package code;
 /** 
 *  Immutable data structure that stores fields of a precise invalidate message 
 **/ 
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

public class PreciseInv implements GeneralInv, SingleWriterInval, Externalizable, Immutable, Comparable
{
  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;
  /*
  private final ObjInvalTarget obj;
  private final AcceptStamp acceptStamp;
  private final AcceptStamp rtAcceptStamp;
  private final boolean embargoed;
  */
  protected final ObjInvalTarget obj;
  protected final AcceptStamp acceptStamp;
  protected final AcceptStamp rtAcceptStamp;
  protected final boolean embargoed;
  protected final boolean committed;

 /** 
 *  empty Constructor for serialization 
 **/ 
  public PreciseInv(){
    
    this.obj = null; 
    
    this.acceptStamp = null;
    
    this.rtAcceptStamp = null;
    
    this.embargoed = false;
    this.committed = false;
  }
  
 /** 
 *  Constructor 
 **/ 
  public PreciseInv(final ObjInvalTarget obj_,
                    final AcceptStamp acceptStamp_,
                    final AcceptStamp rtAcceptStamp_,
                    boolean embargoed_){
    assert(obj_ instanceof Immutable);
    this.obj = obj_;  // obj is immutable
    assert(acceptStamp_ instanceof Immutable);
    this.acceptStamp = acceptStamp_; // acceptStamp is immutable
    assert rtAcceptStamp_ != null;
    assert(rtAcceptStamp_ instanceof Immutable);
    this.rtAcceptStamp = rtAcceptStamp_;
    assert (rtAcceptStamp_.getNodeId().equals(acceptStamp_.getNodeId()));
    this.embargoed = embargoed_;
    this.committed = false;
  }

 /** 
 *  Constructor -- temporaly kept to adapt old codes  
 **/ 
  public PreciseInv(final ObjInvalTarget obj_, final AcceptStamp acceptStamp_){
    assert(obj_ instanceof Immutable);
    this.obj = obj_;  // obj is immutable
    assert(acceptStamp_ instanceof Immutable);
    this.acceptStamp = acceptStamp_; // acceptStamp is immutable

    this.rtAcceptStamp = acceptStamp_;
    this.embargoed = false;
    this.committed = false;
  }


 /** 
 *  Constructor -- temporaly kept to adapt old codes  
 **/ 
  public PreciseInv(final ObjInvalTarget obj_, final AcceptStamp acceptStamp_,
                    final boolean embargoed,
                    final boolean committed){
    assert(obj_ instanceof Immutable);
    this.obj = obj_;  // obj is immutable
    assert(acceptStamp_ instanceof Immutable);
    this.acceptStamp = acceptStamp_; // acceptStamp is immutable

    this.rtAcceptStamp = acceptStamp_;
    this.embargoed = embargoed;
    this.committed = committed;
  }
 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    //System.out.println(PreciseInv.class.getDeclaredFields().length);
    //assert(PreciseInv.class.getDeclaredFields().length == 12);
    assert(PreciseInv.class.getDeclaredFields().length == 12);

    //the fields we custom serialize remain the same
    Field[] f = new Field[5];  
      try{
        f[0] = PreciseInv.class.getDeclaredField("obj");
        f[1] = PreciseInv.class.getDeclaredField("acceptStamp");
        f[2] = PreciseInv.class.getDeclaredField("rtAcceptStamp");
        f[3] = PreciseInv.class.getDeclaredField("embargoed");
        f[4] = PreciseInv.class.getDeclaredField("committed");
        
        assert (f[0].getType().getName().equals("ObjInvalTarget"));
        assert (f[1].getType().getName().equals("AcceptStamp"));
        assert (f[2].getType().getName().equals("AcceptStamp"));
        assert (f[3].getType().getName().equals("boolean"));
        assert (f[4].getType().getName().equals("boolean"));
      }catch(NoSuchFieldException ne){
        assert false;
      }
  }

 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- removed from  
 *                   TaggedOutputStream to here 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("PreciseInv -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

    out.writeObject(this.getObjId().getPath());
    out.writeLong(this.getOffset());
    out.writeLong(this.getLength());
    out.writeLong(this.getAcceptStamp().getLocalClock());

    if(SangminConfig.forkjoin){
      out.writeObject(this.getAcceptStamp().getNodeId());
    } else {
//    out.writeLong(this.getAcceptStamp().getNodeId().getIDint());
      out.writeByte((int)this.getAcceptStamp().getNodeId().getIDint());
    }
    // realtime AcceptStamp & embargoed flag for tact/embargoed write
    out.writeLong(this.getRTAcceptStamp().getLocalClock());
    out.writeBoolean(this.embargoed);
    out.writeBoolean(this.committed);
  }

  
 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --originated from  
 *                   TaggedInputStream 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    String objIdString = (String)(in.readObject());
    long  offset = in.readLong();
    long  length = in.readLong();
    long  localClock = in.readLong();
//    long  nodeIdInt = in.readLong();
    //long  nodeIdInt = in.readByte();
    
    NodeId nodeId;
    if(SangminConfig.forkjoin){
      nodeId = (BranchID)in.readObject();
    } else {
      nodeId = new NodeId(in.readByte());
    }
    
    long realTime = in.readLong();
    boolean  myembargoed = in.readBoolean();
    boolean mycommitted = in.readBoolean();
    /*
      ret = new PreciseInv(new ObjInvalTarget(new ObjId(objIdString),
                                               offset,
                                               length),
                            new AcceptStamp(localClock, new NodeId(nodeIdInt)),
                            new AcceptStamp(realTime, new NodeId(nodeIdInt)),
                            embargoed);
    */
    
    Field[] f = new Field[5];
      
      try{

        f[0] = PreciseInv.class.getDeclaredField("obj");
        f[1] = PreciseInv.class.getDeclaredField("acceptStamp");
        f[2] = PreciseInv.class.getDeclaredField("rtAcceptStamp");
        f[3] = PreciseInv.class.getDeclaredField("embargoed");
        f[4] = PreciseInv.class.getDeclaredField("committed");
      }catch(NoSuchFieldException ne){
        assert false;
      }
      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        assert false;
      }
      try{
        f[0].set(this, new ObjInvalTarget(new ObjId(objIdString),
                                          offset,
                                          length));
        f[1].set(this, new AcceptStamp(localClock, nodeId));
        f[2].set(this, new AcceptStamp(realTime, nodeId));
        f[3].setBoolean(this, myembargoed);
        f[4].setBoolean(this, mycommitted);
      }catch(IllegalArgumentException ie){
        assert false;
      }catch(IllegalAccessException iae){
        assert false;
      }
      
      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        assert false;
      }
  }

 /** 
 *  Return true if this invalidate is embargoed 
 **/ 
  public boolean
  isEmbargoed(){
    return this.embargoed;
  }

  public boolean
  isCommited(){
    return this.committed;
  }
 /** 
 *  Return the accept stamp 
 **/ 
  public final AcceptStamp
  getAcceptStamp(){
    assert(this.acceptStamp instanceof Immutable);
    return(this.acceptStamp);
  }

    public final AcceptStamp getRTAcceptStamp()
    {
      assert this.rtAcceptStamp != null;
      assert(this.rtAcceptStamp instanceof Immutable);
      return(this.rtAcceptStamp);
    }

  public final ObjInvalTarget getObjInvalTarget()
    {
        assert(this.obj instanceof Immutable);
	return(this.obj);
    }

 /** 
 *  getOffset() -- return the offset 
 **/ 
  public final long
  getOffset(){
    return obj.getOffset();
  }

 /** 
 *  getLength() -- return the length 
 **/ 
  public final long
  getLength(){
    return obj.getLength();
  }



 /** 
 *  METHODS INHERITED FROM GENERALINV: 
 **/ 

 /** 
 *  Return the InvalTarget for this invalidate 
 **/ 
    public final InvalTarget getInvalTarget()
    {
	return(this.obj);
    }

	public final ObjId getObjId() {
		return obj.getObjId();
	}

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
    public final AcceptVV getStartVV()
    {
	AcceptVV vv = null;
	AcceptStamp[] uniStamp = null;

	uniStamp = new AcceptStamp[1];
	uniStamp[0] = this.acceptStamp;
	vv = new AcceptVV(uniStamp);
	return(vv);
    }

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
    public final AcceptVV getEndVV()
    {
	return(this.getStartVV()); // For a precise inv, startVV == endVV
    }

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
    public final AcceptVV getRTVV()
    {
	AcceptVV vv = null;
	AcceptStamp[] uniStamp = null;
        
        assert this.rtAcceptStamp != null;
        
	uniStamp = new AcceptStamp[1];
	uniStamp[0] = this.rtAcceptStamp;
	vv = new AcceptVV(uniStamp);
	return(vv);
    }

    //
    // Note: No call for getVV() for a general
    // inval -- need to say whether you mean the start or end
    //

 /** 
 *  Clone this object 
 *  
 *  If we *know* that this object and all objects we reference 
 *  are immutable, then just return this; otherwise perform 
 *      newOne = super.clone();  
 *      now replace any fields in newOne that point to mutable  
 *          objects with a clone of that field 
 *      return newOne 
 *  
 **/ 
    public Object clone()
    {
      assert(this instanceof Immutable);
      return this;
    }

private static boolean warned = false;

private static boolean warned2 = false;

 /** 
 *  Return a new invalidate with mutiple operations performed 
 **/ 
  public GeneralInv
  cloneIntersectInvaltargetChopStartEnd(GeneralInv inv){
    InvalTarget invTarget = null;
    GeneralInv intersectInv = null;

    if(!warned2){
      Env.tbd("Check code for cloneIntersectInvaltargetChopStartEnd()" +
              " for all subclasses of GeneralInv." +
              " I don't trust it (see TODO.txt)");
      warned2 = true;
    }

    // This call should be made for uncommitted invalidates only
    invTarget = this.obj.getIntersection(inv.getInvalTarget());
    assert(invTarget != null);
    if(inv.equals(this)){
      intersectInv = this;
    }else if(invTarget instanceof ObjInvalTarget){
      assert !inv.isCommitted(); //at a specific stamp, either it is a precise inv or a commit inv
      intersectInv = this; //new PreciseInv((ObjInvalTarget)invTarget,
                           //         this.acceptStamp,
                           //         this.rtAcceptStamp,
                           //         (this.embargoed && inv.isEmbargoed()));
    }else if(invTarget instanceof HierInvalTarget){//dummy
      assert (!this.embargoed) || (!inv.isEmbargoed());
      intersectInv = new SingleWriterImpreciseInv((HierInvalTarget)invTarget,
                                                  this.acceptStamp,
                                                  this.acceptStamp,
                                                  this.rtAcceptStamp);
    }else if(invTarget instanceof MultiObjInvalTarget){
      System.err.println("Should not have received a MultiObjInvalTarget " +
                         "as the result of an intersection");
      assert(false);
    }else{
      System.err.println("Invalid inval target type");
      System.err.println("this: " + this);
      System.err.println("inv: " + inv);
      System.err.println("invTarget: " + invTarget);
      assert(false);
    }
    return(intersectInv);
  }

private static boolean warned3 = false;
 /** 
 *  Return the portion of an invalidate for one writer whose 
 *  starting stamp is potentially chopped. 
 **/ 
    public GeneralInv getOneWriterSubsetFrom(NodeId node,
					     long startStampInclusive)
    {
	/*
	  return an invalidation (possibly this) with 
	  interestSet = this.interestSet
	  nodeId = node // assert node was in this.VV
	  startVV{node} == startStampInclusive
	    // assert this.startStamp{node} >= startSTampInclusive
	  endVV{node} == this.endVV{node}
	  (Note: OK to just return this if we meet all of the other criteria)
	*/
	GeneralInv subsetInv = null;

        if(!warned3){
          Env.tbd("Check code for getOneWriterSubsetFrom()"
                  + "==  for startTime seems wrong!");
          warned3 = true;
        }

	assert(node.equals(this.acceptStamp.getNodeId()));
	if(this.acceptStamp.getLocalClock() == startStampInclusive){
	    subsetInv = this;
	}
	return(subsetInv);
    }

 /** 
 *  Return true if the entire invalidation is already accounted 
 *  for by the specified vv 
 **/ 
    public boolean isIncludedBy(VV vv)
    {
	//if we are precise, return vv.includes(this.timestamp)
	//if we are imprecise, return vv.includes(this.*endVV*)
	return(vv.includes(this.acceptStamp));
    }

 /** 
 *  Return true if any component of this invalidation is already 
 *  accounted for by the specified vv 
 **/ 
    public boolean anyComponentIncludedBy(VV vv)
    {
	//if we are precise, return vv.includes(this.timestamp)
	//if we are imprecise, true if for any entry e in *startVV*,
	//  vv.includes(e.timestamp)
	return(vv.includes(this.acceptStamp));
    }

 /** 
 *  Return true if this invalidation is committed 
 **/ 
    public boolean isCommitted()
    {
      return(false);
    }

 /** 
 *  Return true if this invalidate is precise (which it is!) 
 **/ 
    public boolean isPrecise()
    {
	return(true);
    }

 /** 
 *  Return true if this invalidate is precise (which it isn't!) 
 **/ 
    public boolean isBound()
    {
	return(false);
    }


 /** 
 *  isDelete() 
 **/ 
  public boolean
  isDelete(){
    return false;
  }

 /** 
 *  Create a new generalInv that is the union of this and i 
 **/ 
  public GeneralInv
  newUnion(GeneralInv i, SubscriptionSet ss){
    assert (!this.embargoed);
    assert (!i.isEmbargoed());
    /*
      each element of startVV is min of corresponding
                      end        max
      interest set is union of corresponding
      assert(either both are uncommitted or both are committed)
      if both committed 
         assert no gap between csnEnd of one and csnStart of
           next (OK to overlap)
         csnstart = min of both csnStart
         csnEnd = max of both csnMax
    */
    InvalTarget it = null;
    ImpreciseInv impreciseInv = null;
    GeneralInv gi = null;
    AcceptVV newStartVV = null;
    AcceptVV newEndVV = null;
    AcceptVV newRTVV = null;

    it = (i.getInvalTarget()).getUnion(this.obj, ss);

    if((it instanceof ObjInvalTarget) && (!i.equals(this))){
      it = HierInvalTarget.makeHierInvalTarget(this.getObjId().getPath());
    }else if(it instanceof MultiObjInvalTarget){
      // Convert the returned object into a HierInvalTarget because we need
      // to build an Imprecise Invalidate out of the union
      it = ((MultiObjInvalTarget)it).makeHierInvalTarget();
    }
    assert(it instanceof HierInvalTarget);

    // Make an imprecise inval out of this
    newStartVV = this.getStartVV().cloneMinVV(i.getStartVV());
    newEndVV = this.getEndVV().cloneMaxVV(i.getEndVV());
    assert this.getRTVV() != null;
    newRTVV = this.getRTVV().cloneMaxVV(i.getRTVV());
    impreciseInv = new ImpreciseInv((HierInvalTarget)it,
                                    newStartVV,
                                    newEndVV,
                                    newRTVV);
    return(impreciseInv);
  }

 /** 
 *  Return -1, 0, or 1 if this is smaller, equal, or larger than 
 *  gi2, respectively, based on mechanism to provide total orders. 
 **/ 
  public int
  totalOrderCompare(GeneralInv gi2){
    /*
      return -1 if this < gi2 in the total order of the log
      return +1 if this > gi2 in the total order of the log
      return 0 if this and gi2 are equivalent and can be ordered either
      way (both start at the same time and end at the same time)

      we sort by CSN then by min acceptStamp then by serverId

      (1) CSN

      If this.CSN < gi2.CSN, then return -1

      Note that an unassigned CSN has an infinite value)

      For a bulk invalidation, consider the *minimum* CSN included
      in the invalidation.

      If this.CSN == gi2.CSN != CSN_UNASSIGNED, either order 
      is legal. If either is a bulk invalidation, the one with 
      the lower maxCSN goes first. If both have the same maxCSN
      then return 0

      (2) startTime acceptStamp

      assert this.CSN == gi2.CSN == CSN_UNASSIGNED

      if the invalidation is precise, consider its acceptStamp
      if the invalidation is imprecise, consider the startVV and
      look at the minimum entry across all nodeId's in startVV

      if they differ, return -1 or 1

      (3) nodeId
      if both minimum accept stamps in step (2) are equal, 
      then compare the node Ids

      if they differ, return -1 or 1

      (4) endtime acceptStamp

      assert csn, startTime, and nodeId are all the same

      send the "more precise" one first --> look at the endTime
      and the one with the smaller max endtime comes first

      if they differ, return -1 or 1
      if they are the same, return 0 (note that if the invals are
      imprecise, they may still differ in elements other than
      their max or min...but that's ok)
    */
    int result = 0;
    long stamp = Long.MAX_VALUE;
    long minStamp = Long.MAX_VALUE;
    Object gi2StampToken = null;
    Object minStampToken = null;
    AcceptVV gi2AcceptVV = null;
    NodeId nodeId = null;
    VVIterator i = null;

    gi2AcceptVV = gi2.getStartVV();
    for(i = gi2AcceptVV.getIterator(); i.hasMoreElements();){
      gi2StampToken = i.getNext();
      stamp = gi2AcceptVV.getStampByIteratorToken(gi2StampToken);
      if(stamp < minStamp){
        minStamp = stamp;
        minStampToken = gi2StampToken;
      }
    }
    assert(minStampToken != null);
    if(this.acceptStamp.getLocalClock() < minStamp){
      result = -1;
    }else if(minStamp < this.acceptStamp.getLocalClock()){
      result = 1;
    }else{
      nodeId = gi2AcceptVV.getServerByIteratorToken(minStampToken);
      // NOTE: NodeId.compareTo must return {-1,0,1}
      result = this.acceptStamp.getNodeId().compareTo(nodeId);
      if(result == 0){
        // NodeId's are the same; try end stamps
        assert(this.acceptStamp.getNodeId().equals(nodeId));
        if(this.acceptStamp.getLocalClock() <
           gi2.getEndVV().getMaxTimeStamp()){
          result = -1;
        }else if(gi2.getEndVV().getMaxTimeStamp() <
                 this.acceptStamp.getLocalClock()){
          result = 1;
        }else{
          result = 0;
        }
      }
    }
    return(result);
  }

 /** 
 *  METHODS INHERITED FROM SINGLEWRITERINVAL 
 **/ 

 /** 
 *  Return the start vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getStart(){
    return(this.acceptStamp.getLocalClock());
  }

 /** 
 *  Return the end vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getEnd(){
    return(this.acceptStamp.getLocalClock());
  }

 /** 
 *  Return the real vv entry for this writer 
 **/ 
  public long
  getReal(){
    return this.rtAcceptStamp.getLocalClock();
  }

 /** 
 *  Return the ID of the node that did this write 
 **/ 
  public NodeId
  getNodeId(){
    return(this.acceptStamp.getNodeId());
  }

 /** 
 *  Clone this object while chopping off a prefix of the startTime 
 **/ 
  public SingleWriterInval
  cloneChopStartTime(long startTimeOfNew){
    assert(this.acceptStamp.getLocalClock() == startTimeOfNew);
    return((SingleWriterInval)this.clone());
  }

 /** 
 *  Clone this object while chopping off a suffix of the startTime 
 **/ 
  public SingleWriterInval
  cloneChopEndTime(long endTimeOfNew){
    assert(this.acceptStamp.getLocalClock() == endTimeOfNew);
    return((SingleWriterInval)this.clone());
  }

 /** 
 *  Convert to a string 
 **/ 
  public String
  toString(){
    String str = null;

    str = "PreciseInv : <" + this.obj + ", " + this.acceptStamp 
      + ", " + this.rtAcceptStamp + ", " + this.embargoed + ">";
    return(str);
  }

 /** 
 *  memSize() -- 
 *    Estimate the memory footprint of this object in a cheap while not 
 *    very accurate way. 
 *  
 *  Use SizeOf and TestClass to get the number of overheads: 
 *  if delete or add new members, recalculate the number of bytes 
 *  refering to ./test/FootPrintSizeOfStandardObject.txt 
 **/ 
  public long memSize(){
    return 192;
  }

 /** 
 *  Return true if the passed-in object is "equal" to this PreciseInv 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    PreciseInv pi = null;

    // 4 Java-provided fields + 8 declared fields
   if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }
  
  //assert(PreciseInv.class.getDeclaredFields().length == 11): "" +PreciseInv.class.getDeclaredFields().length;
    if(obj instanceof PreciseInv){
      pi = (PreciseInv)obj;
      result = (this.obj.equals(pi.obj) &&
                this.acceptStamp.equals(pi.acceptStamp) &&
                (this.embargoed == pi.embargoed));
    }
    return(result);
  }
  
  public int compareTo(Object o) throws ClassCastException{
    if(o instanceof PreciseInv){
      return this.acceptStamp.compareTo(((PreciseInv)o).acceptStamp);
    }else{
      throw new ClassCastException();
    }
  }

 /** 
 *  Return a hashCode (note: currently unimplemented) 
 **/ 
  public int
  hashCode(){
    assert false : "PreciseInv.hashCode() unimplemented!";
    return(0);
  }
}

//---------------------------------------------------------------------------
/* $Log: PreciseInv.java,v $
/* Revision 1.33  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.32  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.31  2006/09/12 14:57:59  zjiandan
/* Move object custom serialization into individual classes from TaggedOutputStream
/* and TaggedInputStream. The new serialization code consumes a little bit more
/* bytes than the original approach because of java's default object information
/* overhead (I have excluded the field name and type information).
/* The major changes are using java reflection API to serialize "private final"
/* fields (which are very common among our objects that need to custom serialization).
/* Also add sanity check for each class so that we will remember to update
/* the serialization methods when we change the class definitions.
/*
/* */
//---------------------------------------------------------------------------
