package code;
//---------------------------------------------------------------------------
// Represents a set of objects for which we track an lpVV
//---------------------------------------------------------------------------
import java.util.HashMap;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;
import java.util.Enumeration;
import java.io.Serializable;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

public class PreciseSet implements Serializable{ 
// See writeObject, readObject for serialization details

  //-------------------------------------------------------------------------
  // Private data
  //-------------------------------------------------------------------------
  final String name;
  final String fullName;
  CounterVV myLpVV;
  transient Vector liveCons; // See writeObject for serialization details
  CounterVV otherChildrenLpVV;
  transient Vector otherChildrenLiveCons; // See writeObject for serialization
  HashMap children;

  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------
  public
  PreciseSet(String newName, 
             String newFullName,
             VV myLpVV,
             VV otherLpVV){
    assert(newFullName.indexOf(newName) >= 0);
    this.name = newName;
    this.fullName = newFullName;
    this.myLpVV = new CounterVV(myLpVV);
    this.liveCons = new Vector();
    this.otherChildrenLpVV = new CounterVV(otherLpVV);
    this.otherChildrenLiveCons = new Vector();
    this.children = new HashMap();
  }

  //-------------------------------------------------------------------------
  // Constructor -- initialize lpVVs to "beginning of time"
  //-------------------------------------------------------------------------
  public
  PreciseSet(String newName, 
             String newFullName){
    assert(newFullName.indexOf(newName) >= 0);
    this.name = newName;
    this.fullName = newFullName;
    this.myLpVV = (CounterVV)CounterVV.makeVVAllNegatives();
    this.liveCons = new Vector();
    this.otherChildrenLpVV = (CounterVV)CounterVV.makeVVAllNegatives();
    this.otherChildrenLiveCons = new Vector();
    this.children = new HashMap();
  }

  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------
  public
  PreciseSet(String newName, 
             String newFullName,
             VV myLpVV,
             Vector liveCons,
             VV otherLpVV,
             Vector otherLiveCons){
    assert(newFullName.indexOf(newName) >= 0);
    this.name = newName;
    this.fullName = newFullName;
    this.myLpVV = new CounterVV(myLpVV);
    this.liveCons = (Vector)(liveCons.clone());
    this.otherChildrenLpVV = new CounterVV(otherLpVV);
    this.otherChildrenLiveCons = (Vector)(otherLiveCons.clone());
    this.children = new HashMap();
  }

  //-------------------------------------------------------------------------
  // Serialization -- we can't serialize liveCons. Instead, we
  // calculate the current effective lpVV and store/restore that.
  //-------------------------------------------------------------------------
  private void writeObject(ObjectOutputStream out)
    throws IOException{
    //
    // First advance LpVV to reflect current state in active
    // connections.
    //
    advanceMyLpVV();
    advanceOtherChildrenLpVV();

    //
    // Now serialize self (liveCons not serialized but that's
    // OK since their state reflected in lpvv.)
    //
    out.defaultWriteObject();

  }

  //-------------------------------------------------------------------------
  // Serialization -- we can't serialize liveCons. Instead, we
  // calculate the current effective lpVV and store/restore that.
  //-------------------------------------------------------------------------
  private void readObject(ObjectInputStream in)
    throws IOException, ClassNotFoundException{
    in.defaultReadObject();
    this.liveCons = new Vector();
    this.otherChildrenLiveCons = new Vector();
  }

  //-------------------------------------------------------------------------
  // Return the name of this PreciseSet
  //-------------------------------------------------------------------------
  public String
  getName(){
    return(this.name);
  }

  //-------------------------------------------------------------------------
  // Return the full path to this full name
  //-------------------------------------------------------------------------
  public String
  getFullName(){
    return(this.fullName);
  }

  //-------------------------------------------------------------------------
  // Recursively add all nodes contained in "this" and its children to vec
  //-------------------------------------------------------------------------
  public void
  addAllChildren(Vector vec){
    PreciseSet childPS = null;
    Collection childValues = null;
    vec.add(this);
    childValues = this.children.values();
    for(Iterator i = childValues.iterator(); i.hasNext();){
      childPS = (PreciseSet)i.next();
      childPS.addAllChildren(vec);
    }
  }

  //-------------------------------------------------------------------------
  // Return the child with name "name", or null if there is no such child
  //-------------------------------------------------------------------------
  public PreciseSet
  getChild(String name){
    return((PreciseSet)this.children.get(name));
  }

  //-------------------------------------------------------------------------
  // Return a vector of all children
  //-------------------------------------------------------------------------
  public Vector
  getChildrenVector(){
    Vector childVec = null;

    childVec = new Vector();
    for(Iterator i = this.children.values().iterator(); i.hasNext();){
      childVec.add((PreciseSet)i.next());
    }
    return(childVec);
  }

  //-------------------------------------------------------------------------
  // Return the lpvv
  //-------------------------------------------------------------------------
  public AcceptVV
  getMyLpVV(){
    return(this.myLpVV.cloneAcceptVV());
  }

  //-------------------------------------------------------------------------
  // Return the other children lpvv
  //-------------------------------------------------------------------------
  public AcceptVV
  getOtherChildrenLpVV(){
    return(this.otherChildrenLpVV.cloneAcceptVV());
  }

  //-------------------------------------------------------------------------
  // Return the liveCons
  //-------------------------------------------------------------------------
  public Vector
  getLiveCons(){
    return(this.liveCons);
  }

  //-------------------------------------------------------------------------
  // Return the other children liveCons
  //-------------------------------------------------------------------------
  public Vector
  getOtherChildrenLiveCons(){
    return(this.otherChildrenLiveCons);
  }

  //-------------------------------------------------------------------------
  // Return the max(lpvv, liveCons.prevVV)
  //-------------------------------------------------------------------------
  public AcceptVV
  getMyMaxLpVV(){
    CounterVV ret = new CounterVV(this.myLpVV);
    for(Enumeration e = liveCons.elements(); e.hasMoreElements();){
      ConnectionState ic = (ConnectionState) e.nextElement();
      ret.advanceTimestamps(ic.getPrevVV());
    }
    return(ret.cloneAcceptVV());
  }


  //-------------------------------------------------------------------------
  // Advance myLpVV as permitted by connections
  // (This is legal to do at any time. We use it before serializing
  // self to disk.)
  //-------------------------------------------------------------------------
  private void
  advanceMyLpVV(){
    for(Enumeration e = liveCons.elements(); e.hasMoreElements();){
      ConnectionState ic = (ConnectionState) e.nextElement();
      myLpVV.advanceTimestamps(ic.getPrevVV());
    }
  }

  //-------------------------------------------------------------------------
  // Return the max(other children lpvv, otherLiveCons.prevVV)
  //-------------------------------------------------------------------------
  public AcceptVV
  getOtherChildrenMaxLpVV(){
    CounterVV ret = new CounterVV(this.otherChildrenLpVV);
    for(Enumeration e = otherChildrenLiveCons.elements(); e.hasMoreElements();){
      ConnectionState ic = (ConnectionState) e.nextElement();
      ret.advanceTimestamps(ic.getPrevVV());
    }
    return(ret.cloneAcceptVV());
  }

  //-------------------------------------------------------------------------
  // Advance otherChildrenLpVV as permitted by connections
  // (This is legal to do at any time. We use it before serializing
  // self to disk.)
  //-------------------------------------------------------------------------
  private void
  advanceOtherChildrenLpVV(){
    for(Enumeration e = otherChildrenLiveCons.elements(); e.hasMoreElements();){
      ConnectionState ic = (ConnectionState) e.nextElement();
      otherChildrenLpVV.advanceTimestamps(ic.getPrevVV());
    }
  }


  //------------------------------------------------------------------------
  // Note: add conID should be very conservative.
  //       if the PSTree doesn't exactly match the ss,
  //       only add conID to a PSTreeNode if the PSTreeNode is completely included in ss
  //------------------------------------------------------------------------
  public void addLiveCons(SubscriptionSet ss, ConnectionState ic){
    Collection ssChildren = null;
    SubscriptionSet ssChild = null;
    
    PreciseSet psChild = null;

    assert ss.getName().equals(this.getName());
    
    if(ss.containsSelf()){
      if(!this.liveCons.contains(ic)){
        this.liveCons.add(ic);
      }
    }
    if(ss.containsChildren()){
      this.addLiveConsToAllChildren(ic);
    }

    ssChildren = ss.getChildValues();
    for(Iterator i = ssChildren.iterator(); i.hasNext();){
      ssChild = (SubscriptionSet)i.next();
      psChild = this.getChild(ssChild.getName());
      
      if(psChild != null){
        psChild.addLiveCons(ssChild, ic);
      }//otherwise ignore
    } 
  }
  
  //------------------------------------------------------------------------
  // Traverse the subtree and recursively add conID to all liveCons
  // (Protected rather than private to facilitate testing by
  // PreciseSetUnit.)
  //------------------------------------------------------------------------
  protected void addLiveConsToAllChildren(ConnectionState ic){
    PreciseSet ps = null;
    PreciseSet thisChild = null;
    PreciseSet psChild = null;

    if(!this.otherChildrenLiveCons.contains(ic)){
      this.otherChildrenLiveCons.add(ic);
    }
    for(Iterator i = this.children.values().iterator();i.hasNext();){
        thisChild = (PreciseSet)i.next();
        thisChild.addLiveConsToAllChildren(ic);
        if(!thisChild.liveCons.contains(ic)){
          thisChild.liveCons.add(ic);
        }
    }
  }

  //------------------------------------------------------------------------
  // update every node's lpvv whose name is overlapped by ss, and remove
  // connection if exists
  //
  // if ic = null => update overlapping lpvv
  // Note: different from addLiveCons, removeLiveCons can remove incomplete
  //       matching node. i.e. when the exact node doesn't exist, remove it
  //       from the latest ancester's otherChildren
  // 
  //Note: if there's no exactly match PTreeNode,
  //       e.g. if /a/b     and the node /a doesn't have child b
  //        then if ic in /a.otherChildrenLiveCons, 
  //             remove it from the /a.otherChildrenLiveCons and update
  //             /a.otherChildrenLpVV with ic.prevVV(note not newLpVV)
  //------------------------------------------------------------------------
  public void removeLiveCons(SubscriptionSet ss, 
			     ConnectionState ic,
                             VV newLpVV){
    Collection ssChildren = null;
    SubscriptionSet ssChild = null;
    
    PreciseSet psChild = null;
    
    assert ss.getName().equals(this.getName());
    
    if(ss.containsSelf()){
      if(ic != null){
        this.liveCons.remove(ic);  
      }
      this.myLpVV.advanceTimestamps(newLpVV);
    }
    if(ss.containsChildren()){
      this.removeLiveConsFromAllChildren(ic, newLpVV);
    }

    ssChildren = ss.getChildValues();
    for(Iterator i = ssChildren.iterator(); i.hasNext();){
      ssChild = (SubscriptionSet)i.next();
      psChild = this.getChild(ssChild.getName());
      if(psChild != null){
        psChild.removeLiveCons(ssChild, ic, newLpVV);
      } else {// the child doesn't exist
        // It is very important that we need to remove the same connection from
        // the parent's otherChildrenLiveCons. Because the absent of the match child
        // implies that the parent's otherChildren includes this child.
        // we must remove the connection from the otherChildren as well.
        
        if((ic != null) && ( this.otherChildrenLiveCons.remove(ic))){
          //if the connection does exists in the otherChildrenLiveCons
          // update the otherChildrenLpVV with the ic.getPrevVV
          // instead of newLpVV because the newLpVV might be newer.
          // including gi.startVV-1
          this.otherChildrenLpVV.advanceTimestamps(ic.getPrevVV());
        }
        
      }

    } 
  }
  
  //------------------------------------------------------------------------
  // remove conID from all liveCons lists and update lpvv including this.other
  // (Protected rather than private to facilitate testing by
  // PreciseSetUnit.)
  //------------------------------------------------------------------------
  protected void removeLiveConsFromAllChildren(ConnectionState ic,
                                               VV newLpVV){
    PreciseSet ps = null;
    PreciseSet thisChild = null;
    PreciseSet psChild = null;

    if(ic != null){
      this.otherChildrenLiveCons.remove(ic);
    }
    this.otherChildrenLpVV.advanceTimestamps(newLpVV);
    
    for(Iterator i = this.children.values().iterator();i.hasNext();){
        thisChild = (PreciseSet)i.next();
        thisChild.removeLiveConsFromAllChildren(ic, newLpVV);
        if(ic != null){
          thisChild.liveCons.remove(ic);
        }
        thisChild.myLpVV.advanceTimestamps(newLpVV);
        
    }
  }
  
  //-------------------------------------------------------------------------
  // Split this precise set by creating a new child with name "newName"
  //-------------------------------------------------------------------------
  public PreciseSet
  split(String newName){
    PreciseSet newPS = null;

    newPS = (PreciseSet)(this.children.get(newName));
    if(newPS == null){
      newPS = new PreciseSet(newName,
                             this.fullName + "/" + newName,
                             this.otherChildrenLpVV,
                             this.otherChildrenLiveCons,
                             this.otherChildrenLpVV,
                             this.otherChildrenLiveCons);
      this.children.put(newName, newPS);
    }
    return(newPS);
  }

  //-------------------------------------------------------------------------
  // split PSTreeNode if necessary to make PSTree completely match SubSet
  //-------------------------------------------------------------------------
  public void
  split(SubscriptionSet ss){
    //- For each ssChild in ss.getChildren()
    //  - if(!this.children.contains(ssChild.getName()))
    //	            - this.split(ssChild.getName());
    //	- child = this.children.get(ssChild.getName());
    //	- assert child != null;
    //	- child.split(ssChild);
    
    Collection ssChildren = null;
    SubscriptionSet ssChild = null;
    
    PreciseSet psChild = null;
    if(ss.isEmpty()){
      return;
    }

    assert ss.getName().equals(this.getName());
    
    ssChildren = ss.getChildValues();
    for(Iterator i = ssChildren.iterator(); i.hasNext();){
      ssChild = (SubscriptionSet)i.next();
      psChild = this.getChild(ssChild.getName());
      if(psChild == null){
        psChild = this.split(ssChild.getName());
      }
      assert psChild != null;
      psChild.split(ssChild);
    } 
  }

  //-------------------------------------------------------------------------
  // join all this.children, remove this.children, update this.other.lpvv, this.liveCons
  //-------------------------------------------------------------------------
  public void
  join(){
    //    	- For each child in this.children
    //			- child.join();
    //			- this.other.lpVV = min(this.other.lpVV, child.lpVV, child.other.lpVV);
    //			- assert child.children.isEmpty()
    //			- this.other.liveCons = intersection(this.other.liveCons, child.liveCons, 
    //						child.other.liveCons);
    //		- this.children.makeEmpty()
    
    PreciseSet ps = null;
    PreciseSet thisChild = null;
    PreciseSet psChild = null;
    boolean result = false;

    this.otherChildrenLpVV = new CounterVV(this.getChildMinLpVV());
    this.otherChildrenLiveCons = this.getChildCommonLiveCons();
                                   
    this.children.clear();
  }

  //-------------------------------------------------------------------------
  // update myLpVV
  //-------------------------------------------------------------------------  
  public void updateMyLpVV(VV newLpVV){
    this.myLpVV.advanceTimestamps(newLpVV);
  }

  //-------------------------------------------------------------------------
  // update otherChildrenLpVV
  //-------------------------------------------------------------------------  
  public void updateOtherChildrenLpVV(VV newLpVV){
    this.otherChildrenLpVV.advanceTimestamps(newLpVV);
  }

  //-------------------------------------------------------------------------
  // update otherChildrenLpVV and all children's subtree's lpvv and otherlpvv 
  //-------------------------------------------------------------------------  
  public void updateAllChildLpVV(VV newLpVV){
    PreciseSet thisChild = null;
    updateOtherChildrenLpVV(newLpVV);
    for(Iterator i = this.children.values().iterator();i.hasNext();){
        thisChild = (PreciseSet)i.next();
        thisChild.updateMyLpVV(newLpVV);
        thisChild.updateAllChildLpVV(newLpVV);
    }
  }

  //-------------------------------------------------------------------------
  // get min of all "this" node's children's LpVVs
  //-------------------------------------------------------------------------
  public AcceptVV
  getChildMinLpVV(){
        
    PreciseSet ps = null;
    PreciseSet thisChild = null;
    PreciseSet psChild = null;
    
    CounterVV minLpVV = new CounterVV(this.getOtherChildrenMaxLpVV());

    for(Iterator i = this.children.values().iterator();i.hasNext();){
        thisChild = (PreciseSet)i.next();
        minLpVV.setToMinVV(thisChild.getChildMinLpVV());
        minLpVV.setToMinVV(thisChild.getMyMaxLpVV());
    }
    return minLpVV.cloneAcceptVV();
  }
  
  //-------------------------------------------------------------------------
  // get min of all the subtree lpVVs including the root(i.e. "this")
  //-------------------------------------------------------------------------
  public AcceptVV
  getSubtreeMinLpVV(){
        
    PreciseSet ps = null;
    PreciseSet thisChild = null;
    PreciseSet psChild = null;
    
    CounterVV minLpVV = new CounterVV(this.getChildMinLpVV());
    minLpVV.setToMinVV(this.getMyMaxLpVV());
    return minLpVV.cloneAcceptVV();
  }
  
  //-------------------------------------------------------------------------
  // get all children's minLpVV
  //-------------------------------------------------------------------------
  public Vector
  getChildCommonLiveCons(){
        
    Env.tbd("PreciseSet:join() getChildCommonLiveCons instead "
            + "of conservatively set as empty and notify Controller that some connection"
            + " actually doesn't Effectively attached to some subscriptionset");
    return new Vector();
  }
  
  //-------------------------------------------------------------------------
  // Compare whether "o" is equal to this object
  //-------------------------------------------------------------------------
  public boolean
  equals(Object o){
    PreciseSet ps = null;
    PreciseSet thisChild = null;
    PreciseSet psChild = null;
    boolean result = false;

    if(o instanceof PreciseSet){
      ps = (PreciseSet)o;
      result = (this.name.equals(ps.name) &&
                this.fullName.equals(ps.fullName) &&
                (this.myLpVV.equals(ps.getMyLpVV())) &&
                (this.otherChildrenLpVV.equals(ps.getOtherChildrenLpVV()))&&
                (this.liveCons.equals(ps.getLiveCons()))&&
                (this.otherChildrenLiveCons.equals(ps.getOtherChildrenLiveCons()))&&
                (this.children.size() == ps.children.size()));
      for(Iterator i = this.children.values().iterator();
          i.hasNext() && result;){
        thisChild = (PreciseSet)i.next();
        psChild = ps.getChild(thisChild.name);
        result = ((psChild != null) && (thisChild.equals(psChild)));
      }
    }else{
      result = false;
    }
    return(result);
  }

  //-------------------------------------------------------------------------
  // Used for testing
  //-------------------------------------------------------------------------
  public String
  toString(){
    String ret = "(" + this.fullName + ", myLpVV=" + this.myLpVV + ", liveCons={";
    for(Enumeration e = this.liveCons.elements(); e.hasMoreElements();){
      ConnectionState ic = (ConnectionState)(e.nextElement());
      if(ic.getSenderId() != null){
        ret += ic.getSenderId().toString() + ", ";
      }
      else{
        ret += "<fake senderID stream>, ";
      }
    }
    ret += "}, otherChildrenLpVV=" + this.otherChildrenLpVV;
    ret += " otherChildrenLiveCons={";
    for(Enumeration e = this.otherChildrenLiveCons.elements(); e.hasMoreElements();){
      ConnectionState ic = (ConnectionState)(e.nextElement());
      if(ic.getSenderId() != null){
        ret += "senderId " + ic.getSenderId().toString() + ":" + ic.getPrevVV()+", ";
      }
      else{
        ret += "<fake senderID stream>:" + ic.getPrevVV()+ ", ";
      }
    }
    ret += "}";
    
    for(Iterator i = this.children.values().iterator(); i.hasNext();){
      PreciseSet thisChild = (PreciseSet) i.next();
      ret += thisChild.toString();
    }

    ret += ")";
    return ret;
  }

  //-------------------------------------------------------------------------
  // Used for testing
  //-------------------------------------------------------------------------
  public static void
  main(String argv[]){
    System.out.println("PreciseSet unit tests now in junit framework -- run PreciseSetUnit");
  }

}

//---------------------------------------------------------------------------
/* $Log: PreciseSet.java,v $
/* Revision 1.7  2007/07/11 19:08:07  zjiandan
/* clean IncommingConnection
/*
/* Revision 1.6  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.5  2006/08/25 20:31:45  dahlin
/* Added serialization to PreciseSets (plan is to use serialization rather than sync() for persistence). Moved PreciseSet unit tests to junit (PreciseSetUnit)
/*
/* Revision 1.4  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/**/
//---------------------------------------------------------------------------
