package code;
/* RandomAccessState
 * 
 * Provide random access to local stable on-disk state and checkpoint.
 * We use BerkeleyDB as the local store.
 *
 * Concurrency, locking
 * All state is stored in BerkeleyDB, which handles locking and concurrency.
 * All access to this state is within transactions.
 * So, even though this class uses no locks, it is thread safe.
 * 
 * Checkpoints and stability
 * Writes and transactions are not flushed to disk. The call to
 * sync(vv) flushes the state up to at least VV to disk. The reason
 * that we can avoid flushing to disk is that the UniversalFS Log
 * is a redo log of at least everything up to the last call to sync(vv).
 *
 * Performance and limitations:
 * In the current design, even though RandomAccessState allows concurrency,
 * the caller (LocalStore) is a monitor, so only one request will be
 * outstanding at a time. We expect good performance b/c (1) writes 
 * are asynchronous and (2) reads should come from the cache. Note
 * that you will get bad performance if you have a lot of concurrent reads 
 * (serialized in LocalStore) that miss in the cache. A potential
 * solution would be to have LocalStore first issue the read w/o holding
 * the lock as a prefetch and then grab the lock and re-issue read.
 * 
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

import com.sleepycat.je.*;
import com.sleepycat.bind.tuple.TupleBinding;
import com.sleepycat.bind.tuple.TupleOutput;
import com.sleepycat.bind.tuple.TupleInput;
import com.sleepycat.bind.serial.StoredClassCatalog;
import com.sleepycat.bind.serial.SerialBinding;

import java.io.File;
import java.io.IOException;
import java.io.EOFException;
import java.io.Serializable;
import java.io. Externalizable;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.UnsupportedEncodingException;
import java.lang.IllegalArgumentException;
import java.util.Random;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;

public class RandomAccessState{
    
  public static final long NO_NEW_VALID_BYTES = -1;
  public static final long SENTINAL_SMALL_FILE_NUM = -1;
  public static final long SENTINAL_LARGE_FILE_NUM = Long.MAX_VALUE;
  public static final long RAS_SHIP_DONE  = 0xFF10FF13;
  public static final double PRIORITY_NOT_APPLICABLE = -1.0;

  protected static boolean warnTBDLongLengths = false;
  private static boolean warnedExpensiveSanity = false;
  private static final boolean dbgVerbose = false;
  private static final boolean checkScanTime = false;//checkScanCheckpoint latency
  private static final boolean printWriteObjectForCP = DataStore.printWriteObjectForCP;

  private static final boolean testTxnPerformance = false;
  private static final boolean dbgPerformance = false;
  protected boolean doExpensiveSanityChecks;

  private static boolean enableTimingStats = false;
  private static TimingStats timingStats = null;

  private Environment dbEnv;
  private final boolean PRINT_CP_INFO = DataStore.PRINT_CP_INFO;
  private boolean didCleanClose = false;

  
//
// Global state tuple store
//    fnumKey: MAX_FILE_NUM --> max file num
//
  private Database globalStateDB;
  private DatabaseEntry fnumKey;


//
// CLEANVV <nodeID> --> accept stamp
//   
  private Database vvDB;

//
// Store auxiliary checkpoint stuff --
// version vector and ISStatus as of the
// most recent checkpoint. Note that
// cpVV and ISStatus are checkpointed
// periodically, so they may be "behind"
// the per-objec state stored in the
// restof RandomAccessState. The recovery
// code copes with this by replaying
// the replay log from cpVV (which 
// may cause some per-object state updates
// to be executed multiple times.
//
// CPVV_KEY --> serialized CPVV
// ISSTATUS_KEY --> serialized ISStatus
//
  private Database auxCheckpointStuffDB; 

//
// objId --> fileNum
//           lastDeletedAccept
//           lastWriteAccept
//
  private Database perObjStateDB;

//
// (fileNum, offset) --> length
//                       accept
//                       prevaccept
//                       valid
//                       embargoed
//                       embargoedAcceptStamp
//
  private Database perRangeStateDB;

//
// (fileNum, offset) --> byte[]
//
  private Database perRangeBodyDB;

//
// For serialization type info
//
  private Database classDB;
  private StoredClassCatalog classCatalog;
  //private SerialBinding objIdBinding;
  private SerialBinding nodeIdBinding;
  private SerialBinding longBinding;
  private SerialBinding perObjDataBinding;
  private PerRangeKeyBinding perRangeKeyBinding;
  private SerialBinding perRangeDataBinding;

 /** 
 *  AcceptStamp placeholder for "beginning of time" 
 **/ 
  private final static AcceptStamp 
  PREV_ACCEPT_BEGINNING_OF_TIME;// = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, 
                                //                  new NodeId(NodeId.WILDCARD_NODE_ID));


  private final static AcceptStamp 
  ACCEPT_BEGINNING_OF_TIME;// = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, 
                           //                  new NodeId(NodeId.WILDCARD_NODE_ID));
  
  static{
    NodeId nId = new NodeId(NodeId.WILDCARD_NODE_ID);
    if(SangminConfig.forkjoin){
      nId = new BranchID(nId.getIDint());
    }
    PREV_ACCEPT_BEGINNING_OF_TIME =
      new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, nId);
    ACCEPT_BEGINNING_OF_TIME = 
      new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, nId);    
  }
  


  //
  // This class can outrun the garbage collector (see unit test 18
  // in RandomAccessStateUnit.) So, we call System.gc() in 
  // the background after we read or write 
  // START_BG_FORCE_GC_BYTES bytes. We block if System.gc()
  // does not complete at least every BLOCK_FORCE_GC_BYTES
  // bytes.
  //
  private static long START_BG_FORCE_GC_BYTES;
  private static long BLOCK_FORCE_GC_BYTES;
  private static final long GC_FORCER_CHARGE_PER_OP = 1000;
  private GCForcer gcForcer;

  
 /** 
 *  RandomAccessState -- constructor 
 *  
 *  Note that we set "transaction no sync" on the environment. Transactions 
 *  may remain in memory until we call sync(). This is OK because the 
 *  UpdateLog serves as a redo log. 
 **/ 
  public 
  RandomAccessState(String storePath, long cacheSizeBytes)
    throws IOException{

    
    doExpensiveSanityChecks = Env.getDoExpensiveSanityChecks();


    //
    // Initiate a gc as often as 10% of the bytes in cache 
    // are touched. Force a gc at least as often as 20%
    // of byte in cache are touched.
    //
    START_BG_FORCE_GC_BYTES = (long)(cacheSizeBytes * 0.1); 
    BLOCK_FORCE_GC_BYTES = 2 * START_BG_FORCE_GC_BYTES;
    //gcForcer = new GCForcer(START_BG_FORCE_GC_BYTES, BLOCK_FORCE_GC_BYTES);
    //gcForcer.startWorker();

    EnvironmentConfig envConfig = new EnvironmentConfig();
    envConfig.setTransactional(true);
    envConfig.setAllowCreate(true);
    envConfig.setCacheSize(cacheSizeBytes);
    EnvironmentMutableConfig mutableConfig = new EnvironmentMutableConfig();
    mutableConfig.setTxnNoSync(true);
    DatabaseConfig dbConfig = new DatabaseConfig();
    dbConfig.setTransactional(true);
    dbConfig.setAllowCreate(true);
    Transaction txn = null;
    File dbDir = new File(storePath);

    if(doExpensiveSanityChecks && !warnedExpensiveSanity){
      Env.performanceWarning("RandomAccessState -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(!warnTBDLongLengths){
      //
      // The problem here is that invals can be passed to us by remote nodes.
      // Right now a remote node can make us crash (assert false) by handing
      // us a "long" invalidate. The fix is to check to see the length of
      // the invalidate and if it is too long, apply it in 2^32-1 byte 
      // chunks *to the perRangeBody* part. So, updatePerRangeBody()
      // needs to be ready to break things apart and readPerRangeBody()
      // needs to be ready to put things together. (The problem here is that
      // BerkeleyDB returns results in arrays, which in Java are indexed
      // by int not by long.)
      //
      Env.tbd("Fix security bug: "
              + "RandomAccessState assumes inval length < 2^31-1.");
      warnTBDLongLengths = true;
    }

    //
    // Initialize environment and databases
    //
    try{
      dbDir.mkdirs();
      dbEnv = new Environment(dbDir, envConfig);
      dbEnv.setMutableConfig(mutableConfig);
      txn = dbEnv.beginTransaction(null, null);
      globalStateDB = dbEnv.openDatabase(txn, "GlobalState", dbConfig);
      vvDB = dbEnv.openDatabase(txn, "VV", dbConfig);
      auxCheckpointStuffDB = dbEnv.openDatabase(txn, "AuxCheckpointStuff", dbConfig);
      perObjStateDB = dbEnv.openDatabase(txn, "PerObjState", dbConfig);
      perRangeStateDB = dbEnv.openDatabase(txn, "PerRangeState", dbConfig);
      perRangeBodyDB = dbEnv.openDatabase(txn, "PerRangeBody", dbConfig);
      classDB = dbEnv.openDatabase(txn, "StoredClassCatalog", dbConfig);
      classCatalog = new StoredClassCatalog(classDB);
      txn.commit();
      txn = null;
    }
    catch(IllegalArgumentException iae){
      if(txn != null){
        try{
          txn.abort();
        }
        catch(Exception ae){
          Env.warn("Nested exception aborting after illegal argument:" 
                   +  ae.toString());
          ; // Ignore nested exception...we're dying anyhow...
        }
      }
      assert(false);
    }
    catch(DatabaseException dbe){
      String msg = dbe.toString();
      if(txn != null){
        try{
          txn.abort();
        }
        catch(Exception ae){
          msg = msg + ("\nALSO: Nested exception aborting:" +  ae.toString());
        }
      
      }
      throw new IOException("Exception opening berkeleyDB Environment: " 
                            + msg);
    }

    //
    // Initialize binding objects
    //
    nodeIdBinding = new SerialBinding(classCatalog, NodeId.class);
    longBinding = new SerialBinding(classCatalog, Long.class);
    //objIdBinding = new SerialBinding(classCatalog, ObjId.class);
    perObjDataBinding = new SerialBinding(classCatalog, PerObjData.class);
    perRangeKeyBinding = new PerRangeKeyBinding();
    perRangeDataBinding = new SerialBinding(classCatalog, PerRangeData.class);

    //
    // Keys for globalStateDB tuple store
    //
    try{
      fnumKey = new DatabaseEntry("MAX_FNUM".getBytes("UTF-8"));
    }
    catch(UnsupportedEncodingException uee){
      assert(false);
    }

    //
    // Insert sentinals into perRangeState DB to simplify search
    //
    insertSentinals();

    timingStats = new TimingStats();
  }


  
 /** 
 *  RandomAccessState -- constructor -- for testing purposes only 
 *  -- so that nullRAS can initialize 
 **/ 
  protected
  RandomAccessState()
    throws IOException{
    // do nothing

  }    
   

 /** 
 *  enable timing stats -- gathers read/write timing information 
 **/ 
  public void enableTimingStats(){
    enableTimingStats = true;
    timingStats.enableTimingStats();


    //timing stats for apply Body
    timingStats.initialize("applyBody-total");
    timingStats.initialize("applyBody-GCForcer");
    timingStats.initialize("applyBody-convertToInval");

    // timing stats for inval TX
    timingStats.initialize("invalTX-total");
    timingStats.initialize("invalTX-updateVV");
    timingStats.initialize("invalTX-updateObjState");
    timingStats.initialize("invalTX-updatePRS");
    
    // timing stats for PRS
    timingStats.initialize("updatePRS-setCursor");
    timingStats.initialize("updatePRS-clearChunk");
    timingStats.initialize("updatePRS-insertMetaData");
    timingStats.initialize("updatePRS-removePRBody");
    timingStats.initialize("updatePRS-updatePRBody");
    timingStats.initialize("updatePRS-checkCursor");

    // timing stats for apply Inval
    timingStats.initialize("applyInval-total");
    timingStats.initialize("applyInval-GCForcer");

    // timing stats for read
    timingStats.initialize("read-total");
    timingStats.initialize("read-TX");
    timingStats.initialize("read-TX-prs");
    timingStats.initialize("read-TX-getValidBytes");
    timingStats.initialize("read-GCForcer");

  }
   
 /** 
 *  getTimingStats 
 **/ 
  public TimingStats getTimingStats() {
    return timingStats;
  }

  
 /** 
 *  printEnvStats 
 **/ 
  public void printEnvStats(){
    try{
      StatsConfig config = new StatsConfig();
      config.setClear(true);
      System.out.println(dbEnv.getStats(config));
    }catch(Exception e) {
    }
  }

 /** 
 *  finalizer 
 **/ 
  protected void
  finalize(){
    if(!didCleanClose){
      close();
    }
  }


 /** 
 *  close() -- flush everything to disk and close. 
 **/ 
  public void
  close(){
    if(didCleanClose){
      return;
    }
    try{
      globalStateDB.close();
      globalStateDB = null;
      vvDB.close();
      vvDB = null;
      auxCheckpointStuffDB.close();
      auxCheckpointStuffDB = null;
      perObjStateDB.close();
      perObjStateDB = null;
      perRangeStateDB.close();
      perRangeStateDB = null;
      perRangeBodyDB.close();
      perRangeBodyDB = null;
      classDB.close();
      classDB = null;
      classCatalog = null;
      //objIdBinding = null;
      nodeIdBinding = null;
      longBinding = null;
      perObjDataBinding = null;
      perRangeKeyBinding = null;
      perRangeDataBinding = null;
      dbEnv.sync();
      dbEnv.close();
      dbEnv = null;
      //gcForcer.killWorker(); // Make sure GCForcer can be gc'd
      gcForcer = null;
      didCleanClose = true;
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      Env.warn("Exception closing database in RandomAccessState: " + dbe.toString());
      return; // Ignore exception
    }
  }


 /** 
 *  sync() -- make sure that data are safely on disk 
 **/ 
  public void
  sync()
    throws IOException{
    
    try{
      dbEnv.sync();
    }
    catch(DatabaseException e){
      e.printStackTrace();
      throw new IOException(e.toString());
    }
  }


 /** 
 *  read() -- read up to maxLen bytes at specified offset of objId 
 *  
 *  returns 
 *     BodyMsg containing bytes of data on success 
 *     throws NoSuchEntryException if objId does not exist (includes 
 *          case where delete of objId is more recent than any write 
 *          to objId) 
 *     throws EOFException if offset is larger than any write to 
 *          objId (since the last delete) 
 *     throws ReadOfInvalidRangeException if we've seen invalidation 
 *          that overlaps offset but not yet seen corresponding body 
 *     throws ReadOfHoleException( nextBytePosition) 
 *          if read of a "hole"(an offset that has not been written 
 *        since the last delete of the file that is less than some  
 *        other offset that has been written since the last delete) 
 *  
 *  Note: 
 *     The length specified is a maximum. If the offset is overlapped 
 *        by VALID data, we return AT LEAST one byte and AT MOST 
 *        maxLen. In particular, we may return fewer than maxLen bytes 
 *        if another write begins at offset' < offset + maxLen. 
 *  
 *  
 *     On the read of a "hole"  
 *         
 *        throw ReadOfHoleException(nextBytePosition) 
 *         
 *        <<OLD comments: 
 *        we return a 1-byte long result with a value of "0" and 
 *        an accept stamp of "the beginning of time". This case 
 *        corresponds to reading a "hole" in Unix (though we only 
 *        return one byte at a time for simplicity).>> 
 *  
 *      
 *  
 *  Performance note:  
 *      If you do lots of small writes to a file, then read 
 *      results will be broken up into lots of small reads. 
 *      This is for simplicity. It is also a consequence of the 
 *      current interface which returns metadata (which is per write) 
 *      along with the data. Potential solutions are (1) change the 
 *      interface to allow us to return a read result that contains 
 *      a bunch of writes grouped together or (2) change the 
 *      write implementation (at a higher level than this) to  
 *      group together back-to-back writes by a node (that do 
 *      not causally depend on anything from anywhere else). 
 *  
 *       
 **/ 
  public BodyMsg 
  read(ObjId objId, long offset, long maxLen, boolean exactOffset)
    throws NoSuchEntryException, EOFException, ReadOfInvalidRangeException,
	   ReadOfEmbargoedWriteException,
	   ReadOfHoleException{
    timingStats.startTimer("read-total");
    assert(objId != null);
    assert(offset >= 0);
    assert(maxLen > 0); // Note: conceivable that we have to come back and allow 0 len
    // In that case, I would return correct accept stamp w/ 0-len result

    try{ 
      BodyMsg ret = readTransaction(objId, offset, maxLen, exactOffset);
      
      timingStats.startTimer("read-GCForcer");
      //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP + ret.getLength());      
      timingStats.endTimer("read-GCForcer");

      timingStats.endTimer("read-total");
      return ret;
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
      return null;
    }
    catch(IllegalArgumentException iae){
      iae.printStackTrace();
      assert(false);
      return null;
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      de.printStackTrace();
      assert false : "This should *never* happen in current design";
      return null;
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      assert(false);
      return null;
    }
  }


 /** 
 *  Static variables used by readTransaction() 
 **/ 
  /*commented by zjd see read method comments

  protected static AcceptStamp acceptOfHole = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN,
                                                            new NodeId(NodeId.INVALID_NODE_ID));
  private static byte[] holeVal = {0};
  private static ImmutableBytes holeDat = new ImmutableBytes(holeVal);
  */

 /** 
 *  readTransaction() -- do the actual work of a read() 
 **/ 
  private BodyMsg 
  readTransaction(ObjId objId, long offset, long maxLen, boolean exactOffset)
    throws NoSuchEntryException, EOFException, ReadOfInvalidRangeException,
	   ReadOfEmbargoedWriteException,
	   NullPointerException, DeadlockException, 
	   IllegalArgumentException, DatabaseException,
	   ReadOfHoleException{

    timingStats.startTimer("read-TX");

    PerRangeState prs;
    ImmutableBytes bytes;
    BodyMsg result;
    Transaction txn = null;
    try{
      txn = dbEnv.beginTransaction(null, null);

      try{
        timingStats.startTimer("read-TX-prs");
        prs = getPerRangeState(txn, objId, offset);
        timingStats.endTimer("read-TX-prs");

      }
      catch(EOFException eof){ // Not-abnormal result of getPerRangeState
        assert(txn != null);
        txn.commit();
        txn = null;
        throw eof;
      }
      catch(NoSuchEntryException nse){ // Not-abnormal result of getPerRangeState
        assert(txn != null);
        txn.commit();
        txn = null;
        throw nse;
      }
      //added by zjd
      //so that we throw an ReadOfHoleException instead of return a one byte 0 value
      //for the hole
      catch(ReadOfHoleException rhe){ //read of hole 
	assert(txn != null);
        txn.commit();
        txn = null;
        throw rhe;
      }
      /* commentted by zjd -- see above ReadOfHoleException
      if(prs == null){
        //
        // File exists, but no bytes at offset. E.g., read of "hole"
        // return 1-byte result with value "0"
        //
        result = new BodyMsg(objId, offset, 1, acceptOfHole, 
                             holeDat, false);
        txn.commit();
        txn = null;
        return result;
      }  
      */
      assert(prs.getKeyRef().getOffset() <= offset);
      assert(prs.getKeyRef().getOffset() + prs.getDataRef().getLength() > offset);


      if (!exactOffset){ // we want to return the whole data that matches the 
        // per range state, i.e. the whole write record instead
        // of the partial record starting from the read offset.
        // therefore we need to return the data starting from 
        // the per-range-state::offset.
        offset = prs.getKeyRef().getOffset();
      }

      if(prs.getDataRef().isEmbargoed()){
	  throw new ReadOfEmbargoedWriteException();
      }

      if(prs.getDataRef().getValid() == false){
        txn.commit();
        txn = null;
        throw new ReadOfInvalidRangeException(prs.getDataRef().getAccept());
      }

      timingStats.startTimer("read-TX-getValidBytes");
      bytes = getValidBytes(txn, prs, offset, maxLen);
      timingStats.endTimer("read-TX-getValidBytes");


      assert(bytes.getLength() <= maxLen);
      result = new BodyMsg(objId, offset, bytes.getLength(), 
                           prs.getDataRef().getAccept(), bytes, false);
      txn.commit();
      txn = null;
      timingStats.endTimer("read-TX");
      return result;
    }
    finally{
      if(txn != null){
        txn.abort();
      }
    }
  }



 /** 
 *  readMeta() -- read the metadata for the specified offset of  
 *                 the specified file 
 *  
 *  returns 
 *     PreciseInval on success 
 *     throws NoSuchObjectException if objId does not exist (includes 
 *          case where delete of objId is more recent than any write 
 *          to objId) 
 *     throws EOFException if offset is larger than any write to 
 *          objId (since the last delete) 
 **/ 
  public PreciseInv
  readMeta(ObjId objId, long offset)
    throws NoSuchEntryException, EOFException,
	   ReadOfHoleException{
    try{ 
      //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP);
      return readMetaTransaction(objId, offset);
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
    }
    catch(IllegalArgumentException iae){
      assert(false);
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      assert false: "This should *never* happen in current design";
    }
    catch(DatabaseException dbe){
      assert(false);
    }
    assert(false); // Not reached
    return null;
  }



 /** 
 *  readMetaTransaction() -- do the work of readMeta 
 **/ 
  private PreciseInv
  readMetaTransaction(ObjId objId, long offset)
    throws NoSuchEntryException, EOFException,
    NullPointerException, DeadlockException, 
	   IllegalArgumentException, DatabaseException,
	   ReadOfHoleException{

    PerRangeState prs;
    PreciseInv pi;
    ObjInvalTarget t;
    Transaction txn = null;
    try{
      txn = dbEnv.beginTransaction(null, null);
      try{
        prs = getPerRangeState(txn, objId, offset);
      }
      catch(EOFException eof){ // Not-abnormal result of getPerRangeState
        assert(txn != null);
        txn.commit();
        txn = null;
        throw eof;
      }
      catch(NoSuchEntryException nse){ // Not-abnormal result of getPerRangeState
        assert(txn != null);
        txn.commit();
        txn = null;
        throw nse;
      }
      //added by zjd
      //so that we throw an ReadOfHoleException instead of return a one byte 0 value
      //for the hole
      catch(ReadOfHoleException rhe){ //read of hole 
	assert(txn != null);
        txn.commit();
        txn = null;
        throw rhe;
      }

      /* commentted by zjd -- see above ReadOfHoleException

      if(prs == null){ // File exist, but no byte at offset; e.g. read of "hole"
        t = new ObjInvalTarget(objId, offset, 1);
        pi = new PreciseInv(t, acceptOfHole);
      }
      
      else{*/
      t = new ObjInvalTarget(objId, 
			     prs.getKeyRef().getOffset(), 
			     prs.getDataRef().getLength());
      pi = new PreciseInv(t, prs.getDataRef().getAccept());
      
      txn.commit();
      txn = null;
      return pi;
    }
    finally{
      if(txn != null){
        txn.abort();
      }
    }
  }




 /** 
 *  applyInval() -- apply the specified invalidation (which 
 *                  may be bound) to the state 
 *  
 *  Side effect: updates currentVV 
 *  
 *  return the offet of the first newly-valid byte (if any) 
 *    or -1 (NO_NEW_VALID_BYES) if no new bytes became valid.  
 *    (Bytes can become valid if the inval is bound) 
 **/ 
  public long
  applyInval(PreciseInv pi){
    timingStats.startTimer("applyInval-total"); 
    long ret = -1;
    try{
      timingStats.startTimer("applyInval-GCForcer"); 
      //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP);
      timingStats.endTimer("applyInval-GCForcer"); 
      if(Env.EXPT_LOG){
        ObjId oid = pi.getObjId();
        boolean beforeStatus = isValid(oid);
        ret = applyInvalTransaction(pi);
        boolean afterStatus = isValid(oid);
        /*   if(beforeStatus && !afterStatus){
          Env.logValid("INV " + oid.getPath() 
                       + " " + System.currentTimeMillis()
                       +"\n");
                       } */
      }else{
        ret = applyInvalTransaction(pi);
      }
      
      timingStats.endTimer("applyInval-total"); 
      return ret;
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
    }
    catch(IllegalArgumentException iae){
      assert(false);
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      de.printStackTrace();
      assert false: "This should *never* happen in current design";
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      assert(false);
    }
    catch(IOException ie){
      ie.printStackTrace();
      assert false;
    }
    assert(false); // Not reached
    return -999;
  }

 /** 
 *  applyInvalTransaction() -- do the work of applyInval 
 *  
 *  Note: at one point we thought that we could do some sanity 
 *  checking by throwing a CausalOrderException if an invalidation 
 *  *did not* increase VV but *did* advance the acceptstamp 
 *  for its perRangeState. This turns out not to work. If 
 *  an interest set is imprecise, then we won't see its 
 *  invals until "late", and when they arrive, they may not 
 *  advance VV. So, we rely on the ISStatus information for 
 *  causality and we cannot enforce it in the RandomAccessState. 
 **/ 
  private long
  applyInvalTransaction(PreciseInv pi)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    
    
    timingStats.startTimer("invalTX-total"); 
    long newValid;
    Transaction txn = null;
    long fileNum;
    double priority = 0.0;
    BoundInval bi = null;

    int exceptType = 0;
    try{
      txn = dbEnv.beginTransaction(null, null);
      timingStats.startTimer("invalTX-updateVV");
      updateVV(txn, pi.getAcceptStamp());
      timingStats.endTimer("invalTX-updateVV");  
      priority = PRIORITY_NOT_APPLICABLE;
      if(pi.isBound()){
        priority = ((BoundInval)pi).getPriority();
      }
      try{
        timingStats.startTimer("invalTX-updateObjState");  
        fileNum = updatePerObjState(txn,
                                    pi.getObjId(),
                                    priority,
                                    pi.getAcceptStamp(),
                                    null);
        timingStats.endTimer("invalTX-updateObjState");  
        timingStats.startTimer("invalTX-updatePRS");
        newValid = updatePerRangeState(txn, pi, fileNum);
        timingStats.endTimer("invalTX-updatePRS");  
      }
      catch(ExistingDeleteIsNewerException edine){
        //
        // We've already seen a delete that is newer than this
        // write, so ignore this write
        //
        newValid = NO_NEW_VALID_BYTES;
      }
      txn.commit();
      txn = null;
      timingStats.endTimer("invalTX-total"); 
      return newValid;
    }
    catch(DeadlockException de){
      exceptType = 3;
      throw de;
    }
    catch(DatabaseException e){
      exceptType = 1;
       Env.sprinterrln("RandomAccessState::applyInvalTransaction exception: " + e.toString());
        e.printStackTrace();
        if(txn != null){
            txn.abort();
            txn = null;
            System.err.println("RandomAccessState::applyInvalTransaction aborted");
        }
        throw e;
    }
    catch(NullPointerException npe){
      exceptType = 2;
      throw npe;
    }
    
    catch(IllegalArgumentException iae){
      exceptType = 4;
      throw iae;
    }
    finally{
      if(txn != null){
        System.err.println("exceptType: " + exceptType + " before abort txn");
        txn.abort();
        System.err.println("RandomAccessState::applyInvalTransaction aborted");
      }
    }
  }

 /** 
 *  applyInvalTransaction(Transaction, PreciseInv) -- same as above method 
 *  except that the txn is an external parameter.  
 *  -- is used for transactional update checkpoint when checkpoint exchanges 
 **/ 
  private void
  applyInvalTransaction(Transaction txn, PreciseInv pi)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP);
    long fileNum = 0;
    double priority = 0.0;
      
    updateVV(txn, pi.getAcceptStamp());
    try{
      priority = PRIORITY_NOT_APPLICABLE;
      if(pi.isBound()){
        priority = ((BoundInval)pi).getPriority();
      }
      fileNum = updatePerObjState(txn,
                                  pi.getObjId(),
                                  priority,
                                  pi.getAcceptStamp(),
                                  null);
      updatePerRangeState(txn, pi, fileNum);
    }
    catch(ExistingDeleteIsNewerException edine){
      //
      // We've already seen a delete that is newer than this
      // write, so ignore this write
      //
      //newValid = NO_NEW_VALID_BYTES;
    }
    //return newValid;
  }
  
 /** 
 *  applyBody() -- apply the specified body to the state 
 *  
 *  Side effect: updates currentVV 
 *  
 *  return the offet of the first newly-valid byte (if any) 
 *    or -1 (NO_NEW_VALID_BYES) if no new bytes became valid.  
 *    (Bytes can become valid if the inval is bound) 
 *  
 **/ 
  public long
  applyBody(BodyMsg b){
    timingStats.startTimer("applyBody-total");
    long ret = -1;

    timingStats.startTimer("applyBody-GCForcer");
    //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP + b.getLength());
    timingStats.endTimer("applyBody-GCForcer");
   
    timingStats.startTimer("applyBody-convertToInval"); 
    ObjInvalTarget oit = b.getObjInvalTarget();
    BoundInval bi = new BoundInval(oit.getObjId(), oit.getOffset(), 
                                   oit.getLength(),
                                   b.getAcceptStamp(), b.getBody());
    timingStats.endTimer("applyBody-convertToInval");
    
    try{ 
      if(Env.EXPT_LOG){
        ObjId oid = oit.getObjId();
        boolean beforeStatus = isValid(oid);
        ret = applyInvalTransaction(bi); 
        boolean afterStatus = isValid(oid);
        /*
        if(!beforeStatus && afterStatus){
          Env.logValid("BODY " + oid.getPath() 
                       + " " + System.currentTimeMillis()
                       +"\n");
        
                       }*/
      }else{
        ret =  applyInvalTransaction(bi);
      }
      timingStats.endTimer("applyBody-total");
      return ret;
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
    }
    catch(IllegalArgumentException iae){
      iae.printStackTrace();
      assert(false);
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      de.printStackTrace();
      assert false: "This should *never* happen in current design";
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      assert(false);
    }
     catch(IOException ie){
      ie.printStackTrace();
      assert false;
    }
    assert(false); // Not reached
    return -999;
  }



 /** 
 *  delete() -- delete the specified objId (at the specified time) 
 *  
 *  Note that we have to keep records of old files around so  
 *  that if a delete at time <alpha, 2> arrives before a  
 *  write at time <beta, 1>, we realize that the write came 
 *  before the delete and we ignore it. (Conversely, if a  
 *  write at time <gamma, 3> arrives before the delete at time 
 *  <alpha, 2>, we keep gamma's write. 
 *  
 *  We do not, at present, garbage collect these "deleted file" 
 *  placeholders. In the future, one should be able to discard 
 *  such a placeholder when all elements of currentVV 
 *  exceed its acceptStamp (but note that a disconnected node 
 *  can prevent garbage collection for a long time; would 
 *  need to add commit sequence number if we want to guarantee 
 *  eventual garbage collection.). Amol has an idea: 
 *  require creates to generate a "create" record and 
 *  keep a per-object "lastCreateTime". If lastDeleteTime > 
 *  lastCreateTime, you can throw away all state for a file. 
 *  If a write arrives for a file w/o a record (e.g.,  
 *  a file for which either no create ever happened or 
 *  for which delete happened later than create) ignore 
 *  writes. (Also note that delete now beats invalidate 
 *  if it is newer than the last create even if it is older 
 *  than the invalidate.) 
 *  
 *  The postcondition is that perObjStateDB[objId]'s lastDeletedAccept 
 *  field is at least <acceptStamp> and no PerRange elements 
 *  whose accept stamps are less than <acceptStamp> remain in 
 *  perRangeState or perRangeBody. 
 *   
 *  side effect: updates currentVV 
 *  
 **/ 
  public void
  delete(ObjId objId, AcceptStamp accept){
    assert(objId != null);
    assert(accept != null);

    //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP);
    try{ 
      deleteTransaction(objId, accept);
      return;
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
    }
    catch(IllegalArgumentException iae){
      iae.printStackTrace();
      assert(false);
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      de.printStackTrace();
      assert false : "This should *never* happen in current design";
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      assert(false);
    }
  }


 /** 
 *  deleteTransaction() -- do a delete in a transaction 
 **/ 
  private void
  deleteTransaction(ObjId objId, AcceptStamp accept)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    Transaction txn = null;
    long fnum;
  
    if(!warnedNoGCDelete){
      warnedNoGCDelete = true;
      Env.tbd("We do not garbage collect old perObjState after delete"
              + " (see comment in RandomAccessState::delete())");
    }
    try{
      txn = dbEnv.beginTransaction(null, null);
      updateVV(txn, accept);
      try{
        fnum = updatePerObjState(txn,
                                 objId,
                                 PRIORITY_NOT_APPLICABLE,
                                 null,
                                 accept);
        updatePerRangeStateDelete(txn, fnum, accept);
      }
      catch(ExistingDeleteIsNewerException edine){
        //
        // We've already seen a delete newer than this one.
        // No new work to do
        //
      }
      txn.commit();
      txn = null;
      return;
    }
    finally{
      if(txn != null){
        txn.abort();
      }
    }
  }

  private static boolean warnedNoGCDelete = false;

 /** 
 *  updateVV() -- update the currentVV. 
 **/ 
  private void
  updateVV(Transaction txn, AcceptStamp accept)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    Cursor cursor = null;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    AcceptStamp prevAccept;
    NodeId nodeId;
    OperationStatus status;
    long prevAcceptLocal;
    try{
      cursor = vvDB.openCursor(txn, null);
      nodeId = accept.getNodeId();
      assert(nodeId.isValidAndConcrete());
      nodeIdBinding.objectToEntry(nodeId, key);
      status = cursor.getSearchKey(key, data, null);
      if(status == OperationStatus.SUCCESS){
        prevAcceptLocal = ((Long)longBinding.entryToObject(data)).longValue();
        prevAccept = new AcceptStamp(prevAcceptLocal, nodeId);
        if(!prevAccept.gt(accept)){
          longBinding.objectToEntry(new Long(accept.getLocalClock()), 
                                    data);
        
          cursor.put(key, data);
          if(testTxnPerformance){
            System.out.println("updateVV: update vvDB with key.size=" +
                               key.getSize() + " data.size=" + data.getSize());
          }
        }
        return;
      }
      else{
        assert(status == OperationStatus.NOTFOUND);
        longBinding.objectToEntry(new Long(accept.getLocalClock()), 
                                  data);
        
        cursor.put(key, data);
        if(testTxnPerformance){
          System.out.println("updateVV: update vvDB with key.size=" +
                             key.getSize() + " data.size=" + data.getSize());
        }
        return;
      }
    }
    finally{
      if(cursor != null){
        try{
          cursor.close();
          
        }catch(DatabaseException e){
          System.err.println("close cursor in updateVV failed: " + e);
          throw e;
        }
      }
    }
  }




 /** 
 *  getVV() -- return the current VV 
 **/ 
  public VV
  getVV(){
    Transaction txn = null;
    //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP);
    try{
      return getVVTransaction();
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
      return null;
    }
    catch(IllegalArgumentException iae){
      iae.printStackTrace();
      assert(false);
      return null;
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      de.printStackTrace();
      assert false : "This should *never* happen in current design";
      return null;
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      assert(false);
      return null;
    }
  }


 /** 
 *  getVVTransaction() -- make a transaction and return the current VV 
 **/ 
  private VV
  getVVTransaction()
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    Transaction txn = null;
    try{
      txn = dbEnv.beginTransaction(null, null);
      return getVV(txn);
    }
    finally{
      if(txn != null){
        txn.abort();
      }
    }
  }

 /** 
 *  getVV() -- current VV from within a transaction 
 **/ 
  protected VV
  getVV(Transaction txn)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    Cursor cursor = null;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;
    NodeId id;
    Long localClock;
    CounterVV vv = new CounterVV();

    try{
      cursor = vvDB.openCursor(txn, null);
      status = cursor.getFirst(key, data, null);
      while(status == OperationStatus.SUCCESS){
        id = (NodeId)nodeIdBinding.entryToObject(key);
        localClock = (Long)longBinding.entryToObject(data);
        vv.setStampByServer(id, localClock.longValue());
        status = cursor.getNext(key, data, null);
      }
      return vv;
    }
    finally{
      if(cursor != null){
        cursor.close();
      }
    }

  }

 /** 
 *  getVV() -- return the current VV for specified node 
 **/ 
  public long
  getVV(NodeId id){
    Transaction txn = null;
    //gcForcer.payCharge(GC_FORCER_CHARGE_PER_OP);
    try{
      return getVVTransaction(id);
    }
    catch(NullPointerException npe){
      npe.printStackTrace();
      assert(false);
      return -999;
    }
    catch(IllegalArgumentException iae){
      iae.printStackTrace();
      assert(false);
      return -999;
    }
    catch(DeadlockException de){
      //
      // This should *never* happen since in current design
      // there is no concurrency!
      //
      de.printStackTrace();
      assert false : "This should *never* happen in current design";
      return -999;
    }
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      assert(false);
      return -999;
    }
  }


 /** 
 *  getVVTransaction() -- make a transaction and return the current  
 *    local clock for a specified node 
 **/ 
  protected long
  getVVTransaction(NodeId node)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    Transaction txn = null;
    try{
      txn = dbEnv.beginTransaction(null, null);
      return getVV(txn, node);
    }
    finally{
      if(txn != null){
        txn.abort();
      }
    }
  }


 /** 
 *  getVV() -- return the localClock element of VV for the specified  
 *  nodeID or AcceptStamp.BEFORE_TIME_BEGAN if no such element stored. 
 **/ 
  protected long
  getVV(Transaction txn, NodeId nodeId)
    throws  NullPointerException, DeadlockException, IllegalArgumentException, DatabaseException{
    long prevAcceptLocal;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;

    assert(nodeId.isValidAndConcrete());
    nodeIdBinding.objectToEntry(nodeId, key);
    status = vvDB.get(txn, key, data, null);
    if(status == OperationStatus.SUCCESS){
      prevAcceptLocal = ((Long)longBinding.entryToObject(data)).longValue();
      return prevAcceptLocal;
    }
    return AcceptStamp.BEFORE_TIME_BEGAN;
  }



 /** 
 *  scanForUpdates() -- scan through all elements in the store 
 *    and insert any element that (a) intersects is and (b) is newer 
 *    than startVV into upq 
 *  
 **/ 
  public void
  scanForUpdates(UpdatePriorityQueue upq, SubscriptionSet ss, VV startVV)
    throws IOException{

    Env.tbd("RandomAccessState::scanForUpdates (called by DataStore::subscribeUpdates)"
            + " functionality really should be done at 'application level' when "
            + " subscribing to a new subdirectory, if you want to see the old"
            + " updates, scan through them and prefetch them. The problem with"
            + " doing this in scanForUpdates is that work is O(# obj in entire system)"
            + " rather than O(# obj in subdirectory of interest)");

    long lastFNum = SENTINAL_SMALL_FILE_NUM;
    int origPriority;

    Thread myThread = Thread.currentThread();
    origPriority = myThread.getPriority();
    myThread.setPriority(Thread.MIN_PRIORITY);

    scanForUpdates_Internal(upq, ss, startVV);

    myThread.setPriority(origPriority);
  }

 /** 
 *  scanForUpdates_Internal 
 *    finds objects with intersect ss. 
 *    For each such obj, looks for offset which are newer or equal to 
 *    startVV 
 *    inserts the meta data into upq 
 **/ 
  private void
  scanForUpdates_Internal(UpdatePriorityQueue upq, SubscriptionSet ss, VV startVV)
  throws IOException{
    
    boolean dbgThis = false;

    Cursor posCursor = null;
    Cursor prsCursor = null;
    DatabaseEntry posKey = new DatabaseEntry();
    DatabaseEntry posData = new DatabaseEntry();

    if(dbgThis) {
      Env.sprintln("RAS: scanForUpdate called with ss=" + ss + " startVV=" + startVV);
    }
    Env.performanceWarning("Because we don't have a way to tell RAS to " +
                           "get information about all objects from a " +
                           "given directory, we iterate through all " +
                           "objects and send checkpoint data for an " +
                           "object if it happens to lie in the requested " +
                           "directory.");

    try{ 
      String searchKeyStr = null;
      boolean matchPrefix = false;      
      posCursor = perObjStateDB.openCursor(null, null);
      prsCursor = perRangeStateDB.openCursor(null, null);
   
     // 
      //  Splits the SS into individual items
      // 
      String[] ssItem = ss.toString().split(":");
      
      for(int i = 0; i < ssItem.length; i++){
        ssItem[i].trim();
        if(ssItem[i].endsWith("/*")){
          matchPrefix = true;
          //get rid of *
          searchKeyStr = ssItem[i].substring(0, ssItem[i].length()-1); 
        } else {
          matchPrefix = false;
          searchKeyStr = ssItem[i];
        }

        assert searchKeyStr != null;
        
        // look up item in the POS
        if(dbgThis) {
          Env.sprintln("RAS: scanForUpdate:: looking for " + searchKeyStr +
                       "in POS");
        }

        try{
	  posKey = new DatabaseEntry(searchKeyStr.getBytes("UTF-8"));
	}catch(UnsupportedEncodingException e){
	  assert false;
	}

  	OperationStatus status;
	if (matchPrefix){
	  status = posCursor.getSearchKeyRange(posKey, posData, null);
	}else{
	  status = posCursor.getSearchKey(posKey, posData, null);
	}

        // iterate thro each match in the database
	while(status == OperationStatus.SUCCESS){
	  ObjId objId = new ObjId(new String(posKey.getData()));

          if(!matchPrefix){
	    assert objId.getPath().equals(searchKeyStr);
          } else {
             if(!objId.getPath().startsWith(searchKeyStr)){
               if(dbgThis) {
                 Env.sprintln("RAS: scanForUpdate:: found " + objId + 
                   " going to stop looking");
               }
	      break;//stop here: this objId does not contains a prefix of the searchKeyStr
              //and don't need to check next record for perObjState
             }
          }
          

          //
          // Hurray! found a match! Now to check if lastWrite is newer than or equal to startVV
          //  and it has not been deleted (i.e. lastWrite newer than lastDelete)
          //

          PerObjData pod = null;	  
	  pod = (PerObjData)perObjDataBinding.entryToObject(posData);
          AcceptStamp highestWrite = pod.getHighestWriteAccept();
          AcceptStamp highestDelete = pod.getHighestDeleteAccept();
          
          if(dbgThis) {
            Env.sprintln("RAS: scanForUpdate:: found match objId=" + objId +
              " highestWrite=" + highestWrite + " highestDelete=" + highestDelete);
          }

          if(!startVV.includes(highestWrite) &&
             highestWrite.gt(highestDelete)){
            // now to look thro the PerRangeState and insert metabody into upq
            long fileNum = pod.getFileNum();
            scanPRS_Internal(upq, objId, fileNum, startVV, prsCursor);
          }
          // go to the next match
          if(matchPrefix) {
            status = posCursor.getNext(posKey, posData, null);
          } else { // only one match
            break;
          }
        } //while
      } //for
    }catch (DatabaseException de){
      String msg = de.toString();
        //close cursor
        try{
          if(posCursor!= null) {
            posCursor.close();
            posCursor=null;
          }
          if(prsCursor!= null) {
            prsCursor.close();
            prsCursor=null;
          }

        }catch(Exception e){
          msg = msg +"\nAlso: nested Exception" + e.toString();   
          throw new IOException(msg);
        }
    }
    try{
      //close cursor   
      if(posCursor!= null) {
        posCursor.close();
        posCursor=null;
      }
      if(prsCursor!= null) {
        prsCursor.close();
        prsCursor=null;
      }
    }catch(Exception e){
      throw new IOException(e.toString());
    }
    if(dbgThis) {
      Env.sprintln("RAS: scanForUpdate returns");
    }
  }
  

  //-------------------------------------------------------------------
  // scanPRS_Internal
  //   for each fnum /objId, looks throu the PRS records, 
  //   and inserts meta data into the upq if it is newer or equal to  
  //   startVV
  //------------------------------------------------------------------- 

private void
scanPRS_Internal(UpdatePriorityQueue upq, 
                 ObjId objId, 
                 long fileNum , 
                 VV startVV, 
                 Cursor prsCursor) 
    throws DatabaseException{
  boolean dbgThis = false;

  DatabaseEntry prsKey = new DatabaseEntry();
  DatabaseEntry prsData = new DatabaseEntry();
  PerRangeKey prk;
  PerRangeData prd;
  OperationStatus prsStatus = null;
  // 
  // Note: we have set the priority to defaul.  Should it be higher? or lower?
  //
  double priority = Core.DEFAULT_PRIORITY;

  if(dbgThis) {
    Env.sprintln("RAS: scanPRS called for objId= " + objId + 
                 " fNum=" + fileNum + " startVV=" + startVV);
  }

  prk = new PerRangeKey(fileNum, 0);
  perRangeKeyBinding.objectToEntry(prk, prsKey);
  prsStatus = prsCursor.getSearchKeyRange(prsKey, prsData, null);

  // iterate thro' PRS DB for matches
  while(prsStatus == OperationStatus.SUCCESS){
    prk = (PerRangeKey)perRangeKeyBinding.entryToObject(prsKey);

    if(prk.getFNum() != fileNum) {
      if(dbgThis) {
        Env.sprintln("RAS: scanPRS:: no match.  stop looking");
      }
      // no more records
      break;
    } else {
      // found a match, check vv and add it to upq
      prd = (PerRangeData) perRangeDataBinding.entryToObject(prsData);
      if(dbgThis) {
        Env.sprintln("RAS: scanPRS:: found match " + prd);
      }
      if(!startVV.includes(prd.getAccept())) {
        if(dbgThis) {
          Env.sprintln("RAS: scanPRS:: inserting it into upq");
        }
        upq.insert(priority, objId, prk.getOffset(), prd.getLength());
      }
    }
    prsStatus = prsCursor.getNext(prsKey, prsData, null);
  }
}
  


//-------------------------------------------------------------------
// updatePerObjState() -- update lastWriteAccept 
//    and/or lastDeleteAccept for specified
//    object. if necessary, create a new perObjState record 
//    in perObjStateDB.
//
// Postcondition:
//    a record for ObjId exists in perObjStateDB
//    with writeAccept != null --> lastWriteAccept >= writeAccept
//    and deleteAccept != null --> lastDeleteAccept >= deleteAccept
//
// Return:
//    FileNum for objId
//
// Throws:
//    If we have already seen a delete that is newer than writeAccept
//    and at least as new as deleteAccept, throw ExistingDeleteIsNewer
//    exception, indicating that the caller need not update any
//    perRangeState
//
// Note that if we see a write with a lower accept stamp than
// some write we have already seen, the caller still may need to
// update perRangeState because the "earlier" write may update
// a different range of bytes than the "later" write.
//-------------------------------------------------------------------
  private long
  updatePerObjState(Transaction txn,
                    ObjId id,
                    double priority,
                    AcceptStamp writeAccept,
                    AcceptStamp deleteAccept)
    throws  ExistingDeleteIsNewerException,
    NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    Cursor cursor = null;			       
    PerObjData pod;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;
    boolean dirty = false;
    AcceptStamp origWriteAccept, origDeleteAccept;

    if(priority != PRIORITY_NOT_APPLICABLE){
      assert(writeAccept != null);
    }
    if(writeAccept == null){
      writeAccept = ACCEPT_BEGINNING_OF_TIME;
    }else{
      assert(writeAccept.gt(ACCEPT_BEGINNING_OF_TIME));
    }
    if(deleteAccept == null){
      deleteAccept = ACCEPT_BEGINNING_OF_TIME;
    }else{
      assert(deleteAccept.gt(ACCEPT_BEGINNING_OF_TIME));
    }

    try{
      cursor = perObjStateDB.openCursor(txn, null);
      assert id != null;
      //objIdBinding.objectToEntry(id, key);
      try{
	key = new DatabaseEntry(id.getPath().getBytes("UTF-8"));
      }
      catch(UnsupportedEncodingException uee){
	assert(false);
      }
      status = cursor.getSearchKey(key, data, null);
      if(status == OperationStatus.NOTFOUND){
        pod = new PerObjData(makeNewFNum(),
                             priority,
                             writeAccept,
                             deleteAccept);
        origWriteAccept = ACCEPT_BEGINNING_OF_TIME;
        origDeleteAccept = ACCEPT_BEGINNING_OF_TIME;
        dirty = true;
      }
      else{
        assert(status == OperationStatus.SUCCESS);
        pod = (PerObjData)perObjDataBinding.entryToObject(data);
        origWriteAccept = pod.getHighestWriteAccept();
        origDeleteAccept = pod.getHighestDeleteAccept();

        //
        // Having a write and delete with same accept could cause
        // inconsistency if different machines receive them 
        // in different order. (Plus, this shouldn't happen anyhow.)
        //
        Env.remoteAssert(deleteAccept.eq(ACCEPT_BEGINNING_OF_TIME)
                         ||
                         !pod.getHighestWriteAccept().eq(deleteAccept));
        Env.remoteAssert(writeAccept.eq(ACCEPT_BEGINNING_OF_TIME)
                         ||
                         !pod.getHighestDeleteAccept().eq(writeAccept));


        if(pod.getHighestWriteAccept().lt(writeAccept)){
          pod.advanceHighestWriteAccept(writeAccept);
          dirty = true;
        }
        if(pod.getHighestDeleteAccept().lt(deleteAccept)){
          pod.advanceHighestDeleteAccept(deleteAccept);
          dirty = true;
        }
      }
      if(dirty){
        perObjDataBinding.objectToEntry(pod, data);
        status = cursor.put(key, data);

        if(testTxnPerformance){
          System.out.println("updatePerObjStateDB: with key.size=" +
                             key.getSize() + " data.size=" + data.getSize());
        }
        assert(status == OperationStatus.SUCCESS);
      }

      //
      // If we have already seen a delete that is newer than writeAccept
      // and at least as new as deleteAccept, throw ExistingDeleteIsNewer
      // exception, indicating that the caller need not update any
      // perRangeState
      //
      if(writeAccept.lt(origDeleteAccept)
         &&
         deleteAccept.leq(origDeleteAccept)){
        assert(dirty || !dirty); 
        if(dirty){
          //
          // A write that is older than the newest known
          // delete can advance newest known write but
          // still not affect any of the per range state
          //
          assert(!writeAccept.eq(ACCEPT_BEGINNING_OF_TIME));
          assert(writeAccept.lt(origDeleteAccept));
        }
      
        throw new ExistingDeleteIsNewerException();
      }

      return pod.getFileNum();
    }
    finally{
      if(cursor != null){
        try{
          cursor.close();
        }catch(DatabaseException e){
          System.err.println("updatePerObjState cursor close failed: " + e);
          throw e;
        }
      }
    }
  }



//-------------------------------------------------------------------
// makeNewFNum -- claim an unclaimed fnum
//
// The invarient is that at most one objId maps to a given fnum.
// We do the simplest thing -- fnums come from a 63-bit space
// and we just keep a counter of the highest id claimed so far.
// We assume that we will never run out (at a creation rate of
// 1M local file IDs per second, the id space will last 2^43
// seconds or about 2^18 years. So, if you are reading this and
// running out of bits, relay my apologies to King `xkr$im of the
// Great Cockroach Empire or whoever else is running the show
// these days.)
//
// Claiming an fnum done from within its own transaction -- so, it is 
// possible that the caller is part of a transaction that later aborts
// but the  "claim" of the FNUM is not aborted in that case. That's OK -- 
// that fnum just remains unused. We do this to avoid having to
// serialize file creates on this element.
//-------------------------------------------------------------------
  protected long
  makeNewFNum()
    throws DatabaseException{
    long fnum;
    Transaction txn = null;
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;


    try{
      txn = dbEnv.beginTransaction(null, null);
      status = globalStateDB.get(txn, fnumKey, data, null);
      if(status == OperationStatus.NOTFOUND){
        fnum = SENTINAL_SMALL_FILE_NUM;
      }
      else{
        fnum = ((Long)longBinding.entryToObject(data)).longValue();
      }
      fnum++;
      assert(fnum < SENTINAL_LARGE_FILE_NUM); // Sorry, King `xkr$im!
      longBinding.objectToEntry(new Long(fnum), data);
      status = globalStateDB.put(txn, fnumKey, data);
      txn.commit();
      txn = null;
      return fnum;
    }
    finally{
      if(txn != null){
        txn.abort();
      }
    }
  }


//-------------------------------------------------------------------
// dbgMakeNewFNum() -- interface accessible for debugging only.
//-------------------------------------------------------------------
  protected long
  dbgMakeNewFNum()
    throws DatabaseException{
    return makeNewFNum();
  }


//-------------------------------------------------------------------
// updatePerRangeStateDelete() -- update the perRange state to reflect
//     a delete at the specified time
//
// Remove all perRangeState records and perRangeBody records
// for the specified fnum whose accept stamp is smaller than
// deleteAccept
//-------------------------------------------------------------------
  private void 
  updatePerRangeStateDelete(Transaction txn, long fnum, 
                            AcceptStamp deleteAccept)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
  
    OperationStatus status;
    Cursor prsCursor = null;
    Cursor prdCursor = null;
    PerRangeKey prkMeta;
    PerRangeKey prkBody;
    PerRangeData prdMeta;
    DatabaseEntry keyMeta = new DatabaseEntry();
    DatabaseEntry keyBody = new DatabaseEntry();
    DatabaseEntry dataMeta = new DatabaseEntry();
    DatabaseEntry dataBody = new DatabaseEntry();
    boolean deleteWins;

    if(!warnedDeletePerf){
      Env.performanceWarning("RandomAccessState::delete does O(size of file)"
                             + " work since I don't know how to access all"
                             + " of the perRangeData records w/o reading"
                             + " them all");
      warnedDeletePerf = true;
    }
    prkMeta = new PerRangeKey(fnum, 0);
    prkBody = new PerRangeKey(fnum, 0);
    try{
      prsCursor = perRangeStateDB.openCursor(txn, null);
      prdCursor = perRangeBodyDB.openCursor(txn, null);
    
      //
      // Unlike arbitrary offset searches, getSearchKeyRange
      // for offset 0 guaranteed to return first element of
      // fnum (if any);
      //
      perRangeKeyBinding.objectToEntry(prkMeta, keyMeta);
      status = prsCursor.getSearchKeyRange(keyMeta, dataMeta, null);
      assert(status == OperationStatus.SUCCESS);
      prkMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);
      prdMeta = (PerRangeData)perRangeDataBinding.entryToObject(dataMeta);
      perRangeKeyBinding.objectToEntry(prkBody, keyBody);
      status = prdCursor.getSearchKeyRange(keyBody, dataBody, null);
      assert(status == OperationStatus.SUCCESS);
      prkBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);

      //
      // Loop invarients: at the start of each pass, prkMeta,
      // keyMeta, and keyData refer to the object under
      // the prsCursor; and the prdCursor refers to the 
      // first valid range of bytes at or beyond prkMeta
      //
      while(prkMeta.getFNum() <= fnum){
        Env.remoteAssert(!prdMeta.getAccept().eq(deleteAccept));
        if(prdMeta.getAccept().lt(deleteAccept)){
          deleteWins = true;
        }
        else{
          deleteWins = false;
        }

        if(prdMeta.getValid()){
          assert(prkBody.getFNum() == prkMeta.getFNum());
          assert(prkBody.getOffset() == prkMeta.getOffset());
          if(deleteWins){
            status = prdCursor.delete();
            assert(status == OperationStatus.SUCCESS);
          }
          status = prdCursor.getNext(keyBody, dataBody, null);
          assert(status == OperationStatus.SUCCESS);
          prkBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);
        }
        if(deleteWins){
          status = prsCursor.delete();
          assert(status == OperationStatus.SUCCESS);
        }
        status = prsCursor.getNext(keyMeta, dataMeta, null);
        assert(status == OperationStatus.SUCCESS);
        prkMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);
        prdMeta = (PerRangeData)perRangeDataBinding.entryToObject(dataMeta);
      }
    }
    finally{
      if(prsCursor != null){
        prsCursor.close();
      }
      if(prdCursor != null){
        prdCursor.close();
      }
    }


  }


  private static boolean warnedDeletePerf = false;





//-------------------------------------------------------------------
// updatePerRangeState() -- update the perRange state to reflect
//     the specified precise inval
//
// Return the offset of the first byte whose metadata is updated (or
// NO_NEW_VALID_BYTES if none).
//-------------------------------------------------------------------
  private long
  updatePerRangeState(Transaction txn,
                      PreciseInv pi,
                      long fnum)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    long currentOffset;
    long validOffset = NO_NEW_VALID_BYTES;
    long retOffset;
    ClearChunk clobberRet;
    Cursor prsCursor = null;
    Cursor prdCursor = null;
    
    DatabaseEntry keyMeta = new DatabaseEntry();
    DatabaseEntry dataMeta = new DatabaseEntry();
    ObjInvalTarget oit;

    oit = pi.getObjInvalTarget();
    currentOffset = oit.getOffset();

  
    try{
      timingStats.startTimer("updatePRS-setCursor");
      prsCursor = perRangeStateDB.openCursor(txn, null);
      prdCursor = perRangeBodyDB.openCursor(txn, null);

      setCursorsBeforeOrAt(prsCursor, prdCursor, 
                           fnum, currentOffset, 
                           keyMeta, dataMeta);
      timingStats.endTimer("updatePRS-setCursor");
    
      do{
        timingStats.startTimer("updatePRS-clearChunk");
        clobberRet = clearNextChunkToInsert(pi, fnum, currentOffset, 
                                            prsCursor, keyMeta, dataMeta);
        timingStats.endTimer("updatePRS-clearChunk");
        if(clobberRet.getDoApplyMetadata()){
          timingStats.startTimer("updatePRS-insertMetaData");
          insertMetadataChunkNoOverlap(pi, fnum, clobberRet, prsCursor,
                                       keyMeta, dataMeta);
          prsCursor.getNext(keyMeta, dataMeta, null);
          timingStats.endTimer("updatePRS-insertMetaData");
          if(!pi.isBound()){
            // pre: no element before prdCursor overlaps
            timingStats.startTimer("updatePRS-removePRBody");
            removeOverlappingPerRangeBody(prdCursor, fnum, 
                                          clobberRet.getOffset(),
                                          clobberRet.getLength());
            timingStats.endTimer("updatePRS-removePRBody");
            // post: prdCursor refers to element after range
          }
        }
        if(clobberRet.getDoApplyBody()){
          assert(pi.isBound());
          // pre: no element before prdCursor overlaps
          timingStats.startTimer("updatePRS-updatePRBody");
          retOffset = updatePerRangeBody(prdCursor, (BoundInval)pi, fnum, 
                                         clobberRet.getOffset(), 
                                         clobberRet.getLength());
          timingStats.endTimer("updatePRS-updatePRBody");
          // post: prdCursor refers to just inserted element
          if(validOffset == NO_NEW_VALID_BYTES 
             && retOffset != NO_NEW_VALID_BYTES){
            validOffset = retOffset;
          }
        }
        timingStats.startTimer("updatePRS-checkCursor");
        currentOffset = clobberRet.getNextOffset();
        checkCursorAtOrPastOffset(prsCursor, fnum, currentOffset);
        timingStats.endTimer("updatePRS-checkCursor");
      }while(currentOffset < pi.getOffset() + pi.getLength());

      if(doExpensiveSanityChecks){
        checkCoveredAndNoOverlaps(txn, pi, fnum);
        checkPRSPRDMatch(txn, pi, fnum);
      } 

      return validOffset;
    }catch(Exception e){
      e.printStackTrace();
      assert false;
      //throw e;
      return -1;
    }
    finally{
      if(prsCursor != null){
        try{
          prsCursor.close();
        }catch(DatabaseException e){
          System.err.println("updatePerRangeState: prsCursor close failed: "
                             + e);
          throw e;
        }
      }
      if(prdCursor != null){
        try{
          prdCursor.close();
        }catch(DatabaseException e){
          System.err.println("updatePerRangeState: prdCursor close failed: "
                             + e);
          throw e;
        }
      }
    }
  }


//-------------------------------------------------------------------
// setCursorsBeforeOrAt() -- set the metadata and data cursors
//   s.t. they refer to the item before the first item that
//   begins after <fnum, currentOffset> (e.g., the 
//   cursors will point to the last item that begins before
//   or at the specified location.) On exit, keyMeta and dataMeta
//   contain the item under the prsCursor.
//
// This method includes a bunch of extra sanity check code.
//-------------------------------------------------------------------
  private void
  setCursorsBeforeOrAt(Cursor prsCursor, Cursor prdCursor, 
                       long fnum, long currentOffset, 
                       DatabaseEntry keyMeta, DatabaseEntry dataMeta)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    PerRangeKey prk;
    DatabaseEntry keyBody = new DatabaseEntry();
    DatabaseEntry dataBody = new DatabaseEntry();
    PerRangeKey checkPRKMeta, checkPRKBody;
    PerRangeData checkPRD;
    OperationStatus status;
                       
    prk = new PerRangeKey(fnum, currentOffset);

    //
    // Cursor begins at last element that starts *before* totalOffset
    //
    perRangeKeyBinding.objectToEntry(prk, keyMeta);
    perRangeKeyBinding.objectToEntry(prk, keyBody);
    status = prsCursor.getSearchKeyRange(keyMeta, dataMeta, null);
    assert(status == OperationStatus.SUCCESS);
    checkPRKMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);

    //
    // Check: GetSearchKeyRange returns key >= search key
    //
    assert(checkPRKMeta.getFNum() >= fnum);
    assert(checkPRKMeta.getFNum() > fnum 
           || 
           checkPRKMeta.getFNum() == fnum && checkPRKMeta.getOffset() >= currentOffset);
    status = prdCursor.getSearchKeyRange(keyBody, dataBody, null);
    assert(status == OperationStatus.SUCCESS);

    //
    // Check: GetSearchKeyRange returns key >= search key
    //
    checkPRKBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);
    assert(checkPRKBody.getFNum() >= fnum);
    assert(checkPRKBody.getFNum() > fnum 
           || 
           checkPRKBody.getFNum() == fnum && checkPRKBody.getOffset() >= currentOffset);
    status = prsCursor.getPrev(keyMeta, dataMeta, null);
    assert(status == OperationStatus.SUCCESS);
    checkPRKMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);
    assert(checkPRKMeta.getFNum() <= fnum);
    assert(checkPRKMeta.getFNum() < fnum 
           || 
           checkPRKMeta.getFNum() == fnum && checkPRKMeta.getOffset() < currentOffset);
    status = prdCursor.getPrev(keyBody, dataBody, null);
    assert(status == OperationStatus.SUCCESS);
    checkPRKBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);
    assert(checkPRKBody.getFNum() <= fnum);
    assert(checkPRKBody.getFNum() < fnum 
           || 
           checkPRKBody.getFNum() == fnum && checkPRKBody.getOffset() < currentOffset);

    if(checkPRKBody.getOffset() != checkPRKMeta.getOffset()){
      checkPRD = (PerRangeData)perRangeDataBinding.entryToObject(dataMeta);
      assert(checkPRD.getValid() == false);
    }
  
  }

//---------------------------------------------------------------------------
// insertMetadataChunkNoOverlap() -- insert a metadata chunk record
//    at the cursor. Precondition is that clearNextChunkToInsert
//    has cleared out this part of DB, so there must not be
//    an overlap with any existing record.
//
// Postcondition -- the cursor is on the newly inserted item.
//---------------------------------------------------------------------------
  private void
  insertMetadataChunkNoOverlap(PreciseInv pi, long fnum, 
                               ClearChunk chunk, 
                               Cursor cursor, DatabaseEntry key, 
                               DatabaseEntry data)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    OperationStatus status;
    PerRangeKey prk = new PerRangeKey(fnum, chunk.getOffset());
    PerRangeData prd = new PerRangeData(chunk.getLength(),
                                        pi.getAcceptStamp(),
                                        chunk.getPrevAccept(),
                                        pi.isBound() ? true : false,
					pi.isEmbargoed(),
					pi.isCommitted());
    perRangeKeyBinding.objectToEntry(prk, key);
    perRangeDataBinding.objectToEntry(prd, data);
    status = cursor.put(key, data);
    if(testTxnPerformance){
      System.out.println("updatePerRangeState: with key.size=" +
                         key.getSize() + " data.size=" + data.getSize());
    }
    assert(status == OperationStatus.SUCCESS);

    if(doExpensiveSanityChecks){
      status = cursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      PerRangeKey checkPerRangeKey;
      checkPerRangeKey = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      assert(checkPerRangeKey.getFNum() > fnum
             ||
             (checkPerRangeKey.getFNum() == fnum
              &&
              checkPerRangeKey.getOffset() >= prk.getOffset() + prd.getLength()));
      status = cursor.getPrev(key, data, null); // Reestablish exit condition
      assert(status == OperationStatus.SUCCESS);
    }
  }




//---------------------------------------------------------------------------
// clearNextChunkToInsert() -- when inserting a new inval, this call 
//    clears out the next chunk of existing space so that we can do
//    an insert w/o duplicating any range. 
//
// The returned record indicates the offset and length cleared for the
// next insert. If current chunk from <currentOffset> is at least
// as new as pi, then set doApplyMetdata to false in return; caller
// will skip over specified range rather than insert older item.
//
// Cursor management:
//  Invarient is that no item before the cursor could overlap the 
//  remaining inserts. (So, on entry, the cursor could be pointing
//  to the next overlapped item or it could be pointing to a prior
//  item that doesn't overlap the remainder being inserted.)
//  On exit, the cursor points to the next item that could overlap
//  what is being inserted.
//---------------------------------------------------------------------------
  private ClearChunk 
  clearNextChunkToInsert(PreciseInv pi, long fnum, long currentOffset, 
                         Cursor cursor, DatabaseEntry key,
                         DatabaseEntry data)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    OperationStatus status;
    PerRangeKey prk;
    PerRangeData prd;
    DWRet dwret;
    ClearChunk ret;
    long len;



    assert(currentOffset < pi.getOffset() + pi.getLength());

    prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
    prd = (PerRangeData)perRangeDataBinding.entryToObject(data);


    //
    // Cursor is before currentOffset and does not overlap
    // --> advance it. Note -- never should have to advance 
    // by more than one slot.
    //
    if(prk.getFNum() < fnum
       ||
       (prk.getFNum() == fnum 
        && prk.getOffset() + prd.getLength() <= currentOffset)){
      status = cursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
    }
    assert(prk.getFNum() > fnum 
           ||
           (prk.getFNum() == fnum
            && 
            prk.getOffset() + prd.getLength() > currentOffset));


    //
    // Decide dataToUpdate, metadataWins, and prevAccept values
    //
    dwret = decideWinners(pi, prk, prd, fnum, currentOffset);


    //
    // Decide len of next chunk to insert. And, if this chunk wins and 
    // overlaps something that currently exists, remove the overlapping 
    // part. Finally, advance cursor (if necessary) so that it refers
    // to the item that will be *after* the chunk to be inserted.
    // 
    len = setLengthRemoveOverlapAdvanceCursor(cursor, pi, prk, prd, 
                                              fnum, currentOffset, 
                                              dwret.metadataWins, dwret.dataToUpdate,
                                              dwret.prevAccept);
                                            


    ret = new ClearChunk(dwret.metadataWins, 
                         dwret.dataToUpdate, 
                         currentOffset, 
                         len,
                         dwret.prevAccept);

    //
    // Update key, data to establish postcondition 
    // -- item under cursor and stored in key, data
    // is past the chunk we are returning info on.
    //
    status = cursor.getCurrent(key, data, null);
    assert(status == OperationStatus.SUCCESS);
    prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
    prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
    assert(prk.getFNum() >= fnum);
    assert(prk.getFNum() > fnum
           ||
           (prk.getFNum() == fnum
            && 
            (prk.getOffset() >= ret.getOffset() + ret.getLength())));

    return ret;
  } 


//---------------------------------------------------------------------------
// DWRet -- helper class for returning multiple values from decideWinners
//---------------------------------------------------------------------------
  static class 
  DWRet{
    boolean metadataWins;
    boolean dataToUpdate;
    AcceptStamp prevAccept;
  }

//---------------------------------------------------------------------------
// decideWinners() -- decide metadatWins, dataToUpdate, and prevAccept values
//   as helper frunction to clearNextChunkToInsert()
//
// Caller should apply metadata if pi starts before 
// current or if pi is newer 
// OR 
// if pi and existing are same update, but existing
// is invalid and pi is bound 
// OR
// if pi and existing are same update, but existing
// is embargoed and pi is not embargoed 
//
// Caller should apply body if metadataWins or if
// pi and current have same accept stamp and current
// is not valid
//---------------------------------------------------------------------------
  private DWRet
  decideWinners(PreciseInv pi, PerRangeKey prk, PerRangeData prd, 
                long fnum, long currentOffset){
    boolean metadataWins;
    AcceptStamp prevAccept;
    boolean dataToUpdate;
    DWRet ret = new DWRet();

    if(pi.getAcceptStamp().gt(prd.getAccept())
       ||
       fnum < prk.getFNum()
       ||
       currentOffset < prk.getOffset()){
      metadataWins = true;
      if(fnum < prk.getFNum() || currentOffset < prk.getOffset()){
        prevAccept = PREV_ACCEPT_BEGINNING_OF_TIME;
      }
      else{
        prevAccept = (AcceptStamp)prd.getAccept().clone();
      }
    }
    else if(pi.getAcceptStamp().eq(prd.getAccept())
            && 
            pi.isBound()){
      assert(fnum == prk.getFNum());
      assert(currentOffset == prk.getOffset());
      metadataWins = true;
      prevAccept = (AcceptStamp)prd.getPrevAccept().clone();
    }
    else if(!pi.isEmbargoed() 
            && 
            prd.isEmbargoed() 
            && 
            pi.getAcceptStamp().eq(prd.getAccept())){
      assert(fnum == prk.getFNum());
      assert(currentOffset == prk.getOffset());
      metadataWins = true;
      prevAccept = (AcceptStamp)prd.getPrevAccept().clone();
    }else{
      metadataWins = false;
      prevAccept = null;
    }
  
    //
    //if the current in RAS record is embargoed and its acceptStamp
    // == pi but the pi is not embargoed
    // in this case, we should debargo the record since debargoed pi
    // implies a debargo msg is issued.
    //
    if(pi.isBound() 
       && 
       (metadataWins 
        || (!prd.getValid() 
            && prd.getAccept().equals(pi.getAcceptStamp())))){
      dataToUpdate = true;
    }
    else{
      dataToUpdate = false;
    }
    
    ret.dataToUpdate = dataToUpdate;
    ret.prevAccept = prevAccept;
    ret.metadataWins = metadataWins;
    return ret;
  }



//---------------------------------------------------------------------------
// setLengthRemoveOverlapAdvanceCursor() -- 
//
// Decide len of next chunk to insert. And, if this chunk wins and 
// overlaps something that currently exists, remove the overlapping 
// part. Finally, advance cursor (if necessary) so that it refers
// to the item that will be *after* the chunk to be inserted.
// 
//---------------------------------------------------------------------------
  private long
  setLengthRemoveOverlapAdvanceCursor(Cursor cursor, PreciseInv pi, 
                                      PerRangeKey prk, PerRangeData prd, 
                                      long fnum, long currentOffset, 
                                      boolean metadataWins, boolean dataToUpdate,
                                      AcceptStamp prevAccept)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;
    long len;

    //
    // Compute overlaps for next chunk insert
    // case 1: cursor starts at or before currentOffset ends in pi
    // case 2: cursor starts at or before currentOffset ends after pi
    // case 3: cursor starts after currentOffset
    //
    if(prk.getFNum() == fnum                  // case 1
       && prk.getOffset() <= currentOffset
       && prk.getOffset() + prd.getLength() <= pi.getOffset() + pi.getLength()){
      len = prk.getOffset() + prd.getLength() - currentOffset;
      if(metadataWins){
        assert(prevAccept != null);
        chopMetadataTailAdvanceCursor(prk, prd, cursor, fnum, 
                                      currentOffset, len);
        //
        // Performance optimization -- if the next chunk has the
        // same acceptStamp as the chunk we just truncated and abuts
        // it (this can happen if the write that created the just-truncated
        // and next write hit two different older writes so they
        // have different prevAccepts), also chop that chunk
        // so that the new write can be a single long write
        // instead of two (or, recusively more) shorter ones.
        //
        status = cursor.getCurrent(key, data, null);
        assert(status == OperationStatus.SUCCESS);
        prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
        prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
        if(prk.getFNum() == fnum 
           && 
           prk.getOffset() == currentOffset + len
           &&
           prk.getOffset() < pi.getOffset() + pi.getLength()
           &&
           prd.getAccept().eq(prevAccept)){
          ClearChunk nextChunk = clearNextChunkToInsert(pi, fnum,
                                                        currentOffset + len,
                                                        cursor, key, data);
          assert(nextChunk.getDoApplyMetadata() == true);
          assert(nextChunk.getDoApplyBody() == dataToUpdate);
          assert(nextChunk.getOffset() == currentOffset + len);
          assert(nextChunk.getNextOffset() <= pi.getOffset() + pi.getLength());
          assert(nextChunk.getPrevAccept().eq(prevAccept));
          len = len + nextChunk.getLength();
          assert(currentOffset + len == nextChunk.getNextOffset());
        }
      }
      else{
        assert(prevAccept == null);
        status = cursor.getNext(key, data, null);
        prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      }
    }
    else if(prk.getFNum() == fnum             // case 2
            && prk.getOffset() <= currentOffset
            && prk.getOffset() + prd.getLength() > pi.getOffset() + pi.getLength()){
      len = pi.getOffset() + pi.getLength() - currentOffset;
      if(metadataWins){
        assert(prevAccept != null);
        chopMetadataTailAdvanceCursor(prk, prd, cursor, fnum, 
                                      currentOffset, len);
      }
      else{
        assert(prevAccept == null);
        status = cursor.getNext(key, data, null);
        prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      }
    }
    else if(prk.getFNum() > fnum             // case 3
            || prk.getOffset() > currentOffset){
      if(prk.getFNum() == fnum){
        len = min(pi.getOffset() + pi.getLength() - currentOffset,
                  prk.getOffset() - currentOffset);
      }
      else{
        len = pi.getOffset() + pi.getLength() - currentOffset;
      }
      assert(len > 0);
      assert(metadataWins);
      assert(prevAccept.eq(PREV_ACCEPT_BEGINNING_OF_TIME));
      assert(pi.isBound() == dataToUpdate);
    }
    else{
      assert(false); // The above cases are exhaustive
      len = -999;    // Make compiler happy
    }
    return len;
  }

//---------------------------------------------------------------------------
// min()
//---------------------------------------------------------------------------
  private static long
  min(long a, long b){
    return a < b ? a : b;
  }

//---------------------------------------------------------------------------
// chopMetadataTailAdvanceCursor() -- chop (or discard) the metadata
//   record under the cursor.
//
// Precondition -- the item under the cursor starts at or before
//   <fnum, offset> and overlaps <fnum, offset> to <fnum, offset+len>
//
// Postcondition -- the cursor points at the remaining record that
// starts *after* <fnum, offset>
//---------------------------------------------------------------------------
  private void
  chopMetadataTailAdvanceCursor(PerRangeKey prk, PerRangeData prd, 
                                Cursor cursor, 
                                long fnum, long offset, long len)
    throws NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    OperationStatus status;
    long insertOffset, insertLen;
    PerRangeKey insertPRK;
    PerRangeData insertPRD;

    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();

    //
    // Precondition as described at start-of-method comment
    //
    assert(prk.getFNum() == fnum);
    assert(prk.getOffset() <= offset);
    assert(prk.getOffset() + prd.getLength() > offset);

    status = cursor.delete();
    assert(status == OperationStatus.SUCCESS);

    //
    // Insert remainder before (if any)
    //
    if(prk.getOffset() < offset){
      insertPRK = (PerRangeKey)prk.clone();
      insertPRD = (PerRangeData)prd.clone();
      insertLen = offset - prk.getOffset();
      insertPRD.setLength(insertLen);
      perRangeKeyBinding.objectToEntry(insertPRK, key);
      perRangeDataBinding.objectToEntry(insertPRD, data);
      status = cursor.put(key, data);
      assert(status == OperationStatus.SUCCESS);
    }

    //
    // Insert remainder after (if any)
    //
    if(prk.getOffset() + prd.getLength() > offset + len){
      insertPRK = (PerRangeKey)prk.clone();
      insertPRD = (PerRangeData)prd.clone();
      insertPRK.setOffset(offset + len);
      insertPRD.setLength((prk.getOffset() + prd.getLength())
                          - (offset + len));
      assert(insertPRD.getLength() > 0);
      perRangeKeyBinding.objectToEntry(insertPRK, key);
      perRangeDataBinding.objectToEntry(insertPRD, data);
      status = cursor.put(key, data);
      assert(status == OperationStatus.SUCCESS);
    }
    else{
      //
      // Establish postcondition if we did not insert
      // anything after <fnum, offset>
      //
      status = cursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
    }
  
  }




//---------------------------------------------------------------------------
// updatePerRangeBody() -- update the perRangeBodyDB for the specified
//    range; return offset of first updated valid byte (which
//    must always be <offset> since this method assumes inserted
//    item always wins -- see NOTE below.
//
// Precondition -- cursor points at element s.t. no earlier element
//   overlaps insert.
//
// Postcondition -- cursor refers to just-inserted record.
//
// NOTE -- assumes that metadata has been checked and that the
//         inserted inval *wins* over all existing invals from
//         offset to (offset + length - 1).
//---------------------------------------------------------------------------
  private long
  updatePerRangeBody(Cursor cursor, BoundInval bi, 
                     long fnum, long offset, long length)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    byte b[];
    DatabaseEntry data;
    DatabaseEntry key = new DatabaseEntry();
    long aOffset;
    PerRangeKey prKey = new PerRangeKey(fnum, offset);
    OperationStatus status;


    assert(warnTBDLongLengths == true);
    assert(length < Integer.MAX_VALUE); // See warning in constructor!
    ObjInvalTarget oit = bi.getObjInvalTarget();
    aOffset = offset - oit.getOffset();
    assert(aOffset >= 0);
    assert(offset + length <= oit.getOffset() + oit.getLength());
  

    //
    // Remove any overlapping records
    //
    removeOverlappingPerRangeBody(cursor, fnum, offset, length);
  
    // 
    // Insert new record. Ok to get reference to bytes in
    // immutable bytes b/c berkeley DB makes copy on insert.
    //
    b = bi.getBody().dangerousGetReferenceToInternalByteArray(); 
    assert(aOffset < Integer.MAX_VALUE); // see warnTBDLongLengths
    assert(length < Integer.MAX_VALUE); // see warnTBDLongLengths
    data = new DatabaseEntry(b, (int)aOffset, (int)length);
    perRangeKeyBinding.objectToEntry(prKey, key);
    status = cursor.put(key, data);

    assert(status == OperationStatus.SUCCESS);

    //
    // Assert -- we don't overlap with anything before or after
    //
    if(doExpensiveSanityChecks){
      checkNoBodyOverlap(cursor, fnum, offset, length);
    }
  
    return offset;
  }



//---------------------------------------------------------------------------
// checkNoBodyOverlap() -- sanity check that the prev and next items
// (relative to cursor) don't overlap the specified fnum, offset, length
// (and while we're at it that the item under the cursor matches).
//
// Precondition -- the cursor refers to a just-inserted item <fnum, offset>
//
// Upon return, the position of the cursor is unchanged.
//---------------------------------------------------------------------------
  private void
  checkNoBodyOverlap(Cursor origCursor, long fnum, long offset, long length)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    DatabaseEntry data = new DatabaseEntry();
    DatabaseEntry key = new DatabaseEntry();
    PerRangeKey prk;
    OperationStatus status;
    Cursor dupCursor = null;

    try{
      dupCursor = origCursor.dup(true);
    
      //
      // Check prev
      //
      status = dupCursor.getPrev(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      assert(prk.getFNum() < fnum
             ||
             (prk.getFNum() == fnum 
              &&
              prk.getOffset() + data.getSize() <= offset));

      //
      // Check current
      //
      status = dupCursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      assert(prk.getFNum() == fnum);
      assert(prk.getOffset() == offset);
      assert(data.getSize() == length);

      //
      // Check next
      //
      status = dupCursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      assert(prk.getFNum() > fnum
             ||
             (prk.getFNum() == fnum 
              &&
              prk.getOffset() >= offset + length));
    
    
    }
    finally{
      if(dupCursor != null){
        dupCursor.close();
      }
    }
  }


//---------------------------------------------------------------------------
// removeOverlappingPerRangeBody() -- remove per range body records
// or parts of records that overlap.
//
// Precondition -- cursor points at element s.t. no earlier element
//   overlaps insert.
//
// Postcondition -- cursor points at the element after the specified
//   range
//---------------------------------------------------------------------------
  private void
  removeOverlappingPerRangeBody(Cursor cursor, long fnum, long offset, long length)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    DatabaseEntry data = new DatabaseEntry();
    DatabaseEntry key = new DatabaseEntry();
    PerRangeKey prk = new PerRangeKey(fnum, offset);
    OperationStatus status;


    if(!warnedRemove){
      Env.performanceWarning("RandomAccessState::removeOverlappingPerRangeBody()"
                             + " reads body before discarding it. Any way to"
                             + " convince BerkeleyDB to let us read keys w/o bodies?");
      warnedRemove = true;
    }

    //
    // Get the element under cursor
    //
    status = cursor.getCurrent(key, data, null);
    assert(status == OperationStatus.SUCCESS);
    prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);


    //
    // If cursor is completely before offset, advance it 
    //
    while(prk.getFNum() < fnum
          ||
          (prk.getFNum() == fnum 
           && 
           prk.getOffset() + data.getSize() <= offset)){
      status = cursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
    }

    //
    // Clobber anything we overlap
    // Loop invarient: no record before cursor overlaps; record
    //   under cursor either overlaps or is past end
    //
    while(prk.getFNum() == fnum && prk.getOffset() < offset + length){
      long oldFNum, oldOffset;
      oldFNum = prk.getFNum();
      oldOffset = prk.getOffset();
      truncateOrDeleteOverlappingPerRangeBodyAdvanceCursor(cursor, 
                                                           prk, 
                                                           data, 
                                                           offset, 
                                                           length);
      //
      // Each call advances cursor and updates data and prk
      //
      assert(prk.getFNum() > oldFNum || prk.getOffset() > oldOffset);
    
      //
      // Loop invarient (partial)
      //
      assert(prk.getFNum() >= fnum);
      assert(prk.getFNum() > fnum || prk.getOffset() >= offset);
    }

    //
    // Loop exit condition: record under cursor is past end && no
    // record before cursor overlaps.
    //
    assert(prk.getFNum() >= fnum);
    assert(prk.getFNum() > fnum || prk.getOffset() >= offset + length);
    if(doExpensiveSanityChecks){ // Expensive b/c read data bytes
      status = cursor.getPrev(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      assert(prk.getFNum() <= fnum);
      if(prk.getFNum() == fnum){
        assert(prk.getOffset() + data.getSize() <= offset);
      }
      //
      // Re-establish correct cursor postcondition
      //
      status = cursor.getNext(key, data, null);
      assert(status == OperationStatus.SUCCESS);
    }
  }
  private static boolean warnedRemove = false;





//---------------------------------------------------------------------------
// truncateOrDeleteOverlappingPerRangeBodyAdvanceCursor() 
// -- delete or truncate
//    the perRangeBody element that now spans 
//      prk.offset to prk.offset+data.size
//    so that is spans 
//      prk.offset to offset-1
//    and/or
//      (offset + length) to min(offset+length, prk.offset+data.size)
//
// Postcondition:
// The cursor points to the element *after* the original
// <prk.getOffset, prk.getLength> position. So, if the element
// under the cursor was completely deleted, the cursor points
// to the next element. If the element under the cursor had 
// a prefix truncated or a middle chunk removed, the cursor
// points to the surviving suffix.
//
// Also note: upon return, prkInOut and dataInOut refer to the 
// item under the cursor upon return.
//---------------------------------------------------------------------------
  private void
  truncateOrDeleteOverlappingPerRangeBodyAdvanceCursor(Cursor cursor, 
                                                       PerRangeKey prkInOut, 
                                                       DatabaseEntry dataInOut, 
                                                       long offset, 
                                                       long length)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    long chop;
    long newSize;
    long newOffset;
    DatabaseEntry insertKey = new DatabaseEntry();;
    DatabaseEntry tmpKey;
    byte old[];
    OperationStatus status;
    DatabaseEntry insertData = new DatabaseEntry();;
    PerRangeKey insertPrk, tmpPrk;
    long oldPrkOffset = prkInOut.getOffset();
    long oldPrkFNum = prkInOut.getFNum();

    //
    // Delete the old item
    //
    status = cursor.delete();
    assert(status == OperationStatus.SUCCESS);

    //
    // If item extends before, then keep the prefix
    //
    newSize = offset - prkInOut.getOffset();
    if(newSize > 0){
      old = dataInOut.getData();
      perRangeKeyBinding.objectToEntry(prkInOut, insertKey);
      assert(newSize <= Integer.MAX_VALUE); // See warnTBDLongLengths
      assert(newSize <= dataInOut.getSize());
      insertData.setData(old, 0, (int)newSize);
      status = cursor.put(insertKey, insertData);
      assert(status == OperationStatus.SUCCESS);
    }
  
    //
    // If item extends after, then keep the suffix and
    // on exit cursor, prkInOut, and dataInOut refer to
    // suffix. Otherwise, getNext() so that cursor, prkInOut,
    // and dataInOut refer to next item.
    //
    chop = offset + length - prkInOut.getOffset();
    assert(chop < Integer.MAX_VALUE); // see warnTBDLongLengths
    newSize = dataInOut.getSize() - chop;
    assert(newSize < Integer.MAX_VALUE); // see warnTBDLongLengths
    if(newSize > 0){
      newOffset = prkInOut.getOffset() + chop;
      assert(newOffset + newSize == prkInOut.getOffset() + dataInOut.getSize());
      insertPrk = new PerRangeKey(prkInOut.getFNum(), newOffset);
      perRangeKeyBinding.objectToEntry(insertPrk, insertKey);
      old = dataInOut.getData();
      dataInOut.setData(old, (int)chop, (int)newSize);
      status = cursor.put(insertKey, dataInOut);
      assert(status == OperationStatus.SUCCESS);
      prkInOut.setAllFields(insertPrk);
    }
    else{
      tmpKey = new DatabaseEntry();
      status = cursor.getNext(tmpKey, dataInOut, null);
      assert(status == OperationStatus.SUCCESS);
      tmpPrk = (PerRangeKey)perRangeKeyBinding.entryToObject(tmpKey);
      prkInOut.setAllFields(tmpPrk);
    }

    //
    // This function advances cursor and prkInOut and dataInOut
    //
    assert(oldPrkOffset < prkInOut.getOffset() 
           || oldPrkFNum < prkInOut.getFNum());
    return;
  }



//---------------------------------------------------------------------------
// checkCursorAtOrPastOffset() -- verify that the element under
//   the cursor begins at or after <offset> or that it covers
//   offset (the latter case happens when the insert loses to
//   a newer existing write)
//---------------------------------------------------------------------------
  private void
  checkCursorAtOrPastOffset(Cursor prsCursor, long fnum, long offset)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
    Cursor dupCursor = null;
    OperationStatus status;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    PerRangeKey prk;
    PerRangeData prd;

    try{
      dupCursor = prsCursor.dup(true);
      status = dupCursor.getCurrent(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
      prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
      if((prk.getFNum() >= fnum)
         &&
         (prk.getFNum() > fnum
          ||
          (prk.getFNum() == fnum
           && 
           prk.getOffset() >= offset))){
        return;  // Element under cursor begins at or after offset
      }
      if((prk.getFNum() == fnum
          && 
          (prk.getOffset() <= offset)
          &&
          (prk.getOffset() + prd.getLength() > offset))){
        return;  // Element under cursor begins before and ends after
      }
      assert(false);
    }
    finally{
      if(dupCursor != null){
        dupCursor.close();
      }
    }
  }

//---------------------------------------------------------------------------
// checkCoveredAndNoOverlaps() -- check postcondition of insert: 
//   every byte in the byterange is covered by exactly one invalidation
//   record.
//---------------------------------------------------------------------------
  private void
  checkCoveredAndNoOverlaps(Transaction txn, PreciseInv pi, long fnum)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{
  
    Cursor cursor = null;
    PerRangeKey prk;
    PerRangeData prd;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;
    long offset;

    prk = new PerRangeKey(fnum, pi.getOffset());
  
    try{
      cursor = perRangeStateDB.openCursor(txn, null);
      perRangeKeyBinding.objectToEntry(prk, key);
      status = cursor.getSearchKeyRange(key, data, null);
      assert(status == OperationStatus.SUCCESS);
      offset = checkPriorOverlapTo(cursor, fnum, pi.getOffset());
      while(offset < pi.getOffset() + pi.getLength()){
        offset = checkNextOverlapFrom(cursor, fnum, offset);
      }
    }
    finally{
      if(cursor != null){
        cursor.close();
      }
    }
  }

//---------------------------------------------------------------------------
// checkPriorOverlapTo() -- check if the record before the cursor
//   overlaps any prefix of the write starting at offset.
//   Return the first non-overlapped offset. 
//
// Postcondition: the cursor ends up pointing at the *prior*
// element (that is examined here).
//---------------------------------------------------------------------------
  private long
  checkPriorOverlapTo(Cursor cursor, long fnum, long offset)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    PerRangeKey prk;
    PerRangeData prd;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();

    OperationStatus status;
  
    status = cursor.getPrev(key, data, null);
    assert(status == OperationStatus.SUCCESS);
    prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
    prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
    assert(prk.getFNum() < fnum
           ||
           prk.getOffset() < offset);
    if(prk.getFNum() == fnum
       &&
       prk.getOffset() + prd.getLength() > offset){
      offset = prk.getOffset() + prd.getLength();
    }
    return offset;
  }

//---------------------------------------------------------------------------
// checkCurrentOverlapTo() -- sanity check that the element under
//   the next element begins at <fnum, offset> and return offset one
//   byte beyond the element. 
//
// Postcondition: the cursor advances to the element being examined
//---------------------------------------------------------------------------
  private long
  checkNextOverlapFrom(Cursor cursor, long fnum, long offset)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    PerRangeKey prk;
    PerRangeData prd;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();

    OperationStatus status;
  
    status = cursor.getNext(key, data, null);
    assert(status == OperationStatus.SUCCESS);
    prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
    prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
    assert(prk.getFNum() == fnum);
    assert(prk.getOffset() == offset);
    return prk.getOffset() + prd.getLength();
  }

//---------------------------------------------------------------------------
// checkPRSPRDMatch() -- check that the per-range state and
//   per-range body datbases are consistent with one another
//
// prs.valid --> there exists a prd record with the same
//     offset and length
// prs.invalid --> there exists no prd record that overlaps
//     the prs record
//---------------------------------------------------------------------------
  private void
  checkPRSPRDMatch(Transaction txn, PreciseInv pi, long fnum)
    throws  NullPointerException, DeadlockException, 
    IllegalArgumentException, DatabaseException{

    PerRangeKey prkMeta;
    PerRangeKey prkBody;
    PerRangeData prd;
    DatabaseEntry keyMeta = new DatabaseEntry();
    DatabaseEntry keyBody = new DatabaseEntry();
    DatabaseEntry dataMeta = new DatabaseEntry();
    DatabaseEntry dataBody = new DatabaseEntry();
    Cursor prsCursor = null;
    Cursor prdCursor = null;

    OperationStatus status;

    prkMeta = new PerRangeKey(fnum, pi.getOffset());
    prkBody = new PerRangeKey(fnum, pi.getOffset());

    try{
      prsCursor = perRangeStateDB.openCursor(txn, null);
      prdCursor = perRangeBodyDB.openCursor(txn, null);
      perRangeKeyBinding.objectToEntry(prkMeta, keyMeta);
      perRangeKeyBinding.objectToEntry(prkBody, keyBody);
      status = prsCursor.getSearchKeyRange(keyMeta, dataMeta, null);
      assert(status == OperationStatus.SUCCESS);
      status = prdCursor.getSearchKeyRange(keyBody, dataBody, null);
      assert(status == OperationStatus.SUCCESS);

      //
      // First check the prev item
      //
      status = prsCursor.getPrev(keyMeta, dataMeta, null);
      assert(status == OperationStatus.SUCCESS);
      status = prdCursor.getPrev(keyBody, dataBody, null);
      assert(status == OperationStatus.SUCCESS);
      prkMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);
      prd = (PerRangeData)perRangeDataBinding.entryToObject(dataMeta);
      prkBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);
      if(prd.getValid()){
        assert(prkMeta.getFNum() == prkBody.getFNum());
        assert(prkMeta.getOffset() == prkBody.getOffset());
        assert(prd.getLength() == dataBody.getSize());
      }
      else{
        //
        // Prior metadata not valid --> prior body element
        // must be even further back (and not overlap)
        //
        assert(prkMeta.getFNum() >= prkBody.getFNum());
        assert(prkMeta.getFNum() > prkBody.getFNum()
               ||
               (prkMeta.getFNum() == prkBody.getFNum()
                &&
                prkMeta.getOffset() >= 
                prkBody.getOffset() + dataBody.getSize())
               ||
               (prkMeta.getFNum() == SENTINAL_SMALL_FILE_NUM));
      }
      status = prsCursor.getNext(keyMeta, dataMeta, null);
      assert(status == OperationStatus.SUCCESS);
      status = prdCursor.getNext(keyBody, dataBody, null);
      assert(status == OperationStatus.SUCCESS);
      prkMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);
      prd = (PerRangeData)perRangeDataBinding.entryToObject(dataMeta);
      prkBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);

      //
      // Now check everything up to at least pi.getOffset + pi.getLength
      //
      while(prkMeta.getFNum() == fnum
            && 
            prkMeta.getOffset() < pi.getOffset() + pi.getLength()){
        //
        // Body cursor is never behind metadata cursor
        //
        assert(prkBody.getFNum() >= prkMeta.getFNum());
        assert(prkBody.getFNum() > prkMeta.getFNum()
               ||
               (prkBody.getFNum() == prkMeta.getFNum()
                &&
                prkBody.getOffset() >= prkMeta.getOffset()));

        if(prd.getValid()){
          assert(prkBody.getFNum() == prkMeta.getFNum());
          assert(prkBody.getOffset() == prkMeta.getOffset());
          assert(dataBody.getSize() == prd.getLength());
        
          status = prdCursor.getNext(keyBody, dataBody, null);
          assert(status == OperationStatus.SUCCESS);
          prkBody = (PerRangeKey)perRangeKeyBinding.entryToObject(keyBody);
        }
        status = prsCursor.getNext(keyMeta, dataMeta, null);
        assert(status == OperationStatus.SUCCESS);
        prkMeta = (PerRangeKey)perRangeKeyBinding.entryToObject(keyMeta);
        prd = (PerRangeData)perRangeDataBinding.entryToObject(dataMeta);
      }
    }
    finally{
      if(prsCursor != null){
        prsCursor.close();
      }
      if(prdCursor != null){
        prdCursor.close();
      }
    }
  }

//---------------------------------------------------------------------------
// insertSentinals() -- insert sentinal values into perRangeStateDB to 
//   simplify search logic
//---------------------------------------------------------------------------
  private void
  insertSentinals()
    throws IOException{
    Transaction txn;
    Cursor cursor;
    DatabaseEntry key = new DatabaseEntry();
    PerRangeKey prk;
    NodeId invalidNodeId = new NodeId(NodeId.INVALID_NODE_ID);
    if(SangminConfig.forkjoin){
      invalidNodeId = new BranchID(invalidNodeId.getIDint());
    }
    AcceptStamp bogusAccept = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN,
                                              invalidNodeId);
    PerRangeData prd = new PerRangeData(-999, bogusAccept, 
                                        bogusAccept, false, false, false);
    byte sentinalBytes[] = {};
    DatabaseEntry prData = new DatabaseEntry();
    perRangeDataBinding.objectToEntry(prd, prData);
    DatabaseEntry bData = new DatabaseEntry(sentinalBytes);


    OperationStatus status;

    try{
      txn = dbEnv.beginTransaction(null, null);
      prk = new PerRangeKey(SENTINAL_SMALL_FILE_NUM, -1);
      perRangeKeyBinding.objectToEntry(prk, key);
      status = perRangeStateDB.put(txn, key, prData);
      assert(status == OperationStatus.SUCCESS);
      status = perRangeBodyDB.put(txn, key, bData);
      assert(status == OperationStatus.SUCCESS);
      prk = new PerRangeKey(SENTINAL_LARGE_FILE_NUM, -1);
      perRangeKeyBinding.objectToEntry(prk, key);
      status = perRangeStateDB.put(txn, key, prData);
      assert(status == OperationStatus.SUCCESS);
      status = perRangeBodyDB.put(txn, key, bData);
      assert(status == OperationStatus.SUCCESS);
      txn.commit();
    }
    catch(DatabaseException dbe){
      throw new IOException("Exception initializing berkeleyDB Environment: " 
                            + dbe.toString());
    }
  }



//---------------------------------------------------------------------------
// getPerRangeState -- get the PerRangeStateData for the specified 
//   objId, offset
//
// Returns 
//   PerRangStateData on success
//   null on read of "hole" in existing file (file and larger-offset writes
//       exist but no write at offset)
//   throw EOFException if key exists but offset larger than any write
//       to file
//   throws NoSuchEntryException if key does not exist
//   throw NullPointerException, DeadlockException, IllegalArgumentException,
//       or DatabaseException on low-level database exception
//---------------------------------------------------------------------------
  private PerRangeState
  getPerRangeState(Transaction txn, ObjId objId, long offset)
    throws EOFException, NoSuchEntryException,
	   NullPointerException, DeadlockException, IllegalArgumentException, DatabaseException,
	   ReadOfHoleException{
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;
    PerObjData pod;
    PerRangeState prs;
    assert objId != null;
    //objIdBinding.objectToEntry(objId, key);
    try{
      key = new DatabaseEntry(objId.getPath().getBytes("UTF-8"));
     }
    catch(UnsupportedEncodingException uee){
      assert(false);
    }

    status = perObjStateDB.get(txn, key, data, null);
    if(status == OperationStatus.NOTFOUND){
      throw new NoSuchEntryException();
    }
    pod = (PerObjData)perObjDataBinding.entryToObject(data);
    if(pod.getHighestDeleteAccept().gt(pod.getHighestWriteAccept())){
      throw new NoSuchEntryException();
    }
    prs = getPerRangeState(txn, pod.getFileNum(), offset);
    if(prs != null){
      assert(prs.getKeyRef().getFNum() == pod.getFileNum());
      assert(prs.getKeyRef().getOffset() <= offset);
      assert(prs.getKeyRef().getOffset() + prs.getDataRef().getLength() > offset);
      assert(prs.getDataRef().getAccept().gt(pod.getHighestDeleteAccept()));
      assert prs.getDataRef().getAccept().lt(pod.getHighestWriteAccept())
        || prs.getDataRef().getAccept().eq(pod.getHighestWriteAccept())
        : "Returning write for obj that is newer than any write for obj!"
        + " range: " + prs.getDataRef().getAccept() 
        + " v. object: " + pod.getHighestWriteAccept() ;
    }
    return prs;
  }

//---------------------------------------------------------------------------
// getPerObjData -- get the PerObjData for the specified objId
//
// Returns 
//   PerObjData on success
//   throws NoSuchEntryException if key does not exist
//   throw NullPointerException, DeadlockException, IllegalArgumentException,
//       or DatabaseException on low-level database exception
//---------------------------------------------------------------------------
  private PerObjData
  getPerObjData(Transaction txn, ObjId objId)
    throws NoSuchEntryException, NullPointerException,
    DeadlockException, IllegalArgumentException, DatabaseException{

    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status = null;
    PerObjData pod = null;

    assert(txn != null);
    assert objId != null;
    //objIdBinding.objectToEntry(objId, key);
    try{
      key = new DatabaseEntry(objId.getPath().getBytes("UTF-8"));
    }
    catch(UnsupportedEncodingException uee){
      assert(false);
    }
    status = perObjStateDB.get(txn, key, data, null);
    if(status == OperationStatus.NOTFOUND){
      throw new NoSuchEntryException();
    }
    pod = (PerObjData)perObjDataBinding.entryToObject(data);
    return pod;
  }


//---------------------------------------------------------------------------
// getPerRangeState -- get the PerRangeState overlapping the specified 
//   fileNum, offset
//
//
// Caller only calls this if file exists (e.g., fnum exists and at least
//  one write occurred after the last delete). So, we assume as a precondition
//  that at least one per range state record exists.
//
// Returns 
//   PerRangeState on success
//   null on read of "hole" in existing file (file and larger-offset writes
//       exist but no write at offset)
//   throw EOFException if key exists but offset larger than any write
//       to file
//   throw NullPointerException, DeadlockException, IllegalArgumentException,
//       or DatabaseException on low-level database exception
//---------------------------------------------------------------------------
  private PerRangeState
  getPerRangeState(Transaction txn, long fileNum, long offset)
    throws EOFException,
	   NullPointerException, 
	   DeadlockException, 
	   IllegalArgumentException, 
	   DatabaseException,
           ReadOfHoleException{

    Cursor cursor = null;
    boolean couldBeEOF = true;
    PerRangeKey prk;
    PerRangeData prd;
    PerRangeState prs;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    OperationStatus status;

    long nextValidBytePosition = -1;
    try{
      cursor = perRangeStateDB.openCursor(txn, null);
      prk = new PerRangeKey(fileNum, offset);
      perRangeKeyBinding.objectToEntry(prk, key);
      status = cursor.getSearchKeyRange(key, data, null);
      assert(status == OperationStatus.SUCCESS); // b/c sentinal values 
      prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);

      if(prk.getFNum() == fileNum){
        if(prk.getOffset() == offset){
          prs = new PerRangeState(prk, prd);
          return prs;
        }

        assert(prk.getOffset() > offset);
        //
        // There exists a write to this file with *at least*
        // specified offset
        //
	nextValidBytePosition = prk.getOffset();
        couldBeEOF = false;
      }

      assert(prk.getFNum() > fileNum || prk.getOffset() > offset);

      //
      // Current record starts after target location; see if prior record
      // started before.
      //
      status = cursor.getPrev(key, data, null);
      assert(status == OperationStatus.SUCCESS); // b/c sentinals
      prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
      prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);

      assert(prk.getFNum() <= fileNum);
      if(prk.getFNum() < fileNum){
        assert(couldBeEOF == false); // At least one range record exists;
        // (assumed precondition) --> it must be higher offset
	
	//zjd start  -- implement efficient readHole
	assert nextValidBytePosition > offset;
	throw new ReadOfHoleException(nextValidBytePosition);
	//zjd end
        //return null;                 // Read is "hole" of existing file
      }

      assert(prk.getFNum() == fileNum);
      assert(prk.getOffset() < offset); 
      if(prk.getOffset() + prd.getLength() > offset){
        prs = new PerRangeState(prk, prd);
        return prs;
      }

      if(couldBeEOF){
        throw new EOFException();
      }
      else{
	//zjd start  -- implement efficient readHole
	assert nextValidBytePosition > offset;
	throw new ReadOfHoleException(nextValidBytePosition);
	//zjd end
        //return null; // "hole" in file
      }
    }
    finally{
      if(cursor != null){
        cursor.close();
      }
    }
  }

  private static boolean warnedShipRASInefficiency = false;

  //----------------------------------------------------------------------
  // Ship the intersected RAS records
  //----------------------------------------------------------------------
  public void shipRAS(TaggedOutputStream s,
                      SubscriptionSet cpPath,
                      //String[] exclChildNames,
                      VV cpStartVV,
		      boolean withBody)
    throws IOException{
    
    
    long startTime = 0;

    Cursor posCursor = null;
    Cursor prsCursor = null;
    Cursor prbCursor = null;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    DatabaseEntry prsKey = new DatabaseEntry();
    DatabaseEntry prsData = new DatabaseEntry();
    DatabaseEntry posKey = new DatabaseEntry();
    DatabaseEntry posData = new DatabaseEntry();

    PerObjStateForSend pos;
    
    if(checkScanTime){
      startTime = System.currentTimeMillis();
    }
    if(!warnedShipRASInefficiency){
      Env.performanceWarning("Because we don't have a way to tell RAS to " +
                             "get information about all objects from a " +
                             "given directory, we iterate through all " +
                             "objects and send checkpoint data for an " +
                             "object if it happens to lie in the requested " +
                             "directory.");
      warnedShipRASInefficiency = true;
    }

    try{
      posCursor = perObjStateDB.openCursor(null, null);
      prsCursor = perRangeStateDB.openCursor(null, null);
      prbCursor = perRangeBodyDB.openCursor(null, null);
      if(PRINT_CP_INFO){
	Env.dprintln(PRINT_CP_INFO, "RandomAccessState.shipRAS()");
	Env.dprintln(PRINT_CP_INFO, " cpPath = " + cpPath);
      //Env.dprintln(PRINT_CP_INFO,
      //             " exclChildNames.length = " + exclChildNames.length);
	Env.dprintln(PRINT_CP_INFO, " cpStartVV = " + cpStartVV);
      }
      String ss = cpPath.toString();
      String[] ssItem = ss.split(":");
      boolean matchPrefix;
      String searchKeyStr = null;
      for(int i = 0; i < ssItem.length; i ++){
	ssItem[i].trim();
	searchKeyStr = ssItem[i];
	if(ssItem[i].endsWith("/*")){
	  matchPrefix = true;
	  searchKeyStr = ssItem[i].substring(0, ssItem[i].length()-1);//get rid of the *
	}else{
	  matchPrefix = false;
	}
	assert searchKeyStr != null;
	if( PRINT_CP_INFO ){
	  Env.dprintln(PRINT_CP_INFO, "searchKeyStr: " + searchKeyStr);
	}
	try{
	  posKey = new DatabaseEntry(searchKeyStr.getBytes("UTF-8"));
	}catch(UnsupportedEncodingException e){
	  assert false;
	}
	OperationStatus status;
	if (matchPrefix){
	  status = posCursor.getSearchKeyRange(posKey, posData, null);
	}else{
	  status = posCursor.getSearchKey(posKey, posData, null);
	}
	//OperationStatus status = posCursor.getNext(posKey, posData, null);
	while(status == OperationStatus.SUCCESS){
	  //ObjId objId = (ObjId)objIdBinding.entryToObject(posKey);
	  ObjId objId = new ObjId(new String(posKey.getData()));
	  if( PRINT_CP_INFO ){
	    Env.dprintln(PRINT_CP_INFO, " Found objId = " + objId);
	  }
	  
	  //
	  //commented by zjd replace by the new searchKey using prefix match
	  //
	  //Env.dprintln(PRINT_CP_INFO, " ISStatus.liesAnywhereUnder("
	  //	       + cpPath + ", " + objId +"): "
	  //	       + ISStatus.liesAnywhereUnder(cpPath, 
	  //					    //exclChildNames, 
	  //					    objId));
	  //	  if(ISStatus.liesAnywhereUnder(cpPath, 
	  //				//exclChildNames, 
	  //				objId)){
	  if(!matchPrefix){
	    assert objId.getPath().equals(searchKeyStr);
	    if( PRINT_CP_INFO ){
	      Env.dprintln(PRINT_CP_INFO, " find exact match for " + searchKeyStr);
	    }
	  }else{
	    if(!objId.getPath().startsWith(searchKeyStr)){
	      if( PRINT_CP_INFO ){
		Env.dprintln(PRINT_CP_INFO, "stop here: find non prefix match for " + searchKeyStr + " : "+ objId);
	      }
	      break;//stop here: this objId does not contains a prefix of the searchKeyStr
	      //and don't need to check next record for perObjState
	    }
	    if( PRINT_CP_INFO ){
	      Env.dprintln(PRINT_CP_INFO, "continue:find prefix match for " + searchKeyStr + " : "+ objId);
	    }
	  }
	  PerObjData pod = null;
	  
	  pod = (PerObjData)perObjDataBinding.entryToObject(posData);
	  
	  //
	  // filter those pos records whose lastDeleteAccept is smaller
	  // than cpStartVV
	  //
	  // we only check the lastDeleteAccept since the lastWriteAccept
	  // will be updated by applying the per range state data anyway.
	  // we can use this lastWriteAccept for sanityCheck
	  //
	  if((!pod.getHighestDeleteAccept().eq(PREV_ACCEPT_BEGINNING_OF_TIME))
	     &&(!cpStartVV.includes(pod.getHighestDeleteAccept()))){
	    pos = new PerObjStateForSend(objId, 
					 pod.getHighestWriteAccept(),
					 pod.getHighestDeleteAccept());
	    s.writeObject(pos);//perObjState
	    if(printWriteObjectForCP){
	      System.out.println("RAS write :" + pos);
	    }
	  }
	  
	  if(!cpStartVV.includes(pod.getHighestWriteAccept())){
	    //something newer in this object then check the PerRangeState      
	    //otherwise if nothing new for this object, 
	    //just skip the perRangeState search
	    
	    long fileNum = pod.getFileNum();
	    PerRangeKey prk, matchPrk;
	    PerRangeData prd, matchPrd;
	    OperationStatus prsStatus = null;
	    OperationStatus prbStatus = null;
	    
	    prk = new PerRangeKey(fileNum, 0);
	    perRangeKeyBinding.objectToEntry(prk, prsKey);
	    
	    prsStatus = prsCursor.getSearchKeyRange(prsKey, prsData, null);
	    while(prsStatus == OperationStatus.SUCCESS){
	      
	      matchPrk = (PerRangeKey)perRangeKeyBinding.entryToObject(prsKey);
	      
	      if(matchPrk.getFNum() == fileNum){
		matchPrd =
		  (PerRangeData)perRangeDataBinding.entryToObject(prsData);
		// Send this PerRangeData if it is not filtered out
		// by cpStartVV.
		if(!cpStartVV.includes(matchPrd.getAccept())){
		  if((withBody)&&(matchPrd.getValid())){//need to send body
		    DatabaseEntry body = new DatabaseEntry();
		    
		    prbStatus = prbCursor.getSearchKeyRange(prsKey,
							    body,
							    null);
		    assert(prbStatus == OperationStatus.SUCCESS);
		    PerRangeKey bodyPrk = (PerRangeKey)perRangeKeyBinding.entryToObject(prsKey);
		    assert(bodyPrk.getOffset() == matchPrk.getOffset());
		    assert(bodyPrk.getFNum() == matchPrk.getFNum());
		    assert(body.getSize() == matchPrd.getLength()); 
		    BodyMsg bm = new BodyMsg(objId, 
					     matchPrk.getOffset(),
					       body.getSize(),
					     matchPrd.getAccept(),
					     ImmutableBytes.dangerousCreateImmutableBytes(body.getData()),
					     false);
		    s.writeObject(bm);
		    if(printWriteObjectForCP){
			System.out.println("RAS write :" + bm);
			body = null;
		    }
		  }else{
		    PerRangeStateForSend prsfs = new PerRangeStateForSend(objId,
									  matchPrk.getOffset(), 
									  matchPrd.getLength(),
									  matchPrd.getAccept(),
									  matchPrd.isCommitted());
		    s.writeObject(prsfs);
		    if(printWriteObjectForCP){
                        System.out.println("RAS write: " + prsfs);
		    }
		  }
		}
		prsStatus = prsCursor.getNext(prsKey, prsData, null);
	      }else {//no more record
		break;
	      }
	    }
	  }//if something is newer check perrangestate             
	
	  if(matchPrefix){
	    status = posCursor.getNext(posKey, posData, null);
	    if( PRINT_CP_INFO ){
	      Env.dprintln(PRINT_CP_INFO, "continue search for : " + searchKeyStr);
	    }
	  }else{//match exactly one
	    if( PRINT_CP_INFO ){
	      Env.dprintln(PRINT_CP_INFO, "stop search for exact match of : " + searchKeyStr);
	    }
	    break;//no more record exactly match   
	  }
	}
      }//for
      s.writeObject(new Long(RAS_SHIP_DONE));
      if(printWriteObjectForCP){
	System.out.println("RAS write: RAS_SHIPDONE=" + RAS_SHIP_DONE);
      }
    }catch (DatabaseException de){
      String msg = de.toString();
      try{
        if(posCursor != null){
          posCursor.close();
          posCursor = null;
        } 
        if (prsCursor != null){
          prsCursor.close();
          prsCursor = null;
        }
	if (prbCursor != null){
          prbCursor.close();
          prbCursor = null;
        }
      }catch(Exception e){
        msg = msg + ("\nALSO:Nested exception aborting:" + e.toString());
      }
      throw new IOException(msg);
    }

    try{
      if(posCursor != null){
        posCursor.close();
        posCursor = null;
      } 
      if (prsCursor != null){
        prsCursor.close();
        prsCursor = null;
      }
      if (prbCursor != null){
          prbCursor.close();
          prbCursor = null;
      }
    }catch(Exception e){
      String msg = "close cursor error:" + e.toString();
    
      throw new IOException(msg);
    }
    if(checkScanTime){
      System.out.println("CPScanTime: " 
                         + (System.currentTimeMillis()-startTime));
    }
  }

  private static HashMap objIdLastWriter = null;  // For sanity checking
  
  //-------------------------------------------------------------------------
  // Initialize the objIdLastWriter HashMap (sanity checking)
  //-------------------------------------------------------------------------
  private static boolean
  initObjIdLastWriter(){
    RandomAccessState.objIdLastWriter = new HashMap();
    return(true);
  }

  //-------------------------------------------------------------------------
  // Record the highest writer accept stamp for another object
  // (sanity checking)
  //-------------------------------------------------------------------------
  private static boolean
  recordObjIdLastStamp(PerObjStateForSend pos){
    RandomAccessState.objIdLastWriter.put(pos.getObjId(),
                                          pos.getHighestWriteAccept());
    return(true);
  }

  //---------------------------------------------------------------------------
  // For sanity testing, make sure that all highestWriterAcceptStamps are
  // set correctly.
  //---------------------------------------------------------------------------
  private boolean
  lastWriteAcceptStampsMatch(Transaction txn, HashMap objIdLastWriter){
    AcceptStamp senderHighestWriter = null;
    ObjId objId = null;
    boolean result = true;
    PerObjData perObjData = null;

    assert(txn != null);
    try{
      for(Iterator i = RandomAccessState.objIdLastWriter.keySet().iterator();
          i.hasNext() && result;){
        objId = (ObjId)i.next();
        senderHighestWriter = (AcceptStamp)objIdLastWriter.get(objId);
        perObjData = this.getPerObjData(txn, objId);
        result = senderHighestWriter.leq(perObjData.getHighestWriteAccept());
        if (!result){//any pos.lastWriterAccept is less than 
                     //sender.pos.lastWriterAccept
                     // this means that there're some prs haven't arrived.
          break;
        }
      }
      RandomAccessState.objIdLastWriter.clear();
      RandomAccessState.objIdLastWriter = null;
    }catch(NoSuchEntryException e){
      result = false;
    }catch(DeadlockException e){
      result = false;
    }catch(IllegalArgumentException e){
      result = false;
    }catch(DatabaseException e){
      result = false;
    }

    return(result);
  }


  private static boolean warningIsDone = false;

  //-------------------------------------------------------------------------
  // update PerObjStateDB and PerRangeStateDB according to the inputStream
  //-------------------------------------------------------------------------
  public boolean
    updateRAS(LinkedList s, SubscriptionSet path, long txnSize)
    throws IOException{
    boolean dbgMe = false;
    Transaction txn = null;
    boolean checkUpdatePerformance = false;
    int count = 0;
    long start = 0;
    long end = 0;
    long totalTime = 0;

    if(!warningIsDone){
      Env.performanceWarning("RandomAccessState.updateRAS() ideally"
                             + " should update the PerRangeState/PerObjState"
                             + " records in one single transaction"
                             + " however the berkeley db transaction "
                             + " hits a performance drop off when the cache"
                             + " fills up the predecided cache size\n"
                             + " Therefore we have to split the transaction "
                             + " into multiple transactions each will commit"
                             + " when the number of records reach a certain"
                             + " number(right now we have it as CacheSize/10000)"
                             + " \n\n we still have the safety property as"
                             + " we update apply the gapInv first and only"
                             + " update the lpVV at the end of the last txn");
      Env.tbd("RandomAccessState.updateRAS() might end up only update partial"
              + " states of the PerRangeState/PerObjState, in this case, we"
              + " need to inform the controller. But what information is needed"
              + " to avoid resubsribe the whole things?");
      warningIsDone = true;
    }

    assert txnSize > 0;
    
    try{
      txn = dbEnv.beginTransaction(null, null);
      if( PRINT_CP_INFO ){
	Env.dprintln(PRINT_CP_INFO, "RandomAccessState.updateRAS");
      }
      assert(RandomAccessState.initObjIdLastWriter());
      while(count < txnSize){
        //Object o = s.readTaggedObject();
        Object o = null;
        try{
          o = s.remove();
        }catch(NoSuchElementException nsee){
          assert false;//impossible
        }
        
        if(dbgMe){
          System.out.println("------------RAS.updateRAS received: " + o);
        }
        if(o instanceof PerObjStateForSend){
          
          PerObjStateForSend pos = (PerObjStateForSend)o;
	  if( PRINT_CP_INFO ){
	    Env.dprintln(PRINT_CP_INFO, "PerObjStateForSend");
	    Env.dprintln(PRINT_CP_INFO, " objId = " + pos.getObjId());
	    Env.dprintln(PRINT_CP_INFO,
                       " highestWriteAccept = " + pos.getHighestWriteAccept());
	    Env.dprintln(PRINT_CP_INFO,
                       " highestDeleteAccept = " +
                       pos.getHighestDeleteAccept());
	  }

          assert ISStatus.liesAnywhereUnder(path, pos.getObjId());
          
          // in shipRAS we only send POS when pos.deleteAccept > startVV
          // therefore we always need to call delete for any pos
          assert pos.getHighestDeleteAccept().gt(ACCEPT_BEGINNING_OF_TIME);
          if(checkUpdatePerformance){
            start = System.currentTimeMillis();
          }
          delete(pos.getObjId(), pos.getHighestDeleteAccept());
          
          if(checkUpdatePerformance){
            end = System.currentTimeMillis();
            totalTime += end -start;
            count++;
            
            System.out.println("RAS.updateCheckpoint::delete:" + count
                               + " " + (end-start));
          }
	  if(doExpensiveSanityChecks){
	      assert(RandomAccessState.recordObjIdLastStamp(pos));
	  }
        }else if(o instanceof PerRangeStateForSend){
          
          PerRangeStateForSend prs = (PerRangeStateForSend)o;
          assert ISStatus.liesAnywhereUnder(path, prs.getObjId());
          ObjInvalTarget oit = new ObjInvalTarget(prs.getObjId(),
                                                  prs.getOffset(),
                                                  prs.getLength());
          PreciseInv pi = new PreciseInv(oit, prs.getAccept(), false, prs.isCommitted());//don't care of embargoed
          //gc embargoed field
          if(checkUpdatePerformance){
            start = System.currentTimeMillis();
          }
          applyInvalTransaction(txn, pi);
          //applyInvalTransaction(pi);
          if(checkUpdatePerformance){
            end = System.currentTimeMillis();
            totalTime += end -start;
            count++;
            
            System.out.println("RAS.updateCheckpoint::applyInval:" + count
                               + " " + (end-start));
          }
          Env.tbd("RandomAccessState::updateRAS conflict should be detected" +
                  " using PerRangeState.prevAccept info"); 
	  if( PRINT_CP_INFO ){
	    Env.dprintln(PRINT_CP_INFO, "PerRangeStateForSend");
	    Env.dprintln(PRINT_CP_INFO, " objId = " + prs.getObjId());
	    Env.dprintln(PRINT_CP_INFO, " offset = " + prs.getOffset());
	    Env.dprintln(PRINT_CP_INFO, " length = " + prs.getLength());
	    Env.dprintln(PRINT_CP_INFO, " accept = " + prs.getAccept());
             Env.dprintln(PRINT_CP_INFO, " accept = " + prs.getAccept());
	  }
	}else if(o instanceof BodyMsg){
          
          BodyMsg bm = (BodyMsg)o;
          assert ISStatus.liesAnywhereUnder(path, bm.getObjId());
	  ObjInvalTarget oit = bm.getObjInvalTarget();
	  BoundInval bi = new BoundInval(oit.getObjId(), oit.getOffset(), 
					 oit.getLength(),
					 bm.getAcceptStamp(), bm.getBody());          
          if(checkUpdatePerformance){
            start = System.currentTimeMillis();
          }
          applyInvalTransaction(txn, bi);
          //applyInvalTransaction(pi);
          if(checkUpdatePerformance){
            end = System.currentTimeMillis();
            totalTime += end -start;
            count++;
            
            System.out.println("RAS.updateCheckpoint::applyInval:" + count
                               + " " + (end-start));
          }
          Env.tbd("RandomAccessState::updateRAS conflict should be detected" +
                  " using PerRangeState.prevAccept info"); 
	  if( PRINT_CP_INFO ){
	    Env.dprintln(PRINT_CP_INFO, "BodyMsg");
	    Env.dprintln(PRINT_CP_INFO, " objId = " + oit.getObjId());
	    Env.dprintln(PRINT_CP_INFO, " offset = " + oit.getOffset());
	    Env.dprintln(PRINT_CP_INFO, " length = " + oit.getLength());
	    Env.dprintln(PRINT_CP_INFO, " accept = " + bm.getAcceptStamp());
	  }

        }else if (o instanceof Long){
           if( PRINT_CP_INFO ){
	     Env.dprintln(PRINT_CP_INFO, "Done shipping checkpoint");
	   }
          
          if((checkUpdatePerformance)||dbgPerformance){
                        
            System.out.println("RAS.updateCheckpoint::totalTime: " + count
                               + " " + totalTime);
          }
          Env.remoteAssert(((Long)o).longValue() == RAS_SHIP_DONE);
          if(doExpensiveSanityChecks){
	      assert(lastWriteAcceptStampsMatch(txn, objIdLastWriter));
	  }
          txn.commit();
          txn = null;
          return true;
        }else {
          String msg = "Corruptted incomming Checkpoint Stream";
          if(txn != null){
            try{
              txn.abort();
            }catch(DatabaseException de){
              msg +="transaction abort failed" + de;
              assert false;
            }
            System.err.println("updating RAS for cpExchange aborted");
          }
          throw new IOException(msg);
        }
      }
      assert count == txnSize;
      txn.commit();
      txn = null;
      return false;
    }catch(Exception e){
      String msg ="Exception while updating RAS for cpExchange: " +e
                         +"abort all the in-progress RAS updates";
      if(txn != null){
        try{
          txn.abort();
        }catch(DatabaseException de){
          msg += "transaction abort failed";
          assert false;
        }
        System.err.println("updating RAS for cpExchange aborted");
      }
      throw new IOException(msg);

    }finally{
      if(txn != null){
        try{
	  assert txn != null;
          txn.abort();
        }catch(DatabaseException de){
          System.err.println("transaction abort failed");
          assert false;
        }
        System.err.println("updating RAS for cpExchange aborted");
      }
    }
  }
  
  //-----------------------------------------------------------------
  // check if a given obj data status is valid
  //-----------------------------------------------------------------
  private boolean isValid(ObjId oid)
  throws IOException{
    DatabaseEntry oidDE = new DatabaseEntry();
    DatabaseEntry posData = new DatabaseEntry();
    DatabaseEntry prsKey = new DatabaseEntry();
    DatabaseEntry prsData = new DatabaseEntry();
    Cursor prsCursor = null;
    PerObjData pod = null;
    OperationStatus status = null;
    PerRangeKey prk = null, matchPrk;
    PerRangeData matchPrd = null;
    
    try{
      prsCursor = perRangeStateDB.openCursor(null, null);
      assert oid != null;
      //objIdBinding.objectToEntry(oid, oidDE);
      try{
	oidDE = new DatabaseEntry(oid.getPath().getBytes("UTF-8"));
      }
      catch(UnsupportedEncodingException uee){
	assert(false);
      }
      status = perObjStateDB.get(null, oidDE, posData, null);
      if(status == OperationStatus.SUCCESS){
        pod = (PerObjData)perObjDataBinding.entryToObject(posData);
        long fileNum = pod.getFileNum();
        prk = new PerRangeKey(fileNum, 0);
        perRangeKeyBinding.objectToEntry(prk, prsKey);
        
        status = prsCursor.getSearchKeyRange(prsKey, prsData, null);
        while(status == OperationStatus.SUCCESS){
          
          matchPrk = (PerRangeKey)perRangeKeyBinding.entryToObject(prsKey);
        
          if(matchPrk.getFNum() == fileNum){
            matchPrd =
              (PerRangeData)perRangeDataBinding.entryToObject(prsData);
            if(!matchPrd.getValid()){
              prsCursor.close();
              return false;
            }
            status = prsCursor.getNext(prsKey, prsData, null);
          }else {//no more record
            break;
          }
        }
        prsCursor.close();
      }
    }catch (DatabaseException de){
      String msg = de.toString();
      try{
        if (prsCursor != null){
          prsCursor.close();
          prsCursor = null;
        }
      }catch(Exception e){
        msg = msg + ("\nALSO:Nested exception aborting:" + e.toString());
      }
      throw new IOException(msg);
    }
    
    return true;
  }

  //-----------------------------------------------------------------
  // by default all precise inv applied has a default 
  // perRangeData.committed value of "false";
  // 
  // only an explicit commitInv is issued will any perRangeData.committed
  // become "true";
  // 
  // this method is similar to debargo
  // except that we update the perRangeData.committed instead of
  // perRangeData.embargoed
  // and we might update multiple objects for committing multiobjPreciseinv
  //-----------------------------------------------------------------
  public boolean applyCommitInv(CommitInv gi)
  throws IOException{
    InvalTarget targetSet = gi.getInvalTarget();
    boolean ret = false;
    Transaction txn = null;
    Cursor prsCursor = null;
    try{
      txn = dbEnv.beginTransaction(null, null);
      prsCursor = perRangeStateDB.openCursor(txn, null);
      
      if(targetSet instanceof ObjInvalTarget){
      
        ret = this.applyCommitInvInternal(txn,
                                          prsCursor,
                                          (ObjInvalTarget) targetSet,
                                          gi.getTargetAS());
        //tbd do we need to keep the commit sequence for sanityCheck?
      }else{
        assert targetSet instanceof MultiObjInvalTarget;
        MultiObjInvalTarget moit = (MultiObjInvalTarget)targetSet;
        ObjInvalTarget oit = null;
        for(MOITIterator iter = moit.getIterator(true); iter.hasNext();){
          oit = iter.getNext();
          assert oit != null;
          if(this.applyCommitInvInternal(txn, prsCursor, oit, gi.getTargetAS()) && !ret){
            ret = true;
          }
        }
        for(MOITIterator iter = moit.getIterator(false); iter.hasNext();){
          oit = iter.getNext();
          assert false: "commit an object delete operation is not supportted yet";
          //this.state.delete(oit.getObjId(), mopi.getAcceptStamp());
        }
      }
      prsCursor.close();
      prsCursor = null;
      txn.commit();
      txn = null;
    }catch (DatabaseException de){
      String msg = de.toString();
      try{
        if (prsCursor != null){
          prsCursor.close();
          prsCursor = null;
        }
      }catch(Exception e){
        msg = msg + ("\nALSO:Nested exception aborting:" + e.toString());
      }
      if(txn != null){
        try{
          txn.abort();
          txn = null;
        }catch(Exception e){
          msg = msg + ("\nALSO:Nested exception aborting:" + e.toString()); 
        }
      }
      throw new IOException(msg);
    }

    return ret;
  }
   
  private boolean applyCommitInvInternal(Transaction txn, 
                                         Cursor prsCursor,
                                         ObjInvalTarget oit, 
                                         AcceptStamp targetAS)
  throws DatabaseException{
    ObjId oid = oit.getObjId();
    long offset = oit.getOffset();
    long length = oit.getLength();
    AcceptStamp as = targetAS;
    DatabaseEntry oidDE = new DatabaseEntry();
    DatabaseEntry posData = new DatabaseEntry();
    DatabaseEntry prsKey = new DatabaseEntry();
    DatabaseEntry prsData = new DatabaseEntry();
    
    PerObjData pod = null;
    OperationStatus status = null;
    PerRangeKey prk = null, matchPrk;
    PerRangeData matchPrd = null;
    boolean ret = false;
      
      assert oid != null;
      //objIdBinding.objectToEntry(oid, oidDE);
      try{
        oidDE = new DatabaseEntry(oid.getPath().getBytes("UTF-8"));
       }
      catch(UnsupportedEncodingException uee){
        assert(false);
      }
      status = perObjStateDB.get(null, oidDE, posData, null);
      if(status == OperationStatus.SUCCESS){
        pod = (PerObjData)perObjDataBinding.entryToObject(posData);
        long fileNum = pod.getFileNum();
        prk = new PerRangeKey(fileNum, offset);
        perRangeKeyBinding.objectToEntry(prk, prsKey);
        
        status = prsCursor.getSearchKeyRange(prsKey, prsData, null);
        while(status == OperationStatus.SUCCESS){
          
          matchPrk = (PerRangeKey)perRangeKeyBinding.entryToObject(prsKey);
        
          if(matchPrk.getFNum() == fileNum){
            matchPrd =
              (PerRangeData)perRangeDataBinding.entryToObject(prsData);
            if(matchPrk.getOffset() >= offset+length){
              break;
            }else if(!matchPrd.isCommitted()){
                if(matchPrd.getAccept().eq(as)){
                    matchPrd.setCommitted(true);
                    
                    perRangeDataBinding.objectToEntry(matchPrd, prsData);
                    status = prsCursor.delete();
                    assert status == OperationStatus.SUCCESS;
                    status = prsCursor.put(prsKey, prsData);
                    assert status == OperationStatus.SUCCESS;
                    ret = true;
                }
            }
            
            status = prsCursor.getNext(prsKey, prsData, null);
          }else {//no more record
            break;
          }
        }
        
      }
     
    return ret;
  }
  
  //-----------------------------------------------------------------
  // debargo an embargoed write
  //-----------------------------------------------------------------
  public boolean debargo(DebargoMsg dm)
  throws IOException{
      ObjId oid = dm.getObjId();
      long offset = dm.getOffset();
      long length = dm.getLength();
      AcceptStamp as = dm.getAcceptStamp();
    DatabaseEntry oidDE = new DatabaseEntry();
    DatabaseEntry posData = new DatabaseEntry();
    DatabaseEntry prsKey = new DatabaseEntry();
    DatabaseEntry prsData = new DatabaseEntry();
    Cursor prsCursor = null;
    PerObjData pod = null;
    OperationStatus status = null;
    PerRangeKey prk = null, matchPrk;
    PerRangeData matchPrd = null;
    boolean ret = false;
    Transaction txn = null;
    try{
      txn = dbEnv.beginTransaction(null, null);
      prsCursor = perRangeStateDB.openCursor(txn, null);
      assert oid != null;
      //objIdBinding.objectToEntry(oid, oidDE);
      try{
	oidDE = new DatabaseEntry(oid.getPath().getBytes("UTF-8"));
       }
      catch(UnsupportedEncodingException uee){
	assert(false);
      }
      status = perObjStateDB.get(null, oidDE, posData, null);
      if(status == OperationStatus.SUCCESS){
        pod = (PerObjData)perObjDataBinding.entryToObject(posData);
        long fileNum = pod.getFileNum();
        prk = new PerRangeKey(fileNum, offset);
        perRangeKeyBinding.objectToEntry(prk, prsKey);
        
        status = prsCursor.getSearchKeyRange(prsKey, prsData, null);
        while(status == OperationStatus.SUCCESS){
          
          matchPrk = (PerRangeKey)perRangeKeyBinding.entryToObject(prsKey);
        
          if(matchPrk.getFNum() == fileNum){
            matchPrd =
              (PerRangeData)perRangeDataBinding.entryToObject(prsData);
            if(matchPrk.getOffset() >= offset+length){
              break;
            }else if(matchPrd.isEmbargoed()){
		if(matchPrd.getAccept().eq(as)){
		    matchPrd.setEmbargoed(false);
                    
                    perRangeDataBinding.objectToEntry(matchPrd, prsData);
                    status = prsCursor.delete();
                    assert status == OperationStatus.SUCCESS;
                    status = prsCursor.put(prsKey, prsData);
                    assert status == OperationStatus.SUCCESS;
		    ret = true;
		}
	    }
	    
            status = prsCursor.getNext(prsKey, prsData, null);
          }else {//no more record
            break;
          }
        }
        prsCursor.close();
        prsCursor = null;
        txn.commit();
        txn = null;
      }
    }catch (DatabaseException de){
      String msg = de.toString();
      try{
        if (prsCursor != null){
          prsCursor.close();
          prsCursor = null;
        }
      }catch(Exception e){
        msg = msg + ("\nALSO:Nested exception aborting:" + e.toString());
      }
      if(txn != null){
        try{
          txn.abort();
          txn = null;
        }
        catch(Exception e){
          msg = msg + ("\nALSO:Nested exception aborting:" + e.toString()); 
        }
      }
      throw new IOException(msg);
    }
    
    return ret;
  }
  
  //---------------------------------------------------------------------------
  // getValidBytes -- get byte array for specified fileNum, offset, maxLength
  //   Assumes valid bytes exist in one write element for that range.
  //
  // Throws NullPointerException, DeadlockException, IllegalArgumentException, 
  // or DatabaseException on low-level db exception
  //---------------------------------------------------------------------------
  private ImmutableBytes
  getValidBytes(Transaction txn, 
                PerRangeState prs,  // PerRangeState record that overlaps offset
                long offset,        // Desired start
                long maxLen)       // Max len (mey return shorter array)
    throws NullPointerException, DeadlockException, IllegalArgumentException, 
    DatabaseException{
  
    PerRangeKey prk = prs.getKeyRef();
    PerRangeData prd = prs.getDataRef();
    Cursor cursor = null;
    OperationStatus status;
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry value = new DatabaseEntry();
    PerRangeKey prkGot;
    long length, endNonInclusive, skip;
    byte b[];

    assert(maxLen > 0);
    assert(prd.getValid());
    assert(prk.getOffset() <= offset);
    skip = offset - prk.getOffset();
    assert(skip >= 0);
    endNonInclusive = prk.getOffset() + prd.getLength();
    assert(endNonInclusive > offset);

    if(endNonInclusive > offset + maxLen){
      endNonInclusive = offset + maxLen;
    }
    length = endNonInclusive - offset;
    assert(length <= maxLen);
    assert(offset + length <= prk.getOffset() + prd.getLength());
    assert(prk.getOffset() + skip + length == endNonInclusive);
  

    try{
      cursor = perRangeBodyDB.openCursor(txn, null);
    
      perRangeKeyBinding.objectToEntry(prk, key);
      assert(skip <= Integer.MAX_VALUE);   // Limitation of DB API
      assert(length <= Integer.MAX_VALUE);
      value.setPartial((int)skip, (int)length, true);
      status = cursor.getSearchKeyRange(key, value, null);
      assert(status == OperationStatus.SUCCESS); // B/c sentinal
      prkGot = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
    
      //
      // PRS is valid --> byterange is valid
      //
      assert(prkGot.getOffset() == prk.getOffset());
      assert(prkGot.getFNum() == prk.getFNum());
      assert(value.getSize() == length); // b/c setPartial
      b = value.getData();
      assert(b.length == length);
      return ImmutableBytes.dangerousCreateImmutableBytes(b); // b is the only reference
    }
    finally{
      if(cursor != null){
        cursor.close();
      }
    }
  }

//---------------------------------------------------------------------------
// overrideDoExpensiveSanityChecks() -- manually turn on (or off) detailed
//   sanity checks (mainly for testing).
//---------------------------------------------------------------------------
    private void
      overrideDoExpensiveSanityChecks(boolean b){
      doExpensiveSanityChecks = b;
    }



//---------------------------------------------------------------------------
// getAuxCheckpointStuff() -- return stuff needed for the full checkpoint
//---------------------------------------------------------------------------
  public byte[] getAuxCheckpointStuff(String keyS) 
    throws NoSuchEntryException, IOException{
    DatabaseEntry key = new DatabaseEntry(keyS.getBytes());
    DatabaseEntry value = new DatabaseEntry();
    Transaction txn = null;
    OperationStatus status;
    byte[] result;
    try{
      txn = dbEnv.beginTransaction(null, null);
      status = auxCheckpointStuffDB.get(txn, key, value, LockMode.DEFAULT);
      if(status == OperationStatus.NOTFOUND){
        throw new NoSuchEntryException();
      }
      assert(status == OperationStatus.SUCCESS);
      txn.commit();
      txn = null;
      result = value.getData();
      return result;
    }
    catch(DatabaseException de){
      throw new IOException("Caused by RandomAccessState::getAuxCheckpointStuff(): " 
                            + de.toString());
    }
    finally{
      if(txn != null){
        try{
          txn.abort();
        }
        catch(DatabaseException de2){
          Env.warn("Database exception handling error in RandomAccessState::getAuxCheckpointSTuff()"
                   + de2.toString());
        }
      }
    }
  }

//---------------------------------------------------------------------------
// putAuxCheckpointStuff() -- return stuff needed for the full checkpoint
//---------------------------------------------------------------------------
  public void putAuxCheckpointStuff(String keyS, byte[] valueB) 
    throws IOException{
    DatabaseEntry key = new DatabaseEntry(keyS.getBytes());
    DatabaseEntry value = new DatabaseEntry(valueB);
    Transaction txn = null;
    OperationStatus status;


    try{
      //
      // Assume database configured for no duplicates.
      // Each insert overwrites past ones.
      //
      assert(auxCheckpointStuffDB.getConfig().getSortedDuplicates() == false);
      // getUnsortedDuplicates not available in current JE 1.7; uncomment
      // the assert for later versions of JE...
      // assert(auxCheckpointStuffDB.getConfig().getUnsortedDuplicates() == false);

      txn = dbEnv.beginTransaction(null, null);
      status = auxCheckpointStuffDB.put(txn, key, value);
      assert(status == OperationStatus.SUCCESS);
      txn.commit();
      txn = null;
      return;
    }
    catch(DatabaseException de){
      throw new IOException("Caused by RandomAccessState::putAuxCheckpointStuff(): " 
                            + de.toString());
    }
    finally{
      if(txn != null){
        try{
          txn.abort();
        }
        catch(DatabaseException de2){
          Env.warn("Database exception handling error in RandomAccessState::putAuxCheckpointSTuff()"
                   + de2.toString());
        }
      }
    }
  }



//---------------------------------------------------------------------------
// dbgPrintPRB() -- print the perRangeBody records for all offsets for objId
//---------------------------------------------------------------------------
    private void
      dbgPrintPRB(ObjId id, String msg){
      try{
        long fnum = dbgGetFNum(id);
        dbgPrintPRB(fnum, msg);
      }
      catch(NoSuchEntryException e){
        System.err.println("dbgPrintPRS " + id + " does not exist.");
      }
    }

//---------------------------------------------------------------------------
// dbgPrintPRB() -- print the perRangeBody records for all offsets for fnum
//---------------------------------------------------------------------------
    private void
      dbgPrintPRB(long fnum, String msg){

      OperationStatus status;
      DatabaseEntry key = new DatabaseEntry();
      DatabaseEntry data = new DatabaseEntry();
      PerRangeKey prk = new PerRangeKey(fnum-1, Long.MAX_VALUE);
      Cursor cursor = null;
      Transaction txn = null;

      System.err.println("vvvv dbgPrintPRB vvvv");
      System.err.println(msg);
  
      try{
        txn = dbEnv.beginTransaction(null, null);
        cursor = perRangeBodyDB.openCursor(txn, null);
        perRangeKeyBinding.objectToEntry(prk, key);
        status = cursor.getSearchKeyRange(key, data, null);
        assert(status == OperationStatus.SUCCESS); // B/c sentinal
        prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
        while(prk.getFNum() <= fnum){
          if(prk.getFNum() == fnum){
            System.err.print("PRK: " + prk.toString());
            System.err.println(" PRD length: " + data.getSize());
          }
          status = cursor.getNext(key, data, null);
          assert(status == OperationStatus.SUCCESS); // B/c sentinal
          prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
        }
        cursor.close();
        cursor = null;
        txn.commit();
        txn = null;
      }
      catch(DatabaseException de){
        de.printStackTrace();
        assert(false);
      }
      finally{
        if(cursor != null){
          try{
            cursor.close();
          }
          catch(Exception z){
            z.printStackTrace();
            assert(false);
          }
        }
        if(txn != null){
          try{
            txn.abort();
          }
          catch(Exception e){
            e.printStackTrace();
            // Ignore failed abort in dbg routine
          }
        }
      }

  
      System.err.println("^^^^ dbgPrintPRB ^^^^");
    }



//---------------------------------------------------------------------------
// dbgGetFNum() -- get the fnum for objId
//---------------------------------------------------------------------------
    private long
      dbgGetFNum(ObjId id)
      throws NoSuchEntryException{

      DatabaseEntry key = new DatabaseEntry();
      DatabaseEntry data = new DatabaseEntry();
      OperationStatus status;
      PerObjData pod;
      Transaction txn = null;

      try{
        txn = dbEnv.beginTransaction(null, null);
	assert id != null;
        //objIdBinding.objectToEntry(id, key);
	try{
	  key = new DatabaseEntry(id.getPath().getBytes("UTF-8"));
        }
	catch(UnsupportedEncodingException uee){
	  assert(false);
	}
        status = perObjStateDB.get(txn, key, data, null);
        if(status == OperationStatus.NOTFOUND){
          throw new NoSuchEntryException();
        }
        pod = (PerObjData)perObjDataBinding.entryToObject(data);
        return pod.getFileNum();
      }
      catch(DatabaseException e){
        e.printStackTrace();
        assert(false);
      }
      finally{
        if(txn != null){
          try{
            txn.abort();
          }
          catch(Exception e){
            assert(false);
          }
        }
      }
      assert(false); // Notreached. Make compiler happy.
      return -999;
    }

//---------------------------------------------------------------------------
// dbgPrintPRS() -- print the perRangeState records for all offsets for objId
//---------------------------------------------------------------------------
    private void
      dbgPrintPRS(ObjId id, String msg){
      try{
        long fnum = dbgGetFNum(id);
        dbgPrintPRS(fnum, msg);
      }
      catch(NoSuchEntryException e){
        System.err.println("dbgPrintPRS " + id + " does not exist.");
      }
    }

//---------------------------------------------------------------------------
// dbgPrintPRS() -- print the perRangeState records for all offsets for fnum
//---------------------------------------------------------------------------
    private void
      dbgPrintPRS(long fnum, String msg){
      Transaction txn = null;
      try{
        txn = dbEnv.beginTransaction(null, null);
        dbgPrintPRS(fnum, txn, msg);
        txn.commit();
        txn = null;
      }
      catch(DatabaseException dbe){
        dbe.printStackTrace();
        assert(false);
      }
      finally{
        if(txn != null){
          try{
            txn.abort();
          }
          catch(Exception e){
            e.printStackTrace();
            // Ignore failed abort in dbg routine
          }
        }
      }
    }

//---------------------------------------------------------------------------
// dbgPrintPRS() -- print the perRangeState records for all offsets for fnum
//---------------------------------------------------------------------------
    private void
      dbgPrintPRS(long fnum, Transaction txn, String msg){

      OperationStatus status;
      DatabaseEntry key = new DatabaseEntry();
      DatabaseEntry data = new DatabaseEntry();
      PerRangeKey prk = new PerRangeKey(fnum, 0);
      PerRangeData prd;
      Cursor cursor = null;

      System.err.println("vvvv dbgPrintPRS " + msg + " vvvv");

      try{
        cursor = perRangeStateDB.openCursor(txn, null);
        perRangeKeyBinding.objectToEntry(prk, key);
        status = cursor.getSearchKeyRange(key, data, null);
        assert(status == OperationStatus.SUCCESS); // B/c sentinal
        prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
        prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
        while(prk.getFNum() <= fnum){
          if(prk.getFNum() == fnum){
            System.err.print("[PRK: " + prk.toString());
            System.err.println(", PRD: " + prd.toString() + "]");
          }
          status = cursor.getNext(key, data, null);
          assert(status == OperationStatus.SUCCESS); // B/c sentinal
          prk = (PerRangeKey)perRangeKeyBinding.entryToObject(key);
          prd = (PerRangeData)perRangeDataBinding.entryToObject(data);
        }
        cursor.close();
        cursor = null;
      }
      catch(DatabaseException de){
        de.printStackTrace();
        assert(false);
      }
      finally{
        if(cursor != null){
          try{
            cursor.close();
          }
          catch(Exception z){
            z.printStackTrace();
            assert(false);
          }
        }
      }

  
      System.err.println("^^^^ dbgPrintPRS ^^^^");
    }


//---------------------------------------------------------------------------
// destroyAllStateCleanup -- delete all of the files we created (for testing)
//---------------------------------------------------------------------------
  public static void
  destroyAllStateCleanup(String path){
      File dir;
      File files[];
      int ii;
      try{
        dir = new File(path);
        if(!dir.exists()){
          return;
        }
        assert(dir.isDirectory());
        files = dir.listFiles();
        for(ii = 0; ii < files.length; ii++){
          files[ii].delete();
        }
        dir.delete();
      }
      catch(Exception e){
        System.err.println("destroyAllStateCleanup error: " + e.toString());
        e.printStackTrace();
        return;
      }
    }



  

  

  //---------------------------------------------------------------------------
  // self test code moved to RandomAccessStateUnit and RandomAccessStateUnitMT
  //---------------------------------------------------------------------------

}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerRangeKey -- Key for perRangeStateDB and perRangeBodyDB
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class PerRangeKey{
    private long fileNum;
    private long offset;

    protected 
    PerRangeKey(long fn, long off){
      fileNum = fn;
      offset = off;
    }

    protected final long
    getFNum(){
      return fileNum;
    }

    protected final long
    getOffset(){
      return offset;
    }

    protected final void
    setOffset(long o){
      offset = o;
    }

    protected final Object
    clone(){
      return new PerRangeKey(fileNum, offset);
    }

    public final String
    toString(){
      return "[fnum=" + fileNum + " off=" + offset + "]";
    }

    protected final void
    setAllFields(PerRangeKey prk){
      this.offset = prk.getOffset();
      this.fileNum = prk.getFNum();
    }

  }

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerRangeData -- Data for perRangeStateDB
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class PerRangeData implements Serializable{
    private long length;
    private AcceptStamp accept;
    private AcceptStamp prevAccept;
    private boolean valid;
    private boolean embargoed;
    private boolean committed; //by default it is false;
      
    protected 
    PerRangeData(long length_, AcceptStamp accept_, AcceptStamp prevAccept_, 
                 boolean valid_, boolean embargoed_){
  
      length = length_;
      accept = (AcceptStamp)accept_.clone();         // Note: Immutable -- clone returns this
      prevAccept = (AcceptStamp)prevAccept_.clone(); // Note: Immutable -- clone returns this
      valid = valid_;
      embargoed = embargoed_;
      committed = false;
    }

    protected 
    PerRangeData(long length_, AcceptStamp accept_, AcceptStamp prevAccept_, 
                 boolean valid_, boolean embargoed_, boolean committed){
  
      length = length_;
      accept = (AcceptStamp)accept_.clone();         // Note: Immutable -- clone returns this
      prevAccept = (AcceptStamp)prevAccept_.clone(); // Note: Immutable -- clone returns this
      valid = valid_;
      embargoed = embargoed_;
      this.committed = committed;
    }
    
    protected final Object
    clone(){
      assert(accept instanceof Immutable);
      assert(prevAccept instanceof Immutable);
      return new PerRangeData(length, accept, prevAccept, valid, embargoed, committed);
    }

    protected final long
    getLength(){
      return length;
    }

    protected final void
    setLength(long l){
      length = l;
    }

    protected final AcceptStamp
    getAccept(){
      return (AcceptStamp)accept.clone();
    }

    protected final AcceptStamp
    getPrevAccept(){
      return (AcceptStamp)prevAccept.clone();
    }

    protected final boolean
    getValid(){
      return valid;
    }

    protected final void
    setValid(boolean v){
      valid = v;
    }

    protected final boolean
	isEmbargoed(){
	return embargoed;
    }

    protected final void
	setEmbargoed(boolean e){
	embargoed = e;
    }
    protected final boolean
      isCommitted(){
        return committed;
      }
  
    protected final void
      setCommitted(boolean e){
      committed = e;
    }

    public final String
    toString(){
      return "[len=" + length + " accept=" + accept.toString() 
        + " prevAcc=" + prevAccept.toString() 
        + " valid=" + valid 
        + " embargoed=" + embargoed
        + " committed=" + committed
        + "]";
    }

  }


//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerRangeState -- Key + Data for perRangeStateDB
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class PerRangeState{
    private PerRangeKey key;
    private PerRangeData data;

    protected 
    PerRangeState(PerRangeKey k, PerRangeData d){
      key = k;
      data = d;
    }

    protected final PerRangeKey
    getKeyRef(){
      return key;
    }

    protected final PerRangeData
    getDataRef(){
      return data;
    }


  }

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerRangeKeyBinding -- TupleBinding to convert PerRangeKey to
//                             DatabaseEntry
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class PerRangeKeyBinding extends TupleBinding{

//---------------------------------------------------------------------------
// PerRangeKeyBinding -- constructor
//---------------------------------------------------------------------------
    protected PerRangeKeyBinding(){
      super();
    }

//---------------------------------------------------------------------------
// objectToEntry -- Write the PerRangeKey to DatabaseEntry.
// Order is significant (for correct "deserialization" and for sorting).
//---------------------------------------------------------------------------
    public final void 
    objectToEntry(Object o, TupleOutput to){
      PerRangeKey prk = (PerRangeKey)o;
      to.writeLong(prk.getFNum());
      to.writeLong(prk.getOffset());
    }


//---------------------------------------------------------------------------
// entryToObject -- convert DatabaseEntry to PerRangeKey. 
//---------------------------------------------------------------------------
    public final Object 
    entryToObject(TupleInput ti){
      long fileNum;
      long offset;
      PerRangeKey prk;

      fileNum = ti.readLong();
      offset = ti.readLong();
      prk = new PerRangeKey(fileNum, offset);
      return prk;
    }

  }



//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerObjData -- Data for perObjStateDB
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
class PerObjData implements Serializable{
  private long fileNum;
  private double priority;
  private AcceptStamp highestWriteAccept;
  private AcceptStamp highestDeleteAccept;

  protected PerObjData(long fn, double p, AcceptStamp hw, AcceptStamp hd){
    fileNum = fn;
    priority = p;
    highestWriteAccept = (AcceptStamp)hw.clone(); // Immutable --> no copy done
    highestDeleteAccept = (AcceptStamp)hd.clone();// Immutable --> no copy done
  }

  protected final long 
  getFileNum(){
    return fileNum;
  }

  protected final double
  getPriority(){
    return priority;
  }

  protected final AcceptStamp
  getHighestWriteAccept(){
    // Immutable --> no copy done
    return (AcceptStamp)highestWriteAccept.clone();
  }

  protected final void
  advanceHighestWriteAccept(AcceptStamp a){
    assert(a instanceof Immutable);
    assert(a.gt(highestWriteAccept));
    highestWriteAccept = a;
  }

  protected final AcceptStamp
  getHighestDeleteAccept(){
    // Immutable --> no copy done
    return (AcceptStamp)highestDeleteAccept.clone();
  }

  protected final void
  advanceHighestDeleteAccept(AcceptStamp a){
    assert(a instanceof Immutable);
    assert(a.gt(highestDeleteAccept));
    highestDeleteAccept = a;
  }
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerObjStateForSend -- <objId, lastDeleteAccept, lastWriteAccept>
//    of perObjStateDB entry for checkpoint exchange
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class PerObjStateForSend implements Externalizable{
    private ObjId objId;
    private AcceptStamp highestWriteAccept;
    private AcceptStamp highestDeleteAccept;

    protected
    PerObjStateForSend(ObjId objId, AcceptStamp hw, AcceptStamp hd){
      this.objId = objId;
      highestWriteAccept = (AcceptStamp)hw.clone(); // Immutable --> no copy done
      highestDeleteAccept = (AcceptStamp)hd.clone();// Immutable --> no copy done
    }

    //-----------------------------------------------------------------------
    // Constructor for externalization
    //-----------------------------------------------------------------------
    public
    PerObjStateForSend(){
      this.objId = null;
      highestWriteAccept = null;
      highestDeleteAccept = null;
    }

    protected final ObjId 
    getObjId(){
      return objId;
    }

    protected final AcceptStamp
    getHighestWriteAccept(){
      return (AcceptStamp)highestWriteAccept.clone(); // Immutable --> no copy done
    }

    protected final AcceptStamp
    getHighestDeleteAccept(){
      return (AcceptStamp)highestDeleteAccept.clone(); // Immutable --> no copy done
    }

    //-------------------------------------------------------------------------
    // Serialization -- optimization to reduce the pikles in the serialization 
    //                  and reduce the bandwidth for checkpoint exchange
    //-------------------------------------------------------------------------
    public void writeExternal(ObjectOutput out)
      throws IOException{

      out.writeObject(objId.getPath());
      out.writeLong(highestWriteAccept.getLocalClock());
      if(SangminConfig.forkjoin){
        out.writeObject(highestWriteAccept.getNodeId());        
      } else {
        out.writeLong(highestWriteAccept.getNodeId().getIDint());
      }
      out.writeLong(highestDeleteAccept.getLocalClock());
      if(SangminConfig.forkjoin){
        out.writeObject(highestDeleteAccept.getNodeId());
      } else {
        out.writeLong(highestDeleteAccept.getNodeId().getIDint());
      }
  }

  
 
    //-------------------------------------------------------------------------
    // Serialization -- optimization to reduce the pikles in the serialization 
    //                  and reduce the bandwidth for checkpoint exchange
    //-------------------------------------------------------------------------
    public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
      String objIdString = (String)(in.readObject());
      this.objId = new ObjId(objIdString);
      
      long  localClock = in.readLong();
      //long  nodeIdInt = in.readLong();
      NodeId nodeId1;
      if(SangminConfig.forkjoin){
        nodeId1 = (BranchID)in.readObject();
      } else {
        nodeId1 = new NodeId(in.readLong());
      }
      this.highestWriteAccept = new AcceptStamp(localClock, nodeId1);
       
      localClock = in.readLong();
    //nodeIdInt = in.readLong();
      NodeId nodeId2;
      if(SangminConfig.forkjoin){
        nodeId2 = (BranchID)in.readObject();
      } else {
        nodeId2 = new NodeId(in.readLong());
      }      
      this.highestDeleteAccept = new AcceptStamp(localClock, nodeId2);
       
    }
    
    //-----------------------------------------------------------------------
    // Convert to a string
    //-----------------------------------------------------------------------
    public String toString()
    {
	String str = null;

	str = "PerObjStateForSend : <" + this.objId + ", " 
          + this.highestWriteAccept + ", " + this.highestDeleteAccept + ">";
	return(str);
    }

    //-----------------------------------------------------------------------
    // equals()--
    //           compares this object to the specified object. The result
    //           is true if and only if the argument is not null and is 
    //           a PerObjStateForSend object that contains the same members
    //           as this object.
    //-----------------------------------------------------------------------
    public boolean equals(Object obj){
      
      if(! (obj instanceof PerObjStateForSend)){
        return false;  
      }else{
        PerObjStateForSend pos = (PerObjStateForSend)obj;
        if((! this.objId.equals(pos.getObjId()))
           ||(! this.highestWriteAccept.equals(pos.getHighestWriteAccept()))
           ||(! this.highestDeleteAccept.equals(pos.getHighestDeleteAccept()))){
          return false;
        }
        return true;
      }
    }
  }


//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class PerRangeStateForSend -- <objId, offset, length, accept, committed>
// of perRangeStateDB for checkpoint exchange 
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class PerRangeStateForSend implements Externalizable{
    private ObjId objId;
    private long offset;
    private long length;
    private AcceptStamp accept;
    private boolean committed;

    protected 
    PerRangeStateForSend(ObjId objId,
                         long offset,
                         long length,
                         AcceptStamp accept,
                         boolean committed){
      this.objId = objId;
      this.offset = offset;
      this.length = length;
      this.accept = accept;
      this.committed = committed;
    }

   protected
    PerRangeStateForSend(ObjId objId,
                         long offset,
                         long length,
                         AcceptStamp accept){
      this.objId = objId;
      this.offset = offset;
      this.length = length;
      this.accept = accept;
      this.committed = false;
    }


    //-----------------------------------------------------------------------
    // Constructor for externalization
    //-----------------------------------------------------------------------
    public
    PerRangeStateForSend(){
      this.objId = null;
      this.offset = -1L;
      this.length = -1L;
      this.accept = null;
      this.committed = false;
    }

    protected final ObjId
    getObjId(){
      return objId;
    }

    protected final long
    getOffset(){
      return offset;
    }
    
    protected final long
    getLength(){
      return length;
    }

    protected final AcceptStamp
    getAccept(){
      return accept;
    }
    
    protected final boolean
    isCommitted(){
      return committed;
    }

    //-------------------------------------------------------------------------
    // Serialization -- optimization to reduce the pikles in the serialization 
    //                  and reduce the bandwidth for checkpoint exchange
    //-------------------------------------------------------------------------
    public void writeExternal(ObjectOutput out)
      throws IOException{

      out.writeObject(objId.getPath());
      out.writeLong(offset);
      out.writeLong(length);
      out.writeLong(accept.getLocalClock());
      if(SangminConfig.forkjoin){
        out.writeObject(accept.getNodeId()); 
      } else {
        out.writeLong(accept.getNodeId().getIDint());
      }
      out.writeBoolean(committed);
  }

  
 
    //-------------------------------------------------------------------------
    // Serialization -- optimization to reduce the pikles in the serialization 
    //                  and reduce the bandwidth for checkpoint exchange
    //-------------------------------------------------------------------------
    public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
      String objIdString = (String)(in.readObject());
      this.objId = new ObjId(objIdString);
      this.offset = in.readLong();
      this.length = in.readLong();
      long  localClock = in.readLong();
      //long  nodeIdInt = in.readLong();
      NodeId nId;
      if(SangminConfig.forkjoin){
        nId = (BranchID)in.readObject();
      }else{
        nId = new NodeId(in.readLong());
      }
      this.accept = new AcceptStamp(localClock, nId);
      committed = in.readBoolean();
    }
    
    //-----------------------------------------------------------------------
    // Convert to a string
    //-----------------------------------------------------------------------
    public String toString()
    {
	String str = null;

	str = "PerRangeStateForSend : <" + this.objId + ", " 
          + this.offset + ", " + this.length + ", " + this.accept + ", " + this.committed+">";
	return(str);
    }
    
    //-----------------------------------------------------------------------
    // equals()--
    //           compares this object to the specified object. The result
    //           is true if and only if the argument is not null and is 
    //           a PerRangeStateForSend object that contains the same members
    //           as this object.
    //-----------------------------------------------------------------------
    public boolean equals(Object obj){
      
      if(! (obj instanceof PerRangeStateForSend)){
        return false; 
      }else{
        PerRangeStateForSend prs = (PerRangeStateForSend)obj;
        if((! (this.objId.equals(prs.getObjId())))
           ||(! (this.offset == prs.getOffset()))
           ||(! (this.length == prs.getLength()))
           ||(! (this.accept.equals(prs.getAccept())))
           ||( (this.committed != prs.committed))){
          return false;
        }
        return true;
      }
    }
  }
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class ClearChunk -- Helper object to return a bunch of data
//                     elements together in clearNextChunkToInsert()
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class ClearChunk{
    private boolean doApplyMetadata;
    private boolean doApplyData;
    private long  offset;
    private long  len;
    private AcceptStamp prevAccept;

    protected 
    ClearChunk(boolean metadataWins,
               boolean dataToUpdate,
               long o,
               long l,
               AcceptStamp accept){

      if(doApplyMetadata){
        assert(prevAccept != null);
      }
      else{
        assert(prevAccept == null);
      }

      doApplyMetadata = metadataWins;
      doApplyData = dataToUpdate;
      offset = o;
      len = l;
      if(accept != null){
        assert(accept instanceof Immutable);
      }
      prevAccept = accept;   
    }

    protected final boolean
    getDoApplyMetadata(){
      return doApplyMetadata;
    }

    protected final boolean
    getDoApplyBody(){
      return doApplyData;
    }

    protected final long
    getLength(){
      return len;
    }

    protected final long
    getOffset(){
      return offset;
    }

    protected final long
    getNextOffset(){
      return offset + len;
    }

    protected final AcceptStamp
    getPrevAccept(){
      assert(prevAccept instanceof Immutable);

      //
      // It is OK to construct a ClearChunk with prevAccept == null.
      // This happens when doApplyMetadata is false. But,
      // in that case, you should not read prevAccept ever.
      //
      assert(prevAccept != null);
      return prevAccept;
    }

  }


//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
// class ExistingDeleteIsNewerException
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
  class ExistingDeleteIsNewerException extends Exception{
  
//-----------------------------------------------------------------------
// Constructor
//-----------------------------------------------------------------------
    protected ExistingDeleteIsNewerException(){
    }

//-----------------------------------------------------------------------
// Convert this exception to a string
//-----------------------------------------------------------------------
    public String toString(){
      return("ExistingDeleteIsNewerException");
    }

  
  }





//--------------------------------------------------------------------------
/* $Log: RandomAccessState.java,v $
/* Revision 1.64  2008/02/12 20:56:17  nalini
/* *** empty log message ***
/*
/* Revision 1.63  2008/02/05 00:31:00  nalini
/* *** empty log message ***
/*
/* Revision 1.62  2008/01/29 22:16:28  nalini
/* updated stats classes
/*
/* Revision 1.61  2008/01/28 01:40:37  nalini
/* instrumented RAS to gather timing results
/*
/* Revision 1.60  2008/01/23 20:35:27  nalini
/* added null random access state
/*
/* Revision 1.59  2007/07/15 06:21:29  zjiandan
/* optimize bw for cp exchange
/*
/* Revision 1.58  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.57  2007/05/11 02:14:30  nalini
/* updated outgoing body subscription so that it sends bodies from startVV
/*
/* Revision 1.56  2007/05/03 05:34:50  zjiandan
/* *** empty log message ***
/*
/* Revision 1.55  2007/04/05 21:49:25  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.54  2007/04/04 20:17:22  zjiandan
/* fix rmi server cache bug.
/*
/* Revision 1.53  2007/04/02 21:11:39  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.52  2007/03/06 20:57:10  zjiandan
/* optimized RandomAccessState.shipRAS. -- improved a lot even in the case of
/* full scan.
/*
/* Revision 1.50  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.49  2006/12/08 20:58:58  nalini
/* Reverting to old read interface
/*
/* Revision 1.47  2006/10/16 05:44:26  zjiandan
/* Fixed DataStore::applyCheckpoint large lock problem (refer to mike's 2006.10.12.txt), moved IncommingConnection unit tests to junit.
/*
/* Revision 1.46  2006/10/13 22:18:46  dahlin
/* *** empty log message ***
/*
/* Revision 1.45  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.44  2006/08/31 14:54:13  dahlin
/* DataStore restore from checkpoint seems to work now (including ISStatus and cVV)
/*
/* Revision 1.43  2006/08/25 20:31:45  dahlin
/* Added serialization to PreciseSets (plan is to use serialization rather than sync() for persistence). Moved PreciseSet unit tests to junit (PreciseSetUnit)
/*
/* Revision 1.42  2006/08/15 21:46:24  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.41  2006/04/22 22:31:15  zjiandan
/* Completely merged with Runtime.
/*
/* Revision 1.40  2006/04/20 21:41:59  zjiandan
/* add stub to make old code compilng happy.
/*
/* Revision 1.39  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.38  2005/10/17 19:22:29  zjiandan
/* Implemented Non-blocking CP exchange update.
/*
/* Revision 1.37  2005/10/16 04:20:12  zjiandan
/* add "boolean withBody" parameter for Checkpoint exchange.
/*
/* Revision 1.36  2005/10/13 00:24:24  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.35  2005/07/18 05:10:23  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.34  2005/06/14 23:05:08  dahlin
/* I *think* I've fixed the memory leak for the RandomAccessStateUnit tests
/*
/* Revision 1.33  2005/06/13 17:56:23  dahlin
/* Workaround for out of memory error. Most tests pass individually, but if all tests run, later ones fail. Seems to be a problem with not all memory freed across setup/teardown runs. Suspect berkeleydb leak? Workaround is to limit cache size for BerkeleyDB. BUT still have a bug -- memory consumption grows across tests (but more slowly than before.)
/*
/* Revision 1.32  2005/06/10 19:43:20  dahlin
/* Moved RandomAccessState tests to junit
/*
/* Revision 1.31  2005/06/09 17:11:48  dahlin
/* Moving unit tests for RandomAccessState to junit
/*
/* Revision 1.30  2005/06/08 14:14:23  dahlin
/* adding scripts for nightly junit run
/*
/* Revision 1.29  2005/03/22 23:00:03  zjiandan
/* Changes for TactExpt.
/*
/* Revision 1.28  2005/03/16 21:51:17  dahlin
/* RandomAccessState test 18 works with 50MB not with 100MB
/*
/* Revision 1.27  2005/03/16 21:35:24  dahlin
/* Added berekelyDB 1.7.1; added RandomAccessState test 18 stress test memory that fails
/*
/* Revision 1.26  2005/03/09 21:46:35  lgao
/* Fixed amol's error :)
/*
/* Revision 1.25  2005/03/09 21:29:35  nayate
/* Added priority code
/*
/* Revision 1.24  2005/03/07 19:09:11  nayate
/* Added checkpoint filtering by vv
/*
/* Revision 1.23  2005/03/07 07:05:00  nayate
/* Fixed checkpoint shipping/receiving code
/*
/* Revision 1.22  2005/03/04 07:58:44  nayate
/* Minor change
/*
/* Revision 1.21  2005/03/03 19:22:54  nayate
/* Added some checkpoint transfer code
/*
/* Revision 1.20  2005/03/03 08:07:55  nayate
/* Added some checkpoint transfer code
/*
/* Revision 1.19  2005/03/01 10:40:35  nayate
/* First successful compilation
/*
/* Revision 1.18  2005/03/01 06:18:40  nayate
/* Modified for new code
/*
/* Revision 1.17  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.16  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.15  2004/10/22 20:46:55  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.14  2004/10/07 22:20:54  dahlin
/* Fixed some warnings from Findbugs -- none appear to be significant
/*
/* Revision 1.13  2004/10/07 17:09:56  dahlin
/* Cleaned up RandomAccessState and added a couple sanity checks. No functional changes. All self tests (1-16) pass
/*
/* Revision 1.12  2004/10/07 13:51:40  dahlin
/* RandomAccessState passes self tests; about to make a clean-up pass.
/*
/* Revision 1.11  2004/09/24 15:17:35  dahlin
/* Test 15 (performance test) done
/*
/* Revision 1.10  2004/09/24 13:17:57  dahlin
/* RandomAccessState passes through test 14 (including performance tests, random stress test, sync, and delete)
/*
/* Revision 1.9  2004/09/17 20:50:58  dahlin
/* RandomAccessState -- added deletes and tests 7-8 work
/*
/* Revision 1.8  2004/09/15 22:59:53  dahlin
/* Test 6 RandomAccessState works; TBD: delete
/*
/* Revision 1.7  2004/09/10 19:50:10  dahlin
/* RandomAccessState test5 runs
/*
/* Revision 1.6  2004/09/10 16:23:25  dahlin
/* Added optimization to coalesce invals if they have same prevAccept; Passes test 4
/*
/* Revision 1.5  2004/09/10 15:54:49  dahlin
/* RandomAccessState passes test4 (except deliberate tweak of performance bug
/*
/* Revision 1.4  2004/09/08 22:43:20  dahlin
/* Updated RandomAccessState to be more comprehensible; but at present it fails test 2 (endless loop)
/*
/* Revision 1.3  2004/08/25 14:12:31  dahlin
/* Remove causalOrderException from RandomAccessStateLogic b/c hoped-for sanity check logic was wrong (see comments in file)
/*
/* Revision 1.2  2004/08/24 22:21:55  dahlin
/* RandomAccessState self test 3 succeeds
/*
/* Revision 1.1  2004/08/18 22:44:44  dahlin
/* Made BoundInval subclass of PreciseInval; RandomAccessState passes 2 self tests
/*
 */
//--------------------------------------------------------------------------
