package code;
import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;

import com.sleepycat.je.*;
import com.sleepycat.bind.tuple.TupleBinding;
import com.sleepycat.bind.tuple.TupleOutput;
import com.sleepycat.bind.tuple.TupleInput;
import com.sleepycat.bind.serial.StoredClassCatalog;
import com.sleepycat.bind.serial.SerialBinding;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.EOFException;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.lang.IllegalArgumentException;
import java.util.Random;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

/**
 * TBD: Update class name
 */
public class RandomAccessStateUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  private RandomAccessState ras;
  //path = "/var/local/RandomAccessStateTest.db";
  protected static String path = "test" + File.separatorChar + "RandomAccessStateTest.db";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals
  protected static final boolean quick = true;


  //
  // See test 21. Freeing database seems to not
  // completely free cached data. 30MB would 
  // be a more realistic size... Use 2MB to limit
  // cache growth. There still is a problem
  // with memory not being freed across different
  // tests.
  //
  private static long CACHE_SIZE = 2000000;

  /**
   * Basic constructor - called by the test runners.
   */
  public RandomAccessStateUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();

    //
    // wait for previous Unit Tests to be quiet
    //
    Thread.sleep(2000);

    if(verbose){
      System.out.println("RandomAccessStateUnitMT test...");
    }

    System.gc();
    if(true || vverbose){
      memoryReport();
    }

    selfTestCleanup(path); // Start from known state
    ras = new RandomAccessState(path, CACHE_SIZE);
    //ras.overrideDoExpensiveSanityChecks(true);
  }

  protected void tearDown() throws Exception{
    if(ras != null){ // Can be null for test0
      ras.close();
    }
    selfTestCleanup(path);
    if(vverbose){
      System.gc();
      memoryReport();
    }
    super.tearDown();
  }

  public static void memoryReport(){
    System.out.println("TotalMemory: " + Runtime.getRuntime().totalMemory()
		       + " maxMemory: " + Runtime.getRuntime().maxMemory()
		       + " freeMemory: " + Runtime.getRuntime().freeMemory()
      + " usedMemory: " + (Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()));
    if(false){
    	waitForInput();
    }
  }

 /** 
 *  test0 -- create and delete a bunch of RAS instances. Check for 
 *           memory leaks that will thwart junit tests. 
 *  
 *  As of June 2005, this test fails. Something in je is not freeing up  
 *  all resources. Space consumed appears to be 2 bytes per byte 
 *  of max(cached data) of any test (e.g., if we write 10MB of 
 *  data per test, then after the first test 15-20MB are used; 
 *  after the second test, 15-20 are used, etc. (it does not grow with  
 *  each test). 
 *  
 **/ 
  public void test0(){
    int npasses = 5;
    // Sept 2006 -- reduce nfiles from 10K to 1K to 
    // help convince people to run these tests more routinely...
    int nfiles;
    if(quick){
      nfiles = 1000;
    }
    else{
      nfiles = 10000;
    }
    int ii, jj;

    //
    // Should really be some small constant
    // But there appears to be a memory leak in database s.t. 
    // not all of the cache is freed.
    // 
    long MAX_MEM_DELTA = 3*CACHE_SIZE; 
    byte VAL_1 = 1;
    long startMem = Runtime.getRuntime().totalMemory() 
        - Runtime.getRuntime().freeMemory();
    if(verbose){
      System.out.print("Test0...");
    }
    for(ii = 0; ii < npasses; ii++){
      if(vverbose){
        System.out.println("Test0 pass " + ii);
      }
      try{
        checkReadNoSuchEntry("foo");
        for(jj = 0; jj < nfiles; jj++){
          doInval(new Integer(ii*nfiles + jj).toString(), 0, 1000, 4, 4);
          doBody(new String("B" + new Integer(ii*nfiles + jj).toString()), 3000, 1000, 5, 3, VAL_1, 3000);
        }

        ras.close();
        ras = null;
        System.gc();
        selfTestCleanup(path); // Start from known state
        if(vverbose){
          memoryReport();
        }

	//
	// zjd --
	//   the following check failed frequently in a random fashion
	//   try to give it a little bit time to free memory
	//

	try{
	  Thread.currentThread().sleep(2000);
        }catch(InterruptedException ie) {
	  //ok
	}

        long endMem = Runtime.getRuntime().totalMemory() 
            - Runtime.getRuntime().freeMemory();
        if(endMem - startMem > MAX_MEM_DELTA){
          fail("Possible memory leak in RandomAccessState: pass = " + ii 
               + " start=" + startMem
               + " end=" + endMem);
        }
        ras = new RandomAccessState(path, 30000000);
      }
      catch(Exception e){
        fail("Unable to run RandomAccessStateUnit test 0 " + e.toString());
      }
    }
    if(verbose){
      System.out.print("OK. ");
    }
  }

  public static void waitForInput(){
    try{
      while(System.in.available() > 0){
        byte b[] = new byte[1024];
        System.in.read(b);
      }
      System.out.println("\n\nPRESS ANY KEY TO CONTINUE");
      byte b[] = new byte[1];
      System.in.read(b);
    }
    catch(IOException e){
      fail("waitForInput turned on in off-line mode?");
    }
  }
 /** 
 *  test1 -- read non-existing file --> no such object exception 
 **/ 
    public void test1(){
      if(verbose){
        System.out.print("Test1...");
      }
      ObjId id = new ObjId("/test1/bar");
      long offset = 0;
      long maxLen = 999;
      try{
        ras.read(id, offset, maxLen, true);
        assert(false);
      }
      catch(NoSuchEntryException expected){
        if(verbose){
          System.out.print("OK. ");
        }
        return; // Success
      }
      catch(Exception e){
        assert(false);
      }
    }

 /** 
 *  test1b -- read non-existing file --> no such object exception 
 **/ 
    public void test1b(){
      if(verbose){
        System.out.print("Test1b...");
      }
      ObjId id = new ObjId("/test1b/bar");
      long offset = 0;
      try{
        ras.readMeta(id, offset);
        assert(false);
      }
      catch(NoSuchEntryException expected){
        if(verbose){
          System.out.print("OK. ");
        }
        return; // Success
      }
      catch(Exception e){
        assert(false);
      }
    }
  
 /** 
 *  test2 -- test global state (fnum) and global vv by creating a bunch 
 *  of files and peeking at the ending status 
 **/ 
  public void test2(){
    int NFILES = 1000;
    ObjId id;
    ObjInvalTarget target;
    long offset = 0;
    long len = 1;
    PreciseInv pi;
    AcceptStamp accept;
    long localClock;
    NodeId nodeId;
    long node;
    int MAX_NODE = 10;
    int MAX_CLOCK = 1000;
    long checkVV[] = new long[MAX_NODE];
    long startFNum, endFNum;
    Random rng = new Random(4357);
    int ii;
    boolean test2Verbose = false;
    
    if(verbose){
      System.out.print("Test2...");
    }
    for(ii = 0; ii < MAX_NODE; ii++){
      checkVV[ii] = AcceptStamp.BEFORE_TIME_BEGAN;
    }

    try{
      startFNum = ras.makeNewFNum();
      for(ii = 0; ii < NFILES; ii++){
        id = new ObjId("/test2/" + ii);
        target = new ObjInvalTarget(id, offset, len);
        localClock = rng.nextLong();
        if(localClock < 0){
          localClock = localClock * -1;
        }
        localClock = localClock % MAX_CLOCK;
        node = rng.nextLong();
        if(node < 0){
          node = node * -1;
        }
        node = node % MAX_NODE;
        nodeId = new NodeId(node);
        assert(nodeId.isValidAndConcrete());
        accept = new AcceptStamp(localClock, nodeId);
        pi = new PreciseInv(target, accept);
        if(verbose && test2Verbose){
          System.out.println("Test 2: ApplyInval " + pi.toString());
        }
        ras.applyInval(pi);
        if(localClock > checkVV[(int)node]){
          checkVV[(int)node] = localClock;
        }

        //
        // Try to read metadata what we just wrote
        //
        try{
          pi = ras.readMeta(id, offset);
          assert(pi.getObjId().equals(id));
          assert(pi.getObjInvalTarget().getOffset() == offset);
          assert(pi.getObjInvalTarget().getLength() == 1);
        }
        catch(NoSuchEntryException nse){
          assert(false);
        }
        catch(EOFException eof){
          assert(false);
        }
	catch(ReadOfHoleException rhe){
	  assert(false);
	}


        //
        // Try to read metadata after what we just wrote
        //
        try{
          pi = ras.readMeta(id, offset + 1);
          assert(false);
        }
        catch(NoSuchEntryException nse){
          assert(false);
        }
        catch(EOFException eof){
          assert(true); // This is what we expect to happen
        }
	catch(ReadOfHoleException rhe){
	  assert(false);
	}

        //
        // Try to read what we just wrote
        //
        try{
          BodyMsg m = ras.read(id, offset, 100, true);
          assert(false);
        }
        catch(NoSuchEntryException nse){
          assert(false);
        }
        catch(EOFException eof){
          assert(false);
        }
        catch(ReadOfEmbargoedWriteException roewe){
          assert(false);
        }
        catch(ReadOfInvalidRangeException roire){
          assert(true);  // This is what we expect to happen
        }
	catch(ReadOfHoleException rhe){
	  assert false;
	}
	
        
      }
  
      //
      // Now verify that state is as expected
      //
      endFNum = ras.makeNewFNum();
      if(!(endFNum - startFNum == NFILES + 1)){
        fail("Test2 fails: start:" + startFNum
             + " end: " + endFNum
             + " NFILES: " + NFILES);
      }
      for(ii = 0; ii < MAX_NODE; ii++){
        if(!(checkVV[(int)ii] == ras.getVV(null, new NodeId(ii)))){
          fail("Test2 fails: " + ii + " checkVV[] = " 
               + checkVV[(int)ii] + " rasVV[] = "
               + ras.getVV(null, new NodeId(ii)));
        }
      }
      if(verbose){
        System.out.print("OK. ");
      }
      return; // Success!
    }
    catch(DatabaseException e){
      e.printStackTrace();
      fail();
    }
  }



 /** 
 *  test3 -- test range logic 
 *   o invalidate bytes 100-199, 300-399 of file "/test3/foo" 
 *      o read bytes 0-99 --> 0  
 *      o read bytes 100-199 --> ReadOfInvalidRangeException 
 *      o read bytes 200-299 --> 0 
 *      o read bytes 300-399 --> ReadOfInvalidRangeException 
 *      o read bytes 400+ --> EOFException 
 *      o read a bunch of non-existant files and offsets --> no such 
 *        object exception 
 *      o do all of above for readMeta() 
 **/ 
  public void test3(){
    ObjId id;
    ObjInvalTarget target;
    ObjInvalTarget oit;
    PreciseInv pi;
    NodeId nodeId;
    long firstStart = 0;
    long len = 100;
    long secondStart = 100;
    long secondLocal = 1000;
    long thirdStart = 200;
    long fourthStart = 300;
    long fourthLocal = 1001;
    long fifthStart = 400;
    long offset;
    BodyMsg ret;
    ImmutableBytes ib;
    byte b[];
    AcceptStamp accept;
    int ii;

    if(verbose){
      System.out.print("Test3");
    }
    id = new ObjId("/test3/foo");
    nodeId = new NodeId(38);
    target = new ObjInvalTarget(id, secondStart, len);
    accept = new AcceptStamp(secondLocal, nodeId);
    pi = new PreciseInv(target, accept);
    ras.applyInval(pi);
  
    target = new ObjInvalTarget(id, fourthStart, len);
    accept = new AcceptStamp(fourthLocal, nodeId);
    pi = new PreciseInv(target, accept);
    ras.applyInval(pi);

    //
    // Read bytes 0-99 --> "0"
    //
    try{
      for(ii = 0; ii < len; ii++){
        offset = firstStart + ii;
	//add by zjd start:
	try{
	  ret = ras.read(id, offset, len, true);
	}catch (ReadOfHoleException e){
	  //expected;
	  assert e.getNextValidBytePosition() == secondStart:
	  "ReadOfHoleException:"+ e.getNextValidBytePosition() + " expect:"+secondStart;
	  continue;
	}
	assert false;
	//end by zjd

	/*commented by zjd for new readhole interface
        ret = ras.read(id, offset, len, true);
        oit = ret.getObjInvalTarget();
        assert(oit.getObjId().equals(id));
        assert oit.getOffset() == offset: "Ret: " + oit.getOffset() + " expect " + offset;
        assert(oit.getLength() == 1);
        ib = ret.getBody();
        b = ib.getCopyBytes();
        assert(b.length == 1);
        assert(b[0] == 0);
        accept = ret.getAcceptStamp();
        assert(accept.equals(RandomAccessState.acceptOfHole));
	*/
      }
    }    
    catch(Exception e){
      fail();
    }
    for(ii = 0; ii < len; ii++){
      offset = firstStart + ii;
      //add by zjd start:
      try{
	ras.readMeta(id, offset);
      }catch (ReadOfHoleException ee){
	//expected;
	assert ee.getNextValidBytePosition() == secondStart:
	"ReadOfHoleException:"+ ee.getNextValidBytePosition() + " expect:"+secondStart;
	continue;
      }catch(Exception e){
	assert false;
      }
      assert false;
      //end by zjd
      
      //commented by zjd
      //checkReadMeta(ras, id, offset, offset, 1, ras.acceptOfHole);
    }

    //
    // Read bytes 100-199 --> read of invalid rage exceptione
    //
    for(ii = 0; ii < len; ii++){
      offset = secondStart + ii;
      try{
        ret = ras.read(id, offset, len, true);
        fail();
      }
      catch(ReadOfInvalidRangeException roire){
        ; // Expect this -- continue
      }
      catch(Exception e2){
        fail();
      }
    }
    for(ii = 0; ii < len; ii++){
      offset = secondStart + ii;
      checkReadMeta(ras, id, offset, secondStart, len, 
                    new AcceptStamp(secondLocal, nodeId));
    }
    
    //
    // Read bytes 200-299 --> "0"
    //
    try{
      for(ii = 0; ii < len; ii++){
        offset = thirdStart + ii;
		//add by zjd start:
	try{
	  ret = ras.read(id, offset, len, true);
	}catch (ReadOfHoleException e){
	  //expected;
	  assert e.getNextValidBytePosition() == fourthStart:
	  "ReadOfHoleException:"+ e.getNextValidBytePosition() + " expect:"+fourthStart;
	  continue;
	}catch(Exception ee){
	  assert false;
	}
	assert false;
	//end by zjd

	/*commented by zjd for new readhole interface
        ret = ras.read(id, offset, len, true);
        oit = ret.getObjInvalTarget();
        assert(oit.getObjId().equals(id));
        assert(oit.getOffset() == offset);
        assert(oit.getLength() == 1);
        ib = ret.getBody();
        b = ib.getCopyBytes();
        assert(b.length == 1);
        assert(b[0] == 0);
        accept = ret.getAcceptStamp();
        assert(accept.equals(ras.acceptOfHole));
	*/
      }
    }    
    catch(Exception e3){
      assert(false);
    }
    for(ii = 0; ii < len; ii++){
      offset = thirdStart + ii;
      //add by zjd start:
      try{
	ras.readMeta(id, offset);
      }catch (ReadOfHoleException e){
	//expected;
	assert e.getNextValidBytePosition() == fourthStart:
	"ReadOfHoleException:"+ e.getNextValidBytePosition() + " expect:"+fourthStart;
	continue;
      }catch (Exception ee){
	assert false;
      }
      assert false;
      //end by zjd
      
      //commented by zjd
      //checkReadMeta(ras, id, offset, offset, 1, ras.acceptOfHole);
    }
    
    //
    // Read bytes 300-399 --> read of invalid rage exceptione
    //
    for(ii = 0; ii < len; ii++){
      offset = fourthStart + ii;
      try{
        ret = ras.read(id, offset, len, true);
        assert(false);
      }
      catch(ReadOfInvalidRangeException roire2){
        ; // Expect this -- continue
      }
      catch(Exception e4){
        assert(false);
      }
    }
    for(ii = 0; ii < len; ii++){
      offset = fourthStart + ii;
      checkReadMeta(ras, id, offset, fourthStart, len, 
                    new AcceptStamp(fourthLocal, nodeId));
    }
    
    //
    // Read bytes 400+ --> eof exception
    //
    for(ii = 0; ii < len; ii++){
      offset = fifthStart + ii;
      try{
        ret = ras.read(id, offset, len, true);
        assert(false);
      }
      catch(EOFException eofe){
        ; // Expect this -- continue
      }
      catch(Exception e5){
        assert(false);
      }
    }
    for(ii = 0; ii < len; ii++){
      offset = fifthStart + ii;
      checkReadMetaEOF(ras, id, offset);
    }
    
    if(verbose){
      System.out.print("...OK. ");
    }
  }


 /** 
 *  test4 -- check body updates 
 **/ 
    public void
    test4(){
      ObjId id;
      ObjInvalTarget target;
      ObjInvalTarget oit;
      PreciseInv pi;
      NodeId nodeId;
      if(verbose){
        System.out.print("Test 4.");
      }
      byte SHOULD_NOT_SEE = 127;
      byte VAL_1 = 1;
      byte VAL_2 = 122;
      byte VAL_3 = 39;
      byte VAL_4 = 94;
      byte VAL_5 = 55;
      long ret;

      doInval("4", 0, 1000, 4, 4);
      doBody("4", 0, 100, 3, 4, SHOULD_NOT_SEE, ras.NO_NEW_VALID_BYTES);
      checkReadInvalidRange("4", 0, 100);
      checkReadInvalidRange("4", 50, 100);
      checkReadInvalidRange("4", 100, 100);
      doBody("4", 0, 100, 4, 3, SHOULD_NOT_SEE, ras.NO_NEW_VALID_BYTES);
      checkReadInvalidRange("4", 0, 100);
      checkReadInvalidRange("4", 50, 100);
      checkReadInvalidRange("4", 100, 100);
      doBody("4", 0, 1000, 4, 4, VAL_1, 0);
      checkRead("4", 0, 2000, 1000, VAL_1, true);
      checkRead("4", 50, 2000, 950, (byte)(VAL_1 + 50), true);
      doBody("4", 0, 100, 4, 3, SHOULD_NOT_SEE, ras.NO_NEW_VALID_BYTES);
      checkRead("4", 0, 2000, 1000, VAL_1, true);
      checkRead("4", 50, 2000, 950, (byte)(VAL_1 + 50), true);
      doBody("4", 0, 100, 3, 4, SHOULD_NOT_SEE, ras.NO_NEW_VALID_BYTES);
      checkRead("4", 0, 2000, 1000, VAL_1, true);
      checkRead("4", 50, 2000, 950, (byte)(VAL_1 + 50), true);
      doInval("4", 600, 100, 5, 5);
      doInval("4", 700, 100, 5, 6);
      doInval("4", 800, 100, 5, 7);
      doInval("4", 900, 100, 3, 3);
      doBound("4", 500, 100, 9, 9, VAL_3, 500);
      doBound("4", 0, 1000, 6, 5, VAL_4, 0);
      checkRead("4", 0, 1000, 500, VAL_4, true);
      checkRead("4", 100, 1000, 400, (byte)(VAL_4 + 100), true);
      checkRead("4", 500, 1000, 100, VAL_3, true);
      checkRead("4", 550, 1000, 50, (byte)(VAL_3 + 50), true);
      // Comes back in 100 byte chunks b/c different prevAccept
      checkRead("4", 600, 1000, 100, (byte)(VAL_4 + 600), true);
      checkRead("4", 700, 1000, 100, (byte)(VAL_4 + 700), true);
      checkRead("4", 800, 1000, 100, (byte)(VAL_4 + 800), true);
      checkRead("4", 900, 1000, 100, (byte)(VAL_4 + 900), true);

      doBound("4", 0, 900, 8, 8, VAL_5, 0);
      checkRead("4", 0, 1000, 500, VAL_5, true);
      //
      // The following fails b/c we don't coalesce writes with
      // same prevAccept. This is a performance bug possibly worth
      // fixing -- otherwise we potentially get arbitrarily bad 
      // fragmentation
      //
      checkRead("4", 600, 1000, 300, (byte)(VAL_5 + 600), true);
      // Make sure coalesce invals in clearNextChunkToInsert stopped at right place
      checkRead("4", 900, 1000, 100, (byte)(VAL_4 + 900), true);
      if(verbose){
        System.out.print("..OK. ");
      }
    }

 /** 
 *  test5 -- random overlapping writes 
 **/ 
    public void
    test5(){
      int ii;
      if(verbose){
        System.out.print("Test 5.");
      }
      for(ii = 0; ii < 20; ii++){
        test5Worker(ras, ii);
      }
      if(verbose){
        System.out.print("Test 5 Succeeds.");
      }
    }

 /** 
 *  test5Worker -- random overlapping writes 
 **/ 
    private void
    test5Worker(RandomAccessState ras, int pass){
      byte SHOULD_NOT_SEE = 127;
      byte VAL_1 = 1;
      byte VAL_2 = 122;
      byte VAL_3 = 39;
      byte VAL_4 = 94;
      byte VAL_5 = 55;
      byte expectVal;
      long offset;
      long len;
      long offset2;
      long len2;
      long expectOffset;
      long MAX_WRITE_SIZE = 100;
      long FILE_SIZE = 1000;
      long ii;
      String fileName = "5" + "-" + pass;

      if(vverbose){
        System.out.print(".");
      }
      //
      // do 1 write over entire range value 1
      //
      doBody(fileName, 0, FILE_SIZE, 0, 0, VAL_1, 0);

      //
      // do 30 random writes of size (1..100) with increasing (1, i) 
      // timestamp and value 1
      //
      for(ii = 0; ii < 30; ii++){
        offset = random(0, FILE_SIZE - 1);
        len = random(1, MAX_WRITE_SIZE);
        if(len + offset > FILE_SIZE){
          len = FILE_SIZE - offset;
          assert(len > 0);
        }
        doBody(fileName, offset, len, 1, ii, (byte)(VAL_1 + offset), offset);
      }

      //
      // do 1 write of size (1..100) with value 2 with 
      // higher timestamp (3, 1); 
      //
      offset2 = random(0, FILE_SIZE-1);
      len2 = random(1, MAX_WRITE_SIZE);
      if(len2 + offset2 > FILE_SIZE){
        len2 = FILE_SIZE - offset2;
        assert(len2 > 0);
      }
      doBody(fileName, offset2, len2, 3, 1, VAL_2, offset2);
  

      //
      // do 30 random writes with timestamp between first and second
      // step (2, i) value 1; this write will win unless it intersects
      // the write at <offset2, offset2 + len2>
      //
      for(ii = 0; ii < 30; ii++){
        offset = random(0, FILE_SIZE - 1);
        len = random(1, MAX_WRITE_SIZE);
        if(len + offset > FILE_SIZE){
          len = FILE_SIZE - offset;
          assert(len > 0);
        }
        expectOffset = offset;
        if(offset >= offset2 && offset < offset2 + len2){
          if(offset + len <= offset2 + len2){
            expectOffset = ras.NO_NEW_VALID_BYTES;
          }
          else{
            expectOffset = offset2 + len2;
          }
        }
        doBody(fileName, offset, len, 2, ii, 
                       (byte)(VAL_1 + offset), expectOffset);
      }

      //
      // verify read of the (3, 1) write gives 2
      // verify read of all remaining bytes gives 1
      //
      for(offset = 0; offset < FILE_SIZE; offset++){
        if(offset >= offset2 && offset < offset2 + len2){
          expectVal = (byte)(VAL_2 + offset - offset2);
        }
        else{
          expectVal = (byte)(VAL_1 + offset);
        }
        checkRead(fileName, offset, 1, 1, expectVal, true);
      }


    }



 /** 
 *  test6 -- check overlapping logic 
 **/ 
    public void
    test6(){
      byte VAL_1 = 1;
      byte VAL_2 = 122;
      byte VAL_3 = 39;
      byte VAL_4 = 94;
      byte VAL_5 = 55;

      if(verbose){
        System.out.print("Test 6...");
      }
      //
      // New overwrites old
      //

      if(vverbose){
        System.out.print(".");
      }
      // Exact overlap
      doBody("6-1", 43, 99, 1, 1, VAL_1, 43);
      doBody("6-1", 43, 99, 2, 1, VAL_2, 43);
      checkRead("6-1", 43, 100, 99, VAL_2, true);
      checkReadHole("6-1", 42);
      checkReadEOF("6-1", 142);


      // Starts earlier overlap; starts later no overlap,
      // starts earlier no overlap
      if(vverbose){
        System.out.print(".");
      }
      doBody("6-2", 100, 100, 1, 1, VAL_1, 100);
      doBody("6-2", 50, 100, 2, 1, VAL_2, 50);
      doBody("6-2", 200, 100, 3, 1, VAL_3, 200);
      doBody("6-2", 25, 25, 4, 1, VAL_4, 25);
      checkRead("6-2", 25, 100, 25, VAL_4, true);  
      checkRead("6-2", 50, 100, 50, VAL_2, true);
      checkRead("6-2", 75, 100, 25, (byte)(VAL_2 + 25), true);
      checkRead("6-2", 100, 100, 50, (byte)(VAL_2 + 50), true);  
      checkRead("6-2", 125, 100, 25, (byte)(VAL_2 + 75), true);
      checkRead("6-2", 150, 100, 50, (byte)(VAL_1 + 50), true);  
      checkRead("6-2", 175, 100, 25, (byte)(VAL_1 + 75), true);  
      checkRead("6-2", 200, 100, 100, VAL_3, true);
      checkRead("6-2", 225, 100, 75, (byte)(VAL_3 + 25), true);

      if(vverbose){
        System.out.print(".");
      }
      // Ends later overlap -- overwrite all of the
      // prev writes; result comes back in pieces b/c
      // different prev-accept
      doBody("6-2", 0, 1000, 5, 1, VAL_5, 0);
      checkRead("6-2", 0, 1000, 25, VAL_5, true);  
      checkRead("6-2", 25, 1000, 25, (byte)(VAL_5 + 25), true);  
      checkRead("6-2", 50, 1000, 100, (byte)(VAL_5 + 50), true);  
      checkRead("6-2", 100, 1000, 50, (byte)(VAL_5 + 100), true);  
      checkRead("6-2", 150, 1000, 50, (byte)(VAL_5 + 150), true);  
      checkRead("6-2", 200, 1000, 100, (byte)(VAL_5 + 200), true);  
      checkRead("6-2", 300, 1000, 700, (byte)(VAL_5 + 300), true);  

      if(vverbose){
        System.out.print(".");
      }
      // Overwrite prev write. Result comes back as one chunk
      // b/c all have same prev-accept
      doBody("6-2", 0, 1000, 6, 1, VAL_1, 0);
      checkRead("6-2", 0, 3000, 1000, VAL_1, true);  

  
      if(vverbose){
        System.out.print(".");
      }
      // off by 1 bug -- not overlap {start, end} by 1 byte, overlap
      //                 {start,end} by 1 byte
      doBody("6-3", 0, 100, 1, 1, VAL_1, 0);
      doBody("6-3", 101, 99, 2, 1, VAL_2, 101);
      checkRead("6-3", 0, 1000, 100, VAL_1, true);  
      checkRead("6-3", 101, 1000, 99, VAL_2, true);
      checkReadHole("6-3", 100);

      //
      // Old overwrites new
      //
      if(vverbose){
        System.out.print(".");
      }
      doBody("6-4", 43, 99, 2, 1, VAL_1, 43);
      doBody("6-4", 43, 99, 1, 1, VAL_2, ras.NO_NEW_VALID_BYTES);
      checkRead("6-4", 43, 100, 99, VAL_1, true);
      checkReadHole("6-4", 42);
      checkReadEOF("6-4", 142);

      // Starts earlier overlap; starts later no overlap,
      // starts earlier no overlap
      if(vverbose){
        System.out.print(".");
      }
      doBody("6-5", 25, 25, 4, 1, VAL_4, 25);
      doBody("6-5", 200, 100, 3, 1, VAL_3, 200);
      doBody("6-5", 50, 100, 2, 1, VAL_2, 50);
      doBody("6-5", 100, 100, 1, 1, VAL_1, 150);
      checkRead("6-5", 25, 100, 25, VAL_4, true);  
      checkRead("6-5", 50, 100, 100, VAL_2, true);
      checkRead("6-5", 75, 100, 75, (byte)(VAL_2 + 25), true);
      checkRead("6-5", 100, 100, 50, (byte)(VAL_2 + 50), true);  
      checkRead("6-5", 125, 100, 25, (byte)(VAL_2 + 75), true);
      checkRead("6-5", 150, 100, 50, (byte)(VAL_1 + 50), true);  
      checkRead("6-5", 175, 100, 25, (byte)(VAL_1 + 75), true);  
      checkRead("6-5", 200, 100, 100, VAL_3, true);
      checkRead("6-5", 225, 100, 75, (byte)(VAL_3 + 25), true);


      // new overwrites old old: starts before both, starts middle first, 
      // ends middle second, ends after second  
      if(vverbose){
        System.out.print(".");
      }
      doBody("6-6", 25, 25, 1, 1, VAL_1, 25);
      doBody("6-6", 50, 25, 2, 1, VAL_2, 50);
      doBody("6-6", 0, 100, 3, 1, VAL_3, 0);
      checkRead("6-6", 0, 100, 25, VAL_3, true);  
      checkRead("6-6", 25, 100, 25, (byte)(VAL_3+25), true);  
      checkRead("6-6", 50, 100, 25, (byte)(VAL_3+50), true);  
      checkRead("6-6", 75, 100, 25, (byte)(VAL_3+75), true);  
      doBody("6-7", 25, 25, 1, 1, VAL_1, 25);
      doBody("6-7", 50, 25, 2, 1, VAL_2, 50);
      doBody("6-7", 40, 30, 3, 1, VAL_3, 40);
      checkRead("6-7", 25, 100, 15, VAL_1, true);  
      checkRead("6-7", 40, 100, 10, (byte)(VAL_3), true);  
      checkRead("6-7", 50, 100, 20, (byte)(VAL_3+10), true);  
      checkRead("6-7", 70, 100, 5, (byte)(VAL_2+20), true);


      // new overwrites old gap old: 
      if(vverbose){
        System.out.print(".");
      }
      doBody("6-8", 25, 25, 1, 1, VAL_1, 25);
      doBody("6-8", 75, 25, 2, 1, VAL_2, 75);
      doBody("6-8", 0, 200, 3, 1, VAL_3, 0);
      checkRead("6-8", 0, 200, 25, VAL_3, true);  
      checkRead("6-8", 25, 200, 25, (byte)(VAL_3+25), true);  
      checkRead("6-8", 50, 200, 25, (byte)(VAL_3+50), true);  
      checkRead("6-8", 75, 200, 25, (byte)(VAL_3+75), true);  
      checkRead("6-8", 100, 200, 100, (byte)(VAL_3+100), true);  

      doBody("6-9", 25, 25, 1, 1, VAL_1, 25);
      doBody("6-9", 75, 25, 2, 1, VAL_2, 75);
      doBody("6-9", 40, 160, 3, 1, VAL_3, 40);
      checkRead("6-9", 25, 200, 15, VAL_1, true);  
      checkRead("6-9", 40, 200, 10, (byte)(VAL_3), true);  
      checkRead("6-9", 50, 200, 25, (byte)(VAL_3+10), true);  
      checkRead("6-9", 75, 200, 25, (byte)(VAL_3+35), true);  
      checkRead("6-9", 100, 200, 100, (byte)(VAL_3+60), true);  



      if(verbose){
        System.out.print("Test 6 Done.");
      }

    }



 /** 
 *  test7 -- simple delete tests 
 **/ 
  public void
  test7(){
    if(verbose){
      System.out.print("Test7...");
    }
    byte VAL_1 = 1;
    doBody("7-1", 0, 1000, 1, 1, VAL_1, 0);
    checkRead("7-1", 0, 1000, 1000, VAL_1, true);  
    doDelete("7-1", 2, 2);
    checkReadNoSuchEntry("7-1");

    doBody("7-2", 0, 1000, 3, 1, VAL_1, 0);
    doBody("7-2", 1000, 1000, 3, 2, VAL_1, 1000);
    doBody("7-2", 3000, 1000, 3, 3, VAL_1, 3000);
    doDelete("7-2", 4, 4);
    checkReadNoSuchEntry("7-2");

    doBody("7-3", 0, 1000, 5, 1, VAL_1, 0);
    doBody("7-3", 1000, 1000, 5, 2, VAL_1, 1000);
    doBody("7-3", 3000, 1000, 5, 3, VAL_1, 3000);
    doInval("7-3", 2000, 1000, 5, 4);
    doInval("7-3", 9000, 1000, 5, 5);
    doDelete("7-3", 6, 6);
    checkReadNoSuchEntry("7-3");

    ObjId id = new ObjId("/test/7-4");
    doBody("7-4", 0, 1000, 5, 1, VAL_1, 0);
    doBody("7-4", 1000, 1000, 5, 2, VAL_1, 1000);
    doBody("7-4", 3000, 1000, 5, 3, VAL_1, 3000);
    doInval("7-4", 2000, 1000, 5, 4);
    doInval("7-4", 9000, 1000, 5, 5);
    doInval("7-4", 1200, 300, 5, 6);
    doInval("7-4", 1503, 32, 5, 7);
    doBody("7-4", 1300, 100, 5, 8, VAL_1, 1300);
    doInval("7-4", 200, 300, 5, 9);
    doInval("7-4", 503, 32, 5, 10);
    doInval("7-4", 593, 32, 5, 11);
    doInval("7-4", 500, 3, 5, 12);
    doBody("7-4", 250, 252, 5, 13, VAL_1, 250);
    doDelete("7-4", 6, 6);
    checkReadNoSuchEntry("7-4");

    if(verbose){
      System.out.print("..Test7 OK.");
    }
  }

 /** 
 *  test8 -- delete all of the files we have created so far 
 *   
 *  Since Junit fixture cleans up before/after each test, we need 
 *  to create these files first... 
 **/ 
    public void
    test8(){
      String fileList[] = {"/test1/bar", "/test3/foo", "/test/4", "/test/6-1",
                           "/test/6-2", "/test/6-3", "/test/6-4", "/test/6-5", "/test/6-6",
                           "/test/6-7", "/test/6-8", "/test/6-9"};
      String fileList2[] = new String[1000];
      String fileList5[] = new String[20];
      int hh, ii, jj;
      NodeId node = new NodeId(999);
      AcceptStamp accept = new AcceptStamp(Long.MAX_VALUE - 1, node);
      ObjId id;
      BodyMsg read;
      int rcountMax;
      int fl1;
      int fl2;
      if(quick){
        rcountMax = 3;
        fl1 = 10;
        fl2 = 2;
      }
      else{
        rcountMax = 100;
        fl1 = 1000;
        fl2 = 20;
      }

      if(verbose){
        System.out.print("Test 8...");
      }
      for(ii = 0; ii < fl1; ii++){
        fileList2[ii] = "/test2/" + ii;
      }
      for(ii = 0; ii < fl2; ii++){
        fileList5[ii] = "/test/5-" + ii;
      }


      //
      // None of the files should currently exist.
      //
      if(vverbose){
        System.out.println("8-check-pre");
      }
      for(ii = 0; ii < fileList.length; ii++){
        id = new ObjId(fileList[ii]);
        try{
	  if(id.getPath() != null){
            read = ras.read(id, 0, 1, true);
            assert(false);
          }
        }
        catch(NoSuchEntryException nseez){
          ; // This is what is expected
        }
        catch(Exception ez){
          fail();
        }
      }

      for(ii = 0; ii < fileList2.length; ii++){
        id = new ObjId(fileList2[ii]);
        try{
	  if(id.getPath() != null){
	    read = ras.read(id, 0, 1, true);
	    assert(false);
	  }
        }
        catch(NoSuchEntryException nseez2){
          ; // This is what is expected
        }
        catch(Exception ez2){
          fail();
        }
      }

      for(ii = 0; ii < fileList5.length; ii++){
        id = new ObjId(fileList5[ii]);
        try{
	  if(id.getPath() != null){
	    read = ras.read(id, 0, 1, true);
	    assert(false);
	  }
        }
        catch(NoSuchEntryException nseez3){
          ; // This is what is expected
        }
        catch(Exception ez3){
          fail();
          
        }
      }


      //
      // Now, create the files. Do a random number of writes to 
      // each file. Make a couple passes over the whole
      // data set to make sure things are well mixed.
      //
      if(vverbose){
        System.out.println("8-create-pre");
      }
      for(hh = 0; hh < 3; hh++){
        if(vverbose){
          System.out.println("8-create-" + hh);
        }
        for(ii = 0; ii < fileList.length; ii++){
          id = new ObjId(fileList[ii]);
          long rcount = random(1, rcountMax);
          for(jj = 0; jj < rcount; jj++){
            doBodyNoCheckReturn(fileList[ii], random(0, 10000), random(1,1000), 
                                random(0, 999999), 
                                random(0, 10), (byte)42);
          }
        }
        for(ii = 0; ii < fileList2.length; ii++){
          id = new ObjId(fileList2[ii]);
          long rcount = random(1, rcountMax);
          for(jj = 0; jj < rcount; jj++){
            doBodyNoCheckReturn(fileList2[ii], random(0, 10000), random(1,1000), 
                                random(0, 999999), 
                                random(0, 10), (byte)42);
          }
        }
        for(ii = 0; ii < fileList5.length; ii++){
          id = new ObjId(fileList5[ii]);
          long rcount = random(1, rcountMax);
          for(jj = 0; jj < rcount; jj++){
            doBodyNoCheckReturn(fileList5[ii], random(0, 10000), random(1,1000), 
                                random(0, 999999), 
                                random(0, 10), (byte)42);
          }
        }
      }

      //
      // Finally, the delete test itself!
      //

      if(vverbose){
        System.out.println("8-delete-pre");
      }
      for(ii = 0; ii < fileList.length; ii++){
        id = new ObjId(fileList[ii]);
        checkReadExists(fileList[ii]);
	if(id.getPath() != null){
	  ras.delete(id, accept);
	  try{
	    read = ras.read(id, 0, 1, true);
	    assert(false);
	  }
	
	  catch(NoSuchEntryException nsee){
	    ; // This is what is expected
	  }
	  catch(Exception e){
	    fail();
	  }
	}
      }

      for(ii = 0; ii < fileList2.length; ii++){
        id = new ObjId(fileList2[ii]);
        checkReadExists(fileList2[ii]);
	if(id.getPath() != null){
	  ras.delete(id, accept);
	  try{
	    read = ras.read(id, 0, 1, true);
	    assert(false);
	  }
	
	  catch(NoSuchEntryException nsee2){
	    ; // This is what is expected
	  }
	  catch(Exception e2){
	    fail();
	  }
	}
      }

      for(ii = 0; ii < fileList5.length; ii++){
        id = new ObjId(fileList5[ii]);
        checkReadExists(fileList5[ii]);
	if(id.getPath() != null){
	  ras.delete(id, accept);
	  try{
	    read = ras.read(id, 0, 1, true);
	    assert(false);
	  }
	  catch(NoSuchEntryException nsee3){
	    ; // This is what is expected
	  }
	  catch(Exception e3){
	    fail();
	  }
	}
      }
      if(verbose){
        System.out.print("...Test 8 OK.");
      }
      
    }
  

 /** 
 *  test9 -- check overlapping logic -- repeat test 6 but change some bound 
 *           writes to unbound invals 
 **/ 
    public void
    test9(){
      byte VAL_1 = 1;
      byte VAL_2 = 122;
      byte VAL_3 = 39;
      byte VAL_4 = 94;
      byte VAL_5 = 55;

      if(verbose){
        System.out.print("Test 9...");
      }
      
      //
      // New overwrites old
      //


      // Starts earlier overlap; starts later no overlap,
      // starts earlier no overlap
      if(vverbose){
        System.out.print(".");
      }
      doBody("9-2", 100, 100, 1, 1, VAL_1, 100);
      doInval("9-2", 50, 100, 2, 1);
      doBody("9-2", 200, 100, 3, 1, VAL_3, 200);
      doInval("9-2", 225, 25, 4, 1);
      checkReadInvalidRange("9-2", 50, 100);
      checkReadInvalidRange("9-2", 75, 100);
      checkReadInvalidRange("9-2", 100, 100);
      checkRead("9-2", 150, 100, 50, (byte)(VAL_1 + 50), true);  
      checkRead("9-2", 175, 100, 25, (byte)(VAL_1 + 75), true);  
      checkRead("9-2", 200, 100, 25, VAL_3, true);
      checkReadInvalidRange("9-2", 225, 25);
      checkRead("9-2", 250, 100, 50, (byte)(VAL_3 + 50), true);


      // Ends middle overlap -- overwrite some of the
      // prev writes; result comes back in pieces b/c
      // different prev-accept
      if(vverbose){
        System.out.print(".");
      }
      doInval("9-2", 0, 1000, 2, 2);
      checkReadInvalidRange("9-2", 0, 1000);
      checkReadInvalidRange("9-2", 25, 1000);
      checkReadInvalidRange("9-2", 50, 1000);
      checkReadInvalidRange("9-2", 100, 1000);
      checkReadInvalidRange("9-2", 150, 1000);
      checkRead("9-2", 200, 100, 25, VAL_3, true);
      checkReadInvalidRange("9-2", 225, 25);
      checkRead("9-2", 250, 100, 50, (byte)(VAL_3 + 50), true);
      checkReadInvalidRange("9-2", 525, 25);

      // Ends later overlap -- overwrite all of the
      // prev writes; result comes back in pieces b/c
      // different prev-accept
      if(vverbose){
        System.out.print(".");
      }
      doInval("9-2", 0, 1000, 5, 1);
      checkReadInvalidRange("9-2", 0, 1000);
      checkReadInvalidRange("9-2", 25, 1000);
      checkReadInvalidRange("9-2", 50, 1000);
      checkReadInvalidRange("9-2", 100, 1000);
      checkReadInvalidRange("9-2", 150, 1000);
      checkReadInvalidRange("9-2", 200, 1000);
      checkReadInvalidRange("9-2", 300, 1000);

      if(vverbose){
        System.out.print(".");
      }
      // Overwrite prev write. Result comes back as one chunk
      // b/c all have same prev-accept
      doBody("9-2", 0, 1000, 6, 1, VAL_1, 0);
      checkRead("9-2", 0, 3000, 1000, VAL_1, true);  

  
      //
      // Old overwrites new
      //
      if(vverbose){
        System.out.print(".");
      }
      doBody("9-4", 43, 99, 2, 1, VAL_1, 43);
      doInval("9-4", 43, 99, 1, 1);
      checkRead("9-4", 43, 100, 99, VAL_1, true);
      checkReadHole("9-4", 42);
      checkReadEOF("9-4", 142);

      // Starts earlier overlap; starts later no overlap,
      // starts earlier no overlap
      if(vverbose){
        System.out.print(".");
      }
      doBody("9-5", 25, 25, 4, 1, VAL_4, 25);
      doBody("9-5", 100, 25, 3, 1, VAL_3, 100);
      doInval("9-5", 0, 100, 1, 2);
      doInval("9-5", 50, 100, 1, 1);
      doInval("9-5", 150, 100, 1, 3);
      doInval("9-5", 50, 50, 1, 4);
      doBody("9-5", 0, 300, 0, 1, VAL_2, 250);
      checkRead("9-5", 25, 100, 25, VAL_4, true);  
      checkRead("9-5", 100, 100, 25, VAL_3, true);  
      checkRead("9-5", 250, 100, 50, (byte)(VAL_2 + 250), true);  
      checkReadInvalidRange("9-5", 0, 25);
      checkReadInvalidRange("9-5", 10, 15);
      checkReadInvalidRange("9-5", 60, 40);
      checkReadInvalidRange("9-5", 155, 95);


      // new inval overwrites old gap old: 
      if(vverbose){
        System.out.print(".");
      }
      doBody("9-8", 25, 25, 1, 1, VAL_1, 25);
      doBody("9-8", 75, 25, 2, 1, VAL_2, 75);
      doInval("9-8", 0, 200, 3, 1);
      checkReadInvalidRange("9-8", 0, 200);
      checkReadInvalidRange("9-8", 20, 200);
      checkReadInvalidRange("9-8", 25, 200);
      checkReadInvalidRange("9-8", 35, 200);
      checkReadInvalidRange("9-8", 55, 200);
      checkReadInvalidRange("9-8", 75, 200);
      checkReadInvalidRange("9-8", 95, 200);
      checkReadInvalidRange("9-8", 155, 200);

      // new body overwrites old inval gap old inval: 
      if(vverbose){
        System.out.print(".");
      }
      doInval("9-9", 25, 25, 1, 1);
      doInval("9-9", 75, 25, 2, 1);
      doBody("9-9", 0, 200, 3, 1, VAL_3, 0);
      checkRead("9-9", 0, 200, 25, VAL_3, true);  
      checkRead("9-9", 25, 200, 25, (byte)(VAL_3+25), true);  
      checkRead("9-9", 50, 200, 25, (byte)(VAL_3+50), true);  
      checkRead("9-9", 75, 200, 25, (byte)(VAL_3+75), true);  
      checkRead("9-9", 100, 200, 100, (byte)(VAL_3+100), true);  

 
      if(verbose){
        System.out.print("Test 9 Done.");
      }

    }



 /** 
 *  test10 -- old deletes v. new writes, old writes v. new deletes, etc. 
 **/ 
    public void
    test10(){
      if(verbose){
        System.out.print("Test10...");
      }
      
      byte VAL_1 = 1;
      int ii;
      doBody("10-1", 0, 1000, 1, 1, VAL_1, 0);
      checkRead("10-1", 0, 1000, 1000, VAL_1, true);  
      doDelete("10-1", 0, 1);
      checkRead("10-1", 0, 2000, 1000, (byte)(VAL_1), true);  

      // These writes lose to the delete
      doBody("10-4", 0, 1000, 5, 1, VAL_1, 0);
      doBody("10-4", 1000, 1000, 5, 2, VAL_1, 1000);
      doBody("10-4", 3000, 1000, 5, 3, VAL_1, 3000);
      doInval("10-4", 2000, 1000, 5, 4);
      doInval("10-4", 9000, 1000, 5, 5);
      doInval("10-4", 1200, 300, 5, 6);
      doInval("10-4", 1503, 32, 5, 7);

      // these writes will survive
      doBody("10-4", 1300, 100, 6, 8, VAL_1, 1300);
      doInval("10-4", 200, 300, 6, 9);
      doInval("10-4", 503, 32, 6, 10);
      doInval("10-4", 593, 32, 6, 11);
      doInval("10-4", 500, 3, 6, 12);
      doBody("10-4", 250, 252, 6, 13, VAL_1, 250);
      doDelete("10-4", 6, 1);

      // Read and check
      checkRead("10-4", 1300, 200, 100, (byte)(VAL_1), true);  
      for(ii = 0; ii < 252; ii++){
        checkRead("10-4", 250 + ii, 1, 1, (byte)(VAL_1 + ii), true);
      }
      checkReadInvalidRange("10-4", 200, 200);
      checkReadInvalidRange("10-4", 220, 1);
      checkReadInvalidRange("10-4", 503, 1);
      checkReadInvalidRange("10-4", 504, 1);
      checkReadInvalidRange("10-4", 593, 1);
      checkReadInvalidRange("10-4", 600, 1);
      checkReadHole("10-4", 0);
      checkReadHole("10-4", 5);
      checkReadHole("10-4", 15);
      checkReadHole("10-4", 150);
      checkReadHole("10-4", 199);
      checkReadHole("10-4", 800);
      checkReadEOF("10-4", 1400);
      checkReadEOF("10-4", 1401);

      // now issue some writes that are too old to matter
      doBody("10-4", 500, 1000, 0, 0, VAL_1, ras.NO_NEW_VALID_BYTES);
      doBody("10-4", 100, 1000, 0, 1, VAL_1, ras.NO_NEW_VALID_BYTES);
      doBody("10-4", 1000, 1000, 0, 2, VAL_1, ras.NO_NEW_VALID_BYTES);
      doBody("10-4", 0, 1000, 0, 3, VAL_1, ras.NO_NEW_VALID_BYTES);
      doInval("10-4", 500, 1000, 0, 4);
      doInval("10-4", 100, 1000, 0, 5);
      doInval("10-4", 1000, 1000, 0, 6);
  
      // Read and check -- nothing has changed
      checkRead("10-4", 1300, 200, 100, (byte)(VAL_1), true);  
      for(ii = 0; ii < 252; ii++){
        checkRead("10-4", 250 + ii, 1, 1, (byte)(VAL_1 + ii), true);
      }
      checkReadInvalidRange("10-4", 200, 200);
      checkReadInvalidRange("10-4", 220, 1);
      checkReadInvalidRange("10-4", 503, 1);
      checkReadInvalidRange("10-4", 504, 1);
      checkReadInvalidRange("10-4", 593, 1);
      checkReadInvalidRange("10-4", 600, 1);
      checkReadHole("10-4", 0);
      checkReadHole("10-4", 5);
      checkReadHole("10-4", 15);
      checkReadHole("10-4", 150);
      checkReadHole("10-4", 199);
      checkReadHole("10-4", 800);
      checkReadEOF("10-4", 1400);
      checkReadEOF("10-4", 1401);

      if(verbose){
        System.out.print("..Test 10 OK.");
      }
    }


 /** 
 *  test11 -- lots of random reads, writes, deletes. Hope to trigger 
 *            an assertion failure. Also, keep a "shadow" of the 
 *            expected data in memory that we can check reads against. 
 **/ 
    public void
    test11(){
      long passes = 100000;
      if(quick){
        passes = 1000;
      }
      int nFiles = 100;
      int maxOffset = 1000;
      int maxLen = 200;
      long maxTime = passes;
      long maxNodes = 10;
      long percentMax = 100;
      long P_DELETE = 2;
      long P_READ = 10;
      long P_INVAL = 44;
      long P_BODY = 44;

      long ii;
      int iFile;
      int iOffset;
      int iLen;
      byte iVal;
      int iOp;
      long iTime;
      long iNode;
      DBGShadow shadow[][] = new DBGShadow[nFiles][maxOffset + maxLen];
      for(iFile = 0; iFile < nFiles; iFile++){
        for(iOffset = 0; iOffset < maxOffset + maxLen; iOffset++){
          shadow[iFile][iOffset] = new DBGShadow();
        }
      }


      if(verbose){
        System.out.print("Test11...");
      }
      
      for(ii = 0; ii < passes; ii++){
        iFile = (int)random(0, nFiles);
        iOffset = (int)random(0, maxOffset);
        iLen = (int)random(1, maxLen);
        iOp = (int)random(0, percentMax);
        iTime = random(0, ii + 1);  // Gradually advance max possible time so
        // we contiue to see new events throughout
        // Otherwise per-file max delete filters 
        // too many writes.
        iVal = (byte)iTime;
        iNode = random(0, maxNodes);

        if(vverbose && (ii % 100 == 0)){
          System.out.print(".");
        }
        if(iOp < P_DELETE){
          // 
          // Not allowed to have write and delete with same accept stamp
          // Prevent this rare case by giving all deletes an even localTime
          // and all writes an odd localTime
          //
          if(iTime % 2 == 1){
            iTime = iTime + 1;
          }
      
          doDelete("11-" + iFile, iTime, iNode);
          shadowDoDelete(shadow, iFile, iTime, iNode);
        }
        else if(iOp < P_DELETE + P_READ){
          shadowCheckRead("11-" + iFile, shadow, iFile, iOffset, iLen);
        }
        else if(iOp < P_DELETE + P_READ + P_INVAL){
          if(iTime % 2 == 0){
            iTime = iTime + 1;
          }
          doInval("11-" + iFile, iOffset, iLen, iTime, iNode);
          shadowDoInval(shadow, iFile, iOffset, iLen, iTime, iNode);
        }
        else{
          if(iTime % 2 == 0){
            iTime = iTime + 1;
          }
          doBodyNoIncrementValue("11-" + iFile, iOffset, iLen, iTime, iNode, iVal);
          shadowDoBody(shadow, iFile, iOffset, iLen, iTime, iNode, iVal);
        }
      }
      if(verbose){
        System.out.print("Test11 OK.");
      }
    }


 /** 
 *  shadowDoDelete() -- update the shadow array in memory to reflect 
 *                          a delete of the specified file at the specified time 
 **/ 
    private static void
      shadowDoDelete(DBGShadow shadow[][], int iFile, long iTime, long iNode){
      AcceptStamp as = new AcceptStamp(iTime, new NodeId(iNode));
      int fLen = shadow[iFile].length;
      int ii;

      for(ii = 0; ii < fLen; ii++){
        shadow[iFile][ii].applyDelete(as);
      }
    }


 /** 
 *  test12() -- test sync 
 **/ 
  public void
  test12(){
    if(verbose){
      System.out.print("Test12...");
    }
    try{
      ras.sync(); // Just sync right now...
    }
    catch(IOException e){
      fail();
    }
    if(verbose){
      System.out.print("Test12 OK.");
    }
  }





 /** 
 *  test13() -- small file write/create/read/delete test 
 **/ 
  public void test13a(){
    doTest13(false, false, false);
  }
  public void test13b(){
    doTest13(false, false, true);
  }
  public void test13c(){
    doTest13(true, false, true);
  }
  public void test13d(){
    doTest13(false, true, true);
  }


 /** 
 *  doTest13() -- small file write/create/read/delete test 
 *   
 *  Note: we *do* print out timing results even in non-verbose 
 *        mode. This is so that the nightly unit tests will 
 *        contain a log of performance results so if we ever 
 *        have a problem, we can go back and see what it was. 
 **/ 
  private void doTest13(boolean syncEach, boolean syncPhase, boolean doSanity){
    int nFiles = 1000;
    if(quick){
      nFiles = 50;
    }
    long start, end;
    int ii;
    String objId;
    byte b[] = {0};
    ImmutableBytes ib = new ImmutableBytes(b);
    System.out.print("Test13 [small file performance test " 
                     + (syncEach ? " SYNC_EACH" : " NO_SYNC_EACH") 
                     + (syncPhase ? " SYNC_PHASE" : " NO_SYNC_PHASE")
                     + (doSanity ? " DO_EXPENSIVE_SANITY_CHECKS" : " NO_DO_EXPENSIVE_SANITY_CHECKS")
                     + " ]...");
    BodyMsg bm;
    AcceptStamp as = new AcceptStamp(1000, new NodeId(9999999));
    long ret;
    double avgMS;
    CounterVV vv = new CounterVV();
    boolean oldDoSanity = ras.doExpensiveSanityChecks;
    ras.doExpensiveSanityChecks = doSanity;
    
    try{
      
      // 
      // Update VV with this nodeID (outside of timing for fairness
      //
      bm = new BodyMsg(new ObjId("/test-13/foo"), 0, 1, as, ib, false);
      ret = ras.applyBody(bm);
      
      //
      // Create
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        bm = new BodyMsg(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 
                         0, 1, as, ib, false);
        ret = ras.applyBody(bm);
        assert(ret == 0);
        if(syncEach){
          ras.sync();
        }
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("RASUnit 13: Create 1 byte file: " + avgMS + "ms...");
      
      //
      // Overwrite
      //
      as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        bm = new BodyMsg(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 
                         0, 1, as, ib, false);
        ret = ras.applyBody(bm);
        assert(ret == 0);
        if(syncEach){
          ras.sync();
        }
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("RASUnit 13: Overwrite 1 byte file: " + avgMS + "ms...");
      
      
      //
      // Read
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        bm = ras.read(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 0, 1, true);
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("RASUnit 13: Read 1 byte file: " + avgMS + "ms...");
      
      
      //
      // Append
      //
      as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        bm = new BodyMsg(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 
                         1, 1, as, ib, false);
        ret = ras.applyBody(bm);
        assert(ret == 1);
        if(syncEach){
          ras.sync();
        }
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.print("RASUnit 13: Append 1 byte to file: " + avgMS + "ms...");
      
      
      //
      // Delete
      //
      as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        ras.delete(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), as);
        if(syncEach){
          ras.sync();
        }
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.print("RASUnit 13: Delete 2 byte file: " + avgMS + "ms...");
      
      
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
    
    ras.doExpensiveSanityChecks = oldDoSanity;
    if(verbose){
      System.out.print("...Test 13 OK");
    }
  }



 /** 
 *  test14() -- small write/read/delete performance test 
 **/ 
  public void test14a(){
    doTest14(false, false, false);
  }
  public void test14b(){
    doTest14(false, false, true);
  }
  public void test14c(){
    doTest14(true, false, true);
  }
  public void test14d(){
    doTest14(false, true, true);
  }

 /** 
 *  doTest14() -- small write/read/delete performance test 
 *  Note: we *do* print out timing results even in non-verbose 
 *        mode. This is so that the nightly unit tests will 
 *        contain a log of performance results so if we ever 
 *        have a problem, we can go back and see what it was. 
 **/ 
  private void doTest14(boolean syncEach, boolean syncPhase, boolean doSanity){
    int nWrites = 1000;
    if(quick){
      nWrites = 50;
    }
    int nFiles = 1;
    long start, end;
    int ii;
    String objId;
    byte b[] = {0};
    ImmutableBytes ib = new ImmutableBytes(b);
    
    System.out.print("Test14 [small write performance test " 
                     + (syncEach ? " SYNC_EACH" : " NO_SYNC_EACH") 
                     + (syncPhase ? " SYNC_PHASE" : " NO_SYNC_PHASE")
                     + (doSanity ? " DO_EXPENSIVE_SANITY_CHECKS" : " NO_DO_EXPENSIVE_SANITY_CHECKS")
                     + " ]...");

    BodyMsg bm;
    AcceptStamp as = new AcceptStamp(1000, new NodeId(8888888));
    long ret;
    double avgMS;
    CounterVV vv = new CounterVV();
    boolean oldDoSanity = ras.doExpensiveSanityChecks;
    ras.doExpensiveSanityChecks = doSanity;

    try{
      
      // 
      // Update VV with this nodeID (outside of timing for fairness
      //
      bm = new BodyMsg(new ObjId("/test-14/foo"), 0, 1, as, ib, false);
      ret = ras.applyBody(bm);
      
      //
      // Write
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nWrites; ii++){
        bm = new BodyMsg(new ObjId("/test-14/bar" + syncEach + syncPhase + doSanity),
                         ii, 1, as, ib, false);
        ret = ras.applyBody(bm);
        assert(ret == ii);
        if(syncEach){
          ras.sync();
        }
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nWrites;
      System.out.println("RASUnit 14: 1 byte write: " + avgMS + "ms...");
      
      //
      // Overwrite
      //
      as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nWrites; ii++){
        bm = new BodyMsg(new ObjId("/test-14/bar" + syncEach + syncPhase + doSanity),
                         ii, 1, as, ib, false);
        ret = ras.applyBody(bm);
        assert(ret == ii);
        if(syncEach){
          ras.sync();
        }
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nWrites;
      System.out.println("RASUnit 14: 1 byte overwrite: " + avgMS + "ms...");


      //
      // Read
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nWrites; ii++){
        bm = ras.read(new ObjId("/test-14/bar" + syncEach + syncPhase + doSanity), ii, 1, true);
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nWrites;
      System.out.println("RASUnit 14: 1 byte read: " + avgMS + "ms...");


      //
      // Delete
      //
      as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      ras.delete(new ObjId("/test-14/bar" + syncEach + syncPhase + doSanity), as);
      if(syncEach){
        ras.sync();
      }
      if(syncPhase){
        ras.sync();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("RASUnit 14: Delete 1000-byte, 1000-write file: " + avgMS + "ms...");
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }

    ras.doExpensiveSanityChecks = oldDoSanity;
    if(verbose){
      System.out.print("...Test 14 OK");
    }

  }



 /** 
 *  test15() -- large read/write performance test 
 *  
 *  Working set varies from 100000 to 10000000 
 *  File size varies from 1000 to 10000000 
 **/ 
  public void test15a(){
    doTest15(100000, 1000);
  }

  public void test15b(){
    doTest15(1000000, 1000);
  }

  public void test15c(){
    if(!quick){
      doTest15(10000000, 1000);
    }
  }

  public void test15d(){
    doTest15(1000000, 100000);
  }

  public void test15e(){
    if(!quick){
      doTest15(10000000, 100000);
    }
  }

  public void test15f(){
    if(!quick){
      doTest15(10000000, 10000000);
    }
  }


 /** 
 *  doTest15() -- large read/write performance test 
 *  Note: we *do* print out timing results even in non-verbose 
 *        mode. This is so that the nightly unit tests will 
 *        contain a log of performance results so if we ever 
 *        have a problem, we can go back and see what it was. 
 **/ 
  private void doTest15(int iWorkingSet, int iFSize){
    int maxWorkingSet = 10000000;
    if(quick){
      maxWorkingSet = 1000000;
    }
    int passesMaxSize = 2;
    int passesThisSize;
    int iPass;
    int nFiles, iFile;
    long localStamp = 100;
    AcceptStamp as;
    BodyMsg bm;
    byte b[];
    ImmutableBytes ib;
    NodeId writer = new NodeId(4785675);
    long start, end;
    long msPerMicro = 1000;
    int ii;
    long got;
    ObjId obj;
    VV checkVV;
    long totMS;
    double bw;
    long cacheSize = 1000000;

    if(verbose){
      System.out.println("Test 15: Large file read/write performance...");
      System.out.println("   cache: " + cacheSize);
    }

    assert(iWorkingSet >= iFSize);
    assert(maxWorkingSet >= iWorkingSet);


    try{
      passesThisSize = maxWorkingSet / iWorkingSet * passesMaxSize;
      nFiles = iWorkingSet / iFSize;
      assert(nFiles > 0);
      assert(nFiles * iFSize == iWorkingSet); // So the comparison of times is fair
      b = new byte[iFSize];
      for(ii = 0; ii < iFSize; ii++){
        b[ii] = (byte)42;
      }
      ib = new ImmutableBytes(b);

      //
      // Write
      //
      start = System.currentTimeMillis();
      for(iPass = 0; iPass < passesThisSize; iPass++){
        for(iFile = 0; iFile < nFiles; iFile++){
          obj = new ObjId("/test15/" + iFile + "-" + iFSize);
          as = new AcceptStamp(localStamp++, writer);
          bm = new BodyMsg(obj, 0, iFSize,  as, ib, false);
          got = ras.applyBody(bm);
          assert(got == 0);
        }
      }
      checkVV = ras.getVV();
      try{
        ras.sync();
      }
      catch(Exception e){
        e.printStackTrace();
        assert(false);
      }
      end = System.currentTimeMillis();
      totMS = end - start;
      bw = ((double)(iFSize * nFiles * passesThisSize))/((double)(totMS * msPerMicro));
      System.out.println("  Test15 WRITE wss: " 
                         + iWorkingSet
                         + " passes: " + passesThisSize
                         + " fSize: " + iFSize  
                         + " Time: " +  totMS + " ms."
                         + " BW: " + bw + " MB/s");
      
      //
      // Read.
      //
      start = System.currentTimeMillis();
      for(iPass = 0; iPass < passesThisSize; iPass++){
        for(iFile = 0; iFile < nFiles; iFile++){
          obj = new ObjId("/test15/" + iFile + "-" + iFSize);
          bm = ras.read(obj, 0, iFSize, true);
          assert(bm.getLength() == iFSize);
        }
      }
      end = System.currentTimeMillis();
      totMS = end - start;
      bw = ((double)(iFSize * nFiles * passesThisSize))/((double)(totMS * msPerMicro));
      System.out.println("  Test15 READ wss: " 
                         + iWorkingSet
                         + " passes: " + passesThisSize
                         + " fSize: " + iFSize  
                         + " Time: " +  totMS + " ms."
                         + " BW: " + bw + " MB/s");
      
      // 
      // Clean up DB
      //
      ras.close();
      selfTestCleanup(path);
    }
    catch(IOException e){
      fail("Exception in test 15 (check to see if disk is full?)");
    }
    catch(Exception xxe){
      fail();
    }
  }


 /** 
 *  test15b() -- large read/write performance test corresponding to  
 *  URANode::test15 
 *  Note: we *do* print out timing results even in non-verbose 
 *        mode. This is so that the nightly unit tests will 
 *        contain a log of performance results so if we ever 
 *        have a problem, we can go back and see what it was. 
 **/ 
  public void test15alternate(){
    int maxFSize = 1000000;
    int minFSize = 1000000;
    int sizeIncrease = 10;
    int minWorkingSet = 1000000;
    int maxWorkingSet = 1000000;
    int iWorkingSet;
    int passesMaxSize = 2;
    int passesThisSize;
    int iPass, iFSize;
    int nFiles, iFile;
    long localStamp = 100;
    AcceptStamp as;
    BodyMsg bm;
    byte b[];
    ImmutableBytes ib;
    NodeId writer = new NodeId(4785675);
    long start, end;
    long msPerMicro = 1000;
    int ii;
    long got;
    ObjId obj;
    VV checkVV;
    long totMS;
    double bw;
    long cacheSize = 30000000;

    assert(maxWorkingSet >= maxFSize);

    if(verbose){
      System.out.println("Test 15: Large file read/write performance...");
      System.out.println("   cache: " + cacheSize);
    }

    try{
      for(iWorkingSet = minWorkingSet;
          iWorkingSet <= maxWorkingSet; 
          iWorkingSet *= sizeIncrease){
        passesThisSize = maxWorkingSet / iWorkingSet * passesMaxSize;
        for(iFSize = minFSize; iFSize <= maxFSize; iFSize *= sizeIncrease){
          nFiles = iWorkingSet / iFSize;
          if(nFiles == 0){
            assert(iWorkingSet < iFSize);
            continue;
          }
          assert(nFiles * iFSize == iWorkingSet); // So the comparison of times is fair
          b = new byte[iFSize];
          for(ii = 0; ii < iFSize; ii++){
            b[ii] = (byte)42;
          }
          ib = new ImmutableBytes(b);
          b = null;
          //
          // Write
          //
          start = System.currentTimeMillis();
          for(iPass = 0; iPass < passesThisSize; iPass++){
            for(iFile = 0; iFile < nFiles; iFile++){
              obj = new ObjId("/test15/" + iFile + "-" + iFSize);
              as = new AcceptStamp(localStamp++, writer);
              bm = new BodyMsg(obj, 0, iFSize, as, ib, false);
              got = ras.applyBody(bm);
              assert(got == 0);
            }
          }
          checkVV = ras.getVV();
          try{
            ras.sync();
          }
          catch(Exception e){
            e.printStackTrace();
            assert(false);
          }
          end = System.currentTimeMillis();
          totMS = end - start;
          bw = ((double)(iFSize * nFiles * passesThisSize))/((double)(totMS * msPerMicro));
          System.out.println("  Test15 WRITE wss: " 
                             + iWorkingSet
                             + " passes: " + passesThisSize
                             + " fSize: " + iFSize  
                             + " Time: " +  totMS + " ms."
                             + " BW: " + bw + " MB/s");

          //
          // Read.
          //
          start = System.currentTimeMillis();
          for(iPass = 0; iPass < passesThisSize; iPass++){
            for(iFile = 0; iFile < nFiles; iFile++){
              obj = new ObjId("/test15/" + iFile + "-" + iFSize);
              bm = ras.read(obj, 0, iFSize, true);
              assert(bm.getLength() == iFSize);
            }
          }
          end = System.currentTimeMillis();
          totMS = end - start;
          bw = ((double)(iFSize * nFiles * passesThisSize))/((double)(totMS * msPerMicro));
          System.out.println("  Test15 READ wss: " 
                             + iWorkingSet
                             + " passes: " + passesThisSize
                             + " fSize: " + iFSize  
                             + " Time: " +  totMS + " ms."
                             + " BW: " + bw + " MB/s");

        }
      }
    }
    catch(IOException e){
      System.out.println("Exception in test 15 (check to see if disk is full?)");
      System.out.println(e.toString());
      e.printStackTrace();
      assert(false);
    }
    catch(Exception xxe){
      xxe.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  test16() -- concurrency test of makeNewFNum 
 *   
 *  test16 is multi-threaded, so it does not live in this file. 
 *  Instead, it lives in RandomAccessStateUnitMT.java 
 **/ 
  public void test16(){
    if(verbose){
      System.out.println("Test16 is not in this file. "
                         + "Instead, it lives in RandomAccessStateUnitMT.java.");
    }
  }


 /** 
 *  test17() -- test read with offset different from the per-range-state record 
 **/ 
  public void test17(){
    byte VAL_1 = 1;
    byte VAL_2 = 122;
    if(verbose){
      System.out.print("Test17...");
    }
    
    doInval("/0/0/0", 700, 1000, 9345, 9);
    doInval("/0/0/0", 800, 1000, 10001, 9);
    doBody("/0/0/0", 700, 630, 9345, 9, VAL_1, 700);
    doBody("/0/0/0", 800, 200, 10001, 9, VAL_2, 800);
    checkRead("/0/0/0", 720, 630, 80, (byte)(VAL_1 + 20), true);
    checkRead("/0/0/0", 720, 630, 100, VAL_1, false);
    checkRead("/0/0/0", 800, 200, 200, VAL_2, false);
    checkRead("/0/0/0", 999, 200, 1, (byte)(VAL_2 + 199), true);
    checkRead("/0/0/0", 999, 200, 200, VAL_2, false);

    if(verbose){
      System.out.println("...Test17 SUCCEEDS.");
    }
  }



 /** 
 *  test18() -- stress test -- write and read lots of big files. Stress 
 *    tests for memory leaks 
 **/ 
  public void test18a(){
    if(!quick){
      doTest18(10, 200, 1000000);
    }
    else{
      doTest18(5, 10, 500000);
    }
  }

  public void test18b(){
    if(!quick){
      doTest18(10, 2000, 100000);
    }
    else{
      doTest18(5, 100, 50000);
    }
  }


 /** 
 *  doTest18() -- stress test -- write and read lots of big files 
 *  
 *  Note that garbage collection behavior seems to matter.  If we write 
 *  files fast enough, we seem to be able to get ahead of the garbage 
 *  collector and get an out of heap space error. We have *tried* 
 *  to fix this by adding GCForcer to RandomAccessState, but 
 *  the solution doesn't seem 100% solid. Good ideas needed. 
 **/ 
  private void doTest18(int npasses, int nfiles, int fSize){
    String fname;
    long dummyNode = 100;
    byte dummyVal = 24;
    long ifile;
    long ipass;
    long time;
    boolean debugging = true;

    
    if(verbose){
      System.out.println("Test 18 writing " + nfiles + " files each of " + fSize
                         + " bytes (tot: " + nfiles * fSize + ") repeat " + npasses
                         + " times...");
    }
    for(ipass = 0; ipass < npasses; ipass++){
      if(vverbose){
        System.out.print("Test 18 WRITE pass " + ipass + " ...");
      }
      for(ifile = 0; ifile < nfiles; ifile++){
        if(vverbose){
          System.out.print("Write file " + ifile + " ...");
        }
        fname = new String("/test18-" + ifile);
        time = 18000 + ipass*nfiles + ifile;
        doBody(fname, (long)0, fSize, time, dummyNode, dummyVal,
                       0);
      }
    }
    for(ipass = 0; ipass < npasses; ipass++){
      if(vverbose){
        System.out.print("Test 18 READ pass " + ipass + " ...");
      }
      for(ifile = 0; ifile < nfiles; ifile++){
        if(vverbose){
          System.out.print("Read file " + ifile + " ...");
        }
        fname = new String("/test18-" + ifile);
        checkRead(fname, (long)0, fSize, fSize, dummyVal, true);
      }
    }
    if(vverbose){
      System.out.print("Test 18 DELETE pass ...");
    }
    for(ifile = 0; ifile < nfiles; ifile++){
      if(vverbose){
        System.out.print(" file " + ifile + " ...");
      }
      fname = new String("/test18-" + ifile);
      time = 18000 + npasses*nfiles + ifile;
      doDelete(fname, time, dummyNode);
    }

    if(verbose){
      System.out.println("...Test18 SUCCEEDS (NOTE: Not for all cases see above!!!).");
    }
  }


 /** 
 *  test19() -- test shipRAS 
 *               
 *  test scenario: create a bounch of files 
 *                 update a certain range for each file sequentially 
 *                 update another range of each file sequentially again 
 *               
 *                 shipRAS for different combination of path and startvv 
 *                 test if the corresponding PerRangeStates are shipped 
 *                      if the corresponding PerObjectStates are shipped 
 **/ 
  public void test19(){
    /*
    fail("TBD: Write a test of the shipRAS function -- create a "
         + " bunch of files and make sure the thing we ship has the "
         + " right stuff in it (and doesn't have the wrong stuff in it!)");
    */
    if(verbose){
      System.out.print("Test19...");
    }
    byte VAL_1 = 1;
    byte VAL_2 = 122;
    byte VAL_3 = 39;
    byte VAL_4 = 94;
    byte VAL_5 = 55;

    Vector objList = new Vector();
    PerObjStateForSend pos = null;
    PerRangeStateForSend prs = null;

    String outputFileName = "RASTest19.tmp";
    try{
      File tmpFile = new File(outputFileName);
      if(tmpFile.exists()){
	assert(tmpFile.isFile());
	tmpFile.delete();
      }
    }catch(Exception e){
      System.err.println("delete tmp file error: " + e.toString());
      e.printStackTrace();
      return;
    }
    FileOutputStream fos = null;
    TaggedOutputStream tos = null;
    try{
      fos = new FileOutputStream(outputFileName);
      tos = new TaggedOutputStream(fos);
    }catch(Exception e){
      System.err.println("shipRAS outputstream error: " + e.toString());
      e.printStackTrace();
      return;
    }
    String cpPath = "";
    SubscriptionSet cpPathSS = null;
    String[] exclChildNames;
    
    //populate initial state
    
    // testFileName, offset, length, localTime, node, value, expectRet
    // for range 0-4
    doBound("a/a1", 0, 5, 0, 0, VAL_1, 0);
    doBound("a/a2", 0, 5, 1, 0, VAL_1, 0);
    doBound("b/b1", 0, 5, 2, 0, VAL_1, 0);
    doBound("b/b2", 0, 5, 3, 0, VAL_1, 0);

    // for range 5-9
    doBound("a/a1", 5, 5, 4, 0, VAL_2, 5);
    doBound("a/a2", 5, 5, 5, 0, VAL_2, 5);
    doBound("b/b1", 5, 5, 6, 0, VAL_2, 5);
    doBound("b/b2", 5, 5, 7, 0, VAL_2, 5);

    // for range 9-14
    doBound("a/a1", 10, 5, 8, 0, VAL_3, 10);
    doBound("a/a2", 10, 5, 9, 0, VAL_3, 10);
    doBound("b/b1", 10, 5, 10, 0, VAL_3, 10);
    doBound("b/b2", 10, 5, 11, 0, VAL_3, 10);

    //check shipRAS

    // test PerRangeState
    //play with different subscribe path name and excludeChildNames
    //Make sure that the correct set of PerRangeState are sent.
    exclChildNames = new String[0];
    AcceptStamp[] as1 = new AcceptStamp[1];
    as1[0] = new AcceptStamp(-1, new NodeId(0));
    AcceptVV cpStartVV1 = new AcceptVV(as1);
    
    AcceptStamp[] as2 = new AcceptStamp[1];
    as2[0] = new AcceptStamp(4, new NodeId(0));
    AcceptVV cpStartVV2 = new AcceptVV(as2);
    
    NodeId writerId = new NodeId(0);
    
    try{
      /////////////////////
      //case 1 everything
      /////////////////////
      cpPath = "/";
      cpPathSS = SubscriptionSet.makeSubscriptionSet(cpPath + "*");
      //      ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
      
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     0,
                                     5,
                                     new AcceptStamp(0, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     5,
                                     5,
                                     new AcceptStamp(4, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     10,
                                     5,
                                     new AcceptStamp(8, writerId));
      objList.add(prs);
      
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     0,
                                     5,
                                     new AcceptStamp(1, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     5,
                                     5,
                                     new AcceptStamp(5, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     10,
                                     5,
                                     new AcceptStamp(9, writerId));
      objList.add(prs);

      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     0,
                                     5,
                                     new AcceptStamp(2, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     5,
                                     5,
                                     new AcceptStamp(6, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     10,
                                     5,
                                     new AcceptStamp(10, writerId));
      objList.add(prs);

      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     0,
                                     5,
                                     new AcceptStamp(3, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     5,
                                     5,
                                     new AcceptStamp(7, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     10,
                                     5,
                                     new AcceptStamp(11, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      ///////////////////////////////
      //case 2 everything after 4
      ///////////////////////////////
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV2, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV2, false);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     10,
                                     5,
                                     new AcceptStamp(8, writerId));
      objList.add(prs);    
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     5,
                                     5,
                                     new AcceptStamp(5, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     10,
                                     5,
                                     new AcceptStamp(9, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     5,
                                     5,
                                     new AcceptStamp(6, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     10,
                                     5,
                                     new AcceptStamp(10, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     5,
                                     5,
                                     new AcceptStamp(7, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     10,
                                     5,
                                     new AcceptStamp(11, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      ////////////////////////
      // everything for a
      ////////////////////////
      cpPath = "/test/a";
      cpPathSS = SubscriptionSet.makeSubscriptionSet(cpPath + "/*");
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     0,
                                     5,
                                     new AcceptStamp(0, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     5,
                                     5,
                                     new AcceptStamp(4, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     10,
                                     5,
                                     new AcceptStamp(8, writerId));
      objList.add(prs);
      
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     0,
                                     5,
                                     new AcceptStamp(1, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     5,
                                     5,
                                     new AcceptStamp(5, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     10,
                                     5,
                                     new AcceptStamp(9, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      
      ///////////////////////////
      //everything for a after 4
      ///////////////////////////
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV2, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV2, false);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     10,
                                     5,
                                     new AcceptStamp(8, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     5,
                                     5,
                                     new AcceptStamp(5, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     10,
                                     5,
                                     new AcceptStamp(9, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));


      //////////////////////////
      // everything except a
      //////////////////////////
      cpPath = "/test";
      exclChildNames = new String[1];
      exclChildNames[0]= "a";
      cpPathSS = SubscriptionSet.makeSubscriptionSet("/test/b/*");
      // ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     0,
                                     5,
                                     new AcceptStamp(2, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     5,
                                     5,
                                     new AcceptStamp(6, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     10,
                                     5,
                                     new AcceptStamp(10, writerId));
      objList.add(prs);

      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     0,
                                     5,
                                     new AcceptStamp(3, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     5,
                                     5,
                                     new AcceptStamp(7, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     10,
                                     5,
                                     new AcceptStamp(11, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));
      

      ////////////////////////////////
      // everything except a after 4
      ////////////////////////////////
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV2, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV2, false);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     5,
                                     5,
                                     new AcceptStamp(6, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     10,
                                     5,
                                     new AcceptStamp(10, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     5,
                                     5,
                                     new AcceptStamp(7, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     10,
                                     5,
                                     new AcceptStamp(11, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      //////////////
      // nothing
      //////////////
      exclChildNames = new String[2];
      exclChildNames[0] = "a";
      exclChildNames[1] = "b";
      cpPathSS = SubscriptionSet.makeSubscriptionSet("/test/z/*");
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      //test PerObjState
      //issue new operations matters for PerObjState
      //play with different startTime.
      //Make sure that the correct set of PerObjState are sent.
      doDelete("a/a1", 12, 0);
      doBound("a/a1", 0, 5, 13, 0, VAL_5, 0);
      
      //////////////////////////////////////
      // have the delete for per obj state
      //////////////////////////////////////
      exclChildNames = new String[0];
      cpPath = "/";
      cpPathSS = SubscriptionSet.makeSubscriptionSet("/*");
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
      pos = new PerObjStateForSend(new ObjId("/test/a/a1"),
                                   new AcceptStamp(13, writerId),
                                   new AcceptStamp(12, writerId));
      objList.add(pos);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     0,
                                     5,
                                     new AcceptStamp(13, writerId));
      objList.add(prs);
      
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     0,
                                     5,
                                     new AcceptStamp(1, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     5,
                                     5,
                                     new AcceptStamp(5, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a2"),
                                     10,
                                     5,
                                     new AcceptStamp(9, writerId));
      objList.add(prs);

      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     0,
                                     5,
                                     new AcceptStamp(2, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     5,
                                     5,
                                     new AcceptStamp(6, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b1"),
                                     10,
                                     5,
                                     new AcceptStamp(10, writerId));
      objList.add(prs);

      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     0,
                                     5,
                                     new AcceptStamp(3, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     5,
                                     5,
                                     new AcceptStamp(7, writerId));
      objList.add(prs);
      prs = new PerRangeStateForSend(new ObjId("/test/b/b2"),
                                     10,
                                     5,
                                     new AcceptStamp(11, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      /////////////////////////////
      // everything for a after 12
      /////////////////////////////
      cpPath = "/test/a";
      cpPathSS = SubscriptionSet.makeSubscriptionSet(cpPath + "/*");
      AcceptStamp[] as3 = new AcceptStamp[1];
      as3[0] = new AcceptStamp(12, new NodeId(0));
      AcceptVV cpStartVV3 = new AcceptVV(as3);
      // ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV3, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV3, false);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     0,
                                     5,
                                     new AcceptStamp(13, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      //test withBody parameter
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV3, true);
      ras.shipRAS(tos, cpPathSS, cpStartVV3, true);
      byte b[] = new byte[5];
      int ii;
      for(ii = 0; ii < 5; ii++){
        b[ii] = (byte)(VAL_5 + ii);
      }
      ImmutableBytes im = new ImmutableBytes(b);
      BodyMsg bm = new BodyMsg(new ObjId("/test/a/a1"),
			0,
			5,
			new AcceptStamp(13, writerId),
			im,
			false);
      objList.add(bm);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      doInval("a/a1", 0, 5, 14, 0);

      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV3, true);
      ras.shipRAS(tos, cpPathSS, cpStartVV3, true);
      prs = new PerRangeStateForSend(new ObjId("/test/a/a1"),
                                     0,
                                     5,
                                     new AcceptStamp(14, writerId));
      objList.add(prs);
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      ////////////////////////////
      // everything for a after 13
      ////////////////////////////
      AcceptStamp[] as4 = new AcceptStamp[1];
      as4[0] = new AcceptStamp(14, new NodeId(0));
      AcceptVV cpStartVV4 = new AcceptVV(as4);
      //ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV4, false);
      ras.shipRAS(tos, cpPathSS, cpStartVV4, false);
      
      objList.add(new Long(RandomAccessState.RAS_SHIP_DONE));

      tos.close();
      fos.close();
    }catch(Exception e){
      System.err.println("close shipRAS outputstream error: " + e.toString());
      e.printStackTrace();
      return;
    }

    if(!verifyShipRASStream(outputFileName, objList)){
      fail("RandomAccessStateUnit.test19() -- shipRAS output stream doesn't"
           + "match the expected value!");
    }
    if(verbose){
      System.out.print("OK.");
    }
  }

  //------------------------------------------------------------------
  // verifyShipRASStream() -- verify the given outputstream file
  //                          has the same stuff in the same order
  //                          as in the objList
  //------------------------------------------------------------------
  private boolean verifyShipRASStream(String outputFileName, 
                                      Vector objList){
    try{
	File tmpFile = new File(outputFileName);
	assert tmpFile.exists();
        assert tmpFile.isFile();
    }catch(Exception e){
	System.err.println("delete tmp file error: " + e.toString());
        e.printStackTrace();
        return false;
    }

    try{
      FileInputStream fis = new FileInputStream(outputFileName);
      TaggedInputStream tis = new TaggedInputStream(fis);   
      for (Enumeration e = objList.elements() ; e.hasMoreElements() ;) {
	Object expto = e.nextElement();
        Object streamo = tis.readTaggedObject();
        if (expto instanceof PerObjStateForSend){
          assert ((PerObjStateForSend)expto).equals(streamo);
        }else if (expto instanceof PerRangeStateForSend){
          assert ((PerRangeStateForSend)expto).equals(streamo);
	}else if (expto instanceof BodyMsg){
	  assert ((BodyMsg)expto).equals(streamo);
        }else if(expto instanceof Long){
          assert ((Long)expto).equals(streamo);
        }else{
          assert false;
        }
      }
      tis.close();
      fis.close();
      return true;
    }catch(Exception e){
      System.err.println(" updateRAS inputstream  error: " + e.toString());
      e.printStackTrace();
      return false;
    }
    
  }

  //------------------------------------------------------------------
  // verifyShipRASStream() -- verify the given outputstream file
  //                          has the same stuff in the same order
  //                          as in the objList
  //------------------------------------------------------------------
  private boolean verifyShipRASStream(String oldOutputFileName, 
                                      String newOutputFileName){
   
    boolean ret = true;
    try{
	File tmpFile = new File(oldOutputFileName);
	assert tmpFile.exists();
        assert tmpFile.isFile();
        tmpFile = new File(newOutputFileName);
	assert tmpFile.exists();
        assert tmpFile.isFile();
    }catch(Exception e){
	System.err.println("output tmp file error: " + e.toString());
        e.printStackTrace();
        return false;
    }

    try{
      FileInputStream fis = new FileInputStream(oldOutputFileName);
      TaggedInputStream tis = new TaggedInputStream(fis);
      FileInputStream newfis = new FileInputStream(newOutputFileName);
      TaggedInputStream newtis = new TaggedInputStream(newfis);
      
      boolean endOfStream = false;
      Object expto = null;
      Object streamo = null;
      while(true){
        try{
          expto = tis.readTaggedObject();
        }catch(EOFException eof){
          endOfStream = true;
        }
        try{
          streamo = newtis.readTaggedObject();
        }catch(EOFException eof){
          if( endOfStream ){
            break;
          }else{
            ret = false;
            assert false;
            break;
          }
        }
        if (expto instanceof PerObjStateForSend){
          assert ((PerObjStateForSend)expto).equals(streamo);
        }else if (expto instanceof PerRangeStateForSend){
          assert ((PerRangeStateForSend)expto).equals(streamo);
        }else if (expto instanceof BodyMsg){
          assert ((BodyMsg)expto).equals(streamo);
        }else if(expto instanceof Long){
          assert ((Long)expto).equals(streamo);
        }else{
          assert false;
        }
      }
      tis.close();
      fis.close();
      newtis.close();
      newfis.close();
      return true;
    }catch(Exception e){
      System.err.println(" updateRAS inputstream  error: " + e.toString());
      e.printStackTrace();
      return false;
    }
    
  }

  //---------------------------------------------------------------------------
  // test20() -- test updateRAS
  //---------------------------------------------------------------------------
  public void test20(){
    /*
    fail("TBD: Write a test of the updateRAS function -- create a "
         + " bunch of files and make sure the thing we ship has the "
         + " right stuff in it (and doesn't have the wrong stuff in it!)");
    */
    if(verbose){
      System.out.print("Test20...");
    }

    Vector objList = new Vector();
    PerObjStateForSend pos = null;
    PerRangeStateForSend prs = null;
    NodeId writerId = new NodeId(0);
    String objName = "a/a1";

    byte VAL_1 = 1;
    byte VAL_5 = 55;

    doBound(objName, 0, 5, 1, 0, VAL_1, 0);
    checkRead(objName, 0, 5, 5, VAL_1, true);

    /*
    pos = new PerObjStateForSend(new ObjId("/test/" + objName),
				 new AcceptStamp(0, writerId),
				 new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, 
						 new NodeId(NodeId.WILDCARD_NODE_ID)));
    objList.add(pos);
    doUpdateRASWithListOfObjects(objList, "/");//assert false because of 
                                               //no DeleteStamp
    */                                           
    objList.clear();
    pos = new PerObjStateForSend(new ObjId("/test/" + objName),
				 new AcceptStamp(0, writerId),
				 new AcceptStamp(2, writerId));
    
    objList.add(pos);
    doUpdateRASWithListOfObjects(objList, "/");
    checkReadNoSuchEntry(objName);
    
    /*
    objList.clear();
    pos = new PerObjStateForSend(new ObjId("/test/" +objName),
				 new AcceptStamp(3, writerId),
				 new AcceptStamp(2, writerId));
    prs = new PerRangeStateForSend(new ObjId("/test/" + objName),
				   0,
				   3,
				   new AcceptStamp(0, writerId));
    objList.add(pos);
    objList.add(prs);
    doUpdateRASWithListOfObjects(objList, "/"); //expect assert false
                                                // because pos prs lastwriteaccept
                                                // doesn't match

    */
    objList.clear();
    
    pos = new PerObjStateForSend(new ObjId("/test/" + objName),
				 new AcceptStamp(4, writerId),
				 new AcceptStamp(2, writerId));
    prs = new PerRangeStateForSend(new ObjId("/test/" + objName),
				   0,
				   3,
				   new AcceptStamp(3, writerId));
    
    objList.add(pos);
    objList.add(prs);
    prs = new PerRangeStateForSend(new ObjId("/test/" + objName),
				   5,
				   5,
				   new AcceptStamp(4, writerId));
    objList.add(prs);
    doUpdateRASWithListOfObjects(objList, "/"); 
    checkReadInvalidRange(objName, 0, 3);
    checkReadHole(objName, 3);
    checkReadInvalidRange(objName, 5, 5);
    
    
    //test withBody parameter
    objList.clear();
    
    byte b[] = new byte[10];
    int ii;
    for(ii = 0; ii < 10; ii++){
        b[ii] = (byte)(VAL_5 + ii);
    }
    
    ImmutableBytes im = new ImmutableBytes(b);
    BodyMsg bm = new BodyMsg(new ObjId("/test/"+objName),
			     0,
			     10,
			     new AcceptStamp(13, writerId),
			     im,
			     false);
    objList.add(bm);
    doUpdateRASWithListOfObjects(objList, "/"); 
    checkRead(objName, 0, 10, 3, VAL_5, true);
    if(verbose){
	System.out.print("OK");
    }
  }

 /** 
 *  doUpdateRASWithListOfObjects: -- util method for test20 
 *   write list of objects to file like shipRAS,  
 *   call updateRAS to apply those objects. 
 **/ 
  private void doUpdateRASWithListOfObjects(Vector objList, String cpPath){
    String outputFileName = "RASTest20.tmp";
    try{
	File tmpFile = new File(outputFileName);
	if(tmpFile.exists()){
	    assert(tmpFile.isFile());
	    tmpFile.delete();
	}
    }catch(Exception e){
	System.err.println("delete tmp file error: " + e.toString());
        e.printStackTrace();
        return;
    }
    
    try{
      FileOutputStream fos = new FileOutputStream(outputFileName);
      TaggedOutputStream tos = new TaggedOutputStream(fos);
      for (Enumeration e = objList.elements() ; e.hasMoreElements() ;) {
	tos.writeObject(e.nextElement());
      }
      tos.writeObject(new Long(RandomAccessState.RAS_SHIP_DONE));
      tos.close();
      fos.close();
    }catch(Exception e){
      System.err.println(" fake shipRAS outputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
    try{
      FileInputStream fis = new FileInputStream(outputFileName);
      TaggedInputStream tis = new TaggedInputStream(fis);   
      SubscriptionSet cpPathSS = SubscriptionSet.makeSubscriptionSet(cpPath + "*");
      //
      // read all checkpoint related objects from the stream
      // and buffer it at a local vector
      //
      LinkedList objBuff = new LinkedList();
      boolean done = false;
      Object o = null;
      while(!done){
        o = tis.readTaggedObject();
        objBuff.add(o);
        if(o instanceof Long){
          Env.remoteAssert(((Long)o).longValue() == RandomAccessState.RAS_SHIP_DONE);
          done = true;
        }
      }
      tis.close();
      fis.close();

      ras.updateRAS(objBuff, cpPathSS, 100);
      
    }catch(Exception e){
      System.err.println(" updateRAS inputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
  }

 /** 
 *  test21() -- test shipRAS + updateRAS 
 **/ 
  public void test21(){
    
    if(verbose){
      System.out.print("Test21...");
    }
    byte VAL_1 = 1;
    byte VAL_2 = 122;
    byte VAL_3 = 39;
    byte VAL_4 = 94;
    byte VAL_5 = 55;

    // for range 0-4
    doBound("a/a1", 0, 5, 0, 0, VAL_1, 0);
    doBound("a/a2", 0, 5, 1, 0, VAL_1, 0);
    doBound("b/b1", 0, 5, 2, 0, VAL_1, 0);
    doBound("b/b2", 0, 5, 3, 0, VAL_1, 0);

    // for range 5-9
    doBound("a/a1", 5, 5, 4, 0, VAL_2, 5);
    doBound("a/a2", 5, 5, 5, 0, VAL_2, 5);
    doBound("b/b1", 5, 5, 6, 0, VAL_2, 5);
    doBound("b/b2", 5, 5, 7, 0, VAL_2, 5);

    // for range 9-14
    doBound("a/a1", 10, 5, 8, 0, VAL_3, 10);
    doBound("a/a2", 10, 5, 9, 0, VAL_3, 10);
    doBound("b/b1", 10, 5, 10, 0, VAL_3, 10);
    doBound("b/b2", 10, 5, 11, 0, VAL_3, 10);

    doDelete("a/a1", 12, 0);
    doBound("a/a1", 0, 3, 13, 0, VAL_5, 0);
    
    String outputFileName = "RASTest21.tmp";
    try{
	File tmpFile = new File(outputFileName);
	if(tmpFile.exists()){
	    assert(tmpFile.isFile());
	    tmpFile.delete();
	}
    }catch(Exception e){
	System.err.println("delete tmp file error: " + e.toString());
        e.printStackTrace();
        return;
    }
    FileOutputStream fos = null;
    TaggedOutputStream tos = null;
    try{
      fos = new FileOutputStream(outputFileName);
      tos = new TaggedOutputStream(fos);
    }catch(Exception e){
      System.err.println(" fake shipRAS outputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
    //    String[] exclChildNames = new String[0];
    AcceptStamp[] as1 = new AcceptStamp[1];
    as1[0] = new AcceptStamp(-1, new NodeId(0));
    AcceptVV cpStartVV1 = new AcceptVV(as1);
    
    String cpPath = "/";
    SubscriptionSet cpPathSS = SubscriptionSet.makeSubscriptionSet(cpPath);
    try{
      //	ras.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, true);
	ras.shipRAS(tos, cpPathSS, cpStartVV1, true);
   
      tos.close();
      fos.close();
    }catch(Exception e){
      System.err.println(" fake shipRAS outputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
    String path1 = "test" + File.separatorChar + "RandomAccessStateTestTMP.db";
    RandomAccessState newRAS = null;
    FileInputStream fis = null;
    TaggedInputStream tis = null;
    try{
      selfTestCleanup(path1);
      newRAS = new RandomAccessState(path1, CACHE_SIZE); 
      fis = new FileInputStream(outputFileName);
      tis = new TaggedInputStream(fis);
    
      //
      // read all checkpoint related objects from the stream
      // and buffer it at a local vector
      //
      LinkedList objBuff = new LinkedList();
      boolean done = false;
      Object o = null;
      while(!done){
        o = tis.readTaggedObject();
        objBuff.add(o);
        if(o instanceof Long){
          Env.remoteAssert(((Long)o).longValue() == RandomAccessState.RAS_SHIP_DONE);
          done = true;
        }
      }
      tis.close();
      fis.close();
      
      //      newRAS.updateRAS(tis, cpPath, 100);
      newRAS.updateRAS(objBuff, cpPathSS, 100);
      
    }catch(Exception e){
      System.err.println(" updateRAS inputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
    String newOutputFileName = "RASTest21new.tmp";
    try{
	File tmpFile = new File(newOutputFileName);
	if(tmpFile.exists()){
	    assert(tmpFile.isFile());
	    tmpFile.delete();
	}
    }catch(Exception e){
	System.err.println("delete tmp file error: " + e.toString());
        e.printStackTrace();
        return;
    }
    
    try{
      fos = new FileOutputStream(newOutputFileName);
      tos = new TaggedOutputStream(fos);
    }catch(Exception e){
      System.err.println(" fake shipRAS outputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
    try{
      //      newRAS.shipRAS(tos, cpPath, exclChildNames, cpStartVV1, true);
      newRAS.shipRAS(tos, cpPathSS, cpStartVV1, true);
      newRAS.close();
      tos.close();
      fos.close();
    }catch(Exception e){
      System.err.println(" fake shipRAS outputstream  error: " + e.toString());
      e.printStackTrace();
      return;
    }
    
    if(!verifyShipRASStream(outputFileName, newOutputFileName)){
      fail("RandomAccessStateUnit.test21() -- updateRAS doesn't"
           + "create the expected state!");
    }
    if(verbose){
      System.out.print("OK.");
    }
    return;
  }

 /** 
 *  test scan time for different data store size 
 **/ 
  public void test22(){

    if(verbose){
      System.out.print("Test22...");
    }
    long start, end;
    
    int max = 100000;
    if(quick){
      max = 10000;
    }

    for(int i = 100; i < max; i = (int)Math.round(i*2.5)){
      
      repeatedWrite(ras, i+100, i);

      String outputFileName = "RASTest22.tmp";
      try{
	File tmpFile = new File(outputFileName);
	if(tmpFile.exists()){
          assert(tmpFile.isFile());
          tmpFile.delete();
	}
      }catch(Exception e){
	System.err.println("delete tmp file error: " + e.toString());
        e.printStackTrace();
        return;
      }
      FileOutputStream fos = null;
      TaggedOutputStream tos = null;
      try{
        fos = new FileOutputStream(outputFileName);
        tos = new TaggedOutputStream(fos);
      }catch(Exception e){
        System.err.println(" fake shipRAS outputstream  error: " + e.toString());
        e.printStackTrace();
        return;
      }
      String[] exclChildNames = new String[0];
      AcceptStamp[] as1 = new AcceptStamp[1];
      as1[0] = new AcceptStamp(i-1, new NodeId(0));
      AcceptVV cpStartVV1 = new AcceptVV(as1);
    
      String cpPath = "/";
      SubscriptionSet cpPathSS = SubscriptionSet.makeSubscriptionSet("/*");
      try{
        start = System.currentTimeMillis();
        ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
        end = System.currentTimeMillis();
        if(vverbose){
          System.out.println("ScanTime( " + i + " ): " + (end-start));
        } 
        tos.close();
        fos.close();
      }catch(Exception e){
        System.err.println(" fake shipRAS outputstream  error: " 
                           + e.toString());
        e.printStackTrace();
        return;
      }

      ras.close();
      System.gc();
      if(true || vverbose){
        memoryReport();
      }
      selfTestCleanup(path); // Start from known state
      try{
        ras = new RandomAccessState(path, CACHE_SIZE);
      }catch(IOException e){
        System.err.println("create new ras fail: " + e.toString());
        e.printStackTrace();
        return;
      }
      
    }
    if(verbose){
      System.out.print("OK.");
    }
  }


  //---------------------------------------------------------------------------
  // get avg scan time for different dirty files with same data store size
  //---------------------------------------------------------------------------
  public void test23(){
    if(verbose){
      System.out.print("Test23...");
    }
    long start, end, recvStart, recvEnd, totalTime, recvTotalTime;
    
    
    int fileNum = 10000;
    if(quick){
      fileNum = 1000;
    }
    repeatedWrite(ras, fileNum*2, fileNum);

    //for(int i = 100; i <fileNum+1; i = (int)Math.round(i*1.2)){
     for(int i = 5539; i < 5540; i++){  
      totalTime = 0;
      recvTotalTime = 0;
      int pass = 1;
      for(int j = 0; j < pass; j++){
        String outputFileName = "RASTest23.tmp";
        try{
          File tmpFile = new File(outputFileName);
          if(tmpFile.exists()){
            assert(tmpFile.isFile());
            tmpFile.delete();
          }
        }catch(Exception e){
          System.err.println("delete tmp file error: " + e.toString());
          e.printStackTrace();
          return;
        }
        FileOutputStream fos = null;
        TaggedOutputStream tos = null;
        try{
          fos = new FileOutputStream(outputFileName);
          tos = new TaggedOutputStream(fos);
        }catch(Exception e){
          System.err.println(" fake shipRAS outputstream  error: " + e.toString());
          e.printStackTrace();
          return;
        }
        String[] exclChildNames = new String[0];
        AcceptStamp[] as1 = new AcceptStamp[1];
        as1[0] = new AcceptStamp(fileNum+fileNum-i-1, new NodeId(0));
        AcceptVV cpStartVV1 = new AcceptVV(as1);
    
        String cpPath = "/";
        SubscriptionSet cpPathSS = SubscriptionSet.makeSubscriptionSet("/*");
        try{
          start = System.currentTimeMillis();
          ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
          end = System.currentTimeMillis();
          totalTime += (end-start);
          if(vverbose){
            System.out.println("ScanTime_10000( " + i + " ): " + (end-start));
          } 
          tos.close();
          fos.close();
        }catch(Exception e){
          System.err.println(" fake shipRAS outputstream  error: " 
                             + e.toString());
          e.printStackTrace();
          return;
        }

        String path1 = "test" + File.separatorChar + 
          "RandomAccessStateTestTMP.db";
        RandomAccessState newRAS = null;
        FileInputStream fis = null;
        TaggedInputStream tis = null;
        try{
          selfTestCleanup(path1);
          newRAS = new RandomAccessState(path1, CACHE_SIZE); 
          
          repeatedWrite(newRAS, fileNum, fileNum);
          fis = new FileInputStream(outputFileName);
          tis = new TaggedInputStream(fis);
          
          
          recvStart = System.currentTimeMillis();
          
          //
          // read all checkpoint related objects from the stream
          // and buffer it at a local vector
          //
          LinkedList objBuff = new LinkedList();
          boolean done = false;
          Object o = null;
          while(!done){
            o = tis.readTaggedObject();
            objBuff.add(o);
            if(o instanceof Long){
              Env.remoteAssert(((Long)o).longValue() == RandomAccessState.RAS_SHIP_DONE);
              done = true;
            }
          }
          tis.close();
          fis.close();
          
          newRAS.updateRAS(objBuff, cpPathSS, 500);
          recvEnd = System.currentTimeMillis();
          recvTotalTime += (recvEnd-recvStart);
          if(vverbose){
            System.out.println("update_10000( " + i + " ): " 
                               + (recvEnd-recvStart));
          }
          
          newRAS.close();
          newRAS = null;
        }catch(Exception e){
          System.err.println(" updateRAS inputstream  error: " + e.toString());
          e.printStackTrace();
          return;
        }
        
      }//j
      if(vverbose){
        System.out.println("avg_100000( " + i + " ): " + (totalTime/pass)); 
        System.out.println("avg_recv_100000( " + i + " ): " 
                           + (recvTotalTime/pass));
      }
     }//i
     if(verbose){
      System.out.print("OK.");
    }
  }

  //---------------------------------------------------------------------------
  // get average time of shipRAS for the same data store size
  //---------------------------------------------------------------------------
  public void test24(){
    if(verbose){
      System.out.print("Test24...");
    }
    long start, end;

    
    int fileNum = 10000;
    if(quick){
      fileNum = 1000;
    }
    repeatedWrite(ras, fileNum, fileNum);
    for(int i = 1; i < 10; i++){
      String outputFileName = "RASTest24.tmp";
      try{
        File tmpFile = new File(outputFileName);
        if(tmpFile.exists()){
          assert(tmpFile.isFile());
          tmpFile.delete();
        }
      }catch(Exception e){
        System.err.println("delete tmp file error: " + e.toString());
        e.printStackTrace();
        return;
      }
      FileOutputStream fos = null;
      TaggedOutputStream tos = null;
      try{
        fos = new FileOutputStream(outputFileName);
        tos = new TaggedOutputStream(fos);
      }catch(Exception e){
        System.err.println(" fake shipRAS outputstream  error: " 
                           + e.toString());
        e.printStackTrace();
        return;
      }
      String[] exclChildNames = new String[0];
      AcceptStamp[] as1 = new AcceptStamp[1];
      as1[0] = new AcceptStamp(-1, new NodeId(0));
      AcceptVV cpStartVV1 = new AcceptVV(as1);
      
      String cpPath = "/";
      SubscriptionSet cpPathSS = SubscriptionSet.makeSubscriptionSet("/*");
      try{
        start = System.currentTimeMillis();
        ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
        end = System.currentTimeMillis();
        if(vverbose){
          System.out.println("ScanTime_10000: " + (end-start));
        } 
        tos.close();
        fos.close();
      }catch(Exception e){
        System.err.println(" fake shipRAS outputstream  error: " 
                           + e.toString());
        e.printStackTrace();
        return;
      }
    }
    if(verbose){
      System.out.print("OK.");
    }
  }
  
  //---------------------------------------------------------------------------
  // get avg scan time for different dirty files with same data store size
  // simulate experiments for logvsCP
  //---------------------------------------------------------------------------
  public void test23a(){

   if(verbose){
      System.out.print("Test23a...");
    } 
    long start, end;
    int totalWrites = 4000;
    int fileNum = 1500;
    int pass = 10;
    if(quick){
      totalWrites = 1000;
      pass = 3;
    }
    
    repeatedWrite(ras, totalWrites, fileNum);
    for(int i = 10; i < totalWrites+1; i = (int)Math.round(i*1.5)){
      long totalTime = 0;
     
      for(int j = 0; j < pass; j++){
        String outputFileName = "RASTest23a.tmp";
        try{
          File tmpFile = new File(outputFileName);
          if(tmpFile.exists()){
            assert(tmpFile.isFile());
            tmpFile.delete();
          }
        }catch(Exception e){
          System.err.println("delete tmp file error: " + e.toString());
          e.printStackTrace();
          return;
        }
        FileOutputStream fos = null;
        TaggedOutputStream tos = null;
        try{
          fos = new FileOutputStream(outputFileName);
          tos = new TaggedOutputStream(fos);
        }catch(Exception e){
          System.err.println(" fake shipRAS outputstream  error: " 
                             + e.toString());
          e.printStackTrace();
          return;
        }
        String[] exclChildNames = new String[0];
        AcceptStamp[] as1 = new AcceptStamp[1];
        as1[0] = new AcceptStamp(totalWrites-i-1, new NodeId(0));
        AcceptVV cpStartVV1 = new AcceptVV(as1);
    
        String cpPath = "/";
        SubscriptionSet cpPathSS = SubscriptionSet.makeSubscriptionSet("/*");
        try{
          start = System.currentTimeMillis();
          ras.shipRAS(tos, cpPathSS, cpStartVV1, false);
          end = System.currentTimeMillis();
          totalTime += (end-start);
          if(vverbose){
            System.out.println("ScanTime_1500( " + i + " ): " + (end-start));
          } 
          tos.close();
          fos.close();
        }catch(Exception e){
          System.err.println(" fake shipRAS outputstream  error: " 
                             + e.toString());
          e.printStackTrace();
          return;
        }
      }//j
      if(vverbose){
        System.out.println("avg_1500( " + i + " ): " + (totalTime/pass));
      } 
    }//i
    if(verbose){
      System.out.print("OK.");
    }
  }

  //--------------------------------------------------------------------
  // util function for test22 and test23  to create given number of files
  // parmaters: [Num of Writes] [Num of File]
  //--------------------------------------------------------------------
  private void repeatedWrite(RandomAccessState updateRAS, 
                             int writeNum, int totalFile){
    
    int offset = 0; 
    int fileSize = 100;
    int count = 0;
    byte VAL_1 = 22;
    long start = System.currentTimeMillis();
    long nextStart = System.currentTimeMillis();
    long write100Time = 0;
    while(count < writeNum){
      
      String fileName = createFileName(count%totalFile, totalFile);
      doBound(updateRAS, fileName, 0, fileSize, count, 0, VAL_1, 0);      
      /*    
      if(vverbose){
	  System.out.println("Write " + fileName );
      }
      */
      count ++;
      if(vverbose&&(count%100 == 0)){
        nextStart = System.currentTimeMillis();
        write100Time = nextStart- start;
        
        System.out.println("total number of writes so far: " + count
                           + "    last 100 writes took time : " 
                           + write100Time);
        start = nextStart;
      }
    }
  }
  
  private String createFileName(int fileNo, int total){
    assert fileNo < total;
    final boolean dbgme = false; 
    final int BRANCHING_FACTOR = 10;
    int numLen = (int)(Math.ceil(Math.log(total)/Math.log(BRANCHING_FACTOR)));
    int fileLen = 2 * numLen; // including /'s as seperator 
    byte [] bName = new byte[fileLen];
    for(int i=0; i < fileLen; i++){
      if(i%2 == 0){// even position
        bName[i] = '/';
      } else {// odd position
        long base = Math.round(Math.pow(BRANCHING_FACTOR, 
                                        numLen-(i+1)/2));
        bName[i] = (byte) ('0' + fileNo/base);
        fileNo = (int)(fileNo%base);
        if(!(bName[i]>='0' && bName[i]<='9')){
          System.out.println("bName["+i+"]:" + bName[i]
                             + " base:" + base + "fileNo:"+ fileNo + "i: " + i
                             + "bName: " + new String(bName)
                             + " fileLen: " + fileLen + " total: " + total);
        }
        assert(bName[i]>='0' && bName[i]<='9');
      }
    }
    if(dbgme){
      System.out.println("fileNo: " + fileNo + " total: " + total
                         + " fileName:" + new String(bName));
    }
    return new String(bName);
  }


 /** 
 *  test25() -- test ScanFor_Update.  
 *    do a bunch of writes 
 *    experiment with different startVV and see if we can  
 *    get the correct updates 
 **/ 

  public void test25(){
   if(vverbose) System.out.println("RAS Test25: started...");
   int numFiles = 5;  // needs to be >= 2 for case 4 to work
   int numChunks = 2; 
   int sizeOfChunks = 10;

   populateRAS(numFiles, numChunks, sizeOfChunks);
   if(vverbose) System.out.println("RAS Test25: populatedDB RAS");

   try{
     SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/test/a/0:/test/b/*");
     UpdatePriorityQueue upq = new UpdatePriorityQueue(null);
     
     //case 1: scan of updates from beginning  
     if(vverbose) System.out.println("RAS Test25: case 1 started");
     AcceptStamp[] acceptA = { new AcceptStamp(-1, new NodeId(0)),
                               new AcceptStamp(-1, new NodeId(1)) };
     ras.scanForUpdates(upq, ss, new AcceptVV(acceptA));
     checkCase1(upq, numFiles, numChunks, sizeOfChunks);

     //case 2: scan of updates from the middle
     if(vverbose) System.out.println("RAS Test25: case 2 started");
     upq = new UpdatePriorityQueue(null);
     acceptA = new AcceptStamp[]{ new AcceptStamp(numFiles*numChunks/2 - 1, new NodeId(0)),
                                  new AcceptStamp(numFiles*numChunks/2 - 1, new NodeId(1)) };
     
     ras.scanForUpdates(upq, ss, new AcceptVV(acceptA));
     checkCase2(upq, numFiles, numChunks, sizeOfChunks);


     //case 3: scan of updates from the end
     if(vverbose) System.out.println("RAS Test25: case 3 started");
     upq = new UpdatePriorityQueue(null);
     acceptA = new AcceptStamp[]{ new AcceptStamp(numFiles*numChunks- 1, new NodeId(0)),
                                  new AcceptStamp(numFiles*numChunks- 1, new NodeId(1)) };
     
     ras.scanForUpdates(upq, ss, new AcceptVV(acceptA));
     checkCaseEmpty(upq);

     
     //case 4: delete 2 objs -- /test/a/0 -- with lastdelete after lastwrite
     //                      -- /test/a/1 -- with lastdelete before lastwrite
     if(vverbose) System.out.println("RAS Test25: case 4 started");
     doDelete("a/0", numChunks, 0);
     doDelete("a/1", 0, 0);
     ss = SubscriptionSet.makeSubscriptionSet("/test/a/0:/test/a/1");     
     acceptA = new AcceptStamp[]{ new AcceptStamp(-1, new NodeId(0)),
                                  new AcceptStamp(-1, new NodeId(1)) };
     
     upq = new UpdatePriorityQueue(null);
     ras.scanForUpdates(upq, ss, new AcceptVV(acceptA));
     checkCase4(upq, numFiles, numChunks, sizeOfChunks);

     
     //case 5: same as above but with startVV set to the end
     if(vverbose) System.out.println("RAS Test25: case 5 started");
     acceptA = new AcceptStamp[]{ new AcceptStamp(numFiles*numChunks-1, new NodeId(0)),
                                  new AcceptStamp(numFiles*numChunks-1, new NodeId(1)) };
     
     upq = new UpdatePriorityQueue(null);
     ras.scanForUpdates(upq, ss, new AcceptVV(acceptA));
     checkCaseEmpty(upq);

   }catch(Exception e) {
     e.printStackTrace();
     assert(false);
   }
  }

 //---------------------------------------------------------------------------
 // populateRAS() -- for Test 25: populates RAS with writes to /a/* and /b/*
 //---------------------------------------------------------------------------
 
    private void populateRAS(int numFiles, int numChunks, int sizeOfChunks){
    byte value = 65;

    String testFile = null;
    long time = 0;
    // populate /test/a/*  with NodeID 0 in ascending order
    if(vverbose) System.out.println("Populating /test/a");
    for (int i = 0; i < numFiles; i++){
      testFile = "a/" + i;
      for(int j = 0; j< numChunks; j++) {
        doInval(testFile, j*sizeOfChunks, sizeOfChunks, time, 0);
        doBody(testFile, j*sizeOfChunks, sizeOfChunks, time, 0, value, j*sizeOfChunks);
        time++;
      }
    }
    
    time = 0;
    // populate /test/b/*  with NodeID 1 decending order
   if(vverbose) System.out.println("Populating /test/b");
    for (int i = numFiles - 1; i >= 0; i--){
      testFile = "b/" + i;
      for(int j = numChunks - 1; j >= 0; j--) {
        doInval(testFile, j*sizeOfChunks, sizeOfChunks, time, 0);
        doBody(testFile, j*sizeOfChunks, sizeOfChunks, time, 1, value, j*sizeOfChunks);
        time++;
      }
    }
  }


 //---------------------------------------------------------------------------
 // check case1: for test 25 --- populates the expected list and 
 //                             compares it with the received upq 
 //---------------------------------------------------------------------------
 
  private void checkCase1(UpdatePriorityQueue upq, int numFiles, int numChunks, int sizeOfChunks) {
    //
    // Note: the upq will merge the ranges so instead of 
    // two entries (0 to i), (i+1 to j), it will be (0 to j)
    //
    ArrayList expectedBodies = new ArrayList();
    ObjId objId = null;
    MetaBody mBody = null;

    // expect just  /test/a/0
    objId = new ObjId("/test/a/0");
    mBody = new MetaBody(objId, 0, numChunks * sizeOfChunks);
    expectedBodies.add(mBody);
    
    // expect all of /test/b/*
    for (int i = numFiles - 1; i >= 0; i--){
      objId = new ObjId("/test/b/" + i);
      mBody = new MetaBody(objId, 0, numChunks * sizeOfChunks);
      expectedBodies.add(mBody);
    }

    if(vverbose) {
      System.out.println("RAS Test25: upq = ");
      upq.displayQueue();
      System.out.println("RAS Test25: expectedBodies = ");
      System.out.println(expectedBodies.toString());
    }
    compareLists(upq, expectedBodies);
  }

 //---------------------------------------------------------------------------
 // check case2: for test 25 --- populates the expected list and 
 //                             compares it with the received upq 
 //---------------------------------------------------------------------------
 
  private void checkCase2(UpdatePriorityQueue upq, int numFiles, int numChunks, int sizeOfChunks) {
    //
    // Note: the upq will merge the ranges so instead of 
    // two entries (0 to i), (i+1 to j), it will be (0 to j)
    //
    ArrayList expectedBodies = new ArrayList();
    ObjId objId = null;
    MetaBody mBody = null;

    // expect just  nothing from /test/a/*
    
    // expect just the 1st half of test/b/*
    int numChunksRequired = numChunks*numFiles - numChunks*numFiles/2;
    int currFile = 0;
    while (numChunksRequired > 0) {
      assert(currFile < numFiles);
      if (numChunks <= numChunksRequired) {
        // we can add the whole file to the list
        objId = new ObjId("/test/b/" + currFile);      
        mBody = new MetaBody(objId, 0, numChunks * sizeOfChunks);
        expectedBodies.add(mBody);
        numChunksRequired -= numChunks;
      } else {
        // add part of the file
        objId = new ObjId("/test/b/" + currFile);      
        mBody = new MetaBody(objId, 0, numChunksRequired * sizeOfChunks);
        expectedBodies.add(mBody);
        numChunksRequired = 0;
      }
      currFile++;
    }   

    if(vverbose) {
      System.out.println("RAS Test25: upq = ");
      upq.displayQueue();
      System.out.println("RAS Test25: expectedBodies = ");
      System.out.println(expectedBodies.toString());
    }
    compareLists(upq, expectedBodies);
  }

 //---------------------------------------------------------------------------
 // check case empty: for test 25 --- checks if the upq recieved is empty
 //---------------------------------------------------------------------------
 

  private void checkCaseEmpty(UpdatePriorityQueue upq){
   ArrayList expectedBodies = new ArrayList(); 
   if(vverbose) {
     System.out.println("RAS Test25: upq = ");
     upq.displayQueue();
     System.out.println("RAS Test25: expectedBodies = ");
     System.out.println(expectedBodies.toString());
   }
   
   if(!upq.isEmpty()) {
     assert(false) :" Upq should be empty!";
   }
 }
 
 //---------------------------------------------------------------------------
 // check case4: for test 25 --- populates the expected list and 
 //                             compares it with the received upq 
 //---------------------------------------------------------------------------
 
 private void checkCase4(UpdatePriorityQueue upq, int numFiles, int numChunks, int sizeOfChunks) {
    //
    // Note: the upq will merge the ranges so instead of 
    // two entries (0 to i), (i+1 to j), it will be (0 to j)
    //
    ArrayList expectedBodies = new ArrayList();
    ObjId objId = null;
    MetaBody mBody = null;

    // expect just  /test/a/1
    objId = new ObjId("/test/a/1");
    mBody = new MetaBody(objId, 0, numChunks * sizeOfChunks);
    expectedBodies.add(mBody);

    if(vverbose) {
      System.out.println("RAS Test25: upq = ");
      upq.displayQueue();
      System.out.println("RAS Test25: expectedBodies = ");
      System.out.println(expectedBodies.toString());
    }
    compareLists(upq, expectedBodies);
  }
 
 //---------------------------------------------------------------------------
 // compareLIsts -- for test 25 
 //                compares  arry list  with the received upq 
 //---------------------------------------------------------------------------
   
  private void compareLists(UpdatePriorityQueue upq, ArrayList expectedBodies){
    while(!upq.isEmpty()){
      MetaBody mBody = upq.getNext();
      boolean found = expectedBodies.remove(mBody);
      if(vverbose) {
        System.out.println("Body received from upq: " + mBody);
        if(found){
          System.out.println("Body removed from list");
        } else {
          System.out.println("Body not found in list");
        }
      }
    }

    assert(upq.isEmpty());
    assert(expectedBodies.isEmpty()): "UPQ did not receive all expected bodies";
  }


  //---------------------------------------------------------------------------
  // test26() -- test reads and writes with offset > 2^32
  //---------------------------------------------------------------------------
  public void test26(){
    Env.tbd("TBD: Write a test that tests reads and writes with offset > 2^32");
    System.out.println("TBD: Write a test that tests reads and writes with offset > 2^32");
    //    fail("TBD: Write a test that tests reads and writes with offset > 2^32");
  }

  //---------------------------------------------------------------------------
  // shadowDoInval() -- update the shadow array in memory to reflect
  //                        an inval
  //---------------------------------------------------------------------------
    private static void
      shadowDoInval(DBGShadow shadow[][], int iFile, int iOffset, int iLen, 
                        long iTime, long iNode){
      AcceptStamp as = new AcceptStamp(iTime, new NodeId(iNode));
      int fLen = shadow[iFile].length;
      int ii;

      for(ii = iOffset; ii < iOffset + iLen; ii++){
        shadow[iFile][ii].applyInval(as);
      }
    }


  //---------------------------------------------------------------------------
  // shadowDoBody() -- update the shadow array in memory to reflect
  //                       a bound inval
  //---------------------------------------------------------------------------
    private static void
      shadowDoBody(DBGShadow shadow[][], int iFile, int iOffset, int iLen, 
                       long iTime, long iNode, byte iVal){
      AcceptStamp as = new AcceptStamp(iTime, new NodeId(iNode));
      int fLen = shadow[iFile].length;
      int ii;

      for(ii = iOffset; ii < iOffset + iLen; ii++){
        shadow[iFile][ii].applyBody(as, iVal);
      }
    }


//---------------------------------------------------------------------------
// shadowCheckRead() -- check the stored data against the shadow
//---------------------------------------------------------------------------
    private void
    shadowCheckRead(String fileName, DBGShadow shadow[][], 
                      int iFile, int iOffset, int iLen){ 
      int fLen = shadow[iFile].length;
      int ii;
      DBGShadow s;
      byte expect;

      //
      // First do a read using the full range as arguments. Don't know
      // what to expect, but try this odd size read in case it throws
      // an exception.
      //
      checkReadDontCare(fileName, iOffset, iLen);

      // 
      // Now check byte-by-byte state
      //
      for(ii = iOffset; ii < iOffset + iLen; ii++){
        s = shadow[iFile][ii];
        if(s.checkStateDoesNotExist() || s.checkStateDeleted()){
          checkReadNonExistantOrHoleOrEOF(fileName, ii);
        }
        else if(s.checkStateInvalid()){
          checkReadInvalidRange(fileName, ii, 1);
        }
        else if(s.checkStateValid()){
          expect = s.checkValue();
          checkRead(fileName, ii, 1, 1, expect, true);
        }
      }
    }




//---------------------------------------------------------------------------
// doInval() -- apply an inval to the specified range of the specified
//   test file at the specified time.
//---------------------------------------------------------------------------
    private void
      doInval(String testFile, long offset, long length, long localTime, long node){
      ObjId id = new ObjId("/test/" + testFile);
      ObjInvalTarget oit = new ObjInvalTarget(id, offset, length);
      AcceptStamp as = new AcceptStamp(localTime, new NodeId(node));
      PreciseInv pi = new PreciseInv(oit, as);
      long ret;
      ret = ras.applyInval(pi);
      assert(ret == RandomAccessState.NO_NEW_VALID_BYTES);
    }


//---------------------------------------------------------------------------
// doBody() -- apply a body to the specified range of the specified
//   test file at the specified time. Check that the "newbytes" result
//   is as expected.
//---------------------------------------------------------------------------
    private void
      doBody(String testFile, long offset, long length, long localTime, long node,
                 byte value, long expectRet){
      long ret = doBodyNoCheckReturn(testFile, offset, length, localTime, node, value);
      assert ret == expectRet: "doBody: got " + ret 
        + " expected " + expectRet;
    }

//---------------------------------------------------------------------------
// doBodyNoCheckReturn() -- apply a body to the specified range of the specified
//   test file at the specified time. Return the "newbytes" offset
//---------------------------------------------------------------------------
    private long
      doBodyNoCheckReturn(String testFile, long offset, long length, long localTime, 
                              long node, byte value){
  
      assert(RandomAccessState.warnTBDLongLengths == true); // Fix the bug not the warning!
      assert(length < Integer.MAX_VALUE); // Danger -- see warning in constructor!
      ObjId id = new ObjId("/test/" + testFile);
      AcceptStamp as = new AcceptStamp(localTime, new NodeId(node));
      byte b[] = new byte[(int)length];
      int ii;
      for(ii = 0; ii < length; ii++){
        b[ii] = (byte)(value + ii);
      }
      ImmutableBytes im = new ImmutableBytes(b);
      BodyMsg bm = new BodyMsg(id, offset, length, as, im, false);
      long ret = ras.applyBody(bm);
      return ret;
    }


//---------------------------------------------------------------------------
// doBodyNoCheckReturn() -- apply a body to the specified range of the specified
//   test file at the specified time. Return the "newbytes" offset
//---------------------------------------------------------------------------
    private long
      doBodyNoIncrementValue(String testFile, long offset, long length, long localTime, 
                                 long node, byte value){
  
      assert(RandomAccessState.warnTBDLongLengths == true);// Fix the bug not the warning!
      assert(length < Integer.MAX_VALUE); // Danger -- see warning in constructor!
      ObjId id = new ObjId("/test/" + testFile);
      AcceptStamp as = new AcceptStamp(localTime, new NodeId(node));
      byte b[] = new byte[(int)length];
      int ii;
      for(ii = 0; ii < length; ii++){
        b[ii] = (byte)(value);
      }
      ImmutableBytes im = new ImmutableBytes(b);
      BodyMsg bm = new BodyMsg(id, offset, length, as, im, false);
      long ret = ras.applyBody(bm);
      return ret;
    }

//---------------------------------------------------------------------------
// doBound() -- apply a bound inval to the specified range of the specified
//   test file at the specified time. Check that the "newbytes" result
//   is as expected.
//---------------------------------------------------------------------------
    private void
      doBound(String testFile, long offset, long length, long localTime, long node,
                  byte value, long expectRet){
      assert(RandomAccessState.warnTBDLongLengths == true);// Fix the bug not the warning!
      assert(length < Integer.MAX_VALUE); // Danger -- see warning in constructor!
      ObjId id = new ObjId("/test/" + testFile);
      ObjInvalTarget oit = new ObjInvalTarget(id, offset, length);
      AcceptStamp as = new AcceptStamp(localTime, new NodeId(node));
      byte b[] = new byte[(int)length];
      int ii;
      for(ii = 0; ii < length; ii++){
        b[ii] = (byte)(value + ii);
      }
      ImmutableBytes im = new ImmutableBytes(b);
      BoundInval bi = new BoundInval(oit, as, im);
      long ret = ras.applyInval(bi);
      if(ret != expectRet){
        System.out.println("ret : " + ret + " and expect ret: " + expectRet
          + "when applying " + bi);
      }
      assert(ret == expectRet);
    }


//---------------------------------------------------------------------------
// doBound() -- apply a bound inval to the specified range of the specified
//   test file at the specified time. Check that the "newbytes" result
//   is as expected.
// different from above: have the ras as the parameter
//---------------------------------------------------------------------------
    private void
    doBound(RandomAccessState updateRAS, String testFile, long offset, long length, long localTime, long node,
                  byte value, long expectRet){
      assert(RandomAccessState.warnTBDLongLengths == true);// Fix the bug not the warning!
      assert(length < Integer.MAX_VALUE); // Danger -- see warning in constructor!
      ObjId id = new ObjId("/test/" + testFile);
      ObjInvalTarget oit = new ObjInvalTarget(id, offset, length);
      AcceptStamp as = new AcceptStamp(localTime, new NodeId(node));
      byte b[] = new byte[(int)length];
      int ii;
      for(ii = 0; ii < length; ii++){
        b[ii] = (byte)(value + ii);
      }
      ImmutableBytes im = new ImmutableBytes(b);
      BoundInval bi = new BoundInval(oit, as, im);
      long ret = updateRAS.applyInval(bi);
      if(ret != expectRet){
        System.out.println("ret : " + ret + " and expect ret: " + expectRet
          + "when applying " + bi);
      }
      assert(ret == expectRet);
    }

//---------------------------------------------------------------------------
// doDelete -- delete specified file at specified time
//---------------------------------------------------------------------------
    private void
      doDelete(String testFile, long localTime, long node){
      ObjId id = new ObjId("/test/" + testFile);
      AcceptStamp as = new AcceptStamp(localTime, new NodeId(node));
      ras.delete(id, as);
    }


//---------------------------------------------------------------------------
// checkRead -- check that the read is successful and
//   returns the expected result
//---------------------------------------------------------------------------
    private void
    checkRead(String testFile, long offset, long length,
              long expectLength, byte expectFirst, boolean exactOffset){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;
      int ii;
      ImmutableBytes body;
      byte b[];
      ObjInvalTarget target;

      try{
        got = ras.read(id, offset, length, exactOffset);
        target = got.getObjInvalTarget();
        assert(target.getObjId().equals(id));
        if (exactOffset) assert(target.getOffset() == offset);
	if(target.getLength() != expectLength){
	    System.out.println(got);
	}
        assert(target.getLength() == expectLength);
        body = got.getBody();
        b = body.dangerousGetReferenceToInternalByteArray();
        for(ii = 0; ii < expectLength; ii++){
          assert(b[ii] == (byte)(expectFirst + ii));
        }
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }


//---------------------------------------------------------------------------
// checkReadInvalidRange -- check that the read throws invalidRange
//    exception as expected
//---------------------------------------------------------------------------
    private void
    checkReadInvalidRange(String testFile, long offset, long length){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;

      try{
        got = ras.read(id, offset, length, true);
        assert(false);
        return;
      }
      catch(ReadOfInvalidRangeException ire){
        ; // Expected this
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }

//---------------------------------------------------------------------------
// checkReadHole -- check that the read hits a "hole" in the file
//---------------------------------------------------------------------------
    private void
    checkReadHole(String testFile, long offset){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;

      try{
        got = ras.read(id, offset, 1, true);
	//commented by zjd -- see RandomAccessState::read comments
	/*
        assert(got != null);
        assert(got.getLength() == 1);
        assert(got.getBody().getCopyBytes()[0] == 0);
        assert(got.getAcceptStamp().eq(ras.acceptOfHole));
        assert(got.getObjId().equals(id));
        return;
	*/
	
      }catch(ReadOfHoleException rhe){//added by zjd
	return; //expected
      }
      catch(ReadOfInvalidRangeException ire){
        ire.printStackTrace();
        assert(false);
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }


 /** 
 *  checkReadNonExistantOrHoleOrEOF -- check that the read hits a "hole" in the file 
 *                            or EOF (e.g., it hits some location that 
 *                            has not been written for a file that does exist.) 
 **/ 
    private void
    checkReadNonExistantOrHoleOrEOF(String testFile, long offset){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;

      try{
        got = ras.read(id, offset, 100, true);
        //commented by zjd -- see RandomAccessState::read comments
	/*
	assert(got != null);
        assert(got.getLength() == 1);
        assert(got.getBody().getCopyBytes()[0] == 0);
        assert(got.getAcceptStamp().eq(ras.acceptOfHole));
        assert(got.getObjId().equals(id));
        return;
	*/
	assert false;//added by zjd
      }catch(ReadOfHoleException rhe){//added by zjd
	;//Expected this
	return;
      }
      catch(EOFException eof){
        ; // Expected this
        return;
      }
      catch(NoSuchEntryException nse){
        ; // Expected this
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }


 /** 
 *  checkReadNoSuchEntry -- check that the read throws NoSuchEntry 
 *     exception as expected 
 **/ 
    private void
      checkReadNoSuchEntry(String testFile){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;

      try{
        got = ras.read(id, 0, 1, true);
        assert(false);
        return;
      }
      catch(NoSuchEntryException nsee){
        ; // Expected this
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }


 /** 
 *  checkReadExists -- check that specified file exists 
 **/ 
  private void
  checkReadExists(String testFile){
    ObjId id = new ObjId("/test/" + testFile);
    BodyMsg got;

    try{
      got = ras.read(id, 0, 1, true);
      return;
    }
    catch(NoSuchEntryException nsee){
      fail();
    }
    catch(EOFException e){
      ; // Expected this
      return;
    }
    catch(ReadOfEmbargoedWriteException roewe){
      fail();
    }
    catch(ReadOfInvalidRangeException roefe){
      ; // Expected this
      return;
    }
    catch(ReadOfHoleException rhe){
      ;//Expected this
      return;
    }
  }
  

 /** 
 *  checkReadEOF -- check that the read hits EOF 
 **/ 
    private void
      checkReadEOF(String testFile, long offset){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;

      try{
        got = ras.read(id, offset, 100, true);
        assert(false);
        return;
      }
      catch(EOFException eof){
        ; // Expected this
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }


 /** 
 *  checkReadDontCare -- check that the read does something sensible 
 *    (e.g., doesn't die on an internal sanity check) 
 **/ 
    private void
      checkReadDontCare(String testFile, long offset, long len){
      ObjId id = new ObjId("/test/" + testFile);
      BodyMsg got;

      try{
        got = ras.read(id, offset, len, true);
        return;
      }
      catch(NoSuchEntryException nsee){
        ; // Expected this
        return;
      }
      catch(EOFException e){
        ; // Expected this
        return;
      }
      catch(ReadOfEmbargoedWriteException roewe){
        ;
        return;
      }
      catch(ReadOfInvalidRangeException roefe){
        ; // Expected this
        return;
      }
      catch(ReadOfHoleException rhe){
	; //Expected this
	return;
      }
    }




 /** 
 *  checkReadMeta -- check that the read meta is successful and 
 *    returns the expected result 
 **/ 
    private static void
      checkReadMeta(RandomAccessState ras,
                        ObjId id, long offset,
                        long expectOffset, long expectLength,
                        AcceptStamp expectAccept){
      PreciseInv got;
      ObjInvalTarget target;
      try{
        got = ras.readMeta(id, offset);
        target = got.getObjInvalTarget();
        assert(target.getObjId().equals(id));
        assert(target.getOffset() == expectOffset);
        assert(target.getLength() == expectLength);
        assert(got.getAcceptStamp().equals(expectAccept));
        return;
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }



 /** 
 *  checkReadMetaEOF -- check that the read meta throws expected EOF  
 **/ 
    private static void
      checkReadMetaEOF(RandomAccessState ras,
                           ObjId id, long offset){
      PreciseInv got;
      ObjInvalTarget target;
      try{
        got = ras.readMeta(id, offset);
        assert(false);
        return;
      }
      catch(EOFException eof){
        return; // Expected result
      }
      catch(Exception e){
        e.printStackTrace();
        assert false: e.toString();
      }
    }



 /** 
 *  random() -- generate random numbers for test 
 **/ 
    static Random rng = new Random(49957);
    private static long
      random(long from, long to){
      assert(to > from);
      long r = rng.nextLong();
      if(r < 0){
        r = r * -1;
      }
      r = (r % (to - from)) + from;
      assert(r >= from);
      assert(r <= to);
      return r;
    }

 /** 
 *  selfTestCleanup -- delete all of the files we created 
 **/ 
  protected static void
  selfTestCleanup(String path){
    RandomAccessState.destroyAllStateCleanup(path);
  }

  
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(RandomAccessStateUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TestEmpty.testfoo() as a TestCase.
   */
   static public void main(String [] s) {
    String name = "RandomAccessStateUnit";
    System.err.print(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new RandomAccessStateUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }

    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.err.println(name + " self test succeeds");
  }

}



 /** 
 **/ 
 /** 
 *  
 *  class DBGShadow -- used for testing to track (in memory) what the state 
 *                     of a byte on disk should be. 
 *  
 **/ 
 /** 
 **/ 
  class DBGShadow{
    private AcceptStamp as;
    private int state; // STATE_VALID, STATE_INVALID, STATE_DELETED, ...
    private static final int STATE_VALID = 0;
    private static final int STATE_INVALID = 1;
    private static final int STATE_DELETED = 2;
    private static final int STATE_DOES_NOT_EXIST = 3;
    private byte value;

    private final static AcceptStamp 
    ACCEPT_BEGINNING_OF_TIME = new AcceptStamp(AcceptStamp.BEFORE_TIME_BEGAN, 
                                               new NodeId(NodeId.WILDCARD_NODE_ID));

    public
    DBGShadow(){
      as = ACCEPT_BEGINNING_OF_TIME;
      value = 0;
      state = STATE_DOES_NOT_EXIST;
    }

    public void
    applyDelete(AcceptStamp ds){
      assert(ds instanceof Immutable);
      if(ds.gt(as)){
        state = STATE_DELETED;
        as = ds;
      }
    }

    public void
    applyInval(AcceptStamp is){
      assert(is instanceof Immutable);
      if(is.gt(as)){
        state = STATE_INVALID;
        as = is;
      }
    }

    public void
    applyBody(AcceptStamp bs, byte val){
      assert(bs instanceof Immutable);
      if(bs.gt(as) || bs.equals(as)){
        state = STATE_VALID;
        value = val;
        as = bs;
      }
    }

    public boolean
    checkStateDoesNotExist(){
      if(state == STATE_DOES_NOT_EXIST){
        return true;
      }
      return false;
    }

    public boolean
    checkStateDeleted(){
      if(state == STATE_DELETED){
        return true;
      }
      return false;
    }

    public boolean
    checkStateInvalid(){
      if(state == STATE_INVALID){
        return true;
      }
      return false;
    }

    public boolean
    checkStateValid(){
      if(state == STATE_VALID){
        return true;
      }
      return false;
    }

    public byte
    checkValue(){
      assert(state == STATE_VALID);
      return value;
    }


  }
