package code;
 /** 
 *  Represents a connection that sends data and keeps trying to reconnect 
 *  if it fails. 
 **/ 
import java.net.Socket;
import java.io.IOException;
import java.io.*;

public class ReconnectConnection implements SocketMagicConstant
{
  private final long initialBackoffMS = 100;
  private final long maxBackoffMS = 100000;
  private long currentBackoffMS;
  private TaggedOutputStream s;
  private NetAddr netAddr;

  //private boolean constrainedRouter = true;
  private NodeId myNodeId;

 /** 
 *  Constructor 
 **/ 
  public ReconnectConnection(NetAddr na, NodeId myNodeId_)
    {
      this.netAddr = na;
      this.s = null;
      this.currentBackoffMS = initialBackoffMS;
      this.myNodeId = myNodeId_;
    }

  private static boolean warned = false;
 /** 
 *  Constructor 
 **/ 
  public synchronized void send(Object o) throws UnableToConnectException
    {
      Socket skt = null;
      ConstrainedOutputStream cos = null;

      while(true){  // Keep going until write succeeds or max timeout
        if(s == null){
          try{
	    if(OutgoingConnection.USE_HEARTBEATS){
	      skt = new HeartbeatSocket(netAddr.getDNS(),
					netAddr.getPort());
	    }else{
	      skt = new Socket(netAddr.getDNS(),
			       netAddr.getPort());
	    }
            skt.setTcpNoDelay(true);

            // Set constrainedRouter to true to enable bandwidth
            // limiter and delay router
            if(OutgoingConnection.USE_BW_CONSTRAINTS) {
              cos = new ConstrainedOutputStream(skt.getOutputStream(),
                                                Config.getBandwidth(this.myNodeId));
              s = new TaggedOutputStream(cos);
            }
            else {
              s = new TaggedOutputStream(skt.getOutputStream());
              cos = null;
            }

            // Any time we open a connection
            // the first thing we send is
            // the magic number and our nodeId so that the
            // receiving thread knows
            // who is talking.
            s.writeObject(new Long(INVAL_SOCKET_RECONNECT_MAGIC));
            if(!warned){
              Env.warn("Don't use the same magic # for different "
                       + "types of connection (Replace this with "
                       + "RECONNECTION_CONNECTION_MAGIC or something) "
                       + "(and fix the receivers)");
              warned = true;
            }
            s.writeObject(this.myNodeId);
            this.currentBackoffMS = initialBackoffMS;
          }catch(IOException e){
            if(currentBackoffMS > maxBackoffMS){
              throw new UnableToConnectException();
            }
            try{
              Thread.sleep(currentBackoffMS);
              currentBackoffMS *= 2;
            }catch(InterruptedException e2){
              System.err.println("ReconnectConnection: " +
                                 "InterruptedException thrown");
              assert(false);
            }
            s = null; // around we go again
            continue;
          }
        }
        try{         
          s.writeObject(o);
          return;
        }catch(IOException f){
          // write failed -- must be a bad socket
          try{
            s.close();
            s = null;
            System.err.println("Exception " + f + " thrown");
          }
          catch(IOException g){
            // Ignore error on close
          }
        }
      } // While true
    }
}

//---------------------------------------------------------------------------
/* $Log: ReconnectConnection.java,v $
/* Revision 1.17  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.16  2006/06/14 22:50:09  nalini
/* Changed nice sockets to normal sockets for outgoing body connections
/*
/* Revision 1.15  2005/10/13 00:24:24  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.14  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.13  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.12  2004/05/26 09:45:56  arun
/* *** empty log message ***
/*
/* Revision 1.11  2004/05/26 04:42:23  arun
/* *** empty log message ***
/*
/* Revision 1.10  2004/05/18 19:50:44  nayate
/* Changed ReconnectConnection to use ConstrainedOutputStream
/*
/* Revision 1.9  2004/05/14 20:49:00  dahlin
/* SDIMSController logic for Body connections (incoming)
/*
/* Revision 1.8  2004/05/14 00:12:43  zjiandan
/* Changes for second experiments.
/*
/* Revision 1.7  2004/05/11 23:45:44  nayate
/* Made the interface to HeartbeatSocket match that of Socket, and changed
/* classes to use HeartbeatSockets.
/*
/* Revision 1.6  2004/05/11 17:22:04  nayate
/* (1) Added "setTcpNoDelay(true)" calls at socket creation
/* (2) Changed "ObjectOutputStream" to "TaggedOutputStream" where appropriate
/*
/* Revision 1.5  2004/04/30 19:25:09  nayate
/* Changed the class to allow exponential backoff
/*
/* Revision 1.4  2004/04/26 23:08:31  nayate
/* Finished but haven't tested ReconnectConnection.java
/*
/* Revision 1.3  2004/04/19 23:30:05  lgao
/* Initial implementation of the RMI package and modification to Makefile.
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
