package code;
 /** 
 *  SDIMSController 
 *  Note: a lot of interfaces have changed since file creation, not sure if it 
 *  works or not --  Nalini 
 **/ 
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.HashMap;
import java.io.IOException;
import java.io.EOFException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileOutputStream;


public class SDIMSController implements Controller, TimeoutHandler, SpanningChangeHandler
{
    /*

      High level comments:

      (1) One thing that makes this design a bit complex is
      concurrency. The controller is a monitor that acts
      as a central point of control and coordinates lots of
      globally shared data to do this. So, we don't want
      to hold locks while we do blocking network operations.
      So, all of the work that goes to the network gets farmed
      out to worker threads that do the RMIs, SDIMS lookups, 
      etc. We find out that requests have succeeded when
      core calls us back telling us that some data/connection
      has arrived/started; or we find out that requests have
      failed by seeing a timeout.

      (2) Flexibility -- although this thing is designed 
      as an SDIMS controller, the "SDIMS" directory information
      is factored out into modules that can be replaced
      with stub versions for testing. (E.g., a static topology/hierarchy,
      broadcast search, ...)

      (3) SDIMS consistency and reliability -- we assume that once we
      insert something into SDIMS, it stays there across crashes (so
      we dont, for example, rescan the list of locally stored objects
      and re-push them all into SDIMS); we tolerate "stale" reads from
      SDIMS -- if we get an answer that tells us to go to the wrong
      node, we will get a timeout and we will force SDIMS to
      re-aggregate on the retry of the request. (Note: an alternative
      to assuming that the local SDIMS representative has its own
      persistent state and does its own recovery is for SDIMS
      reaggregation to have a way to query the core for the relevant
      state.)

      TBD

      o encapsulate references to SDIMS
      o callbacks from SDIMS when spanning tree parent changes

      0.1 -- SDIMS for read miss

        4 steps
        0.09 all-to-all subscribe updates; no hole filling; randomly
             select peer for reads; randomly select k
             gold replica

             o read retry logic (note: will use different peer, 
               so eventually we hit the right one...)

             o Update version vector on inval iterators to avoid
               resending same data for bidirectional inval
               subscriptions 

             o Update inval iterator to add delay parameter for
               sending on bound invals

             o Update inval iterator to add delay parameter to avoid
               sending on imprecise invals

             o Controller probes for k gold versions and then
               "unbinds" 
  

        0.11 use SDIMS for demand reads
             o Tell SDIMS when I add body to local storage

             o Tell SDIMS when I invalidate/delete body from local
               storage

             o Send demand read requests to target specified by SDIMS;
               retry on timeout

        0.12 add SDIMS for spanning trees; static interest sets
             o Tell SDIMS which interest sets I am subscribed to (to
               form main subscription multicast tree)

             o Ask SDIMS who to connect to for a given interest set to
               form spanning trees

             o Reconnect to spanning tree on connection break

        0.13 add SDIMS for hole filling
             o Tell SDIMS which ranges of accept stamps I have precise
               (to find "root" for hole-filling multicast tree)
 
             o Fill "holes" in interest set to bring lpVV up to cVV on
               demand read/write (or on short delay after hole
               appears); consider delaying applying invals to an
               interest set once a hole appears (to avoid having to
               reapply same invals after hole filled).

             o Tell SDIMS which holes I am currently trying to fill
               (to build hole-filling multicast tree)

        0.14 use SDIMS to fail-over to different gold replicas

             o Detect and deal with failures of "gold" set by sending
               bound to someone else (but don't yet worry about
               "recovering" "dead" gold copies to maintain k copies;
               defer this for now)


      0.2 -- add prefetching

             o SDIMS to track read rate and write rate of objects
               (core tells SDIMS about local reads and writes); this
               should be byte-weighted, right?

             o On local update, use SDIMS to find out priority to
               insert prefetch

             o Add TCP-Nice for prefetch stream; separate prefetch
               stream from demand stream

      0.9 -- add dynamically add interest sets

             o distributed checkpoint recovery (See TODO.txt for list
               of "Requirements for system to fully implement design
               described in paper")

             o Add enclosing interest set to precise set on read/write
               miss 

      1.0 -- add garbage collection: 
             o cache replacement (discard bodies of objects)
             o demote interest set (discard per-object state and make
               imprecise)
             o decommision machine/deal with long failures (and
               reallocate gold copies of data and metadata)
             o log garbage collection + coordinate checkpoints with
               LocalStore   

      Design sketch:

       
      the controller decides who we should connect to and receive updates/
      invalidations/etc. from

      This version of the controller
         (1) Uses SDIMS to locate data on a read miss

             tell sdims when I get a local copy
             tell sdims when I lose/invalidate my local copy
             sdims tracks ranges of bytes
             ask sdims when I get a miss

             issue: scalability -- do we need to bound the amount of
             DRAM consumed? do we need to cache important pointers and
             put the rest on disk? See attenutated bloom filters? 
             [one solution: limit the length in bits of the hash of
             fileID and figure out some way to deal with finding data
             if there is a collision. I think for now this issue can
             be deferred... How does pangea handle this -- does it
             keep pointers in memory or on disk? if in memory, how
             recover on crash? 

             issue: crash recovery -- just the regular SDIMS problem
             -- if a node goes down, don't want to have to do O(#
             objects) work to recover its state... This essentially is
             a generalization of the callback-state recovery problem;
             probably same design-space of solutions: use delayed
             invalidations to mask short term failures that don't
             cause reconfiguration of SDIMS; use per subrange of id
             space <parentID/child, epoch#> to track whether my local
             state is correct or whether I need to query my
             parent/push updates to my parent/query my child/push
             aggregate values to my child for this subrange of keys.

             --> To support this, need core to tell controller when it
             receives/creates local copy of data, when it discards
             local copy of data, when it needs local copy of data

         (2) what to keep precise

              version 0.1: static list (assume all local reads hit this list)

              version 0.9: dynamic add
                  LocalStore's "checkpoint" includes cVV/lpVV for
                  interest sets it is tracking (s.t. on-disk
                  per-object state is at least as new as the
                  corresponding cVV and lpVV). 

                  add an enclosing subdirectory on read/write miss

              version 1.0: dynamic remove
                  remove a subdirectory that has not been referenced
                  "in a long time" and that has received "many more"
                  updates/invals than demand requests (the
                  cost/benefit calculation is something like "to make
                  a given subdirectory precise, I will need to see M
                  messages where M is # of objects in the
                  subdirectory. If I expect to see more than M invals
                  before my next demand access, then it might make
                  sense to let this subdirectory go imprecise and
                  catch up later (of course, this will delay
                  processing the read request so I probably wait at
                  least until X hours have passed and expected number
                  of invals exceeds kM for some k

                  note: we need to make sure that at least j nodes
                  keep each interest set precise (discussed in more
                  detail below)

                  to support this
                  --> core needs to tell controller which directories
                  it is keeping precise; core needs to tell controller
                  when it gets a read miss for imprecise data;
                  controller needs to be able order core to start
                  keeping a new directory precise;

         (3) who to connect to for invals

              tell sdims which interest sets I am keeping precise

              policy question -- I can connect to several nodes that
              have enclosed interest sets, a node that has an exactly
              matching interest set, or a node that has an enclosing
              interest set. How to choose which? Also how does greedy
              construction of this tree compare to optimal? Also, is
              it OK to ignore "freshness" of different nodes'
              subscriptions?

              I think this boils down to two questions: (1) we need to
              maintain a spanning tree for interest set subscriptions
              so that updates percolate properly, (2) we need to
              generalize this spanning tree algorithm to deal
              with the fact that nodes' encoding of interest sets may
              differ. 

              Let's ignore the second problem for now and consider
              just the first one under the assumption that everyone is
              trying to form a spanning tree for *the same* interest
              set. 

              options:

              (1) spanning tree via SDIMS -- Tell SDIMS that I'm
                  interested in the set. SDIMS picks one node per
                  subtree; connect to my parent (bidirectional -- tell
                  my parent about my updates and get updates from my
                  parent). Note that if SDIMS picks me for multiple
                  levels, I connect to all of my multiple
                  parents. Claim: this will create a spanning tree
                  (eventual consistency); Question: load balancing?
                  (Should be OK -- if all N nodes care about an
                  interest set and if SDIMS is a d-ary tree, the root
                  node will see O(logN * d) connections; still, we
                  might want to limit indegree for any node; at any
                  given level, the reduction function could limit
                  indegree by publishing k "subtree roots" if there
                  are more than k*d children at that level (note that
                  the root roots would all have to subscribe to each
                  other...) I think this all works...
 
              (2) Pangea -- SDIMS maintains list of gold controllers
                  for the interest set; gold controllers maintain
                  lists of bronze copies; etc. user
                  controller-to-controller communication to identify
                  nodes to set up spanning tree

             I think both of these work. (1) is easier, so let's look
             at that for now. 

             Suppose we have /a/b/c and a set of nodes A interested in
             /a/*, a set of nodes B interested in /a/b/* and a set of
             nodes C interested in /a/b/c/*. 

             First, we *must* make sure that updates by anyone in C
             are communicated to A and B and that updates by anyone in
             A and B are communicated to everyone in C. At a minimum,
             if we have one node in C connect to at least one node in
             B to exchange invals about /a/b/c/* and at least one node
             in B connect to at least one node in A to exchange invals
             about /a/b/*, we're OK. So, this is simple enough -- if I
             am root for an interest set, I must subscribe to the root
             of the nearest enclosing interest set. (Or we may decide
             to go further and have every level-i root subscribe to
             some other level-i root, but this is not needed and
             actually won't save us anything in the current
             design...although it could reduce latency to see "nearby"
             updates, so maybe it is worthwhile). Also, I believe that
             if no one subscribes to "/", we are still OK -- suppose
             that A is an empty set -- then the root of B will find no
             one to subscribe to, but that's OK... Need to trigger an
             event if, later on, such a node appears.."If I become the
             root of an interest set, tell the roots of the enclosed
             interest sets...")

             An optimization might be to allow a node in C to notice
             that nearest parent in B is much closer than the nearest
             parent in C and subscribe to that parent in B (but still
             for /a/b/c/*) instead. But ensuring complete connectivity
             in this case seems harder. (Is it? Actually I think we're
             OK -- node c can subscribe to someone in B for /a/b/c/*
             which guarantees that they'll see all updates to
             /a/b/c/*; c *must* still advertize that it is subscribed
             to /a/b/c/* and accept children subscribing to it (for
             however many levels); but it does not need to subscribe
             to any parents (no matter how many levels it is chosen as
             root for.) Only issue is that it complicates in-degree
             load balancing (since /a/b aggregation underestimates how
             many nodes are subscribing at each level; this could be
             fixed...but only to a point... suppose there is 1 node
             in B but 1000 nodes in C and all of them want to
             subscribe to the parent in B as a shortcut? A workaround
             is to allow a node to refuse a "cross interest set"
             subscription except those needed for continuity -- e.g.,
             refuse a cross interest set subscription that spans
             levels but allow the level-to-level ones). Also, we lose
             some locality since all of my updates for c now need to
             go to b's root before coming back down; perhaps still
             send updates "up" along normal path... Overall -- I think
             all of this can work, but it is not clear if it buys us
             that much...

             Another optimization might be to allow a node in B to
             notice that the nearest parents in /a/b/c1/* and
             /a/b/c2/* are nearer than the nearest parents in /a/b/*
             and subscribe to them instead (assuming that c1 and c2
             represent all of the interest sets enclosed in b). Two
             challenges here: (1) I think it breaks the argument in
             the previous paragraph (or at least complicates it!) --
             guaranteeing connectivity and loop-freedom seems tougher
             in this case. (2) what happens when a new file "c3" is
             created? We either need to monitor changes in directory
             /a/b and notice when we need to make a new subscription
             or we need some way to subscribe to "/a/b/* except c1 and
             c2? We also should probably have our ISStatus say that we
             are precise for /a/b/c1/* and /a/b/c2/* and not claim to
             be precise for /a/b/*; instead, detect creation of new
             directories.  OK THAT MUST BE IT: local decision can be
             made to subscribe to each sub-interest set and to monitor
             for changes; but if they go that route, they shouldn't
             claim to be subscribed to whole shebang -- instead just
             subscribe to the individual pieces normally; then
             locally, I need to just notice that whenever the
             directory /a/b changes, the controller needs to read the
             new version of the directory to see if there is anything
             else it should subscribt to (so locally, the controller
             knows that we are subscribed to /a/b/*, but the rest of
             the system just thinks we're subscribed to /a/b/c1/* and
             /a/b/c2/*.  Only question is: how do you bootstrap up --
             if I am the first node to be interested in /a/*, all I
             can do is subscribe to all of the /a/b/*'s; so, I guess I
             should still subscribe to the /a/b/*'s but still tell the
             world that I am susbcribed to /a/*; (notice that our
             protocol ensures that it is still "ok" if I miss sending a
             precise update for, say, newly created file /a/b/c3 --
             what I send will be more imprecise than it should be, and
             anyone that receives it will have a hole to fill, but
             consistency is not lost.

      suppose I subscribe as described above, but despite my best
      efforts, someone sends me an imprecise invalidation for data I
      care about. What do I do now that an interest set that I want to
      be precise has become imprecise? 

      3 questions, really (?)
      (1) How do I find a node to go to that can supply me with the
      missing messages (and how do we avoid implosion overwhelming
      such a node?)

      (2) Locally, I may have received PPPPPIPPPPP for some interest
      set; how do I avoid forcing the system to re-send the P's after
      the I (since I already have seen them.)

      (3) How do we maintain the invarient that there is *someone*
      that I can go to that can bring me up to date? (How does garbage
      collection work? Do we need to have k "gold" copies of each
      subdirectory?)



         One option is to go to the horse's mouth -- go to the node(s)
         that participated in the imprecise invalidation and ask them
         for the missing messages. We could keep a bit of local state
         to recognize that replaying our local log through ourself
         would be smart once we've seen these later updates. The
         problem with this is implosion. Instead of connecting
         directly to the node, we could create an SDIMS tree for
         "interest set foo rooted at node bar" (that would be
         different from the default "interest set foo" tree); or maybe
         it is the "interest set foo written by node bar" tree or even
         the "interest set foo written by node bar starting at stamp
         ble" tree...

         option 2: what if I advertise into the tree for each interest
         set the accept stamp range for each node that I am precise
         for; for interest set I, node N will tell the DHT "N is
         current on I for writer w starting at X and ending at Y"
         (maybe we do a bit of filtering so that we don't update Y
         every time a new message arrives; one option is to have a
         special token NOW that means 'I am current up to about the
         current time as far as I know'). Now, the aggregation
         function for interest set I and writer w will track nodes
         that are "precise" for different ranges of accept stamp
         values. If I receive an imprecise invalidation for an
         interest set I am tracking, I can see which servers'
         invalidations I might have missed and even see which accept
         stamp ranges I might have missed; so, query SDIMS to find out
         a nearby node that has the data I want. Two things: (1) if I
         receive an imprecise invalidation spanning k nodes, I can
         either look up all k nodes and subscribe for the missing
         invals for each in parallel, or I can look up one of them and
         subscribe for *all* k nodes from that one -- there may be a
         good chance that it happens to have the data I need; if not,
         try one of the others (with <k-1 nodes in the subscription);
         (2) we still may have an implosion problem, if an imprecise
         inval for node N interest set I happens to get distributed to
         a subtree of nodes subscribing to interest set I, they may
         well see the same node M as the nearest node to fix the
         problem; we may need to announce our intentions to subscribe
         to interest set I node N startTime T to SDIMS and then ask
         SDIMS what parent to use... 

         So far, I like option 2. But it only deals with problem
         1. What about trying to make myself precise with local data
         after I fill a hole?

         option 2.1 -- avoid making holes -- delay in some sense,
         holes are a bug -- I subscribed to I for precise invals at
         got an imprecise one instead. Why? One of two reasons: (1) I
         received an imprecise invalidation that overlaps this
         interest set on a different channel or (2) because some
         upstream bozo ran into case 1 and then sent this imprecise
         one downstream. The former case is annoying; the latter case
         is usually "a bug" -- if that upstream node would have just
         waited a moment, the precise invalidation would have arrived
         (its subscribe to it!), and it could have forwarded the good
         information instead! It seems like the answer is: add a delay
         on invalIterator to wait when it encounters an imprecise
         inval that overlaps its subscription so that there is a
         chance for the precise one to come and refine it.  So, we can
         reduce the number of holes, and we probably should, but what
         do we do once one appears?  

         2.1 cont -- Also on the receiving side, once I see a hole for
         an interest set appear because I got an imprecise inval for A
         from channel for B, delay processing incoming invals for
         interest set A (for some timeout) until the hole is closed
         (?) [Hopefully, the common case is that very soon after
         channel B sends us the imprecise inval that overlaps A, one
         of the A channels sends us the precise inval that fixes the
         problem?] 

         option 2.2 -- hack -- assume that holes are rare so don't
         bother to optimize for this case

         option 2.3 -- hack -- always try to fill hole after locally
         before remotely -- once we fill the hole, try subscribing for
         the missing stuff locally; if that doesn't work, try
         somewhere else.
 
         option 2.4 -- track start/end ranges locally; possibly feed
         them to sdims -- SDIMS could maintain for each interest
         set/writer pair, a accept-range-to-precise-node mapping
         (e.g., I can look up for a given {interest set, writer, and
         accept stamp starting point} a nearby node I can connect to
         to become precise for that range. (We should also (a) tell
         SDIMS when I subscribe to {interest set, writer, and accept
         stamp start} and form a tree using that rather than the first
         table to avoid implosion -- only the root of the second tree
         actually goes to the first one; another option is to form the
         multicast tree on {interest set, writer, accept stamp start,
         node that is supplying data to me} to allow different trees
         for different suppliers; either prevents implosion. (b) do
         the hole avoidance discussed above. (c) if I get an imprecise
         inval covering several writers, probably still good to just
         look up a node that can fill 1 writer's hole and subscribe to
         all missing writers from that node -- only create anther
         connection if that one fails to completely fill the hole


         So, now it looks like 2.4 (+2.1) are a solid answer to the
         first two questions. Last question: how do we make sure that
         there is *someone* that can fill the hole. 

         This is actually straightforward, I think. We *already* have
         decided that there need to be at least k nodes that store the
         data body for each write; those nodes necessarily also store
         the precise metadata


         So, for all of this to work, we need
            --> on demand read to imprecise, make the interest set
                precise
            --> for the hole-filling case, need controller to ask core
                when hole is filled and it can revert back to relying
                on regular stream for that interest set



         (4) unbind policy -- making sure there are at least k bound
             copies of data

             mechanism: at least k chosen neighbors for each interest
             set need to subscribe for invals with "no delay"; (anyone
             else that subscribes for invals probably wants to put an
             Delay_unbind parameter that delays bound messages for up
             to the specified amount of time); on a write, controller
             needs to first sync() with those k neighbors and then
             once we know that they have the data, issue an "unbind"
             directive
 
             2 questions: how to choose those k nodes and how to deal
             with the case that one or more of them fails. 

	     version 0.1: Just manually select for each node k buddies
	     and list them in the config file. 

             Version .99: randomly select k buddies per node; if you
             detect that a buddy has become unreachable, immediately
             select a different one for new writes. Now the trick is
             to notice when for a given item, too few copies
             exist. Try this: for each object for which I hold a bound
             copy, register the bound copy with SDIMS and then ask
             SDIMS to tell us when the set of holders of the bound
             copies change. Controller keeps a *local* *persistent*
             list of other holders of bound copies of each object that
             I have a bound copy of (don't want to depend on SDIMS for
             persistent storage of this info); if we detect that node
             N has died, need to scan through my data to find out
             which data now has fewer than k copies and need to select
             one of the controllers to create an additional copy

                So, SDIMS will keep a mapping from {objID, range} to
                list of "gold" copies

                Each controller with a bound copy will maintain local
                copy of this list

                SDIMS will keep a list of "live" nodes; if we have any
                bound invals from that node, subscribe for
                notifications of that node going dead (how best to do
                this in SDIMS?)

                if controller hears that a node is dead, go through
                bound all objects stored both here and at that node;
                if my ID is closest to objId, then my job to select a
                new node to make a new bound copy (or after timeout,
                all nodes try)

                send a bound inval to the specified node (we're going
                to need a new "subscribeSendBound" interface to send
                bound invalidations matching a set of criteria to a
                particular node and for that node to bind them (even
                though they may be "old"))



         (5) What to prefetch
              use SDIMS to track read and write rate for inserts to
              upq

              --> keep per object read rate and write rate (weighted
                  by bytes)
                  on enqueue to UPQ, lookup read rate and write rate
                  from SDIMS to set priority 

         (6) who to connect to for updates

         (7) Who is primary (for commits)?

              no longer needed

         (8) cache replacement policy
              local decision but also need to make sure at least k
              good copies

              --> need to mark in local metadata in DataStore which
                  data are bound
              --> need interface for sending bound to someone else if
                  I want to evacuate

         (9) log garbage collection policy
         (10) local precise state garbage collection policy

             what do I need to do to guarantee that there exists some
             node that can make me precise?

               one answer: if I am responsible for holding bound
               invalidations for interest set I writer W, then I am
               responsible for keeping log of all writes to I by W for
               after acceptstamp{W, T} until I know that at least k
               nodes have lpVV for I s.t. I.lpVV[W] > T; and if I have
               lpVV[W] = T', I cannot discard the per-object state
               unless I know at least k other nodes have lpVV[W] >= T'

             (also, if I have bound updates in a subdirectory that I
             want to make imprecise and throw away the per-object
             metadata, I need to send those updates to someone else)

         (11) Key questions about appropriateness of SDIMS --
              (1) reconnection time (what if my parent dies -- what
              metadata do I have to update to reintegrate to system);
              (2) do we need to page to disk?, (3) load, (4)
              is eventual consistency enough for this app? (How do
              design controller so that eventual consistency is enough)
    */


private Core core;
private ReadDirectory readDir;
private SpanningDirectory spanningDir;

private TimeoutQueue timeouts;
private InvalSpanningList invalSpanningList;
private RMIClient rmiClient;
private WorkQueue spanningTreeWork;
private SDIMSInterface sdims;
private PendingDemandList pendingDemand;
private WorkQueue demandReadWork;
private long readRequestId;
private WorkQueue directoryUpdatesWork;
private LinkedList interestSetsICareAbout;
private Thread spanningTreeWatchdog;
private final static boolean TBD = false;
private HashMap boundWrites; // AcceptStamp --> BoundStatus
private boolean doPushUpdates;

private final boolean dbgSpanningTree = true;
private final String dstTag = "DBG Spanning Tree in SDIMSController:";
private final boolean dbgVerboseInvals = false;
private final String dbgvi = "DBG verbose invals SDIMSController:";

private int N_DIRECTORY_UPDATE_WORKERS;
private int N_DEMAND_READ_WORKERS;     
private int N_SPANNING_TREE_WORKERS;   
private long MAX_DELAY_NONOVERLAPPING_MS;

/*
 *------------------------------------------------------------------
 *
 * SDIMSController --
 *
 *          Constructor
 *
 * Arguments:
 *      Core core -- the core that we control
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public SDIMSController(Core core_, ReadDirectory readDir_, SpanningDirectory spanDir_,
		       RMIClient rmiClient_, int minNodesToForceBoundWritesTo,
		       boolean doPushUpdates_,
		       int nDirUpdateWorkers, int nDemandReadWorkers, 
		       int nSpanningTreeWorkers, long maxDelayNonOverlappingMS )
{  
    int ii;

    this.core = core_;
    this.readDir = readDir_;
    this.spanningDir = spanDir_;
    this.rmiClient = rmiClient_;
    this.readRequestId = 0;
    this.spanningTreeWatchdog = null;

    Env.warn("TBD: Controller needs to delete the interfaces:\n"
	     + "   informCommitStreamInitiated\n"
	     + "   informUnbindStreamInitiated\n"
	     + "   informCommitStreamTerminated\n"
	     + "   informUnbindStreamTerminated");
    Env.warn("TBD: Controller needs to add the interfaces:\n"
	     + "   informOutgoingInvalSreamInitiated\n"
	     + "   informOutgoingInvalSreamTerminated\n"
	     + "   (and core needs to call at appropriate times)\n");

    N_DIRECTORY_UPDATE_WORKERS = nDirUpdateWorkers;
    N_DEMAND_READ_WORKERS = nDemandReadWorkers;
    N_SPANNING_TREE_WORKERS = nSpanningTreeWorkers;
    MAX_DELAY_NONOVERLAPPING_MS = maxDelayNonOverlappingMS;

    this.doPushUpdates = doPushUpdates_;

    this.timeouts = new TimeoutQueue();
    TimeoutQueueWorker timeoutWorker = new TimeoutQueueWorker(timeouts, this);
    timeoutWorker.start();
    this.invalSpanningList = new InvalSpanningList();
    

    this.directoryUpdatesWork = new WorkQueue("DirectoryUpdates");
    for(ii = 0; ii < N_DIRECTORY_UPDATE_WORKERS; ii++){
	DirectoryUpdateWorker duWorker;
	duWorker = new DirectoryUpdateWorker(directoryUpdatesWork, 
					     readDir);
	duWorker.start();
    }
    

    this.pendingDemand = new PendingDemandList();
    this.demandReadWork = new WorkQueue("DemandReadWork");
    for(ii = 0; ii < N_DEMAND_READ_WORKERS; ii++){
        DemandReadWorker dWorker = new DemandReadWorker(this, demandReadWork, 
	                                                timeouts, rmiClient, 
							readDir,
                                                        core.getMyNodeId());
        dWorker.start();
    }
    
    this.spanningTreeWork = new WorkQueue("SpanningTreeWork");
    for(ii = 0; ii < N_SPANNING_TREE_WORKERS; ii++){
        SpanningTreeWorker stWorker = new SpanningTreeWorker(this, 
							     spanningTreeWork, 
	                                                     timeouts, 
							     rmiClient, 
							     spanningDir,
							     core);
        stWorker.start();
    }
    
    //
    // Join spanning tree for each interest set
    //
    InterestRegion ir = Config.getInterestSet(core.getMyNodeId());
	SubscriptionSet ss = ir.makeSubscriptionSet();
    //core.registerIS(is);
    subscribeInvalsSpanning(ss);
    interestSetsICareAbout = new LinkedList();
    interestSetsICareAbout.addFirst(ss);
    /* 
     * Future:
     *
     ISIterator isi = null; 
     isi = core.getInterestSets();
     while(isi.hasNext()){
       InterestSet is = isi.getNext();
       interestSetsICareAbout.addFirst(is);
       subscribeInvalsSpanning(is);
     }
     *
     */


    


    //
    // Support for unbinds
    //
    BoundStatus.setGoldCount(minNodesToForceBoundWritesTo);
    boundWrites = new HashMap();
    Env.performanceWarning("TBD: For bound writes we should set up "
			   + "fast-path channels to bound writes to "
			   + "k neighbors; these channels should "
			   + "have delayForBound = 0 and should "
			   + "be filtered to only send writes for "
			   + "this writer; we will also want to "
			   + "update our logic that notices when "
			   + "connections fail to notice the difference "
			   + "between this bound channel failing and "
			   + "a spanning tree connection failing and "
			   + "to re-open the bound channel connection.");



    Env.warn("SDIMSController TBD: prefetching channels");
    Env.warn("SDIMSController TBD: prefetching channel formation assumes "
	     + "channels do signallng to avoid sending redundant");
    Env.warn("SDIMSController TBD: hole filling for invalidations");
    Env.warn("SDIMSController TBD: prefetching statistics");
    Env.performanceWarning("SDIMSController TBD: refind spanning tree "
			   + "so that if two nodes have multiple connections "
			   + "for multiple interest sets, they can "
			   + "combine those connections");
      
}

 /** 
 **/ 
  //    inform methods for send-checkpoint protocol
 /** 
 **/ 
  
 /** 
 *  called in SubscribeInvalWorker when !(this.omitVV < startVV) i.e 
 *  gap exist between the sender's omitVV and the requested stream's  
 *  startVV. It implies that the ongoing subscribe inval stream is  
 *  failed. 
 *  
 *  What expected in the controller is to make the decision whether 
 *  to subscribe inval from other nodes for the receiver or invoke 
 *  a subscribeCheckpoint request to the same node optionally followed  
 *  by a subscribeInval request with a higher startVV 
 **/ 
  public void informGapExistForSubscribeInv(NodeId invReceiver,
                                            SubscriptionSet ss,
                                            VV startVV,
                                            VV omitVV){
    Env.printDebug("informGapExistForSubscribeInv");
  }

 /** 
 *  inform Outgoing Checkpoint Stream initiated called in CPSendWorker 
 **/ 
  public void
  informOutgoingCheckpointStreamInitiated(NodeId targetNodeId,
                                          String cpPath,
                                          String[] exclChildNames,
                                          VV startVV,
                                          boolean placeholderWriterSet){
    Env.printDebug("informOutgoingCheckpointStreamInitiated");
  }

 /** 
 *  inform Outgoing Checkpoint Stream initiated called in CPSendWorker 
 **/ 
  public void
  informOutgoingCheckpointStreamTerminated(NodeId receiverNodeId, 
                                           String cpPath,
                                           String[] exclChildNames,
                                           VV startVV, 
                                           boolean placeholderWriterSet){
    Env.printDebug("informOutgoingCheckpointStreamTerminated");
  }

 /** 
 *  inform a Checkpoint Stream initiated called in CPRecvWorker 
 **/ 
  public void
  informCheckpointStreamInitiated(NodeId senderNodeId, 
                                  String cpPath,
                                  String[] exclChildNames,
                                  VV vvStart, 
                                  boolean placeholderWriterSet){
    Env.printDebug("informCheckpointStreamInitiated");
  }

 /** 
 *  inform a Checkpoint Stream terminated called in CPRecvWorker 
 **/ 
  public void
  informCheckpointStreamTerminated(NodeId senderNodeId, 
                                   String cpPath,
                                   String[] exclChildNames,
                                   VV vvStart,
                                   boolean placeholderWriterSet){
    Env.printDebug("informCheckpointStreamTerminated");
  } 

 /** 
 *  inform a Checkpoint Stream apply status called in CPRecvWorker 
 **/ 
  public void
  informCheckpointStreamReceiveStatus(NodeId senderNodeId, 
                                      String cpPath,
                                      String[] exclChildNames,
                                      VV vvStart, 
                                      boolean applyStatus,
                                      boolean placeholderWriterSet){
    Env.printDebug("informCheckpointStreamTerminated"); 
  }

 /** 
 *  inform subscribe invalidate for a certain subscription set failed 
 **/ 
  public void
  informSubscribeInvalFailed(NodeId senderNodeId,
                             NodeId receiverNodeId,
                             SubscriptionSet ss,
                             VV vvStart){
    Env.printDebug("informSubscribeInvalFailed");
  }

 /** 
 *  inform subscribe invalidate for a certain subscription set failed 
 **/ 
  public void
  informSubscribeInvalSucceeded(NodeId senderNodeId,
                                NodeId receiverNodeId,
                                SubscriptionSet ss,
                                VV vvStart){
    Env.printDebug("informSubscribeInvalSucceeded");
    // need to update SDIMS
  }



 /** 
 *  inform successfully adding subscriptionSet to OutgoingConnection's 
 *  invalidateIterator's interestset 
 **/ 
  public void 
  informOutgoingInvalStreamInitiated(NodeId receiverId, 
				     VV startVV,
				     boolean placeholderWriteSet){
    Env.printDebug("informOutgoingInvalStreamInitiated");
  }


 /** 
 *  inform successfully removed subscriptionSet from 
 *  OutgoingConnection's invalIterator's interest set 
 **/ 
  public void
  informOutgoingSubscribeInvalTerminated(NodeId receiverId,
					 SubscriptionSet subscriptionSet){

    Env.printDebug("informOutgoingSubscribeInvalTerminated");
    // need to update this method so that it informs SDIMS
  }


 /** 
 *  inform established an outgoing body connection to targetNodeId 
 **/ 
  public void
  informOutgoingBodyStreamInitiated(NodeId receiverID, 
				    boolean placeholderWriterSet){
    Env.printDebug("informOutgoingBodyStreamInitiated");
  }


				
 /** 
 *   inform outgoing body connection terminated 
 **/ 
  public void 
  informOutgoingBodyStreamTerminated(NodeId receiverNodeId,
				     SubscriptionSet is,
				     boolean placeholderWriteSet){
    Env.printDebug("informOutgoingBodyStreamTerminated");
  }

 /** 
 *  inform SS was added to a incomming body stream 
 **/ 
  public void
  informSubscribeBodySucceeded(NodeId senderNodeId,
                               SubscriptionSet ss){
    Env.printDebug("informSubscribeBodySucceeded");
  }

 /** 
 *  inform SS was removed from an incomming body stream 
 **/ 
  public void
  informSubscribeBodyRemoved(NodeId senderNodeId,
                             SubscriptionSet ss){
    Env.printDebug("informSubscribeBodyRemoved");
  }


  

/*
 *------------------------------------------------------------------
 *
 * handleTimeout --
 *
 *          Called by TimeoutQueueWorker when a timeout occurs.
 *
 * Arguments:
 *      Object event -- the request that needs to be retried.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void
handleTimeout(Object event)
{
    Class cl = event.getClass();

    if(cl.getName().equals("PendingDemandRequest")){
	handleDemandReadTimeout((PendingDemandRequest)event);
	return;
    }
    else if(cl.getName().equals("SubscribeInvalRequest")){
	handleSubscribeInvalsTimeout((SubscribeInvalRequest)event);
	return;
    }

    assert(false); // Unexpected type 
    return;
}


/*
 *------------------------------------------------------------------
 *
 * informBodyStreamInitiated --
 *
 *          A body stream was initiated to this node. Cancel
 *          retry timer.
 *
 *          Note -- senders aggregate body streams across
 *          interest sets, so this stream may match
 *          multiple spanning trees.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informBodyStreamInitiated(NodeId senderNodeId)
{
    if(dbgSpanningTree){
	Env.inform(dstTag + " enter informBodyStreamInitiated() for " 
		   + senderNodeId.toString());
    }

    List matchingSpanning = invalSpanningList.find(senderNodeId);
    if(matchingSpanning == null || matchingSpanning.size() == 0){
	// Connection is not from my parent in spanning tree.
	//
	// Most likely, this is a connection from one of my children.
	// (Another possibility is that this is an old connection that 
	// has arrived late -- after we gave up on it and went to someone
	// else, in which case we end up with a redundant connection. 
	// But that should be rare and does no major harm.)
	//
	// For debugging purposes, keep track of these connections
	// to (probable) children.
	//

	Env.sprintln("informBodyStreamInitiated() called because " + senderNodeId + "connected to us");
	  
	if(warnHIISTSI){
	    Env.warn("TBD: SDIMSController should track child subscriptions "
		     + "for debugging");
	    warnHIISTSI = true;
	}
	/* TBD:
	   childBodySpanningTree.insert(nodeId);
	   return;
	*/
	return;
    }



    ListIterator li = matchingSpanning.listIterator();
    if(li == null){
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " informBodyStreamInitiated() finds no matching spanning " 
		       + senderNodeId.toString());
	}
    }

    while(li.hasNext()) {
	SubscribeInvalRequest r = (SubscribeInvalRequest)li.next();

	assert(r != null); // Why? MDD

	//
	// Connection is from parent in spanning tree, but already
	// connected. Probably, we had a timeout and retry?
	//
	if(r.getPushChannelStatus() == SubscribeInvalRequest.STATUS_CONNECTED){
	    Env.performanceWarning("SDIMSController detects possible redundant spanning "
				   + "prefetch body tree connection. "
				   + "We may be sending the same data "
				   + "multiple times. This really should be fixed!");
	    /* 
	     * TBD:
	     * core.controllerKillRedundantBodyStream(senderNodeId, is, ALL_WRITERS); 
	     */
	    return;
	}


	//
	// Connection is an expected incoming connection from my parent
	// in spanning tree.
	//
	assert(r != null);
	Env.sprintln("Changing pushChannel status of " + r + " from PENDING to CONNECTED");
	// Arun: commented this since currently PushChannel is not being 
	// activated. Hopefully, this doesn't break anything.
	//Env.remoteAssert(r.getPushChannelStatus() == SubscribeInvalRequest.STATUS_PENDING);
	r.setPushChannelStatus(SubscribeInvalRequest.STATUS_CONNECTED);
	Env.sprintln("Changed pushChannel status of " + r + " from PENDING to CONNECTED");

    }

}

/*
 *------------------------------------------------------------------
 *
 * informBodyStreamTerminated --
 *
 *          A body stream was terminated. May need to retry/
 *          reopen connection.
 *
 *          Note -- senders aggregate body streams across
 *          interest sets, so this stream may match
 *          multiple spanning trees.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that had been connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warnIBST = false;

public synchronized void 
informBodyStreamTerminated(NodeId senderNodeId)
{
    if(dbgSpanningTree){
	Env.inform(dstTag 
		   + " informBodyStreamTerminated()  " 
		   + senderNodeId.toString());
    }

    List matchingSpanning = invalSpanningList.find(senderNodeId);
    if(matchingSpanning == null || matchingSpanning.size() == 0){
	//
	// This is not a connection we care to maintain (anymore?)
	//
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " informBodyStreamTerminated() finds no matching spanning " 
		       + senderNodeId.toString());
	}

	return;
    }

    //
    // Retry connection
    //
    ListIterator li = matchingSpanning.listIterator();
    while(li.hasNext()){
	SubscribeInvalRequest r = (SubscribeInvalRequest)li.next();

	r.setPushChannelStatus(SubscribeInvalRequest.STATUS_FAILED);
	Env.sprintln("Stream " + r + " terminated and is being restarted");
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " informBodyStreamTerminated() restarting " 
		       + senderNodeId.toString()
		       + " r:" + r.toString());
	}
	spanningTreeWork.insert(r); // hand request off to a thread

	if(!warnIBST){
	    Env.performanceWarning("SDIMSController::informBodyStreamTerminated reconnects"
				   + " but reconnectionConnection might already be doing"
				   + " the reconnect --> make sure we don't end up with"
				   + " multiple body connections");
	    warnIBST = true;
	}
	return;
    }


}


/*
 *------------------------------------------------------------------
 *
 * informReverseBodyStreamTerminated --
 *
 *            .
 *
 * Arguments:
 *      --
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void informOutgoingSubscribeBodyTerminated(NodeId targetNodeId, 
							       SubscriptionSet ss)
{
    if(dbgSpanningTree){
	Env.inform(dstTag 
		   + " informOutgoingSubscribeBodyTerminated() entered " 
		   + targetNodeId.toString());
    }

    //assert(placeholderWriterSet);
    SubscribeInvalRequest r = invalSpanningList.find(targetNodeId, ss);
	
    //
    // This is the reverse connection to my parent in the spanning tree
    //
    if(r != null){
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " informOutgoingSubscribeBodyTerminated() "
		       + "-- lost connectin to parent" 
		       + targetNodeId.toString()
		       + " r: " + r.toString());
	}
	r.setReversePushChannelStatus(SubscribeInvalRequest.STATUS_FAILED);
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " informOutgoingSubscribeBodyTerminated() "
		       + "-- lost connectin to parent" 
		       + targetNodeId.toString()
		       + " r: " + r.toString());
	}
	Env.sprintln("Reverse connection - getting updates - from " + targetNodeId);
	spanningTreeWork.insert(r);
    }
    
}


/*
 *------------------------------------------------------------------
 *
 * informOutgoingSubscribeBodyInitiated --
 *
 *          .
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that had been connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void informOutgoingSubscribeBodyInitiated(NodeId targetNodeId, 
							      SubscriptionSet ss) 
{
    //
    // See if this is a reverse connection in spanning tree
    //
    if(dbgSpanningTree){
	Env.inform(dstTag 
		   + " informOutgoingBodyStreamInitiated() "
		   + targetNodeId.toString());
    }

    //assert(placeholderWriterSet); /* if writer set != all nodes{ return} */
    if(true){
	SubscribeInvalRequest r = invalSpanningList.find(targetNodeId, ss);
	if(r == null){
	    // not a connection to parent in spanning tree
	    return;
	}
	
	r.setReversePushChannelStatus(SubscribeInvalRequest.STATUS_CONNECTED);
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " informOutgoingBodyStreamInitiated() "
		       + targetNodeId.toString()
		       + " r: " + r.toString());
	    
	}
    }
}

/*
 *------------------------------------------------------------------
 *
 * informSyncReplyStreamInitiated --
 *
 *          A sync reply stream was initiated to this node. Cancel
 *          retry timer.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private boolean warnisrsi = false;
public synchronized void 
informSyncRplyStreamInitiated(NodeId senderNodeId)
{
    if(warnisrsi == false){
	Env.warn("TBD: SDIMSController::informSyncReplyStreamInitiated()");
	warnisrsi = true;
    }
}

/*
 *------------------------------------------------------------------
 *
 * informSyncReplyStreamTerminated --
 *
 *          A sync reply stream was terminated. May need to retry/
 *          reopen connection.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that had been connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private boolean warnisrst = false;
public synchronized void 
informSyncRplyStreamTerminated(NodeId senderNodeId)
{
    if(warnisrst == false){
	Env.warn("TBD: SDIMSController::informSyncReplyStreamTerminated()");
	warnisrst = true;
    }
}



/*
 *------------------------------------------------------------------
 *
 * informCommitStreamInitiated, informUnbindStreamInitiated 
 * informCommitStreamTerminated, informUnbindStreamTerminated --
 *          OBSOLETE -- DELETE FROM ALL CONTROLLERS
 *
 * Arguments:
 *      NA
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informCommitStreamInitiated(NodeId senderNodeId)
{
    assert(TBD); // This interface needs to be deleted from Controller!
}
public synchronized void 
informUnbindStreamInitiated(NodeId senderNodeId)
{
    assert(TBD); // This interface needs to be deleted from Controller!
}
public synchronized void 
informCommitStreamTerminated(NodeId senderNodeId)
{
    assert(TBD); // This interface needs to be deleted from Controller!
}
public synchronized void 
informUnbindStreamTerminated(NodeId senderNodeId)
{
    assert(TBD); // This interface needs to be deleted from Controller!
}






/*
 *------------------------------------------------------------------
 *
 * recvSyncReply --
 *
 *          Received a sync() reply for a specific accept stamp.
 *          Once we receive enough of these, we can unbind it.
 *
 * Arguments:
 *      AcceptStamp acceptStamp - identifies the write (that we issued)
 *          that has made it to a node.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
recvSyncReply(AcceptStamp acceptStamp, NodeId senderNodeId)
{
  	Env.sprintln("Received sync reply with accept stamp " + acceptStamp + " from " + senderNodeId);
  
    Env.remoteAssert(acceptStamp.getNodeId().equals(core.getMyNodeId()));
    BoundStatus status = (BoundStatus)boundWrites.get(acceptStamp);
    if(status == null){
	return; // Already unbound
    }
    boolean done = status.update(senderNodeId);
    if(done){
	UnbindMsg unbind = new UnbindMsg(status.getObjInvalTarget(), 
					 acceptStamp);
	core.applyUnbind(unbind);
	boundWrites.remove(acceptStamp);
    }
}

/*
 *------------------------------------------------------------------
 *
 * informRecvPushBody --
 *
 *          Received pushed body of a range of bytes in a file.
 *          Tell SDIMS we have it.
 *
 * Arguments:
 *      BodyMsg msg -- identifies the file and range of bytes we have
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informReceivePushBody(BodyMsg msg)
{
    //
    // Update SDIMS directory
    //
    DirectoryUpdate du = new DirectoryUpdate(DirectoryUpdate.INSERT, msg);
    directoryUpdatesWork.insert(du);
}

/*
 *------------------------------------------------------------------
 *
 * informRecvInval --
 *
 *          Received inval for a range of bytes in a file.
 *          Tell SDIMS we have no longer have it *UNLESS*
 *          this is a bound inval, in which case tell SDIMS
 *          that we do have it.
 *
 * Arguments:
 *      GeneralInv inv -- identifies the affected file and range of bytes
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warniri = false;
public synchronized void 
informReceiveInval(GeneralInv inv, NodeId senderId)
{
    //
    // Update directory
    //
    DirectoryUpdate du;
    if(dbgVerboseInvals){
	Env.inform(dbgvi + " got inval " + inv.toString());
    }

    // check if inv is precise and newer than data on localstore
    if(inv.isPrecise() && !inv.isBound()) {
      assert(inv instanceof PreciseInv);
      PreciseInv pi = (PreciseInv) inv;
      ObjInvalTarget oit = pi.getObjInvalTarget();
      PreciseInv ei;

      try {
        ei = core.readMeta(oit.getObjId(), oit.getOffset());
      } 
      catch(ObjNotFoundException e){
        // This inval is newer than what we have; ignore exception and continue
        ei = null;
      } 
      catch(EOFException e){
        // This inval is newer than what we have; ignore exception and continue
        ei = null;
      }
      catch(ReadOfHoleException rhe){
	//This inval is newer; ignore exception and continue
	ei = null;
      }
      catch (IOException e) {
        ei = null;
        e.printStackTrace();
        assert(false);
      }
      
      if(ei!= null && !pi.getAcceptStamp().gt(ei.getAcceptStamp())){
        // In this case, local write is showing up as an
        // invalidate to yourself and polluting directory
        // maintained by SDIMS.
        return;	
      }
    }
	
    if(inv.isBound()){
      try{
        du = new DirectoryUpdate(DirectoryUpdate.INSERT, inv);
      }
      catch(IsImpreciseException e){
        assert(false);
        return;
      }
    }
    else{
      try{
        du = new DirectoryUpdate(DirectoryUpdate.INVAL, inv);
      }
      catch(IsImpreciseException e){
        return; // Can't insert imprecise inval to directory
      }
    }
    directoryUpdatesWork.insert(du);
}


/*
 *------------------------------------------------------------------
 *
 * informLocalWrite --
 *
 *          We just did a local write. Tell SDIMS we have a copy
 *          of the specified range of bytes. Also, create a
 *          BoundStatus object to track who has received the
 *          bound invalidation so we can decide when to unbind
 *          the inval.
 *
 * Arguments:
 *      ObjId objId -- object updated
 *      long offset -- start of range of bytes
 *      long length -- length of update
 *      boolean isBound -- if the write was bound
 *      boolean isEmbargoed -- if the write was embargoed
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void informLocalWrite(ObjId objId, long offset, long length,
					  AcceptStamp as, boolean isBound,
                                          boolean isEmbargoed)
{
    //
    // Update SDIMS directory
    //
    DirectoryUpdate du = new DirectoryUpdate(DirectoryUpdate.INSERT, 
					     objId, offset, length);
    directoryUpdatesWork.insert(du);

    //
    // Track bound state
    //
    BoundStatus status = new BoundStatus(objId, offset, length, as);
    assert(boundWrites.get(as) == null);
    boundWrites.put(as, status);
    if(boundWrites.size() > 1000 && boundWrites.size() % 1000 == 0){
	Env.warn("Unexpectedly large number of bound writes oustanding: " 
		 + boundWrites.size());
    }
}

/*
 *------------------------------------------------------------------
 *
 * informLocalDelete --
 *
 *          We just did a local delete. Tell SDIMS we no longer 
 *          have a copy of the specified object (for any range
 *          of bytes.)
 *
 * Arguments:
 *      ObjId objId -- object deleted
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void informLocalDelete(ObjId objId)
{
    //
    // Update SDIMS directory
    //
    DirectoryUpdate du = new DirectoryUpdate(DirectoryUpdate.INVAL, 
					     objId, 0, Long.MAX_VALUE);
    directoryUpdatesWork.insert(du);
}

/*
*-----------------------------------------------------------------------
*    Fake method
*-----------------------------------------------------------------------
*/
    public void informLocalReadImprecise(ObjId objId, long offset, long length)
    {
	Env.printDebug("informLocalReadImprecise");
    }

/*
*-----------------------------------------------------------------------
*    Fake method
*-----------------------------------------------------------------------
*/
    public void informLocalReadInvalid(ObjId objId, long offset, long length,
				       AcceptStamp inval)
    {
	Env.printDebug("informLocalReadInvalid");
    }


/*
 *------------------------------------------------------------------
 *
 * informDemandMiss --
 *
 *          We just tried to read an object and had a miss.
 *          Ask SDIMS where to send a demand read request, send
 *          it, and set a timeout event to trigger retries.
 *
 *          We considered including the accept stamp as an argument
 *          but decided against doing so (for now.) (1) we don't always 
 *          know [e.g., a miss for a new] and (2) it may change between
 *          request issue and reply recv time (e.g., new invalidations 
 *          arrive in the mean time). So, at best it would be a hint 
 *          -- but not a terribly useful one -- we probably always want 
 *          to go get the newest one we can lay our hands on anyhow.
 *          So, we rely on thread in core to kick us if new data comes
 *          back and it needs us to re-issue request (that will cancel
 *          the timeout). (See informReceiveDemandReply).
 *
 * Arguments:
 *      ObjId objId -- object updated
 *      long offset -- start of range of bytes
 *      long length -- length of update
 *
 * Postcondition:
 *      There is a pending request for at least the first byte
 *      of the read.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warnedidrm = false;

public synchronized void 
informDemandReadMiss(ObjId objId, long offset, long length)
{
    if(!warnedidrm){
	Env.performanceWarning("On a demand miss, if length is just a"
			       + " few bytes, do we want to increase it?"
			       + " Do we want to issue a prefetch?");
	warnedidrm = true;
    }

    PendingDemandRequest pdr = new PendingDemandRequest(objId, offset, 
							length, readRequestId++,
							new AcceptStamp(-1, new NodeId(0)));
    /*
     * If read overlaps another read, make sure that first byte of
     * this request is issued with existing or new request. 
     * --> If this offset is less than pending offset, then
     * issue this request (but shorten the length). If pending offset
     * is later than this one, shorten then length so they no longer
     * overlap. Note that it is OK to shorten this length even if
     * newThis + pending shorter than originalThis; we only have
     * to guarantee to get first byte.
     */
    boolean isCompletelyRedundant = pendingDemand.insertAndRemoveRedundant(pdr);
    if(isCompletelyRedundant){
	return;
    }
	Env.sprintln("Read miss for " + objId + " : [" + offset + "," + length + ". Asking SDIMS to find a copy");
    demandReadWork.insert(pdr);
    return;
}

/*
 *------------------------------------------------------------------
 *
 * informDemandReply --
 *
 *          Stop retrying; we now have local copy of data.
 *          Note -- the copy that just arrived may turn out to be 
 *          "too old" to be useful. In that case the core should 
 *          first call informReceiveDemandReply  so that we know to 
 *          stop retrying that request and then call informDemandReadMiss 
 *          so we know to issue a new request.
 *    
 * Arguments:
 *      BodyMsg msg -- identifies what we received
 *
 * Results:
 *      Postcondition is that there is no longer a pending
 *      request whose first byte is covered by bodyMsg.
 *      (Note that we may cancel multiple requests and 
 *      that not all of them will be fully satisfied. But
 *      every request we cancel will have had its first
 *      byte satisfied.)
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informReceiveDemandReply(NodeId senderId, BodyMsg msg)
{
    // 
    // remove corresponding request from pendingDemand
    // note -- if this is a partial reply, no need to re-issue; caller will do so
    //
    pendingDemand.remove(msg);

    //
    // Update SDIMS directory
    //
    DirectoryUpdate du = new DirectoryUpdate(DirectoryUpdate.INSERT, msg);
    directoryUpdatesWork.insert(du);

} 

/*
 *------------------------------------------------------------------
 *
 * informDemandReadHit
 *
 *         To support controller cache replacement policy
 *         [TBD: Add interface to core: "discardBody()" to discard
 *         the body of an object from local storage and "removeIS()"
 *         to discard an interest set (which allows us to discard
 *          the per-object metadata like last known acceptStamp, etc.]
 *
 * Arguments:
 *      ObjId id -- referenced object
 *      long offset -- start
 *      long length -- length
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informDemandReadHit(ObjId objId, long offset, long length)
{
    /*
     * Do nothing for now. Later we will update cache replacement state
     */
}


/*
 *------------------------------------------------------------------
 *
 * handleDemandReadTimeout --
 *
 *          If the data have not arrived, then retry.
 *
 * Arguments:
 *      PendingDemandRead pdr -- the read that timed out.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected synchronized void
handleDemandReadTimeout(PendingDemandRequest pdr)
{

    if(pendingDemand.isPresent(pdr) == false){
	//
	// Data arrived before timeout
	//
	return;
    }
    if(pdr.getRetryCount() > PendingDemandRequest.MAX_RETRY_COUNT){
	pendingDemand.remove(pdr);
	assert(TBD);/* TBD: core.demandReadFailed(pdr);*/
	return;
    }
    pdr.incrementRetryCount();
	Env.sprintln("Demand read miss timeout for " + pdr.getObjId() + " : [" + pdr.getOffset() + "," + pdr.getLength() + ", retrying...");
    demandReadWork.insert(pdr);
    return;

}


/*
 *------------------------------------------------------------------
 *
 * informDemandImprecise --
 *         
 *          We have received a demand read for an interest set that is
 *          currently imprecise. The controller may need to identify 
 *          an enclosing interest set to make precise (either an interest
 *          set that the local core is already tracking or make a call
 *          to the local core telling core to begin tracking a new
 *          interest set [TBD: add this call to core].) Then, the controller
 *          may need to contact remote nodes and tell them to set up
 *          a subscription to this node. If objId belongs to an interest
 *          set that the core is already tracking, enclosingIS identifies
 *          it and lpVV indicates the last precise VV (from which the new
 *          subscription must start.) On the other hand, if the
 *          object does not belong to an interest set being tracked,
 *          enclosingIS and lpVV will be null. In that case, the controller
 *          has to tell the core to start tracking a new IS and when
 *          the controller subscribes this node to a stream of invals,
 *          this stream must start at time 0 OR we need to get a checkpoint
 *          from another node [TBD: add checkpoint transfer interface
 *          to node and core.]
 *         
 *          Note: informDemandImprecise brings the ISStatus up to date
 *          and informDemandRead brings the object up to date; normally, 
 *          if a core hits an imprecise interest set, it calls informDemandImprecise
 *          first and then calls informDemandReadMiss immediately (so that
 *          both requests can be doing their work in parallel)
 *
 * Arguments:
 *      ObjId objId -- the referenced object
 *      InterestSet enclosingIS -- the enclosing imprecise interest set we
 *              are tracking (--> need to fill the hole) or null if we 
 *              are not tracking any enclosing interest set (--> need
 *              to set up new interest set to track)
 *      VV lpVV -- last precise version vector for enclsoing interest set
 *              or null if no enclosing interest set
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informDemandImprecise(ObjId objId, 
		      PreciseSet enclosingPS, 
		      VV lpVV)
{
    assert(enclosingPS == null || lpVV != null); // enclosingIS != null --> lpVV != null
    Env.warn("TBD: SDIMSController::informDemandImprecise");
    assert(TBD);
}


/*
 *------------------------------------------------------------------
 *
 * informBecameImprecise --
 *
 *          An interest set we are tracking has become imprecise.
 *          May want to fix hole now or may want to wait until
 *          a demand read comes along.
 *          Update SDIMS with range for which we are precise.
 *
 * Arguments:
 *      InterestSet is -- this IS is currently imprecise
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informBecameImprecise(PreciseSet ps)
{
    Env.warn("TBD: SDIMSController::informBecameImprecise");
    Env.warn("TBD: Controller probably needs to add a lpVV argument to informBecameImprecise");
}

/*
 *------------------------------------------------------------------
 *
 * informBecamePrecise --
 *
 *          An interest set we are tracking has become precise.
 *          Update SDIMS with range for which we are precise.
 *
 * Arguments:
 *      InterestSet is -- this IS is currently imprecise
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informBecamePrecise(PreciseSet ps)
{
    Env.warn("TBD: SDIMSController::informBecamePrecise");
    Env.warn("TBD: Controller probably needs to add a lpVV argument to informBecamePrecise");
}














/*
 *------------------------------------------------------------------
 *
 * subscribeInvalsSpanning --
 *
 *          Attache to the spanning tree for the specified interest
 *          set.
 *
 * Arguments:
 *      InterestSet is -- the interest set we care about.
 *
 * Postcondition:
 *      There is subscription or a pending subscription for 
 *      the specified interest set OR we discover that we
 *      are the root of the spanning tree and we subscribe
 *      to the spanning tree of the enclosing interest set
 *      (if is is not the root IS).
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private void
subscribeInvalsSpanning(SubscriptionSet ss)
{
    if(dbgSpanningTree){
	Env.inform(dstTag + " enter subscribeInvalSpanning() for " 
		   + ss.toString());
    }

    if(spanningTreeWatchdog == null){
	spanningTreeWatchdog = new SpanningTreeWatchdog(this);
	spanningTreeWatchdog.start();
    }

    //
    // All of our spanning tree maintenance code in SDIMSController
    // assumes we form one spanning tree per subdirectory
    //
    assert(ss.containsExactlyOneSubdirectory()); 

    VV startVV = core.getLpVV(ss); 

    NodeId target = new NodeId(NodeId.WILDCARD_NODE_ID);
    SubscribeInvalRequest r = new SubscribeInvalRequest(ss, startVV, target,
							doPushUpdates);
    
    boolean itsNew = invalSpanningList.insert(r);
    if(!itsNew){
	if(dbgSpanningTree){
	    Env.inform(dstTag + " exit subscribeInvalSpanning() " 
		       + " --- already working on it ---"
		       + "is: " + ss.toString()
		       + "r: " + r.toString());
	}
	return; // We already have a request out to subscribe to this IS
    }

   Env.sprint("Attaching to the spanning tree for interest set " + ss.toString());
   spanningTreeWork.insert(r); // hand request off to a thread

    if(dbgSpanningTree){
	Env.inform(dstTag + " exit subscribeInvalSpanning() -- enqueud " 
		   + r.toString());
    }

}

/*
 *------------------------------------------------------------------
 *
 * handleSubscribeInvalsTimeout --
 *
 *          An attempt to set up an invalidation subscription timed
 *          out. Called by timeout thread.
 *
 * Arguments:
 *      SubscribeInvalRequest r -- the request that timed out
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected synchronized void 
handleSubscribeInvalsTimeout(SubscribeInvalRequest r)
{

    if(dbgSpanningTree){
	Env.inform(dstTag + " enter handleSubscribeInvalTimeout() for " 
		   + r.toString());
    }
  
    //
    // Assert -- we always believe we have exactly one
    // connection to one parent
    //
    SubscribeInvalRequest check;
    check = invalSpanningList.find(r.getTarget(), r.getSubscriptionSet());
    assert(check != null);
    if(!check.equals(r)){
	//
	// We've already gotten a callback that the old parent has
	// changed to a new one... This timeout is no longer relevant.
	//

	if(dbgSpanningTree){
	    Env.inform(dstTag + " exit handleSubscribeInvalTimeout() " 
		       + "no work needed (returning)");
	}
	return;
    }

    
    //
    // See if any of the four connections -- inval down, inval reverse
    // body down, body reverse -- needs attention.
    //
    if(r.getStatus() != SubscribeInvalRequest.STATUS_CONNECTED
       ||
       r.getReverseConnectionStatus() != SubscribeInvalRequest.STATUS_CONNECTED
       ||
       (r.getDoPush() && (r.getPushChannelStatus() 
			  != SubscribeInvalRequest.STATUS_CONNECTED))
       ||
       (r.getDoPush() && (r.getReversePushChannelStatus() 
			  != SubscribeInvalRequest.STATUS_CONNECTED))){
	
	if(r.getRetryCount() > SubscribeInvalRequest.WARNING_RETRY_COUNT){
	    Env.warn("SDIMSController::handleSubscribeInvalsTimeout() "
		     + "sees suspicious # of timeouts. Do we need to "
		     + "add code to deal with case where we cannot subscribe "
		     + "to an interest set? (e.g., call core to tell any "
		     + "pending requests to return error?");
	}
	r.setPendingToFailed();
	r.incrementRetryCount();

	if(dbgSpanningTree){
	    Env.inform(dstTag + " exit handleSubscribeInvalTimeout enqueuing " 
		       + r.toString());
	}

	Env.sprintln("Inval subscription attempt " + r + " timed out, retrying...");
	spanningTreeWork.insert(r); // hand request off to a thread to retry  
    }
    else{
	// 
	// All needed connections are happy. No need to take
	// any action on this timeout.
	//
	return;
    }


}


/*
 *------------------------------------------------------------------
 *
 * handleRootSpanningTree --
 *
 *          SDIMS determined that I am the root of the spanning
 *          tree for this interest set. Cancel the subscription
 *          request for this interest set and instead subscribe
 *          to the next enclosing interest set (unless this 
 *          interest set is root interest set.)
 *
 * Arguments:
 *      SubscribeInvalRequest r -- the request for which we discovered
 *                                 that we are the root of the spanning 
 *                                 tree
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected synchronized void
handleRootSpanningTree(SubscribeInvalRequest r)
{
    //
    // All of our spanning tree maintenance code in SDIMSController
    // assumes we form one spanning tree per subdirectory
    //
    assert(r.getSubscriptionSet().containsExactlyOneSubdirectory()); 
    String isAsString;

    if(dbgSpanningTree){
	Env.inform(dstTag + " enter handleRootSpanningTree() with " 
		   + r.toString());
    }


    if(r.getStatus() != SubscribeInvalRequest.STATUS_I_AM_ROOT_OF_SPANNING_TREE){
	// status has changed between call issue and arrival here
	if(dbgSpanningTree){
	    Env.inform(dstTag + " exit handleRootSpanningTree() --  " 
		       + "I'm not root");
	}

	return;
    }

    //
    // Code assumes interest set corresponds to a subdirectory
    //
    Class ISClass = r.getSubscriptionSet().getClass();
    assert(ISClass.getName().equals("SubscriptionSet")); 
    SubscriptionSet ss = (SubscriptionSet)(r.getSubscriptionSet());

    //
    // Subscribe to the parent interest set.
    //
    SubscriptionSet parentSS = ss.getParentSS();
    if(parentSS == null){
	//
	// This is root interest set. No need to go further.
	//
	if(dbgSpanningTree){
	    Env.inform(dstTag + " exit handleRootSpanningTree() -- " 
		       + " This is root directory");
	}

	//assert(r.getInterestSet().toString().equals("(/)"));
	return;
    }

    if(dbgSpanningTree){
	Env.inform(dstTag + " handleRootSpanningTree() -- " 
		   + " subscribe to enclosing directory " 
		   + parentSS.toString());
    }

    subscribeInvalsSpanning(parentSS);
}


/*
 *------------------------------------------------------------------
 *
 * notifySpanningChange --
 *
 *          Called when the spanning directory notices
 *          that our parent in the spanning tree for
 *          some interest set changed.
 *
 *          Remove the old record, close down the old connections,
 *          and re-attach to spanning tree
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void
    notifySpanningChange(String type, String name, 
			Object value )
{
    if(dbgSpanningTree){
	Env.inform(dstTag + " enter notifySpanningChange()");
    }


    assert(TBD);
    /*
      assert type is spannig tree
      is = name
      NodeId new parent = value
      SubscribeInvalRequest r = invalSpanningList.find(is)
      if(r == null){
        return; // we no longer care about this is
      }
      if(parent.equals(newParent)){
         return; // We already know this parent
      }
      if(old parent was me and new parent isn't me)
        renclosingIS = invalSpanningList.find(enclosing is)
        if renclosingIS is not on interestSetsICareAbout list
           remove renclosingIs from invalSpanningList
           shut down connections to and from enclosing spanning tree
    
      remove r from invalSpanningList
      tell core to shut down connections to and from old parent
      subscribeInvalsSpanning(is)
    */
}

/*
 *------------------------------------------------------------------
 *
 * informSubscribeInvalInitiated --
 *
 *          An inval stream was initiated to this node. Cancel
 *          retry timer and update status.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that connected to us
 *      InterestSet is -- the interest set subscribed to
 *
 *      Note : with OLD API
 *      VV startVV -- the startVV used for the connection
 *      boolean placeholderWriterSet -- in the future, we will
 *         support subscriptions that specify a "writerSet"
 *         (like an interest set) that specifies that we want
 *         to see precise invals from writers in the set
 *         and imprecise outside). This placeholder is
 *         to remind us that this code will depend on
 *         that information once that feature is supported.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warnediisi = false;
public synchronized void 
informInvalStreamInitiated(NodeId senderNodeId, SubscriptionSet ss,
			   VV vvStart, boolean placeholderWriterSet)
{
    if(dbgSpanningTree){
	Env.inform(dstTag + " enter informInvalStreamInitiated() with " 
		   + senderNodeId.toString()
		   + " ss: " + ss.toString());
    }


    if(!warnediisi){
	Env.performanceWarning("TBD: need to add logic to bidirectional "
			       + "inval streams to avoid sending back what "
			       + "they just sent us!");

	Env.warn("TBD: Controller::informInvalStreamInitiated probably needs to "
		 + "include list of writers as arguments"
		 + "so we can distinguish hole filling from spanning tree connections.");

	assert(placeholderWriterSet); /*  this code only handles spanning tree case*/

	warnediisi = true;
    }



     //
     // Right now only handle spanning tree case
     //
     assert(placeholderWriterSet);
     handleInformIncomingSpanningTreeStreamInitiated(senderNodeId, ss);
}


/*
 *------------------------------------------------------------------
 *
 *  handleInformIncomingSpanningTreeStreamInitiated--
 *
 *          An inval stream was initiated to this node as
 *          part of spanning tree for interest set. Cancel
 *          retry timer and update status.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that connected to us
 *      InterestSet is -- the interest set covered
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */

private static boolean warnHIISTSI = false;

private void
handleInformIncomingSpanningTreeStreamInitiated(NodeId senderNodeId, 
						SubscriptionSet ss)
{


    SubscribeInvalRequest r = invalSpanningList.find(senderNodeId, ss);

    if(dbgSpanningTree){
	Env.inform(dstTag 
		   + " enter handleInformIncomingInvalStreamInitiated() with " 
		   + " sender: " + senderNodeId.toString()
		   + " ss: " + ss.toString()
		   + " request: " 
		   + (r==null?"NULL":r.toString()));
    }

    //
    // Case 1:
    //
    // Connection is not from my parent in spanning tree.
    //
    // Most likely, this is a connection from one of my children.
    // (Another possibility is that this is an old connection that 
    // has arrived late -- after we gave up on it and went to someone
    // else, in which case we end up with a redundant connection. 
    // But that should be rare and does no major harm.)
    //
    // For debugging purposes, keep track of these connections
    // to (probable) children.
    //
    if(r == null){ 
	if(warnHIISTSI){
	    Env.warn("TBD: SDIMSController should track child subscriptions "
		     + "for debugging");
	    warnHIISTSI = true;
	}
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " exit handleInformIncomingInvalStreamInitiated() " 
		       + " -- child's upward connection -- "
		       + " sender: " + senderNodeId.toString()
		       + " ss: " + ss.toString()
		       + " request: " 
		       + (r==null?"NULL":r.toString()));
	}

	/* TBD:
	SpanningTreeKey key = new SpanningTreeKey(is, senderNodeId);
	childSpanningTree.insert(key);
	*/
	return;
     }

    //
    // Case 2:
    //
    // Connection is from parent in spanning tree, but already
    // connected. Probably, we had a timeout and retry.
    //
    if(r.getStatus() == SubscribeInvalRequest.STATUS_CONNECTED){
	Env.performanceWarning("SDIMSController detects possible redundant spanning "
			       + "tree connection. We may be sending the same data "
			       + "multiple times. This really should be fixed!");
	/* 
	 * TBD:
	 * core.controllerKillRedundantInvalStream(senderNodeId, is, ALL_WRITERS); 
	 */
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " exit handleInformIncomingInvalStreamInitiated() " 
		       + " -- already connected -- "
		       + " sender: " + senderNodeId.toString()
		       + " ss: " + ss.toString()
		       + " request: " + r.toString());
	}

	return;
    }


    //
    // Case 3:
    //
    // Connection is an expected incoming connection from my parent
    // in spanning tree.
    //
    assert(r != null);
    Env.remoteAssert(r.getStatus() == SubscribeInvalRequest.STATUS_PENDING);
    r.setStatus(SubscribeInvalRequest.STATUS_CONNECTED);
    if(dbgSpanningTree){
	Env.inform(dstTag 
		   + " exit handleInformIncomingInvalStreamInitiated() " 
		   + " -- successful connection -- "
		   + " sender: " + senderNodeId.toString()
		   + " ss: " + ss.toString()
		   + " request: " + r.toString());
    }


}


/*
 *------------------------------------------------------------------
 *
 * informOutgoingSubscribeInvalInitiated --
 *
 *          Called when we start to make a connection to 
 *          a remote node. (Connection may not succeed.)
 *
 *          We track outgoing connections to parent in spanning
 *          tree.
 *
 * Arguments:
 *      NodeId targetNodeId -- the node we are connecting to
 *      InterestSet is -- the interest set subscribed to
 *      VV startVV -- the startVV used for the connection
 *      
 *      Note: old API needed another argument
 *      boolean placeholderWriterSet -- in the future, we will
 *         support subscriptions that specify a "writerSet"
 *         (like an interest set) that specifies that we want
 *         to see precise invals from writers in the set
 *         and imprecise outside). This placeholder is
 *         to remind us that this code will depend on
 *         that information once that feature is supported.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informOutgoingSubscribeInvalInitiated(NodeId targetNodeId, 
				      SubscriptionSet ss)
{

    if(dbgSpanningTree){
	SubscribeInvalRequest r = invalSpanningList.find(targetNodeId, ss);
	Env.inform(dstTag 
		   + " enter informOutgoingSubscribeInvalInitiated() " 
		   + " -- successful connection -- "
		   + " target: " 
		   + (targetNodeId==null?"null":targetNodeId.toString())
		   + " ss: " 
		   + (ss==null?"null":ss.toString())
		   + " request: " 
		   + (r==null?"null":r.toString()));
    }

    //
    // See if this is a reverse connection in spanning tree
    //
    //assert(placeholderWriterSet); /* if writer set != all nodes{ return} */
    if(true){
	SubscribeInvalRequest r = invalSpanningList.find(targetNodeId, ss);
	if(r == null){
	    // not a connection to parent in spanning tree
	    return;
	}
	
	r.setReverseConnectionStatus(SubscribeInvalRequest.STATUS_CONNECTED);
    }
}


/*
 *------------------------------------------------------------------
 *
 * informInvalStreamTerminated --
 *
 *          An inval stream was terminated. If this was a "downward"
 *          link on interest set spanning tree to us, restart connection.
 *          (If it was an "upward" link on IS spanning tree, don't
 *          bother, sender will restart).
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that had been connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informInvalStreamTerminated(NodeId senderNodeId, SubscriptionSet ss, 
			    VV startVV, boolean placeholderWriterSet,
			    StreamId sId)
{
    assert(placeholderWriterSet);
    //
    // Retry connection
    //
    SubscribeInvalRequest r = invalSpanningList.find(senderNodeId, ss);

    if(dbgSpanningTree){
	Env.inform(dstTag 
		   + " enter informInvalStreamTerminated() " 
		   + " sender: " + senderNodeId.toString()
		   + " ss: " + ss.toString()
		   + " vv: " + startVV.toString()
		   + " request: " 
		   + (r==null?"NULL":r.toString()));
    }


    if(r != null){ // This is upward link on spanning tree
	VV newStartVV = core.getLpVV(ss);
	Env.performanceWarning("SDIMSController should ask core fore updated "
			       + "startVV when reconnecting to avoid resendig "
			       + "the same stuff!");
	r.reset(newStartVV);
	if(dbgSpanningTree){
	    Env.inform(dstTag 
		       + " in informInvalStreamTerminated() " 
		       + " --- Retry connection --- "
		       + " sender: " + senderNodeId.toString()
		       + " ss: " + ss.toString()
		       + " vv: " + startVV.toString()
		       + " request: " + r.toString());
	}
	spanningTreeWork.insert(r); // hand request off to a thread
	return;
    }
    else{
	//
	// This is not a connection we care to maintain (anymore)
	//
    }
}



/*
 *------------------------------------------------------------------
 *
 * informOutgoingInvalStreamTerminated --
 *
 *          An outgoing inval stream was terminated. If it is
 *          a "upward" link on per-IS spanning tree, restart it.
 *          Otherwise, don't worry -- receiver will restart it.
 *
 * Arguments:
 *      NodeId senderNodeId -- the node that had been connected to us
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void 
informOutgoingInvalStreamTerminated(NodeId receiverNodeId, SubscriptionSet ss,
				    VV startVV, boolean placeholderWriterSet)
{
    if(dbgSpanningTree){
	SubscribeInvalRequest r = invalSpanningList.find(receiverNodeId, ss);
	Env.inform(dstTag 
		   + " enter informOutgoingInvalStreamTerminated() " 
		   + " receiver: " + receiverNodeId.toString()
		   + " ss: " + ss.toString()
		   + " vv: " + startVV.toString()
		   + " request: " 
		   + (r==null?"NULL":r.toString()));
    }


    assert(placeholderWriterSet);
    if(placeholderWriterSet){
	SubscribeInvalRequest r = invalSpanningList.find(receiverNodeId, ss);
	
	//
	// This is the reverse connection to my parent in the spanning tree
	//
	if(r != null){
	    r.setReverseConnectionStatus(SubscribeInvalRequest.STATUS_FAILED);

	    if(dbgSpanningTree){
		Env.inform(dstTag 
			   + " in informOutgoingInvalStreamTerminated() " 
			   + " --- Retry connection ---"
			   + " receiver: " + receiverNodeId.toString()
			   + " ss: " + ss.toString()
			   + " vv: " + startVV.toString()
			   + " request: " + r.toString());
	    }

	    spanningTreeWork.insert(r);
	}
    }
}




/*
 *------------------------------------------------------------------
 *
 *  dbgGetCopyISIAmInterestedIn --
 *
 *          Return a copy of the list of interest sets I am interested
 *          in (for the spanning tree). Used for debugging/sanity
 *          checking by SpanningTreeWatchdog.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected synchronized List
dbgGetCopyISIAmInterestedIn()
{
    return (List)interestSetsICareAbout.clone();
}


/*
 *------------------------------------------------------------------
 *
 *  dbgGetSpanningTreeStatus --
 *
 *          For the spacified interest set, check the status
 *          of our connection to the spanning tree.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      A long that acts as a bitvector with the following bits set/cleared:
 *        SpanningTreeWatchdog.CONTROLLER_CARES_THIS_IS
 *        SpanningTreeWatchdog.CONTROLLER_THINKS_UP
 *        SpanningTreeWatchdog.CONTROLLER_THINKS_REVERSE_UP
 *        SpanningTreeWatchdog.CORE_THINKS_UP
 *        SpanningTreeWatchdog.CORE_THINKS_REVERSE_UP
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected synchronized long
dbgGetSpanningStatus(SubscriptionSet ss)
{
    long status = 0;
    if(!(interestSetsICareAbout.contains(ss))){
	assert((status & SpanningTreeWatchdog.CONTROLLER_CARES_THIS_IS) == 0);
	return status;
    }
    status = status = status | SpanningTreeWatchdog.CONTROLLER_CARES_THIS_IS;
    assert((status & SpanningTreeWatchdog.CONTROLLER_CARES_THIS_IS) != 0);

    SubscribeInvalRequest check;
    check = invalSpanningList.find(ss);
    if(check == null){
	assert((status & SpanningTreeWatchdog.CONTROLLER_THINKS_UP) == 0);
	assert((status & SpanningTreeWatchdog.CONTROLLER_THINKS_REVERSE_UP) 
	       == 0);
    }
    else{
	if(check.getStatus() == SubscribeInvalRequest.STATUS_CONNECTED){
	    status = status | SpanningTreeWatchdog.CONTROLLER_THINKS_UP;
	    assert((status & SpanningTreeWatchdog.CONTROLLER_THINKS_UP) != 0);
	}
	else{
	    assert((status & SpanningTreeWatchdog.CONTROLLER_THINKS_UP) == 0);
	}
	if(check.getReverseConnectionStatus() == SubscribeInvalRequest.STATUS_CONNECTED){
	    status = status | SpanningTreeWatchdog.CONTROLLER_THINKS_REVERSE_UP;
	    assert((status & SpanningTreeWatchdog.CONTROLLER_THINKS_REVERSE_UP) != 0);
	}
	else{
	    assert((status & SpanningTreeWatchdog.CONTROLLER_THINKS_REVERSE_UP) == 0);
	}
	NodeId parent = check.getTarget();
	if(!parent.isValid()){
	    assert(check.getStatus() != SubscribeInvalRequest.STATUS_CONNECTED);
	    assert(check.getReverseConnectionStatus() != SubscribeInvalRequest.STATUS_CONNECTED);
	}
	else{
	    Env.warn("TBD: Sanity check code in SDIMSController:dbgGetSpanningStatus"); // assert(TBD);
	    /* if(core.getConnectedTo(is, parent)){
	         status = status | SpanningTreeWatchdog.CORE_THINKS_UP;
	         assert((status & SpanningTreeWatchdog.CORE_THINKS_UP) != 0);
               else{
	         assert((status & SpanningTreeWatchdog.CORE_THINKS_UP) == 0);
               }
	       if(core.getConnectedFrom(is, parent)){
	         status = status | SpanningTreeWatchdog.CORE_THINKS_REVERSE_UP;
	         assert((status & SpanningTreeWatchdog.CORE_THINKS_REVERSE_UP) != 0);
               else{
	         assert((status & SpanningTreeWatchdog.CORE_THINKS_REVERSE_UP) == 0);
               }
	    */
	}
    }
    //
    // Only bits set are those we have defined
    //
    assert((status & ~(SpanningTreeWatchdog.CONTROLLER_CARES_THIS_IS 
			| SpanningTreeWatchdog.CONTROLLER_THINKS_UP
			| SpanningTreeWatchdog.CONTROLLER_THINKS_REVERSE_UP
			| SpanningTreeWatchdog.CORE_THINKS_UP
			| SpanningTreeWatchdog.CORE_THINKS_REVERSE_UP)) == 0);
    return status;
    

}









public static void runEnterprise(String s[]) {


  	assert(s.length == 3);
  
	System.out.println("SDIMSController WAN enterprise experiment..");

    System.out.println("My worker id = " + s[1]);

	// process comman line inputs

	// first argument tells us whethere to use SDIMS or trvial
	// controller or oracle controller.
	int type = 0;
	if(s[2].equals("SDIMS")) {
		System.out.println("Using trivial controller..");
		type = Controller.SDIMS_CONTROLLER;
	}
	else if(s[2].equals("ORACLE")) {
		System.out.println("Using oracle controller..");
		type = Controller.ORACLE_CONTROLLER;
	}
	else if(s[2].equals("CODA")) {
		System.out.println("Using coda controller..");
		type = Controller.CODA_CONTROLLER;
	}
	else if(s[2].equals("BAYOU")) {
		System.out.println("Using bayou controller..");
		type = Controller.BAYOU_CONTROLLER;
	}
	else {
		System.out.println("invalid controller, must be one of SDIMS, ORACLE, CODA, BAYOU");
		assert(false);
	}

	Core core;
	NodeId n = new NodeId((new Integer(s[1])).intValue());
	int me = (new Integer(n.toString())).intValue();
	URANode node = new URANode(s[0],
				   n,
				   type);
	core = node.getCore();
	assert(node != null);
	LocalInterface local = node.getLocalInterface();
	assert(local != null);
	
	int NUM_IS = 2;
	int NUM_WRITES = 2000;
	int WRITES_PER_IS = 1000;
	int WRITE_SIZE = 1000;


	byte[] buf = new byte[WRITE_SIZE];
	for(int i=0; i<WRITE_SIZE; i++) {
		buf[i] = 'a';
	}


	SubscriptionSet[] iss = new SubscriptionSet[NUM_IS];
	// first make a certain number of interest sets
	for(int i = 0; i<NUM_IS; i++) {
		iss[i] = SubscriptionSet.makeSubscriptionSet("/" + i);
	}

	// Loop below is a NO-OP
	// Node 1 does NUM_WRITES local writes equally distributed across all
	// directories. Currently, node 1 is the only node in the system.
	/*
	if(core.getMyNodeId().toString().equals("1")) {
		for(int i=0; i<NUM_WRITES; i++) {
		  	// determine which file to write to
			int dirnum = NUM_WRITES / WRITES_PER_IS;
			int filenum = NUM_WRITES % WRITES_PER_IS;
			ObjId obj = new ObjId((new Integer(dirnum)).toString() +
				"/" + (new Integer(filenum)).toString());
			//local.write(obj, 0, WRITE_SIZE, buf, false);

		}
	}
	*/

	try {
	// Node 1 does writes in /1/* and /4/*
	if(me == 1) {
	  	for(int i=0; i<WRITES_PER_IS; i++) {
			ObjId obj = new ObjId(("/1/" + (new Integer(i)).toString()));
			local.write(obj, 0, WRITE_SIZE, buf, false);

		}
		for(int i=0; i<WRITES_PER_IS; i++) {
			ObjId obj = new ObjId(("/4/" + (new Integer(i)).toString()));
			local.write(obj, 0, WRITE_SIZE, buf, false);

		}
	}

	} catch(Exception e) {
		System.out.println("Exception while doing local write " + e);
		e.printStackTrace();
		assert(false);
	}


	// all nodes sync up to see all invals
	NodeId node1 = new NodeId(1);
	NodeId node4 = new NodeId(4);
	try {
	local.sync(new AcceptStamp(WRITES_PER_IS-1, node1));
	local.sync(new AcceptStamp(WRITES_PER_IS-1, node4));
	} catch (Exception e) {
		System.out.println("Exception occurred while syncing " + e);
		assert(false);
	}
	// have seen all invalildates at this point


	// node 4 reads half of node 1's writes, ie one interest set's
	// worth of data. Then 4 writes a single write for 5 to know it's
	// done.
	try {
	if(me == 4) {
		for(int i = 0; i<WRITES_PER_IS/NUM_IS; i++) {
		  	ObjId obj = new ObjId("/1/" + (new Integer(i)).toString());
			local.informDemandReadMiss(obj, 20L, 40L);
			local.readBlockInvalid(obj, 0, WRITE_SIZE, true);
		}

		ObjId obj4 = new ObjId("/4/0");
		long got = local.write(obj4, 0, WRITE_SIZE, buf, false);
	}
	} catch(Exception e) {
		System.out.println("Exception occured during remote read " + e);
	}
	// at this point node 1 has read all of node 4's writes


	// node 5 does reads of writes that node 4 read from 1. A clever
	// controller would detect this on the fly and direct 5 to 4 as
	// opposed to 1, the original writer and central server.
	try {

	// first syncs with 4 having read its reads from 1
	local.sync(new AcceptStamp(NUM_WRITES, node4));
	  
	if(me == 5) {
	  	// read writes done by 4
		for(int i = 0; i<NUM_WRITES/NUM_IS; i++ ) {
		  	ObjId obj = new ObjId("/4/" + (new Integer(i)).toString());
			local.informDemandReadMiss(obj, 0L, WRITE_SIZE);
			local.readBlockInvalid(obj, 0, WRITE_SIZE, true);
		}
	}
	} catch (ObjNotFoundException e) {
		System.out.println("File not found exception " + e);
		assert(false);
	} catch(IOException e) {
		System.out.println("IO exception " + e);
		assert(false);
	} catch(InterruptedException e) {
		System.out.println("Interrupted exception " + e);
		assert(false);
	} catch(ReadOfHoleException rhe){
	  System.out.println(rhe.toString());
	  assert(false);
	}
	 
}





public static void main(String s[])
{

  	runEnterprise(s);
	System.exit(1);

	////////////////////////////////////////////////

  
    System.out.print("SDIMSController self test...");
    //
    // We assume a particular formatting of InterestSet.toString()
    // in handleRootSpanningTree(). We also assume particular
    // behavior about getParentIS().
    //
    Env.warn("WARNING: turned off some sanity checks without thinking it through");

    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/foo/bar");
    assert(ss.containsExactlyOneSubdirectory());
    //    assert(is.toString().equals("(/foo/bar)"));
    SubscriptionSet parent = ss.getParentSS();
    //    assert(parent.toString().equals("(/foo)"));
    ss = SubscriptionSet.makeSubscriptionSet("/");
    assert(ss.containsExactlyOneSubdirectory());
    //    assert(is.toString().equals("(/)"));
    parent = ss.getParentSS();
    assert(parent == null);


    System.err.println("WARNING: redirecting stderr to stdout");
    System.out.println("WARNING: redirecting stderr to stdout");
    System.setErr(System.out); // Redirect stderr to stdout



    int arg;

    if(s.length == 1){
      System.out.println("Worker ID = " + s[0]);
      arg = new Long(s[0]).intValue();
    }
    else{
      arg = 0;
    }

    if(arg == 0) {
      	System.out.println("I am master");
      	// 
      	// Test spanning tree functionality 
      	// 
      	try{
		Runtime r = Runtime.getRuntime(); 
		String args[] = new String[2]; 
		args[0] = "runSDIMSControllerTest-worker"; 
		// args[1] = new Long(1).toString(); 
		// Process spanningTester1 = r.exec(args); 
		args[1] = new Long(2).toString(); 
		Process spanningTester2 = null;
		if(true){
		    spanningTester2 = r.exec(args); 
		}
		args[1] = new Long(3).toString(); 
		Process spanningTester3 = r.exec(args); 
		
		InputStream z;
		FileOutputStream fos;
		TestPump p;
		//	InputStream z = spanningTester1.getInputStream();
		//	FileOutputStream fos = new FileOutputStream("test/tmp1");
		//	TestPump p = new TestPump(z, fos);
		//	p.start();
	
		if(true){
		    z = spanningTester2.getInputStream();
		    fos = new FileOutputStream("test/tmp2");
		    p = new TestPump(z, fos);
		    p.start();
		}
	
		z = spanningTester3.getInputStream();
		fos = new FileOutputStream("test/tmp3");
		p = new TestPump(z, fos);
		p.start();
		
		try{ 
		    //	  spanningTester1.waitFor();
		    //testSpannerWorker(1);
		    testDemandReadMiss(1);
		    if(true){
			spanningTester2.waitFor(); 
		    }
		    spanningTester3.waitFor();
		} catch(InterruptedException e){ 
		  	assert(false); 
		  }
	    } catch(IOException f){
			f.printStackTrace();
			System.exit(-1);
	      }
    }
    else{
      //testSpannerWorker(arg);
	  testDemandReadMiss(arg);
    }
    System.out.println("SDIMSController self test: *TBD* unbind, "
		       + " *TBD*read miss --> FAILED/TBD <---");
    System.exit(-1);
    //    System.out.println("Self test done (" + arg + ")");
    System.exit(0);
}

private static void testDemandReadMiss(int id) {
	Core core;
    System.out.println("My worker id = " + id);

	final int SIZE = 100;
	final int N = 100;
	NodeId n = new NodeId(id);
	URANode node = new URANode("test/SDIMS-spanning-test.config",
				   n,
				   Controller.TRIVIAL_CONTROLLER);
	core = node.getCore();
	assert(node != null);
	LocalInterface local = node.getLocalInterface();
	assert(local != null);
	byte b[] = new byte[SIZE];
	int ii;
	for(ii = 0; ii < SIZE; ii++){
	    b[ii] = 42;
	}

	ObjId[] obj = new ObjId[100];
	for(int i=0; i<N; i++) {
		obj[i] = new ObjId("/" + id + "/" + i);
	}
	
	ObjId o1 = new ObjId("/" + id + "/1");
	ObjId o2 = new ObjId("/" + id + "/2");
	ObjId o3 = new ObjId("/" + id + "/3");

	Env.warn("Test is fragile -- write should really return PreciseInv");

	try{
	    local.write(o1, 0, SIZE, b, false);
	    local.write(o2, 0, SIZE, b, false);
	    local.write(o3, 0, SIZE, b, false);
	    //
	    // Now try sync-ing with each object
	    // for all three nodes.
	    //
	    Env.warn("SDIMSController self test assumes accept"
		     + " stamps start at zero and increase by 1");
	    BlockTest bt = new BlockTest(local, new AcceptStamp(0, new NodeId(0)));
	    bt.start();
	    for(ii = 1; ii <= 3; ii++){
		long id2 = ii;
		NodeId target = new NodeId(id2);
		Env.sprintln("SDIMSController self test: " + id 
				   + " starting to wait for " + id2
				   + "'s invals.");

		local.sync(new AcceptStamp(0, target));
		local.sync(new AcceptStamp(1, target));
		local.sync(new AcceptStamp(2, target));

		Env.sprintln("SDIMSController self test: " + id 
				   + " received " + id2
				   + "'s invals.");

		Env.warn("Earlier, when we waited for a non-existant node "
			 + "(node 0), "
                         + "UpdateLog::waitUntilTentativeAppliedInternal()"
			 + " created a new entry in VV and returned immediately"
			 + "(Correct behavior should be block forever?");
	    }
	}
	catch(IOException e){
	    assert(false);
	}
	catch(InterruptedException f){
	    assert(false);
	}

	Env.warn("Not done yet -- need to add sync to local interface and sync on other threads' writes");

	System.out.println("DONE! All invals arrived at node " + id + "\n");
	try{
	    // Wait long enough for my info to propagate to other nodes
	    Thread.sleep(10000);
	}
	catch(InterruptedException z){
	    // No problem
	}

	try {

	// Do demand reads for all reads here 
	// - local reads first
	System.out.println("Starting demand reads, local hits first..");
	BodyMsg b1 = local.readBlockInvalid(o1, 20L, 40L, true);
	System.out.println(b1);
	b1 = local.readBlockInvalid(o2, 20L, 40L, true);
	System.out.println(b1);
	b1 = local.readBlockInvalid(o3, 20L, 40L, true);
	System.out.println(b1);

	// - remote reads, the real test
	System.out.println("Starting remote reads..");
	for(ii=1; ii<=3; ii++) {
	  	if(ii != id) { // not me
		  	for(int j=1; j<=3; j++) {
			  	ObjId o = new ObjId("/" + ii + "/" + j);
				System.out.println("Calling informDemandReadMiss to look for " + o.getPath());
				long starttime = System.currentTimeMillis();
				local.informDemandReadMiss(o, 20L, 40L);
				while(true) {
					b1 = local.readBlockInvalid(o, 20L, 40L, true);
					long endtime = System.currentTimeMillis();
					System.out.println("Read latency = " + (endtime - starttime));
					if(b1 != null) break;
				}
				System.out.println(b1);
			}
		}
	}

	System.out.println("");

	} catch (ObjNotFoundException e) {
	  	assert(false);
	} catch (IOException e) {
		assert(false);
	} catch(ReadOfHoleException e){
	  assert false;
	}

	try{
	    // Wait long enough for my info to propagate to other nodes
	    Thread.sleep(40000);
	}
	catch(InterruptedException z){
	    // No problem
	}

}

private static void testSpannerWorker(int id)
{
    Core core;
    System.out.println("My worker id = " + id);

	final int SIZE = 100;
	NodeId n = new NodeId(id);
	URANode node = new URANode("test/SDIMS-spanning-test.config",
				   n,
				   Controller.TRIVIAL_CONTROLLER);
	core = node.getCore();
	assert(node != null);
	LocalInterface local = node.getLocalInterface();
	assert(local != null);
	byte b[] = new byte[SIZE];
	int ii;
	for(ii = 0; ii < SIZE; ii++){
	    b[ii] = 42;
	}
	ObjId o1 = new ObjId("/" + id + "/1");
	ObjId o2 = new ObjId("/" + id + "/2");
	ObjId o3 = new ObjId("/" + id + "/3");

	Env.warn("Test is fragile -- write should really return PreciseInv");

	try{
	    local.write(o1, 0, SIZE, b, false);
	    local.write(o2, 0, SIZE, b, false);
	    local.write(o3, 0, SIZE, b, false);
	    //
	    // Now try sync-ing with each object
	    // for all three nodes.
	    //
	    Env.warn("SDIMSController self test assumes accept"
		     + " stamps start at zero and increase by 1");
	    BlockTest bt = new BlockTest(local, new AcceptStamp(0, new NodeId(0)));
	    bt.start();
	    for(ii = 1; ii <= 3; ii++){
			long id2 = ii;
			NodeId target = new NodeId(id2);
			System.out.println("SDIMSController self test: " + id 
					   + " starting to wait for " + id2
					   + "'s invals.");
			local.sync(new AcceptStamp(0, target));
			local.sync(new AcceptStamp(1, target));
			local.sync(new AcceptStamp(2, target));
			System.out.println("SDIMSController self test: " + id 
					   + " received " + id2
					   + "'s invals.");
			Env.warn("Earlier, when we waited for a non-existant node "
				 + "(node 0), "
	                         + "UpdateLog::waitUntilTentativeAppliedInternal()"
				 + " created a new entry in VV and returned immediately"
				 + "(Correct behavior should be block forever?");
	    }
	}
	catch(IOException e){
	    assert(false);
	}
	catch(InterruptedException f){
	    assert(false);
	}

	Env.warn("Not done yet -- need to add sync to local interface and sync on other threads' writes");

	System.out.println("DONE! All invals arrived at node " + id + "\n");
	try{
	    // Wait long enough for my info to propagate to other nodes
	    Thread.sleep(10000);
	}
	catch(InterruptedException z){
	    // No problem
	}
}

public void informCheckpointApplied(NodeId senderId, boolean status) {
	// TODO Auto-generated method stub
	
}

}

class BlockTest extends Thread
{
    LocalInterface local;
    AcceptStamp shouldBlock;

public BlockTest(LocalInterface local, AcceptStamp shouldBlock)
    {
	this.local = local;
	this.shouldBlock = shouldBlock;
    }
    
public void run()
    {
	System.out.println("Try sync of " + shouldBlock + " (should block)");
	try{
	    local.sync(shouldBlock);
	}
	catch(InterruptedException e){
	}
	System.out.println("**** TEST FAILS *** blocking sync returned");
	assert(false);
    }
}

class TestPump extends Thread
{
    InputStream s;
    OutputStream o;
public TestPump(InputStream s, OutputStream o)
{
    this.s = s;
    this.o = o;
}

public void run()
  {
    int got;
    byte b[] = new byte[80];
    while(true){
      try{
	got = s.read(b);
	if(got > 0){
	    o.write(b, 0, got);
	}
	if(got < 0){
	    System.err.println("Pump sees eof while writing to " + o.toString());
	    return;
	}
      } catch(IOException e){
	e.printStackTrace();
	assert(false);
      }
    }
    
    
  }
}

//---------------------------------------------------------------------------
/* $Log: SDIMSController.java,v $
/* Revision 1.71  2007/04/05 21:49:25  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.70  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.69  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.68  2006/12/08 21:36:26  nalini
/* reverting to the old read interface
/*
/* Revision 1.66  2006/10/31 21:43:33  nalini
/* added control msgs to body streams
/*
/* Revision 1.65  2006/08/15 21:46:24  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.64  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.63.2.1  2006/06/02 22:18:13  nalini
/* Supports addition and removeal of SS from Outgoing Body Streams
/*
/* Revision 1.63  2006/04/23 22:44:59  zjiandan
/* change informReceiveInval to have senderId info.
/*
/* Revision 1.62  2006/04/21 05:09:13  zjiandan
/* add inform methods.
/*
/* Revision 1.61  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.60  2005/10/13 00:24:24  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.58  2005/03/03 08:07:55  nayate
/* Added some checkpoint transfer code
/*
/* Revision 1.57  2005/03/01 10:40:35  nayate
/* First successful compilation
/*
/* Revision 1.56  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.55  2004/10/22 20:46:55  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.54  2004/07/26 20:03:39  dahlin
/* Fixed typos from windows checkin so it will compile under Linux
/*
/* Revision 1.53  2004/05/27 04:39:37  arun
/* *** empty log message ***
/*
/* Revision 1.52  2004/05/27 04:34:56  arun
/* *** empty log message ***
/*
/* Revision 1.51  2004/05/27 02:42:13  arun
/* *** empty log message ***
/*
/* Revision 1.50  2004/05/27 01:59:23  arun
/* *** empty log message ***
/*
/* Revision 1.49  2004/05/27 01:41:43  arun
/* enterprise experiment
/*
/* Revision 1.48  2004/05/26 21:55:25  arun
/* *** empty log message ***
/*
/* Revision 1.47  2004/05/26 09:45:56  arun
/* *** empty log message ***
/*
/* Revision 1.46  2004/05/26 04:37:39  arun
/* *** empty log message ***
/*
/* Revision 1.45  2004/05/24 21:16:23  arun
/* minor corrections to make unbind messages with sdims work
/*
/* Revision 1.44  2004/05/24 05:33:33  arun
/* *** empty log message ***
/*
/* Revision 1.43  2004/05/24 01:34:51  arun
/* added check in SDIMSController to check if inval is newer than existing body on store to prevent SDIMS from polluting its directory when it gets an inval for its own local write
/*
/* Revision 1.42  2004/05/23 20:15:30  arun
/* *** empty log message ***
/*
/* Revision 1.41  2004/05/23 10:32:25  arun
/* added support for startVV based update priority queue
/*
/* Revision 1.40  2004/05/23 03:27:08  arun
/* *** empty log message ***
/*
/* Revision 1.39  2004/05/23 00:25:22  arun
/* only fresh writes get inserted in UPQ. Some debugging code related to SDIMS
/*
/* Revision 1.38  2004/05/22 10:10:12  arun
/* minor changes to notifyUpq logic. Correction of store behaviour while handling bound invals.
/*
/* Revision 1.37  2004/05/22 06:46:19  arun
/* minor corrections
/*
/* Revision 1.36  2004/05/21 23:22:25  arun
/* *** empty log message ***
/*
/* Revision 1.35  2004/05/21 03:09:22  ypraveen
/* Arguments and their types added for notifySpanningChange routine
/*
/* Revision 1.34  2004/05/20 04:48:41  dahlin
/* Finished SDIMS controller self test for spanning tree; also basic spanning watchdog works
/*
/* Revision 1.33  2004/05/20 04:10:57  dahlin
/* Passes SDIMSController spanning tree self test
/*
/* Revision 1.32  2004/05/20 01:44:07  dahlin
/* Looks like spanning trees are getting set up with TrivialSpanningTreeDirectory (though more testing remains)
/*
/* Revision 1.31  2004/05/20 00:50:39  dahlin
/* Turn off dbg output for stuff I just checked in
/*
/* Revision 1.30  2004/05/20 00:47:34  dahlin
/* Fixed several bugs to make inval log exchange work (biggest one: ISIterator handling case when an single-writer log exists but has no records in it; also added some debugging tools
/*
/* Revision 1.29  2004/05/19 04:34:55  dahlin
/* SDIMSController test compiles and runs but deadlocks
/*
/* Revision 1.28  2004/05/19 02:25:44  dahlin
/* Testing fix
/*
/* Revision 1.27  2004/05/19 02:14:44  dahlin
/* Making progress on SDIMSController spanning test
/*
/* Revision 1.26  2004/05/14 21:20:00  dahlin
/* Retry logic for upward path on body push spanning tree
/*
/* Revision 1.25  2004/05/14 20:49:00  dahlin
/* SDIMSController logic for Body connections (incoming)
/*
/* Revision 1.24  2004/05/14 19:23:21  dahlin
/* Cleanup spanning tree logic; add update bodies
/*
/* Revision 1.23  2004/05/13 21:58:14  dahlin
/* simple unbind implemented
/*
/* Revision 1.22  2004/05/13 20:09:10  dahlin
/* When core has applied a bound inval, it tells originator of the write
/*
/* Revision 1.21  2004/05/13 18:49:17  dahlin
/* Clean SDIMSController by moving SpanningTree directory out as separate class (rather than using raw SDIMSInterface)
/*
/* Revision 1.20  2004/05/13 00:17:43  dahlin
/* Small refinements to SDIMSController
/*
/* Revision 1.19  2004/05/12 16:58:01  dahlin
/* Finished design review with lei, amol, jiandan, arun, and mike
/*
/* Revision 1.18  2004/05/11 00:49:20  dahlin
/* Updated TBD for SDIMSController
/*
/* Revision 1.17  2004/05/11 00:46:56  dahlin
/* Added read-miss-directory updates to SDIMSController
/*
/* Revision 1.16  2004/05/10 23:55:42  dahlin
/* Moved SDIMS read requests into ReadDirectory class for modularity and extensibility
/*
/* Revision 1.15  2004/05/10 22:45:47  dahlin
/* All unit: target tests succeed
/*
/* Revision 1.14  2004/05/10 21:14:15  dahlin
/* Getting close to something that compiles w/o errors...
/*
/* Revision 1.13  2004/05/10 20:48:19  dahlin
/* Clarified RMI exceptions; full version of (stub) DemandReadWorker
/*
/* Revision 1.12  2004/05/10 18:56:32  dahlin
/* Created PendingDemandList and PendingDemandRequest (needed by SDIMSController)
/*
/* Revision 1.11  2004/05/09 20:09:14  dahlin
/* Updated unit tests for WorkQueue, TimeoutQueue, TimeoutQueueWorker, DirectoryInterestSet, SubscribeInvalRequest, InvalSpanningList, AcceptStamp, NodeId, and AcceptVV
/*
/* Revision 1.10  2004/05/09 02:24:16  dahlin
/* Complete coding and unit testing of InvalSpanningList and SubscribeInvalRequest
/*
/* Revision 1.9  2004/05/08 22:18:57  dahlin
/* Partially complete version of SDIMSController (it should compile w/o error, though)
/*
v/* Revision 1.8  2004/05/07 00:40:27  dahlin
/* Began pseudo-code; have state machine for subscribeInvals
/*
/* Revision 1.7  2004/05/06 16:02:36  dahlin
/* Created stub versions of all methods including comment outlining what needs to be done
/*
/* Revision 1.6  2004/05/06 14:25:33  dahlin
/* Fix so it will compile
/*
/* Revision 1.5  2004/05/06 13:58:51  dahlin
/* Fixed types so it will compile
/*
/* Revision 1.4  2004/05/05 23:42:13  dahlin
/* Conversation w/ praveen; update plan for incrementally building it
/*
/* Revision 1.3  2004/05/02 17:23:54  dahlin
/* Complete first draft of SDIMSController.java design -- includes high-level TBD but no pseudo-code
/*
/* Revision 1.2  2004/04/30 17:45:46  dahlin
/* First-cut answers to the major design issues that I know of for SDIMSController
/*
/* Revision 1.1  2004/04/30 00:09:27  dahlin
/* Starting to work through details in prose discussion of design options
/*
/* Revision 1.2  2004/04/15 20:04:24  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
