package code;
 /** 
 *  Represent single-writer data stored from an imprecise invalidate 
 **/ 
public class SingleWriterImpreciseInv implements SingleWriterInval, Immutable
{
  private final InvalTarget invalTarget;
  private final AcceptStamp startTime;
  private final AcceptStamp endTime;
  private final AcceptStamp rtAcceptStamp;

 /** 
 *  Constructor -- temporary for compatable with old code 
 *        note: realTime is set to endTime if not specified 
 **/ 
  public SingleWriterImpreciseInv(InvalTarget invalTarget_,
                                  AcceptStamp startTime_,
                                  AcceptStamp endTime_){
    assert(invalTarget_ instanceof Immutable);
    this.invalTarget = invalTarget_; // InvalTargets are immutable
    assert(startTime_ instanceof Immutable);
    this.startTime = startTime_;     // AcceptStamp is immutable
    this.endTime = endTime_;         // AcceptStamp is immutable
    assert(this.startTime.getNodeId().equals(this.endTime.getNodeId()));
    rtAcceptStamp = endTime_;
  }

 /** 
 *  Constructor 
 **/ 
  public SingleWriterImpreciseInv(InvalTarget invalTarget_,
                                  AcceptStamp startTime_,
                                  AcceptStamp endTime_,
                                  AcceptStamp rtAcceptStamp_){
    assert(invalTarget_ instanceof Immutable);
    this.invalTarget = invalTarget_; // InvalTargets are immutable
    assert(startTime_ instanceof Immutable);
    this.startTime = startTime_;     // AcceptStamp is immutable
    this.endTime = endTime_;         // AcceptStamp is immutable
    assert(this.startTime.getNodeId().equals(this.endTime.getNodeId()));
    assert(rtAcceptStamp_ != null);
    assert(rtAcceptStamp_ instanceof Immutable);
    assert(this.startTime.getNodeId().equals(rtAcceptStamp_.getNodeId()));
    rtAcceptStamp = rtAcceptStamp_;
  }
  
 /** 
 *  Return the start accept stamp 
 **/ 
  public final AcceptStamp
  getStartAcceptStamp(){
    return(this.startTime);
  }

 /** 
 *  Return the end accept stamp 
 **/ 
  public final AcceptStamp
  getEndAcceptStamp(){
    return(this.endTime);
  }

 /** 
 *  METHODS INHERITED FROM GENERALINV: 
 **/ 

 /** 
 *  Return the InvalTarget for this invalidate 
 **/ 
  public InvalTarget
  getInvalTarget(){
    return(this.invalTarget);
  }

 /** 
 *  Generate and return an acceptVV wrapper around startTime 
 **/ 
  public AcceptVV
  getStartVV(){
    AcceptVV vv = null;
    AcceptStamp[] uniStamp = null;

    uniStamp = new AcceptStamp[1];
    uniStamp[0] = this.startTime;
    vv = new AcceptVV(uniStamp);
    return(vv);
  }

 /** 
 *  Generate and return an acceptVV wrapper around acceptStamp 
 **/ 
  public AcceptVV
  getEndVV(){
    AcceptVV vv = null;
    AcceptStamp[] uniStamp = null;

    uniStamp = new AcceptStamp[1];
    uniStamp[0] = this.endTime;
    vv = new AcceptVV(uniStamp);
    return(vv);
  }

 /** 
 *  Return the realTime stampe version vector 
 **/ 
  public final AcceptVV
  getRTVV(){
    AcceptVV vv = null;
    AcceptStamp[] uniStamp = null;

    uniStamp = new AcceptStamp[1];
    uniStamp[0] = this.rtAcceptStamp;
    vv = new AcceptVV(uniStamp);
    return(vv);
  }

 /** 
 *  Clone this object 
 *  
 *  If we *know* that this object and all objects we reference 
 *  are immutable, then just return this; otherwise perform 
 *      newOne = super.clone();  
 *      now replace any fields in newOne that point to mutable  
 *          objects with a clone of that field 
 *      return newOne 
 *  
 **/ 
  public Object
  clone(){
    // All of our members are immutable, so we can make a shallow copy
    return(new SingleWriterImpreciseInv(this.invalTarget,
                                        this.startTime,
                                        this.endTime));
  }

 /** 
 *  Return a new invalidate with mutiple operations performed 
 **/ 
  public GeneralInv
  cloneIntersectInvaltargetChopStartEnd(GeneralInv inv){
    InvalTarget invTarget = null;
    GeneralInv intersectInv = null;
    PreciseInv preciseInv = null;
    BoundInval boundInv = null;
    MultiObjPreciseInv mopi = null;

    long invStartStamp = 0;
    long invEndStamp = 0;
    long invRealStamp = 0;
    NodeId nodeId = null;
    AcceptStamp invStartAcceptStamp = null;
    AcceptStamp invEndAcceptStamp = null;
    AcceptStamp invRealAcceptStamp = null;

    assert(inv.getStartVV().includes(this.startTime));

    invTarget = this.invalTarget.getIntersection(inv.getInvalTarget());
    assert(invTarget != null);
    
    //simple strategies: if inv is precise, check the InvalTarget should be inclusive,
    //then just return the precise inv: preciesInv, BoundInv, MultiObjPreciseInv, CommitInv
    if(inv.isPrecise()){
      assert inv instanceof PreciseInv
      || inv instanceof BoundInval
      || inv instanceof MultiObjPreciseInv
      || inv instanceof CommitInv;
      assert invTarget.equals(inv.getInvalTarget()):
        "this = " + this + "intersect invTarget=" + invTarget + "inv=" + inv;
      assert inv.getStartVV().includes(this.getStartVV()): "inv= " + inv + " this=" + this;
      assert this.getEndVV().includes(inv.getEndVV());
      intersectInv = inv;
    }else{
      assert invTarget instanceof HierInvalTarget: "invTarget=" + invTarget + " inv= " + inv + " this=" + this;
      nodeId = this.startTime.getNodeId();
      try{
        invStartStamp = inv.getStartVV().getStampByServer(nodeId);
        invEndStamp = inv.getEndVV().getStampByServer(nodeId);
        invRealStamp = inv.getRTVV().getStampByServer(nodeId);
      }catch(NoSuchEntryException e){
        System.err.println("inv has an invalid start/end time");
        assert(false);
      }
      assert(invStartStamp >= this.startTime.getLocalClock());
      assert(invEndStamp <= this.endTime.getLocalClock());
      invStartAcceptStamp = new AcceptStamp(invStartStamp, nodeId);
      invEndAcceptStamp = new AcceptStamp(invEndStamp, nodeId);
      invRealAcceptStamp = new AcceptStamp(invRealStamp, nodeId);

      intersectInv = new SingleWriterImpreciseInv((HierInvalTarget)invTarget,
                                                  invStartAcceptStamp,
                                                  invEndAcceptStamp,
                                                  invRealAcceptStamp);
    }
    /*
     * 
    else if(invTarget instanceof ObjInvalTarget){
      assert(inv.isPrecise());
      assert(inv.getStartVV().equals(inv.getEndVV()));
      if(inv instanceof DeleteInv){
        DeleteInv deleteInv = (DeleteInv)inv;
        intersectInv = new DeleteInv(((ObjInvalTarget)invTarget).getObjId(),
                                     deleteInv.getAcceptStamp(),
                                     deleteInv.getRTAcceptStamp(),
                                     false);
      }else if(inv instanceof BoundInval){
        // This is not true -- see comments in BoundInval.method()
	// zjd Feb 2007
	//
        // Here BoundInv becomes unbinded because if we've already received
        // impreciseinv("this") includes this boundinv, it means that
        // somewhere
        // in the system the unbind msg is already created. Therefore it's ok
        // to just unbind it here.
        //
        boundInv = (BoundInval)inv;

        intersectInv = boundInv;//add by zjd Feb 2007
	//see above commented by zjd
        //
        //intersectInv = new PreciseInv((ObjInvalTarget)invTarget,
        //                              boundInv.getAcceptStamp(),
        //                              boundInv.getRTAcceptStamp(),
        //                              false);
	
      }else if(inv instanceof MultiObjPreciseInv){
        // Pass the call onto MultiObjPreciseInv
        mopi = (MultiObjPreciseInv)inv;
        intersectInv = mopi.cloneIntersectInvaltargetChopStartEnd(this);
      }else{
        assert(inv instanceof PreciseInv);
        preciseInv = (PreciseInv)inv;
        intersectInv = new PreciseInv((ObjInvalTarget)invTarget,
                                      preciseInv.getAcceptStamp(),
                                      preciseInv.getRTAcceptStamp(),
                                      false);
      }
    }
    */
    return(intersectInv);
  }

 /** 
 *  Return the portion of an invalidate for one writer whose 
 *  starting stamp is potentially chopped. 
 **/ 
  public GeneralInv getOneWriterSubsetFrom(NodeId node,
                                           long startStampInclusive){
    /*
      return an invalidation (possibly this) with 
      interestSet = this.interestSet
      nodeId = node // assert node was in this.VV
      startVV{node} == startStampInclusive
      // assert this.startStamp{node} >= startSTampInclusive
      endVV{node} == this.endVV{node}
      (Note: OK to just return this if we meet all of the other criteria)
    */
    SingleWriterImpreciseInv subsetInv = null;
    AcceptStamp newStartAcceptStamp = null;

    assert(this.startTime.getNodeId().equals(node));
    assert(startStampInclusive >= this.startTime.getLocalClock());
    assert(startStampInclusive <= this.endTime.getLocalClock());

    newStartAcceptStamp = new AcceptStamp(startStampInclusive, node);
    subsetInv = new SingleWriterImpreciseInv(this.invalTarget,
                                             newStartAcceptStamp,
                                             this.endTime);
    return(subsetInv);
  }

 /** 
 *  Return true if the entire invalidation is already accounted 
 *  for by the specified vv 
 **/ 
  public boolean
  isIncludedBy(VV vv){
    //if we are precise, return vv.includes(this.timestamp)
    //if we are imprecise, return vv.includes(this.*endVV*)
    return(vv.includes(this.endTime));
  }

 /** 
 *  Return true if any component of this invalidation is already 
 *  accounted for by the specified vv 
 **/ 
  public boolean
  anyComponentIncludedBy(VV vv){
    //if we are precise, return vv.includes(this.timestamp)
    //if we are imprecise, true if for any entry e in *startVV*,
    //  vv.includes(e.timestamp)
    return(vv.includes(this.startTime));
  }

 /** 
 *  Return true if this invalidation is committed 
 **/ 
  public boolean
  isCommitted(){
    // Ensure that either the entire message is uncommitted, or it
    // is committed.
    return(false);
  }

 /** 
 *  Return true if this invalidate is precise (which it isn't!) 
 **/ 
  public boolean
  isPrecise(){
    return(false);
  }

 /** 
 *  Return true if this invalidate is embargoed (which it isn't!) 
 **/ 
  public boolean
  isEmbargoed(){
    return(false);
  }

 /** 
 *  Return true if this invalidate is bound (which it isn't!) 
 **/ 
  public boolean
  isBound(){
    return(false);
  }

 /** 
 *  Return true if this invalidate is a delete (which it isn't!) 
 **/ 
  public boolean
  isDelete(){
    return(false);
  }

 /** 
 *  Create a new generalInv that is the union of this and i 
 **/ 
  public GeneralInv
  newUnion(GeneralInv i, SubscriptionSet ss){
    /*
      create a new generalInv that is the union of this and i
        each element of startVV is min of corresponding
                     end        max
      interest set is union of corresponding
      assert(either both are uncommitted or both are committed)
      if both committed 
        assert no gap between csnEnd of one and csnStart of
          next (OK to overlap)
        csnstart = min of both csnStart
        csnEnd = max of both csnMax
    */
    InvalTarget it = null;
    ImpreciseInv impreciseInv = null;
    AcceptVV newStartVV = null;
    AcceptVV newEndVV = null;
    AcceptVV newRTVV = null;
    ObjId objId = null;

    assert (!i.isEmbargoed());
    // Create an InvalTarget for the union
    it = this.invalTarget.getUnion(i.getInvalTarget(), ss);
    assert(it != null);

    if((it instanceof ObjInvalTarget) && (!i.equals(this))){
      objId = ((ObjInvalTarget)it).getObjId();
      it = HierInvalTarget.makeHierInvalTarget(objId.getPath());
    }else if(it instanceof MultiObjInvalTarget){
      // Convert the returned object into a HierInvalTarget because we need
      // to build an Imprecise Invalidate out of the union
      it = ((MultiObjInvalTarget)it).makeHierInvalTarget();
    }
    assert(it instanceof HierInvalTarget);

    newStartVV = this.getStartVV().cloneMinVV(i.getStartVV());
    newEndVV = this.getEndVV().cloneMaxVV(i.getEndVV());
    newRTVV = this.getRTVV().cloneMaxVV(i.getRTVV());
    impreciseInv = new ImpreciseInv((HierInvalTarget)it,
                                    newStartVV,
                                    newEndVV,
                                    newRTVV);
    return(impreciseInv);
  }

 /** 
 *  Return -1, 0, or 1 if this is smaller, equal, or larger than 
 *  gi2, respectively, based on mechanism to provide total orders. 
 **/ 
  public int
  totalOrderCompare(GeneralInv gi2){
    /*
      return -1 if this < gi2 in the total order of the log
      return +1 if this > gi2 in the total order of the log
      return 0 if this and gi2 are equivalent and can be ordered either
      way (both start at the same time and end at the same time)

      we sort by CSN then by min acceptStamp then by serverId

      (1) CSN

      If this.CSN < gi2.CSN, then return -1
           
      Note that an unassigned CSN has an infinite value)

      For a bulk invalidation, consider the *minimum* CSN included
      in the invalidation.

      If this.CSN == gi2.CSN != CSN_UNASSIGNED, either order 
      is legal. If either is a bulk invalidation, the one with 
      the lower maxCSN goes first. If both have the same maxCSN
      then return 0

      (2) startTime acceptStamp

      assert this.CSN == gi2.CSN == CSN_UNASSIGNED

      if the invalidation is precise, consider its acceptStamp
      if the invalidation is imprecise, consider the startVV and
      look at the minimum entry across all nodeId's in startVV

      if they differ, return -1 or 1

      (3) nodeId
      if both minimum accept stamps in step (2) are equal, 
      then compare the node Ids

      if they differ, return -1 or 1

      (4) endtime acceptStamp

      assert csn, startTime, and nodeId are all the same

      send the "more precise" one first --> look at the endTime
      and the one with the smaller max endtime comes first

      if they differ, return -1 or 1
      if they are the same, return 0 (note that if the invals are
      imprecise, they may still differ in elements other than
      their max or min...but that's ok)
    */
    int result = 0;
    NodeId thisMinNodeId = null;
    NodeId gi2MinNodeId = null;

    if(this.startTime.getLocalClock() <
       gi2.getStartVV().getMinTimeStamp()){
      result = -1;
    }else if(gi2.getStartVV().getMinTimeStamp() <
             this.startTime.getLocalClock()){
      result = 1;
    }else{
      // Minimum entries in timestamps are equals, CSNs are equal
      assert(this.startTime.getLocalClock() ==
             gi2.getStartVV().getMinTimeStamp());
      thisMinNodeId = this.startTime.getNodeId();
      gi2MinNodeId = this.getMinNodeId(gi2.getStartVV());
      result = thisMinNodeId.compareTo(gi2MinNodeId);
      if(result == 0){
        assert(thisMinNodeId.equals(gi2MinNodeId));
        if(this.endTime.getLocalClock() <
           gi2.getEndVV().getMaxTimeStamp()){
          result = -1;
        }else if(gi2.getEndVV().getMaxTimeStamp() <
                 this.endTime.getLocalClock()){
          result = 1;
        }else{
          result = 0;
        }
      }
    }
    return(result);
  }

 /** 
 *  METHODS INHERITED FROM SINGLEWRITERINVAL 
 **/ 

 /** 
 *  Return the start vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getStart(){
    return(this.startTime.getLocalClock());
  }

 /** 
 *  Return the end vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getEnd(){
    return(this.endTime.getLocalClock());
  }

 /** 
 *  Return the end vv entry for this writer (start == end for precise) 
 **/ 
  public long
  getReal(){
    return(this.rtAcceptStamp.getLocalClock());
  }

 /** 
 *  Return the node Id  
 **/ 
  public NodeId
  getNodeId(){
    return(this.startTime.getNodeId());
  }

 /** 
 *  Clone this object while chopping off a prefix of the startTime 
 **/ 
  public SingleWriterInval
  cloneChopStartTime(long startTimeOfNew){
    SingleWriterImpreciseInv impreciseInv = null;
    AcceptStamp newStartTime = null;

    assert((this.startTime.getLocalClock() <= startTimeOfNew) &&
           (startTimeOfNew <= this.endTime.getLocalClock()));
    newStartTime = new AcceptStamp(startTimeOfNew,
                                   this.startTime.getNodeId());
    impreciseInv = new SingleWriterImpreciseInv(this.invalTarget,
                                                newStartTime,
                                                this.endTime);
    return(impreciseInv);
  }

 /** 
 *  Clone this object while chopping off a prefix of the endTime 
 **/ 
  public SingleWriterInval
  cloneChopEndTime(long endTimeOfNew){
    SingleWriterImpreciseInv impreciseInv = null;
    AcceptStamp newEndTime = null;

    assert((this.startTime.getLocalClock() <= endTimeOfNew) &&
           (endTimeOfNew <= this.endTime.getLocalClock()));
    newEndTime = new AcceptStamp(endTimeOfNew,
                                 this.endTime.getNodeId());
    impreciseInv = new SingleWriterImpreciseInv(this.invalTarget,
                                                this.startTime,
                                                newEndTime);
    return(impreciseInv);
  }

 /** 
 *  Convert to a string representation 
 **/ 
  public String
  toString(){
    String str = null;

    str = ("SingleWriterImpreciseInv : <" + this.invalTarget +
           ", " + this.startTime + ", " + this.endTime +
           ", " + this.rtAcceptStamp + ">");
    return(str);
  }

 /** 
 *  Return the node Id of vv which has the smallest accept stamp 
 **/ 
  private static final NodeId
  getMinNodeId(VV vv){
    Object token = null;
    VVIterator vvIter = null;
    long minTimeStamp = Long.MAX_VALUE;
    long timeStamp = 0;
    NodeId minNodeId = null;

    vvIter = vv.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      timeStamp = vv.getStampByIteratorToken(token);
      if(timeStamp < minTimeStamp){
        minTimeStamp = timeStamp;
        minNodeId = vv.getServerByIteratorToken(token);
      }
    }
    assert(minNodeId != null);
    return(minNodeId);
  }

 /** 
 *  memSize() -- 
 *    Estimate the memory footprint of this object in a cheap while not 
 *    very accurate way +/- 1~7 bytes.  
 *  
 *  Use SizeOf and TestClass to get the number of overheads: 
 *  whenever delete or add new members, recalculate the number of bytes 
 *  refering to ./test/FootPrintSizeOfStandardObject.txt 
 **/ 
  public long memSize(){
    assert invalTarget instanceof HierInvalTarget;
    int numHIT = ((HierInvalTarget)(invalTarget)).size();
    return 120 + 192*numHIT;
  }

 /** 
 *  Return true if "this" equals "obj" 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    SingleWriterImpreciseInv bi = null;

    if(obj instanceof SingleWriterImpreciseInv){
      bi = (SingleWriterImpreciseInv)obj;
      result = (this.invalTarget.equals(bi.invalTarget) &&
                this.startTime.equals(bi.startTime) &&
                this.endTime.equals(bi.endTime));
    }
    return(result);
  }

  public ImpreciseInv cloneImpreciseInv(){
    
    return new ImpreciseInv(this.invalTarget, this.getStartVV(),
        this.getEndVV());
  }
  
  
}


//---------------------------------------------------------------------------
/* $Log: SingleWriterImpreciseInv.java,v $
/* Revision 1.30  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.29  2007/03/02 09:43:46  zjiandan
/* Modified in-mem zip operation for BoundInval to adapt to the new semantics
/* of BoundInval.
/*
/* Revision 1.28  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.27  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.26  2005/07/18 05:10:23  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* */
//---------------------------------------------------------------------------
