package code;
import java.util.HashSet;

 /** 
 *  Implement a linked list of per-writer uncommitted invalidate 
 **/ 
public class SingleWriterLogUncommitted implements SingleWriterLog
{
  final static boolean debugging = false;
  long minCounter;
  long maxCounter;
  InvalListItem newest;
  InvalListItem oldest;
  NodeId myNodeId;
  
  //to store those iterators that are waiting
  //for a new update of this writer
  HashSet<InMemLogIterator> registeredIters;
  
  long totalItem;
  long totalBodySize;
  long totMemCharge;

  private static final boolean dbg = false;
  private static final String dbgm = "DBG: SingleWriterLogUncommitted";
  private final static boolean doExpensiveSanityChecks = Env.getDoExpensiveSanityChecks();
  public static final long UNINITIALIZED_COUNTER = AcceptStamp.BEFORE_TIME_BEGAN;

  // 
  // Caller must hold lock when accessing public methods
  //
  
 /** 
 *   Constructor 
 **/ 
  public 
  SingleWriterLogUncommitted(NodeId n){
    minCounter = UNINITIALIZED_COUNTER;
    maxCounter = UNINITIALIZED_COUNTER;
    newest = null;
    oldest = null;
    myNodeId = n;
    totalItem = 0;
    totalBodySize = 0;
    totMemCharge = 0;
    registeredIters = new HashSet<InMemLogIterator>();
  }

 /** 
 *  Get the estimated memory size for this in-mem per-writer-log 
 **/ 
  public long memSize(){
    return this.totMemCharge;
  }

 /** 
 *   return minCounter 
 **/ 
  public long getMinCounter(){
    return minCounter;
  }

 /** 
 * 	return maxCounter   
 **/ 
  public long getMaxCounter(){
    return maxCounter;
  }
	
 /** 
 *   return oldest 
 **/ 

  public InvalListItem getOldest(){
    return oldest;
  }

 /** 
 * 	return newest   
 **/ 
  public InvalListItem getNewest(){
    return newest;
  }
	
 /** 
 * 	return NodeId representing this log's writer  
 **/ 
  public NodeId getNodeId(){
    return myNodeId;
  }
	

 /** 
 *   public Methods for SingleWriterLog interface 
 **/ 
	
 /** 
 *  return the non-gap-filling item on the list  
 *  with the lowest start time s.t. endTime > start 
 *  or null if not exists. 
 *  
 *  Linear search cost: O(n) 
 **/ 
  public GeneralInv getNextByStart(long start){
    InvalListItem item = getNextItemByStart(start);
    GeneralInv ret = null;
    if(item != null){
      ret = item.getInv();
    }
    return ret;
  }	
    
 /** 
 *  Chop off all the items with End <= tailEnd such that remaining  
 *  first item with start = tailEnd +1; 
 *  
 *  return the chopped mem size 
 **/ 
  public long chopTail(long tailEnd){
    long beforeChop = this.totMemCharge;
    InvalListItem item = findLastItemNewerThan(tailEnd);
    if(item == null){//no newer item, truncate everything
	    
      assert(maxCounter <= tailEnd);
	    
      minCounter = UNINITIALIZED_COUNTER;
      maxCounter = UNINITIALIZED_COUNTER;
      newest = null;
      oldest = null;
      totalItem = 0;
      
      this.totMemCharge = 0;
      
      return beforeChop;
    }
	
    assert(item.getInv() instanceof SingleWriterInval);
    assert(((SingleWriterInval)(item.getInv())).getEnd() > tailEnd);
    /* 
     don't chop existing item -- because chopping existing one doesn't save space
     and will violate the invariant that each item's start will remaining
     the same which is the basice assumption for SingleWriterLogPointers to work.
    if(((SingleWriterInval)(item.getInv())).getStart() <= tailEnd){
      item.setInv(((SingleWriterInval)(item.getInv())).cloneChopStartTime(tailEnd+1));
    }
    */
    item.setOlder(null);
    
    long newStart = ((SingleWriterInval)item.getInv()).getStart();
    InvalListItem currentItem = oldest;
    while(currentItem != null){
      SingleWriterInval curInv = (SingleWriterInval)(currentItem.getInv());
      if (curInv.getEnd() >= newStart){
        break;
      }
      totalItem--;
      this.totMemCharge -= currentItem.memSize();
      if (curInv instanceof MultiObjPreciseInv){
        totalBodySize -= ((MultiObjPreciseInv)(curInv)).getLength();
      }else if (curInv.isBound()){
	totalBodySize -= ((BoundInval)(curInv)).getLength();
      }
      currentItem = currentItem.getNewer();
    }
    oldest = null;
    //System.gc(); moved to UpdateLog.gcWorker
    oldest = item;
    minCounter = ((SingleWriterInval)(item.getInv())).getStart();
		
    if (doExpensiveSanityChecks){
      verifyInvariants();
    }
    
    assert (beforeChop - this.totMemCharge) >=0;
    return beforeChop - this.totMemCharge;
  }
	
 /** 
 *   Return first item with endTime > t 
 *   or null if no such item 
 *   Linear search from the start:cost O(n) 
 *   
 *   used by gc so search from start 
 **/ 
  private InvalListItem findLastItemNewerThan(long t){
    InvalListItem currentItem = oldest;
    
    //the whole list is at most as new as t
    if (maxCounter <= t) {
      return null; 
    }
	
    while (currentItem  != null){   
      
      if (((SingleWriterInval)(currentItem.getInv())).getEnd()> t){
        return currentItem;
      } 
      currentItem = currentItem.getNewer();
    }
		
    return null;
  }

 /** 
 *   Find the item s.t. startTime <= t <= endTime  
 *   or null if no such item 
 *  
 *  linear forward search from head:oldest 
 **/ 
  private InvalListItem 
  findItemMatchingStartFromGivenPointer(long t, InvalListItem startItem){
      
    InvalListItem currentItem = startItem;
    //the whole list is larger than t
    if ((minCounter > t) || (maxCounter < t)){
      return null;
    }
    
    if(t == maxCounter){
      return newest;
    }
    
    
    if(t >= startItem.getInv().getStart()){//search towards newest    
      while (currentItem!=null){   
              
        if ((((SingleWriterInval)(currentItem.getInv())).getStart() <= t) 
            && (((SingleWriterInval)(currentItem.getInv())).getEnd() >= t)){//found
          return currentItem;
        }
      
        currentItem = currentItem.getNewer();
      }
    }else{//search towards oldest
      while (currentItem!=null){   
        
        if ((((SingleWriterInval)(currentItem.getInv())).getStart() <= t) 
            && (((SingleWriterInval)(currentItem.getInv())).getEnd() >= t)){//found
          return currentItem;
        }
      
        currentItem = currentItem.getOlder();
      }
    }

    return null;//not found
  }
 /** 
 *   Find the item s.t. startTime <= t <= endTime  
 *   or null if no such item 
 *  
 *  linear forward search from head:oldest 
 **/ 
  private InvalListItem findItemMatchingStartFromHead(long t){
	 
    InvalListItem currentItem = oldest;
    if ((minCounter > t)){// || (maxCounter < t)){
      return null;
    }
    
    //System.out.println();
    while (currentItem!=null){   
      	
      if ((((SingleWriterInval)(currentItem.getInv())).getStart() <= t) 
          && (((SingleWriterInval)(currentItem.getInv())).getEnd() >= t))
      {
        return currentItem;
      }
      currentItem = currentItem.getNewer();
      //System.out.print(".");
    }
    //System.out.println();
    return null;
  }
    
 /** 
 *   Find the item s.t. startTime <= t <= endTime  
 *   or null if no such item 
 *  
 *   linear backward search from tail:newest 
 **/ 
  private InvalListItem findItemMatchingStartFromTail(long t){
         
    InvalListItem currentItem = newest;
    if ((minCounter > t) || (maxCounter < t)){
      return null;
    }
    //System.out.println();
    while (currentItem!=null){   
          
      if ((((SingleWriterInval)(currentItem.getInv())).getStart() <= t) 
          && (((SingleWriterInval)(currentItem.getInv())).getEnd() >= t))
      {
        return currentItem;
        
      }
      currentItem = currentItem.getOlder();
      //System.out.print(".");
    }

    //System.out.println();
    return null;
  }
    
  //
  // Postcondition is that the specified invalidation has been
  // merged into the list. 
  // 
  // The list *never* has any *gaps* or *overlaps*. That is, 
  // each item on the list has a start time that is
  // the end time of the next older item on the list's end time + 1
  // and it has an end time that is the start time of the next
  // newer item - 1 (unless it is the oldest/newest item).
  //
  // But, the incoming message may, in fact, leave a gap or cause an 
  // overlap. How do we handle this?
  //
  // Gaps
  //
  // If the incoming message is *newer* than any item currently on 
  // the list and leaves a gap with the newest item on the list,
  // create a dummy entry to fill the gap. The dummy entry has
  // the needed start and end times to fill the gap, and it
  // has an *empty* interest set.
  //

  // If the incoming message is *older* than any item currently on 
  //  inv.startTime < min
  // the list, it is an error (unless the list is empty). (Even if no gap!)
  // Call Env.RemoteAssert() and throw a CausalOrderException
  // (this should percolate up to the invalStreamWorker which
  // should sever the connection. (Why is this an error? Because
  // it indicates that we don't have a causal view of this writer's
  // actions.
  //
  // Because we filled gaps when new messages arrived, no gap
  // can exist between items on the list.
  //
  // Overlap
  //
  // If two invalidations overlap, we need to *intersect* them
  // so that we don't lose any information. The structure of the
  // system ensures that a given stream contains no causal gaps
  // and we can use this knowledge to refine our local view
  // of the log as messages arrive.
  //
  // The key is that for any overlapping period of time, 
  // the interest set of any invalidation we see (or infer)
  // must include *all* affected interest sets/objects, so
  // if we see two different invalidations that overlap in
  // time, we know that the set of affected objects must
  // be the *intersection* of the interest sets of the
  // two invalidations. And, this applies to "inferred"
  // invalidations whose target interest set is the empty set. 
  //
  // note: for nonoverlapping invalidation accumulation across overlapping 
  //       invalidations, e.g. the accumulation Delay parameter. The above
  //       statements won't be true any more. Because the overlapping 
  //       invalidations arrive after the accumulated non overlapping 
  //       imprecise invalidation
  //  

 /** 
 *   Merge inv into current log, split items if necessary. 
 *   return the estimated mem overhead created by merging the inv. 
 **/ 
  public long 
  merge(GeneralInv inv_)
    throws CausalOrderException{

    long beforeMerge = this.totMemCharge;

    //see above comments for requirements and plan; see code for
    //committed list (above) for a way to structure this logic
    // note: now when intersect we only get the intersect of interest set.
    // also need to add dumn inv to filling gaps.
		
    assert(inv_ instanceof SingleWriterInval);
    SingleWriterInval inv = (SingleWriterInval) inv_;
    assert( myNodeId.equals(inv.getNodeId()) );
    assert(!inv.isCommitted());
    assert(inv.getStart()>=UNINITIALIZED_COUNTER);
		
    if ((newest != null) && (inv.getStart() > maxCounter +1)){
      
      assert(inv.getEnd() > maxCounter +1);
         
      //create dummy item
      SingleWriterInval dummyInv = null;
      dummyInv = new SingleWriterImpreciseInv(new HierInvalTarget(),
					      new AcceptStamp(maxCounter+1, myNodeId),
					      new AcceptStamp(inv.getStart()-1, myNodeId));

      InvalListItem dummy = new InvalListItem(dummyInv);
      
      totalItem++;
      this.totMemCharge += dummy.memSize();

      newest.setNewer(dummy);
      dummy.setOlder(newest);
      newest = dummy;
      assert(dummyInv.getEnd() == inv.getStart() -1);
      maxCounter = dummyInv.getEnd();
      if(doExpensiveSanityChecks){
        verifyNodeInvariants(dummy);
      }
    }
		
    if (oldest != null){
      assert(minCounter == ((SingleWriterInval)(oldest.getInv())).getStart());
    }
    // if inv is older than any item in the list, causal order violated
    if ( inv.getStart() < minCounter){
      //
      // to support clustering delay, we should allow the disorder arrive of 
      // invalidate
      // -- old code
      //throw new CausalOrderException("per-writer uncommitted log gap detected");
      //
      
      Env.tbd("To support clustering unintereted invs across interested invs"
              + " i.e. delay sending the accumulated impreciseinv, "
              + " Per-Writer-logs have to allow disorder arrival of invs"
              + " for example, the interested preciseinv arrives before "
              + " the arrival of a big imprecise inv whose startVV is smaller"
              + " than the preciseInv. This relaxation requires zip to "
              + " combine the overlapped invs while for liveness and accurate"
              + " information, it should take the intersection. One option"
              + "(mike proposed) is to batch the arriving invs and reorder "
              + " them. Need to rethink about it if we do need the DELAY "
              + " feature. If finally we decide not to support DELAY, need "
              + " to disable this trunk of code.");

      //create dummy item
      SingleWriterInval dummyInv = 
        new SingleWriterImpreciseInv(new HierInvalTarget(),
                                     new AcceptStamp(inv.getStart(), myNodeId),
                                     new AcceptStamp(minCounter-1, myNodeId));

      InvalListItem dummy = new InvalListItem(dummyInv);
      totalItem++;
      this.totMemCharge += dummy.memSize();
      
      oldest.setOlder(dummy);
      dummy.setNewer(oldest);
      oldest = dummy;
      assert(dummyInv.getStart() ==  inv.getStart());
      minCounter = dummyInv.getStart();
      if(doExpensiveSanityChecks){
        verifyNodeInvariants(dummy);
      }
    }			
    
    //everything is fine now, start zipping.
    InvalListItem start = findItemMatchingStartFromTail(inv.getStart());
    zip(start, inv);
    return this.totMemCharge - beforeMerge;

  }

 /** 
 *  Recursively "zip" together inv with current 
 *  list, "decomposing" invals that inv has more 
 *  info about as we go 
 *   
 *  return the mem overhead created by zipping the inv  
 **/ 
  private void
  zip(InvalListItem matchStart, SingleWriterInval inv){
    MultiObjPreciseInv mopi = null;
    MOITBoundEntry[] entries = null;
    ImmutableBytes ib = null;
    byte[] data = null;

    if(matchStart == null){
      assert((newest == null) 
             || (((SingleWriterInval)(newest.getInv())).getEnd() == inv.getStart() - 1));
			
      InvalListItem current = new InvalListItem(inv);
      if (inv.isBound()){
        if(inv instanceof MultiObjPreciseInv){
          mopi = (MultiObjPreciseInv)inv;
          entries = mopi.getBoundEntriesDangerous();
          for(int i = 0; i < entries.length; i++){
            ib = entries[i].getImmutableBytes();
            data = ib.dangerousGetReferenceToInternalByteArray();
            totalBodySize += data.length;
          }
          entries = null;
        }else{
          totalBodySize += ((BoundInval)(inv)).getLength();
        }
      }
      totalItem++;    
      this.totMemCharge += current.memSize();

      current.setOlder(newest); //advance newest to current
      
      if(current.getOlder() != null){
        current.getOlder().setNewer(current);
      } else{// this is the first one
        oldest = current;
        minCounter = inv.getStart();
      }
      newest = current;
      long newEnd = ((SingleWriterInval)(current.getInv())).getEnd();
      if(maxCounter < newEnd){
        maxCounter = newEnd;//something new
        callbackIterators(current);
      }
      if (doExpensiveSanityChecks) {
        verifyInvariants();
      }
      return;
    }else{
      //postcondition of matchStart
      SingleWriterInval matchInv = (SingleWriterInval)(matchStart.getInv());
      assert(matchInv.getStart() <= inv.getStart());
      assert(matchInv.getEnd() >= inv.getStart());
      if (doExpensiveSanityChecks) {
        verifyNodeInvariants(matchStart);
      }
      
      SingleWriterInval remainder = intersectOne(matchStart, inv);
      // find the matchStart -- because the old one might have been splitted
      // into two items with the original InvalListItem point to the part < start
      // therefore, we need to make sure that the "matchStart" still point to the
      // exact start.
      InvalListItem  matchStartNext = matchStart.getNewer();
      long newStart = ((SingleWriterInval)(matchStart.getInv())).getStart();
      if(newStart != inv.getStart()){//the matchStart has changed
        assert matchStartNext != null;
        assert ((SingleWriterInval) (matchStartNext.getInv())).getStart() == inv.getStart();
        matchStart = matchStartNext;//make matchStart point to the right item
        matchStartNext = matchStart.getNewer();
      }
      
      if(remainder != null){
        assert(matchStartNext ==null)
          ||(((SingleWriterInval)(matchStartNext.getInv())).getStart() == remainder.getStart()); 
        zip(matchStart.getNewer(), remainder);
      }
      if (doExpensiveSanityChecks) {
        verifyNodeInvariants(matchStart);
      }
      return;
    }
  }



 /**
  * tell all registered iterator that new item is available
  * @param current
  */
    
  private void callbackIterators(InvalListItem current){
    
    //System.out.println("new inval: "+ inv + "registeredIters=" + registeredIters.size());
    if(!registeredIters.isEmpty()){//notify Iterator which are waiting for new invalidation
      //
      //callback for the registered InMemLogIterators
      //
      Object[] iters = registeredIters.toArray();
      HashSet<InMemLogIterator> newIters = new HashSet<InMemLogIterator>();
      for(int i = 0; i < iters.length; i ++){
        try{
          ((InMemLogIterator)iters[i]).addPointer(myNodeId, current);
          //System.out.println("iters[" + i + "]: add " + current.getInv());
          
        }catch(StaleInvalidationException e){
          // can't remove yet, still waiting for new update
          // because iter.cvv is still larger than current.getEndVV();
          //System.out.println("iters[" + i + "]: remain registered ");
          newIters.add((InMemLogIterator)(iters[i]));
        }
       
      }
      registeredIters.clear();//remove all satisfied iterators
      registeredIters.addAll(newIters);//add back those do not get new available items
      newIters = null;
    }
  }

 
  //
  // Combine one invalidation with one element of the list.
  // This may replace the element on the list with up
  // to three pieces -- (1) the part before the new part
  // (2) the common part that intersects between both
  // (3) the part after the new part. 
  // If the new invalidation ends *after* the current list
  // item ends, then return an invalidation representing 
  // the remainder (the part that doesn't intersect
  //
 /** 
 *   Decompose matchStart to merge inv if necessary 
 *   return inv.newerThan matchStart part and matchStart always 
 *   refer to the last part of its decomposed parts. 
 **/ 
  private SingleWriterInval 
  intersectOne(InvalListItem matchStart, SingleWriterInval inv){
    SingleWriterInval matchStartInv = (SingleWriterInval)(matchStart.getInv());
    // matchStart invariants
    assert(matchStartInv.getStart() <= inv.getStart());
    assert(matchStartInv.getEnd() >= inv.getStart());
    SingleWriterInval remainder = null;
    SingleWriterInval choppedinv = inv;
		
    if(inv.getEnd() > matchStartInv.getEnd()){
      remainder = inv.cloneChopStartTime(matchStartInv.getEnd() + 1);
      choppedinv = inv.cloneChopEndTime(matchStartInv.getEnd());
    }else{
      remainder = null;
    }

    SingleWriterInval replacement1;
    SingleWriterInval replacement2;
    SingleWriterInval replacement3;
    
    if(matchStartInv.getStart() < inv.getStart()){
      replacement1 = matchStartInv.cloneChopEndTime(inv.getStart()-1);
      assert(replacement1 != null);
      
    }
    else{
      //because matchstart invariant startTime<= inv.starttime <=EndTime
      assert(matchStartInv.getStart() == inv.getStart());

      replacement1 = null;
    }
		
    assert(matchStart.getInv() instanceof SingleWriterInval);
    
    replacement2 = (SingleWriterInval)matchStartInv.cloneIntersectInvaltargetChopStartEnd(choppedinv);
    if(matchStartInv.getEnd() > inv.getEnd()){
      replacement3 = matchStartInv.cloneChopStartTime(inv.getEnd()+1);
      
      assert(remainder == null);
    }
    else{
      replacement3 = null;
    }

    //insert replacements into the per-writer-log
    // fix the start with InvalListItem
    
    if(replacement1 != null){//insert replace1 into the list before matchstart
    //replace matchStart inv with replacement2
      // should recalculate the memSize as the invalidate is changed
      this.totMemCharge -= matchStart.memSize();
      matchStart.setInv(replacement1);
      this.totMemCharge += matchStart.memSize();
      assert(replacement2!=null);// there should always be nonempty overlap parts 
      // because of matchstart invariant
      // therefore we don't need to increase the count for replacement2
      // as it should have been counted previously before merge the inv
      //
      InvalListItem newNewer = new InvalListItem(replacement2);
      
      newNewer.setOlder(matchStart);
      newNewer.setNewer(matchStart.getNewer());
      
      
      if (matchStart.getNewer()==null)
      {//original matchStart is the newest
        newest = newNewer;
        assert(maxCounter >= ((SingleWriterInval)(newNewer.getInv())).getEnd());
      } else { matchStart.getNewer().setOlder(newNewer);}
      
      matchStart.setNewer(newNewer);
      
      if (doExpensiveSanityChecks){
        verifyNodeInvariants(matchStart);
      }
      
      //update the matchStart to point to the match start item
      //which is always replacement2
      matchStart = matchStart.getNewer();
      
      assert matchStart == newNewer;
      totalItem++;
      this.totMemCharge += newNewer.memSize();
    }else{
      this.totMemCharge -= matchStart.memSize();
      matchStart.setInv(replacement2);
      this.totMemCharge += matchStart.memSize();
    }

    if(replacement3 != null){//insert replacement3 into list
      assert(remainder == null);
      InvalListItem newNewer = new InvalListItem(replacement3);
      newNewer.setNewer(matchStart.getNewer());
      newNewer.setOlder(matchStart);

      totalItem++;
      this.totMemCharge += newNewer.memSize();

      if (matchStart.getNewer()==null)
      {//original matchStart is the newest
        newest = newNewer;
		    
        //maxCounter should still be the same as newNewer is
        //the last part of original matchStart
        assert(maxCounter == ((SingleWriterInval)(newNewer.getInv())).getEnd());
      } else { matchStart.getNewer().setOlder(newNewer);}
		   
      matchStart.setNewer(newNewer);
      if (doExpensiveSanityChecks){
        verifyNodeInvariants(newNewer);
      }
    }
	
    if(debugging){
      System.out.println("replacement1: "+replacement1);
      System.out.println("replacement2: " + replacement2);
      System.out.println("replacement3: " + replacement3);
    }
    if (doExpensiveSanityChecks) {
      verifyNodeInvariants(matchStart);
    }
    return remainder;
  }


  public InvalListItem findEmbargoedInval(DebargoMsg unbind){
		
    assert( myNodeId.equals(unbind.getNodeId()) );
    //scan down list from newest to oldest
    InvalListItem cur = newest;
    
    while ((cur != null) && 
           (((SingleWriterInval)(cur.getInv())).getStart() >= unbind.getTime()))
    {
      if ((((SingleWriterInval)(cur.getInv())).getStart() == unbind.getTime()) 
          && (cur.getInv().isEmbargoed())){
        // ??????
        //will the BoundInv still be instanceof BoundInv when apply it to 
        // local per-writer log?
	
	assert(cur.getInv().getInvalTarget() instanceof ObjInvalTarget);
        assert(unbind.getInvalTarget() instanceof ObjInvalTarget);
	ObjInvalTarget cInvTarget = (ObjInvalTarget)(cur.getInv().getInvalTarget());
        assert((cInvTarget).equals(((ObjInvalTarget)(unbind.getInvalTarget()))));
        
        return cur;
      }
      cur = cur.getOlder();
    }
    return null;
  }

 /** 
 *  -- depracated find the bound invalidates in this per-writer-log 
 **/ 
  public InvalListItem findBoundInval(UnbindMsg unbind){
    GeneralInv gi = null;
    MultiObjPreciseInv mopi = null;

    assert( myNodeId.equals(unbind.getNodeId()) );
    //scan down list from newest to oldest
    InvalListItem cur = this.newest;

    if(cur != null){
      gi = cur.getInv();
    }
    while((gi != null) &&
          (((SingleWriterInval)gi).getStart() >= unbind.getTime())){
      if(((SingleWriterInval)gi).getStart() == unbind.getTime()){
        if(cur.getInv() instanceof BoundInval){
          // ??????
          //will the BoundInv still be instanceof BoundInv when apply it to 
          // local per-writer log?
          assert(cur.getInv().getInvalTarget() instanceof ObjInvalTarget);
          assert(unbind.getInvalTarget() instanceof ObjInvalTarget);
	  ObjInvalTarget cInvTarget = (ObjInvalTarget)(cur.getInv().getInvalTarget());
          assert(cInvTarget.equals(((ObjInvalTarget)(unbind.getInvalTarget()))));
          return cur;
        }else if(cur.getInv().isBound()){
          assert(cur.getInv() instanceof MultiObjPreciseInv);
          mopi = (MultiObjPreciseInv)cur.getInv();
          if(mopi.hasBoundEntry(unbind.getInvalTarget())){
            return cur;
          }
        }
      }
      cur = cur.getOlder();
      if(cur != null){
        gi = cur.getInv();
      }
    }
    return null;
  }

 /** 
 *  unbind a BoundInval if exists. 
 *  return true if the BoundInval exists 
 *         false if it doesn't exist 
 **/ 
  public boolean unbind(UnbindMsg unbind){
    InvalListItem oldBound = this.findBoundInval(unbind);
    if(oldBound == null){
      return false;
    }
    assert(oldBound != null);
    assert(oldBound.getInv() instanceof BoundInval);
    this.totMemCharge -= oldBound.memSize();
    oldBound.replaceBoundWithUnbound(unbind);
    this.totMemCharge += oldBound.memSize();
    return true;
  }

 /** 
 *  calculate the number of valid items whose start > commitPoint 
 **/ 
  public long getOrderError(long commitPoint){
    long currOE = 0;
    InvalListItem nextItem = findLastItemNewerThan(commitPoint);
    while(nextItem != null){
    
      if (!nextItem.getInv().getInvalTarget().isEmpty()){//it's not dummy inv
        currOE++;//each iterm should be a precise inv, because this single writer
                 //must be ME.
      }
      nextItem = nextItem.getNewer();
    }
    return currOE;
  }

 /** 
 *   Verify Individual Item invariants: no gap between neighbors 
 **/ 
  private void verifyNodeInvariants(InvalListItem current){
    //no gaps in time between current and next earlier/later node
    //same writer for this node and earlier/later nodes
	
    assert(current != null);
    if(current.getOlder() != null){

      assert(((SingleWriterInval)(current.getOlder().getInv())).getEnd() 
             == ((SingleWriterInval)(current.getInv())).getStart()-1)
             : current.getOlder().getInv().toString() 
             + " current=" + current.getInv().toString();
    }
    if(current.getNewer() != null){
      assert(((SingleWriterInval)(current.getNewer().getInv())).getStart() 
             == ((SingleWriterInval)(current.getInv())).getEnd() + 1);
		
    }
    return;
  }
	
 /** 
 *   Verify List invariants 
 **/ 
  public void
  verifyInvariants(){
    if ((oldest == null ) || (newest == null)) {//if Log is empty
      assert((oldest==null) && (newest==null)
             && (minCounter == UNINITIALIZED_COUNTER) && (maxCounter == UNINITIALIZED_COUNTER));
    }else {
      assert((minCounter <= maxCounter) 
	     && (minCounter == ((SingleWriterInval)(oldest.getInv())).getStart())
             &&(maxCounter == ((SingleWriterInval)(newest.getInv())).getEnd()));
			
      //verify every item in the list one by one
      InvalListItem currentItem = oldest;
      while (currentItem!=null){   
        verifyNodeInvariants(currentItem);
        currentItem = currentItem.getNewer();
      }
	    
    }
  }
  
 /** 
 *  Return a string representation 
 **/ 
  public final String toString(){
    return "In-Mem-Log: \n memSize: " + totMemCharge + "\n" +oldest;
  }
  
 /** 
 *   Module Test 
 **/ 
  public static void main(String[]args)
    throws Exception{
    //moved to junit
  }
  
 /** 
 *  return the non-gap-filling item on the list  
 *  with the lowest start time s.t. endTime > start 
 *  or null if not exists. 
 *  
 *  Linear search cost: O(n) 
 **/ 
  public InvalListItem getNextItemByStart(long start){
    // return the item on the list 
    // with the lowest start time s.t. endTime is > start
    
    // We may have "inferred" some writes with empty 
    // interest set to "fill gaps"; skip those
    //
    if(dbg){
      Env.inform(dbgm + " & call getNextByStart & " + start
                 + " newest: " 
                 + (newest == null ? "NULL" : newest.getInv().toString())
                 + " oldest: "
                 + (oldest == null ? "NULL" : oldest.getInv().toString()));
    }

    InvalListItem currentItem = oldest;
    
    //the whole list is at most as new as t
    if (maxCounter <= start) {
      return null; 
    }
                
    while (currentItem  != null){
      SingleWriterInval nextInv = (SingleWriterInval)(currentItem.getInv());
      if ((nextInv.getEnd()> start)&& !(nextInv.getInvalTarget().isEmpty())){
        return currentItem;
      } 
      currentItem = currentItem.getNewer();
    }
    if(dbg){
      Env.inform(dbgm + " & getNextByStart returns <null> &" + start);
    }            
    return null;
  }

  public void register(InMemLogIterator inMemLogIterator){
    this.registeredIters.add(inMemLogIterator);    
  }

  public void registerAll(HashSet<InMemLogIterator> activeIters){
    this.registeredIters.addAll(activeIters);
  }

  public void remove(InMemLogIterator iter){
    this.registeredIters.remove(iter);
  }
}


//---------------------------------------------------------------------------
/* $Log: SingleWriterLogUncommitted.java,v $
/* Revision 1.27  2007/09/10 23:52:22  zjiandan
/* upgrade to newest BerkeleyDB je version.
/* */
//---------------------------------------------------------------------------
