package code;

 /** 
 *  Periodically sanity check local view of spanning tree 
 **/ 
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Iterator;

public class SpanningTreeWatchdog extends Thread
{
private SDIMSController controller;
private static final long INITIAL_SCAN_DELAY_MS = 15000;
private static final long SCAN_INTERVAL_MS = 15000;
private static final long TOLERATE_STATUS_DOWN_COUNT = 1;
private HashMap status; // SubscriptionSet --> {lastOKScan}
private long scanCount;
private static final boolean verbose = true;


public static final long CONTROLLER_CARES_THIS_IS = 1;
public static final long CONTROLLER_THINKS_UP = 1<<1;
public static final long CONTROLLER_THINKS_REVERSE_UP = 1<<2;
public static final long CORE_THINKS_UP = 1<<3;
public static final long CORE_THINKS_REVERSE_UP = 1<<4;


public SpanningTreeWatchdog(SDIMSController controller_)
{
    this.controller = controller_;
    this.status = new HashMap();
    scanCount = 0;

    assert(CONTROLLER_CARES_THIS_IS == 1);
    assert(CONTROLLER_THINKS_UP == 2);
    assert(CONTROLLER_THINKS_REVERSE_UP == 4);
    assert(CORE_THINKS_UP == 8);
    assert(CORE_THINKS_REVERSE_UP == 16);
}

public void run()
{
    try{
	Thread.sleep(INITIAL_SCAN_DELAY_MS);
    }
    catch(InterruptedException e){
	Env.warn("SpanningTreeWatchdog interrupted during initial sleep");
    }
    while(true){
	scanCount++;
	doCheck();
	try{
	    Thread.sleep(SCAN_INTERVAL_MS);
	}
	catch(InterruptedException e){
	    Env.warn("SpanningTreeWatchdog interrupted during interval sleep");
	}
    }
}

private void
doCheck()
{
    ListIterator li;

    List currentList = controller.dbgGetCopyISIAmInterestedIn();
    removeStatusNoLongerInterestedIn(currentList);
    try{
	li = currentList.listIterator(0);
    }catch(IndexOutOfBoundsException e){
	return;
    }
    while(li.hasNext()){
	SubscriptionSet ss = (SubscriptionSet)li.next();
	checkOne(ss);
    }
}

private void
removeStatusNoLongerInterestedIn(List currentList)
{
    Iterator ii;
    SubscriptionSet ss = null;
    SubscriptionSet sanityLastRemove = null;
    try{
	ii = status.keySet().iterator();
    }catch(IndexOutOfBoundsException e){
	return;
    }
    while(ii.hasNext()){
	ss = (SubscriptionSet)ii.next();
	if(!currentList.contains(ss)){
	    ii.remove();
	    if(verbose){
		Env.inform("SpanningTreeWatchdog: " + ss.toString() 
			   + " removed from interestSetsICareAbout at " 
			   + System.currentTimeMillis());
	    }
	    sanityLastRemove = ss;
	}
    }
    if(sanityLastRemove != null){
	assert(!status.containsKey(sanityLastRemove));
    }
    return;
}

private void
checkOne(SubscriptionSet ss)
{
    Long lastConnected = (Long)status.get(ss);
     if(lastConnected == null){
	status.put(ss, new Long(scanCount-1));
	if(verbose){
	    Env.inform("SpanningTreeWatchdog: " + ss.toString()
		       + " added to interestSetsICareAbout at "
		       + System.currentTimeMillis());
	}
	lastConnected = (Long)status.get(ss);
    }
    long stat = controller.dbgGetSpanningStatus(ss);
    if((stat & CONTROLLER_CARES_THIS_IS) == 0){
	//
	// Our list is stale
	//
	Env.warn("SpanningTreeWatchdog: " + ss.toString()
		 + " stale entry not checked");
	return;
    }
    if((stat & CONTROLLER_THINKS_UP) != 0){
	if(scanCount - lastConnected.longValue() > TOLERATE_STATUS_DOWN_COUNT){
	    Env.warn("SpanningTreeWatchdog: " + ss.toString()
		     + " back up at "
		     + System.currentTimeMillis());
	}
	status.put(ss, new Long(scanCount));
    }
    else{
	assert((stat & CONTROLLER_THINKS_UP) == 0);
	if(scanCount - lastConnected.longValue() >= TOLERATE_STATUS_DOWN_COUNT){
	    long approxDown = (scanCount - lastConnected.longValue())*SCAN_INTERVAL_MS;
	    long lastUp = System.currentTimeMillis() - approxDown;
	    Env.warn("SpanningTreeWatchdog: " + ss.toString()
		     + " down at "
		     + System.currentTimeMillis()
		     + " last seen/assumed up at "
		     + lastUp
		     + "( " 
		     + approxDown/1000 
		     + " s)");
	}
    }
    if(((stat & CONTROLLER_THINKS_UP) != 0) 
       && ((stat & CONTROLLER_THINKS_REVERSE_UP) != 0)
       ||
       ((stat & CONTROLLER_THINKS_UP) == 0) 
       && ((stat & CONTROLLER_THINKS_REVERSE_UP) == 0)){
	//
	// Forward and reverse consistent
	//
    }
    else{
	Env.warn("SpanningTreeWatchdog: inconsistent view of forward "
		 + "and reverse for interest set "
		 + ss.toString()
		 + " Controller thinks forward connection is "
		 + (((stat & CONTROLLER_THINKS_UP) != 0) ? " UP " : " DOWN ")
		 + " Controller thinks reverse connection is " 
		 + (((stat & CONTROLLER_THINKS_REVERSE_UP) != 0) ? " UP " : " DOWN "));
    }
    if(((stat & CONTROLLER_THINKS_UP)!= 0) && ((stat & CORE_THINKS_UP) != 0)
       ||
       ((stat & CONTROLLER_THINKS_UP) == 0) && ((stat & CORE_THINKS_UP) == 0)){
	//
	// Controller and core have consistent view of connection
	//
    }
    else{
	Env.warn("SpanningTreeWatchdog: inconsistent view of interest set "
		+ ss.toString()
		+ " Controller thinks connection is "
		+ (((stat & CONTROLLER_THINKS_UP) != 0) ? " UP " : " DOWN ")
		+ " Core thinks connection is " 
		 + (((stat & CORE_THINKS_UP) != 0) ? " UP " : " DOWN "));
    }
    if(((stat & CONTROLLER_THINKS_REVERSE_UP) != 0) && ((stat & CORE_THINKS_REVERSE_UP) != 0)
       ||
       ((stat & CONTROLLER_THINKS_REVERSE_UP) == 0) && ((stat & CORE_THINKS_REVERSE_UP) == 0)){
	//
	// Controller and core have consistent view of connection
	//
    }
    else{
	Env.warn("SpanningTreeWatchdog: inconsistent view of interest set "
		+ ss.toString()
		+ " Controller thinks connection is "
		+ (((stat & CONTROLLER_THINKS_REVERSE_UP) != 0) ? " UP " : " DOWN ")
		+ " Core thinks connection is " 
		 + (((stat & CORE_THINKS_REVERSE_UP) != 0) ? " UP " : " DOWN "));
    }


}



}

//---------------------------------------------------------------------------
/* $Log: SpanningTreeWatchdog.java,v $
/* Revision 1.5  2005/03/15 05:26:47  nayate
/* Modified to use the new InterestSet code
/*
/* Revision 1.4  2005/03/01 10:18:14  nayate
/* Modified for new code
/*
/* Revision 1.3  2004/05/20 04:48:41  dahlin
/* Finished SDIMS controller self test for spanning tree; also basic spanning watchdog works
/*
/* Revision 1.2  2004/05/20 04:16:59  dahlin
/* Fixed ^M formatting
/*
/* Revision 1.1  2004/05/13 18:49:17  dahlin
/* Clean SDIMSController by moving SpanningTree directory out as separate class (rather than using raw SDIMSInterface)
/*
 */
//---------------------------------------------------------------------------
