package code;

 /** 
 **/ 
import java.rmi.*;

public class SpanningTreeWorker extends Thread
{
private final static boolean TBD = false;

private SDIMSController controller;
private WorkQueue work;
private TimeoutQueue timeouts;
private RMIClient rmiClient;
private SpanningDirectory directory;
private Core core;

public
SpanningTreeWorker(SDIMSController controller_, 
		   WorkQueue work_,
		   TimeoutQueue timeouts_,
		   RMIClient rmiClient_,
		   SpanningDirectory directory_, 
                   Core core_)
{   
    this.controller = controller_;
    this.work = work_;
    this.timeouts = timeouts_;
    this.rmiClient = rmiClient_;
    this.directory = directory_;
    this.core = core_;
}


public void 
run()
{
    SubscribeInvalRequest r;
    long timeoutMS;
    NodeId target;

    while(true){
	try{
	    r = (SubscribeInvalRequest)work.getNext();
	    timeoutMS = r.getTimeout();

		//System.out.println("SpanningTreeWorker looping");

	    //
	    // Get the target to connect to.
	    // Note that if we have successfully connected any
	    // of the incoming connections and this is just the
	    // first retry, use the old SDIMS value rather
	    // than forcing reaggragation
	    //
	    if(r.getRetryCount() == 1 
	       &&
	       (r.getStatus() == SubscribeInvalRequest.STATUS_CONNECTED 
		|| r.getPushChannelStatus() == SubscribeInvalRequest.STATUS_CONNECTED)){
		//
		// keep old target for now
		//
		target = r.getTarget();
		assert(target != null);
		assert(target.isValid());
	    }
	    else{ // Find new target
		target = directory.getTargetAndSubscribeNotifyChanges(r, controller);
		if(!target.equals(r.getTarget())){
		    coreCloseConnections(r);
		    r.setTargetAndResetStatus(target);
		}
	    }

	    if(target == null || !target.isValid()){
		//
		// SDIMS couldn't tell us who to connect to
		// Retry after a timeout.
		//
		timeouts.insert(timeoutMS, r);
		continue;
	    }
	    assert(r.getTarget().isValid());

	    //
	    // I am root -- issue a request to subscribe to parent
	    // interest set. In this case, don't need to subscribe
	    // to a parent.
	    //
	    if(r.getTarget().equals(core.getMyNodeId())){
		r.setStatus(SubscribeInvalRequest.STATUS_I_AM_ROOT_OF_SPANNING_TREE);
		r.setReverseConnectionStatus(SubscribeInvalRequest.STATUS_I_AM_ROOT_OF_SPANNING_TREE);
		r.setPushChannelStatus(SubscribeInvalRequest.STATUS_I_AM_ROOT_OF_SPANNING_TREE);
		r.setReversePushChannelStatus(SubscribeInvalRequest.STATUS_I_AM_ROOT_OF_SPANNING_TREE);
		controller.handleRootSpanningTree(r);
		continue;
            }

	    //
	    // Issue async connect request to parent and create
	    // timeout to check to make sure connection happens.
	    //
	    if(r.getStatus() == SubscribeInvalRequest.STATUS_FAILED){
		r.setStatus(SubscribeInvalRequest.STATUS_PENDING);
		boolean timedOut = issueRMI(r.getSubscriptionSet(), 
					    r.getStartVV(),
					    r.getTarget(), 
					    core.getMyNodeId());
		if(timedOut){
		    //
		    // We already paid a timeout; re-issue request w/o 
		    // timeout. Unless we've had a bunch of retries
		    // already (e.g., if we can contact remote
		    // node that is not accepting connections,
		    // it replies quickly).
		    //
		    if(r.getRetryCount() < 5){
			timeouts.insert(0, r);
		    }
		    else{
			timeouts.insert(timeoutMS, r);
		    }
		    continue;
		}
	    }

	    //
	    // Also request to set up the reverse connection
	    //
	    if(r.getReverseConnectionStatus() 
	       == SubscribeInvalRequest.STATUS_FAILED){
		r.setStatus(SubscribeInvalRequest.STATUS_PENDING);
		VV remoteStartVV = getRemoteStartVV(r.getSubscriptionSet(), 
						    r.getTarget());
		issueRMI(r.getSubscriptionSet(), remoteStartVV,
			 core.getMyNodeId(), r.getTarget());
	    }


	    //
	    // Set up push update channels along spanning tree
	    // 
	    if(r.getDoPush()){
		setupPushChannels(r);
	    }

	    //
	    // Requests succeed or fail; either way, retry after
	    // timeout
	    //
	    timeouts.insert(timeoutMS, r);
	    continue;
	}
	catch(RuntimeException e){
	    Env.warn("SpanningTreeWorker had exception: "
		     + e.toString() + "...Skipping request and continuing.");
	    e.printStackTrace();
	}
    }
}


/*
 *------------------------------------------------------------------
 *
 * coreCloseConnections --
 *
 *          Tell the core to close the spanning tree (and update)
 *          connections; we are choosing a new parent.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warnCCC = false;

private void
    coreCloseConnections(SubscribeInvalRequest r)
{
    if(!warnCCC){
	Env.warn("TBD: SpanningTreeWorker calls Core.closeInvalConnection() "
		 + "and Core.closeBodyConnection()");
	warnCCC = true;
    }

    /* TBD:
      core.closeInvalConnection(r.getInterestSet(), myNodeId, r.target, all writers);
      core.closeInvalConnection(r.getInterestSet(), r.target, myNodeId, all writers);
      if(r.doPushBody){
        core.closeBodyConnection(r.getInterestSet(), myNodeId, r.target, all writers);
        core.closeBodylConnection(r.getInterestSet(), r.target, myNodeId, all writers);
      }
    */
      

}



/*
 *------------------------------------------------------------------
 *
 * issueRMI --
 *
 *         Issue an RMI to subscribe to the inval stream.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Return true if we time out. Return false otherwise.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warned = false;
private boolean
issueRMI(SubscriptionSet ss, VV startVV, NodeId from, NodeId to)
{

    long DUMMY_CSN = CSNContainer.DUMMY_CSN;
    if(!warned){
	Env.performanceWarning("MAX_DELAY_MS used in SpanningTreeWorker should "
			       + " really be a parameter supplied at run time");
	warned = true;
    }
    int MAX_DELAY_MS = 30000;

    try{
	rmiClient.subscribe(ss,
			    startVV,
			    
			    from,
			    to,
			    Config.getDNS(to), 
			    Config.getPortInval(to), 
			    86400000,
			    MAX_DELAY_MS,
                            0);
	return false;
    }
    //
    // For exceptions that reflect communications errors, treat
    // them as timeouts. Retry.
    //
    catch(RMINetworkException e){
	return true;
    }
    catch(RMIApplicationException f){
	Env.remoteAssert(f);
	return true;
    }
}


/*
 *------------------------------------------------------------------
 *
 * setupPushChannels --
 *
 *         If the streams are not connected, 
 *         subscribe to the update stream
 *         from parent and for parent to subscribe to update
 *         stream from me.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private void 
setupPushChannels(SubscribeInvalRequest pdr)
{
    NodeId target = pdr.getTarget();
    NodeId myNodeId = core.getMyNodeId();
    assert(target.isValid());
    assert(!pdr.equals(myNodeId));
    assert(pdr.getDoPush());
    if(pdr.getPushChannelStatus() 
       == SubscribeInvalRequest.STATUS_FAILED){
	pdr.setPushChannelStatus(SubscribeInvalRequest.STATUS_PENDING);
	issueSubscribePushRMI(pdr.getSubscriptionSet(), pdr.getStartVV(),
			      target, myNodeId);
    }
    if(pdr.getReversePushChannelStatus() 
       == SubscribeInvalRequest.STATUS_FAILED){
	pdr.setPushChannelStatus(SubscribeInvalRequest.STATUS_PENDING);
	VV remoteStartVV = getRemoteStartVVBody(pdr.getSubscriptionSet(), 
						pdr.getTarget());
       issueSubscribePushRMI(pdr.getSubscriptionSet(), remoteStartVV,
			     myNodeId, target);
    }
}



/*
 *------------------------------------------------------------------
 *
 * issueSubscribePushRMI --
 *
 *         Issue RMIs to subscribe to the update stream
 *         from parent and for parent to subscribe to update
 *         stream from me.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Return true if we time out. Return false otherwise.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warned2 = false;
private boolean
issueSubscribePushRMI(SubscriptionSet ss, VV startVV, NodeId from, NodeId to)
{
    long DUMMY_CSN = CSNContainer.DUMMY_CSN;

    if(!warned2){
	Env.performanceWarning("MAX_DELAY_MS used in SpanningTreeWorker should "
			       + " really be a parameter supplied at run time");
	warned2 = true;
    }
    int MAX_DELAY_MS = 30000;

    try{
	rmiClient.subscribeBody(ss,
				startVV,
				from,
				to,
				Config.getDNS(to), 
				Config.getPortBody(to), 
				86400000);
	return false;
    }
    //
    // For exceptions that reflect communications errors, treat
    // them as timeouts. Retry.
    //
    catch(RMINetworkException e){
	return true;
    }
    catch(RMIApplicationException f){
	Env.remoteAssert(f);
	return true;
    }
}



/*
 *------------------------------------------------------------------
 *
 * getRemoteStartVV --
 *
 *          We are about to start sending invals to the specified
 *          node for the specified interest set. Decide what
 *          startVV to use. 
 *
 * ***LIMITATION***:
 *          Eventually, this should be an RMI. For now, just
 *          always start at the very beginning.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warnGRSVV = false;
private VV
getRemoteStartVV(SubscriptionSet ss, NodeId node)
{
    if(!warnGRSVV){
	Env.performanceWarning("SpanningTreeWorker always makes connections to "
			       + "parent with startVV of -1! Really need to RMI "
			       + "to parent to find right startVV for this IS");
	warnGRSVV = true;
    }
    assert(node.isValid());
    VV vv = AcceptVV.makeVVAllNegatives();
    return vv;
}

/*
 *------------------------------------------------------------------
 *
 * getRemoteStartVV --
 *
 *          We are about to start sending invals to the specified
 *          node for the specified interest set. Decide what
 *          startVV to use. 
 *
 * ***LIMITATION***:
 *          Eventually, this should be an RMI. For now, just
 *          always start at the very beginning.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static boolean warnGRSVVB = false;
private VV
getRemoteStartVVBody(SubscriptionSet ss, NodeId node)
{
    if(!warnGRSVV){
	Env.performanceWarning("SpanningTreeWorker always makes connections to "
			       + "parent with startVV of 0! Really need to RMI "
			       + "to parent to find right startVV for this IS");
	warnGRSVVB = true;
    }
    assert(node.isValid());
    VV vv = AcceptVV.makeVVAllNegatives();
    return vv;
}



public static void main(String s[])
{
    System.err.print("SpanningTreeWorker self test...");

    System.err.println("SpanningTreeWorker self test succeeds");
    System.exit(0);
    
}

}
//---------------------------------------------------------------------------
/* $Log: SpanningTreeWorker.java,v $
/* Revision 1.19  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.18  2005/10/13 00:24:24  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.17  2005/03/01 10:40:35  nayate
/* First successful compilation
/*
/* Revision 1.16  2005/03/01 08:34:12  nayate
/* Modified for new code
/*
/* Revision 1.15  2005/01/13 20:55:40  zjiandan
/* Reorganized sosp experiments files into sosp subdirectory under experiments.
/*
/* Revision 1.14  2004/05/24 10:37:33  arun
/* rectified some logic for updating ISStatus. Changes for dealing with bound invals
/* using sdims.
/*
/* Revision 1.13  2004/05/23 00:25:22  arun
/* only fresh writes get inserted in UPQ. Some debugging code related to SDIMS
/*
/* Revision 1.12  2004/05/20 04:48:41  dahlin
/* Finished SDIMS controller self test for spanning tree; also basic spanning watchdog works
/*
/* Revision 1.11  2004/05/20 02:12:40  dahlin
/* Fixing conflict with amol
/*
/* Revision 1.10  2004/05/20 02:08:47  arun
/* *** empty log message ***
/*
/* Revision 1.9  2004/05/20 01:44:08  dahlin
/* Looks like spanning trees are getting set up with TrivialSpanningTreeDirectory (though more testing remains)
/*
/* Revision 1.8  2004/05/20 00:47:35  dahlin
/* Fixed several bugs to make inval log exchange work (biggest one: ISIterator handling case when an single-writer log exists but has no records in it; also added some debugging tools
/*
/* Revision 1.7  2004/05/14 19:23:21  dahlin
/* Cleanup spanning tree logic; add update bodies
/*
/* Revision 1.6  2004/05/13 18:49:17  dahlin
/* Clean SDIMSController by moving SpanningTree directory out as separate class (rather than using raw SDIMSInterface)
/*
/* Revision 1.5  2004/05/10 22:45:47  dahlin
/* All unit: target tests succeed
/*
/* Revision 1.4  2004/05/10 20:48:19  dahlin
/* Clarified RMI exceptions; full version of (stub) DemandReadWorker
/*
/* Revision 1.3  2004/05/09 22:21:11  dahlin
/* Unit test and stub implementation for spanningTreeWorker
/*
/* Revision 1.2  2004/05/09 14:32:16  dahlin
/* Fix the ^M formatting
/*
/* Revision 1.1  2004/05/08 22:20:17  dahlin
/* Partially complete version of SDIMSController (it should compile w/o error, though)
/*
 */
//---------------------------------------------------------------------------
