package code;

//---------------------------------------------------------------------------
/* SplitJoinUnit.java
 *
 * Unit test for split join scenario:
 *  
 *   Four nodes, alpha, beta, gamma, delta
 *
 *   (1) alpha issues random writes to a, b, c
 *   
 *   (2) beta subscribes for a from alpha
 *       gamma subscribes for b from alpha
 *   
 *   (3) delta subscribes * from beta and gamma
 *       ==> delta reads /a, /c precise
 *           
 *  This class is delta.
 *  see papers/osdi2004/submitted.pdf splitjoin example for more details
 */
//---------------------------------------------------------------------------

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class SplitJoinUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals
  private Process rmiregistry;
  private Process alpha;
  private Process beta;
  private Process gamma;

  private BarrierServer barrier;
  public static final int BARRIER_PORT = 9723;
  
  
  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public SplitJoinUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    if(verbose){
      System.out.println("SplitJoinUnit test...");
    }
    System.gc();
    if(vverbose){
      RandomAccessStateUnit.memoryReport();
    }
    /* 
     * TBD: Insert other fixture code here
     * e.g., this.objectUnderTest = new ClassToBeTested(...);
     */
    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    Thread.sleep(2000);

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    if(verbose){
      System.out.println("rmiregistry started");
    }
    Thread.sleep(2000);

    //
    // Start barrier server
    //
    barrier = new BarrierServer(BARRIER_PORT, 4, BarrierServer.LOOP_FOREVER);
    barrier.start();

    //
    // Set up config files
    //
    makePractiConfig(CONFIG_PATH);

    //
    // Set the connection style
    //
    OutgoingConnectionWorker.setDbgWithFileOutputStream(false);

    //
    // Start helper processes
    //
    alpha = startHelper(ALPHA_ID);
    beta = startHelper(BETA_ID);
    gamma = startHelper(GAMMA_ID);
  }


  /* 
   * Config info
   */
  public static final String CONFIG_PATH = "test" + File.separatorChar 
  + "tmp.SplitJoinUnit.config";
  public static final long ALPHA_ID = 100;
  public static final long BETA_ID = 200;
  public static final long GAMMA_ID = 300;
  public static final long DELTA_ID = 400;
  

  public static final String ALPHA_IS = "/a:/b:/c";
  public static final String BETA_IS = "/a";
  public static final String GAMMA_IS = "/c";
  public static final String DELTA_IS = "/a:/b:/c";
  
  public static final int TOTALWRITES = 1000;

  /*
   * Just make a 4-node config file on localhost in the 
   * specified path
   */
  protected void  makePractiConfig(String path){
    
    Config.createEmptyConfig();
    long NODE_0_ID = ALPHA_ID;
    long NODE_1_ID = BETA_ID;
    long NODE_2_ID = GAMMA_ID;
    long NODE_3_ID = DELTA_ID;
    
    String NODE_0_IP = "localhost";
    String NODE_1_IP = "localhost";
    String NODE_2_IP = "localhost";
    String NODE_3_IP = "localhost";

    String NODE_0_IS = ALPHA_IS;
    String NODE_1_IS = BETA_IS;
    String NODE_2_IS = GAMMA_IS;
    String NODE_3_IS = DELTA_IS;

    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
			    NODE_0_IP,
			    1588,
			    1589,
			    1591,
			    1592,
			    1590,
			    "test" + File.separatorChar + "local-" + 
			    NODE_0_ID + ".db",
			    NODE_0_IS,
			    -1L,
			    NODE_0_IP,
			    1593,
			    1594,
			    -1,
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
			    NODE_1_IP,
			    1688,
			    1689,
			    1691,
			    1692,
			    1690,
			    "test" + File.separatorChar + "local-" + 
			    NODE_1_ID+".db",
			    NODE_1_IS,
			    -1L,
			    NODE_1_IP,
			    1693,
			    1694,
			    -1,
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
  
    Config.addOneNodeConfig(new NodeId(NODE_2_ID),
			    NODE_2_IP,
			    1788,
			    1789,
			    1791,
			    1792,
			    1790,
			    "test" + File.separatorChar + "local-" + 
			    NODE_2_ID+".db",
			    NODE_2_IS,
			    -1L,
			    NODE_2_IP,
			    1793,
			    1794,
			    -1,
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_3_ID),
			    NODE_3_IP,
			    1888,
			    1889,
			    1891,
			    1892,
			    1890,
			    "test" + File.separatorChar + "local-" + 
			    NODE_3_ID+".db",
			    NODE_2_IS,
			    -1L,
			    NODE_3_IP,
			    1893,
			    1894,
			    -1,
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
    Config.writeToFile(path);
  }



  //---------------------------------------------------------------------------
  // Start helper process. Set global writer to refer to helper process.
  //---------------------------------------------------------------------------
  private Process startHelper(long myNodeId){
    Process helper = null;
    //
    // Start the helper PRACTI node
    // "make foo.unit" runs "java foo" with appropriate arguments
    //
    // Use "make" to figure out what arguments needed to
    // set up class paths, etc. Grab the command line
    // rather than just run make b/c we want a handle
    // on the helper process so we can kill it later.
    //
    try{
      helper = Runtime.getRuntime().exec("./runClass.sh code.SplitJoinUnitHelper " + myNodeId);
      if(verbose){
        System.out.println("helper-" + myNodeId + " started");
      }
      
      setUpHelperIO(helper, "helper-" + myNodeId);
    }
    catch(Exception ex){
      ex.printStackTrace();
      assert(false);
    }
    return helper;
  }

  //---------------------------------------------------------------------------
  // Wait for the helper process to create the files
  // I want to read. Spawn "/dev/null" threads to
  // consume itstheir output so it doesn't block.
  //---------------------------------------------------------------------------
  private void setUpHelperIO(Process helper, String tag){
    InputStream is = helper.getInputStream();
    Thread dn = new DvNull(is, "SplitJoinUnitHelper " + tag + " stdout", 
                           verbose, vverbose);
    dn.start();
    is = helper.getErrorStream();
    dn = new DvNull(is, "SplitJoinUnitHelper " + tag + " stderr",
                    verbose, vverbose);
    dn.start();
  }



  //---------------------------------------------------------------------------
  // Kill helper process and kill rmiregistry
  //---------------------------------------------------------------------------
  protected void tearDown() throws Exception{
    if(alpha != null){
      alpha.destroy();
    }
    if(verbose){
      System.out.println("helper alpha terminated");
    }

    if(beta != null){
      beta.destroy();
    }
    if(verbose){
      System.out.println("helper beta terminated");
    }

    if(gamma != null){
      gamma.destroy();
    }

    if(verbose){
      System.out.println("helper gamma terminated");
    }
    rmiregistry.destroy();
    if(verbose){
      System.out.println("rmiregistry terminated");
    }
    
    super.tearDown();
  }






  //--------------------------------------------------------------------------- 
  // Test split join
  //
  // this is for Delta
  // (1) wait for alpha, beta, gamma up and running
  // 
  // (2) wait for beta/gamma subscribe from alpha
  // 
  // (3) wait for alpha write and beta/gamma done
  // 
  // (4) subscribe for /a:/b:/c from beta, gemma
  // 
  // (5) read /a /c-> precise /b-->imprecise
  // 
  // (6) subscribe from myself
  //
  // (7) wait Untill read /b Precise
  //
  //---------------------------------------------------------------------------
  public void testSplitJoin(){
    
    BarrierClient bc = new BarrierClient("127.0.0.1", BARRIER_PORT, 0);
    NodeId myNodeId = new NodeId(DELTA_ID);
    NodeId alphaNodeId = new NodeId(ALPHA_ID);
    NodeId betaNodeId = new NodeId(BETA_ID);
    NodeId gammaNodeId = new NodeId(GAMMA_ID);
    
    AcceptStamp finalStamp = new AcceptStamp(TOTALWRITES-1, alphaNodeId);

    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet(DELTA_IS);
    if(vverbose){
      System.out.println("DELTA ID: " + myNodeId.toString());
    }
    assert(myNodeId.equals(new NodeId(DELTA_ID)));
    URANode node = new URANode(CONFIG_PATH, myNodeId, 
                               Controller.NULL_CONTROLLER, 
			       true,//cleanDB 
			       true);//noSynLog

    // make sure the accumulate time is long enough so that we don't
    // get PInvals when we are expecting ImpInval
    node.getCore().setMaxAccumulateTime(2000); 

    LocalInterface li = node.getLocalInterface();
    RMIClient ci = node.getRMIClientInterface();
    try{
      if(verbose){
        System.out.println("Hello from SplitJoinUnit -- DELTA");
      }

      if(verbose){
        System.out.println("barrier 1 wait for everyone up and runing");
      }

      bc.sendBarrierRequest(0, 0); // 1 all nodes up and running
      if(verbose){
        System.out.println(" 1 All nodes up and running.");
      }

      bc.sendBarrierRequest(0, 0); // 2 beta and gamma subscribed from alpha
      if(verbose){
        System.out.println(" 2 Beta and gamma subscribe from alpha");
      }
      
      
      bc.sendBarrierRequest(0, 0); // 3 beta and gamma subscribed from alpha
      if(verbose){
        System.out.println(" 3 Beta and gamma subscribe from alpha done");
      }

      //
      // me, Delta, subscribes from Beta
      //
      try{
	ci.subscribeInval(betaNodeId, myNodeId, ss, 
			  AcceptVV.makeVVAllNegatives(), false);
      }
      catch(Exception e){
	System.out.println("Exception on subscribe: " + e.toString());
	assert(false);
      }
      
      if(vverbose){
	System.out.println("Delta::subscribe of subscribeId= " + ss
			   + " from Beta done.");
      }


      //
      // me, Delta, subscribes from Gamma
      //
      try{
	ci.subscribeInval(gammaNodeId, myNodeId, ss, 
			  AcceptVV.makeVVAllNegatives(), false);
      }
      catch(Exception e){
	System.out.println("Exception on subscribe: " + e.toString());
	assert(false);
      }
      
      if(vverbose){
	System.out.println("Delta::subscribe of subscribeId= " + ss
			   + " from Gamma done.");
      }
      

      //
      // wait syncCheck read /a|/c precise
      //
      syncCheck(finalStamp, node);
      
      //make sure /a or /c precise
      waitForPreciseInvalid(new ObjId("/a"), li, "DELTA");
      waitForPreciseInvalid(new ObjId("/c"), li, "DELTA");
      
      if(verbose){
	System.out.println(" both /a and /b are precise at Delta");
      }

      try{
	Thread.sleep(500); // Wait before shutup so that 
	//all invalidates applied and the Stream close won't complain
      }
      catch(InterruptedException e){
      }

      bc.sendBarrierRequest(0, 0); // 4 helpers can die now
      if(verbose){
        System.out.println(" 4 helpers can die now.");
      }      
      
      
    } 
    finally{
      //
      // Make sure we release all of the ports so that next
      // unit test can run.
      //
      node.shutdown();
      node = null;
      System.gc();
    }
    
  }

  //---------------------------------------------------------------------------
  // syncCheck
  //---------------------------------------------------------------------------
  public static void syncCheck(AcceptStamp as, URANode unode){
    while(true){
      try{
	unode.getCore().syncCheck(as);
	break;
      }catch(InterruptedException ie){
	//ignore
      }
    } 
  }
  
  //---------------------------------------------------------------------------
  // waitForPreciseInvalid
  //---------------------------------------------------------------------------
  public static void waitForPreciseInvalid(ObjId checkObj, 
					   LocalInterface checkLi, 
					   String tag){
    int count=0;
    int maxtry = 100;
    if(verbose){
      System.out.println(tag + "waitForPreciseInvalid " + checkObj + " start");
    }
    while(true){
      count++;
      
      if(vverbose){
	System.out.println(tag + " waitForPreciseInvalid " + checkObj 
			   + " tried " + count + " times");
      }
      if(count>=maxtry){
	
	fail(tag + " waitForPreciseInvalid("+checkObj 
	     + ") reached maxtry and failed");
      }
      try{
	BodyMsg b = checkLi.read(checkObj, 0, 1, false, false);
	if(b == null){
	  assert(false); // Timeout
	}
	assert false;
      }
      catch(ReadOfInvalidRangeException roire){
	if(checkLi.isPrecise(checkObj)){
	  if(vverbose){
	    System.out.println(tag + " waitForPreciseInvalid " + checkObj
			       + " done.");
	  }
	  return;
	}else{
	  if(vverbose){
	    System.out.println(tag + " waitForPreciseInvalid " 
			       + checkObj + " still imprecise");
	  }
	}
      }
      catch(ReadOfHoleException rohe){
	assert(false);
      }
      catch(EOFException eof){
	assert(false);
      }
      catch(IOException iof){
	assert(false);
      }
      catch(ObjNotFoundException onfe){
	// This is OK -- we have not yet received the inval -- retry
      }
      try{
	Thread.sleep(200); // Wait before retry
      }
      catch(InterruptedException e){
      }
    }
    
  }

  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(SplitJoinUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   SplitJoinUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "SplitJoinUnit";
    System.err.print(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          verbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new SplitJoinUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);

    System.exit(0);
  }

}

//---------------------------------------------------------------------------
/*$Log: SplitJoinUnit.java,v $
/*Revision 1.2  2007/07/04 21:47:48  zjiandan
/*fix unittests
/**/
//---------------------------------------------------------------------------


