package code;

 /** 
 *  SubscribeInvalRequest -- SDIMSController uses this to track which inval 
 *  subscriptions we have out. We share state between the SDIMSController and 
 *  worker thread using these objects, so make methods synchronized. 
 **/ 
import java.lang.reflect.Field;

public class SubscribeInvalRequest{

    private SubscriptionSet ss;
    private VV startVV;
    private NodeId target;  // Node that we are going to ask for this stream
    //TBD:    private List of NodeId writers;

    public final static int STATUS_PENDING = 1197; // A request is outstanding
    public final static int STATUS_FAILED = 1198; // Request has failed or timed out
    public final static int STATUS_CONNECTED = 1199; 
    public final static int STATUS_I_AM_ROOT_OF_SPANNING_TREE = 1200;
    private int status; 
    private int reverseStatus;
    private boolean doPushChannel;
    private int pushChannelStatus;
    private int reversePushChannelStatus;


    private int retryCount;
    public final static int WARNING_RETRY_COUNT = 10;
    private static final long firstRetryMS = 5000;
    private static float retryIncreaseRate = (float)1.2;
    


    /*
     *------------------------------------------------------------------
     *
     * constructor --
     *
     *          Constructor
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    public SubscribeInvalRequest(SubscriptionSet ss, VV startVV, NodeId target,
				 boolean doPushChannel_)
    {
	this.ss = ss;
	this.startVV = startVV;
	this.target = target;
	this.status = STATUS_FAILED;
	this.reverseStatus = STATUS_FAILED;
	retryCount = 0;
	this.doPushChannel = doPushChannel_;
	this.pushChannelStatus = STATUS_FAILED;
	this.reversePushChannelStatus = STATUS_FAILED;


	// TBD: add list of writers we want to subscribe to
    } 


    /*
     *------------------------------------------------------------------
     *
     * equals --
     *
     *          Two requests are equal if all of their fields are equal.
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    public synchronized boolean 
	equals(SubscribeInvalRequest r)
    {
	//
	// Don't forget to update this when you change the definition
	// of the object
	//
	Field f[] = r.getClass().getDeclaredFields();
	assert(f.length == 18);

	//
	// We hold lock on this. Also need to hold lock on 
	// r.
	//
	// Note that recursive locks work in Java, so a == a
	// is true and works properly.
	//
	return r.checkEquals(ss, startVV, target, status, reverseStatus, 
			     doPushChannel, pushChannelStatus, 
			     reversePushChannelStatus,
			     retryCount);
    }

    /*
     *------------------------------------------------------------------
     *
     * checkEquals --
     *
     *          Helper function that checks all non-final fields for equality.
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    private boolean checkEquals(SubscriptionSet ss, VV startVV, NodeId target, 
				int status, int reverseStatus, boolean doPushChannel,
				int pushChannelStatus, int reversePushChannelStatus, 
				int retryCount)
    {
	//
	// Don't forget to update this when you change the definition
	// of the object
	//
	Field f[] = this.getClass().getDeclaredFields();
	if(f.length != 18){
	    System.err.println("Changed fields in SubscribeInvalRequest " + f.length);
	    assert(f.length == 18);
	}

	if(!this.ss.equals(ss)){
	    return false;
	}
	if(!this.startVV.equals(startVV)){
	    return false;
	}
	if(this.target != target){
	    return false;
	}
	if(this.status != status){
	    return false;
	}
	if(this.reverseStatus != reverseStatus){
	    return false;
	}
	if(this.doPushChannel != doPushChannel){
	    return false;
	}
	if(this.pushChannelStatus != pushChannelStatus){
	    return false;
	}
	if(this.reversePushChannelStatus != reversePushChannelStatus){
	    return false;
	}
	if(this.retryCount != retryCount){
	    return false;
	}
	return true;
    }


/*
 *------------------------------------------------------------------
 *
 * toString() --
 *
 *          description.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
    public synchronized String
	toString()
    {
	//
	// Don't forget to update this when you change the definition
	// of the object
	//
	Field f[] = this.getClass().getDeclaredFields();
	assert(f.length == 18);


	StringBuffer b = new StringBuffer();
	b.append("(SubscribeInvalRequest: ss: " + ss.toString());
	b.append(" startVV: " + startVV.toString());
	b.append(" target: " + target.toString());
	b.append(" status: " + statusToString(status));
	b.append("/" + statusToString(reverseStatus));
	b.append(" doPush: " + doPushChannel);
	b.append(" push: " + statusToString(pushChannelStatus));
	b.append("/" + statusToString(reversePushChannelStatus));
	b.append(" retryCount: " + retryCount);
	b.append(")");
	return b.toString();
    }

    private String
    statusToString(int st)
    {
	switch(st){
	case STATUS_PENDING:
	    return "PENDING";
	case STATUS_FAILED:
	    return "FAILED";
	case STATUS_CONNECTED:
	    return "CONNECTED";
	case STATUS_I_AM_ROOT_OF_SPANNING_TREE:
	    return "ROOT";
	default:
	    assert(false);
	    return "-999";
	}
    }

    /*
     *------------------------------------------------------------------
     *
     * hashCode --
     *
     *          If you want to use as a hash key, you need to come up
     *          with a hash function that is compatible with equals
     *          (e.g., one that computes across the fields).
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    public synchronized int hashCode()
    {
	assert(false);// Be sure to update self test if this changes
	return -1; 
    }


    /*
     *------------------------------------------------------------------
     *
     * clone --
     *
     *          Prevent accidental copying of this thing -- we use 
     *          SubscribeInvalRequest to share data between worker thread
     *          and SDIMSController, so make sure we pass references
     *          around not pointers.
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    protected Object clone()
    {
	assert(false);
	return null;
    }


    /*
     *------------------------------------------------------------------
     *
     * GetSubscriptionSet --
     *
     *          Return a reference to the interest set.
     *
     * Arguments:
     *      None.
     *
     * Results:
     *      None.
     *
     * Side effects:
     * 
     *      Returning a pointer to our internal state is dangerous.
     *      What we want to do is call the clone() method. If InterestSet
     *      knows that it is immutable, clone can just return a pointer
     *      to the original, but if that is not known, then clone gives a copy.
     *      Nice and modular, no?
     *
     *------------------------------------------------------------------
     */
    public synchronized SubscriptionSet getSubscriptionSet()
    {
	return (SubscriptionSet)ss.clone();
    }


    /*
     *------------------------------------------------------------------
     *
     * getStartVV --
     *
     *          Return a reference to the startVV
     *
     * Arguments:
     *      None.
     *
     * Results:
     *      None.
     *
     * Side effects:
     * 
     *      Returning a pointer to our internal state is dangerous.
     *      What we want to do is call the clone() method. If VV
     *      knows that it is immutable, clone can just return a pointer
     *      to the original, but if that is not known, then clone gives a copy.
     *      Nice and modular, no?
     *
     *------------------------------------------------------------------
     */
    public synchronized VV getStartVV()
    {
      return(new CounterVV(startVV));
      //return (VV)startVV.clone();
    }


    /*
     *------------------------------------------------------------------
     *
     * reset --
     *
     *          Called when the connection goes down and we need to
     *          re-open the connection. Clear the retry count (we succeeded
     *          before, so we get to start with small timeout), reset
     *          the status, and update startVV.
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    public synchronized void reset(VV vv)
    {
	retryCount = 0;
	status = STATUS_FAILED;
	startVV = vv;
    }







    public synchronized long getTimeout()
    {
	int ii;
	long to = firstRetryMS;
	for(ii = 0; ii < retryCount; ii++){
	    to *= retryIncreaseRate;
	}
	return to;
    }

    public synchronized NodeId getTarget()
    {
	return (NodeId)target.clone(); 
    }

    public synchronized void setTargetAndResetStatus(NodeId t)
    {
	NodeId oldTarget = target;
	target = t;
	//
	// if target changed, reset status
	//
	if(!t.equals(oldTarget)){
	    status = STATUS_PENDING;
	    reverseStatus = STATUS_PENDING;
	    pushChannelStatus = STATUS_PENDING;
	    reversePushChannelStatus = STATUS_PENDING;
	    Env.warn("SubscribeInvalRequest target changed -- do we need to gc the connections?");
	}

    }

    public synchronized void incrementRetryCount()
    {
	retryCount++;
    }

    public synchronized void setPendingToFailed()
    {
	if(status == STATUS_PENDING){
	    status = STATUS_FAILED;
	}
	if(reverseStatus == STATUS_PENDING){
	    reverseStatus = STATUS_FAILED;
	}
	if(pushChannelStatus == STATUS_PENDING){
	    pushChannelStatus = STATUS_FAILED;
	}
	if(reversePushChannelStatus == STATUS_PENDING){
	    reversePushChannelStatus = STATUS_FAILED;
	}
    }


    public synchronized int getRetryCount()
    {
	return retryCount;
    }
    
    public synchronized void setStatus(int newStatus)
    {
	assert(newStatus >= STATUS_PENDING 
	       && newStatus <= STATUS_I_AM_ROOT_OF_SPANNING_TREE);
	status = newStatus;
	return;
    }

    public synchronized int getStatus()
    {
	return status;
    }

    public synchronized void setReverseConnectionStatus(int newStatus)
    {
	assert(newStatus >= STATUS_PENDING 
	       && newStatus <= STATUS_I_AM_ROOT_OF_SPANNING_TREE);
	reverseStatus = newStatus;
	return;
    }

    public synchronized int getReverseConnectionStatus()
    {
	return reverseStatus;
    }

    public synchronized boolean getDoPush()
    {
	return doPushChannel;
    }

    public synchronized long getPushChannelStatus()
    {
	return pushChannelStatus;
    }


    public synchronized long getReversePushChannelStatus()
    {
	return reversePushChannelStatus;
    }

    public synchronized void setPushChannelStatus(int s)
    {
	pushChannelStatus = s;
    }


    public synchronized void setReversePushChannelStatus(int s)
    {
	reversePushChannelStatus = s;
    }

    /*
     *------------------------------------------------------------------
     *
     * main --
     *
     *          Simple unit tests/sanity checks
     *
     * Arguments:
     *      type1 arg1 -- description.
     *
     * Results:
     *      None.
     *
     * Side effects:
     *      None.
     *
     *------------------------------------------------------------------
     */
    public static void main(String s[])
    {
	System.err.print("SubscribeInvalRequest unit test....");

	SubscriptionSet ss1 = SubscriptionSet.makeSubscriptionSet("/foo/bar");
	SubscriptionSet ss2 = SubscriptionSet.makeSubscriptionSet("/foo/flim");
	SubscriptionSet ss3 = SubscriptionSet.makeSubscriptionSet("/foo/bar");
	assert(ss1.equals(ss3));
	assert(ss1.equals(ss1));
	assert(!ss1.equals(ss2));

	NodeId n1 = new NodeId(1);
	NodeId n2 = new NodeId(99);
	NodeId n3 = new NodeId(1);
	NodeId n4 = new NodeId(93);

	AcceptStamp as1 = new AcceptStamp(43, n1);
	AcceptStamp as2 = new AcceptStamp(98, n2);
	AcceptStamp as3 = new AcceptStamp(43, n3);
	AcceptStamp as4 = new AcceptStamp(9900, n4);
	AcceptStamp aa1[] = new AcceptStamp[2];
	aa1[0] = as1;
	aa1[1] = as2;
	AcceptStamp aa2[] = new AcceptStamp[2];
	aa2[0] = as1;
	aa2[1] = as4;
	AcceptStamp aa3[] = new AcceptStamp[2];
	aa3[0] = as3;
	aa3[1] = as2;
    
    
	AcceptVV vv1 = new AcceptVV(aa1);
	AcceptVV vv2 = new AcceptVV(aa2);
	AcceptVV vv3 = new AcceptVV(aa3);
    
	SubscribeInvalRequest sir1 = new SubscribeInvalRequest(ss1, vv1, n1, false);
	SubscribeInvalRequest sir2 = new SubscribeInvalRequest(ss2, vv1, n1, false);
	SubscribeInvalRequest sir3 = new SubscribeInvalRequest(ss3, vv1, n1, false);
	assert(sir1.equals(sir1));
	assert(!sir1.equals(sir2));
	assert(sir3.equals(sir3));
	// No need to test hash code -- calling it asserts false...

	sir1 = new SubscribeInvalRequest(ss1, vv1, n1, false);
	sir2 = new SubscribeInvalRequest(ss1, vv2, n1, false);
	sir3 = new SubscribeInvalRequest(ss3, vv3, n1, false);
	assert(sir1.equals(sir1));
	assert(!sir1.equals(sir2));
	assert(sir3.equals(sir3));

	sir1 = new SubscribeInvalRequest(ss1, vv1, n1, false);
	sir2 = new SubscribeInvalRequest(ss1, vv1, n2, false);
	sir3 = new SubscribeInvalRequest(ss3, vv3, n3, false);
	assert(sir1.equals(sir1));
	assert(!sir1.equals(sir2));
	assert(sir3.equals(sir3));

	System.err.println("...SubscribeInvalRequest unit test succeeds.");
	System.exit(0);
    }


}


//---------------------------------------------------------------------------
/* $Log: SubscribeInvalRequest.java,v $
/* Revision 1.10  2005/03/16 03:45:19  nayate
/* Minor change
/*
/* Revision 1.9  2005/03/01 08:50:30  nayate
/* Modified for new code
/*
/* Revision 1.8  2004/05/20 00:47:35  dahlin
/* Fixed several bugs to make inval log exchange work (biggest one: ISIterator handling case when an single-writer log exists but has no records in it; also added some debugging tools
/*
/* Revision 1.7  2004/05/14 19:23:21  dahlin
/* Cleanup spanning tree logic; add update bodies
/*
   /* Revision 1.6  2004/05/10 20:48:19  dahlin
   /* Clarified RMI exceptions; full version of (stub) DemandReadWorker
   /*
   /* Revision 1.5  2004/05/09 22:21:11  dahlin
   /* Unit test and stub implementation for spanningTreeWorker
   /*
   /* Revision 1.4  2004/05/09 20:09:14  dahlin
   /* Updated unit tests for WorkQueue, TimeoutQueue, TimeoutQueueWorker, DirectoryInterestSet, SubscribeInvalRequest, InvalSpanningList, AcceptStamp, NodeId, and AcceptVV
   /*
   /* Revision 1.3  2004/05/09 14:41:24  dahlin
   /* Fix the ^M formatting
   /*
   /* Revision 1.2  2004/05/09 02:24:16  dahlin
   /* Complete coding and unit testing of InvalSpanningList and SubscribeInvalRequest
   /*
*/
//---------------------------------------------------------------------------
