package code;

 /** 
 *  SubscriptionSet: Represents the set of all data a stream is subscribed 
 *  to listen to 
 **/ 
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;
import java.util.Set;
import java.util.Vector;
import java.io.Externalizable;

import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

public class SubscriptionSet implements Immutable, Cloneable, Externalizable{

 /** 
 *  Constant 
 **/ 
  private static int EMPTY_SS_HASHCODE = 2312;

 /** 
 *  Private data 
 **/ 
  private final String name;
  private final HashMap children;
  private final boolean includesSelf;
  private final boolean includesChildren;

 /** 
 *  empty constructor for serialization 
 **/ 
  public
  SubscriptionSet(){
    this.name = null;
    this.children = new HashMap(0);
    this.includesSelf = false;
    this.includesChildren = false;
  }

 /** 
 *  Constructor 
 **/ 
  private
  SubscriptionSet(String newName,
                  HashMap newChildren,
                  boolean newIncludesSelf,
                  boolean newIncludesChildren){
    this.name = newName;
    this.children = newChildren;
    this.includesSelf = newIncludesSelf;
    this.includesChildren = newIncludesChildren;
  }

 /** 
 *  Constructor (Build an SS from a HierInvalTarget) 
 **/ 
  public
  SubscriptionSet(HierInvalTarget hit){
    HierInvalTarget hitChild = null;
    HashMap newChildren = null;
    SubscriptionSet ssChild = null;
    Collection hitChildrenValues = null;

    if(!hit.isEmpty()){
      // Copy most of the fields as they are
      this.name = hit.getName();
      this.includesSelf = hit.containsSelf();
      this.includesChildren = hit.containsChildren();
      newChildren = new HashMap();

      // Copy over the children (recursively creating new SubscriptionSets)
      hitChildrenValues = hit.getChildren().values();
      for(Iterator i = hitChildrenValues.iterator(); i.hasNext();){
        hitChild = (HierInvalTarget)i.next();
        ssChild = new SubscriptionSet(hitChild);
        newChildren.put(ssChild.getName(), ssChild);
      }
      this.children = newChildren;
    }else{
      // hit was null; make an empty SubscriptionSet
      this.name = null;
      this.children = new HashMap();
      this.includesSelf = false;
      this.includesChildren = false;
    }
  }

 /** 
 *  Return true if this node has no children 
 **/ 
  public boolean
  hasNoChildren(){
    return(this.isEmpty() || this.children.isEmpty());
  }

 /** 
 *  Return true if this node has no children. Return null if no such child. 
 **/ 
  public SubscriptionSet
  getChild(String childName){
    assert(!this.isEmpty());
    return((SubscriptionSet)this.children.get(childName));
  }

 /** 
 *  Clone this object 
 **/ 
  public Object
  clone(){
    return(this);
  }

 /** 
 *  Return a hash code 
 **/ 
  public int
  hashCode(){
    int code = 0;
    SubscriptionSet ssChild = null;

    // Make sure we get an assert failure if we add anything to this class
    // and don't use it when computing a hashCode
    assert(this.getClass().getDeclaredFields().length == 6);
    if(!this.isEmpty()){
      code = (this.name.hashCode() ^
              (new Boolean(this.includesSelf)).hashCode() ^
              (new Boolean(this.includesChildren).hashCode()));
      for(Iterator i = this.children.values().iterator(); i.hasNext();){
        ssChild = (SubscriptionSet)i.next();
        code = code ^ ssChild.hashCode();
      }
    }else{
      code = SubscriptionSet.EMPTY_SS_HASHCODE;
    }
    return(code);
  }

 /** 
 *  Return true if "o" equals this object 
 **/ 
  public boolean
  equals(Object o){
    SubscriptionSet ss = null;
    boolean result = false;
    String childName = null;
    SubscriptionSet ssChild = null;
    SubscriptionSet thisChild = null;

    // Make sure we get an assert failure if we add anything to this class
    // and don't use it when computing a hashCode
    assert(this.getClass().getDeclaredFields().length == 6):
      this.getClass().getDeclaredFields().length;
    if(o instanceof SubscriptionSet){
      ss = (SubscriptionSet)o;
      if(this.isEmpty()){
        result = ss.isEmpty();
      }else if(this.name.equals(ss.name) &&
               (this.includesSelf == ss.includesSelf) &&
               (this.includesChildren == ss.includesChildren) &&
               (this.children.size() == ss.children.size())){
        result = true;
        for(Iterator i = this.children.keySet().iterator();
            i.hasNext() && result;){
          childName = (String)i.next();
          ssChild = ss.getChild(childName);
          thisChild = this.getChild(childName);
          result = (ssChild != null) && (thisChild.equals(ssChild));
        }
      }else{
        result = false;
      }
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Return the name of this SubscriptionSet node 
 **/ 
  public String
  getName(){
    assert(!this.isEmpty());
    return(this.name);
  }

 /** 
 *  Return true if this node contains all children 
 **/ 
  public boolean
  containsChildren(){
    assert(!this.isEmpty());
    return(this.includesChildren);
  }

 /** 
 *  Return true if this node includes itself in the SubscriptionSet 
 **/ 
  public boolean
  containsSelf(){
    assert(!this.isEmpty());
    return(this.includesSelf);
  }

 /** 
 *  Return a Collection view of the children of this node 
 **/ 
  public Collection
  getChildValues(){
    assert(!this.isEmpty());
    return(this.children.values());
  }

 /** 
 *  Take the union of "this" and "ss" (copied from HierInvalTarget.java) 
 **/ 
  public SubscriptionSet
  getCompleteUnion(SubscriptionSet ss){
    SubscriptionSet unionSS = null;
    SubscriptionSet thisChild = null;
    SubscriptionSet ssChild = null;
    SubscriptionSet newChild = null;
    Collection thisChildValues = null;
    Collection ssChildValues = null;
    HashMap newChildren = null;

    assert(this != null);
    assert(ss != null);
    if(this.isEmpty()){
      unionSS = ss;
    }else if(ss.isEmpty()){
      unionSS = this;
    }else{
      assert(this.name.equals(ss.name));
      if(this.children.isEmpty()){
        if(this.includesChildren){
          unionSS = new SubscriptionSet(this.name,
                                        this.children,
                                        this.includesSelf || ss.includesSelf,
                                        true);
        }else{
          assert(this.includesSelf);
          unionSS = new SubscriptionSet(ss.name,
                                        ss.children, 
                                        true,
                                        ss.includesChildren);
        }
      }else if(ss.children.isEmpty()){
        if(ss.includesChildren){
          unionSS = new SubscriptionSet(ss.name,
                                        ss.children,
                                        this.includesSelf || ss.includesSelf,
                                        true);
        }else{
          assert(ss.includesSelf);
          unionSS = new SubscriptionSet(this.name,
                                        this.children,
                                        true,
                                        this.includesChildren);
        }
      }else{
        thisChildValues = this.children.values();
        newChildren = new HashMap();
        for(Iterator i = thisChildValues.iterator(); i.hasNext();){
          thisChild = (SubscriptionSet)i.next();
          ssChild = (SubscriptionSet)ss.getChild(thisChild.name);
          if(ssChild != null){
            newChild = thisChild.getCompleteUnion(ssChild);
          }else{
            newChild = (SubscriptionSet)thisChild.clone();
          }
	  assert newChild instanceof SubscriptionSet:newChild;
          newChildren.put(thisChild.name, newChild);
        }
        ssChildValues = ss.children.values();
        for(Iterator i = ssChildValues.iterator(); i.hasNext();){
          ssChild = (SubscriptionSet)i.next();
          if(!newChildren.containsKey(ssChild.name)){
            newChild = (SubscriptionSet)ssChild.clone();
	    assert newChild instanceof SubscriptionSet:newChild;
            newChildren.put(ssChild.name, newChild);
          }
        }
        unionSS = new SubscriptionSet(this.name,
                                      newChildren,
                                      this.includesSelf || ss.includesSelf,
                                      false);
      }
    }
    return(unionSS);
  }

 /** 
 *  Remove "ss" from "this" 
 *  
 *  Note: Since a SubscriptionSet is represented as a tree of included 
 *  children, the remove operation can only work if the removed 
 *  SubscriptionSet is a substree of the original set. If the user enables 
 *  the strict flag, an exception is thrown when this method is called 
 *  for an unremovable SubscriptionSet; otherwise the method uses a 
 *  best-effort approach to removing a subtree 
 **/ 
  public SubscriptionSet
  remove(SubscriptionSet ss, boolean strict)
    throws IllegalRemoveSubscriptionSetException{
    SubscriptionSet result = null;
    SubscriptionSet thisChild = null;
    SubscriptionSet ssChild = null;
    SubscriptionSet newChild = null;
    Collection thisChildValues = null;
    HashMap newChildren = null;
    boolean newIncludesSelf = false;
    boolean newIncludesChildren = false;
    String errStr = null;

    // name, children, includesSelf, includesChildren
    assert(this != null);
    assert(ss != null);
    if(ss.isEmpty() || this.isEmpty()){
      result = this;
    }else{
      assert(this.name.equals(ss.name));
      newChildren = new HashMap();
      newIncludesSelf = this.includesSelf && (!ss.includesSelf);
      if(ss.hasNoChildren()){
        assert(ss.includesChildren || ss.includesSelf);
        newIncludesChildren = this.includesChildren && (!ss.includesChildren);
        if(!ss.includesChildren){
          thisChildValues = this.children.values();
          for(Iterator i = thisChildValues.iterator(); i.hasNext();){
            thisChild = (SubscriptionSet)i.next();
	    assert thisChild instanceof SubscriptionSet:thisChild;
            newChildren.put(thisChild.name, thisChild.clone());
          }
        }
      }else if(this.hasNoChildren()){
        // ss has children; this does not
        assert(this.includesChildren || this.includesSelf);
        assert(!ss.includesChildren);
        newIncludesChildren = false;
        if(strict && this.includesChildren && (!ss.hasNoChildren())){
          // ss has children but we don't know the further structure
          // of this tree so we don't know how to remove those children.
          // Throw an exception
          errStr = "Cannot remove " + ss + " from " + this;
          throw new IllegalRemoveSubscriptionSetException(errStr);
        }
      }else{
        // this and ss both have children
        newIncludesChildren = false;
        thisChildValues = this.children.values();
        for(Iterator i = thisChildValues.iterator(); i.hasNext();){
          thisChild = (SubscriptionSet)i.next();
          ssChild = (SubscriptionSet)ss.getChild(thisChild.name);
          if(ssChild != null){
            newChild = thisChild.remove(ssChild, strict);
            if(!newChild.isEmpty()){
              assert(newChild.name.equals(thisChild.name));
	      assert newChild instanceof SubscriptionSet :newChild;
              newChildren.put(newChild.name, newChild);
            }
          }else{
            newChild = thisChild.remove(SubscriptionSet.makeEmptySet(),
                                        strict);
            assert(newChild.name.equals(thisChild.name));
	    assert newChild instanceof SubscriptionSet:newChild;
            newChildren.put(newChild.name, newChild);
          }
        }
      }
      if((!newIncludesSelf) &&
         (!newIncludesChildren) &&
         newChildren.isEmpty()){
        // This is an empty node.
        result = SubscriptionSet.makeEmptySet();
      }else{
        result = new SubscriptionSet(this.name,
                                     newChildren,
                                     newIncludesSelf,
                                     newIncludesChildren);
      }
    }
    return(result);
  }

 /** 
 *  Return true if this SubscriptionSet is empty 
 **/ 
  public boolean
  isEmpty(){
    return(this.name == null);
  }

 /** 
 *  Return the intersection of "ss" and "this" 
 **/ 
  public SubscriptionSet
  getIntersection(SubscriptionSet ss){
    SubscriptionSet result = null;
    SubscriptionSet thisChild = null;
    SubscriptionSet ssChild = null;
    SubscriptionSet newChild = null;
    Collection thisChildValues = null;
    Collection ssChildValues = null;
    HashMap newChildren = null;
    boolean newIncludesSelf = false;
    boolean newIncludesChildren = false;

    // name, children, includesSelf, includesChildren
    assert(this != null);
    assert(ss != null);
    if(ss.isEmpty() || this.isEmpty()){
      result = SubscriptionSet.makeEmptySet();
    }else{
      assert(this.name.equals(ss.name));
      newChildren = new HashMap();
      newIncludesSelf = this.includesSelf && ss.includesSelf;
      newIncludesChildren = this.includesChildren && ss.includesChildren;

      if(ss.hasNoChildren()){
        assert(ss.includesChildren || ss.includesSelf);
        if(ss.includesChildren){
          thisChildValues = this.children.values();
          for(Iterator i = thisChildValues.iterator(); i.hasNext();){
            thisChild = (SubscriptionSet)i.next();
	    assert thisChild instanceof SubscriptionSet :thisChild;
            newChildren.put(thisChild.name, thisChild.clone());
          }
        }
      }else if(this.hasNoChildren()){
        // ss has children; this does not
        assert(this.includesChildren || this.includesSelf);
        if(this.includesChildren){
          ssChildValues = ss.children.values();
          for(Iterator i = ssChildValues.iterator(); i.hasNext();){
            ssChild = (SubscriptionSet)i.next();
	    assert ssChild instanceof SubscriptionSet :ssChild;
            newChildren.put(ssChild.name, ssChild.clone());
          }
        }
      }else{
        // this and ss both have children
        thisChildValues = this.children.values();
        for(Iterator i = thisChildValues.iterator(); i.hasNext();){
          thisChild = (SubscriptionSet)i.next();
          ssChild = (SubscriptionSet)ss.getChild(thisChild.name);
          if(ssChild != null){
            newChild = thisChild.getIntersection(ssChild);
            if(!newChild.isEmpty()){
              assert(newChild.name.equals(thisChild.name));
	      assert newChild instanceof SubscriptionSet :newChild;
              newChildren.put(newChild.name, newChild);
            }
          }
        }
      }
      if((!newIncludesSelf) &&
         (!newIncludesChildren) &&
         newChildren.isEmpty()){
        // This is an empty node.
        result = SubscriptionSet.makeEmptySet();
      }else{
        result = new SubscriptionSet(this.name,
                                     newChildren,
                                     newIncludesSelf,
                                     newIncludesChildren);
      }
    }
    return(result);
  }

 /** 
 *  Return the intersection of ss and this 
 *  Note: Copied from above implementation 
 **/ 
  public SubscriptionSet
  getIntersection(HierInvalTarget hit){
    SubscriptionSet result = null;
    SubscriptionSet newChild = null;
    SubscriptionSet thisChild = null;
    HierInvalTarget hitChild = null;
    Collection thisChildValues = null;
    Collection hitChildValues = null;
    HashMap newChildren = null;
    boolean newIncludesSelf = false;
    boolean newIncludesChildren = false;

    // name, children, includesSelf, includesChildren
    assert(this != null);
    assert(hit != null);
    if(hit.isEmpty() || this.isEmpty()){
      result = SubscriptionSet.makeEmptySet();
    }else{
      assert(this.name.equals(hit.getName()));
      newChildren = new HashMap();
      newIncludesSelf = this.includesSelf && hit.containsSelf();
      newIncludesChildren = this.includesChildren && hit.containsChildren();

      if(hit.getChildren().isEmpty()){
        assert(hit.containsChildren() || hit.containsSelf());
        if(hit.containsChildren()){
          thisChildValues = this.children.values();
          for(Iterator i = thisChildValues.iterator(); i.hasNext();){
            thisChild = (SubscriptionSet)i.next();
	    assert thisChild instanceof SubscriptionSet :thisChild;
            newChildren.put(thisChild.name, thisChild.clone());
          }
        }
      }else if(this.hasNoChildren()){
        // hit has children; this does not
        assert(this.includesChildren || this.includesSelf);
        if(this.includesChildren){
          hitChildValues = hit.getChildren().values();
          for(Iterator i = hitChildValues.iterator(); i.hasNext();){
            hitChild = (HierInvalTarget)i.next();
            newChildren.put(hitChild.getName(), new SubscriptionSet(hitChild));
          }
        }
      }else{
        // this and hit both have children
        thisChildValues = this.children.values();
        for(Iterator i = thisChildValues.iterator(); i.hasNext();){
          thisChild = (SubscriptionSet)i.next();
          hitChild = hit.getChild(thisChild.name);
          if(hitChild != null){
            newChild = thisChild.getIntersection(hitChild);
            if(!newChild.isEmpty()){
              assert(newChild.name.equals(thisChild.name));
	      assert newChild instanceof SubscriptionSet :newChild;
              newChildren.put(newChild.name, newChild);
            }
          }
        }
      }
      if((!newIncludesSelf) &&
         (!newIncludesChildren) &&
         newChildren.isEmpty()){
        // This is an empty node.
        result = SubscriptionSet.makeEmptySet();
      }else{
        result = new SubscriptionSet(this.name,
                                     newChildren,
                                     newIncludesSelf,
                                     newIncludesChildren);
      }
    }
    return(result);
  }

 /** 
 *  Return a vector of each path that is a part of this SS tree 
 **/ 
  public Vector
  getFullPaths(){
    Vector vec = null;

    vec = new Vector();
    this.getFullPaths("", vec);
    return(vec);
  }

 /** 
 *  writeExternal 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{
    Vector v = getFullPaths();
    String s;
    if(v.size() == 0){
      s = "";
    }
    else{
      s = internalToString(v);
      assert(!s.equals(""));
    }
    out.writeUTF(s);
  }

 /** 
 *  readExternal 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException, ClassNotFoundException{

    String s = (String)in.readUTF();
    if(s.equals("")){
      return; // de-serializer created empty subscription set to start us off...
    }
    //
    // ss is what we want this to look like
    //
    SubscriptionSet ss = makeSubscriptionSet(s);


    //
    // Need to set "final" fields.
    //
    assert(this.getClass().getDeclaredFields().length == 6);
    Field field[] = new Field[4];
    try{
      field[0] = this.getClass().getDeclaredField("name");
      field[1] = this.getClass().getDeclaredField("children");
      field[2] = this.getClass().getDeclaredField("includesSelf");
      field[3] = this.getClass().getDeclaredField("includesChildren");
      
      AccessibleObject.setAccessible(field, true);

      field[0].set(this, ss.name);
      field[1].set(this, ss.children);
      field[2].setBoolean(this, ss.includesSelf);
      field[3].setBoolean(this, ss.includesChildren);
      
      AccessibleObject.setAccessible(field, false);
    }catch(NoSuchFieldException ne){
      ne.printStackTrace();
      assert false;
    } catch (SecurityException se){
      se.printStackTrace();
      assert false;
    }catch(IllegalArgumentException ie){
      ie.printStackTrace();
      assert false;
    }catch(IllegalAccessException iae){
      iae.printStackTrace();
      assert false;
    }

    ss = null;
  }
      
      


 /** 
 *  convert a non-empty vector to a string -- used for toString and 
 *  also serialization 
 **/ 
  public String internalToString(Vector vec){
    String str = null;
    str = (String)vec.get(0);
    for(int i = 1; i < vec.size(); i++){
      str += (":" + (String)vec.get(i));
    }
    return str;
  }
  
 /** 
 *  Used for testing 
 **/ 
  public String
  toString(){
    String str = null;
    Vector vec = null;

    vec = this.getFullPaths();
    if(vec.isEmpty()){
      str = "(Empty)";
    }else{
      str = internalToString(vec);
    }
    return(str);
  }

 /** 
 *  Make a SubscriptionSet from an array of paths (copied from 
 *  HierInvalTarget) 
 **/ 
  public static SubscriptionSet
  makeSubscriptionSet(String[] allNames){
    String completeName = "";
    SubscriptionSet ss = null;

    if(allNames.length == 0){
      ss = new SubscriptionSet(null, new HashMap(), false, false);
    }else{
      completeName = allNames[0];
      for(int i = 1; i < allNames.length; i++){
        completeName = completeName + ":" + allNames[i];
      }
      ss = SubscriptionSet.makeSubscriptionSet(completeName);
    }
    return(ss);
  }

 /** 
 *  Make a SubscriptionSet from the given ":"-separated set of paths 
 *  (copied from HierInvalTarget) 
 **/ 
  public static SubscriptionSet
  makeSubscriptionSet(String completeName){
    int startIndex = 0;
    int endIndex = 0;
    boolean done = false;
    String fullName = null;
    FileNameTokenizer fnt = null;
    SubscriptionSet ss = null;
    SubscriptionSet newSS = null;

    startIndex = -1;
    endIndex = 0;
    done = false;
    while(!done){
      endIndex = completeName.indexOf(":", startIndex + 1);
      if(endIndex >= 0){
        fullName = completeName.substring(startIndex + 1, endIndex);
      }else{
        fullName = completeName.substring(startIndex + 1);
        done = true;
      }
      startIndex = endIndex;
      //note: if completName doesn't start with "/", then 
      //      the next statement will invoke a system assertion error
      //      
      //System.out.println("fullName = " + fullName);
      fnt = new FileNameTokenizer(fullName);
      if(ss == null){
        ss = SubscriptionSet.makeSubscriptionSet(fnt);
      }else{
        newSS = SubscriptionSet.makeSubscriptionSet(fnt);
        ss = ss.getCompleteUnion(newSS);
      }
    }
    return(ss);
  }

 /** 
 *  Make a SubscriptionSet from an array of paths (copied from 
 *  HierInvalTarget) 
 **/ 
  public static SubscriptionSet
  makeEmptySet(){
    SubscriptionSet emptySet = null;

    emptySet = SubscriptionSet.makeSubscriptionSet(new String[0]);
    return(emptySet);
  }

 /** 
 *  Return true if this SubscriptionSet contains only one subdirectory 
 *  and nothing else (not even intermediate nodes). 
 *  This method is used by SDIMSController. 
 **/ 
  public boolean
  containsExactlyOneSubdirectory(){
    boolean result = false;
    SubscriptionSet ss = null;
    Iterator iter = null;

    if(!this.isEmpty()){
      result = true;
      assert(this.name.equals(""));
      ss = this;
      while((!ss.children.isEmpty()) && result){
        if((ss.children.size() == 1) && (!ss.includesSelf)){
          iter = ss.children.values().iterator();
          ss = (SubscriptionSet)iter.next();
          assert(!iter.hasNext());
        }else{
          // Either "ss" has more than one child or it is
          // included in the SubscriptionSet.
          result = false;
        }
      }
    }else{
      result = false;
    }
    return(result);
  }

 /** 
 *  Return a clone of this SubscriptionSet that ends at the parent node. 
 *  This method is used by SDIMSController. 
 **/ 
  public SubscriptionSet
  getParentSS(){
    SubscriptionSet parentSS = null;
    SubscriptionSet parentSSNode = null;
    Iterator iter = null;
    SubscriptionSet ss = null;

    assert(!this.isEmpty());
    assert(this.name.equals(""));
    assert(this.containsExactlyOneSubdirectory());
    parentSSNode = null;
    ss = this;
    while(!ss.children.isEmpty()){
      assert(ss.children.size() == 1);
      iter = ss.children.values().iterator();
      parentSSNode = ss;
      ss = (SubscriptionSet)iter.next();
      assert(!iter.hasNext());
    }
    if(parentSSNode != null){
      parentSS = this.cloneUntilEndSS(parentSSNode,
                                      ss.includesSelf,
                                      ss.includesChildren);
    }
    return(parentSS);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    //moved to SubscriptionSetUnit.java
  }

 /** 
 *  unit test helper funcitons 
 **/ 
  
 /** 
 *  get children size 
 **/ 
  public int
  getChildrenSize(){
    assert this.children != null;
    return this.children.size();
  }
  
 /** 
 *  Used for testing; helper function called by toString() and getFullPaths() 
 **/ 
  public void
  getFullPaths(String pathPrefix, Vector vec){
    String childStr = null;
    Set childKeys = null;
    String childName = null;
    SubscriptionSet child = null;

    if(!this.isEmpty()){
      if(this.includesSelf){
        vec.add(pathPrefix + this.name);
      }
      if(this.includesChildren){
        assert(this.children.isEmpty());
        vec.add(pathPrefix + this.name + "/*");
      }else{
        childKeys = this.children.keySet();
        for(Iterator i = childKeys.iterator(); i.hasNext();){
          childName = (String)i.next();
	  //System.out.println("childName:"+childName);
          child = (SubscriptionSet)this.getChild(childName);
          assert(child != null);
          child.getFullPaths(pathPrefix + this.name + "/", vec);
        }
      }
    }
  }

  //-------------------------------------------------------------------------
  // Return a new SubscriptionSet for a given path
  // (copied from HierInvalTarget).
  //-------------------------------------------------------------------------
  public static SubscriptionSet
  makeSubscriptionSet(FileNameTokenizer fnt){
    String token = null;
    String nextToken = null;
    SubscriptionSet parentSS = null;
    SubscriptionSet ss = null;
    SubscriptionSet firstSS = null;
    boolean done = false;

    token = fnt.getNextToken();
    assert((token != null) && (token.equals("")));
    parentSS = null;
    done = false;
    while(!done){
      nextToken = fnt.getNextToken();
      if(nextToken == null){
        ss = new SubscriptionSet(token, new HashMap(), true, false);
        done = true;
      }else if(nextToken.equals("*")){
        ss = new SubscriptionSet(token, new HashMap(), false, true);
        assert(fnt.getNextToken() == null);
        done = true;
      }else{
        // Neither token nor nextToken is null
        ss = new SubscriptionSet(token, new HashMap(), false, false);
      }
      if(parentSS != null){
	assert ss instanceof SubscriptionSet;
        parentSS.children.put(token, ss);
      }else{
        firstSS = ss;
      }
      parentSS = ss;
      token = nextToken;
    }
    return(firstSS);
  }

  

  
  

  //-------------------------------------------------------------------------
  // Return a SubscriptionSet that ends at the parent node
  //-------------------------------------------------------------------------
  public SubscriptionSet
  cloneUntilEndSS(SubscriptionSet endSS,
                  boolean addSelf,
                  boolean addChildren){
    HashMap newChildren = null;
    Iterator iter = null;
    SubscriptionSet ssChild = null;
    SubscriptionSet thisClone = null;
    SubscriptionSet ssChildClone = null;

    // Have already asserted containsExactlyOneSubdirectory()
    newChildren = new HashMap();
    if(this != endSS){
      iter = this.children.values().iterator();
      assert(iter.hasNext());
      ssChild = (SubscriptionSet)iter.next();
      assert(!iter.hasNext());
      ssChildClone = ssChild.cloneUntilEndSS(endSS, addSelf, addChildren);
      assert ssChildClone instanceof SubscriptionSet :ssChildClone;
      newChildren.put(ssChildClone.getName(), ssChildClone);
      thisClone = new SubscriptionSet(this.name,
                                      newChildren,
                                      false,
                                      false);
    }else{
      thisClone = new SubscriptionSet(this.name,
                                      newChildren,
                                      addSelf,
                                      addChildren);
    }
    return(thisClone);
  }
}

//---------------------------------------------------------------------------
/* $Log: SubscriptionSet.java,v $
/* Revision 1.22  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/* */
//---------------------------------------------------------------------------
