package code;

/* SyncRequestUnit
 *
 *  Tests if the sync request works for AS and VV
 *
 * (C) 2007 Copyright -- See the file COPYRIGHT for additional details
 */

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class SyncRequestUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = true; // Start/end of test
  protected static boolean vverbose = true; // Test internals

  private Process rmiregistry;

  private NodeId writer1Id = new NodeId(0);
  private NodeId writer2Id = new NodeId(1);
  private NodeId receiverId = new NodeId(2); 

  private RMIClient w1_rmiClient;  //writer1
  private RMIClient w2_rmiClient;  //writer2
  private RMIClient r_rmiClient;  //receiver

  private URANode w1_node; //writer1
  private URANode w2_node; //writer2
  private URANode r_node;  //receiver

  private WaitObj w1_wObj; //writer1
  private WaitObj w2_wObj; //writer2
  private WaitObj r_wObj; //receiver

  private static int SIZE_OF_WRITE = 20;
  private static byte VALUE = 65; //"A";
  private static long WRITES_PER_PHASE = 10;

 /** 
 * Basic Constructor - called by test runners 
 **/ 

  public SyncRequestUnit(final String s){
    super (s);
  }

 /** 
 * Set up environment in which tests must run 
 **/ 
  protected void setUp() throws Exception {
    super.setUp();

    //Env.verifyAssertEnabled();

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    makePractiConfig();

    //Initialize writer1
    w1_rmiClient = new RMIClient();
    Core w1_fakeCore = new Core(w1_rmiClient, false, true, writer1Id, false);
    w1_fakeCore.testSkipRecoverLocalState();
    w1_wObj = new WaitObj();
    Controller w1_controller = new SyncRequestUnitController(w1_fakeCore, w1_wObj);
    w1_node = new URANode(w1_fakeCore, w1_controller, w1_rmiClient);
    if(vverbose) System.out.println("Initialized writer 1");


    //Initialize writer2
    w2_rmiClient = new RMIClient();
    Core w2_fakeCore = new Core(w2_rmiClient, false, true, writer2Id, false);
    w2_fakeCore.testSkipRecoverLocalState();
    w2_wObj = new WaitObj();
    Controller w2_controller = new SyncRequestUnitController(w2_fakeCore, w2_wObj);
    w2_node = new URANode(w2_fakeCore, w2_controller, w2_rmiClient);
    if(vverbose) System.out.println("Initialized writer 2");

    //Initialize receiver
    r_rmiClient = new RMIClient();
    Core r_fakeCore = new Core(r_rmiClient, false, true, receiverId, false);
    r_fakeCore.testSkipRecoverLocalState();
    r_wObj = new WaitObj();
    Controller r_controller = new SyncRequestUnitController(r_fakeCore, r_wObj);
    r_node = new URANode(r_fakeCore, r_controller, r_rmiClient);
    if(vverbose) System.out.println("Initialized reader");

    //set up inval streams
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
    r_rmiClient.subscribeInval(writer1Id, receiverId, ss, AcceptVV.makeVVAllNegatives(), false);
    r_rmiClient.subscribeInval(writer2Id, receiverId, ss, AcceptVV.makeVVAllNegatives(), false);
    if(vverbose) System.out.println("Set up subscriptions");
    Thread.sleep(200);

  }

//---------------------------------------------------------------------------
//Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
    r_rmiClient.removeSubscribeInval(writer1Id, receiverId, ss);
    r_rmiClient.removeSubscribeInval(writer2Id, receiverId, ss);
    Thread.sleep(200);
    w1_node.shutdown();
    w2_node.shutdown();
    r_node.shutdown();
    rmiregistry.destroy();
    if(vverbose) System.out.println("rmiregistry terminated");
    super.tearDown();


  }


  public void test() {
    //
    // Note: we put these two tests together for now because the
    // clean up of the sockets do not work very cleanly leading to 
    // socket used exception
    //
    syncAcceptStampTest();
    syncVVTest();
    //fixed.
    //System.out.println("SyncRequestUnit Succeeded.. any assertion errors can be ignored.  
    //TBD: need to implement clean shutdown to get rid of the errors");

  }


//---------------------------------------------------------------------------
//test syncing of AcceptStamp
//-- writer1 makes several writes 
//-- writer1 waits for receiver sync reply
//-- writer2 makes writes
//-- writer2 waits for receiver to sync
//-- writer 2 writes
//-- writer 1 waits for receiver to sync
//---------------------------------------------------------------------------
  public void syncAcceptStampTest(){
    try{
      LocalInterface w1_li = w1_node.getLocalInterface();
      LocalInterface w2_li = w2_node.getLocalInterface();
      LocalInterface r_li = r_node.getLocalInterface();

      byte[] b = new byte[SIZE_OF_WRITE];
      for(int i = 0; i < SIZE_OF_WRITE; i++){
        b[i] = VALUE;
      }
      ImmutableBytes imb = new ImmutableBytes(b);
      ObjId objId =  new ObjId("/obj");

      //
      // Phase 1 -- writer 1 writes and requests for sync
      //
      if(vverbose) System.out.println("AS Phase 1");
      for(int i=0; i < WRITES_PER_PHASE; i++){
        w1_li.write(objId, 0, SIZE_OF_WRITE, imb, false);
      }

      AcceptStamp as = new AcceptStamp(WRITES_PER_PHASE - 1, writer1Id);
      w1_rmiClient.requestSync(receiverId, writer1Id, as);
      w1_wObj.waitOnObj(); // wait to receive sync reply
      w1_wObj.reset();
      //Check receiver's VV
      if(vverbose) System.out.println("Receiver VV: " + r_node.getCurrentVV());
      assert(r_node.getCurrentVV().getStampByServer(writer1Id) == as.getLocalClock()): "Problem in phase 1 of AS test";
      if(vverbose) System.out.println("done");


      //
      //Phase 2 -- writer 2 writes and request for sync
      //
      if(vverbose) System.out.println("AS Phase 2");
      for(int i=0; i < WRITES_PER_PHASE; i++){
        w2_li.write(objId, 0, SIZE_OF_WRITE, imb, false);
      }

      as = new AcceptStamp(WRITES_PER_PHASE - 1, writer2Id);
      w2_rmiClient.requestSync(receiverId, writer2Id, as);
      w2_wObj.waitOnObj(); // wait to receive sync reply
      w2_wObj.reset();
      //Check receiver's VV
      if(vverbose) System.out.println("Receiver VV: " + r_node.getCurrentVV());
      assert(r_node.getCurrentVV().getStampByServer(writer2Id) == as.getLocalClock()): "Problem in phase 2 of AS test";
      if(vverbose) System.out.println("done");



      //
      //Phase 3-- writer 2 writes and writer1  request for sync
      //
      if(vverbose) System.out.println("AS Phase 3");
      for(int i=0; i < WRITES_PER_PHASE; i++){
        w2_li.write(objId, 0, SIZE_OF_WRITE, imb, false);
      }

      as = new AcceptStamp(WRITES_PER_PHASE*2 - 1, writer2Id);
      w2_rmiClient.requestSync(receiverId, writer1Id, as);
      w1_wObj.waitOnObj(); // wait to receive sync reply
      w1_wObj.reset();
      //Check receiver's VV
      if(vverbose) System.out.println("Receiver VV: " + r_node.getCurrentVV());
      assert(r_node.getCurrentVV().getStampByServer(writer2Id)==as.getLocalClock()): "Problem in phase 3 of AS test";
      if(vverbose) System.out.println("done");

    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    } 

  }



//---------------------------------------------------------------------------
//test syncing of VV
//-- writer1 makes several writes 
//-- writer1 waits for receiver sync reply
//-- writer2 makes writes
//-- writer2 waits for receiver to sync
//-- writer2 writes
//-- writer1 waits for receiver to sync
//---------------------------------------------------------------------------
  public void syncVVTest(){
    try{
      LocalInterface w1_li = w1_node.getLocalInterface();
      LocalInterface w2_li = w2_node.getLocalInterface();
      LocalInterface r_li = r_node.getLocalInterface();

      byte[] b = new byte[SIZE_OF_WRITE];
      for(int i = 0; i < SIZE_OF_WRITE; i++){
        b[i] = VALUE;
      }
      ImmutableBytes imb = new ImmutableBytes(b);
      ObjId objId =  new ObjId("/obj");

      //
      // Note: if the two tests run separately, these values should be -1
      //
      long w1_iniAS = WRITES_PER_PHASE -1;
      long w2_iniAS = WRITES_PER_PHASE*2 -1;
      long r_iniAS = -1;

      AcceptStamp[] asV = new AcceptStamp[3];
      asV[0] = new AcceptStamp(w1_iniAS, writer1Id);
      asV[1] = new AcceptStamp(w2_iniAS, writer2Id);
      asV[2] = new AcceptStamp(r_iniAS, receiverId);


      //
      // Phase 1 -- writer 1 writes and requests for sync
      //
      if(vverbose) System.out.println("VV Phase 1");
      for(int i=0; i < WRITES_PER_PHASE; i++){
        w1_li.write(objId, 0, SIZE_OF_WRITE, imb, false);
      }

      asV[0] = new AcceptStamp(WRITES_PER_PHASE + w1_iniAS, writer1Id);
      AcceptVV vv = new AcceptVV(asV);
      w1_rmiClient.requestSync(receiverId, writer1Id, vv);
      w1_wObj.waitOnObj(); // wait to receive sync reply
      w1_wObj.reset();
      //Check receiver's VV
      if(vverbose) System.out.println("Receiver VV: " + r_node.getCurrentVV());
      assert(r_node.getCurrentVV().equals(vv)): "Problem in phase 1 of VV Test";
      if(vverbose) System.out.println("done");


      //
      //Phase 2 -- writer 2 writes and request for sync
      //
      if(vverbose) System.out.println("VV Phase 2");
      for(int i=0; i < WRITES_PER_PHASE; i++){
        w2_li.write(objId, 0, SIZE_OF_WRITE, imb, false);
      }

      asV[1] = new AcceptStamp(WRITES_PER_PHASE + w2_iniAS, writer2Id);
      vv = new AcceptVV(asV);
      w2_rmiClient.requestSync(receiverId, writer2Id, vv);
      w2_wObj.waitOnObj(); // wait to receive sync reply
      w2_wObj.reset();
      //Check receiver's VV
      if(vverbose) System.out.println("Receiver VV: " + r_node.getCurrentVV());
      assert(r_node.getCurrentVV().equals(vv)): "Problem in phase 2 of VV Test";
      if(vverbose) System.out.println("done");



      //
      //Phase 3-- writer 2 writes and writer1  request for sync
      //
      if(vverbose) System.out.println("VV Phase 3");
      for(int i=0; i < WRITES_PER_PHASE; i++){
        w2_li.write(objId, 0, SIZE_OF_WRITE, imb, false);
      }

      asV[1] = new AcceptStamp(WRITES_PER_PHASE*2 + w2_iniAS, writer2Id);
      vv = new AcceptVV(asV);
      w2_rmiClient.requestSync(receiverId, writer1Id, vv);
      w1_wObj.waitOnObj(); // wait to receive sync reply
      w1_wObj.reset();
      //Check receiver's VV
      if(vverbose) System.out.println("Receiver VV: " + r_node.getCurrentVV());
      assert(r_node.getCurrentVV().equals(vv)): "Problem in phase 3 of VV Test";
      if(vverbose) System.out.println("done");

    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    } 

  }



//---------------------------------------------------------------------------
//Makes the config file for the experiment
//---------------------------------------------------------------------------
  private static void makePractiConfig(){
    Config.createEmptyConfig();
    Config.readKeys();
    Config.addOneNodeConfig(new NodeId(0),
        "localhost",
        4588,
        4589,
        4591,
        4592,
        4590,
        "synctest-node0.db",
        "/*",
        -1L,
        "localhost",
        4593,
        4594,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(1),
        "localhost",
        4688,
        4689,
        4691,
        4692,
        4690,
        "synctest-node1.db",
        "/*",
        -1L,
        "localhost",
        4693,
        4694,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(2),
        "localhost",
        4388,
        4389,
        4391,
        4392,
        4390,
        "synctest-node2.db",
        "/*",
        -1L,
        "localhost",
        4393,
        4394,
        -1,
        Config.CACHE_SIZE_BYTES_DEFAULT,
        Config.MAX_LOG_DISK_SIZE_BYTES,
        Config.MAX_LOG_MEM_SIZE_BYTES);

  }

  public static Test suite(){
    TestSuite suite = new TestSuite(SyncRequestUnit.class);
    return suite;
  }

  public static void main(String s[]) {
    String name = "SyncRequestUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new SyncRequestUnit("test" + s[ii]));
        }

      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    System.exit(0);
  }


//---------------------------------------------------------------------------
//Internal Class -- SyncRequestUnitController
//Overloads the receivedSyncReply method so notify blocked threads
//---------------------------------------------------------------------------
  public class SyncRequestUnitController extends LocalController {

    private WaitObj wObj;
    private boolean dbg = false;

    public SyncRequestUnitController(Core core, WaitObj wObj){
      super(core);
      this.wObj = wObj;
    }

    public void recvSyncReply(AcceptStamp as, NodeId sender) {
      if( dbg ){
        Env.dprintln(dbg, "SyncRequestUnitController: received Sync reply from"
            + sender + " with as = " + as);
      }
      wObj.notifyThreads();
    }
  }
//---------------------------------------------------------------------------
//Internal Class -- SyncThread
//Thread which waits for a sync Reply
//---------------------------------------------------------------------------

  public class SyncThread extends Thread{
    WaitObj  wObj;

    public SyncThread(WaitObj wObj){
      this.wObj = wObj;
    }

    public void run() {
      try{
        wObj.waitOnObj();
        wObj.reset();
      }catch(Exception e) {
        e.printStackTrace();
      }
    }
  }
}
//---------------------------------------------------------------------------
/* $Log: SyncRequestUnit.java,v $
/* Revision 1.7  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.6  2007/06/04 21:41:00  zjiandan
/* expose stream catchup type CP|LOG option to rmiClient.subscribeInval().
/*
/* Revision 1.5  2007/04/02 21:11:39  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.4  2007/01/08 22:17:09  nalini
/* unit test bugfix
/*
/* Revision 1.3  2007/01/07 03:03:48  nalini
/* getting unit tests to work
/*
/* Revision 1.2  2007/01/06 02:03:24  nalini
/* minor junit  bug fixes
/*
/* Revision 1.1  2007/01/05 01:19:54  nalini
/* support for sync with VV added
/*
 */
//---------------------------------------------------------------------------

