package code;

 /** 
 *  Implementation of a worker thread that handles receiving a stream 
 *  of sync requests 
 **/ 
import java.net.Socket;
import java.io.IOException;
import java.io.ObjectInputStream;

public class SyncRplyRecvWorker extends Thread implements SocketMagicConstant
{
  private TaggedInputStream s;
  private Core core;
  private Controller controller;
  private Socket underlyingSocket;

  private static boolean warnedTBD = false;

 /** 
 *  Constructor 
 **/ 
  public SyncRplyRecvWorker(TaggedInputStream s,
                            Core core, 
                            Controller controller,
                            Socket underlyingSocket)
    {
      this.s = s;
      this.core = core;
      this.controller = controller;
      this.underlyingSocket = underlyingSocket;

      if(!warnedTBD){
        Env.warn("TBD: Sync requests should include in their body"
                 + " the name of the node that is to provide "
                 + " the sync (not just the node that is requesting"
                 + " the sync. And replies should also include "
                 + " the node that generated the sync reply (rather"
                 + " than requiring the receiver to get this info"
                 + " from the stream. This is in keeping with"
                 + " our notion that all messages should be"
                 + " self-describing.");
        warnedTBD = true;
      }
    }

 /** 
 *  Main thread method 
 **/ 
  public void run()
    {
      NodeId senderNodeId = null;

      //
      // Read sender nodeId from stream.
      //
      try{
        Long magic = (Long)s.readTaggedObject();
        if((magic.longValue() != INVAL_SOCKET_MAGIC) &&
           (magic.longValue() != INVAL_SOCKET_RECONNECT_MAGIC)){
          Env.remoteAssert(false, "descriptive error msg");
        }
        senderNodeId = (NodeId)s.readTaggedObject();
      }catch(Exception e){
        // 
        // Attempt to connect failed and we don't even know
        // who it was that was trying. Nothing to do but
        // have this thread die. If this was important,
        // then a controller should timeout and retry. 
        //
        this.closeStreamAndSocket(null,
                                  "SyncRplyRecvWorker cannot handshake " +
                                  "with sender " + e);
        return; // Thread exit.
      }
      controller.informSyncRplyStreamInitiated(senderNodeId);

      while(true){
        try{
          AcceptStamp msg = (AcceptStamp)s.readTaggedObject();
          controller.recvSyncReply(msg, senderNodeId);
        }catch(IOException e){
          this.closeStreamAndSocket(senderNodeId,
                                    "SyncRplyRecvWorker interrupted " +
                                    "while receiving data " + e);
          return;
        }catch(ClassNotFoundException e){
          this.closeStreamAndSocket(senderNodeId,
                                    "SyncRplyRecvWorker interrupted " +
                                    "while receiving data " + e);
          return;
        }
      }
    }

 /** 
 *  Cleanly close the stream and socket 
 **/ 
  private void closeStreamAndSocket(NodeId node, String msg)
    {
      try{
        s.close();
        underlyingSocket.close();
      }
      catch(Exception e){
        // Ignore errors when trying to close
      }
      if(node != null){
        controller.informSyncRplyStreamTerminated(node);
      }
      Env.event("SyncRplyRecvWorker closing stream: " + msg);
    }
}

//---------------------------------------------------------------------------
/* $Log: SyncRplyRecvWorker.java,v $
/* Revision 1.9  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.8  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.7  2005/01/18 22:49:43  zjiandan
/* Rewrited Class Serialization for network deliveringto reduce the overall bandwidth.
/*
/* Revision 1.6  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.5  2004/05/20 15:51:45  dahlin
/* Added TBD to make sync requests and replies self-describing
/*
/* Revision 1.4  2004/05/13 21:58:14  dahlin
/* simple unbind implemented
/*
/* Revision 1.3  2004/04/26 20:04:24  nayate
/* Finished but didn't test classes that receive invals, updates, and
/* sync messages.
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
