package code;

 /** 
 *  Sorted queue of timeouts 
 **/ 
import java.util.TreeMap;
import java.util.Comparator;
import java.util.Iterator;

public class TimeoutQueue
{

private long sequence;
private TreeMapValueSearch pending;

public TimeoutQueue()
{
    pending = new TreeMapValueSearch(new TimeKey(-1, -1));
    sequence = 0;
}


/*
 *------------------------------------------------------------------
 *
 * Insert --
 *
 *          Insert an event into the sorted list of events.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void
insert(long delay, Object event)
{
    long wakeupTimeMS = System.currentTimeMillis() + delay;
    TimeKey key = new TimeKey(wakeupTimeMS, sequence);
    sequence++;
    pending.put(key, event);
    notify();
    return;
}

/*
 *------------------------------------------------------------------
 *
 * getNext --
 *
 *          Block until the next timeout's time arrives. Then *remove*
 *          and return the object associated with the time.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized Object
getNext()
{
    long delay;
    delay = getDelay();
    while(delay > 0){
	try{
	    wait(delay); // Wait for timeout or signal
	}
	catch(InterruptedException e){
	}
        delay = getDelay();
    }
    assert(pending.size() > 0);
    TimeKey k = (TimeKey)pending.firstKey();
    Object ret = pending.remove(k);
    return ret;
}


/*
 *------------------------------------------------------------------
 *
 * getDelay --
 *
 *          Return the delay that we should sleep until the
 *          first enqueued timeout occurs or return a really
 *          large number if no timeouts are pending.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private long
getDelay()
{
    if(pending.size() == 0){
	return Long.MAX_VALUE;
    }

    long now = System.currentTimeMillis();
    TimeKey k = (TimeKey)pending.firstKey();
    long when = k.getTimeMS();
    return when - now;
}


/*
 *------------------------------------------------------------------
 *
 * find --
 *          iterates thro' the objects in the queue and  locates
 *          first the one which matches the input
 *
 * Arguments:
 *      Object obj -- object to match against
 *
 * Results:
 *      returns the Object in the queue
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */

public synchronized Object 
find(Object obj) {
  return pending.findByValue(obj);
}



/*
 *------------------------------------------------------------------
 *
 * remove --
 *          iterates thro' the objects in the pending and removes
 *          the first one which matches the input
 *
 * Arguments:
 *      Object obj -- object to match against
 *
 * Results:
 *      Returns the removed object
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */

public synchronized Object 
remove(Object obj) {
  Object pendingObj = pending.removeByValue(obj);
  return pendingObj;
}


public String toString() {
  //dangerous to use
  return pending.toString();
}
/*
 *------------------------------------------------------------------
 *
 * main --
 *
 *          Simple unit test.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public static void main(String s[])
{
    TimeoutQueue q = new TimeoutQueue();

    long delay;
    Long item;

    System.out.print("TimeoutQueue self test...");
    long startMS = System.currentTimeMillis();

    delay = 1000;
    item = new Long(1);
    q.insert(delay, item);

    delay = 3000;
    item = new Long(3);
    q.insert(delay, item);

    delay = 5000;
    item = new Long(5);
    q.insert(delay, item);

    delay = 7000;
    item = new Long(7);
    q.insert(delay, item);

    delay = 9000;
    item = new Long(9);
    q.insert(delay, item);


    delay = 2000;
    item = new Long(2);
    q.insert(delay, item);

    delay = 4000;
    item = new Long(4);
    q.insert(delay, item);

    delay = 6000;
    item = new Long(6);
    q.insert(delay, item);

    delay = 8000;
    item = new Long(8);
    q.insert(delay, item);

    delay = 10000;
    item = new Long(10);
    q.insert(delay, item);
    delay = 10000;
    item = new Long(11);
    q.insert(delay, item);
    item = new Long(12);
    q.insert(delay, item);
    item = new Long(13);
    q.insert(delay, item);
    item = new Long(14);
    q.insert(delay, item);

    int ii;
    for(ii = 1; ii < 15; ii++){
	item = (Long)q.getNext();
	System.out.print(item + " ");
	if(item.intValue() != ii){
	    System.err.println("Possible error in TimeoutQueue unit test"
			       + " (or we got desheduled for a *really* long time)");
	    System.exit(-1);
	}
    }
    long doneMS = System.currentTimeMillis();
    if(doneMS - startMS < 10000){
	System.err.println("ERROR in TimeoutQueue self test -- didn't wait long enough");
	System.exit(-1);
    }
    if(doneMS - startMS > 13000){
	System.err.println("Possible error in TimeoutQueue self test"
			   + " -- waited longer than expected (or we got"
			   + " descheduled for a *really* long time)");
	System.exit(-1);
    }

    //Testing finding and removal of queue items.
    delay = 1000;
    item = new Long(15);
    delay = 1100;
    q.insert(delay, item);
    item = new Long(16);
    delay = 1200;
    q.insert(delay, item);
    item = new Long(17);
    delay = 1300;
    q.insert(delay, item);
    item = new Long(18);
    delay = 1400;
    q.insert(delay, item);
    item = new Long(19);
    delay = 1500;
    q.insert(delay, item);
    
    if(q.find(new Long(18)) == null){
      System.err.println("ERROR in TimeoutQueue self test -- "
                         + " could not find element");
      System.exit(-1);
    }
    if(q.find(new Long(15)) == null){
      System.err.println("ERROR in TimeoutQueue self test -- "
                         + " could not find element");
      System.exit(-1);
    }

    q.remove(new Long(17));
    if(q.find(new Long(17)) != null){
      System.err.println("ERROR in TimeoutQueue self test -- "
                         + " did not remove correctly");
      System.exit(-1);
    }

    item = (Long) q.getNext();
    System.out.print(item+ " ");
    item = (Long) q.getNext();
    System.out.print(item+ " ");
    item = (Long) q.getNext();
    System.out.print(item+ " ");
    if(item.longValue() != 18){
      System.err.println("Error in TimeoutQueue self test -- "
                         + "did not remove correctly");
      System.exit(-1);
    }


    item = (Long) q.getNext();
    System.out.print(item+ " ");


    System.err.println("TimeoutQueue self test succeeds");
    System.exit(0);
}


}







 /** 
 *  $Log: TimeoutQueue.java,v $ 
 *  Revision 1.6  2007/04/02 21:11:39  zjiandan 
 *  snapshort for sosp2007. 
 *  
 *  Revision 1.5  2007/03/08 21:41:17  nalini 
 *  total revamp of P2Runtime, update subscriptions removed, retry logic changed 
 *  
 *  Revision 1.4  2006/09/12 22:18:05  dahlin 
 *  Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests. 
 *  
 *  Revision 1.3  2006/04/20 03:52:53  zjiandan 
 *  Callbacks merged with runTime. 
 *  
 *  Revision 1.2  2004/05/10 20:54:04  dahlin 
 *  Fixed ^M formatting 
 *  
 *  Revision 1.1  2004/05/08 22:20:17  dahlin 
 *  Partially complete version of SDIMSController (it should compile w/o error, though) 
 *  
 **/ 
