package code;

/* TwoFileObjectStore.java
 * 
 * Store an object as two files, so that
 * at any given time one of them is guaranteed to be uncorrupted
 *
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */



import java.io.OutputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.StreamCorruptedException;
import java.io.IOException;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.util.Vector;

public class TwoFileObjectStore{

private boolean newerNotCorrupt;
private long versionNumber;
private String newerFileName;
private String olderFileName;
private String fileName_1;
private String fileName_2;
private static final String testFileName = "test/TwoFileObjectStore-testfile";
private static final String TESTPATH = "test/test_TwoFileObjectStore/";

//private static final String testFileName = "/var/local/TwoFileObjectStore-testfile";
//private static final String TESTPATH = "/var/local/test_TwoFileObjectStore/";
public static final String RESERVED = "---!@#$TFOS%^&*---";
private static final String suffix1 = RESERVED + "1.txt";
private static final String suffix2 = RESERVED + "2.txt";
private static final long DELETED_FLAG = -999;
private static final String DELETED_FILE = "BOGUS_TwoFileObjectStore_NAME";
private boolean pathOK;


//
// Number of file open/closes to allow between explicit gc calls.
// Zero for no explicit gc calls.
//
private static final int gcFrequency = 0; 
private static int gcCounter = 0;

 /** 
 *  Constructor: Build the object.
 
 **/ 
public TwoFileObjectStore(String fileName)
throws ReservedPathNameException{
  if(fileName.indexOf(RESERVED) != -1){
    throw new ReservedPathNameException("Files stored in TwoFileObjectStore can"
                                        + " not contain the string \""
                                        + RESERVED + "\"");
  }
  File file1 = null;
  File file2 = null;
  FileInputStream file1Stream = null;
  FileInputStream file2Stream = null;
  fileName_1 = (new File(fileName + suffix1)).getAbsolutePath();
  fileName_2 = (new File(fileName + suffix2)).getAbsolutePath();
  
  this.newerNotCorrupt = true;
  this.newerFileName = null;
  this.olderFileName = null;
  this.versionNumber = 0;
  this.pathOK = false; 
  
  this.newerFileName = this.getNewerFileName();
  this.versionNumber = this.getVersionNumber(this.newerFileName);
  this.olderFileName = ((this.newerFileName.equals(this.fileName_1)) ?
                        this.fileName_2 : this.fileName_1);
}
  
 /** 
 *  Return the name of the main file (for debugging puposes)
 
 **/ 
public String getFileName(){
  assert(versionNumber != DELETED_FLAG);
  return(this.fileName_1.substring(0,fileName_1.length()-4));
}

 /** 
 *  Read the object from the file.
 
 **/ 
public synchronized Object readObject()
  throws StreamCorruptedException,
  FileNotFoundException,
  IOException,
  ClassNotFoundException{
  assert(versionNumber != DELETED_FLAG);
  Object retObject = null;
  this.newerNotCorrupt = true;
  try{
    retObject = this.readObjectFromFile(this.newerFileName);
    this.newerNotCorrupt = true;
    return(retObject);
  }catch(StreamCorruptedException e){
    throw(new StreamCorruptedException("TwoFileObjectStore:" +
                                       " Newer file corrupted"));
  }catch(EOFException e){
    throw(new EOFException("TwoFileObjectStore: Newer file EOF"));
  }catch(FileNotFoundException e){
    throw(new FileNotFoundException("TwoFileObjectStore: " + e));
  }catch(IOException e){
    throw(new IOException("TwoFileObjectStore: " + e));
  }
}

 /** 
 *  Read the object aggressively; i.e., try to recover the
 
 *  file from the older copy if the newer one is corrupt.
 
 *  Note that the new one should not be corrupt.
 
 **/ 
public synchronized Object readObjectAggressively()
  throws StreamCorruptedException,
  FileNotFoundException,
  IOException,
  ClassNotFoundException{
  assert(versionNumber != DELETED_FLAG);
  boolean deleteSucceeded = false;
  Object retObject = null;
	
  this.newerNotCorrupt = true;
  try{
    retObject = this.readObjectFromFile(this.newerFileName);
    this.newerNotCorrupt = true;
    return(retObject);
  }catch(StreamCorruptedException e){
    // The newer file was corrupted.
    this.newerNotCorrupt = false;
    try{
      return(this.readObjectFromFile(this.olderFileName));
    }catch(StreamCorruptedException e1){
      // The older file was also corrupted, so we give up
      throw(new StreamCorruptedException("TwoFileObjectStore:" +
                                         " Both files corrupted"));
    }catch(EOFException e1){
      // The older file was empty, so again we give up
      throw(new EOFException("TwoFileObjectStore:" +
                             " Valid file contained no data"));
    }
  }catch(EOFException e){
    // The newer file did not have any data, so check the older one.
    try{
      this.newerNotCorrupt = false;
      return(readObjectFromFile(this.olderFileName));
    }catch(StreamCorruptedException e1){
      // The older file was corrupted, the new had no data
      this.newerNotCorrupt = false;
      throw(new EOFException("TwoFileObjectStore: Valid file" +
                             " contained no data"));
    }catch(EOFException e1){
      // Both files contained no data
      this.newerNotCorrupt = true;
      throw(new EOFException("TwoFileObjectStore: Neither file" +
                             " contained data"));
    }
  }
}

 /** 
 *  Write the object
 
 **/ 
public synchronized void writeObject(Object obj) throws IOException{
  assert(versionNumber != DELETED_FLAG);
  // Invariant: The newer file should not corrupted.
  assert obj != null : "TwoFileObjectStore: writeObject: null passed in.";
  assert this.newerNotCorrupt :
    "TwoFileObjectStore: writeObject: Invariant Condition does not hold.";
  ObjectOutputStream objectOS = null;
  FileOutputStream fileOS = null;
  String str = null;

  if(!pathOK){
    //
    // We've not used this path yet, so ensure that
    // parent directories exist.
    //
    createAncestorPath(this.olderFileName);
    pathOK = true;
  }

  try{
    fileOS = new FileOutputStream(this.olderFileName, false);
    // Swap the two file names
    str = this.newerFileName;
    this.newerFileName = this.olderFileName;
    this.olderFileName = str;

    objectOS = new ObjectOutputStream(fileOS);
	    
    this.versionNumber++;
    objectOS.writeLong(this.versionNumber);
    objectOS.writeObject(obj);
    objectOS.flush();
    FileDescriptor fd = fileOS.getFD();
    fd.sync();
  }catch(FileNotFoundException e){
    // One or both of the files we are trying to write to cannot be
    // created either because they already exists as directories or
    // for another file-system related reason (permissions, etc.).
    assert false : "TwoFileObjectStore: writeObject: " + getFileName() +
      " " + e;
  }finally{
    if(objectOS != null){
      try{
        objectOS.close();
      }catch(IOException e){
        // Do nothing
      }
      objectOS = null;
    }
    if(fileOS != null){
      try{
        fileOS.close();
      }catch(IOException e){
        // Do nothing
      }
      fileOS = null;
    }
    gcIfNeeded();
  }
}


 /** 
 *  Delete the underlying files and clear all fields.
 
 **/ 
public synchronized void delete(){
  if(versionNumber == DELETED_FLAG){
    return;
  }

  File f1 = new File(newerFileName);
  f1.delete();
  File f2 = new File(olderFileName);
  f2.delete();

  this.newerNotCorrupt = false;
  this.versionNumber = DELETED_FLAG;
  this.newerFileName = DELETED_FILE;
  this.olderFileName = DELETED_FILE;
  this.fileName_1 = DELETED_FILE;
  this.fileName_2 = DELETED_FILE;
      
}


 /** 
 *  Return the most updated of the files.  If one of them is
 
 *  corrupted, return the non-corrupted one.
 
 **/ 
private synchronized String getNewerFileName(){
  String newerFileName = null;
  File file1 = null;
  File file2 = null;
  long version = 0L;
	
  file1 = new File(this.fileName_1);
  file2 = new File(this.fileName_2);
	
  // NOTE: lastModified returns 0 for a non-existent file
  if(file1.lastModified() > file2.lastModified()){
    // Either file2 does not exist or file1 is newer
    if(this.fileNotCorrupted(this.fileName_1)){
      newerFileName = this.fileName_1;
    }else{
      newerFileName = this.fileName_2;
    }
  }else if(file2.lastModified() > file1.lastModified()){
    // Either file1 does not exist or file2 is newer
    if(this.fileNotCorrupted(this.fileName_2)){
      newerFileName = this.fileName_2;
    }else{
      newerFileName = this.fileName_1;
    }
  }else{
    // Either neither exists or both have the same timestamp
    version = getVersionNumber(this.fileName_1);
    if(version >= 0){
      if(version >= getVersionNumber(this.fileName_2)){
        // File 1 had a higher version or file 2 is corrupt
        newerFileName = this.fileName_1;
      }else{
        // File 2 had a higher version or file 1 is corrupt
        newerFileName = this.fileName_2;
      }
    }else{
      newerFileName = this.fileName_2;
    }
  }
  file1 = null;
  file2 = null;
  return(newerFileName);
}

 /** 
 *  Return the version number from the current file
 
 *  On return values:
 
 *  -1: File was not found
 
 *  -2: Stream corrupted
 
 *  -3: IOException
 
 *  Otherwise, return the version number
 
 **/ 
private synchronized long getVersionNumber(String fileName){
  File file = null;
  long versionNumber = 0L;
  FileInputStream fileIS = null;
  ObjectInputStream objectIS = null;

  try{
    fileIS = new FileInputStream(fileName);
    objectIS = new ObjectInputStream(fileIS);
    versionNumber = objectIS.readLong();
  }catch(FileNotFoundException e){
    versionNumber = -1L;
  }catch(StreamCorruptedException e){
    versionNumber = -2L;
  }catch(IOException e){
    versionNumber = -3L;
  }finally{
    if(objectIS != null){
      try{
        objectIS.close();
      }catch(IOException e){
        // Do nothing
      }
      objectIS = null;
    }

    if(fileIS != null){
      try{
        fileIS.close();
      }catch(IOException e){
        // Do nothing
      }
      fileIS = null;
    }

    gcIfNeeded(); // Free up file and stream resources
  }
  return(versionNumber);
}

 /** 
 *  Return true if the file:
 
 *  1) exists,
 
 *  2) is in the correct format, and
 
 *  3) is readable
 
 **/ 
private synchronized boolean fileNotCorrupted(String fileName){
  long version = 0;

  version = this.getVersionNumber(fileName);
  if(version < 0){
    return false;
  }
  try{
    Object o = readObjectFromFile(fileName);
  }
  catch(FileNotFoundException fnf){
    assert(false); // We just read the version number and hold the lock!
  }
  catch(StreamCorruptedException e){
    return false;
  }
  catch(IOException f){
    return false;
  }
  catch(ClassNotFoundException cnf){
    return false; 
  }
  return true;
}

 /** 
 *  Read an object from a file; used by readObject()
 
 **/ 
private synchronized Object readObjectFromFile(String fileName)
throws StreamCorruptedException,
	       FileNotFoundException,
	       IOException,
	       ClassNotFoundException{
  ObjectInputStream objectIS = null;
  Object object = null;
  FileInputStream fileIS = null;
  long version = 0;

  try{
    fileIS = new FileInputStream(fileName);
    objectIS = new ObjectInputStream(fileIS);
    version = objectIS.readLong();
    object = objectIS.readObject();
    return(object);
  }finally{
    if(objectIS != null){
      try{
        objectIS.close();
      }catch(IOException e){
        // Do nothing
      }
      objectIS = null;
    }
    if(fileIS != null){
      try{
        fileIS.close();
      }catch(IOException e){
        // Do nothing
      }
      fileIS = null;
    }
    gcIfNeeded();    // Force a garbage collection to free up resources
  }
}


 /** 
 *  createAncestorPath() -- make file creation/writes automatic
 
 *  by creating ancestor path if needed
 
 **/ 
private static void 
createAncestorPath(String path)
throws IOException{
  boolean success;

  File full = new File(path);
  File parent = full.getParentFile();
  if(!parent.exists()){
    success = parent.mkdirs();
    if(!success){
      throw new IOException("Cannot create ancestor paths for " + path);
    }
  }
}


 /** 
 *  gcIfNeeded() -- call System.gc() once per every <gcFrequency> 
 
 *         open/close calls
 
 * 
 
 *  This call is here as a hack to work around a garbage collection
 
 *  bug in some jvms (e.g., sun's JDK 1.2 environment). The
 
 *  problem is that file descriptors don't seem to get garbage
 
 *  collected on close(), so if a program opens and closes lots
 
 *  of files, you run out of file descriptors (this problem
 
 *  manifests itself as an OutOfMemoryError).
 
 **/ 
private static synchronized void
gcIfNeeded(){
  assert(gcFrequency >= 0);
  if(gcFrequency == 0){
    return;
  }
  gcCounter++;
  if(gcCounter == gcFrequency){
    gcCounter = 0;
    System.gc();
  }
}




 /** 
 *  For testing purposes; corrupt the file.
 
 **/ 
private static void corruptFile(String fileName) throws IOException{
  FileOutputStream fileOS = null;
  try{
    fileOS = new FileOutputStream(fileName, false);
    fileOS.write(new byte[50]);
  }finally{
    if(fileOS != null){
      try{
        fileOS.close();
      }catch(IOException e){
        // We failed to close the file, so we can't do anything here.
      }
      fileOS = null;
    }
    gcIfNeeded();  // Force garbage collection to free resources.
  }
}

 /** 
 *  For testing purposes; make the given file empty
 
 **/ 
private static void emptyFile(String fileName) throws IOException{
  ObjectOutputStream objectOS = null;
  FileOutputStream  fileOS = null;
  try{
    fileOS = new FileOutputStream(fileName, false);
    objectOS = new ObjectOutputStream(fileOS);
    objectOS.flush();
  }finally{
    if(objectOS != null){
      try{
        objectOS.close();
      }catch(IOException e){
        // We failed to close the file, so we can't do anything here.
      }
      objectOS = null;
    }
    if(fileOS != null){
      try{
        fileOS.close();
      }catch(IOException e){
        // We failed to close the file, so we can't do anything here.
      }
      fileOS = null;
    }
    gcIfNeeded();  // Force garbage collection to free resources.
  }
}

 /** 
 *  Check the basic funtionality of the read and write of objects..
 
 **/ 
private static void selfTest1(TwoFileObjectStore f){	
// f2 will be the newer file (due to the way the constructor is written)
  Vector a = new Vector(2);
  a.add(0, new Integer(12));
  a.add(1, new Integer(1978));
  try{
    f.writeObject(a); // Should write to f1
  }catch(FileNotFoundException e){
    System.out.println("TwoFileObjectStore: The files could not be" +
                       " written to");
    return;
  }catch(IOException e){
    System.out.println("TwoFileObjectStore: IOException : " + e);
    return;
  }
	
  // Read from the files
  Vector b;
  try{
    b = (Vector)f.readObjectAggressively(); // read from f1
    f.writeObject(b);           // Make f2 also have a copy of the data.
  }catch(StreamCorruptedException e){
    System.out.println("TwoFileObjectStore: " + e);
    return;
  }catch(FileNotFoundException e){
    System.out.println("TwoFileObjectStore: " + e);
    return;
  }catch(IOException e){
    System.out.println("TwoFileObjectStore: " + e);
    return;
  }catch(ClassNotFoundException e){
    System.out.println("TwoFileObjectStore: " + e);
    return;
  }

  // See if what was returned matches what was saved
  if(b.size() != a.size()){
    System.out.println ("TwoFileObjectStore: Sizes of the saved" +
                        " and returned object differ");
    return;
  }
  for(int i = 0; i < a.size(); i++){
    if(((Integer)b.get(i)).intValue() != ((Integer)a.get(i)).intValue()){
      System.out.println("TwoFileObjectStore: The saved and returned" +
                         " object differ in content");
      return;
    }
  }
  // POSTCONDITION: f2 will be newer and will contain the same data as f1.
  System.out.print("selfTest1 passed...");
}

 /** 
 *  Check for -> "Don't read from a corrupted file even if it is newer"...
 
 *  Should be done after selfTest1
 
 **/ 
private static void selfTest2(TwoFileObjectStore f){
  File f1 = null;
  File f2 = null;
  Vector b = null;
  Vector a = null;

  try{
    /** Test1:
     *    Newer file is corrupted and older is correct
     *    Corrupt file1 and make it newer.
     *    Make sure we can read the object without exceptions.
     *  Postcondition:
     *    file1 is corrupt. delete problems
     */
    f1 = new File((new File(testFileName + suffix1)).getAbsolutePath());
    f2 = new File((new File(testFileName + suffix2)).getAbsolutePath());
    corruptFile(f1.getAbsolutePath());
    assert f1.lastModified() >= f2.lastModified():
      "selfTest2: f1 should have been newer";
    // Set the file variables to null so that readObject can delete
    // the associated files.
    f1 = null;
    f2 = null;
    b = (Vector)f.readObjectAggressively();
    a = (Vector)f.readObjectAggressively();
    assert b != null : "selfTest2: b was null";
    // System.out.println("Here 1");
  }catch(Exception e){
    assert false : "selfTest2: Something wrong when accessing" +
      " the file system." + e;
  }
	
  try{
    /** Test2:
     *    Both files are corrupted
     *    Corrupt file1 and make it newer.
     *    Corrupt file2 and make it newer.
     *    Make sure that we get an assert failure since both files are corrupt.
     *  Postcondition:
     *    Both files are corrupted. f2 will be newer.
     */
    b = null;
    f1 = new File((new File(testFileName + suffix1)).getAbsolutePath());
    f2 = new File((new File(testFileName + suffix2)).getAbsolutePath());
    corruptFile(f2.getAbsolutePath());
    assert f2.lastModified() >= f1.lastModified() :
      "selfTest2: f2 should have been newer";
    b = (Vector)f.readObjectAggressively();
    assert b != null : "selfTest2: b was null";
  }catch(StreamCorruptedException e){
    // Do nothing here; we expected this.
    // System.out.println("Here 2");
  }catch(Exception e){
    assert false : "selfTest2: Something wrong when accessing" +
      " the file system.";
  }

  try{
    /** Test3:
     *   Newer file is corrupted and older file is
     *     empty => EOFException thrown in readObject()
     *   f2 is newer and corrupted. make
     *    1) f1 empty and
     *    2) f2 corrupt (to make it newer)
     *  PostCondition:
     *    same as 1) and 2) above
     */
    emptyFile(f1.getAbsolutePath());
    corruptFile(f2.getAbsolutePath());
    b = (Vector)f.readObjectAggressively();
  }catch(EOFException e){
    // System.out.println("selfTest2: EOFException as expected.");
    // System.out.println("Here 3");
  }catch(Exception e){
    System.out.println("selfTest2: " + e);
    assert false : "Should not have received this";
  }

  /** Test 4:
   *    Newer file is (made) empty and older file is correct.
   *  PostCondition:
   *    f1 is empty and f2 has the object
   *    Should be able to read the object correctly.
   */
  try{
    // Make both files correct again by writing twice
    f1 = new File((new File(testFileName + suffix1)).getAbsolutePath());
    f2 = new File((new File(testFileName + suffix2)).getAbsolutePath());
	    
    // f2 is newer at this point
    // FOR TESTING make variable true!!
    f.newerNotCorrupt = true;
    f.writeObject(a); // write to f1
    f.writeObject(a); // write to f2
    emptyFile(f1.getAbsolutePath()); // make f1 newer and empty
    b = (Vector)f.readObjectAggressively(); // read from f2
    assert b != null : "selfTest2: Read the NULL Object !";
    assert b.equals(a) :
      "selfTest2: Reading the Object is problematic!";
    // System.out.println("Here 4");
  } catch(Exception e){
    System.out.println ("selfTest2: " + e);
  }
	
  /** Test 5:
   *    Newer file is (made) empty and older file is corrupted
   *    f1 is new and empty, so make f2 corrupt and then f1 empty again
   *  PostCondition:
   *    f1 is new and empty, f2 is corrupt
   *    Should get an EOFException
   */
  try{
    // Make both files correct again by writing twice
    corruptFile(f2.getAbsolutePath()); // f2 is corrupted
    // f2 is newer at this point
    emptyFile(f1.getAbsolutePath()); // make f1 newer and empty

    b = (Vector)f.readObjectAggressively(); // read from f2 which is corrupted
    assert false : "selfTest2: Should've thrown an EOFException";
  }catch(EOFException e){
    //System.out.println("selfTest2: Got a new EOFException as expected" +
    //" for TestCase 5.");
    //System.out.println("Here 5");
  }catch(Exception e){
    System.out.println ("selfTest2: " + e);
    assert false : "Should not have received this";
  }

  /** Test 6:
   *    Both newer file and older files are made empty
   *    f1 is new and empty, so make f2 empty
   *  PostCondition:
   *    same as above!!
   *    Should get an EOFExcpetion
   */
  try{
    // Make both files correct again by writing twice
    emptyFile(f2.getAbsolutePath()); // f2 is empty
    b = (Vector)f.readObjectAggressively(); // read from f2 which is corrupted
    assert false : "selfTest2: Should've thrown an EOFException";
    System.out.println("Here 6");
  }catch(EOFException e){
    //System.out.println("selfTest2: Got a new EOFException as" +
    //	       " expected for TestCase 6.");
  }catch(Exception e){
    System.out.println ("selfTest2: " + e);
    assert false : "Should not have have received this";
  }
  System.out.print("selfTest2 passed...");
}


 /** 
 *  Test delete
 
 **/ 
private static void testDelete(TwoFileObjectStore f){
  boolean gotAssertion = false;
  f.delete();
  try{
    Object o = f.readObject();
  }
  catch(AssertionError e){
    gotAssertion = true; // Expected
  }
  catch(StreamCorruptedException zz){
    assert(false);
  }
  catch(FileNotFoundException fdsr){
    assert(false);
  }
  catch(IOException esdf){
    assert(false);
  }
  catch(ClassNotFoundException desdf){
    assert(false);
  }
  assert(gotAssertion);
  gotAssertion = false;
  try{
    FileInputStream fis = new FileInputStream(testFileName + suffix1);
  }
  catch(FileNotFoundException z){
    gotAssertion = true; // Expected
  }
  assert(gotAssertion);
  gotAssertion = false;
  try{
    FileInputStream fis = new FileInputStream(testFileName + suffix2);
  }
  catch(FileNotFoundException z){
    gotAssertion = true; // Expected
  }
  assert(gotAssertion);

  try{
    f = new TwoFileObjectStore(testFileName);
  }
  catch(ReservedPathNameException rpne){
    rpne.printStackTrace();
    assert(false);
  }
  assert f.versionNumber == -1 : "Version number expected -1 got " + f.versionNumber;
}

 /** 
 *  Test for corrupt newer file on constructor
 
 **/ 
private static void selfTest3(){
  String s = new String("Test object");
  TwoFileObjectStore f = null;
  try{
    f = new TwoFileObjectStore(testFileName);
  }
  catch(ReservedPathNameException rpne){
    rpne.printStackTrace();
    assert(false);
  }
  f.delete();
  try{
    f = new TwoFileObjectStore(testFileName);
  }
  catch(ReservedPathNameException rpne){
    rpne.printStackTrace();
    assert(false);
  }
  try{
    f.writeObject(s);
    f.writeObject(s);
  }
  catch(IOException ze){
    assert(false);
  }
  String newerFileName = f.getNewerFileName();
  f = null;

  //
  // Now corrupt the newer file
  //
  FileOutputStream fos = null;
  try{
    fos = new FileOutputStream(newerFileName);
  }
  catch(FileNotFoundException fdf){
    assert(false);
  }
  String gibberish = new String("Gibberish");
  try{
    fos.write(gibberish.getBytes()); // Note -- raw write bytes overwrites version
    fos.close();
  }
  catch(IOException ioe){
    assert(false);
  }

  try{
    f = new TwoFileObjectStore(testFileName);
  }
  catch(ReservedPathNameException rpne){
    rpne.printStackTrace();
    assert(false);
  }
}

 /** 
 *  Test for garbage collection -- do we run out of file descriptors
 
 *  if we use this class?
 
 **/ 
private static void selfTest4(){
  int ifile, ii;
  int NINTER = 10;
  int NFILE = 4097;
  long start, end;
  long write = 0L, read = 0L;
  double avg;

  System.out.print("Begin selfTest4() [stress test]...");

  TwoFileObjectStore stores[] = new TwoFileObjectStore[NFILE];
  start = System.currentTimeMillis();
  for(ifile = 0; ifile < NFILE; ifile++){
    String name = new String(TESTPATH + ifile / 100 + "/" + ifile % 100);
    try{
      stores[ifile] = new TwoFileObjectStore(name);
    }
    catch(ReservedPathNameException rpne){
      rpne.printStackTrace();
      assert(false);
    }
  }
  end = System.currentTimeMillis();
  avg = ((double)(end - start))/((double)NFILE);
  System.out.print(" [avg create time: " + avg + " ms]");
  String test = new String("Test data");
  for(ii = 0; ii < NINTER; ii++){
    start = System.currentTimeMillis();
    for(ifile = 0; ifile < NFILE; ifile++){
      try{
        stores[ifile].writeObject(test);
      }
      catch(IOException ioe){
        ioe.printStackTrace();
        assert(false);
      }
    }
    end = System.currentTimeMillis();
    write += (end - start);
    start = System.currentTimeMillis();
    for(ifile = 0; ifile < NFILE; ifile++){
      try{
        Object o = stores[ifile].readObject();
      }
      catch(Exception e){
        assert(false);
      }
    }
    end = System.currentTimeMillis();
    read += (end - start);
    System.out.print(".");
  }
  avg = ((double)(write))/((double)(NINTER * NFILE));
  System.out.print(" [avg write time: " + avg + " ms]");
  avg = ((double)(read))/((double)(NINTER * NFILE));
  System.out.print(" [avg read time: " + avg + " ms]");
  for(ifile = 0; ifile < NFILE; ifile++){
    stores[ifile].delete();
  }

  //
  // Just for comparison -- how long do "empty" calls to gc()
  // cost?
  //
  start = System.currentTimeMillis();
  for(ifile = 0; ifile < 100; ifile++){
    System.gc();
  }
  end = System.currentTimeMillis();
  avg = ((double)(end - start))/((double)(100));
  System.out.print(" [estimated gc time per raw gc call: " + avg + " ms]");
      
  System.out.print("test 4 passes");
}


 /** 
 *  selfTest5 -- test illegal pathname
 
 **/ 
private static void 
selfTest5(){
  TwoFileObjectStore f = null;
  System.out.print("Begin selfTest5() ...");
  try{
    f = new TwoFileObjectStore(testFileName + RESERVED);
  }
  catch(ReservedPathNameException rpne){
    assert(true);
    System.out.print("test 5 passes");
    return;
  }
  assert(false);
  
}

 /** 
 *  Test the class
 
 **/ 
public static void main(String[] argv){
  // Test createAncestorPath()
  try{
    FileOutputStream mdd1 = new FileOutputStream("test/foo");
    createAncestorPath("test/bar/baz/ble");
    FileOutputStream mdd2 = new FileOutputStream("test/bar/baz/ble");
  }
  catch(FileNotFoundException fnf){
    fnf.printStackTrace();
    assert(false);
  }
  catch(IOException e){
    e.printStackTrace();
    assert(false);
  }
      
  // Do some testing
  System.out.print("TwoFileObjectStore self test...");
  TwoFileObjectStore f = null;
  try{
    f = new TwoFileObjectStore(testFileName);
  }
  catch(ReservedPathNameException rpne){
    rpne.printStackTrace();
    assert(false);
  }

  selfTest1(f);
  selfTest2(f);
  testDelete(f);
  f.delete();
  selfTest3();
  selfTest4();
  selfTest5();
  System.out.println("TwoFileObjectStore self test PASSED");
}

}
