package code;

import code.security.*;
import code.security.holesync.filter.Filter;
import code.security.holesync.filter.SubscriptionSetFilter;
 /** 
 *  Test bandwidth to subscribe to 1000 objects, 100 of which have
 
 *  been modified. Test created for sosp 2007 submission microbenchmark.
 
 **/ 

//package utils.junit;

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class TwoNodesSubscriptionUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean printStats = false;
  protected static boolean verbose = true; // Start/end of test
  protected static boolean vverbose = true; // Test internals
  private Process rmiregistry;
  private Process helper;
  private BarrierServer barrier;
  public static final int BARRIER_PORT = 9743;
  public static int ni = 1, nj = 1, nk = 1;
  public static int printFreq1 = 10;
  public static int printFreq2 = 10;
  public static int NUM_DUMMY_NODES = 0;

  public static String COARSE_PATH = "/coarse-data";
  public static String FINE_PATH_PREFIX = "/fine-data";
  public static String FINE_PATH_LOG = FINE_PATH_PREFIX + "ForLogCatchup";
  public static String FINE_PATH_CP = FINE_PATH_PREFIX + "ForCPCatchup";
  public static String LAST_STAGE_PATH = "/lastStage";
  public static boolean useSecure = (SangminConfig.securityLevel == SangminConfig.COMPLETE);

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public TwoNodesSubscriptionUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    if(verbose){
      System.out.println("TwoNodesSubscriptionUnit test...");
    }
    System.gc();
    if(vverbose){
      RandomAccessStateUnit.memoryReport();
    }
    /* 
     * TBD: Insert other fixture code here
     * e.g., this.objectUnderTest = new ClassToBeTested(...);
     */
    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    Thread.sleep(2000);

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    if(verbose){
      System.out.println("rmiregistry started");
    }
    Thread.sleep(2000);

    //
    // Start barrier server
    //
    barrier = new BarrierServer(BARRIER_PORT, 2, BarrierServer.LOOP_FOREVER);
    barrier.start();

    //
    // Set up config files
    //
    makeDummyConfig(CONFIG_PATH);

    //
    // Set the connection style
    //
    OutgoingConnectionWorker.setDbgWithFileOutputStream(false);

    //
    // Start helper process
    //
    startHelper();

  }


  /* 
   * Config info
   */
  public static String CONFIG_PATH = "test" + File.separatorChar 
  + "tmp.TwoNodesSubscriptionUnit.config";
  public static long UNIT_NODE_ID = 10;
  public static long HELPER_NODE_ID = 20;
  public static long DUMMY_NODE_ID = 30;
  

  /*
   * Just make a 2-node config file on localhost in the 
   * specified path
   */
  protected void  makeDummyConfig(String path){

    // System.out.println("DBG: user.dir = " //MDD Debug jswat working dir
    // + System.getProperties().getProperty("user.dir")); 

    Config.createEmptyConfig();
    NodeId id = new NodeId(HELPER_NODE_ID);
    Config.addOneNodeConfig(id, "localhost", 2578, 2579, 2580, 2581, 2582,
			    "test" + File.separatorChar + "tmp.TwoNodesSubscriptionUnitHelper.db",
			    "/*",
			    -1L,
			    "localhost.cs.utexas.edu",
			    2583,
			    2584,
			    -1, 
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    id = new NodeId(UNIT_NODE_ID);
    Config.addOneNodeConfig(id, "localhost", 2478, 2479, 2480, 2481, 2482,
			    "test" + File.separatorChar + "tmp.TwoNodesSubscriptionUnitHelper.db",
			    "/*",
			    -1L,
			    "localhost.cs.utexas.edu",
			    2483,
			    2484,
			    -1, 
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    int ii;
    for(ii = 0; ii < NUM_DUMMY_NODES; ii++){
      id = new NodeId(DUMMY_NODE_ID + ii);
      Config.addOneNodeConfig(id, "localhost-" + ii, 9478, 9479, 9480, 9481, 9482,
			    "test" + File.separatorChar + "tmp.TwoNodesSubscriptionUnitHelper.db",
			    "/*",
			    -1L,
			    "localhost.cs.utexas.edu",
			    2483,
			    2484,
			    -1, 
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    }


    Config.writeToFile(path);
  }



  //---------------------------------------------------------------------------
  // Start helper process. Set global writer to refer to helper process.
  //---------------------------------------------------------------------------
  private void startHelper(){
    //
    // Start the helper PRACTI node
    // "make foo.unit" runs "java foo" with appropriate arguments
    //
    // Use "make" to figure out what arguments needed to
    // set up class paths, etc. Grab the command line
    // rather than just run make b/c we want a handle
    // on the helper process so we can kill it later.
    //
    try{
      helper = Runtime.getRuntime().exec("make TwoNodesSubscriptionUnitHelper.unit");
      if(verbose){
        System.out.println("helper started");
      }
      setUpHelperIO(helper, "only helper");
    }
    catch(Exception ex){
      ex.printStackTrace();
      assert(false);
    }
  }

  //---------------------------------------------------------------------------
  // Wait for the helper process to create the files
  // I want to read. Spawn "/dev/null" threads to
  // consume itstheir output so it doesn't block.
  //---------------------------------------------------------------------------
  private void setUpHelperIO(Process helper, String tag){
    InputStream is = helper.getInputStream();
    Thread dn = new DvNull(is, "TwoNodesSubscriptionUnitHelper " + tag + " stdout", 
                           verbose, vverbose);
    dn.start();
    is = helper.getErrorStream();
    dn = new DvNull(is, "TwoNodesSubscriptionUnitHelper " + tag + " stderr",
                    verbose, vverbose);
    dn.start();
  }



  //---------------------------------------------------------------------------
  // Kill helper process and kill rmiregistry
  //---------------------------------------------------------------------------
  protected void tearDown() throws Exception{
    if(helper != null){
      helper.destroy();
    }
    if(verbose){
      System.out.println("helper terminated");
    }
    rmiregistry.destroy();
    if(verbose){
      System.out.println("rmiregistry terminated");
    }
    
    super.tearDown();
    try{
     Thread.sleep(2000);
    }catch(InterruptedException ie){
      //ignore
    }
  }






  //--------------------------------------------------------------------------- 
  // Test coarse-grained subscriptions. 
  //
  // To initialize state, Unit create 1000 objects
  // (/data-coarse/[0..9]/[0..9]/[0..9]. Then helper subscribe
  // to /* from unit and wait for /data-coarse/9/9/9 to arrive.
  //
  //
  // start helper and wait for writing to complete
  //---------------------------------------------------------------------------
  public void testCoarse(){
    int pass;
    boolean catchupWithCP;

    BarrierClient bc = new BarrierClient("127.0.0.1", BARRIER_PORT, 0);
    NodeId myNodeId = new NodeId(UNIT_NODE_ID);
    NodeId helperNodeId = new NodeId(HELPER_NODE_ID);
    if(vverbose){
      System.out.println("UNIT ID: " + myNodeId.toString());
    }
    assert(myNodeId.equals(new NodeId(UNIT_NODE_ID)));
    URANode node;
    if(useSecure){
      SubscriptionSetFilter ssf = new SubscriptionSetFilter(SubscriptionSet.makeSubscriptionSet("/*"));
      LinkedList<Filter> f = new LinkedList<Filter>();
      f.add(ssf);
        
      node = new SecureURANode(CONFIG_PATH, myNodeId, 
          Controller.NULL_CONTROLLER, 
          true,//cleanDB 
          true, new code.security.liveness.TrustedServerTraceLivenessFilter(null), f);//noSynLog
    }else{
       node = new URANode(CONFIG_PATH, myNodeId, 
          Controller.NULL_CONTROLLER, 
          true,//cleanDB 
          true);//noSynLog
    }
  

    //second set of writes
    
    // make sure the accumulate time is long enough so that we don't
    // get PInvals when we are expecting ImpInval
    node.getCore().setMaxAccumulateTime(2000); 

    LocalInterface li = node.getLocalInterface();
    RMIClient ci = node.getRMIClientInterface();
    try{
      if(verbose){
        System.out.println("Hello from TwoNodesSubscriptionUnit");
      }

      write(COARSE_PATH, 0, ni-1, 0, nj-1, 0, nk-1, li);
    
      write(FINE_PATH_LOG, 0, ni-1, 0, nj-1, 0, nk-1, li);
      
//      write(FINE_PATH_CP, 0, ni-1, 0, nj-1, 0, nk-1, li);
      
      
      
      if(verbose){
        System.out.println("barrier 1");
      }

      bc.sendBarrierRequest(0, 0); // 1 We are up and running and have written; helper is up
      if(verbose){
        System.out.println("Both nodes up and running. Initial "+ ni*nj*nk + "  writes done.");
      }

      bc.sendBarrierRequest(0, 0); // 2 Helper has subscribed and received ni*nj*nk invals
      if(verbose){
        System.out.println("Helper has subscribed and received " +  ni*nj*nk + " invals from " + COARSE_PATH);
      }
      
      bc.sendBarrierRequest(0, 0); // 3 Helper has subscribed and received ni*nj*nk invals
      if(verbose){
        System.out.println("Helper has subscribed and received last inval from " + FINE_PATH_LOG);
      }
      
      bc.sendBarrierRequest(0, 0); // 4 Helper has subscribed and received ni*nj*nk invals
      if(verbose){
        System.out.println("Helper has subscribed and received last invals from " + FINE_PATH_CP);
      }
      
      bc.sendBarrierRequest(0, 0); // 5 Helper has subscribed and received ni*nj*nk invals
      if(verbose){
        System.out.println("Wait for helper to subscribe the last path " + LAST_STAGE_PATH);
      }
      write(LAST_STAGE_PATH, 0, ni-1, 0, nj-1, 0, nk-1, li);
      bc.sendBarrierRequest(0, 0); // 6 Helper has subscribed and received ni*nj*nk invals
      if(verbose){
        System.out.println("Helper has subscribed and received last invals from " + LAST_STAGE_PATH);
      }
      
      
      if(verbose){
        System.out.println("Unit is done. ");
      }
      /*
       
      bc.sendBarrierRequest(0, 0); 
      if(verbose){
        System.out.println("Helper has updated " + ni*nj + " objects");
      }

      resetStats();
      if(verbose){
	System.out.println("1 wait for " + COARSE_PATH + "/" + (ni-1) + "/" + (nj-1) + "/0");
      }
      
      catchupWithCP = false;
      subscribeAndWait(COARSE_PATH + "/*",
                       helperNodeId, myNodeId, 
                       COARSE_PATH + "/" + (ni-1) + "/" + (nj-1) + "/0",
                       li, ci, 
                       catchupWithCP);
      if(verbose){
	System.out.println("1 wait for  return " + COARSE_PATH + "/" + (ni-1) + "/" + (nj-1) + "/0");
      }
      printStats("TOTAL_BW_COARSE_UNIT "+ ni*nj*nk);

      if(verbose){
        System.out.println("Unit has received " + ni*nj + " invals. Helper will grab stats");
      }
      bc.sendBarrierRequest(0, 0); // 98 Tell helper to grab stats
      bc.sendBarrierRequest(0, 0); // 99 Make sure helper has stats


      //
      // Fine phase. Do it twice -- first with log catchup then with checkpoint.
      //
      int ipass;
      String passTag;
      for(ipass = 0; ipass < 2; ipass++){
        if(ipass == 0){
          passTag = "LOG";
	  catchupWithCP = false;
        }
        else{
          passTag = "CHECKPOINT";
	  catchupWithCP = true;
        }
        
        String FINE_PATH = FINE_PATH_PREFIX + passTag;

        if(verbose){
          System.out.println("Doing write of " + FINE_PATH + "/9/9/99");
        }
        write(FINE_PATH, 0, ni-1, 0, nj-1, 0, nk-1, li);
        bc.sendBarrierRequest(0, 0); // 100; did write of ni*nj*nk objects
        bc.sendBarrierRequest(0, 0); // 101; helper did read of ni*nj*nk objects
        bc.sendBarrierRequest(0, 0); // 102; helper did write of ni*nj objects
        resetStats();
        bc.sendBarrierRequest(0, 0); // 103; everyone reset stats
        long start = System.currentTimeMillis();
        int ii, jj, kk;
        int count = 0;
        for(ii = 0; ii < ni; ii++){
          for(jj = 0; jj < nj; jj++){
            for(kk = 0; kk < nk; kk++){
              String objId = ii + "/" + jj + "/" + kk;
	      if(verbose){
		System.out.println("2 wait for " + FINE_PATH + "/" + objId);
	      }
              subscribeAndWait(FINE_PATH + "/" + objId,
                               helperNodeId, myNodeId, 
                               FINE_PATH + "/" + objId,
                               li, ci, catchupWithCP);
	      if(verbose){
		System.out.println("2 wait for return " + FINE_PATH + "/" + objId);
	      }
              bc.sendBarrierRequest(0, 0); // 104; unit done reading [count]
              count++;
              printStats("TOTAL_BW_FINE_UNIT " + passTag + " " + count);
              bc.sendBarrierRequest(0, 0); // 105; done stats [count]
            }
            System.out.print(".");
          }
        }
	if(verbose){
	  System.out.println("done (" + passTag + ".)");
	}
        long end = System.currentTimeMillis();

        bc.sendBarrierRequest(0, 0); // 106; done

        if(verbose == false && vverbose == false && printStats == false 
           && (end - start > 1000 * ni * nj * nk)){
          assert(false):"per subscribe takes:" + (end-start)+ ": " + (1000 * ni * nj * nk); // 1000ms per subscribe should be *plenty*
        }
      }
       */
      try{
        Thread.sleep(5000);
      }
      catch(InterruptedException z){
      }
     
    } 
    finally{
      //
      // Make sure we release all of the ports so that next
      // unit test can run.
      //
      node.shutdown();
    }
    
  }




 /** 
 *  Write -- write 1-byte items to /prefix/[is..ie]/[js..je]/[ks..ke].
 
 *  Return final accept stamp.
 
 **/ 
  public static void write(String prefix, 
                           int is, int ie,
                           int js, int je,
                           int ks, int ke,
                           LocalInterface li){
    int ii, jj, kk;

    if(vverbose){
      System.out.println("Writing initial data: " + is + " to " + ie + " etc.");
    }
    for(ii = is; ii <= ie; ii++){
      for(jj = js; jj <= je; jj++){
        for(kk = ks; kk <= ke; kk++){
          ObjId oid = new ObjId(prefix + "/" + ii + "/" + jj + "/" + kk);
          byte b[] = new byte[1];
          b[0] = 42;
          try{
            if(vverbose){
              System.out.println("Writing " + oid);
            }
            long got = li.write(oid, 0, b.length, b, false);
            Assert.affirm(got == b.length);
          }
          catch(IOException e){
            System.out.println("IO Exception in TwoNodesSubscriptionUnit.write("
                               + oid.toString()
                               + ") -- "
                               + e.toString());
	    assert false;
          }
        }
      }
    }
    li.syncStateToDisk(); 
  }


 /** 
 *  subscribeAndWait -- subscribe to path and wait for inval to arrive 
 
 *  indicated specified object written by helper.
 
 * 
 
 *  Interface doesn't let us see accept stamp of invalid stuff (whoops).
 
 *  Initially this is valid; wait for it to become invalid.
 
 **/ 
  public static void subscribeAndWait(String subscribeId, 
                                      NodeId sender, NodeId receiver,
                                      String waitId, 
                                      LocalInterface li,
                                      RMIClient ci,
				      boolean catchupWithCP){

    ObjId sid = new ObjId(subscribeId);
    ObjId wid = new ObjId(waitId);

    if(vverbose){
      System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait for sid = " + subscribeId
                         + " waitId= " + waitId);
    }

    //
    // waitID should not be invalid before we subscribe
    //
    try{
      
      BodyMsg b = li.read(wid, 0, 1, false, false, 10000);
    }
    catch(ReadOfInvalidRangeException roire){
      assert(false):wid + roire.toString();
    }
    catch(ReadOfHoleException rohe){
      assert(false);
    }
    catch(EOFException eof){
      assert(false);
    }
    catch(IOException iof){
      assert(false);
    }
    catch(ObjNotFoundException onfe){
      // This is OK -- we have not received the inval
    }

    if(vverbose){
      System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait read of waitId= " + waitId
                         + " done.");
    }

    //
    // Subscribe
    //
    try{
      //
      // this subscription has a problem, because it will by default set the startVV[me]=infinity
      // then the subscriptionset will never be attached.
      //
      //ci.subscribeInval(sender, receiver.getIDint(), subscribeId, catchupWithCP);
      
         ci.subscribeInval( sender,
          receiver,
          SubscriptionSet.makeSubscriptionSet(subscribeId),
          AcceptVV.makeVVAllNegatives(),
          catchupWithCP);
           
    }
    catch(Exception e){
      e.printStackTrace();
      System.out.println("Exception on subscribeAndWait RMI: " + e.toString());
      assert(false);
    }

    if(vverbose){
      System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait subscribe of subscribeId= " + subscribeId
                         + " done.");
    }

    while(true){
      //
      // wait for waitID to become invalid
      //
      try{
	
        BodyMsg b = li.read(wid, 0, 1, false, false, ni*nj*nk);
        if(b == null){
          assert(false); // Timeout
        }
        // Data still valid from my earlier write -- retry
      }
      catch(ReadOfInvalidRangeException roire){
        if(vverbose){
          System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait wait-invalid of waitId= " + waitId
                             + " done.");
        }
        return;
      }
      catch(ReadOfHoleException rohe){
        assert(false);
      }
      catch(EOFException eof){
        assert(false);
      }
      catch(IOException iof){
        System.out.println(iof);
        assert(false);
      }
      catch(ObjNotFoundException onfe){
        // This is OK -- we have not yet received the inval -- retry
      }
      try{
        Thread.sleep(10); // Wait before retry
      }
      catch(InterruptedException e){
      }
    }
  }
  
  
 /** 
 * subscribeAndWait -- subscribe to path and wait for inval to arrive  
 * indicated specified object written by helper. 
 *  
 * Interface doesn't let us see accept stamp of invalid stuff (whoops). 
 * Initially this is valid; wait for it to become invalid. 
 **/ 
 public static void subscribe(String subscribeId, 
                                     NodeId sender, NodeId receiver,
                                     
                                     LocalInterface li,
                                     RMIClient ci,
                                     boolean catchupWithCP){

   ObjId sid = new ObjId(subscribeId);
   

   //
   // Subscribe
   //
   try{
     //
     // this subscription has a problem, because it will by default set the startVV[me]=infinity
     // then the subscriptionset will never be attached.
     //
     //ci.subscribeInval(sender, receiver.getIDint(), subscribeId, catchupWithCP);
     
        ci.subscribeInval( sender,
         receiver,
         SubscriptionSet.makeSubscriptionSet(subscribeId),
         AcceptVV.makeVVAllNegatives(),
         catchupWithCP);
          
   }
   catch(Exception e){
     e.printStackTrace();
     System.out.println("Exception on subscribeAndWait RMI: " + e.toString());
     assert(false);
   }

   if(verbose){
     System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait subscribe of subscribeId= " + subscribeId
                        + " done.");
   }

  
 }

 
 /** 
 * subscribeAndWait -- subscribe to path and wait for inval to arrive  
 * indicated specified object written by helper. 
 *  
 * Interface doesn't let us see accept stamp of invalid stuff (whoops). 
 * Initially this is valid; wait for it to become invalid. 
 **/ 
public static void waitObj(String waitId, 
                           LocalInterface li){

  
  ObjId wid = new ObjId(waitId);

  if(verbose){
    System.out.println("TwoNodesSubscriptionUnit::waitObj: " 
                       + " waitId= " + waitId);
  }

  //
  // waitID should not be invalid before we subscribe
  //
  try{
    
    BodyMsg b = li.read(wid, 0, 1, false, false, 10000);
  }
  catch(ReadOfInvalidRangeException roire){
    assert(false):wid + roire.toString();
  }
  catch(ReadOfHoleException rohe){
    assert(false);
  }
  catch(EOFException eof){
    assert(false);
  }
  catch(IOException iof){
    assert(false);
  }
  catch(ObjNotFoundException onfe){
    // This is OK -- we have not received the inval
  }

  if(vverbose){
    System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait read of waitId= " + waitId
                       + " done.");
  }

 

  while(true){
    //
    // wait for waitID to become invalid
    //
    try{
      
      BodyMsg b = li.read(wid, 0, 1, false, false, ni*nj*nk);
      if(b == null){
        assert(false); // Timeout
      }
      // Data still valid from my earlier write -- retry
    }
    catch(ReadOfInvalidRangeException roire){
      if(vverbose){
        System.out.println("TwoNodesSubscriptionUnit::UpdateAndWait wait-invalid of waitId= " + waitId
                           + " done.");
      }
      return;
    }
    catch(ReadOfHoleException rohe){
      assert(false);
    }
    catch(EOFException eof){
      assert(false);
    }
    catch(IOException iof){
      System.out.println(iof);
      assert(false);
    }
    catch(ObjNotFoundException onfe){
      // This is OK -- we have not yet received the inval -- retry
    }
    try{
      Thread.sleep(10); // Wait before retry
    }
    catch(InterruptedException e){
    }
  }
}

 /** 
 *  resetStats
 
 **/ 
  public static void resetStats(){
    Stats.reset();
  }

 /** 
 *  printStats
 
 **/ 
  public static void printStats(String finalTag){
    Hashtable h = Stats.getStats();
    Enumeration e = h.keys();
    long tot = 0;
    while(e.hasMoreElements()){
      String key = (String)e.nextElement();
      Long value = (Long)h.get(key);
      if(printStats){
        System.out.println("Stats: " + key + " " + value);
      }
      tot += value.longValue();
    }
    if(printStats){
      System.out.println(finalTag + " " + tot);
    }
  }


  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(TwoNodesSubscriptionUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   TwoNodesSubscriptionUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "TwoNodesSubscriptionUnit";
    System.err.print(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          verbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new TwoNodesSubscriptionUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);

    System.exit(0);
  }

}



