package code;

import java.util.*;

 /** 
 *  class SSUPQ is simply an (ss, upq) pair. 
 **/ 
class SSUPQ {
  private SubscriptionSet ss;
  private UpdatePriorityQueue upq;

 /** 
 *  Constructor 
 **/ 
  public
  SSUPQ(SubscriptionSet ss_, UpdatePriorityQueue upq_){
    this.ss = ss_;
    upq = upq_;
  }

 /** 
 *  Return true if "pair" equals "this" 
 **/ 
  public boolean
  equals(SSUPQ pair) {
    return(this.ss.equals(pair.ss) && (this.upq == pair.upq));
  }

 /** 
 *  Return the SubscriptionSet for this SSUPQ object 
 **/ 
  public SubscriptionSet
  getSS(){
    return(this.ss);
  }

 /** 
 *  Sets the SubscriptionSet for this SSUPQ object 
 **/ 
  public void
  setSS(SubscriptionSet ss_){
    this.ss = ss_;
  }

 /** 
 *  Return the UpdatePriorityQueue for this SSUPQ object 
 **/ 
  public UpdatePriorityQueue
  getUpq() {
    return(this.upq);
  }
}


 /** 
 *  class UPQNotifier is simply a linkedlist of SSUPQ pairs. 
 **/ 
class UPQNotifier {
  LinkedList notifyList;	

 /** 
 *  Data members 
 **/ 
  public final long UPDATE = 0;
  public final long INVAL = 1;
  private static final boolean dbg = false;

 /** 
 *  Constructor 
 **/ 
  public
  UPQNotifier() {
    notifyList = new LinkedList();	
  }

 /** 
 *  Return true if all UPQs are empty 
 **/ 
  public synchronized boolean
  isAllUPQEmpty(){
    boolean isEmpty = true;
    ListIterator li = notifyList.listIterator();
    while(li.hasNext()){
      UpdatePriorityQueue upq = ((SSUPQ)li.next()).getUpq();
      if(!upq.isEmpty()){
        isEmpty = false;
        break;
      }
    }
    return isEmpty;
  }

 /** 
 *  add() adds a pair if it's not already present. 
 **/ 
  public synchronized boolean
  add(SubscriptionSet ss, UpdatePriorityQueue upq){
    
    boolean added = false;

    SSUPQ pair = find(upq);

    if(pair != null){
      pair.setSS(pair.getSS().getCompleteUnion(ss));
      added = true;
    }
    else {
       pair = new SSUPQ(ss, upq);
       notifyList.add(0, pair);
       added = true;
    }
    if( dbg ){
      Env.dprintln(dbg, "UPQ notifier: added "+ ss + " for "+upq.getDestId());
    }
    return added;
  }

 /** 
 *  remove() removes a pair and returns true is it was present. 
 **/ 
  public synchronized boolean
  remove(SubscriptionSet ss, UpdatePriorityQueue upq) {
    SSUPQ pair = find(upq);

    if(pair == null  || pair.getSS().getIntersection(ss).isEmpty()){
      return false;
    }
    
    try{
      pair.setSS(pair.getSS().remove(ss, true));
    }catch(IllegalRemoveSubscriptionSetException e){
      assert false : ("" + e);
    }

    //
    // for efficiency
    //
    if (pair.getSS().isEmpty()) {
      notifyList.remove(pair);
    }
    if( dbg ){
      Env.dprintln(dbg, "UPQ notifier: removed "+ ss + " for  "+ upq.getDestId());
    }
    return true;
  }

 /** 
 *  notify() inserts bound update into corresponding upq which then 
 *  signals threads waiting on it. 
 **/ 
  public synchronized void
  notifyUpq(BoundInval bi) {
    if( dbg ){
      Env.dprintln(dbg, "UPQNotifier::notifyUPQ bi called");
    }
    for(ListIterator i = notifyList.listIterator(); i.hasNext(); ) {
      Env.dprinterrln(dbg, "UPQNotifier::notifyUPQ bi iter");
      SSUPQ pair = (SSUPQ)i.next();

      if( dbg ){
	Env.dprintln(dbg, "UPQNotifier:: comparing "+ bi.getInvalTarget() +
		   "and "+ pair.getSS());
      }

      if(bi.getInvalTarget().intersects(pair.getSS())) {
        // insert into priority queue	
	if( dbg ){
	  Env.dprintln(dbg, "UPQNotifier::added to Queue");
	}

        InvalTarget it = bi.getInvalTarget();
        assert(it instanceof ObjInvalTarget);
        ObjInvalTarget oit = (ObjInvalTarget)it;

        pair.getUpq().insert(bi.getPriority(),
                             oit.getObjId(),
                             oit.getOffset(),
                             oit.getLength());
      }
    }
    Env.dprinterrln(dbg, "UPQNotifier::notifyUPQ bi returns");
  }

 /** 
 *  same as above except it takes BodyMsg as argument. 
 **/ 
  public synchronized void
  notifyUpq(BodyMsg body){
    Env.dprinterrln(dbg, "UPQNotifier::notifyUPQ bm called");
    for(ListIterator i = notifyList.listIterator(); i.hasNext(); ) {
      Env.dprinterrln(dbg, "UPQNotifier::notifyUPQ bm iter");
      SSUPQ pair = (SSUPQ)i.next();
      

      if(body.getObjInvalTarget().intersects(pair.getSS())) {
        // insert into priority queue	
        ObjInvalTarget oit = body.getObjInvalTarget();
	if( dbg ){
	  Env.dprintln(dbg, "UPQNotifier::added to Queue");
	}
        pair.getUpq().insert(body.getPriority(),
                             oit.getObjId(),
                             oit.getOffset(),
                             oit.getLength());
      }
    }
    Env.dprinterrln(dbg, "UPQNotifier::notifyUPQ bm returns");
  }

 /** 
 *  Remove a particular object for a given receiver 
 **/ 
  public synchronized void
  notifyUPQCancelPrefetch(ObjId id, long offset, long length, NodeId destId){
    SSUPQ pair = null;
    UpdatePriorityQueue upq = null;

    Env.dprinterrln(dbg, "UPQNotifier::notifyUPQCancelPrefetch called");
    for(ListIterator i = notifyList.listIterator(); i.hasNext(); ) {
      pair = (SSUPQ)i.next();
      upq = pair.getUpq();

      if(upq.getDestId().equals(destId)){
        upq.cancel(id, offset, length);
      }
    }
    Env.dprinterrln(dbg, "UPQNotifier::notifyUPQCancelPrefetch returns");
  }

 /** 
 *  find(destId) finds the pair which matches the destId 
 **/ 
  private SSUPQ
  find(NodeId destId){
    SSUPQ pair = null;
    UpdatePriorityQueue upq = null;

    for(ListIterator i = notifyList.listIterator(); i.hasNext();){
      pair = (SSUPQ) i.next();
      upq = pair.getUpq();

      if(upq.getDestId().equals(destId)){
	return pair;
      }
    }
    return null;
  }

 /** 
 *  find(upq) finds the pair which matches the destId of the queue 
 *  We have assumed that there is only 1 upq for a destination 
 **/ 
  private SSUPQ
  find(UpdatePriorityQueue upq_){
    NodeId destId = upq_.getDestId();
    return find(destId);
  }

 /** 
 *  For testing 
 **/ 

  public void displayNotifier(){
    SSUPQ pair = null;
    UpdatePriorityQueue upq = null;

    for(ListIterator i = notifyList.listIterator(); i.hasNext(); ) {
      pair = (SSUPQ)i.next();
      upq = pair.getUpq();
      System.out.println("destId=" + upq.getDestId() + " ss="+pair.getSS());
      upq.displayQueue();
    }
  }

 /** 
 *  Testing of add and remove from notifier 
 **/ 

  public static void main(String args[]){
    
    UpdatePriorityQueue upq1 = new UpdatePriorityQueue(new NodeId(1));
    UpdatePriorityQueue upq2 = new UpdatePriorityQueue(new NodeId(2));
    
    SubscriptionSet ss1 = SubscriptionSet.makeSubscriptionSet("/apple");
    SubscriptionSet ss2 = SubscriptionSet.makeSubscriptionSet("/banana");
    SubscriptionSet ss3 = SubscriptionSet.makeSubscriptionSet("/cherry");
    SubscriptionSet ss4 = SubscriptionSet.makeSubscriptionSet("/durian");
    

    UPQNotifier notifier = new UPQNotifier();
    
    System.out.println("Testing add....");
    notifier.add(ss1,upq1);
    notifier.add(ss2,upq1);
    notifier.add(ss3,upq2);
    
    notifier.displayNotifier();
    
    System.out.println("Testing remove....");      
    notifier.remove(ss4, upq1);
    System.out.println("Test 1: should be same as b4");
    notifier.displayNotifier();
    
    notifier.remove(ss2,upq1);
    System.out.println("Test 2: removing " + ss2 + " from "+upq1.getDestId());
    notifier.displayNotifier();    
  
    notifier.remove(ss3,upq2);
    System.out.println("Test 3: node2 should be empty");
    notifier.displayNotifier();
    
    

    
  }
      

}
