package code;

import java.io.*;
import java.util.*;

 /** 
 **/ 
public class URANode
{
  //
  // Members (all protected)
  //
  protected Core core;
  protected RMIServerImpl rmiServer;         
  protected SocketServer socketServer;
  protected Controller controller;
  protected LocalInterface localInterface;
  protected RMIClient rmiClient;

  protected URANode(){
    
  }
 /** 
 *  cleanDb is used for testing purpose. True -- clean all the involved db 
 **/ 
  public URANode(String localConfigFile,
                 NodeId myId,
                 int controllerType,
                 boolean cleanDb){
    this(localConfigFile, myId, controllerType, cleanDb, false);
  }

 /** 
 *  cleanDb is used for testing purpose. True -- clean all the involved db 
 **/ 
  public URANode(String localConfigFile,
                 NodeId myId,
                 int controllerType,
                 boolean cleanDb,
		 boolean noSyncLog){
    try{
      Config.readConfig(localConfigFile);
      rmiClient = new RMIClient();

      //???????add filterOn to parameter later
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, myId, noSyncLog);
      controller = makeController(controllerType, core, rmiClient);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("" + e);
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  cleanDb is used for testing purpose. True -- clean all the involved db 
 **/ 
  public URANode(String localConfigFile,
                 NodeId myId,
                 Controller c,
                 boolean cleanDb,
		 boolean noSyncLog){
    try{
      Config.readConfig(localConfigFile);
      rmiClient = new RMIClient();
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, myId, noSyncLog);
      controller = c;
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("" + e);
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  nullRAS is used to pick a null random access state 
 **/ 
  public URANode(String localConfigFile,
                 NodeId myId,
                 int controllerType,
                 boolean cleanDb,
		 boolean noSyncLog, 
		 boolean nullRAS){
 
  try{
      Config.readConfig(localConfigFile);
      rmiClient = new RMIClient();

      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, myId, noSyncLog, nullRAS);
      controller = makeController(controllerType, core, rmiClient);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("" + e);
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  cleanDb is used for testing purpose. True -- clean all the involved db 
 **/ 
  public URANode(String localConfigFile,
                 NodeId myId,
                 int controllerType,
                 boolean cleanDb,
		 boolean noSyncLog,
		 boolean logLIOpToFile,
		 String logLIOpFileName){
    try{
      Config.readConfig(localConfigFile);
      rmiClient = new RMIClient();

      //???????add filterOn to parameter later
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, myId, noSyncLog);
      controller = makeController(controllerType, core, rmiClient);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core, logLIOpToFile, logLIOpFileName);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("" + e);
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  cleanDb is used for testing purpose. by default it's false. i.e. recover 
 *  from disk 
 **/ 
  public URANode(String localConfigFile, NodeId myId, int controllerType){

    try{
      Config.readConfig(localConfigFile);
      rmiClient = new RMIClient();

      //???????add filterOn to parameter later
      boolean filterOn = true;
      boolean cleanDb = false; // by default, recover from disk
      core = new Core(rmiClient, filterOn, cleanDb, myId, false);
      controller = makeController(controllerType, core, rmiClient);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("" + e);
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  Attempt clean shutdown. Mostly used for unit tests so that 
 *  we can run multiple tests sequentially within same process. 
 *  Not extensively tested 
 **/ 
  public void shutdown(){
    //
    //  Note we need to shutdown the outgoing connections (RMIServer) 
    //  before we close the core, otherwise the outgoing connections
    //  will try to poll for invals and there will be an assert error
    //  in the update log.
    //
    rmiServer.close();
    rmiServer = null;
    socketServer.shutdown();
    socketServer = null;
    core.syncStateToDisk();
    core.close();
    core = null;
  }

 /** 
 *  Provide a simple interface that allows us to pass in our own 
 *  implementations of a controller and core 
 **/ 
  public
  URANode(Core newCore, Controller newController, RMIClient newRMIClient){
    try{
      this.core = newCore;
      this.controller = newController;
      this.rmiClient = newRMIClient;

      this.rmiServer = new RMIServerImpl(core, controller);
      this.rmiServer.start();
      this.socketServer = new SocketServer(core, controller);
      this.localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      this.core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("" + e);
      e.printStackTrace();
      assert(false);
    }
  }

  protected Controller makeController(int type, Core core, RMIClient client){
    if(type == Controller.NULL_CONTROLLER){
      return new LocalController(core);
    }
    else if (type == Controller.TRIVIAL_CONTROLLER){
      return new SDIMSController(core, 
                                 new RandomReadDirectory(),
                                 new TrivialSpanningDirectory(core.getMyNodeId()),
                                 rmiClient,
                                 0, 
                                 true,
                                 2, 2, 2, 100);
    }
    else if (type == Controller.SDIMS_CONTROLLER){
      SDIMSInterface sdims = new SDIMSInterface(core.getMyNodeId()) ;
      //
      // start sdims here 
      //
      sdims.start() ;

      return new SDIMSController(core, 
                                 new SDIMSReadDirectory(sdims),
                                 new SDIMSSpanningDirectory(sdims),
                                 rmiClient,
                                 0, 
                                 false,
                                 2, 2, 2, 100);
    }
    assert(false);
    return null;
  }





 /** 
 **/ 
  public LocalInterface getLocalInterface(){
    assert(localInterface != null);
    return localInterface;
  }

 /** 
 *  getRMIClientInterface() -- return reference to RMI client interface 
 **/ 
  public RMIClient getRMIClientInterface(){
    assert(rmiClient != null);
    return rmiClient;
  }

  public VV getCurrentVV(){
    return core.getCurrentVV();
  }

 /** 
 *  Return a newly allocated Immutable byte array with "len" dummy bytes 
 *  (for testing) 
 **/ 
  public static ImmutableBytes populateByteArray(long len)
    {
      byte[] buffer = null;
      byte[] preambleBytes = null;
      String preamble = null;
      byte letter = (byte)'a';
      int index = 0;

      preamble = "For your universal replication needs: URS! ";
      preambleBytes = preamble.getBytes();
      buffer = new byte[(int)len];
      index = 0;
      while(index < Math.min((int)len, preambleBytes.length)){
        buffer[index] = preambleBytes[index];
        index++;
      }
      while(index < len){
        buffer[index] = letter;
        letter++;
        if(letter > 'z'){
          letter = 'a';
        }
        index++;
      }
      ImmutableBytes b = ImmutableBytes.dangerousCreateImmutableBytes(buffer);
      buffer = null;
      return(b);
    }


 /** 
 *  Populating writes from a file 
 **/ 
  public void populateLocalWrites(InputStream is){
    boolean dbgWriteScript = false;
    // global variables for emulating access rate of writes
    long accumulatedTimeMillis = 0;
    long nextWriteExpectedTime = System.currentTimeMillis();
    long startTime = System.currentTimeMillis();
    long writeNum = 0;

    Env.printDebug("Enter action:  ");
    Env.printDebug("Comments       : c [comments]");
    Env.printDebug("write          : w [ObjId] [offset] [length]"
                   +" [bytes] [bound])");
    Env.printDebug("GetAcceptStamp : g [NodeId]");
    Env.printDebug("Read           : r [ObjId] [offset] [length]");
    //get the highest number of the specific Node
    Env.printDebug("Sync           : s [NodeId] [Time]");
    Env.printDebug("SyncBody       : b");
    Env.printDebug("test status    : t");
    Env.printDebug("sleep          : l [durationMS]");
    Env.printDebug("end            : e");
//        System.err.println("populateLocalWrites:");
    BufferedReader din = null;
		
    try{
	    		
      din = new BufferedReader(new InputStreamReader(is));
		
	    
      String input;
      for(input = din.readLine(); input!= null; input=din.readLine()) {
        Env.printDebug("");
        System.err.println("Input (" + core.getMyNodeId()+ "): " + input);
        byte[] action = input.getBytes();
		
        if(action[0] == 'w'){
          writeNum++;
          StringTokenizer st = new StringTokenizer(input.substring(2));
          ObjId objId = new ObjId(st.nextToken());
          long offset = (new Long(st.nextToken())).longValue();
          long len = (new Long(st.nextToken())).longValue();
          ImmutableBytes buffer = populateByteArray(len);
          boolean bound = false;
          int b = (new Integer(st.nextToken())).intValue();
          if (b==1) bound = true;
          int accessInterval = (new Integer(st.nextToken())).intValue();
                    
          if(accessInterval <=0){//no specific access interval
            localInterface.write(objId, offset, len, buffer, bound);
                    
          } else { 
            long diff = nextWriteExpectedTime - System.currentTimeMillis();
            while(diff>0){
              try{
                if(dbgWriteScript){
                  System.err.println("wait for next write arriving:" + diff);
                }
                Thread.sleep(diff);
              } catch (InterruptedException e){
                Env.printDebug("Interrupted while waiting to send data. Will check for schedule again!");
              }
              diff = nextWriteExpectedTime - System.currentTimeMillis();
            }
            assert (diff<=0); 
            if(dbgWriteScript){
              System.err.println("write @ " + System.currentTimeMillis());
            }
            localInterface.write(objId, offset, len, buffer, bound);
            nextWriteExpectedTime = (long) ( System.currentTimeMillis() 
                                             + accessInterval
                                             + diff);
            //note:it's possible that the given accessInterval is too
            // small to succeed. In this case, the actual accessInterval
            // will be the latency of one write.. 
          }
			
        } else if (action[0] == 'g'){//get acceptStamp
          long duration = nextWriteExpectedTime - startTime;
          if ((dbgWriteScript) && (duration>0)){
            
            System.err.println("AVG Access Rate: " + 
                               (((double)(writeNum)/(double)(duration))*1000) + "/s");
          }
          StringTokenizer st = new StringTokenizer(input.substring(2));
          NodeId nodeId = new NodeId((new Long(st.nextToken())).longValue());
          //System.err.println("Node " + core.getMyNodeId()+ " are about to get acceptStamp");
          CoordCommPacket ccp = new CoordCommPacket(CoordCommPacket.ACCEPT_STAMP,
                                                    core.getAcceptStamp(nodeId));
          
          ccp.sendSelf(System.out);
          

        } else if (action[0] == 'd'){//get acceptStamp
		    
          CoordCommPacket ccp = new CoordCommPacket(CoordCommPacket.SYNC_DONE,
                                                    null);
          System.err.println("Node " + core.getMyNodeId()+ " are about to send SYNC_DONE");
          ccp.sendSelf(System.out);
          System.err.println("Node " + core.getMyNodeId()+ "sent SYNC_DONE");
		    
        } else if (action[0] == 'r'){//blockRead send READ_DONE at the end
          System.err.println("Node[" + core.getMyNodeId() + "] start to read");
          StringTokenizer st = new StringTokenizer(input.substring(2));
          ObjId objId = new ObjId(st.nextToken());
          long offset = (new Long(st.nextToken())).longValue();
          long len = (new Long(st.nextToken())).longValue();
          BodyMsg bm = null;
          try{
            bm = core.readBlockInvalid(objId, offset, len, true, true, -1);
            System.err.println("Node[" + core.getMyNodeId() + "] finishes read");
          }
          catch(IOException e){
            e.printStackTrace();
            assert(false);
          }
          catch(Exception e){
            //
            // ObjNotFoundException or EOFException
            //
            assert(false); // Fragile. Is there a better way to handle this?
          }
          assert(bm != null);
          /*
            System.err.println("read AcceptStamp:" + 
            bm.getAcceptStamp());
          */
          System.err.println("node [" + core.getMyNodeId() + "] read result: " + bm.getAcceptStamp());
                    
          CoordCommPacket ccp = new CoordCommPacket(CoordCommPacket.READ_DONE,
                                                    bm.getAcceptStamp());
          ccp.sendSelf(System.out);

        } else if (action[0] == 'n'){//NonBlockRead do not send READ_DONE at the end
          StringTokenizer st = new StringTokenizer(input.substring(2));
          ObjId objId = new ObjId(st.nextToken());
          long offset = (new Long(st.nextToken())).longValue();
          long len = (new Long(st.nextToken())).longValue();
          BodyMsg bm = null;
          try{
            bm = core.readBlockInvalid(objId, offset, len, true, true, -1);
          }
          catch(IOException e){
            e.printStackTrace();
            assert(false);
          }
          catch(Exception e){
            //
            // ObjNotFoundException or EOFException
            //
            e.printStackTrace();
            assert(false); // Fragile. Is there a better way to handle this?
          }
          assert(bm != null);
          /*
            System.err.println("read AcceptStamp:" + 
            bm.getAcceptStamp());
          */
          //System.err.println("node [" + core.getMyNodeId() + "] read result: " + bm);
                    
        } else if (action[0] == 's'){//Sync
          StringTokenizer st = new StringTokenizer(input.substring(2));
          NodeId nodeId = new NodeId((new Long(st.nextToken())).longValue());
          long stamp = (new Long(st.nextToken())).longValue();
          AcceptStamp as = new AcceptStamp(stamp, nodeId);
          core.syncCheck(as);
		  
          CoordCommPacket ccp = new CoordCommPacket(CoordCommPacket.SYNC_DONE,
                                                    null);
          ccp.sendSelf(System.out);
        } else if (action[0] == 'o'){//Sync localstore to disk
          core.syncStateToDisk();
        } else if (action[0] == 't'){
          core.checkState();

        } else if (action[0] == 'l'){
          StringTokenizer st =
            new StringTokenizer(input.substring(2));
          long sleepDurationMS = Long.parseLong(st.nextToken());
          try{
            Thread.sleep(sleepDurationMS);
          }catch(InterruptedException e){
            System.err.println("" + e);
            e.printStackTrace();
          }

        } else if (action[0] == 'e'){
          System.exit(-1);

        } else if (action[0] == 'c'){
          // Do nothing
        } else if (action[0] == 'v'){// get VV
          VV cvv = core.getCurrentVV();
          CoordCommPacket ccp = new CoordCommPacket(CoordCommPacket.CURRENT_VV, cvv);
          ccp.sendSelf(System.out);
        } else if (action[0] == 'b'){// sync on body stream
          assert(false); // MDD: This code is broken -- someone commented out calls 
          // to DataStore::waitAllUPQClear()
          core.startSyncBodyCheck();
          // response will be sent back within the call.
        }
      }
    } catch (Exception e){
      e.printStackTrace();
      Env.printDebug("Wrong input format ...");
    }	
  }

 /** 
 *  Make and send an awake signal on standard output 
 **/ 
  public void sendAwakeSignal()
    {
      CoordCommPacket ccp = null;

      ccp = new CoordCommPacket(CoordCommPacket.AWAKE_SIGNAL, null);
      try{
        ccp.sendSelf(System.out);
        System.err.println("Sent self over stdout stream...");
      }catch(IOException e){
        System.err.println("Error writing to standard output: " + e);
        e.printStackTrace();
        assert(false);
      }
    }

 /** 
 *  getCore 
 **/ 
  public Core getCore(){
    return core;
  }

 /** 
 *  get Controller 
 **/ 
  public Controller getController(){
    return controller;
  }

 /** 
 *  Sanity Check 
 **/ 
  protected void sanityCheck(){
    //core.cleanEnv(); //clean up everything(log, store)
    //test13(false, false, false);
    //test13(true, false, false);
    test13(false, true, false);
    //test14(false, false, false);
    //test14(true, false, false);
    
    test14(false, true, false);
   
    test15(true);//bound
    test15(false);//unbound
//    core.cleanEnv();
  }

 /** 
 *  test13() -- small file write/create/read/delete test 
 **/ 
  protected void
  test13(boolean syncEach, boolean syncPhase, boolean bound){
    int nFiles = 1000;
    long start, end;
    int ii;
    String objId;
    byte b[] = {0};
    ImmutableBytes ib = new ImmutableBytes(b);
    System.out.println("Test13 [small file performance test " 
                       + (syncEach ? " SYNC_EACH" : " NO_SYNC_EACH") 
                       + (syncPhase ? " SYNC_PHASE" : " NO_SYNC_PHASE")
                       + (bound ? " BOUND" : " UNBOUND")
                       + " ]...");
    BodyMsg bm;
//  AcceptStamp as = new AcceptStamp(1000, new NodeId(9999999));
    long ret;
    double avgMS;
    CounterVV vv = new CounterVV();

    try{
    
      /*
      // 
      // Update VV with this nodeID (outside of timing for fairness
      //
      bm = new BodyMsg(new ObjId("/test-13/foo"), 0, 1, -999, as, ib, false);
      ret = ras.applyBody(bm);
      */

      //
      // Create
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        /*
          bm = new BodyMsg(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 
          0, 1, -999, as, ib, false);
          ret = ras.applyBody(bm);
        */
        ret = localInterface.write(new ObjId("/test-13/" + ii + syncEach + syncPhase),
                                   0, 1, ib, bound);
        assert(ret == 1);
        if(syncEach){
          core.syncStateToDisk();
        }
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("Create 1 byte file: " + avgMS + "ms...");

      //
      // Overwrite
      //

//    as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        /*
          bm = new BodyMsg(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 
          0, 1, -999, as, ib, false);
          ret = ras.applyBody(bm);
        */
        ret = localInterface.write(new ObjId("/test-13/" + ii + syncEach + syncPhase),
                                   0, 1, ib, bound);
        assert(ret == 1);
        if(syncEach){
          //ras.sync(vv);
          core.syncStateToDisk();
        }
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("Overwrite 1 byte file: " + avgMS + "ms...");


      //
      // Read
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
      
        bm = localInterface.read(new ObjId("/test-13/" + ii + syncEach + syncPhase), 
                                 0, 1, true, true);
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("Read 1 byte file: " + avgMS + "ms...");


      //
      // Append
      //
      //as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        /*
          bm = new BodyMsg(new ObjId("/test-13/" + ii + syncEach + syncPhase + doSanity), 
          1, 1, -999, as, ib, false);
          ret = ras.applyBody(bm);
        */
        ret = localInterface.write(new ObjId("/test-13/" + ii + syncEach + syncPhase),
                                   1, 1, ib, bound);
        assert(ret == 1);
        if(syncEach){
          core.syncStateToDisk();
        }
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("Append 1 byte to file: " + avgMS + "ms...");


      //
      // Delete
      //
      //as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nFiles; ii++){
        localInterface.delete(new ObjId("/test-13/" + ii + syncEach + syncPhase));
        if(syncEach){
          core.syncStateToDisk();
        }
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("Delete 2 byte file: " + avgMS + "ms...");


    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
    System.out.println("...Test 13 OK");
  }







 /** 
 *  test14() -- small write/read/delete performance test 
 **/ 
  protected void
  test14(boolean syncEach, boolean syncPhase, boolean bound){

    int nWrites = 10;
    int nFiles = 1;
    long start, end;
    int ii;
    String objId;
    byte b[] = {0};
    ImmutableBytes ib = new ImmutableBytes(b);
    System.out.print("Test14 [small write performance test " 
                     + (syncEach ? " SYNC_EACH" : " NO_SYNC_EACH") 
                     + (syncPhase ? " SYNC_PHASE" : " NO_SYNC_PHASE")
                     //+ (doSanity ? " DO_EXPENSIVE_SANITY_CHECKS" : " NO_DO_EXPENSIVE_SANITY_CHECKS")
                     + (bound ? " BOUND" : " UNBOUND")
                     + " ]...");
    BodyMsg bm;
  
    long ret;
    double avgMS;
    CounterVV vv = new CounterVV();
    //boolean oldDoSanity = ras.doExpensiveSanityChecks;
    //ras.doExpensiveSanityChecks = doSanity;

    try{

      /*
      // 
      // Update VV with this nodeID (outside of timing for fairness
      //
      bm = new BodyMsg(new ObjId("/test-14/foo"), 0, 1, -999, as, ib, false);
      ret = ras.applyBody(bm);
      */
    
      //
      // Write
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nWrites; ii++){
        /*
          bm = new BodyMsg(new ObjId("/test-14/bar" + syncEach + syncPhase + doSanity),
          ii, 1, -999, as, ib, false);
          ret = ras.applyBody(bm);
        */
        ret = localInterface.write(new ObjId("/test-14/bar" + syncEach + syncPhase), 
                                   ii, 1, ib, bound);
        assert(ret == 1);
        if(syncEach){
          core.syncStateToDisk();
        }
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nWrites;
      System.out.println("1 byte write: " + avgMS + "ms...");

      //
      // Overwrite
      //
      //as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      for(ii = 0; ii < nWrites; ii++){
        /*
          bm = new BodyMsg(new ObjId("/test-14/bar" + syncEach + syncPhase + doSanity),
          ii, 1, -999, as, ib, false);
          ret = ras.applyBody(bm);
        */
        ret = localInterface.write(new ObjId("/test-14/bar" + syncEach + syncPhase), 
                                   ii, 1, ib, bound);
        assert(ret == 1);
        if(syncEach){
          core.syncStateToDisk();
        }
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nWrites;
      System.out.println("1 byte overwrite: " + avgMS + "ms...");


      //
      // Read
      //
      start = System.currentTimeMillis();
      for(ii = 0; ii < nWrites; ii++){
        bm = localInterface.read(new ObjId("/test-14/bar" + syncEach + syncPhase ), ii, 1, true, true);
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nWrites;
      System.out.println("1 byte read: " + avgMS + "ms...");


      //
      // Delete
      //
      //as = new AcceptStamp(as.getLocalClock() + 1, new NodeId(9999999));
      start = System.currentTimeMillis();
      localInterface.delete(new ObjId("/test-14/bar" + syncEach + syncPhase));
      if(syncEach){
        core.syncStateToDisk();
      }
      if(syncPhase){
        core.syncStateToDisk();
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)nFiles;
      System.out.println("Delete 1000-byte, 1000-write file: " + avgMS + "ms...");
    }
    catch(Exception e){
      e.printStackTrace();
      assert(false);
    }

    //ras.doExpensiveSanityChecks = oldDoSanity;
    System.out.println("...Test 14 OK");

  }

 /** 
 *  test15() -- large read/write performance test 
 **/ 
  protected void
  test15(boolean bound){
    int maxFSize = 1000000;
    int minFSize = 1000000;
    int sizeIncrease = 10;
    int minWorkingSet = 1000000;
    int maxWorkingSet = 1000000;
    int iWorkingSet;
    int passesMaxSize = 2;
    int passesThisSize;
    int iPass, iFSize;
    int nFiles, iFile;
  
    BodyMsg bm;
    byte b[];
    ImmutableBytes ib;
    NodeId writer = new NodeId(4785675);
    long start, end;
    long msPerMicro = 1000;
    int ii;
    long got;
    ObjId obj;
    VV checkVV;
    long totMS;
    double bw;
    long cacheSize = Config.getCacheSizeBytes(core.getMyNodeId());

    assert(maxWorkingSet >= maxFSize);

    System.out.println("Test 15: Large file read/write performance...");
    System.out.println("   cache: " + cacheSize);

    try{
      for(iWorkingSet = minWorkingSet;
          iWorkingSet <= maxWorkingSet; 
          iWorkingSet *= sizeIncrease){
        System.out.print(".");
        passesThisSize = maxWorkingSet / iWorkingSet * passesMaxSize;
        for(iFSize = minFSize; iFSize <= maxFSize; iFSize *= sizeIncrease){
          nFiles = iWorkingSet / iFSize;
          if(nFiles == 0){
            assert(iWorkingSet < iFSize);
            continue;
          }
          assert(nFiles * iFSize == iWorkingSet); // So the comparison of times is fair
          b = new byte[iFSize];
          for(ii = 0; ii < iFSize; ii++){
            b[ii] = (byte)42;
          }
          ib = new ImmutableBytes(b);
          b = null;
          //
          // We create a new DB for each test b/c otherwise
          // we will fill up the disk pretty quickly...
          //
          /*
            selfTestCleanup(path); // Start from known state
            RandomAccessState ras = new RandomAccessState(path, cacheSize);
          */

          //core.cleanEnv(); //delete db files
          //
          // Write
          //
          start = System.currentTimeMillis();
          for(iPass = 0; iPass < passesThisSize; iPass++){
            for(iFile = 0; iFile < nFiles; iFile++){
              obj = new ObjId("/test15/" + iFile + "-" + iFSize);
              /*
                as = new AcceptStamp(localStamp++, writer);
                bm = new BodyMsg(obj, 0, iFSize, -999, as, ib, false);
                got = ras.applyBody(bm);
              */
              got = localInterface.write(obj, 0, iFSize, ib, bound);//note: set
              //bound to true or false can tell
              // the relative cost for logging
            
              assert(got == iFSize);
            }
          }
//        checkVV = ras.getVV();
          try{
            core.syncStateToDisk();
          }
          catch(Exception e){
            e.printStackTrace();
            assert(false);
          }
          end = System.currentTimeMillis();
          totMS = end - start;
          bw = ((double)(iFSize * nFiles * passesThisSize))/((double)(totMS * msPerMicro));
          System.out.println("  Test15 " + 
                             (bound ? "BOUND " : "UNBOUND ") 
                             + "WRITE wss: " + iWorkingSet
                             + " passes: " + passesThisSize
                             + " fSize: " + iFSize  
                             + " Time: " +  totMS + " ms."
                             + " BW: " + bw + " MB/s");

          System.out.print(",");
          //
          // Read.
          //
          start = System.currentTimeMillis();
          for(iPass = 0; iPass < passesThisSize; iPass++){
            for(iFile = 0; iFile < nFiles; iFile++){
              obj = new ObjId("/test15/" + iFile + "-" + iFSize);
              bm = localInterface.read(obj, 0, iFSize, true, true);
              assert(bm.getLength() == iFSize);
            }
          }
          end = System.currentTimeMillis();
          totMS = end - start;
          bw = ((double)(iFSize * nFiles * passesThisSize))/((double)(totMS * msPerMicro));
          System.out.println("  Test15 READ wss: " 
                             + iWorkingSet
                             + " passes: " + passesThisSize
                             + " fSize: " + iFSize  
                             + " Time: " +  totMS + " ms."
                             + " BW: " + bw + " MB/s");

/*
// 
// Clean up DB
//
ras.close();
selfTestCleanup(path);
*/
        }
      }
    }
    catch(IOException e){
      System.out.println("Exception in test 15 (check to see if disk is full?)");
      System.out.println(e.toString());
      e.printStackTrace();
      assert(false);
    }
    catch(Exception xxe){
      xxe.printStackTrace();
      assert(false);
    }
  
  }

 /** 
 *  random() -- generate random numbers for test 
 **/ 
  static Random rng = new Random(49957);
  protected static long
  random(long from, long to){
    assert(to > from);
    long r = rng.nextLong();
    if(r < 0){
      r = r * -1;
    }
    r = (r % (to - from)) + from;
    assert(r >= from);
    assert(r <= to);
    return r;
  }

 /** 
 *  Test Single Node Capacity for large writes 
 **/ 
  public void largeWritesTest(int fileSize, boolean bound){
    long fileNum = 0;
    try{
      Runtime rt = Runtime.getRuntime();
      while (true){
        byte b[] = new byte[fileSize];
        for(int ii = 0; ii < fileSize; ii++){
          b[ii] = (byte)42;
        }
        ImmutableBytes ib = new ImmutableBytes(b);
        b = null;
        long got = localInterface.write(new ObjId("/a"), 0, fileSize, ib, bound);
        assert got == fileSize;
        fileNum++;
        System.out.println("write " + fileNum + " files with size: " + fileSize); 
        
        System.out.println("Free Mem in JVM:" + rt.freeMemory() +"\n"
                           + "Total Mem in JVM:" + rt.totalMemory());
      }
    } catch (IOException e){
      System.out.println("Exception in test 15 (check to see if disk is full?)");
      System.out.println(e.toString());
      e.printStackTrace();
      assert(false);
    }catch(Exception xxe){
      xxe.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  
 **/ 
  public void maxFileSize(int fileSize, boolean bound){
    try{
      Runtime r = Runtime.getRuntime();
      Thread.sleep(10);
      System.err.println("1:" + (r.totalMemory() - r.freeMemory())); 
      byte b[] = new byte[fileSize];
      for(int ii = 0; ii < fileSize; ii++){
        b[ii] = (byte)42;
      }
      ImmutableBytes ib = new ImmutableBytes(b);
      b = null;
      System.gc();
      Thread.sleep(10);
      System.err.println("2:" + (r.totalMemory() - r.freeMemory())); 
      long got = localInterface.write(new ObjId("/a"), 0, fileSize, ib, bound);
      ib = null;
      System.gc();
      Thread.sleep(10);
      System.err.println("3:" + (r.totalMemory() - r.freeMemory())); 
      core.syncStateToDisk();
      Thread.sleep(10);
      System.gc();
      System.err.println("4:" + (r.totalMemory() - r.freeMemory())); 
      assert got == fileSize;
    } catch (IOException e){
      System.out.println("Exception in maxFileSize(check to see if disk is full?)");
      System.out.println(e.toString());
      e.printStackTrace();
      assert(false);
    }catch(Exception xxe){
      xxe.printStackTrace();
      assert(false);
    }
  }
    
  
 /** 
 *  Program start point 
 **/ 
  public static void
  main(String[] argv){
    boolean newCCdbg = false;
    if (argv.length <2){
      Env.printDebug("Usage: [configfile] [localNodeId] [SanityCheck]");
      System.exit(-1);
    }
    boolean cleanDb;
    if (argv.length == 2){
      cleanDb = false; //-- recover from disk
      URANode ura = new URANode(argv[0], 
                                new NodeId((new Long(argv[1])).longValue()),
                                Controller.NULL_CONTROLLER,
                                cleanDb);
      Env.printDebug("Waiting for the coordinator to start...");
      if(newCCdbg){
        System.err.println("about to send Awake Signal...");
      }
      ura.sendAwakeSignal();
      ura.populateLocalWrites(System.in);
    }else{//sanityCheck
      cleanDb = true; //-- cleanup dbs, start from fresh disk for testing
      URANode ura = new URANode(argv[0], 
                                new NodeId((new Long(argv[1])).longValue()),
                                Controller.NULL_CONTROLLER,
                                cleanDb);
      System.out.print("SanityCheck start...");
      if (argv[2].equalsIgnoreCase("SanityCheck")){
        ura.sanityCheck();
      }else if (argv[2].equalsIgnoreCase("maxFileSize")){
        assert(argv.length >= 5);
        boolean bound = false;
        if (new Integer(argv[4]).intValue()==1){
          bound = true;
        }
        ura.maxFileSize(new Integer(argv[3]).intValue(), bound);
      }else if (argv[2].equalsIgnoreCase("largeWrite")){
        assert(argv.length >= 5);
        boolean bound = false;
        if(new Integer(argv[4]).intValue()==1){
          bound = true;
        } 
        ura.largeWritesTest(new Integer(argv[3]).intValue(), bound);
      }
      System.out.println("Sanitycheck Succeeded"); 
      System.exit(0);
    }
  }
}

//---------------------------------------------------------------------------
/* $Log: URANode.java,v $
/* Revision 1.71  2008/01/23 20:35:27  nalini
/* added null random access state
/*
/* Revision 1.70  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
/* Revision 1.69  2007/09/10 23:52:22  zjiandan
/* upgrade to newest BerkeleyDB je version.
/*
/* Revision 1.68  2007/05/31 06:02:01  zjiandan
/* add AllPreciseSetsUnit
/*
/* Revision 1.67  2007/04/05 21:49:25  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.66  2007/03/15 21:58:31  dahlin
/* Added experimetn to test BW for subscribe for SOSP paper
/*
/* Revision 1.65  2007/03/11 04:16:54  zjiandan
/* Add timeout into read interface and expose acceptstamp in LocalInterface.write.
/*
/* Revision 1.64  2007/02/23 20:54:28  zjiandan
/* Fixed mem leak problems in NFS2Interface and some other bugs.
/* Andrew benchmark passed, performance still needs to tune.
/*
/* Revision 1.63  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.62  2007/01/05 01:18:41  nalini
/* support for sync with VV added
/*
/* Revision 1.61  2006/12/08 21:36:26  nalini
/* reverting to the old read interface
/*
/* Revision 1.59  2006/11/01 17:04:32  nayate
/* Added a parameter to control log sync-ing
/*
/* Revision 1.58  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.57  2006/06/13 16:35:48  dahlin
/* *** empty log message ***
/*
/* Revision 1.56  2005/10/13 00:24:25  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.55  2005/03/22 23:00:03  zjiandan
/* Changes for TactExpt.
/*
/* Revision 1.54  2005/03/15 21:15:34  zjiandan
/* Automatic GC checked in.
/*
/* Revision 1.53  2005/03/14 22:12:14  nayate
/* Added a new URANode constructor for experiments
/*
/* Revision 1.52  2005/03/07 21:00:08  lgao
/* Planet lab exp check-in
/*
/* Revision 1.51  2005/03/04 05:52:19  nayate
/* Minor changes
/*
/* Revision 1.50  2005/03/01 07:48:23  nayate
/* Modified for new code
/*
/* Revision 1.49  2005/03/01 07:45:26  nayate
/* Modified for new code
/*
/* Revision 1.47  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.46  2005/01/18 22:49:43  zjiandan
/* Rewrited Class Serialization for network deliveringto reduce the overall bandwidth.
/*

/* Revision 1.45  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.44  2004/11/02 22:24:33  zjiandan
/* add utility methods for core recovery self test.
/*
/* Revision 1.43  2004/10/22 20:46:56  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.42  2004/10/13 17:41:36  zjiandan
/* Initial implementation of PersistentLog tested with DataStore stubs.
/*
/* TBD: test recovery with integrated DataStore and RandomAccessState.
/*
/* Revision 1.41  2004/07/28 20:18:26  dahlin
/* encapsulated byte[] bodies in ImmutableBytes for safety
/*
/* Revision 1.40  2004/05/26 04:37:55  arun
/* *** empty log message ***
/*
/* Revision 1.39  2004/05/26 01:59:37  nayate
/* Added a log sync command
/*
/* Revision 1.38  2004/05/25 08:16:42  lgao
/* *** empty log message ***
/*
/* Revision 1.37  2004/05/25 06:30:04  nayate
/* Added code to register interest sets at program startup
/*
/* Revision 1.36  2004/05/25 01:51:34  ypraveen
/* changed SDIMS_SPANTREE_CONTROLLER name to SDIMS_CONTROLLER
/*
/* Revision 1.35  2004/05/24 10:37:33  arun
/* rectified some logic for updating ISStatus. Changes for dealing with bound invals
/* using sdims.
/*
/* Revision 1.34  2004/05/24 05:38:17  zjiandan
/* Add InvalIteratorFilter Class.
/*
/* Revision 1.33  2004/05/24 03:08:58  lgao
/* Add syncBodyCheck...
/*
/* Revision 1.32  2004/05/24 01:34:51  arun
/* added check in SDIMSController to check if inval is newer than existing body on store to prevent SDIMS from polluting its directory when it gets an inval for its own local write
/*
/* Revision 1.31  2004/05/23 23:54:13  ypraveen
/* Added SDIMSReadDir instead of Random in making SDIMS_SPANTREE_CONTROLLER
/*
/* Revision 1.30  2004/05/23 23:45:05  zjiandan
/* Add SyncBody
/*
/* Revision 1.29  2004/05/23 21:15:35  arun
/* informLocalWrite added to URANode, LocalInterface on local write.
/*
/* Revision 1.28  2004/05/23 10:30:34  nayate
/* Changed URANode to register "/*" as its interest set
/*
/* Revision 1.27  2004/05/23 05:28:30  ypraveen
/* Added a new option in makeController -- SDIMS_SPANTREE_CONTROLLER -- that uses sdims
/*
/* Revision 1.26  2004/05/22 02:42:37  nayate
/* Changes in blocking demand reads
/*
/* Revision 1.25  2004/05/21 13:37:20  nayate
/* Minor changes (removed some debugging information)
/*
/* Revision 1.24  2004/05/21 13:11:38  nayate
/* Fixed applyWrite bug
/*
/* Revision 1.23  2004/05/21 11:57:14  nayate
/* Added a blocking read call
/*
/* Revision 1.22  2004/05/21 11:03:03  lgao
/* Add a method to return current VV
/*
/* Revision 1.21  2004/05/21 06:25:01  zjiandan
/* Add read.
/*
/* Revision 1.20  2004/05/21 01:53:38  zjiandan
/* Add readBody script handler.
/*
/* Revision 1.19  2004/05/20 19:27:49  nayate
/* Added testing prints
/*
/* Revision 1.18  2004/05/20 04:10:57  dahlin
/* Passes SDIMSController spanning tree self test
/*
/* Revision 1.17  2004/05/19 22:53:35  zjiandan
/* Return null when sync finish.
/*
/* Revision 1.16  2004/05/19 22:38:41  zjiandan
/* Handle getAcceptStamp and Sync request scripts from Coordinator
/*
/* Revision 1.15  2004/05/19 20:31:12  nayate
/* Changed code to use stdin and stdout to communicate with the Coordinator
/*
/* Revision 1.14  2004/05/19 03:09:01  dahlin
/* Fixed default constructor for URANode
/*
/* Revision 1.13  2004/05/19 02:14:44  dahlin
/* Making progress on SDIMSController spanning test
/*
/* Revision 1.12  2004/05/18 08:54:25  nayate
/* Added a new "l" script command which makes this URANode to sleep
/*
/* Revision 1.11  2004/05/17 19:05:58  nayate
/* Changed URANode to parse scripts where the user does not specify write data
/*
/* Revision 1.10  2004/05/14 19:23:21  dahlin
/* Cleanup spanning tree logic; add update bodies
/*
/* Revision 1.9  2004/05/14 00:12:43  zjiandan
/* Changes for second experiments.
/*
/* Revision 1.8  2004/05/13 20:09:10  dahlin
/* When core has applied a bound inval, it tells originator of the write
/*
/* Revision 1.7  2004/05/13 01:13:19  nayate
/* Changed URANode to wait 5 seconds before starting to allow other structures
/* to get initialized
/*
/* Revision 1.6  2004/05/11 03:49:45  zjiandan
/* Made some changes to get the first version to work.
/*
/* Revision 1.5  2004/05/01 00:04:37  zjiandan
/* First Successful Integration Test!!:)
/*
/* Revision 1.4  2004/04/30 19:08:21  nayate
/* Fixed version
/*
/* Revision 1.3  2004/04/27 07:59:05  lgao
/* Initial implementation for UpdatePriorityQueue
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
