package code;

 /** 
 *  Unbind Message used to unbind BoundInval in log 
 *   -- propagated by log exchange same as invalidates 
 **/ 
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//reflection class used for workaround custom serialize private final fields
import java.lang.reflect.Field;
import java.lang.reflect.AccessibleObject;

// For testing
import java.io.ByteArrayOutputStream;
import java.io.ByteArrayInputStream;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

public final class UnbindMsg implements Externalizable, Immutable
{

  private static boolean doExpensiveSanityChecksForRead = Env.getDoExpensiveSanityChecks();
  private static boolean doExpensiveSanityChecksForWrite = Env.getDoExpensiveSanityChecks();
  private static boolean warnedExpensiveSanity = false;

  final static boolean debugging = false;
  private final ObjInvalTarget objId;
  private final AcceptStamp timestamp;
    
 /** 
 *  empty Constructor for serialization 
 **/ 
  public UnbindMsg(){
    
    this.objId = null; 
    
    this.timestamp = null;
  }

 /** 
 *  Constructor 
 **/ 
  public UnbindMsg(ObjInvalTarget objId_, AcceptStamp timestamp_){
    assert(objId_ instanceof Immutable);
    this.objId = objId_;
    assert(timestamp_ instanceof Immutable);
    this.timestamp = timestamp_; 
  }
  
 /** 
 *  Do sanity check for the custom object serialization 
 *  so that we don't forget updating the readObject and writeObject 
 *  when the object definition is changed 
 **/ 
  final private void doSanityCheck(){
    //System.out.println(UnbindMsg.class.getDeclaredFields().length);
    assert(UnbindMsg.class.getDeclaredFields().length == 7);

    //the fields we custom serialize remain the same
    Field[] f = new Field[2];  
      try{
        f[0] = UnbindMsg.class.getDeclaredField("objId");
        f[1] = UnbindMsg.class.getDeclaredField("timestamp");
                
        assert (f[0].getType().getName().endsWith("ObjInvalTarget")): f[0].getType().getName();
        assert (f[1].getType().getName().endsWith("AcceptStamp"));
        
      }catch(NoSuchFieldException ne){
        assert false;
      }
  }

 /** 
 *  Serialization -- optimization to reduce the pikles in the serialization  
 *                   and improve the performance -- removed from  
 *                   TaggedOutputStream to here 
 **/ 
  public void writeExternal(ObjectOutput out)
    throws IOException{

    if(doExpensiveSanityChecksForWrite && !warnedExpensiveSanity){
      Env.performanceWarning("UnbindMsg -- doExpensiveSanityChecks on");
      warnedExpensiveSanity = true;
    }

    if(doExpensiveSanityChecksForWrite){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      doExpensiveSanityChecksForWrite = false;
    }

    out.writeObject(this.getObjId().getPath());
    out.writeLong(this.getOffset());
    out.writeLong(this.getLength());
    out.writeLong(this.getTime());
    if(SangminConfig.forkjoin){
      out.writeObject(this.getNodeId()); 
    } else {
      out.writeLong(this.getNodeId().getIDint());
    }
    
  }

  
 /** 
 *  Serialization --  optimization to reduce the pikles in the serialization  
 *                   and improve the performance --originated from  
 *                   TaggedInputStream 
 **/ 
  public void readExternal(ObjectInput in)
    throws IOException,
    ClassNotFoundException{
    
    if(doExpensiveSanityChecksForRead){
      //
      // Don't forget to update this when you change the definition
      // of the object
      //
      //System.out.println(PreciseInv.class.getDeclaredFields().length);
      doSanityCheck();
      //turn off
      doExpensiveSanityChecksForRead = false;
    }

    String objIdString = (String)(in.readObject());
    long  offset = in.readLong();
    long  length = in.readLong();
    long  localClock = in.readLong();
    
    //long  nodeIdInt = in.readLong();
    NodeId nodeId;
    if(SangminConfig.forkjoin){
      nodeId = (BranchID)in.readObject();
    } else {
      nodeId = new NodeId(in.readLong());
    }
      
    Field[] f = new Field[2];
      
      try{

        f[0] = UnbindMsg.class.getDeclaredField("objId");
        f[1] = UnbindMsg.class.getDeclaredField("timestamp");
        
      }catch(NoSuchFieldException ne){
        assert false;
      }
      try{
        AccessibleObject.setAccessible(f, true);
      } catch (SecurityException se){
        assert false;
      }
      try{
        f[0].set(this, new ObjInvalTarget(new ObjId(objIdString),
                                          offset,
                                          length));
        f[1].set(this, new AcceptStamp(localClock, nodeId));
        
      }catch(IllegalArgumentException ie){
        assert false;
      }catch(IllegalAccessException iae){
        assert false;
      }
      
      try{
        AccessibleObject.setAccessible(f, false);
      } catch (SecurityException se){
        assert false;
      }
  }

 /** 
 *  Clone this object 
 **/ 
  public Object clone(){
    assert(this instanceof Immutable);
    return this;
  }
    
 /** 
 *  Get NodeId 
 **/ 
  public NodeId getNodeId(){
    return timestamp.getNodeId();
  }

 /** 
 *  Get Time 
 **/ 
  public long getTime(){
    return timestamp.getLocalClock();
  }
  
 /** 
 *  Get AcceptStamp 
 **/ 
  public AcceptStamp getAcceptStamp(){
    return timestamp;
  }
 	
 /** 
 *  Get InvalTarget 
 **/ 
  public ObjInvalTarget getInvalTarget(){
    return objId;
  }

 /** 
 *  Get ObjId 
 **/ 
  public final ObjId getObjId() {
    return objId.getObjId();
  }

 /** 
 *  getOffset() -- return the offset 
 **/ 
  public final long
  getOffset(){
    return objId.getOffset();
  }

 /** 
 *  getLength() -- return the length 
 **/ 
  public final long
  getLength(){
    return objId.getLength();
  }


 /** 
 *  Return true if the passed-in object is "equal" to this UnbindMsg 
 **/ 
  public boolean
  equals(Object obj){
    boolean result = false;
    UnbindMsg pi = null;

    
    if(obj instanceof UnbindMsg){
      pi = (UnbindMsg)obj;
      result = (this.objId.equals(pi.objId) &&
                this.timestamp.equals(pi.timestamp));
    }
    return(result);
  }

  
	
 /** 
 *  Return a string representation 
 **/ 
  public final String toString(){
    String str = null;
    str = "UnbindMsg:(";
    str += this.objId;
    str += ", ";
    str += this.timestamp;
    str += ")";
    return str;
  }
 
}

//---------------------------------------------------------------------------
/* $Log: UnbindMsg.java,v $
/* Revision 1.13  2006/09/13 02:44:40  zjiandan
/* move unit test to XUnit.java for junit test.
/*
/* Revision 1.12  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/*
/* Revision 1.11  2006/04/04 15:59:59  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.10  2005/07/18 05:10:23  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.9  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.8  2004/10/13 17:41:36  zjiandan
/* Initial implementation of PersistentLog tested with DataStore stubs.
/*
/* TBD: test recovery with integrated DataStore and RandomAccessState.
/*
/* Revision 1.7  2004/07/28 14:27:35  dahlin
/* Added sanity checks for immutable objects
/*
/* Revision 1.6  2004/05/24 21:16:24  arun
/* minor corrections to make unbind messages with sdims work
/*
/* Revision 1.5  2004/04/26 20:59:51  zjiandan
/* Finished UpdateLog and partially tested(InterestSet intersect not complete yet
/* waiting for the local interface to finish design of InterstSet).
/* TodoList: Test Parallel InvalIterators and Inserters.
/*
/* Revision 1.4  2004/04/23 17:57:43  zjiandan
/* Finished and Tested
/*
/* Revision 1.3  2004/04/21 17:37:41  zjiandan
/* Miner Change to make UpdateLog.java compile successfully.
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
