package code;

 /** 
 *  Stores update messages waiting to be applied 
 **/ 
import java.util.HashMap;
import java.util.TreeMap;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.util.Map;
import java.util.Iterator;
import java.util.Set;
import java.util.LinkedList;
import java.io.PrintStream;

public class UpdateBuffer
{
 /** 
 *  Constants 
 **/ 
  public final static int FAIL_ON_FULL = 1;
  public final static int STOP_ON_FULL = 2;
  public final static int LRU_REPLACE_ON_FULL = 3;
  public final static long INFINITE_BUFFER_BYTES = 0;

 /** 
 *  Data members 
 **/ 
  private long maxSizeBytes;
  private int gcPolicy;
  private HashMap table;
  private long sizeBytes;

 /** 
 *  Constructor 
 **/ 
  public
  UpdateBuffer(long newMaxSizeBytes, int newGCPolicy){
    this.maxSizeBytes = newMaxSizeBytes;
    this.table = new HashMap();
    this.sizeBytes = 0;
    this.gcPolicy = newGCPolicy;
    assert((this.gcPolicy >= UpdateBuffer.FAIL_ON_FULL) &&
           (this.gcPolicy <= UpdateBuffer.LRU_REPLACE_ON_FULL));
    assert (this.gcPolicy != UpdateBuffer.LRU_REPLACE_ON_FULL) :
      "Unimplemented";
  }

 /** 
 *  Add this msg to the buffer and handle possible overflow 
 **/ 
  public synchronized boolean
  add(BodyMsg bodyMsg){
    NodeId nodeId = null;
    AcceptStamp as = null;
    TreeMap treeMap = null;
    boolean added = false;
    LinkedList msgList = null;

    if((this.sizeBytes + bodyMsg.getBody().getLength() <= this.maxSizeBytes) ||
       (this.maxSizeBytes == UpdateBuffer.INFINITE_BUFFER_BYTES)){
      if(this.maxSizeBytes != UpdateBuffer.INFINITE_BUFFER_BYTES){
        this.sizeBytes += bodyMsg.getBody().getLength();
      }
      as = bodyMsg.getAcceptStamp();
      nodeId = as.getNodeId();
      treeMap = (TreeMap)this.table.get(nodeId);
      if(treeMap == null){
        treeMap = new TreeMap();
        this.table.put(nodeId, treeMap);
      }
      msgList = (LinkedList)treeMap.get(new Long(as.getLocalClock()));
      if(msgList == null){
        msgList = new LinkedList();
        treeMap.put(new Long(as.getLocalClock()), msgList);
      }
      assert(!msgList.contains(bodyMsg));
      msgList.addLast(bodyMsg);
      added = true;
    }else{
      added = false;
      if(this.gcPolicy == FAIL_ON_FULL){
        assert(false);
      }
    }
    return(added);
  }

 /** 
 *  Remove and return a list of all objects from the buffer that have a 
 *  lower (or equal) accept stamp than endVV 
 *  
 *  Note: This method should be called when applying the corresponding 
 *  invalidate at its end time 
 **/ 
  public LinkedList
  remove(AcceptVV endVV){
    VVIterator vvIter = null;
    Object token = null;
    NodeId nodeId = null;
    long maxStamp = 0;
    TreeMap treeMap = null;
    Long stampLong = null;
    BodyMsg bodyMsg = null;
    LinkedList msgList = null;
    LinkedList removedBodies = null;

    removedBodies = new LinkedList();
    vvIter = endVV.getIterator();
    while(vvIter.hasMoreElements()){
      token = vvIter.getNext();
      nodeId = endVV.getServerByIteratorToken(token);
      maxStamp = endVV.getStampByIteratorToken(token);
      treeMap = (TreeMap)this.table.get(nodeId);
      if(treeMap != null){
        assert(!treeMap.isEmpty());
        try{
          stampLong = (Long)treeMap.firstKey();
          while(stampLong.longValue() <= maxStamp){
            msgList = (LinkedList)treeMap.remove(stampLong);
            while(!msgList.isEmpty()){
              bodyMsg = (BodyMsg)msgList.removeFirst();
              if(this.maxSizeBytes != UpdateBuffer.INFINITE_BUFFER_BYTES){
                this.sizeBytes -= bodyMsg.getBody().getLength();
              }
              assert(this.sizeBytes >= 0);
              removedBodies.addLast(bodyMsg);
            }
            stampLong = (Long)treeMap.firstKey();
          }
        }catch(NoSuchElementException e){
          // Map is empty
          assert(treeMap.isEmpty());
          this.table.remove(nodeId);
        }
      }
    }
    return(removedBodies);
  }

 /** 
 *  Return true if there is any entry in the buffer that has a timestamp 
 *  less than endVV 
 **/ 
  public synchronized boolean
  containsAny(AcceptVV endVV){
    boolean found = false;
    VVIterator vvIter = null;
    Object token = null;
    NodeId nodeId = null;
    long maxStamp = 0;
    Long stampLong = null;
    TreeMap treeMap = null;

    found = false;
    vvIter = endVV.getIterator();
    while((vvIter.hasMoreElements()) && (!found)){
      token = vvIter.getNext();
      nodeId = endVV.getServerByIteratorToken(token);
      maxStamp = endVV.getStampByIteratorToken(token);
      treeMap = (TreeMap)this.table.get(nodeId);
      if(treeMap != null){
        assert(!treeMap.isEmpty());
        stampLong = (Long)treeMap.firstKey();
        found = (stampLong.longValue() <= maxStamp);
      }
    }
    return(found);
  }

 /** 
 *  Return true if the buffer has every entry invalidated by this 
 *  MultiObjPreciseInv 
 **/ 
  public boolean
  containsAll(MultiObjPreciseInv mopi){
    boolean result = false;
    boolean found = false;
    AcceptStamp as = null;
    NodeId nodeId = null;
    TreeMap treeMap = null;
    LinkedList msgList = null;
    MultiObjInvalTarget moit = null;
    ObjInvalTarget oit = null;
    BodyMsg bodyMsg = null;

    result = false;
    as = mopi.getAcceptStamp();
    nodeId = as.getNodeId();
    treeMap = (TreeMap)this.table.get(nodeId);
    if(treeMap != null){
      msgList = (LinkedList)treeMap.get(new Long(as.getLocalClock()));
      if(msgList != null){
        result = true;
        moit = mopi.getMultiObjInvalTarget();
        for(MOITIterator moitIter = moit.getIterator(true);
            moitIter.hasNext() && result;){
          oit = moitIter.getNext();
          found = false;
          for(ListIterator msgListIter = msgList.listIterator(0);
              msgListIter.hasNext() && (!found);){
            bodyMsg = (BodyMsg)msgListIter.next();
            found = oit.equals(bodyMsg.getObjInvalTarget());
          }
          result = found;
        }
      }
    }
    return(result);
  }

 /** 
 *  Print this class (used for testing) 
 **/ 
  public void
  print(PrintStream ps){
    Iterator it1 = null;
    Iterator it2 = null;
    Iterator it3 = null;
    Map.Entry entry1 = null;
    Map.Entry entry2 = null;
    Set entrySet1 = null;
    Set entrySet2 = null;
    Long stampLong = null;
    BodyMsg bodyMsg = null;
    TreeMap treeMap = null;
    LinkedList msgList = null;

    ps.println("maxSizeBytes = " + this.maxSizeBytes);
    ps.println("gcPolicy = " + this.gcPolicy);
    ps.println("sizeBytes = " + this.sizeBytes);
    entrySet1 = this.table.entrySet();
    for(it1 = entrySet1.iterator(); it1.hasNext();){
      entry1 = (Map.Entry)it1.next();
      ps.println("NodeId = " + (NodeId)entry1.getKey());
      treeMap = (TreeMap)entry1.getValue();
      for(it2 = treeMap.entrySet().iterator(); it2.hasNext();){
        entry2 = (Map.Entry)it2.next();
        stampLong = (Long)entry2.getKey();
        msgList = (LinkedList)entry2.getValue();
        for(it3 = msgList.listIterator(0); it3.hasNext();){
          bodyMsg = (BodyMsg)it3.next();
          ps.println(" " + stampLong + ":" + bodyMsg.getObjId());
        }
      }
    }
  }

 /** 
 *  Print this class (used for testing) 
 **/ 
  public String
  toString(){
    Iterator it1 = null;
    Iterator it2 = null;
    Iterator it3 = null;
    Map.Entry entry1 = null;
    Map.Entry entry2 = null;
    Set entrySet1 = null;
    Set entrySet2 = null;
    Long stampLong = null;
    BodyMsg bodyMsg = null;
    TreeMap treeMap = null;
    LinkedList msgList = null;
    String str = null;
    NodeId nodeId = null;

    /*
      ps.println("maxSizeBytes = " + this.maxSizeBytes);
      ps.println("gcPolicy = " + this.gcPolicy);
      ps.println("sizeBytes = " + this.sizeBytes);
    */
    str = "UpdateBuffer-->\n";
    entrySet1 = this.table.entrySet();
    for(it1 = entrySet1.iterator(); it1.hasNext();){
      entry1 = (Map.Entry)it1.next();
      //ps.println("NodeId = " + (NodeId)entry1.getKey());
      nodeId = (NodeId)entry1.getKey();
      treeMap = (TreeMap)entry1.getValue();
      for(it2 = treeMap.entrySet().iterator(); it2.hasNext();){
        entry2 = (Map.Entry)it2.next();
        stampLong = (Long)entry2.getKey();
        msgList = (LinkedList)entry2.getValue();
        for(it3 = msgList.listIterator(0); it3.hasNext();){
          bodyMsg = (BodyMsg)it3.next();
          str += ("  " + bodyMsg.getObjId() +
                  ":(" + stampLong +
                  ", " + nodeId + ")\n");
          //ps.println(" " + stampLong + ":" + bodyMsg.getObjId());
        }
      }
    }
    return(str);
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing UpdateBuffer.java...");
    UpdateBuffer.testAdd();
    UpdateBuffer.testContainsAny();
    UpdateBuffer.testRemove();
    UpdateBuffer.testContainsAll();
    UpdateBuffer.testAddMultiple();
    System.out.println("...Finished");
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testAdd(){
    ObjId objId1 = null;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    BodyMsg bodyMsg3 = null;
    AcceptStamp as1 = null;
    UpdateBuffer updateBuffer1 = null;
    boolean added = false;
    ImmutableBytes ib1 = null;
    ImmutableBytes ib2 = null;

    updateBuffer1 = new UpdateBuffer(50, UpdateBuffer.STOP_ON_FULL);
    as1 = new AcceptStamp(110, new NodeId(500));
    ib1 = new ImmutableBytes(new byte[49]);
    objId1 = new ObjId("/a");
    bodyMsg1 = new BodyMsg(objId1, 10, 100, as1, ib1, false);
    added = updateBuffer1.add(bodyMsg1);
    assert(added);

    ib2 = new ImmutableBytes(new byte[1]);
    bodyMsg2 = new BodyMsg(objId1, 100, 200, as1, ib2, false);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);

    bodyMsg3 = new BodyMsg(objId1, 200, 300, as1, ib2, false);
    added = updateBuffer1.add(bodyMsg3);
    assert(!added);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testContainsAny(){
    ObjId objId1 = null;
    ObjId objId2 = null;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    AcceptStamp as1 = null;
    AcceptStamp as2 = null;
    AcceptStamp[] asArray = null;
    ImmutableBytes ib1 = null;
    ImmutableBytes ib2 = null;
    UpdateBuffer updateBuffer1 = null;
    AcceptVV vv1 = null;
    boolean containedOne = false;
    boolean added = false;

    updateBuffer1 = new UpdateBuffer(100, UpdateBuffer.STOP_ON_FULL);
    as1 = new AcceptStamp(100, new NodeId(1));
    ib1 = new ImmutableBytes(new byte[50]);
    objId1 = new ObjId("/a");
    bodyMsg1 = new BodyMsg(objId1, 10, 100, as1, ib1, false);
    added = updateBuffer1.add(bodyMsg1);
    assert(added);

    // Test 1: Use the same accept stamp as that of the object
    asArray = new AcceptStamp[1];
    asArray[0] = new AcceptStamp(100, new NodeId(1));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(containedOne);

    // Test 2: Use an earlier accept stamp as that of the object
    asArray[0] = new AcceptStamp(99, new NodeId(1));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(!containedOne);

    // Test 3: Use a later accept stamp as that of the object
    asArray[0] = new AcceptStamp(101, new NodeId(1));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(containedOne);

    // Test 4: Use an incomparable accept stamp to that of the object
    asArray[0] = new AcceptStamp(101, new NodeId(2));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(!containedOne);

    // Test 5: Use a larger-dimension AcceptVV that contains the object stamp
    asArray = new AcceptStamp[2];
    asArray[0] = new AcceptStamp(100, new NodeId(1));
    asArray[1] = new AcceptStamp(200, new NodeId(2));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(containedOne);

    // Test 6: Use a larger-dimension AcceptVV
    // Add a second object
    // Make sure the second object is contained but not the first
    objId2 = new ObjId("/b");
    as2 = new AcceptStamp(100, new NodeId(2));
    ib2 = new ImmutableBytes(new byte[50]);
    bodyMsg2 = new BodyMsg(objId2, 10, 100, as2, ib2, true);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);
    asArray = new AcceptStamp[2];
    asArray[0] = new AcceptStamp(99, new NodeId(1));
    asArray[1] = new AcceptStamp(200, new NodeId(2));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(containedOne);

    // Test 7: Make sure neither the second nor the first object is contained
    // in the 2D VV
    asArray = new AcceptStamp[2];
    asArray[0] = new AcceptStamp(99, new NodeId(1));
    asArray[1] = new AcceptStamp(99, new NodeId(2));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(!containedOne);

    // Test 8: Make sure the first object is contained in the 2D VV
    asArray = new AcceptStamp[2];
    asArray[0] = new AcceptStamp(200, new NodeId(1));
    asArray[1] = new AcceptStamp(99, new NodeId(2));
    vv1 = new AcceptVV(asArray);
    containedOne = updateBuffer1.containsAny(vv1);
    assert(containedOne);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testRemove(){
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    UpdateBuffer updateBuffer1 = null;
    AcceptStamp[] asArray1 = null;
    AcceptStamp[] asArray2 = null;
    AcceptStamp[] asArray3 = null;
    AcceptVV endVV1 = null;
    AcceptVV endVV2 = null;
    AcceptVV endVV3 = null;
    LinkedList removedBodies = null;
    boolean added = false;

    bodyMsg1 = new BodyMsg(new ObjId("/a"),
                           10,
                           100,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[100]),
                           true);
    bodyMsg2 = new BodyMsg(new ObjId("/b"),
                           20,
                           200,
                           new AcceptStamp(100, new NodeId(2)),
                           new ImmutableBytes(new byte[200]),
                           false);

    updateBuffer1 = new UpdateBuffer(500, UpdateBuffer.STOP_ON_FULL);
    added = updateBuffer1.add(bodyMsg1);
    assert(added);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);

    // Test 1: Test remove() for both objects
    asArray1 = new AcceptStamp[2];
    asArray1[0] = new AcceptStamp(100, new NodeId(1));
    asArray1[1] = new AcceptStamp(100, new NodeId(2));
    endVV1 = new AcceptVV(asArray1);
    removedBodies = updateBuffer1.remove(endVV1);
    assert(removedBodies.size() == 2);
    assert(removedBodies.contains(bodyMsg1));
    assert(removedBodies.contains(bodyMsg2));

    // Test 2: Add both objects back and try removing just the first
    added = updateBuffer1.add(bodyMsg1);
    assert(added);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);
    asArray2 = new AcceptStamp[2];
    asArray2[0] = new AcceptStamp(50, new NodeId(1));
    asArray2[1] = new AcceptStamp(100, new NodeId(2));
    endVV2 = new AcceptVV(asArray2);
    removedBodies = updateBuffer1.remove(endVV2);
    assert(removedBodies.size() == 1);
    assert(removedBodies.contains(bodyMsg2));

    // Test 3: Add that object back and try removing the other
    added = updateBuffer1.add(bodyMsg2);
    assert(added);
    asArray3 = new AcceptStamp[2];
    asArray3[0] = new AcceptStamp(100, new NodeId(1));
    asArray3[1] = new AcceptStamp(99, new NodeId(2));
    endVV3 = new AcceptVV(asArray3);
    removedBodies = updateBuffer1.remove(endVV3);
    assert(removedBodies.size() == 1);
    assert(removedBodies.contains(bodyMsg1));

    // Test 4: Add that object back and try removing neither
    added = updateBuffer1.add(bodyMsg1);
    assert(added);
    asArray3 = new AcceptStamp[2];
    asArray3[0] = new AcceptStamp(99, new NodeId(1));
    asArray3[1] = new AcceptStamp(99, new NodeId(2));
    endVV3 = new AcceptVV(asArray3);
    removedBodies = updateBuffer1.remove(endVV3);
    assert(removedBodies.isEmpty());
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testContainsAll(){
    UpdateBuffer updateBuffer1 = null;
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    MultiObjPreciseInv mopi1 = null;
    MultiObjInvalTarget moit1 = null;
    ObjInvalTarget[] invs = null;
    ObjInvalTarget[] dels = null;
    boolean contains = false;
    boolean added = false;

    // Add two bodies
    updateBuffer1 = new UpdateBuffer(500, UpdateBuffer.STOP_ON_FULL);
    bodyMsg1 = new BodyMsg(new ObjId("/a"),
                           10,
                           100,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[100]),
                           true);
    bodyMsg2 = new BodyMsg(new ObjId("/b"),
                           20,
                           200,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[200]),
                           false);
    updateBuffer1 = new UpdateBuffer(500, UpdateBuffer.STOP_ON_FULL);
    added = updateBuffer1.add(bodyMsg1);
    assert(added);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);

    // Test 1: See if containsAll works when checking for only those bodies
    invs = new ObjInvalTarget[2];
    invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 100);
    invs[1] = new ObjInvalTarget(new ObjId("/b"), 20, 200);
    dels = new ObjInvalTarget[0];
    moit1 = new MultiObjInvalTarget(invs, dels);
    mopi1 = new MultiObjPreciseInv(moit1,
                                   new AcceptStamp(100, new NodeId(1)),
                                   new AcceptStamp(100, new NodeId(1)),
                                   false,
                                   new MOITBoundEntry[0]);
    contains = updateBuffer1.containsAll(mopi1);
    assert(contains);

    // Test 2: See if containsAll fails when checking a proper superset
    invs = new ObjInvalTarget[3];
    invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 100);
    invs[1] = new ObjInvalTarget(new ObjId("/b"), 20, 200);
    invs[2] = new ObjInvalTarget(new ObjId("/c"), 30, 300);
    dels = new ObjInvalTarget[0];
    moit1 = new MultiObjInvalTarget(invs, dels);
    mopi1 = new MultiObjPreciseInv(moit1,
                                   new AcceptStamp(100, new NodeId(1)),
                                   new AcceptStamp(100, new NodeId(1)),
                                   false,
                                   new MOITBoundEntry[0]);
    contains = updateBuffer1.containsAll(mopi1);
    assert(!contains);

    // Test 3: See if containsAll works when checking for a proper subset
    invs = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/a"), 10, 100);
    dels = new ObjInvalTarget[0];
    moit1 = new MultiObjInvalTarget(invs, dels);
    mopi1 = new MultiObjPreciseInv(moit1,
                                   new AcceptStamp(100, new NodeId(1)),
                                   new AcceptStamp(100, new NodeId(1)),
                                   false,
                                   new MOITBoundEntry[0]);
    contains = updateBuffer1.containsAll(mopi1);
    assert(contains);

    // Test 4: See if containsAll fails when checking for a non-overlapping set
    invs = new ObjInvalTarget[1];
    invs[0] = new ObjInvalTarget(new ObjId("/d"), 50, 400);
    dels = new ObjInvalTarget[0];
    moit1 = new MultiObjInvalTarget(invs, dels);
    mopi1 = new MultiObjPreciseInv(moit1,
                                   new AcceptStamp(100, new NodeId(1)),
                                   new AcceptStamp(100, new NodeId(1)),
                                   false,
                                   new MOITBoundEntry[0]);
    contains = updateBuffer1.containsAll(mopi1);
    assert(!contains);
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testAddMultiple(){
    BodyMsg bodyMsg1 = null;
    BodyMsg bodyMsg2 = null;
    BodyMsg bodyMsg3 = null;
    UpdateBuffer updateBuffer1 = null;
    AcceptVV endVV1 = null;
    AcceptStamp[] asArr1 = null;
    LinkedList removedBodies = null;
    boolean added = false;

    // Test 1:
    //   Add two body messages with the same AcceptStamp
    //   See if we can remove both using that stamp
    bodyMsg1 = new BodyMsg(new ObjId("/a"),
                           10,
                           100,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[100]),
                           true);
    bodyMsg2 = new BodyMsg(new ObjId("/b"),
                           20,
                           200,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[200]),
                           false);

    updateBuffer1 = new UpdateBuffer(500, UpdateBuffer.STOP_ON_FULL);
    added = updateBuffer1.add(bodyMsg1);
    assert(added);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);

    asArr1 = new AcceptStamp[1];
    asArr1[0] = new AcceptStamp(100, new NodeId(1));
    endVV1 = new AcceptVV(asArr1);
    removedBodies = updateBuffer1.remove(endVV1);
    assert(removedBodies.size() == 2);
    assert(removedBodies.contains(bodyMsg1));
    assert(removedBodies.contains(bodyMsg2));

    // Test 2:
    //   Add two body messages with the same AcceptStamp
    //   Add another body with a different stamp
    //   See if we can remove all three using a single endVV
    bodyMsg1 = new BodyMsg(new ObjId("/a"),
                           10,
                           100,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[100]),
                           true);
    bodyMsg2 = new BodyMsg(new ObjId("/b"),
                           20,
                           200,
                           new AcceptStamp(100, new NodeId(1)),
                           new ImmutableBytes(new byte[200]),
                           false);
    bodyMsg3 = new BodyMsg(new ObjId("/c"),
                           30,
                           300,
                           new AcceptStamp(100, new NodeId(2)),
                           new ImmutableBytes(new byte[300]),
                           false);

    updateBuffer1 = new UpdateBuffer(800, UpdateBuffer.STOP_ON_FULL);
    added = updateBuffer1.add(bodyMsg1);
    assert(added);
    added = updateBuffer1.add(bodyMsg2);
    assert(added);
    added = updateBuffer1.add(bodyMsg3);
    assert(added);

    asArr1 = new AcceptStamp[2];
    asArr1[0] = new AcceptStamp(100, new NodeId(1));
    asArr1[1] = new AcceptStamp(100, new NodeId(2));
    endVV1 = new AcceptVV(asArr1);
    removedBodies = updateBuffer1.remove(endVV1);
    assert(removedBodies.size() == 3);
    assert(removedBodies.contains(bodyMsg1));
    assert(removedBodies.contains(bodyMsg2));
    assert(removedBodies.contains(bodyMsg3));
  }
}
