package code;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.Condition;

import java.io.*;
import javax.naming.TimeLimitExceededException;

import code.branchDetecting.BranchID;
import code.security.SangminConfig;

import com.sleepycat.je.DatabaseException;

public class UpdateLog{
  /*
   * Maintain a persistent log of all updates applied
   * to local state.
   * 
   * This is about my 5th redesign of this thing...
   * (see abandoned-log-design.txt, abandoned-log-design-2.txt)
   *
   * Some of the tricky bits
   *  (1) complex logic for merging a new imprecise invalidation
   *      with existing log -- new one might be committed or
   *      uncommitted and overlap a bunch of existing committed
   *      or uncommitted things already in log; 
   *  (2) furthermore as described in the updates of 3.30.2004, 
   *      this merge logic *does* need to "decompose" overlapping
   *      invalidations so that we don't lose causality information. 
   *      (Merging invalidations is a union operation, we now need
   *      a corresponding intersection operation for decomposing
   *      overlapping invalidations I1 and I2:
   *              New1, New2, New3 = split(I1, I2)
   *                [w/o loss of generality, assume I1.start <= I2.start]
   *                if I1 overlap I2 in time (e.g., I1.end >= I2.start)
   *                  New1 = {I1.start, I2.start, I1.InterestSet}
   *                  New2 = {I2.start, MIN(I1.end, I2.end),
   *                          I1.InterestSet INTERSECT I2.InterestSet}
   *                  New3 = (I2.end < I1.end
   *                         ? {I2.end, I1.end, I1.InterestSet}
   *                         : {I1.end, I2.end, I2.InterestSet}
   *  (3) Maintaining a single global log is tricky because
   *      of the complex ordering rules across multi-writer
   *      imprecise invalidations potentially with a startCSN,
   *      endsCSN, startVV, and endVV
   *  (4) Searching for the "next" invalidation to transmit
   *      by an inval iterator may be expensive -- if we have per
   *      writer logs (rather than a more complex global log)
   *      then figuring out which item to send might require O(#nodes)
   *      work to examine #nodes per-writer logs; also, once
   *      we pick the per-writer log, if it is just a linked list
   *      than scanning down the list to the next item to send
   *      might take O(# items on list) work. In the worst case,
   *      sending n updates from a system with m nodes could
   *      take work (O n^2 m). On the other hand, we could
   *      maintain a globally ordered list, but that is much more complex; 
   *      and we could maintain some sort of search index for each
   *      node's updates, but that is also more complex...
   *  (5) Synchronization is a bit tricky -- in several designs we tried
   *      to cache state about the log in the InvalIterator, but
   *      doing so is dangerous because then that state can be affected
   *      both by the INvalIterator thread and by new inserts into
   *      the log
   *  (6) What to do when we receive an invalidation "in the past"
   *      compared to the current position of an invalitorator. 
   *      Send it? Ignore it?
   *  (7) How to coordinate between the in-memory log data structures 
   *      and the on-disk persistent log? 
   *
   * Our solution is to keep in memory 1 log of committed writes in CSN 
   * order and #nodes logs of per-writer, uncommitted writes in acceptStart
   * order. 
   *
   *  (1,2) The complexity of each of these logs and of the decomposition 
   *      logic is reduced compared to trying to maintain a global log 
   *      where each element might have multiple start and end times.
   *      Because we "decompose" writes on the fly, a node's log
   *      always the most detailed view the node could possibly have
   *      given what it has seen so far --> when the node forwards this
   *      log to someone else, no information is lost.
   *
   *  (3) We don't maintain a global log
   *
   *  (4) *PERFORMANCE IS A LIMITATION* -- after getting bogged down 
   *      so many times with designs that we spent hour or days on
   *      before realizing they were incorrect, we have decided to
   *      make *correctness and simplicity the overriding concerns*
   *      So, we *do not* try to maintain a globally ordered unified
   *      log in addition to the per-writer logs and we *do not* provide
   *      auxiliary index structures (e.g., a tree) to speed up searching
   *      for the right place to start searching a linked list. 
   *      When an InvalIterator calls GetNext(csn, VV), the call may
   *      first have to scan through #nodes entries to find the the right
   *      per-writer log to send from and then it may need to scan through
   *      #writes entries in that per-writer log to find the right record
   *      to send. Fix this later if it becomes an issue, but *get something
   *      working correctly first!*
   *
   *  (5) The simplified design simplifies synchronization -- the InvalIterator's
   *      getNext() call and the invalRecver's applyInval() call are
   *      now made from "above" this layer through this monitor.
   *      (Some earlier designs had this layer call the InvalIterator
   *      with news about updates to this layer's state; the problem was
   *      that we then either risked deadlock or unsafe sharing
   *      (since invalIterator and this layer call each other).
   * 
   *      One other point on synchronization -- all of the inval records
   *      in the log *must* be immutable since we hand references to them
   *      out to the iterator. 
   *
   *  (6) Each invalIterator has a current position in each log; if a new
   *      invalidation arrives and lands earlier than ii's current
   *      position (e.g., ii earlier sent an imprecise inval and now an
   *      overlapping precise inval arrives with "more details"),
   *      ii *ignores this new information*. In earlier designs, I foolishly
   *      tried to send these refined  views in some cases but I now realize
   *      (a) if we're not really careful about this, we could get endless
   *      forwarding loops of the same invalidation across nodes and (b)
   *      sending this new information doesn't really help the receiver 
   *      (the receive can update its log to be more accurate, but its 
   *      local views of the InterestSets can't be updated with this out-of-sequence
   *      data. So, the rule is -- each InvalIterator confines itself to
   *      sending a causal stream; if a node wants to "refine" something
   *      it saw in the past (e.g., to handle a demand read), then that node
   *      needs to set up some *new* invalIterators with the appropriate interest
   *      sets and appropriate start times.
   *
   *  (7) The above discussion applies *soley* to the in-memory view of the
   *      data. The on-disk persistent log is *much* simplier -- it is just
   *      a redo log of all changes we have applied to the in-memory view.
   *      Whenever a new message arrives, append it to the on-disk log before
   *      returning from applying it to the in-memory version. The constructor
   *      simply does a log-replay of the on-disk log into the in-memory version.
   */

 
  String logPath;
  PersistentLog checkpoint;

  HashMap<NodeId, SingleWriterLogUncommitted> perWriterLogs;//per-writer-log
  HashSet<InMemLogIterator> activeIters;//used to track iterator pointers
  MailBoxList mailBoxList;//used to notify OutgoingConnectionWorker of new invalidations
  protected long currentLamportClock;
  protected NodeId myNodeId;
  
  protected CounterVV currentVV;
  CounterVV diskOmitVV; //any invalidate <= diskOmitVV is gced
  //note the omitVV is inclusive

  CounterVV inMemOmitVV;//any invalidate <= inMemOmitVV is gced

  
    
  long unbindCounter;
  long debargoCounter;
  HashMap<Long, UnbindMsg> recentUnbinds;
  HashMap<Long, DebargoMsg> recentDebargo;

  private static final boolean vverbose = false;
  private static final boolean verbose = false;
  private static final boolean dbg = false;

  //for garbage collect
  int iiCount; //the current live InvalIterator on this log
  static final int MAX_UNBINDS_SAVED = 1000; //for Garbage Collect old unbinds
  static final int MAX_DEBARGOS_SAVED = 1000;//for Garbage Collect old debargos
  private final static boolean dbgBoundHops = false;

  private final static boolean dbgii = false;
  private final static String dbgiimsg = "DBG UpdateLog inval iterator:";

  private final static boolean dbgGC = false;
  final static boolean debugging = false;
  private final static boolean dbgLock = false;

  private final static boolean dbgkeyset = false;
  private final static String dbgksmsg = "DBG UpdateLog keyset:";
  private final static boolean dbgInv = false;
  private final static boolean logging = true;
  private final static boolean doExpensiveSanityChecks = false;
  // Note that it is OK to "lose" an unbind
  // and forget to forward it to someone.
  // (Though doing so may hurt performance).


  public final ReentrantLock lock = new ReentrantLock();
  protected final Condition reduceOE = lock.newCondition();
  protected final Condition moreInval = lock.newCondition();
  protected final Condition advanceCVV = lock.newCondition();

  //the following members are for garbage collection
  private long lwm;
  private long hwm;
  private long ldisk;
  private long hdisk;

  private long totMemCharge;
  private long totDiskCharge;
  private final Condition gcDone = lock.newCondition();
  private final Condition gcStart = lock.newCondition();
  
  private final int GC_MAX_TRY_BEFORE_WARN = 10;
  private final static int MAX_OUTGOINGSTREAMS = 100;
  //
  //it is possible that the garbage collection can't make it
  //for the MAX_MEM or MAX_DISK goal because BoundVV is too small
  //have the GCWorker at most try GC_MAX_TRY_BEFORE_CLAIM_FAIL
  //times before throwing the GCFailedException
  //
  private final int GC_MAX_TRY_BEFORE_CLAIM_FAIL = 100;

  private final double DEFAULT_GC_DISK_RATE = 0.5;
  private final double DEFAULT_GC_MEM_RATE = 0.5; 
  
  private final long DEFAULT_MEM_FORCE_SLACK = 0;
  private final double DEFAULT_START_FORCE_RATE = 0.9;//force-start
  private final long DEFAULT_DISK_FORCE_SLACK = 0;
  private Thread gcWorker;
  private boolean timeToDie;
  
  private static int pwcount = 0;//performance warning counts
  private static final int pwmax = 10;//max performance warning counts
  private static boolean tbdPrinted = false;

  
 /** 
 *  Constructor 
 **/ 
  public UpdateLog(String logPath_, NodeId myNodeId_, boolean noSync) 
    throws IOException, CantRecoverCheckpointException, 
    SecurityException, NoSuchEntryException{
    
    Env.performanceWarning("UpdateLog uses simple" 
                           +"[worst case O(#nodes * #updates)] "
                           + "algorithm for searching for "
                           + "next eligible inval to apply.");
    this.myNodeId = myNodeId_;
    this.mailBoxList = new MailBoxList(MAX_OUTGOINGSTREAMS);
    Config.getLogDiskSizeBytes(this.myNodeId);
    logPath = logPath_;
    checkpoint = new PersistentLog(logPath, 
                                   Config.getCacheSizeBytes(this.myNodeId),
                                   false,
                                   noSync);
    diskOmitVV = new CounterVV(checkpoint.getOmitVV());
    inMemOmitVV = new CounterVV(diskOmitVV);
    currentVV = new CounterVV(diskOmitVV);

    if ( verbose )
      Env.dprintln(verbose, "UpdateLog::UpdateLog diskOmitVV="
                 + diskOmitVV.toString());

    perWriterLogs = new HashMap<NodeId, SingleWriterLogUncommitted>();
    activeIters = new HashSet<InMemLogIterator>();
    
    currentLamportClock = 0;

     //for Garbage Collection
    hwm = Config.getLogMemSizeBytes(this.myNodeId) - DEFAULT_MEM_FORCE_SLACK;
    lwm = Math.round(hwm * DEFAULT_START_FORCE_RATE);
    
    hdisk = Config.getLogDiskSizeBytes(this.myNodeId) - DEFAULT_DISK_FORCE_SLACK;
    ldisk = Math.round(hdisk * DEFAULT_START_FORCE_RATE);

    assert hwm > lwm;
    assert hdisk > ldisk;

    totMemCharge = 0;
    totDiskCharge = checkpoint.diskSize();
    gcWorker = null;
    
    
    if ( dbgGC )
      Env.dprintln(dbgGC, "lwm: " + lwm 
                 +" hwm: " + hwm
                 +" hdisk: " + hdisk
                 +" ldisk: " + ldisk
                 +" totMemCharge: " + totMemCharge
                 +" totDiskCharge: " + totDiskCharge);
    
    timeToDie = false;
    
    unbindCounter = 0;
    debargoCounter = 0;
    recentUnbinds = new HashMap<Long, UnbindMsg>();
    recentDebargo = new HashMap<Long, DebargoMsg>();
    LogIterator li = null;
    //
    //recover persistentlog into in-mem per-writer logs
    //
    try{
      if ( verbose )
	Env.dprintln(verbose, "UpdateLog::UpdateLog() iterating redo log");
      li = checkpoint.getLogIterator();
      Object nextEvent = li.getNext();
      while (nextEvent != null){
        tbdPrinted = Env.tbd(tbdPrinted,
                             "As the in-mem per-writer-log is garbage collected"
                             + " independently from persistent log, the recovery"
                             + " of in-mem per-writer-log from persistent log" 
                             + " might construct a in-mem per-writer-log larger than"
                             + " the constrained size for the mem-log, because"
                             + " at the recovery time, the garbage collection is not"
                             + " started yet. One option would be starting the "
                             + " the garbage collection before the recovery, but the"
                             + " problem is that garbage collection needs datastore"
                             + " checkpoint vv and UpdateLog which doesn't exist yet"
                             + " at the in-mem-recovery time as in current design.");
        if (nextEvent instanceof GeneralInv){
          if ( vverbose )
	    Env.dprintln(vverbose, "UpdateLog::UpdateLog scanning through events: " 
                       + ((GeneralInv)nextEvent).toString()
                       + " [currentVV="
                       + currentVV.toString()
                       + "]");
          try{
            applyInvalInternal((GeneralInv)nextEvent);
          }catch (CausalOrderException ce){
            ce.printStackTrace();
            assert false; 
          }
        } else if(nextEvent instanceof UnbindMsg){
          
          applyUnbindInternal((UnbindMsg)nextEvent);
        } else if(nextEvent instanceof DebargoMsg){
          assert( nextEvent instanceof DebargoMsg);
          applyDebargoInternal((DebargoMsg)nextEvent);
        }else{
          assert false;//should never go here.
        }
        nextEvent = li.getNext();
      }
    } catch (DatabaseException de){
      if(li != null){
          
        try{
          li.close();
        } catch (Exception ae){
          de.printStackTrace();
          System.err.println("Nested Exception while closing Logiterator: ");
          ae.printStackTrace();
          assert false;
        }
        li = null;
      }
      de.printStackTrace();
      assert false;
    }
    finally{
      if(li != null){
        try{
          li.close();
        }catch (Exception ae){
          ae.printStackTrace();
          assert false;
        }
        li = null;
      }
    }
    // at this point all in mem per-writer log should be there
    // and all the corresponding currentVV, currentLamportClock
    // should be there correctly.
	
    if(doExpensiveSanityChecks){
      verifyInvariants();
    }
    iiCount = 0;
    //iilist = new Vector();
  }

 /** 
 *  startGCWorker() -- kick off the worker thread. Must be called 
 *                   after initialization. 
 **/ 
  public void startGCWorker(DataStore store){
    lock.lock();
    try{
      gcWorker = new GCWorker(this, store);
      gcWorker.start();
    }
    finally{
      lock.unlock();
    }
  }

 /** 
 *  killGCWorker() -- Tell the worker thread to die. 
 **/ 
  public void killGCWorker(){
    lock.lock();
    try{
      timeToDie = true;
      gcStart.signal();
    }
    finally{
      lock.unlock();
    }
  }

 /** 
 *  waitForWork() -- called by the worker thread 
 **/ 
  public void waitForWork() throws InterruptedException{
    lock.lock();
    try{
      while(!timeToDie && totMemCharge <= lwm
            && totDiskCharge <= ldisk){
        gcStart.awaitUninterruptibly();
      }
      if(timeToDie){
        throw new InterruptedException("GCWorker should die now.");
      }
      return; 
    }
    finally{
      lock.unlock();
    }
  }

 /** 
 *  doneWithWork() -- called by the worker thread when done 
 **/ 
  public void doneWithWork(AcceptVV boundVV)
    throws IOException{
    
    if ( dbgLock || dbgGC )
      Env.dprintln(dbgLock||dbgGC, "UpdateLog::doneWithWork about to start");
    lock.lock();
    long start = System.currentTimeMillis();
    
    if ( dbgLock || dbgGC )
      Env.dprintln(dbgLock||dbgGC, "UpdateLog::doenWithWork start");
    try{
      int count = 0;
      do{
	if (count > GC_MAX_TRY_BEFORE_WARN){
	  Env.warn("gc tried " + count + " times to gc");
	}

	if (count > GC_MAX_TRY_BEFORE_CLAIM_FAIL){
	  throw new IOException("GC worker has to give up after trying "+count+" times"
				+ " with totMemCharge=" + totMemCharge 
				+ " totDiskCharge=" + totDiskCharge
				+ " hwm=" + hwm + " hdisk=" + hdisk);
	}
	
	if(totMemCharge > lwm){
	  long totCutted = truncateMem(DEFAULT_GC_MEM_RATE, 
				       boundVV);
	  System.gc(); 
	  totMemCharge -= totCutted;
	  if ( dbgGC )
	    Env.dprintln(dbgGC, "GCed Mem: " + totCutted 
		       + " totMem after GC: " + totMemCharge);
	}
	
	if(totDiskCharge > ldisk){
	  truncateDisk(DEFAULT_GC_DISK_RATE,
		       boundVV);
	  totDiskCharge = checkpoint.diskSize();
	  if (dbgGC)
	    Env.dprintln(dbgGC, "totDisk after GC: " + totDiskCharge);
	  
	}
	count++;
	
      }while((totMemCharge > hwm) || (totDiskCharge > hdisk));
      
      gcDone.signalAll();
      long end = System.currentTimeMillis();
      if ( dbgGC || dbgLock ) 
	Env.dprintln(dbgGC||dbgLock, "UpdateLog::doneWithWork takes " + (end-start));
      
    }
    
    finally{
      lock.unlock();
    }
  }

  public AcceptStamp commit(InvalTarget targetSet, 
                            AcceptStamp targetAS,
                            AcceptStamp realStamp){
    AcceptStamp ret = this.commitInternal(targetSet, targetAS, realStamp);
    notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates
    return ret;
  }
  
  public AcceptStamp commitInternal(InvalTarget targetSet, 
                                    AcceptStamp targetAS,
                                    AcceptStamp realStamp){
    lock.lock();
    try{
      GeneralInv inv;
      long checkLamportClock = currentLamportClock;
      
      AcceptStamp acceptStamp = new AcceptStamp(currentLamportClock, myNodeId);
      
      
      inv = new CommitInv(targetSet,
                          acceptStamp,
                          targetAS,
                          realStamp);
      
      
      try{
        applyInvalAtomic(inv);
      }catch(CausalOrderException e){
        e.printStackTrace();
        assert false;
        //Assert.affirm(false, "Causal order exception on local write " 
        //              + e.toString());
      }
      
      if(doExpensiveSanityChecks){
        verifyInvariants();
        assert(checkLamportClock < currentLamportClock);
      }
      if(dbgInv){
        Env.dprintln(dbgInv, "write " + acceptStamp 
                     + " @" + System.currentTimeMillis());
      }
      return acceptStamp;
    }finally{
      lock.unlock();
    }
  }
  
  
 /** 
 *  Write multiple objects atomically (include Order Error Bound parameter) 
 *  This is a wrap that's not thread-safe and notify  
 *  OutgoingConnectionMailbox 
 **/ 
  public AcceptStamp
  writeMulti(MultiWriteEntry[] mwe,
             AcceptStamp realStamp,
             long targetOE,
             boolean embargoed){
    AcceptStamp ret = this.writeMultiInternal(mwe, 
                                              realStamp, 
                                              targetOE,
                                              embargoed);
    notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates
    return ret;
  
  }
  
 /** 
 *  Write multiple objects atomically (include Order Error Bound parameter) 
 **/ 
  public AcceptStamp
  writeMultiInternal(MultiWriteEntry[] mwe,
             AcceptStamp realStamp,
             long targetOE,
             boolean embargoed){
    if ( dbgLock )
      Env.dprintln(dbgLock, "UpdateLog::writeMulti about to enter");
    lock.lock();
    if ( dbgLock )
      Env.dprintln(dbgLock, "UpdateLog::writeMulti start");
    try{
      AcceptStamp ret = null;

      
      SingleWriterLogUncommitted l =  null;
      l = (SingleWriterLogUncommitted)perWriterLogs.get(myNodeId);
      if(l != null){
        while(l.getOrderError(currentVV.getMinTimeStamp()) >= targetOE){
          try{
            if(dbg){
              System.err.println("Block because of Order Error violation" +
                                 "currOE: " +
                                 l.getOrderError(currentVV.getMinTimeStamp()) +
                                 "targetOE: " + targetOE);
            }
            reduceOE.await();
          }catch (InterruptedException ie){
            Env.sprinterrln("write waiting for targetOE is interrupted !");
          }
        }
        if(dbg){
          System.err.println("stop (blocking because of Order Error " +
                             "violation)\ncurrOE: " +
                             l.getOrderError(currentVV.getMinTimeStamp()));
        }
      }

      ret = this.writeMultiInternal(mwe, realStamp, embargoed);
      if ( dbgLock )
	Env.dprintln(dbgLock, "UpdateLog::writeMulti finish.");
      
      return ret;
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  add Order Error Bound for local write 
 **/ 
  public AcceptStamp
  write(ObjId objId,
        long offset,
        long len,
        double priority,
        AcceptStamp realStamp,
        ImmutableBytes buffer,
        boolean bound,
        long maxBoundHops,
        long targetOE,
        boolean embargoed){
    AcceptStamp ret = this.writeInternal(objId, 
                                         offset, 
                                         len, 
                                         priority, 
                                         realStamp, 
                                         buffer, 
                                         bound, 
                                         maxBoundHops, 
                                         targetOE, 
                                         embargoed);
    this.notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates
    return ret;
  }
  
 /** 
 *  add Order Error Bound for local write 
 **/ 
  public AcceptStamp
  writeInternal(ObjId objId,
        long offset,
        long len,
        double priority,
        AcceptStamp realStamp,
        ImmutableBytes buffer,
        boolean bound,
        long maxBoundHops,
        long targetOE,
	boolean embargoed){
    lock.lock();
    try{
      AcceptStamp ret = null;

      
      SingleWriterLogUncommitted l =  null;
      l = (SingleWriterLogUncommitted)perWriterLogs.get(myNodeId);
      if(l != null){
        while(l.getOrderError(currentVV.getMinTimeStamp()) >= targetOE){
          try{
            if(dbg){
              System.err.println("Block because of Order Error violation " +
                                 "currOE: " +
                                 l.getOrderError(currentVV.getMinTimeStamp()) +
                                 "targetOE: " + targetOE);
            }
            reduceOE.await();
          }catch (InterruptedException ie){
            Env.sprinterrln("write waiting for targetOE is interrupted !");
          }
        }
        if(dbg){
          System.err.println("stop (blocking because of Order Error " +
                             "violation)\ncurrOE: " +
                             l.getOrderError(currentVV.getMinTimeStamp()));
        }
      }

      ret = this.writeInternal(objId,
                       offset,
                       len,
                       priority,
                       realStamp,
                       buffer,
                       bound,
                       maxBoundHops,
                       embargoed);
      return ret;
    }finally{
      lock.unlock();
    }
    
  }

  // following fields are for write throughput measurement
  protected int count = 0;
  protected Timer timer;
  protected TimeoutHandler timeoutHandler;
  protected long start = 0, end = 0;

  class TimeoutHandler extends TimerTask{
    long start = 0;
    public TimeoutHandler(long startTime){
      start = startTime;
    }
    
    public void run(){
      long end = System.currentTimeMillis();
      System.out.println("## Write Throughput is :" 
          + (count*1000/(end-start))
          + "per Second");
    }
  }

 /** 
 *  Create and apply BoundInv for local write 
 **/ 
  public AcceptStamp
  write(ObjId objId,
        long offset,
        long len,
        double priority,
        AcceptStamp realStamp,
        ImmutableBytes buffer,
        boolean bound,
        long maxBoundHops,
        boolean embargoed){
    
    if(SangminConfig.writeThroughPutMeasure){
      if(count==0){
        timer = new Timer();
        timeoutHandler = new TimeoutHandler(System.currentTimeMillis());       
        timer.schedule(timeoutHandler, SangminConfig.writeThroughputExpDuration);
        //schedule a timer
      }
    }
    
    AcceptStamp ret = this.writeInternal(objId, 
                                 offset, 
                                 len, 
                                 priority, 
                                 realStamp, 
                                 buffer, 
                                 bound, 
                                 maxBoundHops, 
                                 embargoed);
    this.notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates    

    if(SangminConfig.writeThroughPutMeasure){
      count++;
    }
        
    return ret;
  }
  
 /** 
 *  Create and apply BoundInv for local write 
 **/ 
  public AcceptStamp
  writeInternal(ObjId objId,
        long offset,
        long len,
        double priority,
        AcceptStamp realStamp,
        ImmutableBytes buffer,
        boolean bound,
        long maxBoundHops,
	boolean embargoed){
    lock.lock();
    try{
      GeneralInv inv;
      long checkLamportClock = currentLamportClock;
      
      AcceptStamp acceptStamp = new AcceptStamp(currentLamportClock, myNodeId);
      
      if(bound){
        assert(SangminConfig.securityLevel == SangminConfig.NONE);
        inv = new BoundInval(objId,
                             offset,
                             len,
                             acceptStamp, 
                             buffer,
                             priority,
                             realStamp,
                             maxBoundHops,
                             embargoed);
      }else{
        inv = new PreciseInv(new ObjInvalTarget(objId, offset, len),
                             acceptStamp,
                             realStamp,
                             embargoed);
      }
      
      try{
        applyInvalAtomic(inv);
      }catch(CausalOrderException e){
        e.printStackTrace();
        assert false;
        //Assert.affirm(false, "Causal order exception on local write " 
        //              + e.toString());
      }
      
      if(doExpensiveSanityChecks){
        verifyInvariants();
        assert(checkLamportClock < currentLamportClock);
      }
      if(dbgInv){
        Env.dprintln(dbgInv, "write " + acceptStamp 
                     + " @" + System.currentTimeMillis());
      }
      return acceptStamp;
    }finally{
      lock.unlock();
    }
   
  }

 /** 
 *  Create and apply BoundInv for local write 
 **/ 
  public AcceptStamp
  writeMulti(MultiWriteEntry[] mwe, AcceptStamp realStamp, boolean embargoed){
    AcceptStamp ret = this.writeMultiInternal(mwe, realStamp, embargoed);
    this.notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates
    return ret;
  }
 /** 
 *  Create and apply BoundInv for local write 
 **/ 
  public AcceptStamp
  writeMultiInternal(MultiWriteEntry[] mwe, AcceptStamp realStamp, boolean embargoed){
    if ( dbgLock )
      Env.dprintln(dbgLock, "UpdateLog::writeMulti about to enter");
    
    lock.lock();
    if ( dbgLock )
      Env.dprintln(dbgLock, "UpdateLog::writeMulti start");
    try{
      GeneralInv inv;
      long checkLamportClock = currentLamportClock;

      AcceptStamp acceptStamp = new AcceptStamp(currentLamportClock, myNodeId);

      inv = new MultiObjPreciseInv(mwe, acceptStamp, realStamp, embargoed);
      try{
        applyInvalAtomic(inv);
      }catch(CausalOrderException e){
        e.printStackTrace();
        assert false;
        //Assert.affirm(false, "Causal order exception on local write " 
        //              + e.toString());
      }
      
      if(doExpensiveSanityChecks){
        verifyInvariants();
        assert(checkLamportClock < currentLamportClock);
      }
      if(dbgInv){
        Env.dprintln(dbgInv, "write " + acceptStamp 
                     + " @" + System.currentTimeMillis());
      }
      if ( dbgLock ) 
	Env.dprintln(dbgLock, "UpdateLog::writeMulti finish");
      return acceptStamp;
    }finally{
      lock.unlock();
    }
   
  }
    
 /** 
 *  Create and apply DeleteInv for local delete call 
 **/ 
  public AcceptStamp 
  delete(ObjId path){
    AcceptStamp ret = this.deleteInternal(path);
    this.notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates
    return ret;
  }
 /** 
 *  Create and apply DeleteInv for local delete call 
 **/ 
  public AcceptStamp 
  deleteInternal(ObjId path){
    lock.lock();
    try{
      long checkLamportClock = currentLamportClock;
      
      AcceptStamp acceptStamp = new AcceptStamp(currentLamportClock, 
                                                myNodeId);
      //currentLamportClock++;  done in applyInvalInternal
      DeleteInv del = new DeleteInv(path, acceptStamp);
      
      try{
        applyInvalAtomic(del);
      }catch(Exception e){
        e.printStackTrace();
        assert false;
        //Assert.affirm(false, "Causal order exception on local write " 
        //              + e.toString());
      }

      if(doExpensiveSanityChecks){
        verifyInvariants();
        assert(checkLamportClock < currentLamportClock);
      }
      return acceptStamp;
    }finally{
      lock.unlock();
    }

  }

 /** 
 *  apply general invalidate message into local logs 
 **/ 
  public void
  applyInval(GeneralInv inv)
    throws CausalOrderException{
    AcceptVV oldCVV = this.getCurrentVV();
    this.applyInvalAtomic(inv);
    if(!oldCVV.includes(this.getCurrentVV())){//something new
      this.notifyMailBoxOfNewUpdates();//notify OutgoingConnection of new updates
    }
  }
 /** 
 *  apply general invalidate message into local logs 
 **/ 
  public void
  applyInvalAtomic(GeneralInv inv)
    throws CausalOrderException{
    lock.lock();
    try{
      try{
        checkpoint.append(inv); //We know there will be no exceptions with this
                                //inval and we are about to start changing 
                                //local state.
        totDiskCharge = checkpoint.diskSize();
        //invoke garbage collection if necessary
        if(totDiskCharge >ldisk){
          gcStart.signal();
          
          if ( dbgGC )
	    Env.dprintln(dbgGC, "signal gc disk with charge: " 
                       + totDiskCharge + " log.currentVV: " 
                       + this.currentVV);
            
          
        }
        while(totDiskCharge > hdisk){
          if(pwcount < pwmax){
            Env.performanceWarning("LogGC payCharge -- " 
              +"caller blocks b/c gc not done yet");
            pwcount++;
            if(pwcount == pwmax){
              Env.performanceWarning("Future LogGC warnings suppressed");
            }
          }
          gcDone.awaitUninterruptibly();
        }
      }catch(DatabaseException de){
        de.printStackTrace();
        assert false:" While node " + myNodeId + " applying inv=" + inv;
        //Assert.affirm(false, "DatabaseException while appending inval " 
        //            + de.toString());
      }
      if(logging){
        applyInvalInternal(inv); 
      }
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  apply invalidate into local logs 
 **/ 
  private void
  applyInvalInternal(GeneralInv recvInv)
    throws CausalOrderException{
    GeneralInv inv = recvInv;
    lock.lock();
    try{

      
      boolean doBroadcast = false;
      long minBefore = this.currentVV.getMinTimeStamp(); 
      // 
      // Note: The GeneralInv message can be a precise invalidation,
      // an imprecise invalidation, or a bound invalidation (e.g., 
      // a precise invalidation *bound* to the message). 
      // In the third case, don't forget to mark the message
      // as "bound" in the log and to update the locally stored
      // copy. Also note -- no need to send update to subscribers
      // since it is bound, we will send the update with the inval
      // (though we might want to enqueue the update to subscribers
      // when we get an unbind message, though that risks sending
      // the same body twice, as well...perhaps track to whom
      // we have sent bound updates...

      // Note: before we do anything, we will check if it is a
      // bound invalidation. 
      // If it is a bound inval and the
      //  maxBoundHops == Long.MAX_VALUE => do nothing
      //  maxBoundHops > 0 =>decrement it by 1
      //  maxBoundHops = 0 => we change it to a PreciseInval
      //
      if ( dbgBoundHops )
	Env.dprintln(dbgBoundHops, "Recv Inval " + inv);

      if(recvInv instanceof BoundInval) {
        if ( dbgBoundHops )
	  Env.dprintln(dbgBoundHops, "inval is bound");
        BoundInval bInv = (BoundInval) recvInv;
        long maxBoundHops = bInv.getMaxBoundHops();
        if(maxBoundHops == Long.MAX_VALUE) {
          // do nothing
        }
        else if(maxBoundHops > 0){
          // decrement by 1
          inv = new BoundInval(bInv.getObjInvalTarget(),
                               bInv.getAcceptStamp(),
                               bInv.getBody(),
                               bInv.getPriority(),
                               maxBoundHops-1);
        } else if(maxBoundHops == 0) {
        	assert(SangminConfig.securityLevel == SangminConfig.NONE);
          // change it to a Precise Inval
          inv = new PreciseInv(bInv.getObjInvalTarget(),
                               bInv.getAcceptStamp(),
                               bInv.getRTAcceptStamp(),
                               bInv.isEmbargoed());
        }
      }

      // do the same thing for multiObjPreciseInv
      if(recvInv instanceof MultiObjPreciseInv) {
        MultiObjPreciseInv mopInv = (MultiObjPreciseInv) recvInv;
        if (mopInv.hasBound()){
          if ( dbgBoundHops )
	    Env.dprintln(dbgBoundHops, "inval is bound");        
          long maxBoundHops = mopInv.getMaxBoundHops();
          if(maxBoundHops == Long.MAX_VALUE) {
            // do nothing
          }
          else if(maxBoundHops > 0){
            // decrement by 1
            inv = new MultiObjPreciseInv(mopInv.getMultiObjInvalTarget(),
                                         mopInv.getAcceptStamp(),
                                         mopInv.getRTAcceptStamp(),
                                         mopInv.isEmbargoed(),
                                         maxBoundHops - 1,
                                         mopInv.getBoundEntriesDangerous());

        } else if(maxBoundHops == 0) {
          // change it to a Precise Inval
            inv = mopInv.cloneUnbound();
          }
        }
      }
                            
      if ( dbgBoundHops )
	Env.dprintln(dbgBoundHops, "Going to Apply " + inv);

      if(doExpensiveSanityChecks){
        verifyInvariants();
      }
      
      //
      // ignore if already omitted by in-mem-log
      // 
      if (inv.isIncludedBy(inMemOmitVV))
        return;
      
      assert(!inv.isCommitted());
      //
      // Previous commits that we have already seen take precedence over
      // this inval, so we chop of its tail at commitVV and
      // apply the rest to the per-writer logs
      //
      
      //
      // Any sanity checks on input need to be done here. Once we
      // start changing local state, then the full change must succeed.
      //
      //any error checks needed?
      
      //
      // Input makes sense -- make the change!
      //
      
      
      // inval and we are about to start changing local state.
      //foreach (nodeId in inv.getStartVV){
      VVIterator startVVI = inv.getStartVV().getIterator();
      Object token = null;
      NodeId nodeId = null;
      
      while (startVVI.hasMoreElements()){
        token = startVVI.getNext();
        nodeId = inv.getStartVV().getServerByIteratorToken(token);
        
        try{
          
          long thisStart = inv.getStartVV().getStampByServer(nodeId);
          long thisEnd = inv.getEndVV().getStampByServer(nodeId);
          if(thisEnd <= AcceptStamp.BEFORE_TIME_BEGAN) continue; //trivial invalidate
          long omittedStart = -1;
          //truncate the inval by omittedStart
          try{
            omittedStart = inMemOmitVV.getStampByServer(nodeId);
          }catch (NoSuchEntryException nsee){
            omittedStart = -1;
          }
          if(omittedStart > thisEnd){
            continue;
          }

          if(omittedStart > thisStart) {
            thisStart = omittedStart;
          }
          assert thisEnd >= thisStart;
          
          SingleWriterInval uncommittedPart = 
            (SingleWriterInval)inv.getOneWriterSubsetFrom(nodeId,
                                                          thisStart);
          assert uncommittedPart != null;
          SingleWriterLogUncommitted l =  
            (SingleWriterLogUncommitted)perWriterLogs.get(nodeId);
          long oldMax;
          if (l != null){
            //Env.printDebug("l.getMinCounter()=" + l.getMinCounter());
            //Env.printDebug("thisStart=" + thisStart);
            //Env.printDebug("uncommittedPart" + uncommittedPart);
            
            oldMax = l.getMaxCounter();
          } else { //if no per writer log for this node Id, create one
            l = new SingleWriterLogUncommitted(nodeId);
            addSingleWriterLog(nodeId, l);
            if(dbgkeyset){
              Env.inform(dbgksmsg + " applyinvalinternal add to keyset " 
                         + " added node: " + nodeId.toString()
                         + " resulting keyset " 
                         + perWriterLogs.keySet().toString());
            }
            oldMax = l.getMaxCounter();
            assert(oldMax == -1);
          }
          
          long newMemCharge = l.merge((GeneralInv)uncommittedPart);
          totMemCharge += newMemCharge;
          
          //invoke garbage collection if necessary
          if(totMemCharge > lwm){
            gcStart.signal();
            if ( dbgGC )
	      Env.dprintln(dbgGC, "signal gc Mem with charge: " 
                         + totMemCharge + " log.currentVV: " 
                         + this.currentVV);
            
            
          }
          while(totMemCharge > hwm){
            if(pwcount < pwmax){
              Env.performanceWarning("LogGC payCharge -- " 
                + "caller blocks b/c gc not done yet");
              pwcount++;
              if(pwcount == pwmax){
                Env.performanceWarning("Future LogGC warnings suppressed");
              }
            }
            gcDone.awaitUninterruptibly();
          }
        
          if(l.getMaxCounter() > oldMax){
            doBroadcast = true;
          }
          if(doExpensiveSanityChecks){
            l.verifyInvariants();
          }
          //update local states
          
          currentVV.advanceTimestamps(inv.getEndVV());
          //System.out.println("DEBUG: counterVV advanced" + currentVV);
          
          if (inv.getEndVV().getMaxTimeStamp() >= currentLamportClock ){
            currentLamportClock = inv.getEndVV().getMaxTimeStamp()+1;
          }	
          
          
        }catch(NoSuchEntryException e){
          //die.
          //we need to carefully check the input and if there is a problem
          //throw the exception (and ignore this inval) *before* we start 
          //changing our local state. 
          Env.printDebug("no such entry Exception:" +e);
          e.printStackTrace();
          System.exit(-1);		
        }
      }//while
      
      if(minBefore < this.currentVV.getMinTimeStamp()){//more inval committed
        reduceOE.signalAll(); //wake up waiting writes blocked for Order Error 
      }

      if(doBroadcast){//cvv is advanced
        // wake up waiting invalIterators calling getNext 
        // and threads blocks waiting for tentative applied, i.e. sync to a
        // specific CVV.
        moreInval.signalAll();
        advanceCVV.signalAll();
      }
      
      if(doExpensiveSanityChecks){
        verifyInvariants();
      }
      
    }finally{
      lock.unlock();
    }
  }
    
  //
  // Called by invalIterators. As noted above, current algorithm
  // is simple, but potentially inefficient
  //
  // If we have an unbind message whose local counter exceeds
  //   unbindCount, return it
  // If we have a committed write whose csn exceeds csn,
  //   return it
  // If we have one or more uncommitted writes whose
  //   accept stamp exceeds the corresponding accept stamp
  //   in vv, return the one with the *lowest* accept stamp

 /** 
 *  Return a GeneralInval or an UnbindMsg 
 **/ 
  public Object 
  getNext(long unbindCount, VV vv, long timeout)
    throws TimeLimitExceededException, OmittedVVException, IOException{
    lock.lock();
    if(timeToDie){
      throw new IOException("UpdateLog time-to-die");
    }
    try{
      return getNext(unbindCount, MAX_UNBINDS_SAVED, vv, timeout);
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  Return a GeneralInval or an UnbindMsg or an DebargoMsg 
 **/ 
  public Object 
  getNext(long unbindCount, long debargoCount, VV vv, long timeout)
    throws TimeLimitExceededException, OmittedVVException, IOException{
    boolean dbgMe = false;
    
    lock.lock();
    
    try{
      long wakeTime = System.currentTimeMillis() + timeout;
      if(wakeTime <0){
	wakeTime = Long.MAX_VALUE;
      }
      if(timeToDie){
	throw new IOException("UpdateLog time-to-die");
      }
      NodeId n = nextCausal(unbindCount, debargoCount, vv);
      if ( dbgMe ){
	Env.dprintln(dbgMe, "nextCausal: "+ unbindCount + " ," + debargoCount
		     +" ," +vv + " updateLog=" + this.toString()
		     + " \n ---- return " + n);
      }
      
      NodeId invalidNodeId = new NodeId(NodeId.INVALID_NODE_ID); 
      if(SangminConfig.forkjoin)
        invalidNodeId = new BranchID(invalidNodeId.getIDint());
      
      while((System.currentTimeMillis() < wakeTime) 
            && n.equals(invalidNodeId)){ 
        // Nothing available and not reach timeout yet --> wait and try again

	timeout = wakeTime - System.currentTimeMillis();
        if (timeout <1){
          timeout = 1;
        }
	try{
	  moreInval.awaitNanos(timeout*1000);//timeout is a millous unit
        }catch(InterruptedException ie){
	  //ignore
	}
	if(timeToDie){
	  throw new IOException("UpdateLog time-to-die");
	}
        n = nextCausal(unbindCount, debargoCount, vv);
        if ( dbgMe ){
	  Env.dprintln(dbgMe, "nextCausal: "+ unbindCount + " ," + debargoCount
		       +" ," +vv + " updateLog=" + this.toString()
		       + " \n ---- return " + n);
	}
        if(dbgii){
          Env.inform(dbgiimsg + " * while() nextCausal returns * "
                     + "node: " 
                     + n.toString());
        }
        
      }
      
      if(dbgii){
        Env.inform(dbgiimsg + " * exit while() * "
                   + "node: " 
                   + n.toString());
      }
      
      
      if(n.equals(invalidNodeId)){
        
        assert(System.currentTimeMillis() >= wakeTime);
	//System.out.println("wakeTime =" + wakeTime);
        throw new TimeLimitExceededException();
      }
      
      NodeId unbind_pseudo_nodeId = new NodeId( NodeId.UNBIND_PSEUDO_NODE);
      NodeId debargo_pseudo_nodeId = new NodeId( NodeId.DEBARGO_PSEUDO_NODE);
      if(SangminConfig.forkjoin){
        unbind_pseudo_nodeId = new BranchID(unbind_pseudo_nodeId.getIDint());
        debargo_pseudo_nodeId = new BranchID(debargo_pseudo_nodeId.getIDint());
      }
      
      if(n.equals(unbind_pseudo_nodeId)){
        UnbindMsg u = null;
        
        while((u == null)&&(unbindCount < unbindCounter-1)){
          unbindCount++;
          u = (UnbindMsg)recentUnbinds.get(new Long(unbindCount));
        }
        assert(u!=null);
        //nextCausal shouldn't have returned UNBIND_PSEUDO_NODE
        return (UnbindMsg) u.clone();
      }

      if(n.equals(debargo_pseudo_nodeId)){
        DebargoMsg u = null;
        
        while((u == null)&&(debargoCount < debargoCounter-1)){
          debargoCount++;
          u = (DebargoMsg)recentDebargo.get(new Long(debargoCount));
        }
        assert(u!=null);
        
        return (DebargoMsg) u.clone();
      }
      if(dbgii){
        Env.inform(dbgiimsg + " * getNext() found new * "
                   + "from node: " 
                   + n.toString());
      }
      
      SingleWriterLogUncommitted l =  null;
      l = (SingleWriterLogUncommitted)perWriterLogs.get(n);
      assert(l != null);
      GeneralInv i = null;
      long start;
      
      try{
        if(dbgii){
          Env.inform(dbgiimsg + " * getNext() has l * "
                     + "from node: " 
                     + n.toString());
        }
        if (!vv.containsNodeId(n)){//n is an unknown nodeId to vv
          start = l.getMinCounter()-1;
        } 
        else {
          start = vv.getStampByServer(n);
        }
        
        if(dbgii){
          Env.inform(dbgiimsg + " * getNext() has start * "
                     + start);
        }
        
        i = l.getNextByStart(start);
        
        if(dbgii){
          Env.inform(dbgiimsg + " * getNext() got i to return * "
                     + (i==null?"<NULL>":i.toString()));
        }
        assert(i != null);
        assert(i instanceof SingleWriterInval);
        assert(!i.isCommitted()); 
        assert(!vv.containsNodeId(n) 
               ||((SingleWriterInval)i).getEnd() > vv.getStampByServer(n));
        if(dbgii){
          Env.inform(dbgiimsg + " * getNext() still has i to rtrn * "
                     + i.toString());
        }
        
	
      } catch (NoSuchEntryException e){
        assert false; //should never happen
      }
      //
      // We may have "inferred" some writes with empty 
      // interest set to "fill gaps"; skip those
      //
      //CHECK THAT WE DO NOT RETURN EMPTY INFERRED WRITES;
      //assert(!i.isDummy());
      
      if(dbgii){
        Env.inform(dbgiimsg + " * returning i * "
                   + i.toString());
      }
      
      return (GeneralInv)i.clone(); 
      // Passing out pointer to someone who won't be holding lock
    }finally{
      lock.unlock();
    }
  }
  
 /** 
 *  check next causal message for given (debargoCuont, unbindCount, vv) 
 *  return NodeId.DEBARGO_PSEUDO_NODE if there's Debargo messege>debargoCount 
 *  return NodeId.UNBIND_PSEUDO_NODE if there's Unbind messege > unbindCount 
 *  otherwise return a node id that has at least one causally-later 
 *   message 
 **/ 
  private NodeId
  nextCausal(long unbindCount, long debargoCount, VV vv)
    throws OmittedVVException{
    
    NodeId invalid_nodeId = new NodeId( NodeId.INVALID_NODE_ID);
    NodeId unbind_pseudo_nodeId = new NodeId( NodeId.UNBIND_PSEUDO_NODE);
    NodeId debargo_pseudo_nodeId = new NodeId( NodeId.DEBARGO_PSEUDO_NODE);
    if(SangminConfig.forkjoin){
      invalid_nodeId = new BranchID(invalid_nodeId.getIDint());
      unbind_pseudo_nodeId = new BranchID(unbind_pseudo_nodeId.getIDint());
      debargo_pseudo_nodeId = new BranchID(debargo_pseudo_nodeId.getIDint());
    }
    
    
    if(dbgii){
      Env.inform(dbgiimsg + " * next causal called * "
                 + " count: " + unbindCount
                 + " debargo: " + debargoCount
                 + " vv: " + vv.toString()
                 + " currentVV: " + currentVV.toString()
                 + " perWriterLog keyset: "
                 + perWriterLogs.keySet().toString());
    }
    if(debargoCount < debargoCounter - 1){
      if(dbgii){
        Env.inform(dbgiimsg + " * return unbind * ");
      }
      return debargo_pseudo_nodeId;
    }else if (unbindCount < unbindCounter-1){
      if(dbgii){
        Env.inform(dbgiimsg + " * return unbind * ");
      }
      return unbind_pseudo_nodeId;
    }else{
      //
      // check if the current omitvv exceeds any part of the 
      // parameter vv here. If so, the garbage collector has
      // go beyond the current vv of this iterator. The stream
      // will no longer be causal after VV. any stream should
      // stop by getting the omittedVVException created here
      //
      if(this.inMemOmitVV.isAnyPartGreaterThan(vv)){
        System.err.println("!!!!!!this.inMemOmitVV: " + this.inMemOmitVV
                           + " invalIterater.vv: " + vv
                           + " log.currentVV:"+ this.currentVV);
        throw new OmittedVVException("Garbage collector outruns inval outgoing"
          + "stream. Stop stream!");
      }
      long min = Long.MAX_VALUE;
      NodeId winner = invalid_nodeId;

      assert perWriterLogs !=null : "perWriterLogs null";
      assert perWriterLogs.keySet() != null : "keyset is null";
      Iterator<NodeId> keyi = perWriterLogs.keySet().iterator();
      NodeId nodeId = null;
      SingleWriterLogUncommitted l = null;
	    
      while (keyi.hasNext()){
        do{
          nodeId = (NodeId) keyi.next();
          l = (SingleWriterLogUncommitted)perWriterLogs.get(nodeId);
          if(dbgii){
            Env.inform(dbgiimsg + " * keyi.hasNext returned * "
                       + "node: " + nodeId.toString()
                       + "maxCounter: " + l.getMaxCounter());
          }
        }while(keyi.hasNext() 
               && l.getMaxCounter() == 
               SingleWriterLogUncommitted.UNINITIALIZED_COUNTER);

        if(!keyi.hasNext()
           && l.getMaxCounter() == 
           SingleWriterLogUncommitted.UNINITIALIZED_COUNTER){
          continue; // Will leave loop

        }
		
        assert(l != null);
        assert(l.getNewest() != null); 
		
        if(dbgii){
          Env.inform(dbgiimsg + " * consider nodeId * "
                     + nodeId.toString()
                     + " nodeId min: " + l.getMinCounter()
                     + " overall min: " + min);
        }

        try{
          if (!vv.containsNodeId(nodeId)){ //this NodeId is not
            // known by the vv yet 
            if(dbgii){
              Env.inform(dbgiimsg + " * node not in VV * "
                         + " node: " 
                         + nodeId.toString());
            }
            if (l.getMinCounter() < min){  // I am lowest so far
              winner = nodeId;
              min = l.getMinCounter();
            }else if ((l.getMinCounter() == min) 
                      && (nodeId.compareTo(winner)==-1)){
              winner = nodeId;
            }
          }
          else{//vv contains this nodeId
            if(l.getMaxCounter() 
               > 
               vv.getStampByServer(nodeId)){ // Something new 
              if(dbgii){
                Env.inform(dbgiimsg 
                           + " * next causal found new * "
                           + "from node: " 
                           + nodeId.toString());
              }

              long mystart;
              mystart = ((SingleWriterInval)
                         (l.getNextByStart(vv.getStampByServer(nodeId)))).getStart();
              if (mystart < min){  // I am lowest so far
                winner = nodeId;
                min = mystart;
              }else if ((mystart == min) 
                        && (nodeId.compareTo(winner)==-1)){
                winner = nodeId;
              }
            }
            else{
              if(dbgii){
                Env.inform(dbgiimsg + " * nothing new * "
                           + "from node: " 
                           + nodeId.toString());
              }
            }
          }

        } catch (NoSuchEntryException e) {
          System.err.println("No Such Entry Exception: " +e);
          e.printStackTrace();
        }
      }//while

      if(dbgii){
        Env.inform(dbgiimsg + " * return winner * "
                   + "node: " 
                   + winner.toString());
      }
      //System.out.println("log:" +log);
      //System.out.println("NextCausal("+unbind")"
      
      return winner;
    }
  }
  //note: not exactly contains. only that we've learned about it.
    
 /** 
 *  Check if this node has already seen invalidate "inv" 
 **/ 
  public boolean
  alreadyContains(GeneralInv inv){
    lock.lock();
    try{
      return currentVV.includes(inv.getEndVV());
    }finally{
      lock.unlock();
    }
  }
 /** 
 *  Return currentVV, for testing 
 **/ 
  public AcceptVV getCurrentVV(){
    lock.lock();
    try{
      return currentVV.cloneAcceptVV();
    }finally{
      lock.unlock();
    }
  }
  
 /** 
 *  Return myNodeId, for testing 
 **/ 
  public NodeId getMyNodeId(){
    
      return myNodeId;
    
  } 

 /** 
 *  Wait till inval with t applied 
 **/ 
  public void
  waitUntilTentativeApplied(AcceptStamp t)
    throws InterruptedException{
    lock.lock();
    try{
      waitUntilTentativeAppliedInternal(t);
    }finally{
      lock.unlock();
    }
  }
    
 /** 
 *  Internal wait till inval with t applied 
 **/ 
  private void
  waitUntilTentativeAppliedInternal(AcceptStamp t)
    throws InterruptedException{
    lock.lock();
    try{
      if (t==null) return;
      NodeId n = t.getNodeId();
      SingleWriterLogUncommitted l = 
        (SingleWriterLogUncommitted)perWriterLogs.get(n);
      if (l == null){
        l = new SingleWriterLogUncommitted(n);
        addSingleWriterLog(n, l);
        
        assert(l.getMaxCounter() <= t.getLocalClock());
        if(dbgkeyset){
          Env.inform(dbgksmsg 
                     + " waitUntilTentativeAppliedInternal add to keyset " 
                     + " added accept: " + t.toString()
                     + " resulting keyset " 
                     + perWriterLogs.keySet().toString());
        }
      }
      
      while(l.getMaxCounter() < t.getLocalClock()){
        advanceCVV.await();
      }
      assert(l.getMaxCounter() >= t.getLocalClock());
      if(dbgkeyset){
        Env.inform(dbgksmsg 
                   + " waitUntilTentativeAppliedInternal returning " 
                   + " accept: " + t.toString()
                   + " maxCounter: " + l.getMaxCounter());
      }
      
      return;
    }finally{
      lock.unlock();
    }
  }
    
 /** 
 *  Wait till inval with vv applied 
 **/ 
  public void
  waitUntilTentativeApplied(VV vv)
    throws InterruptedException{
    lock.lock();
    try{
      VVIterator startVVI = vv.getIterator();
      Object token = null;
      NodeId nodeId = null;
      while (startVVI.hasMoreElements())
      {
        token = startVVI.getNext();
        nodeId = vv.getServerByIteratorToken(token);
        assert(vv.containsNodeId(nodeId));
        try{
          AcceptStamp s = new AcceptStamp(vv.getStampByServer(nodeId), 
                                          nodeId);
          waitUntilTentativeAppliedInternal(s);
        } catch(NoSuchEntryException e){
          Env.printDebug("No Such Entry Exception " + e);
          e.printStackTrace();
          System.exit(-1);
        }
        
      }
    }finally{
      lock.unlock();
    }
  }
    
  //
  // A "bound" invalidation must be forwarded *with*
  // its body. An invalidation is initially bound
  // until the controller decides it has gotten
  // to "enough" nodes. Then we start distributing
  // "unbind" messages.
  //
  // We use the following strategy for distributing
  // unbind messages: when I receive an unbind, forward
  // that unbind to all nodes that are currently subscribed
  // to see invalidations from me. Note 
  //   (1) A node may "miss" an unbind message if it
  //       unsubscribes (or the connection is broken)
  //       after it receives a bound inval but before
  //       it receives the "unbind" msg. This is OK --
  //       it does not hurt correctness and though
  //       it may hurt performance by forcing the system
  //       to forward extra bodies, it should be
  //       rare.
  //   (2) Since the unbind messages come via the
  //       FIFO inval channel, if an unbind arrives
  //       before an inval, then it is safe to ignore
  //       the unbind (since we know that when the
  //       inval is forwarded, it will have been unbound
  //       already.)
  //

 /** 
 *  apply unbind message 
 **/ 
  public void 
  applyUnbind(UnbindMsg unbind){	
    lock.lock();
    try{
      //
      // log the unbind into stable storage first
      //
      try {
        checkpoint.append(unbind);
      } catch (DatabaseException de){
        
        de.printStackTrace();
        assert false;
        //Assert.affirm(false, "DatabaseException while apply inval " 
        //              + de.toString());
        
      }
      
      applyUnbindInternal(unbind);
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  apply debargo messagae 
 **/ 
  public void 
  applyDebargo(DebargoMsg dm){	
    lock.lock();
    try{
      //
      // log the unbind into stable storage first
      //
      try {
        checkpoint.append(dm);
      } catch (DatabaseException de){
        
        de.printStackTrace();
        assert false;
        //Assert.affirm(false, "DatabaseException while apply inval " 
      //              + de.toString());
        
      }
      
      applyDebargoInternal(dm);
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  apply unbind message internally 
 **/ 
  private void 
  applyUnbindInternal(UnbindMsg unbind){
    lock.lock();
    try{
      
      // Do we have to handle the case that the write is committed?
      
      // no. We don't if we assume that when a write is committed, we think
      // it's safe to unbind it, i.e. an
      // unbind messeage is issued. 
      // !!!!!!!!!!!!!!!!!!Remember to notify Controller to send 
      //unbind message when a commit is issued
      
      assert(unbind.getNodeId() != null);
      SingleWriterLogUncommitted l = 
        (SingleWriterLogUncommitted)perWriterLogs.get(unbind.getNodeId());
      if(l == null){
        Env.remoteAssert(false, 
                         "Received unbind for serverId I don't know about " 
                         + unbind.getNodeId().toString());
        return; // Ignore irrelevant unbind
      }
	
      boolean unbindSucceed = l.unbind(unbind);
      if(!unbindSucceed){
        // 
        // No action needed -- either we have already seen 
        // the "unbind" or we have not seen the inval (in which
        // case the sender that sent this unbind will
        // already have marked it as unbound, so if they
        // send it to us later, it will already be unbound
        //
        Env.performanceWarning("spurious unbind");
        return;
      }
      
      assert(unbindCounter < Long.MAX_VALUE); // We don't handle 
      // wraparound
      recentUnbinds.put(new Long(unbindCounter), unbind);
      unbindCounter++;
      if(recentUnbinds.size() > MAX_UNBINDS_SAVED){ // Bound mem consumption
        //	recentUnbinds.remove(unbindCounter - MAX_UNBINDS_SAVED);
        Env.tbd("Unbind garbage collecting!!!");
      }
      if(doExpensiveSanityChecks){
        verifyInvariants();
      }
      moreInval.signalAll(); // new resources for getNext to send
      return;
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  apply unbind message internally 
 **/ 
  private void 
  applyDebargoInternal(DebargoMsg dm){
    lock.lock();
    try{
      assert(dm.getNodeId() != null);
      SingleWriterLogUncommitted l = 
        (SingleWriterLogUncommitted)perWriterLogs.get(dm.getNodeId());
      if(l == null){
        Env.remoteAssert(false, 
                         "Received unbind for serverId I don't know about " 
                       + dm.getNodeId());
        return; // Ignore irrelevant debargo msg
      }
      
      InvalListItem oldEmbargoed = l.findEmbargoedInval(dm);
      if(oldEmbargoed == null){
        // 
        // No action needed -- either we have already seen 
        // the "unbind" or we have not seen the inval (in which
        // case the sender that sent this unbind will
        // already have marked it as unbound, so if they
        // send it to us later, it will already be unbound
        //
        Env.performanceWarning("spurious debargo message");
        return;
      }
      assert(oldEmbargoed != null);
      assert(oldEmbargoed.getInv() instanceof PreciseInv);
      
      oldEmbargoed.debargo();
      assert(debargoCounter < Long.MAX_VALUE); // We don't handle 
      // wraparound
      recentDebargo.put(new Long(debargoCounter), dm);
      debargoCounter++;
      if(recentDebargo.size() > MAX_UNBINDS_SAVED){ // Bound mem consumption
        //	recentUnbinds.remove(unbindCounter - MAX_UNBINDS_SAVED);
        Env.tbd("Debargo garbage collecting!!!");
      }
      if(doExpensiveSanityChecks){
        verifyInvariants();
      }
      moreInval.signalAll(); // new resources for getNext to send
      return;
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  make an normal iterator of this log 
 **/ 
  public InvalIterator
  makeInvalIterator(SubscriptionSet ss, VV startVV, 
                    long maxForceMS,
                    long maxDelayMS)
    throws OmittedVVException{
    lock.lock();
    try{
      if (inMemOmitVV.includesAnyPartOf(startVV)){
        throw new OmittedVVException();
      }
      assert(inMemOmitVV.equals(CounterVV.makeVVAllNegatives()) //nothing omitted
             ||!inMemOmitVV.includesAnyPartOf(startVV));
      InvalIterator ii = null;
      
      ii = new InvalIterator(startVV,
			     ss,
			     unbindCounter-1,
			     maxForceMS,
			     maxDelayMS,
			     this);
      
      iiCount ++;
      //iilist.add(ii);
      return ii;
    }finally{
      lock.unlock();
    }
    
  }

 /** 
 *  make a catchup iterator of this log if forCatchup is true 
 *  otherwise just return a normal invaliterator 
 **/ 
  public CatchupInvalIterator
  makeCatchupInvalIterator(SubscriptionSet ss, VV startVV, 
                    long maxForceMS,
                    long maxDelayMS,
		    VV endVV)
    throws OmittedVVException{
    lock.lock();
    try{
      if (inMemOmitVV.includesAnyPartOf(startVV)){
        throw new OmittedVVException();
      }
      assert(inMemOmitVV.equals(CounterVV.makeVVAllNegatives()) //nothing omitted
             ||!inMemOmitVV.includesAnyPartOf(startVV));
      CatchupInvalIterator ii = null;
      
      ii = new CatchupInvalIterator(startVV,
				    ss,
				    unbindCounter-1,
				    maxForceMS,
				    maxDelayMS,
				    this,
				    endVV);
      
      iiCount ++;
      //iilist.add(ii);
      return ii;
    }finally{
      lock.unlock();
    }
    
  }
  
  
 /** 
 *  remove inval iterator 
 **/ 
  public void 
  removeInvalIterator(InvalIterator ii){
    lock.lock();
    try{
      iiCount --;
      //iilist.remove(ii);
      assert iiCount >=0; //something must be wrong if it goes to negtive.
      // removeInvalIterator must be called unexpectedly
      if (iiCount == 0) {
        //notifyAll(); //ready for garbage collection
      }
    }finally{
      lock.unlock();
    }
  }


 /** 
 *  remove inval iterator 
 **/ 
  public void 
  removeInvalIterator(CatchupInvalIterator ii){
    lock.lock();
    try{
      iiCount --;
      //iilist.remove(ii);
      //assert iilist.size() == iiCount;
      assert iiCount >=0; //something must be wrong if it goes to negtive.
      // removeInvalIterator must be called unexpectedly
      if (iiCount == 0) {
        //notifyAll(); //ready for garbage collection
      }
    }finally{
      lock.unlock();
    }
  }

  
  public void
  addMailBox(OutgoingConnectionMailBox box){
    mailBoxList.add(box);
  }
  
  public void
  removeMailBox(OutgoingConnectionMailBox box){
    this.removeInMemLogIterator((InMemLogIterator)(box.getIterator()));
    mailBoxList.remove(box);
  }
  
  public void
  notifyMailBoxOfNewUpdates(){
    if(mailBoxList != null){
      mailBoxList.receiveNewUpdates();
    }
  }
  
 /** 
 *  Verify Invariants for Updatelog 
 **/ 
  public void
  verifyInvariants(){
    //verify local state invariants
    	
    currentVV.includes(inMemOmitVV);
    currentVV.includes(diskOmitVV);
    assert((currentLamportClock == 0) 
           ||(currentLamportClock == currentVV.getMaxTimeStamp()+1));
	  	
    //verify each uncommitted per-writer log
    NodeId curId;
    SingleWriterLogUncommitted curLog;
	
    VVIterator keyi = currentVV.getIterator();
    while (keyi.hasMoreElements()){
      curId = (NodeId) keyi.getNext();
      curLog = ( SingleWriterLogUncommitted)perWriterLogs.get(curId);
	    
      try{
        if ((curLog != null)&&(curLog.getMaxCounter()!=-1 )){
          //curLog is not empty
		    
          //check continuous of commited and uncommitted per writer-log
          //Env.printDebug("commitVV:" + commitVV);
          //Env.printDebug("commitVV.containsNodeId(curId):" 
          //+ commitVV.containsNodeId(curId));
          /*	    
                   assert((!commitVV.containsNodeId(curId)) //no commit entry
                   || (curLog.getMinCounter()==-1)   //no uncommit entry
                   ||(curLog.getMinCounter() > 
                   commitVV.getStampByServer(curId)));
          */
    
          //check currentVV
          assert((currentVV.containsNodeId(curId)) 
                 && (curLog.getMaxCounter() == 
                     currentVV.getStampByServer(curId)));
		    
          // check uncommitted log invariants
          if(doExpensiveSanityChecks){
            curLog.verifyInvariants();
          }
        }
        /*
          else{
          assert((!commitVV.containsNodeId(curId) 
          && (!currentVV.containsNodeId(curId)))
          ||(commitVV.getStampByServer(curId) == 
          currentVV.getStampByServer(curId)));
          }
        */
      } catch (NoSuchEntryException e){
        Env.printDebug("no such entry Exception:" +e);
        e.printStackTrace();
        System.exit(-1);	
      }
	    
    }
	
  }
  
    
 /** 
 *  Return a string representation 
 **/ 
  public final String 
  toString(){
   String str = "";
   /*
    str = checkpoint.toString();
      
    str += "\n Current Log: \n";
    str += "currentVV: " + currentVV +"\n";
    str += "curretnLamportClock: " + currentLamportClock +"\n";
    str += "myNodeId: " + myNodeId +"\n";
    str += "unbindCounter:" +unbindCounter +"\n";
    str += "recentUnbinds:" + recentUnbinds +"\n";
	
    
      if (committed.getOldest() != null){
      str += "committed: " + committed.getOldest() +"\n";
      }else { str += "committed: Empty" +"\n";}
    */

    Iterator<NodeId> keyi = perWriterLogs.keySet().iterator();
    NodeId nodeId = null;
    SingleWriterLogUncommitted l = null;
	
    while (keyi.hasNext()){
      nodeId = (NodeId) keyi.next();
	    
      l = (SingleWriterLogUncommitted)perWriterLogs.get(nodeId);
      str += "****uncommitted["+nodeId + "]****: \n MemSize: " + l.memSize() + "\n"
        + l.getOldest() +"\n";
    }
    return(str);
  }
  
 /** 
 *  sync log contents to disk 
 **/ 
  public void
  syncToDisk(){
    
    try{
      checkpoint.sync();
    }
    catch(Exception e){
      e.printStackTrace();
      //throw new IOException(e.toString());
      assert false;
    }
  }

 /** 
 *  garbage collect persistentlog and in-mem per writer log any records 
 *  <= tailVV 
 **/ 
  public  AcceptVV truncate(VV tailVV)
    throws IOException {
    lock.lock();
    try{
      AcceptVV newOmitVV = null;
      //garbage collect persistentLog
      newOmitVV = checkpoint.truncateCompletely(tailVV);
      
      VVIterator tailVVI = newOmitVV.getIterator();
      Object token = null;
      NodeId nodeId = null;
      
      while (tailVVI.hasMoreElements()){
        token = tailVVI.getNext();
        nodeId = newOmitVV.getServerByIteratorToken(token);
        long thisTail = -1;
        try{
          thisTail = newOmitVV.getStampByServer(nodeId);
        }catch(NoSuchEntryException nsee){
          //impossible.
          assert false;
          System.exit(1);
        }
        SingleWriterLogUncommitted l =  
          (SingleWriterLogUncommitted)perWriterLogs.get(nodeId);
        if (l != null){
          l.chopTail(thisTail);
        } 
      }
      inMemOmitVV.advanceTimestamps(newOmitVV);
      diskOmitVV.advanceTimestamps(newOmitVV);
      return newOmitVV;
    }finally{
      lock.unlock();
    }
  }  
  
 /** 
 *   truncate the per-Writer-Log into "x" percentage of original size. 
 *  post condition:  
 *     logSize = rate * orginal Size. 
 **/ 
  public AcceptVV truncate(double rate)
    throws IOException {
    lock.lock();
    try{
      CounterVV newOmitVV = new CounterVV();
      
      NodeId nodeId = null;
      
      Iterator<NodeId> keyi = perWriterLogs.keySet().iterator();
      SingleWriterLogUncommitted l = null;
      long newStart;
      
      while (keyi.hasNext()){
        nodeId = (NodeId) keyi.next();
        
        l = (SingleWriterLogUncommitted)perWriterLogs.get(nodeId);
        long start = l.getMinCounter();
        
        if (start != SingleWriterLogUncommitted.UNINITIALIZED_COUNTER){
          long end = l.getMaxCounter();
          newStart = (long)((1-rate)*start + rate*end);
          l.chopTail(newStart);
          newOmitVV.setStamp(new AcceptStamp(newStart, nodeId));
        }
      }
      
      //garbage collect persistentLog
      checkpoint.truncateCompletely(newOmitVV.cloneAcceptVV());
      inMemOmitVV.advanceTimestamps(newOmitVV);
      diskOmitVV.advanceTimestamps(newOmitVV);
      return newOmitVV.cloneAcceptVV();
    }finally{
      lock.unlock();
    }
  }


 /** 
 *    
 *  truncate the per-Writer-Log records that are less than upBound till  
 *  the remain log is "x" percentage of original size. 
 *    
 *  post condition:  
 *      it might truncate less than "1-x" percentage of the log because 
 *      more than "x" log are large than upBound 
 *   
 *  return the total mem size truncated. 
 **/ 
  public long truncateMem(double rate, VV upBound)
    throws IOException {
    lock.lock();
    
    if ( dbgGC)
      Env.dprintln(dbgGC, "start truncate Mem-- upBound: " + upBound
                 + " cvv: " + currentVV 
                 + " inMemOmitVV: " + inMemOmitVV);
    
    long cutted = 0;
    try{
      CounterVV newOmitVV = new CounterVV();
      
      NodeId nodeId = null;
      
      Iterator<NodeId> keyi = perWriterLogs.keySet().iterator();
      SingleWriterLogUncommitted l = null;
      long newStart;
    
      while (keyi.hasNext()){
        nodeId = (NodeId) keyi.next();
      
        l = (SingleWriterLogUncommitted)perWriterLogs.get(nodeId);
        long start = l.getMinCounter();
        
        if (start != SingleWriterLogUncommitted.UNINITIALIZED_COUNTER){
          long end = l.getMaxCounter();
          //compute the new start of the log after the trucation
          newStart = (long)((1-rate)*start + rate*end);
          try{
            long upBoundStamp = upBound.getStampByServer(nodeId);
            if(upBoundStamp < newStart){
              newStart = upBoundStamp;
            }
          }catch(NoSuchEntryException e){
            // the checkpoint vv hasn't learned the invals about this nodeid
            // shouldn't truncate any of this per writer log
            continue;
          }
          cutted += l.chopTail(newStart);
          newOmitVV.setStamp(new AcceptStamp(newStart, nodeId));
        }
      }

      //checkpoint.truncateCompletely(newOmitVV.cloneAcceptVV());
      inMemOmitVV.advanceTimestamps(newOmitVV);
      
      if ( dbgGC )
	Env.dprintln(dbgGC, "Finish mem truncation -- upBound: " + upBound
                   + " cvv: " + currentVV 
                   + " inMemOmitVV: " + inMemOmitVV
                   + " total size gain: " + cutted);
        
      return cutted;
    }finally{
      lock.unlock();
    }
  }  

 /** 
 *    
 *  truncate the per-Writer-Log records that are less than upBound till  
 *  the remain log is "x" percentage of original size. 
 *    
 *  post condition:  
 *      it might truncate less than "1-x" percentage of the log because 
 *      more than "x" log are large than upBound 
 *   
 *  return the total mem size truncated. 
 **/ 
  public void truncateDisk(double rate, VV upBound)
    throws IOException {

    lock.lock();
    if ( dbgGC )
      Env.dprintln(dbgGC, "start truncate disk-- upBound: " + upBound
                 + " cvv: " + currentVV 
                 + " diskOmitVV: " + diskOmitVV
                 + " diskSize: " + checkpoint.diskSize());
    
    try{
      long newStart, start, end, upBoundStart;
      CounterVV newOmitVV = new CounterVV();
      Object token = null;
      //calculate the new omit vv
      for(VVIterator i = this.currentVV.getIterator(); i.hasMoreElements();){
        token = i.getNext();
        end = this.currentVV.getStampByIteratorToken(token);
        try{
          start = this.diskOmitVV.getStampByServer((NodeId)token);
        }catch(NoSuchEntryException e){
          start = 0;
        }
        newStart = (long)((1-rate)*start + rate*end);
        try{
          upBoundStart = upBound.getStampByServer((NodeId)token);
          if(newStart > upBoundStart){
            newStart = upBoundStart;
          }
        }catch(NoSuchEntryException e){
          newStart = AcceptStamp.BEFORE_TIME_BEGAN;//does not truncate any 
          // entry writen by this nodeId;
        }
        newOmitVV.setStamp(new AcceptStamp(newStart, (NodeId)token));
      }        
      assert newOmitVV.includes(this.diskOmitVV);
      
      //truncate it
      checkpoint.truncateCompletely(newOmitVV);
      diskOmitVV.advanceTimestamps(newOmitVV);
      if ( dbgGC )
	Env.dprintln(dbgGC, "finish disk-- upBound: " + upBound
                       + " cvv: " + currentVV 
                       + " diskOmitVV: " + diskOmitVV
                       + " diskSize: " + checkpoint.diskSize());
      
    }finally{
      lock.unlock();
    }
    
  }

 /** 
 *  get diskOmitVV 
 **/ 
  public AcceptVV 
  getDiskOmitVV(){
    lock.lock();
    try{
      return diskOmitVV.cloneAcceptVV();
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  get inMemOmitVV 
 **/ 
  public AcceptVV 
  getInMemOmitVV(){
    lock.lock();
    try{
      return inMemOmitVV.cloneAcceptVV();
    }finally{
      lock.unlock();
    }
  }

 /** 
 *  close() 
 **/ 
  public void 
  close(){
    try{
      lock.lock();
      
      killGCWorker();
      perWriterLogs.clear();
      perWriterLogs = null;
      recentUnbinds.clear();
      recentUnbinds = null;
      recentDebargo.clear();
      recentDebargo = null;
      checkpoint.close();
      System.gc();
   
    }finally{
      lock.unlock();
    }
    
  }

  
    
  
 /** 
 *  Module test 
 **/ 
  public static void main(String[]argv){
    //self test moved to junit
    try{
      
      String configFile = "ufs-local.config" ;
      Config.readConfig(configFile);
        
      
      System.out.println(" UpdateLog debugging interface...");
      testInterface();
  
      
    } catch (Exception e){
      e.printStackTrace();
      System.exit(-1);
    }
  }
   
  
	
 /** 
 *  interactive test interface 
 **/ 
  private static void testInterface(){
    boolean end = false;
    try{
      UpdateLog log = new UpdateLog("/projects/lasr/space0/zjd/ufsenv", 
                                    new NodeId(9),
                                    false);
      System.out.println("Enter action:  ");
      System.out.println("Note: for interestSet assume always two objIds, "
                         +"for vv always two nodes components");
      System.out.println("write       : w [ObjId] [offset] [length] [bytes]"
                         +" [bound])");
      System.out.println("ApplyInv    : a [objId] [objId]"
                         +" [nodeId1] [nodeId2] [startTime1] [startTime2]" 
                         +"[endTime1] [endTime2]");
      System.out.println("getNext     : n [unbindCount] [nodeId1] "
                         +"[nodeId2] [startTime1] [startTime2] [timeout]");
      System.out.println("waitUntil   : u [nodeId] [Time] [nodeId] [Time]");
      System.out.println("applyunbind : b [objId] [offset] [range] [time] "
                         +"[nodeid]");
      System.out.println("InvIterator : i [#] [nodeId1] [startTime1]"
                         + "[nodeId2] [startTime2] [objId] [objId]"
                         +" [ubc] [stop#]");
      System.out.println("showlog     : s");
      System.out.println("end         : e");
	    
      InputStreamReader tempReader=new InputStreamReader(System.in);
      BufferedReader reader=new BufferedReader(tempReader);
      while(!end){
        String input=reader.readLine();
        System.out.println(" ");
        System.out.println("Input: " + input);
        if (input == null) return;
        byte[] action = input.getBytes();
			
        if(action[0] == 'w'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          ObjId objId = new ObjId(st.nextToken());
          long offset = (new Long(st.nextToken())).longValue();
          long len = (new Long(st.nextToken())).longValue();
          byte[] buffer = st.nextToken().getBytes();
          ImmutableBytes ib = new ImmutableBytes(buffer);
          boolean bound = false;
          int b = (new Integer(st.nextToken())).intValue();
          if (b==1) bound = true;
          log.write(objId, offset, len, Core.DEFAULT_PRIORITY, null, ib, bound,
                    Long.MAX_VALUE, false);
        } else if (action[0] == 'a'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
			
          String objId1 = st.nextToken();
          String objId2 = st.nextToken();
          long nodeId1 = (new Long(st.nextToken())).longValue();
          long nodeId2 = (new Long(st.nextToken())).longValue();
          long startTime1 = (new Long(st.nextToken())).longValue();
          long startTime2 = (new Long(st.nextToken())).longValue();
          long endTime1 = (new Long(st.nextToken())).longValue();
          long endTime2 = (new Long(st.nextToken())).longValue();
		    
          String[] dir=new String[2];
          dir[0] = objId1;
          dir[1] = objId2;
		    
          AcceptStamp[] as1 = new AcceptStamp[2];
          as1[0] = new AcceptStamp(startTime1, new NodeId(nodeId1));
          as1[1] = new AcceptStamp(startTime2, new NodeId(nodeId2));
		    
          AcceptStamp[] as2 = new AcceptStamp[2];
          as2[0] = new AcceptStamp(endTime1, new NodeId(nodeId1));
          as2[1] = new AcceptStamp(endTime2, new NodeId(nodeId2));
		    
          ImpreciseInv ii =
            new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                             new AcceptVV(as1),
                             new AcceptVV(as2));
          log.applyInval(ii);
		    
        } else if (action[0] == 'n'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
		    
          long ubc = (new Long(st.nextToken())).longValue();
          
          long nodeId1 = (new Long(st.nextToken())).longValue();
          long nodeId2 = (new Long(st.nextToken())).longValue();
          long startTime1 = (new Long(st.nextToken())).longValue();
          long startTime2 =(new Long(st.nextToken())).longValue();
          long timeout = (new Long(st.nextToken())).longValue();
		    
          AcceptStamp[] as1 = new AcceptStamp[2];
          as1[0] = new AcceptStamp(startTime1, new NodeId(nodeId1));
          as1[1] = new AcceptStamp(startTime2, new NodeId(nodeId2));
          AcceptVV acceptVV1 = new AcceptVV(as1);
		    
          Object obj;
          obj = log.getNext(ubc, acceptVV1, timeout);
          if (obj instanceof GeneralInv){
            System.out.println((GeneralInv)obj);
          }else if (obj instanceof UnbindMsg){
            System.out.println((UnbindMsg)obj);
          } else { 
	    assert(false);
	  }
		    
        } else if (action[0] == 'u'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          //waitUntil   : u [nodeId] [Time]
          long nodeId1 = (new Long(st.nextToken())).longValue();
          long time1 = (new Long(st.nextToken())).longValue();
          long nodeId2 = (new Long(st.nextToken())).longValue();
          long time2 = (new Long(st.nextToken())).longValue();
			    
          AcceptStamp[] as1 = new AcceptStamp[2];
          as1[0] = new AcceptStamp(time1, new NodeId(nodeId1));
          as1[1] = new AcceptStamp(time2, new NodeId(nodeId2));
          AcceptVV acceptVV1 = new AcceptVV(as1);
		    
          log.waitUntilTentativeApplied(acceptVV1);
		    
        } else if (action[0] == 'b'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          //applyunbind : b [objId] [offset] [range] [time] [nodeid]
          String objId = st.nextToken();
          long offset = (new Long(st.nextToken())).longValue();
          long range = (new Long(st.nextToken())).longValue();
          long time = (new Long(st.nextToken())).longValue();
          long nodeId = (new Long(st.nextToken())).longValue();
		    
          UnbindMsg ubm = new UnbindMsg(new ObjInvalTarget(new ObjId(objId), 
                                                           offset, range), 
                                        new AcceptStamp(time, 
                                                        new NodeId(nodeId)));
		    
          log.applyUnbind(ubm);
        } else if (action[0] == 't'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          
          long t1 = (new Long(st.nextToken())).longValue();
          long t2 = (new Long(st.nextToken())).longValue();
          long t3 = (new Long(st.nextToken())).longValue();
          
          AcceptStamp[] as1 = new AcceptStamp[3];
          as1[0] = new AcceptStamp(t1, new NodeId(9));
          as1[1] = new AcceptStamp(t2, new NodeId(10));
          as1[2] = new AcceptStamp(t3, new NodeId(11));
          
          AcceptVV acceptVV1 = new AcceptVV(as1);
          
          log.truncate(acceptVV1);
          
          System.out.println("diskOmitVV: " + log.getDiskOmitVV());
          
        } else if (action[0] == 'g'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          
          double r = (new Double(st.nextToken())).doubleValue();
          
          log.truncate(r);
          
          System.out.println("DiskOmitVV: " + log.getDiskOmitVV());
          System.out.println("log: " +log);
        } else if (action[0] == 's'){   
          System.out.println(log);
	} else if (action[0] == 'e'){
          end = true;
        }
      }
       
    } catch (Exception e){
      e.printStackTrace();
      System.out.println("Wrong input format ...");
    }
  }

 /** 
 **/ 
 /** 
 *  Class GCWorker 
 **/ 
 /** 
 **/ 
  static int killCount = 0;
  static int maxKillWarn = 3;
  class GCWorker extends Thread{
    private UpdateLog log;
    private DataStore store;

    private GCWorker(UpdateLog l, 
                     DataStore s){
      log = l;
      store = s;
    }

    public void run(){
      
      while(true){
        try{
          log.waitForWork();
        }
        catch(InterruptedException e){
          //
          // No more work to do. Someone called LogGC.killWorker()
          //
          killCount++;
          if(killCount < maxKillWarn){
            Env.inform("LogGC worker thread exits b/c LogGC.killWorker called");
            if(killCount == maxKillWarn){
              Env.inform("LogGC will suppress future killWorker notifications");
            }
          }
          return; // Thread exits
        }
        
        try{
          store.syncToDisk();
          //Env.tbd("UpdateLog garbage collection might stall because DataStore"
          //      + " checkpoint vv is behind. Need to add logic to coordinate"
          //      + " garbage collection and checkpoint sync to disk activity");
          log.doneWithWork(store.getCPVV());
        }catch(IOException e){
          System.err.println(e.toString());
          assert false;
          System.exit(-1);
        }
      }
    }
  } 
   
  
  private 
  void addSingleWriterLog(NodeId nid,
                          SingleWriterLogUncommitted log){
    log.registerAll(activeIters);
    perWriterLogs.put(nid, log);
  }

  
  /**
   * This method allows outside world to create an iterator
   * to scan the per-writer-log.
   * 
   * @param excludedStartVV
   * @return
   */
  public InMemLogIterator makeInMemLogIterator(AcceptVV excludedStartVV){
    lock.lock();
    try{
      InMemLogIterator ret = new InMemLogInternalIterator(excludedStartVV);
      activeIters.add(ret);
      return ret;
    }finally{
      lock.unlock();
    }
  }
  
  public void removeInMemLogIterator(InMemLogIterator iter){
    lock.lock();
    try{
      Object[] allLogs = null;
      if(perWriterLogs != null){
        allLogs = perWriterLogs.values().toArray();
        for (int i = 0; i < allLogs.length; i++){
          ((SingleWriterLogUncommitted)(allLogs[i])).remove(iter);
        }
      }
      if(activeIters != null){
        activeIters.remove(iter);
      }
    }finally{
      lock.unlock();
    }
  }
  
  
  /**
  *
  * This class is an external interator of UpdateLog.
  * Its main function: getNext() iterates the entire per-writer-logs of
  * a node and returns each SingleWriterInval stored in the in-mem log
  * one by one according to their causal order defined by
  * each SingleWriterInval's start time which is of the
  * format <writer, time>. And it returns null at the end.
  *
  * Difference between original InvalIterator and InMemLogIterator:
  *     - performance: Besides cvv, it keeps the references of the
  *         next item to return for each writer so that the getNext will
  *         be start from the last pointer instead of from the very begining.
  *
  *         What's more, we precalculate the nextItem for each writer.
  *         whenever iter.getNext() is called, it directly returns the min
  *         of all current nextItem, and prefetch the next item to replace
  *         the one to return.
  *
  *     - semantics:
  *        . InMemLogIterator is block free.
  *          It won't block. If no more item to return, it simply returns null.
  *
  *        . InMemLogIterator won't accumulate any SingleWriterInv into
  *          a general imprecise invalidation.
  *          It only faithfully reports all the SingleWriterInv in the
  *          current per-writer-log. Each SingleWriterInv is ordered by
  *          it's start acceptStamp.
  *
  * @author zjiandan
  *
  */
  class InMemLogInternalIterator implements InMemLogIterator{
    
    private SingleWriterLogPointers nextPointers;

    private CounterVV iterCVV; //summarize all sent invals

    private final int id; //for debug
    
    /**
     * called by UpdateLog.makeInMemLogIterator() which
     * is under a synchronized block
     * so we don't put lock here
     */
    private InMemLogInternalIterator(AcceptVV excludedStartVV){
      //populate the next items for each writer
      nextPointers = getNextPointers(excludedStartVV);
      iterCVV = new CounterVV(excludedStartVV);
      id = activeIters.size();
      //System.out.println("iter "+ id + "created.startVV=" + excludedStartVV);
    }
    
    private SingleWriterLogPointers getNextPointers(AcceptVV excludedStartVV){
      SingleWriterLogPointers ret = new SingleWriterLogPointers();
      SingleWriterLogUncommitted[] allLogs = null;
      long start = -1;
      NodeId id = null;
      InvalListItem item = null;
      allLogs = perWriterLogs.values().toArray(new SingleWriterLogUncommitted[0]);
      for (int i = 0; i < allLogs.length; i++){
        id = allLogs[i].getNodeId();
        try{
          start = excludedStartVV.getStampByServer(id);
        }catch(NoSuchEntryException e){
          start = -1;
        }

        item = allLogs[i].getNextItemByStart(start);
        if(item != null){
          ret.add(id, item);
        }else{//no new item yet
          allLogs[i].register(this);
        }
      }
      return ret;
    }
    
    public boolean hasNext(){
      lock.lock();
      try{
        return !(nextPointers.size()==0);
      }finally{
        lock.unlock();
      }
    }

    /**
     * assumption
     * all nextItems are already stored in the corresponding
     * InMemLogIterator
     *
     * just get the item with the minimum start
     * advance cvv
     * prefetch the next item on the same writer to replace
     * the one about to be sent
     */
    public SingleWriterInval next(){
      lock.lock();
      try{
        if(doExpensiveSanityChecks){
          sanityCheck();
        }
        SingleWriterInval ret = null;
        //System.out.println("iter<"+id+"> ---before next:" + nextPointers.toString());
        InvalListItem nextItem = nextPointers.min();
      
        if(nextItem != null){
          ret = (SingleWriterInval)nextItem.getInv();
          NodeId nid = ret.getNodeId();
          //update the pointers
          assert !this.iterCVV.includes(ret.getStartVV());
          this.iterCVV.advanceTimestamps(ret.getEndVV());
          InvalListItem newNextItem = nextItem.getNewer();
          if(newNextItem != null){
            while(newNextItem.isDummy()){
              newNextItem = newNextItem.getNewer();
              assert newNextItem != null;//no single writer log 
                                         //will end with a dummy inv
            }
            nextPointers.advance(nid, newNextItem);
          }else{//no new inval --> reach the end of the writer's updates

            //register in per-writer-log so that it will get a new one
            //whenever the writer gets a new update
          //  System.out.println("\n iter<" + id + "> end of log: going to remove the pointer" + nextItem.getInv());
            SingleWriterLogUncommitted log = perWriterLogs.get(nid);
            //System.out.println("\n iter<" + id + "> registered itself to log "+ nid);
            log.register(this);
            nextPointers.remove(nid, nextItem);
          }
        }
        if(doExpensiveSanityChecks){
          sanityCheck();
        }
        //System.out.println("<"+id+">---after return next=" + ret 
        //    + "\n pointers= " + nextPointers.toString());
        return ret;
      }finally{
        lock.unlock();
      }
    }

    public void remove()
    throws UnsupportedOperationException{
      throw new UnsupportedOperationException("InMemLogIterator.remove() is not supported"); 
    }
    
    /**
     * called by singleWriterLogUncommited.zip()
     * to update nextPointers.
     * It is supposed to be called within a synchronized block.
     */
    public void
    addPointer(NodeId nodeId, InvalListItem item)
    throws StaleInvalidationException{
      if(this.iterCVV.includes(item.getInv().getEndVV())){
        throw new StaleInvalidationException("Attempt to add old inv as new item.");
      }
      //System.out.println("iter " + "<" + id + " > get New item:" + nextPointers.toString());
      nextPointers.add(nodeId, item);
      if(doExpensiveSanityChecks){
        sanityCheck(nodeId);
        sanityCheck();
      }
    }

    /**
     * called by singleWriterLogUncommited.zip()
     * to update nextPointers.
     * It is supposed to be called within a synchronized block.
     */
    private InvalListItem 
    getPointer(NodeId id){
      return nextPointers.getInvalListItem(id);
    }
    
    public String toString(){
     String ret = " Iter<"+ id + "> ";
     ret += iterCVV;
     ret += ", ";
     ret += nextPointers;
     return ret;
    }
    
    public AcceptVV getCurrentVV(){
      return iterCVV.cloneAcceptVV();
    }
    private void
    sanityCheck(){
      
      SingleWriterLogPointers expectedPointers = null;
      expectedPointers = this.getNextPointers(iterCVV.cloneAcceptVV());
      assert expectedPointers.equals(nextPointers):"ID=" + id 
      + "nextPointers=" + nextPointers + " expectedPointers="+ expectedPointers;
    }

    private void
    sanityCheck(NodeId id){
      SingleWriterLogUncommitted slog = perWriterLogs.get(id);
      assert slog != null;
      long start = -1;
      try{
        start = iterCVV.getStampByServer(id);
      }catch(NoSuchEntryException e){
        start = -1;
      }
      InvalListItem item = slog.getNextItemByStart(start);
      assert item == nextPointers.getInvalListItem(id);
    }
    
  }


  


  
}


	
		


//---------------------------------------------------------------------------
/* $Log: UpdateLog.java,v $
/* Revision 1.69  2007/09/10 23:52:22  zjiandan
/* upgrade to newest BerkeleyDB je version.
/*
/* Revision 1.68  2007/08/05 04:43:54  zjiandan
/* SocketServer shutdown quietly
/*
/* Revision 1.67  2007/07/03 07:57:54  zjiandan
/*  more unit tests
/*
/* Revision 1.66  2007/06/30 00:08:08  zjiandan
/* *** empty log message ***
/*
/* Revision 1.65  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.64  2007/04/11 07:18:33  zjiandan
/* Fix SubscribeBWUnit test.
/*
/* Revision 1.63  2007/04/02 21:11:39  zjiandan
/* snapshort for sosp2007.
/* */
//---------------------------------------------------------------------------





