package code;

 /** 
 *  Maintain a priority queue of updates 
 **/ 
import java.util.TreeMap;
import java.util.StringTokenizer;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Collection;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.*;

public class UpdatePriorityQueue{

  //
  // A priority queue of references to update bodies 
  // to be sent to some destination.
  // 
  // We support partial writes of a file by accumulating
  // the ranges of changed bytes for each objId. One simplifying
  // assumption -- we only keep one priority across all
  // writes to the same file. For now, we just keep the most
  // recent one. 
  //
  private static final boolean dbg = false;
  private static final boolean dbgRange = false;
 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    boolean end = false;
    Config.readConfig("ufs.config");
    UpdatePriorityQueue upq = new UpdatePriorityQueue(null);

    System.out.println("Enter action:");
    System.out.println("  Insert a range (i [priority] [ObjId] " +
                       "[offset] [length])");
    System.out.println("  Cancel a range (c [ObjId] [offset] [length])");
    System.out.println("  Retrieve the first range (n)");
    System.out.println("  Print ranges (p)");
    System.out.println("  Exit (e)");
    try{
      InputStreamReader tempReader = new InputStreamReader(System.in);
      BufferedReader reader = new BufferedReader(tempReader);

      while(!end){
        String input=reader.readLine();
        System.out.println(" ");
        System.out.println("Input: " + input);
        byte[] action = input.getBytes();
        if(action[0] == 'i'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          double p = (new Double(st.nextToken())).doubleValue();
          String id = st.nextToken();
          int start = (new Integer(st.nextToken())).intValue();
          int length = (new Integer(st.nextToken())).intValue();
          upq.insert(p, new ObjId(id), start, length);
        } else if (action[0] == 'c'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
          String id = st.nextToken();
          int start = (new Integer(st.nextToken())).intValue();
          int length = (new Integer(st.nextToken())).intValue();
          upq.cancel(new ObjId(id), start, length);
        } else if (action[0] == 'n'){
          upq.getNext();
        } else if (action[0] == 'p'){
          upq.displayQueue();
        } else if (action[0] == 'e'){
          end = true;
        }
      }
    } catch (Exception e){
      e.printStackTrace();
      System.out.println("Wrong input format ...");
    }
  }

 /** 
 *  Data members 
 **/ 
  private final int MAX_ITEMS = 10000;  // Max number of updates to track
  private TreeMap priorityList;  // java.util.TreeMap
  private Hashtable mapID2Ranges;
  private NodeId destId;

 /** 
 *  Return the NodeId for which this UpdatePriorityQueue was made 
 **/ 
  public NodeId
  getDestId(){
    return destId;
  }	

 /** 
 *  Return the NodeId for which this UpdatePriorityQueue was made 
 **/ 
  public
  UpdatePriorityQueue(/*Core core_, */NodeId destId_){
    //core = core_;
    priorityList = new TreeMap(new UpdatePriorityComparator());
    mapID2Ranges = new Hashtable();
    destId = destId_;
  }

 /** 
 *  Return true if the priority queue is empty 
 **/ 
  public synchronized boolean
  isEmpty(){
    Env.dprinterrln(dbg, "UpdatePriorityQueue::isempty called ");
    if(priorityList.size() == 0){
      Env.dprinterrln(dbg, "UpdatePriorityQueue::isempty return t ");
      return true;
    }else{
      Env.dprinterrln(dbg, "UpdatePriorityQueue::isempty return f ");
      return false;
    }
  }

 /** 
 *  Called when we learn of a new relevant update 
 **/ 
  public synchronized void
  insert(double priority, ObjId id, long start, long length){

    Env.dprinterrln(dbg, "UpdatePriorityQueue::insert called for " + id+":"+start+":"+length);

    UpdateRangeList updates;
    assert length > 0;
    assert start >= 0;

    //System.err.println(" -- UPQ: Received notification for write to "
    //+ id + " : [" + start + "," + length + "]");

    if(!mapID2Ranges.containsKey(id)){ 
      //System.err.println(" -- UPQ: Not contains list for " + id +
      //" already");
      updates = new UpdateRangeList(priority, id);
      mapID2Ranges.put(id, updates);
      priorityList.put(updates, updates);
    }else{	    
      //System.err.println(" -- UPQ: Contains list for " + id + " already");
      Object key = mapID2Ranges.get(id);
      updates = (UpdateRangeList)priorityList.get(key);
      assert updates != null;
      if(updates.getPriority() != priority){
        updates = (UpdateRangeList)priorityList.remove(key);
        mapID2Ranges.remove(id);
        updates.setPriority(priority);
        key = (Object)updates;
        mapID2Ranges.put(id, key);
        priorityList.put(key, updates);
      }
    }

    assert(updates != null);
    assert(updates.getPriority() == priority);
    updates.addRange(start, length);
    //  System.err.println(priorityList.size() + " " + updates.size());
    this.notifyAll();
    //updates.displayList();
    Env.dprinterrln(dbg, "UpdatePriorityQueue::insert returns ");
    if((dbgRange)&&(id.getPath().equals("/u/57"))){
      System.err.println("(" + id + ", " + start + ", " + length + ")");
      displayQueue();
    }
    return;
  }

 /** 
 *  Remove a range from this priority queue 
 **/ 
  public synchronized void
  cancel(ObjId id, long start, long length){
    /*
      TBD: we got an invalidation for a range of a file -- cancel
      any pending push of that data
    */
    Env.dprinterrln(dbg, "UpdatePriorityQueue::cancel called ");
    try{
      assert length > 0;
      assert start >= 0;
      UpdateRangeList updates;
      Object key = mapID2Ranges.get(id);
      if(key == null){
        Env.dprinterrln(dbg, "UpdatePriorityQueue::cancel return 1 ");
        return;
      }
      updates = (UpdateRangeList)this.priorityList.get(key);
      
      //System.err.println("upq received cancel for " + id + " : [" +
      // start + "," + length + "]");
      updates.cancelRange(start, length);
      if(updates.isEmpty()){
        priorityList.remove(key);
        mapID2Ranges.remove(id);
        //System.err.println("cancelled received inval from upq");
      }
    }catch(Exception e){
      System.err.println("UpdatePriorityQueue::cancel exiting because: " + e);
      e.printStackTrace();
      assert(false);
    }
    Env.dprinterrln(dbg, "UpdatePriorityQueue::cancel return 2 ");
  }

 /** 
 *  Get meta-information for the next range on the priority queue 
 **/ 
  public synchronized MetaBody
  getNext(){
    //find next updates by priority in priorityList
    //get next range from updates
    // form a metabody
    //delete range from updates
    Env.dprinterrln(dbg, "UpdatePriorityQueue::getNext called ");
    long t1, t2, t3, t4;
    //BodyMsg bodyMsg = null;
    MetaBody metaBody = null;
    UpdateRangeList updates;
    while(priorityList.size() == 0){
      try{
        Env.dprinterrln(dbg, "UpdatePriorityQueue::getNext wait() ");
        wait();
      } catch (InterruptedException e){
        System.err.println("Unexpected interrupt occured !");
        e.printStackTrace();
      }
    }
    assert priorityList.size() != 0;
    assert mapID2Ranges.size() != 0;
    t1 = System.currentTimeMillis();
    Object key = priorityList.firstKey();
    updates = (UpdateRangeList)priorityList.get(key);
    t2 = System.currentTimeMillis();
    assert updates != null;
    metaBody = new MetaBody(updates.getOID(),
                            updates.nextRangeStart(),
                            updates.nextRangeLength());

    t3 = System.currentTimeMillis();
    updates.removeNextRange();
    if(updates.isEmpty()){
      priorityList.remove(key);
      mapID2Ranges.remove(updates.getOID());
    }
    t4 = System.currentTimeMillis();
    Env.dprinterrln(dbg, "UpdatePriorityQueue::getNextreturns ");
    //System.err.println((t2-t1) + " " + (t3-t2) + " " + (t4-t3));
    //return bodyMsg;
    return(metaBody);
  }
    
 /** 
 *  Print the current priority queue 
 **/ 
  public synchronized void
  displayQueue(){
    boolean end = false;
    UpdateRangeList updates;
    Iterator li = priorityList.values().iterator();
    while(li.hasNext()){
      updates = (UpdateRangeList)li.next();
      assert updates != null;
      updates.displayList();
    }
  }
}

//---------------------------------------------------------------------------
/* $Log: UpdatePriorityQueue.java,v $
/* Revision 1.30  2007/05/11 02:14:30  nalini
/* updated outgoing body subscription so that it sends bodies from startVV
/*
/* Revision 1.29  2006/08/15 21:46:24  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.28  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.27.2.1  2006/06/02 22:18:13  nalini
/* Supports addition and removeal of SS from Outgoing Body Streams
/*
/* Revision 1.27  2006/04/04 16:00:00  nayate
/* Added the ability to (1) delay invalidates, and (2) support transactional updates.
/*
/* Revision 1.26  2005/10/13 00:24:25  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.25  2005/07/18 05:10:23  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.24  2005/03/16 00:45:51  nayate
/* Turned off debug printing
/*
/* Revision 1.23  2005/03/08 22:58:53  nayate
/* Changed priorities to doubles from ints
/*
/* Revision 1.22  2005/03/07 19:10:36  nayate
/* Minor changes
/*
/* Revision 1.21  2004/10/22 20:46:56  dahlin
/* Replaced TentativeState with RandomAccessState in DataStore; got rid of 'chain' in BodyMsg; all self-tests pass EXCEPT (1) get compile-time error in rmic and (2) ./runSDIMSControllerTest fails [related to (1)?]
/*
/* Revision 1.20  2004/07/26 20:03:39  dahlin
/* Fixed typos from windows checkin so it will compile under Linux
/*
/* Revision 1.19  2004/05/26 21:57:59  arun
/* *** empty log message ***
/*
/* Revision 1.18  2004/05/26 19:58:08  nayate
/* Fixed deadlock bug
/*
/* Revision 1.17  2004/05/26 09:45:56  arun
/* *** empty log message ***
/*
/* Revision 1.16  2004/05/25 23:04:58  lgao
/* *** empty log message ***
/*
/* Revision 1.15  2004/05/25 10:35:50  arun
/* Added support for chaining, bound invals and unbinding.
/*
/* Revision 1.14  2004/05/25 08:16:42  lgao
/* *** empty log message ***
/*
/* Revision 1.13  2004/05/24 10:37:34  arun
/* rectified some logic for updating ISStatus. Changes for dealing with bound invals
/* using sdims.
/*
/* Revision 1.12  2004/05/24 05:38:17  zjiandan
/* Add InvalIteratorFilter Class.
/*
/* Revision 1.11  2004/05/24 03:08:58  lgao
/* Add syncBodyCheck...
/*
/* Revision 1.10  2004/05/24 01:34:51  arun
/* added check in SDIMSController to check if inval is newer than existing body on store to prevent SDIMS from polluting its directory when it gets an inval for its own local write
/*
/* Revision 1.9  2004/05/20 01:06:11  arun
/* added some code for recovery support in store. Added maxVV definition in COunterVV etc.
/*
/* Revision 1.8  2004/05/15 00:23:17  lgao
/* Add flag to distingurish demand fetch from prefetch data.
/*
/* Revision 1.7  2004/05/13 20:09:10  dahlin
/* When core has applied a bound inval, it tells originator of the write
/*
/* Revision 1.6  2004/05/12 17:12:36  lgao
/* Modify the getNext() to block when no data to return
/*
/* Revision 1.5  2004/04/30 19:27:03  nayate
/* Uncommented some previously commented code
/*
/* Revision 1.4  2004/04/27 22:56:02  lgao
/* Adding the uniform testing input format into the UpdatePriorityQueue.
/* The input is a file fed from stdin to the testing program.
/*
/* Revision 1.3  2004/04/27 07:59:05  lgao
/* Initial implementation for UpdatePriorityQueue
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
