package code;

 /** 
 **/ 
// Synchronization: no need; caller holds updatePriorityQueue lock

public class UpdateRangeList{

 /** 
 *  Data members 
 **/ 
  private double priority;
  private UpdateRangeList.Range headRange;
  private ObjId oid;

 /** 
 *  Data members 
 **/ 
  public
  UpdateRangeList(double p, ObjId id){
    priority = p;
    this.headRange = null;
    oid = id;
  }

 /** 
 *  Set the priority of this update range 
 **/ 
  public void
  setPriority(double p){
    this.priority = p;
  }

 /** 
 *  Return the priority of this update range 
 **/ 
  public double
  getPriority(){
    return(this.priority);
  }

 /** 
 *  Return the ID of the updated object 
 **/ 
  public ObjId
  getOID(){
    return(this.oid);
  }

 /** 
 *  Insert a new range to be updated in the list of ranges 
 *  
 *  Merging this range into the list handles overlaps properly -- if 
 *  the preceeding range overlaps, make one range spanning both; if 
 *  subsequent range overlaps, make one range spanning both; if this 
 *  overlaps with more than one, make one range spanning all 
 **/ 
  public void
  addRange(long start, long length){
    Range insertRange = null;
    Range range = null;
    Range prevRange = null;
    long end = 0;

    end = start + length - 1;
    assert((start >= 0) && (end >= start));
    if((this.headRange == null) || (start <= (this.headRange.end + 1))){
      // The new range should be put at the front of the list
      insertRange = new Range(start, end, this.headRange);
      this.headRange = insertRange;
    }else{
      // The new range goes elsewhere in the list
      prevRange = this.headRange;
      range = this.headRange.nextRange;
      while((range != null) && (start > (range.end + 1))){
        prevRange = range;
        range = range.nextRange;
      }
      insertRange = new Range(start, end, prevRange.nextRange);
      prevRange.nextRange = insertRange;
    }

    // Merge the new range with subsumed and overlapping ranges
    range = insertRange.nextRange;
    while((range != null) && ((insertRange.end + 1) >= range.start)){
      insertRange.start = Math.min(insertRange.start, range.start);
      insertRange.end = Math.max(insertRange.end, range.end);
      insertRange.removeNext();
      range = insertRange.nextRange;
    }
    assert(this.checkListIntegrity());
  }

 /** 
 *  Remove a range from the list of ranges 
 **/ 
  public void
  cancelRange(long start, long length){
    Range range = null;
    long end = 0;
    long leftStart = 0;
    long leftEnd = 0;
    long rightStart = 0;
    long rightEnd = 0;
    Range prevRange = null;
    Range leftRange = null;
    Range rightRange = null;

    end = start + length - 1;
    assert((start >= 0) && (end >= start));

    // Add a dummy head range
    this.headRange = new Range(-1, -1, this.headRange);
    prevRange = this.headRange;
    range = this.headRange.nextRange;
    while(range != null){
      // Remove each range and replace it with its left and right components
      prevRange.removeNext();
      leftStart = range.start;
      leftEnd = Math.min(range.end, start - 1);
      if(leftStart <= leftEnd){
        // Add this range in
        leftRange = new Range(leftStart, leftEnd, prevRange.nextRange);
        prevRange.nextRange = leftRange;
        prevRange = leftRange;
      }
      rightStart = Math.max(range.start, end + 1);
      rightEnd = range.end;
      if(rightStart <= rightEnd){
        // Add this range in
        rightRange = new Range(rightStart, rightEnd, prevRange.nextRange);
        prevRange.nextRange = rightRange;
        prevRange = rightRange;
      }
      range = prevRange.nextRange;
    }
    // Remove the dummy head block
    range = this.headRange;
    this.headRange = this.headRange.nextRange;
    range.nextRange = null;
    assert(this.checkListIntegrity());
  }

 /** 
 *  Return true if there are no ranges in the list 
 **/ 
  public boolean
  isEmpty(){
    return(this.headRange == null);
  }

 /** 
 *  Get the start of the next range 
 **/ 
  public long
  nextRangeStart(){
    assert(!this.isEmpty());
    return(this.headRange.start);
  }

 /** 
 *  Get the length of the next range 
 **/ 
  public long
  nextRangeLength(){
    assert(!this.isEmpty());
    return(this.headRange.end - this.headRange.start + 1);
  }

 /** 
 *  Remove the next range 
 **/ 
  public void
  removeNextRange(){
    Range oldHead = null;

    assert(!this.isEmpty());
    oldHead = this.headRange;
    this.headRange = this.headRange.nextRange;
    oldHead.nextRange = null;
  }

 /** 
 *  Print the range list 
 **/ 
  public void
  displayList(){
    Range range = null;

    System.out.println(priority + "  " + oid.toString() + " : ");
    range = this.headRange;
    while(range != null){
      System.out.println("\t" + range.start +
                         " " + (range.end - range.start + 1));
      range = range.nextRange;
    }
  }

 /** 
 *  Used for testing 
 **/ 
  public static
  void main(String[] argv){
    UpdateRangeList url = null;

    Env.verifyAssertEnabled();
    System.out.println("Testing UpdateRangeList...");
    UpdateRangeList.testAdd();
    UpdateRangeList.testCancel();
    UpdateRangeList.testMisc();
    System.out.println("...finished");
  }

 /** 
 *  Assert that the list is valid 
 **/ 
  private boolean
  checkListIntegrity(){
    Range range = null;

    range = this.headRange;
    while(range != null){
      assert(range.start >= 0);
      assert(range.end >= range.start);
      if(range.nextRange != null){
        // Make sure our intervals are (1) disjoint and (2) ordered
        assert((range.end + 1) < range.nextRange.start);
      }
      range = range.nextRange;
    }
    return(true);
  }

 /** 
 *  Test the addRange method 
 **/ 
  private static void
  testAdd(){
    UpdateRangeList url = null;

    // (1) Create a range list
    url = new UpdateRangeList(0.1, new ObjId("obj1"));
    url.addRange(5, 10);
    assert((url.headRange != null) &&
           (url.headRange.start == 5) &&
           (url.headRange.end == 14) &&
           (url.headRange.nextRange == null));

    // (2) Add an immediately following range
    url.addRange(15, 5);
    assert((url.headRange != null) &&
           (url.headRange.start == 5) &&
           (url.headRange.end == 19) &&
           (url.headRange.nextRange == null));

    // (3) Add a subsumed entry
    url.addRange(10, 5);
    assert((url.headRange != null) &&
           (url.headRange.start == 5) &&
           (url.headRange.end == 19) &&
           (url.headRange.nextRange == null));

    // (4) Add a disjoint entry
    url.addRange(100, 5);
    assert((url.headRange != null) &&
           (url.headRange.start == 5) &&
           (url.headRange.end == 19) &&
           (url.headRange.nextRange != null) &&
           (url.headRange.nextRange.start == 100) &&
           (url.headRange.nextRange.end == 104) &&
           (url.headRange.nextRange.nextRange == null));

    // (5) Add an entry that subsumes both entries
    url.addRange(5, 100);
    assert((url.headRange != null) &&
           (url.headRange.start == 5) &&
           (url.headRange.end == 104) &&
           (url.headRange.nextRange == null));

    // (6) Add an entry that touches on the left
    url.addRange(3, 2);
    assert((url.headRange != null) &&
           (url.headRange.start == 3) &&
           (url.headRange.end == 104) &&
           (url.headRange.nextRange == null));

    // (7) Add an entry that touches on the right
    url.addRange(105, 50);
    assert((url.headRange != null) &&
           (url.headRange.start == 3) &&
           (url.headRange.end == 154) &&
           (url.headRange.nextRange == null));

    // (8) Add an entry disjointly to the left and right
    url.addRange(1, 1);
    url.addRange(156, 1);
    assert((url.headRange != null) &&
           (url.headRange.start == 1) &&
           (url.headRange.end == 1) &&
           (url.headRange.nextRange != null) &&
           (url.headRange.nextRange.start == 3) &&
           (url.headRange.nextRange.end == 154) &&
           (url.headRange.nextRange.nextRange != null) &&
           (url.headRange.nextRange.nextRange.start == 156) &&
           (url.headRange.nextRange.nextRange.end == 156) &&
           (url.headRange.nextRange.nextRange.nextRange == null));

    // (9) Add an entry that subsumes those exactly
    url.addRange(1, 156);
    assert((url.headRange != null) &&
           (url.headRange.start == 1) &&
           (url.headRange.end == 156) &&
           (url.headRange.nextRange == null));

    // (10) Add an entry that subsumes the result completely
    url.addRange(0, 158);
    assert((url.headRange != null) &&
           (url.headRange.start == 0) &&
           (url.headRange.end == 157) &&
           (url.headRange.nextRange == null));

    // (11) Add an entry that is later and then squeeze in a piece exactly
    url.addRange(160, 40);
    url.addRange(158, 2);
    assert((url.headRange != null) &&
           (url.headRange.start == 0) &&
           (url.headRange.end == 199) &&
           (url.headRange.nextRange == null));
  }

 /** 
 *  Test the cancel method 
 **/ 
  private static void
  testCancel(){
    UpdateRangeList url = null;

    url = new UpdateRangeList(0.1, new ObjId("obj1"));

    // (1) Remove a range from the middle of the first range
    url.addRange(10, 10);
    url.cancelRange(12, 7);
    assert((url.headRange != null) &&
           (url.headRange.start == 10) &&
           (url.headRange.end == 11) &&
           (url.headRange.nextRange != null) &&
           (url.headRange.nextRange.start == 19) &&
           (url.headRange.nextRange.end == 19) &&
           (url.headRange.nextRange.nextRange == null));

    // (2) Remove a range overlapping the front
    url.cancelRange(10, 1);
    assert((url.headRange != null) &&
           (url.headRange.start == 11) &&
           (url.headRange.end == 11) &&
           (url.headRange.nextRange != null) &&
           (url.headRange.nextRange.start == 19) &&
           (url.headRange.nextRange.end == 19) &&
           (url.headRange.nextRange.nextRange == null));

    // (3) Remove a range subsuming the front
    url.cancelRange(5, 10);
    assert((url.headRange != null) &&
           (url.headRange.start == 19) &&
           (url.headRange.end == 19) &&
           (url.headRange.nextRange == null));

    // (4) Remove a range subsuming the remaining list exactly
    url.cancelRange(19, 1);
    assert(url.headRange == null);

    // (5) Remove a range that chops the entry on the right
    url.addRange(10, 10);
    url.cancelRange(15, 10);
    assert((url.headRange != null) &&
           (url.headRange.start == 10) &&
           (url.headRange.end == 14) &&
           (url.headRange.nextRange == null));

    // (6) Remove a range that chops the entry on the left
    url.cancelRange(5, 7);
    assert((url.headRange != null) &&
           (url.headRange.start == 12) &&
           (url.headRange.end == 14) &&
           (url.headRange.nextRange == null));

    // (7) Build disjoint ranges, remove one completely, and one partially
    url.addRange(20, 2);
    url.addRange(25, 2);
    url.cancelRange(20, 6);
    assert((url.headRange != null) &&
           (url.headRange.start == 12) &&
           (url.headRange.end == 14) &&
           (url.headRange.nextRange != null) &&
           (url.headRange.nextRange.start == 26) &&
           (url.headRange.nextRange.end == 26) &&
           (url.headRange.nextRange.nextRange == null));
  }

 /** 
 *  Test miscellaneous methods 
 **/ 
  public static void
  testMisc(){
    UpdateRangeList url = null;

    url = new UpdateRangeList(0.1, new ObjId("obj1"));
    assert(url.getPriority() == 0.1);
    url.setPriority(0.2);
    assert(url.getPriority() == 0.2);
    assert(url.getOID().equals(new ObjId("obj1")));
    assert(url.isEmpty());

    url.addRange(100, 200);
    assert(!url.isEmpty());
    assert(url.nextRangeStart() == 100);
    assert(url.nextRangeLength() == 200);
    url.removeNextRange();
    assert(url.isEmpty());
  }

 /** 
 *  Internal class to define ranges 
 **/ 
  static class Range{

 /** 
 *  Constructor 
 **/ 
    public
    Range(long newStart, long newEnd, Range newNextRange){
      this.start = newStart;
      this.end = newEnd;
      this.nextRange = newNextRange;
    }

 /** 
 *  Remove the next entry 
 **/ 
    public boolean
    removeNext(){
      boolean removed = false;
      Range oldNextRange = null;

      if(this.nextRange != null){
        oldNextRange = this.nextRange;
        this.nextRange = this.nextRange.nextRange;
        oldNextRange.nextRange = null; // To help Java garbage collector
        removed = true;
      }
      return(removed);
    }

 /** 
 *  Data members 
 **/ 
    public long start;
    public long end;
    public Range nextRange;
  }
}

//---------------------------------------------------------------------------
/* $Log: UpdateRangeList.java,v $
/* Revision 1.7  2005/10/16 10:18:59  nayate
/* Rewrote UpdateRangeList
/*
/* Revision 1.6  2005/03/08 22:58:53  nayate
/* Changed priorities to doubles from ints
/*
/* Revision 1.5  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.4  2004/04/27 22:56:02  lgao
/* Adding the uniform testing input format into the UpdatePriorityQueue.
/* The input is a file fed from stdin to the testing program.
/*
/* Revision 1.3  2004/04/27 07:59:05  lgao
/* Initial implementation for UpdatePriorityQueue
/*
/* Revision 1.2  2004/04/15 20:04:25  nayate
/* New Makefile; added provision to allow CVS to append file modification
/* logs to files.
/* */
//---------------------------------------------------------------------------
