package code.branchDetecting;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;


public class BranchDetector{

  SegmentCheckInterface senderSegmentChecker;
  SegmentCheckInterface receiverSegmentChecker;

  public BranchDetector(SegmentCheckInterface ssc){
    senderSegmentChecker = ssc;
  }

  public void registerReceiverSegementChecker(SegmentCheckInterface rsc){
    receiverSegmentChecker = rsc;
  }
  
  public void detectBranches(){
    BranchKnowledge rK = receiverSegmentChecker.getKnowledge();
    BranchKnowledge sK = senderSegmentChecker.getKnowledge();
    assert sK != null;
    assert rK != null;
    for(Long nodeId : sK.getNodeIDSet()){
      SegmentGroups gsSender = sK.getSegmentGroups(nodeId);
      SegmentGroups gsReceiver = rK.getSegmentGroups(nodeId);
      if(gsReceiver!=null){
        detectBranches(gsReceiver, gsSender);
      }      
    }
  }

  private void detectBranches(SegmentGroups gsReceiver, SegmentGroups gsSender){
    assert gsReceiver.getNodeId() == gsSender.getNodeId();
    Iterator<SegmentGroup> receiverIter = gsReceiver.getIterator();
    Iterator<SegmentGroup> senderIter = gsSender.getIterator();
    if(!receiverIter.hasNext() || !senderIter.hasNext()){
      return;
    }
    SegmentGroup gReceiver = receiverIter.next();//gsReceiver.getNextGroup();
    SegmentGroup gSender = senderIter.next();//gsSender.getNextGroup();

    try{
      while(gReceiver != null && gSender != null){      
        if(gReceiver.getStartTS() == gSender.getStartTS()){

          for(Segment rs : gReceiver.getSegments()){
            Segment ss = gSender.getSegment(rs.bid);

            if(ss == null){
              // Probably Do Nothing
              continue;
            } 
            assert rs.bid.equals(ss.bid);

            try{

              if(rs.getEndTS() < ss.getEndTS()){
                HashTSTuple startOfSSeg = senderSegmentChecker.checkCompatibility(rs, ss);
                if(startOfSSeg != null){
                  gsSender.addSegment(
                      new Segment(startOfSSeg.timeStamp, ss.getEndTS(),
                          new BranchID(ss.bid.getIDint(), startOfSSeg.timeStamp, startOfSSeg.hash),
                          ss.getEndHash(),ss.isLeaf()));

                  ss.setEndTS(rs.getEndTS());
                  ss.setEndHash(rs.getEndHash());
                  ss.makeNonLeaf();
                  senderSegmentChecker.createNewBranch(ss.bid, startOfSSeg.timeStamp);
                }                          
              } else if(rs.getEndTS() > ss.getEndTS()){
                HashTSTuple startOfRSeg = receiverSegmentChecker.checkCompatibility(ss, rs);
                if( startOfRSeg != null){
                  gsReceiver.addSegment(
                      new Segment(startOfRSeg.timeStamp, rs.getEndTS(),
                          new BranchID(rs.bid.getIDint(), startOfRSeg.timeStamp, startOfRSeg.hash),
                          rs.getEndHash(),rs.isLeaf()));

                  rs.setEndTS(ss.getEndTS());
                  rs.setEndHash(ss.getEndHash());
                  rs.makeNonLeaf();
                  receiverSegmentChecker.createNewBranch(rs.bid, startOfRSeg.timeStamp);
                }
              } else {
                senderSegmentChecker.checkCompatibility(rs,ss);
              }

            } catch (ConflictingSegmentsException e){
              /* There has been a fork.. */
              LinkedList<HashTSTuple> res = findForkPoint(rs,ss);
              assert res != null;
              HashTSTuple endOfCommonSeg = res.removeFirst();
              HashTSTuple startOfRSeg = res.removeFirst();
              HashTSTuple startOfSSeg = res.removeFirst();
              
              assert endOfCommonSeg.timeStamp < startOfRSeg.timeStamp : 
                "End:"+endOfCommonSeg.timeStamp+",RStart:"+startOfRSeg.timeStamp;
              assert endOfCommonSeg.timeStamp < startOfSSeg.timeStamp :
                "End:"+endOfCommonSeg.timeStamp+",SStart:"+startOfSSeg.timeStamp;

              gsSender.addSegment(
                  new Segment(startOfSSeg.timeStamp, ss.getEndTS(),
                      new BranchID(ss.bid.getIDint(), startOfSSeg.timeStamp, startOfSSeg.hash),
                      ss.getEndHash(),ss.isLeaf()));

              ss.setEndTS(endOfCommonSeg.timeStamp);
              ss.setEndHash(endOfCommonSeg.hash);
              ss.makeNonLeaf();

              senderSegmentChecker.createNewBranch(ss.bid, startOfSSeg.timeStamp);
              
              gsReceiver.addSegment(
                  new Segment(startOfRSeg.timeStamp, rs.getEndTS(),
                      new BranchID(rs.bid.getIDint(), startOfRSeg.timeStamp, startOfRSeg.hash),
                      rs.getEndHash(),rs.isLeaf()));

              rs.setEndTS(endOfCommonSeg.timeStamp);
              rs.setEndHash(endOfCommonSeg.hash);
              rs.makeNonLeaf();
              
              receiverSegmentChecker.createNewBranch(rs.bid, startOfRSeg.timeStamp);



            }

          }
          gReceiver = receiverIter.next();//gsReceiver.getNextGroup();
          gSender = senderIter.next();//gsSender.getNextGroup();
        } else if (gReceiver.getStartTS() < gSender.getStartTS()){
          gReceiver = receiverIter.next();//gsReceiver.getNextGroup();
        } else {
          gSender = senderIter.next();//gsSender.getNextGroup();
        }            

      }
    }catch(NoSuchElementException e){

    }

  }

  private LinkedList<HashTSTuple> findForkPoint(Segment receiverSeg, Segment senderSeg){
    Segment shortSeg,longSeg;
    SegmentCheckInterface shortSegIF, longSegIF;
    if(receiverSeg.getEndTS() < senderSeg.getEndTS()){
      shortSeg = receiverSeg;
      longSeg = senderSeg;
      shortSegIF = receiverSegmentChecker;
      longSegIF = senderSegmentChecker;
    } else {
      shortSeg = senderSeg;
      longSeg = receiverSeg;
      shortSegIF = senderSegmentChecker;
      longSegIF = receiverSegmentChecker;
    }

    BranchID bid = shortSeg.bid;
    long startTS = shortSeg.getStartTS();
    long endTS = shortSeg.getEndTS();    
    do{
      long center = (startTS+endTS)/2;
      HashTSTuple t1 = 
        shortSegIF.getSummaryHashAtOrBefore(bid, center);
      HashTSTuple t2 = 
        longSegIF.getSummaryHashAtOrBefore(bid, center);
      if(t1.equals(t2)){
        HashTSTuple t3 = shortSegIF.getSummaryHashAfter(bid, t1.timeStamp);
        assert t3.timeStamp > t1.timeStamp : "t3 : " + t3 + ", t1 : " + t1;
        HashTSTuple t4 = longSegIF.getSummaryHashAfter(bid, t1.timeStamp);
        assert t4.timeStamp > t1.timeStamp : "t4 : " + t4 + ", t1 : " + t1;
        if(!t3.equals(t4)){
          // We've found a forking point
          LinkedList<HashTSTuple> ret = new LinkedList<HashTSTuple>();
          ret.add(t1);
          if(receiverSeg.getEndTS() < senderSeg.getEndTS()){
            ret.add(t3);
            ret.add(t4);
          } else {
            ret.add(t4);
            ret.add(t3);
          }
          return ret;
        } else {
          // go forward
          startTS = center;
        }        
      } else {
        // go backward
        endTS = center;        
      }
    } while(startTS != endTS);
    assert false;
    return null;
  }

}
