package code.branchDetecting;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.Map.Entry;

import code.NoSuchEntryException;
import code.NodeId;
import code.VVIterator;
import code.security.SecureInv;
import code.security.SecurePreciseInv;
import code.simulator.*;

public class BranchKnowledge implements Serializable{

  // mapping from NodeId to SegmentGroup
  private HashMap<Long,SegmentGroups> knowledge;


  public BranchKnowledge(){
    knowledge = new HashMap<Long,SegmentGroups>();
  }

  /**
   * Return a SegmentGroups object corresponding nid not branchID;
   * If no such SegmentGroups exists, return null.
   * @param nid 
   */
  public SegmentGroups getSegmentGroups(NodeId nid){
    // We should use NodeId not BranchID such that all branches rooted
    // from a same nodeId are grouped together 
    return knowledge.get(nid.getIDint());
    
  }
  public SegmentGroups getSegmentGroups(Long nid){
    // We should use NodeId not BranchID such that all branches rooted
    // from a same nodeId are grouped together 
    return knowledge.get(nid);    
  }
  
  public Set<Long> getNodeIDSet(){
    return knowledge.keySet();
  }
  
  /**
   * Returns a segment with BranchID bid.
   * If no such segment exists, returns null
   * @param bid
   */
  public Segment getSegment(BranchID bid){
    SegmentGroups sgs = knowledge.get(bid.getIDint());
    if(sgs==null) return null;
    return sgs.getSegment(bid);
  }
  
  public Segment removeSegment(BranchID bid){
    SegmentGroups sgs = knowledge.get(bid.getIDint());
    if(sgs==null) return null;
    return sgs.removeSegment(bid);
  }
  
  /**
   * Update this branch knowledge so that its segments reflect
   * this invalidation.
   * hash is a hash 
   * @param si
   * @param hash
   */
  public void applyInval(SecureSimPreciseInv si){
    NodeId nid = si.getNodeId();
    Hash hash = si.generateMyHash();
    BranchID bid;
    if(!( nid instanceof BranchID)){
      assert false;
      bid = new BranchID(nid.getIDint(),-1, Hash.NullHash);
    }  else {
      bid = (BranchID)nid;
    }
    SegmentGroups sgs;

    if(knowledge.containsKey(nid.getIDint())){
      sgs = knowledge.get(nid.getIDint());
    } else {
      sgs = new SegmentGroups(nid.getIDint());
      knowledge.put(nid.getIDint(), sgs);
    }    
    try{
      assert bid!=null;
      assert hash!=null;
      assert si!=null;
      assert si.getEndVV()!=null;
      assert si.getEndVV().getStampByServer(bid)>=0;
      assert sgs!=null;
      // princem: assert that I have exactly one entry in my DVV with the same node Id as mine 
      // and only if my accept stamp is a forked branch
      // no non-local entry is allowed only when my branch is the default branch
      NodeId n = si.getNodeId();
      assert n instanceof BranchID;
      BranchID creator = (BranchID)n;
      if(!si.getDVV().containsNodeId(creator)){
        VVIterator vvi = si.getDVV().getIterator();
        BranchID parent = null;
        while(vvi.hasMoreElements()){
          BranchID b = (BranchID)vvi.getNext();
          if(b.getIDint() == bid.getIDint() && !b.equals(bid)){
            if(parent != null){
              assert false;
            }else{
              parent = b;
            }
          }
        
        }
        sgs.updateSegment(bid, si.getEndVV().getStampByServer(bid), hash, parent);
      }else{
        sgs.updateSegment(bid, si.getEndVV().getStampByServer(bid), hash, null);
      }
    }catch(NoSuchEntryException e){
      e.printStackTrace();
      assert false;
    }
  }

  public String toString(){
    return knowledge.toString();
  }

  public boolean equals(Object o){
    if(!(o instanceof BranchKnowledge)){
      return false;
    }else{
      BranchKnowledge bk = (BranchKnowledge)o;
      return bk.knowledge.equals(this.knowledge);
    }
  }
  
  public BranchKnowledge clone(){
    BranchKnowledge ret = new BranchKnowledge();
    Iterator<Entry<Long,SegmentGroups>> i = knowledge.entrySet().iterator();
    while(i.hasNext()){
      Entry<Long,SegmentGroups> e = i.next();
      ret.knowledge.put(e.getKey(), e.getValue().clone());
    }

    return ret;
  }

}
