package code.branchDetecting;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.TreeMap;
import java.util.Map.Entry;

import code.simulator.*;

public class SegmentGroup implements Comparable<SegmentGroup>, Serializable{

  /**
   *  time stamp of beginning of segments belonging to this group
   */
  private long startTS;
  
  /**
   * a set of segments belonging to this group
   */
  private TreeMap<BranchID,Segment> segmentsSet; 
  
  public SegmentGroup(long sTS){
    this.startTS = sTS;
    segmentsSet = new TreeMap<BranchID,Segment>();
  }
  
  /**
   * Add a segment s into this group.
   * s.startTS should be equal to this group's start TS
   * @param s
   */
  public void addSegment(Segment s){
    assert s.startTS == this.startTS;
    assert s.bid.startTS == this.startTS;

    segmentsSet.put(s.bid, s);
  }
  
  /**
   * Returns a segment with BranchID bid.
   * If no such segment exists, returns null
   * @param bid
   */
  public Segment getSegment(BranchID bid){
    return segmentsSet.get(bid);
  }
  
  public Segment removeSegment(BranchID bid){
    return segmentsSet.remove(bid);
  }
  
  public Collection<Segment> getSegments(){
    return segmentsSet.values();
  }
  
  /**
   * Update a segment with bid as new invalidation is applied.
   * That is, it will only lengthen a existing segment or create
   * a new segment if there is no segment with BranchID bid
   * @param bid BranchID of the updating segment
   * @param endTS new end time stamp
   * @param endHash new hash value at the new end time stamp
   */
  public void updateSegment(BranchID bid, long endTS, Hash endHash){
    Segment s;
    if(segmentsSet.containsKey(bid)){
      s = segmentsSet.get(bid);
      if(s.getEndTS() >= endTS){
        return;
      }
      assert s.isLeaf();
      s.setEndTS(endTS);
      s.setEndHash(endHash);
    } else {
      s = new Segment(endTS,endTS,bid,endHash,true);
    }
    segmentsSet.put(bid,s);
  }
  
  /**
   * If this group contains segment with bid, make it non-leaf
   * @param bid BranchID being searched
   * @return true if it found a segment with its BranchID bid, false otherwise
   */
  public boolean makeNonLeaf(BranchID bid){
    Segment seg = segmentsSet.get(bid);
    if(seg!=null){
      seg.makeNonLeaf();
      return true;
    }
    return false;
  }
  
  public int compareTo(SegmentGroup sg){
    return this.startTS<sg.startTS?-1:
      this.startTS>sg.startTS?1:0;    
  }
  
  public boolean equals(Object o){
    if(!(o instanceof SegmentGroup)) return false;
    SegmentGroup sg = (SegmentGroup)o;
    return sg.startTS==this.startTS && this.segmentsSet.equals(sg.segmentsSet);
  }
  
  public String toString(){
    return "Segment Group(Start TS = " + startTS + ")\n" + segmentsSet.toString() + "\n";
  }
  
  public long getStartTS(){
    return startTS;
  }
  
  public SegmentGroup clone(){
    SegmentGroup ret = new SegmentGroup(this.startTS);
    Iterator<Entry<BranchID,Segment>> i = segmentsSet.entrySet().iterator();
    while(i.hasNext()){
      Entry<BranchID,Segment> e = i.next();
      ret.segmentsSet.put(e.getKey(), e.getValue().clone());
    }
    return ret;
  }
  
  

}
