package code.branchDetecting;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.PriorityQueue;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.Map.Entry;

import code.simulator.*;
import code.NodeId;

public class SegmentGroups implements Serializable{
  /**
   * NodeId to which all segments in this groups belong
   */
  private long nodeId; 
  /**
   * Map<starting time stamp,SegmentGroup> 
   */
  private TreeMap<Long,SegmentGroup> map;
  
  private long lastTS=-10; // only for debugging purpose
  
  public SegmentGroups(long nodeId){
    this.nodeId = nodeId;
    map = new TreeMap<Long,SegmentGroup>();
  }
  
  public long getNodeId(){
    return nodeId;
  }
  
  /**
   * @return an iterator that iterates over SegmentGroup objects 
   * in an ascending order of their startTS
   */
  public Iterator<SegmentGroup> getIterator(){
    return (new TreeSet<SegmentGroup>(map.values())).iterator();        
  }
  
  /**
   * Returns a segment with BranchID bid.
   * If no such segment exists, returns null
   * @param bid
   */
  public Segment getSegment(BranchID bid){
    SegmentGroup sg = map.get(bid.startTS);
    if(sg == null) return null;
    return sg.getSegment(bid);
  }
  
  public Segment removeSegment(BranchID bid){
    SegmentGroup sg = map.get(bid.startTS);
    if(sg == null) return null;
    return sg.removeSegment(bid);
  }
  
  /**
   * Add a segment to this collection of segment groups
   * If a corresponding group for this segment does not exist,
   *  it will be created
   */
  public void addSegment(Segment s){
    assert s.startTS > lastTS;
    
    SegmentGroup sg;
    if(map.containsKey(s.startTS)){
      sg = map.get(s.startTS);      
    } else {
      sg = new SegmentGroup(s.startTS);
    }
    
    sg.addSegment(s);
    map.put(s.startTS, sg);    
  }
  
  /**
   * Update a segment as new invalidation is applied/received
   * @param bid branchID of the segment being updated
   * @param endTS new end time stamp of the segment
   * @param endHash new hash of the segment
   */
  public void updateSegment(BranchID bid, long endTS, Hash endHash, BranchID parent){    
    SegmentGroup sg = map.get(bid.startTS);
    if(sg == null){
      sg = new SegmentGroup(bid.startTS);
      if(parent!=null){
        SortedMap<Long,SegmentGroup> submap = map.headMap(bid.startTS);
        SegmentGroup[] a = new SegmentGroup[submap.size()];
        a = submap.values().toArray(a);
        // search for its parent and make it non-leaf 
        for(SegmentGroup sg2 : a){
          // break if we find out the parent
          if(sg2.makeNonLeaf(parent))
            break;
        }
      }
      map.put(bid.startTS, sg);      
    }
    sg.updateSegment(bid, endTS, endHash);
  }
  
  public String toString(){
    String ret = "";
    Iterator<Entry<Long,SegmentGroup>> i = map.entrySet().iterator();
    
    while(i.hasNext()){
      Entry<Long,SegmentGroup> e = i.next();
      ret += "startTS : " + e.getKey() + "\n";
      ret += e.getValue().toString() + "\n";
    }
    
    return ret;
  }
  
  /**
   * Return a deep copy of this object
   */
  public SegmentGroups clone(){
    SegmentGroups ret = new SegmentGroups(this.nodeId);
    ret.lastTS = this.lastTS;
    Iterator<Entry<Long,SegmentGroup>> i = map.entrySet().iterator();
    while(i.hasNext()){
      Entry<Long,SegmentGroup> e = i.next();
      ret.map.put(e.getKey(), e.getValue().clone());
    }
    
    return ret;
  }
  
  public boolean equals(Object o){
    if(!(o instanceof SegmentGroups)){
      return false;
    }else{
      SegmentGroups sg = (SegmentGroups)o;
      return nodeId == sg.nodeId && this.map.equals(sg.map);
    }
  }
  
  
}
