 /** 
/* BayouTest
 *
 * 2-node test for Bayou
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class BayouTest {
  private BayouNode node1;

  private BayouFSLocalInterface node1_interface;
  
  protected static long NODE_1_ID = 0;
  protected static long NODE_2_ID = 1;
  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.Bayou.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.Bayou.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.Bayou.nodemap";
  protected String OVERLOG_PATH = "Bayou.olg";


  protected static int sizeOfWrites = 20;
  protected static byte value_1 = 65; //"A";
  protected static byte value_2 = 66; //"B";
  protected static byte value_3 = 67; //"C";
  protected static long timeToWait = 2000; // time to retry a read after objNotFound
  protected static long waitAntiEntropyEnd = 10000; // time to wait to make sure anti-entropy
                                                  // has ended
  protected static long antiEntropyPeriod = 10000; // time between antientropies

  static private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 3000;
  static private BarrierClient c;

  
  private boolean dbg = true;

 

  
 /** 
 *  Constructor 
 **/ 
 
  public BayouTest(String overlogPath) {
    OVERLOG_PATH = overlogPath;
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(NODEID_MAP_PATH);

    node1 = new BayouNode(CONFIG_PATH, 
			  CONFIG_P2_PATH,
			  new NodeId(NODE_1_ID),
			  true,
			  OVERLOG_PATH,
			  NODEID_MAP_PATH);

     node1_interface = node1.getFSInterface();
  }

 /** 
 *  Start the nodes 
 **/ 

  public void startNodes() throws Exception {
    node1.start();
    Thread.currentThread().sleep(5000);
    System.out.println("Started node 1");

  }

 /** 
 *  Shutdown the nodes 
 **/ 

  public void shutdownNodes() throws Exception {
    node1.shutdown();
  }



 /** 
 *  Make configuration files 
 **/ 
  private static void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.BayouTest-" + 
			    NODE_1_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(NODE_2_ID),
			   "localhost",
			   9888,
			   9889,
			   9891,
			   9892,
                            9890,
                           "test" + File.separatorChar + "tmp.BayouTest-" + 
			   NODE_2_ID+".db",
			   "/*",
			   -1L,
			   "localhost",
			   9893,
			   9894,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

  private static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_1_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_2_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_1_ID), new OverlogId("localhost:5000"));
    NodeIdMap.add(new NodeId(NODE_2_ID), new OverlogId("localhost:5001"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
//  Converts values to byte arrays
//---------------------------------------------------------------------------

  private byte[] convertToArray(byte value){
    byte[] b = new byte[sizeOfWrites];
    for(int i = 0; i < sizeOfWrites; i++) {
      b[i] = value;
    }
    return b;
  }


//---------------------------------------------------------------------------
//  Writes a value to the specified interfac
//---------------------------------------------------------------------------

  private void write(BayouFSLocalInterface fsInterface, ObjId objId, byte[] b)
    throws IOException{
    fsInterface.write(objId, 0, sizeOfWrites, b);
  }

//---------------------------------------------------------------------------
// Reads an object from the specified interface
// If objNotFound -- sleeps for some time and tries again
//---------------------------------------------------------------------------

  private ImmutableBytes read(BayouFSLocalInterface fsInterface, ObjId objId) {
    ImmutableBytes b=null;;
    try{
      while (true) {
	try{
	  b = fsInterface.read(objId, 0, sizeOfWrites).getBody();
	  break;
	}catch(ObjNotFoundException e){
	  //wait until you recieve obj
	  Env.dprintln(dbg, "ObjNotFoundException for " + objId);
	  Thread.currentThread().sleep(timeToWait);
	}
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    return b;
  }


//---------------------------------------------------------------------------
// Runs a simple test to make sure that anti-entropy works and disconnects
//  as expected
//
//  stage 1:
//  Node 2 writes /b
//
//  stage 2:
//  Node 1 tries to read /b  until anti-entropy occurs
//
//  stage 3:
//  Node 2 waits for a while and writes to /b
//  Node 1 tries to read /b 
//  (should be the old value if anti-entropy hasn't occured)
//  
// 
//---------------------------------------------------------------------------

 public void simpleTest(){
    try{
      System.out.println("Test Started");
      Env.dprintln(dbg, "Starting test");

      ObjId objId_b = new ObjId("/b");

      byte[] value_1_arr = convertToArray(value_1);
      byte[] value_2_arr = convertToArray(value_2);

      ImmutableBytes value_1_imb = new ImmutableBytes(value_1_arr);
      ImmutableBytes value_2_imb = new ImmutableBytes(value_2_arr);

      ImmutableBytes imb;
     
      //
      // stage 1:
      //
      Env.dprintln(dbg, "Finished Stage 1");

      //
      // stage 2:
      //
      imb = read(node1_interface, objId_b);
      Env.dprintln(dbg, "Stage 2:");
      Env.dprintln(dbg, "...Exptected Value: " + value_1_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_1_imb)): "Read does not match expected value";
      
      c.sendBarrierRequest(0, NODE_1_ID); // anti-entropy ended 


      //
      // stage 3:
      //

      c.sendBarrierRequest(0, NODE_1_ID); // helper finished writing

      Thread.currentThread().sleep(500);
      imb = read(node1_interface, objId_b);
      Env.dprintln(dbg, "Stage 3:");
      Env.dprintln(dbg, "...Exptected Value: " + value_1_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_1_imb)): "Read does not match expected value";
 
      Env.dprintln(dbg, "Finished Stage 3");

      System.out.println("Test passed");
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
// Runs a simple test to make sure the nodes work
//
// stage 1:
// Node 1 writes objects into /c dir
// Node 2 writes objects into /d dir
//
// stage 2:
// Node 1 tries to read /d (if no antientropy -- will be objNot found)
// Node 2 tries to read /c (f no antientropy -- will be objNot found)
//
// stage 3:
// Node1 updates /c, and writes to /p
// Node2 updates /d and writes to /q
//
// stage 4:
// Node1 tries to read /q (until objNot found is cleared then reads /d)
// Node2 tries to read /p (until objNot found is cleared then reads /c)
//---------------------------------------------------------------------------


  public void simpleTest2(){
    try{
      System.out.println("Test Started");
      Env.dprintln(dbg, "Starting test");

      ObjId objId_c = new ObjId("/c");
      ObjId objId_d = new ObjId("/d");
      ObjId objId_p = new ObjId("/p");
      ObjId objId_q = new ObjId("/q"); 

      byte[] value_2_crr = convertToArray(value_2);
      byte[] value_3_crr = convertToArray(value_3);

      ImmutableBytes value_2_imb = new ImmutableBytes(value_2_crr);
      ImmutableBytes value_3_imb = new ImmutableBytes(value_3_crr);

      ImmutableBytes imb;
     
      //
      // stage 1:
      //
      write(node1_interface, objId_c, value_2_crr);      
      Env.dprintln(dbg, "Finished Stage 1");

      //
      // stage 2:
      //
      imb = read(node1_interface, objId_d);
      Env.dprintln(dbg, "Stage 2:");
      Env.dprintln(dbg, "...Exptected Value: " + value_2_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_2_imb)): "Read does not match expected value";
      
      Thread.currentThread().sleep(waitAntiEntropyEnd);

      //
      // stage 3:
      //
      write(node1_interface, objId_c, value_3_crr);      
      write(node1_interface, objId_p, value_3_crr);
     
 
      Env.dprintln(dbg, "Finished Stage 3");

      //
      // stage 4:
      //
      
      Env.dprintln(dbg, "Stage 4:");
      //Note: we need to read q before b so that we know that
      // the 2nd round of anti-entropy has occured
      imb = read(node1_interface, objId_q);
      Env.dprintln(dbg, "...Exptected Value: " + value_3_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_3_imb)): "Read does not match expected value";
      imb = read(node1_interface, objId_d);
      Env.dprintln(dbg, "...Exptected Value: " + value_3_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_3_imb)): "Read does not match expected value";

      Env.dprintln(dbg, "Finished Stage 4");
     
      System.out.println("Test passed");
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

  public static void main(String[] args) {
    try {
      Env.verifyAssertEnabled();
      
      if(args.length < 1) {
	System.out.println("Usage: java BayouTest <overlog file>");
	System.exit(0);
      }
     
      String fileName = args[0];


     //
     // Start barrier server - for client and server to synch with each other
     //
    
      barrierServer = new BarrierServer(BARRIER_PORT, 2, 6); // for simple test
     barrierServer.start();

     c =  new BarrierClient("localhost", 
                                          BARRIER_PORT,
                                          (int) NODE_1_ID);

      BayouTest test = new BayouTest(fileName);
      c.sendBarrierRequest(0, NODE_1_ID); // Inform helper that stuff has be set up


      test.startNodes();
      c.sendBarrierRequest(0, NODE_1_ID); // Inform helper that overlog node has been started

      test.simpleTest();
      c.sendBarrierRequest(0, NODE_1_ID); // finished test 1
      test.simpleTest2();
      c.sendBarrierRequest(0, NODE_1_ID); // Inform helper that test has completed shutting down
      
      test.shutdownNodes();
      System.exit(0);
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }
  
}
//---------------------------------------------------------------------------
/* $Log: BayouTest.java,v $
/* Revision 1.5  2007/09/12 19:07:16  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.4  2007/03/09 03:01:39  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.3  2007/01/15 20:26:05  nalini
/* Bayou and NFS works
/*
/* Revision 1.2  2007/01/12 22:30:08  nalini
/* Fixed informRemovedSubscription bug
/*
/* Revision 1.1  2007/01/12 19:09:32  nalini
/* Bayou case-study created
/*
 */
 /** 
