 /** 
/* BayouTest_Helper
 *
 * 2-node test for Bayou
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class BayouTest_Helper {
  private BayouNode node2;

  private BayouFSLocalInterface node2_interface;
    
  private boolean dbg = true;
  static private BarrierClient c;

 /** 
 *  Constructor 
 **/ 
 
  public BayouTest_Helper(String overlogPath){


    node2 = new BayouNode(BayouTest.CONFIG_PATH, 
			  BayouTest.CONFIG_P2_PATH,
			  new NodeId(BayouTest.NODE_2_ID),
			  true,
			  overlogPath,
			  BayouTest.NODEID_MAP_PATH);

    node2_interface = node2.getFSInterface();

  }

 /** 
 *  Start the nodes 
 **/ 

  public void startNodes() throws Exception {
    node2.start();
    Thread.currentThread().sleep(5000);
    System.out.println("Started Node 2");

  }

 /** 
 *  Shutdown the nodes 
 **/ 

  public void shutdownNodes() throws Exception {
    node2.shutdown();
  }


 /** 
 *   Converts values to byte arrays 
 **/ 

  private byte[] convertToArray(byte value){
    byte[] b = new byte[BayouTest.sizeOfWrites];
    for(int i = 0; i < BayouTest.sizeOfWrites; i++) {
      b[i] = value;
    }
    return b;
  }


 /** 
 *   Writes a value to the specified interfac 
 **/ 

  private void write(BayouFSLocalInterface fsInterface, ObjId objId, byte[] b)
    throws IOException{
    fsInterface.write(objId, 0, BayouTest.sizeOfWrites, b);
  }

 /** 
 *  Reads an object from the specified interface 
 *  If objNotFound -- sleeps for some time and tries again 
 **/ 

  private ImmutableBytes read(BayouFSLocalInterface fsInterface, ObjId objId) {
    ImmutableBytes b=null;;
    try{
      while (true) {
	try{
	  b = fsInterface.read(objId, 0, BayouTest.sizeOfWrites).getBody();
	  break;
	}catch(ObjNotFoundException e){
	  //wait until you recieve obj
	  Env.dprintln(dbg, "ObjNotFoundException for " + objId);
	  Thread.currentThread().sleep(BayouTest.timeToWait);
	}
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    return b;
  }


 /** 
 *  Runs a simple test to make sure that anti-entropy works and disconnects 
 *   as expected 
 *  
 *   stage 1: 
 *   Node 2 writes /b 
 *  
 *   stage 2: 
 *   Node 1 tries to read /b until anti entropy occurs 
 *  
 *   stage 3: 
 *   Node 2 waits for a while and writes to /b  
 *   Node 1 tries to read /b  
 *   (should be the old value if anti-entropy hasn't occured) 
 *    
 *   
 **/ 

  public void simpleTest(){
    try{
      System.out.println("Node 2 Test Started");
      Env.dprintln(dbg, "Starting test");

      ObjId objId_b = new ObjId("/b");


      byte[] value_1_arr = convertToArray(BayouTest.value_1);
      byte[] value_2_arr = convertToArray(BayouTest.value_2);

      ImmutableBytes value_1_imb = new ImmutableBytes(value_1_arr);
      ImmutableBytes value_2_imb = new ImmutableBytes(value_2_arr);

      ImmutableBytes imb;
     
      //
      // stage 1:
      //
      write(node2_interface,objId_b, value_1_arr);
      Env.dprintln(dbg, "Finished Stage 1");

      //
      // stage 2:
      //

      c.sendBarrierRequest(0, BayouTest.NODE_2_ID); // anti-entropy ended 
      Env.dprintln(dbg, "Finished Stage 2");

      //
      // stage 3:
      //
      
      Thread.currentThread().sleep(BayouTest.waitAntiEntropyEnd);
      write(node2_interface,objId_b, value_2_arr);
      Env.dprintln(dbg, "Finished writing");
      c.sendBarrierRequest(0, BayouTest.NODE_2_ID); // anti-entropy ended 

      System.out.println("Test passed");
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }


 /** 
 *  Runs a simple test to make sure the nodes work 
 *  
 *  stage 1: 
 *  Node 1 writes objects into /c dir 
 *  Node 2 writes objects into /d dir 
 *  
 *  stage 2: 
 *  Node 1 tries to read /d (if no antientropy -- will be objNot found) 
 *  Node 2 tries to read /c (f no antientropy -- will be objNot found) 
 *  
 *  stage 3: 
 *  Node1 updates /c, and writes to /p 
 *  Node2 updates /d and writes to /q 
 *  
 *  stage 4: 
 *  Node1 tries to read /q (until objNot found is cleared then reads /d) 
 *  Node2 tries to read /p (until objNot found is cleared then reads /c) 
 **/ 


  public void simpleTest2(){
    try{
      System.out.println("Node 2 Test Started");
      Env.dprintln(dbg, "Starting test");

      ObjId objId_c = new ObjId("/c");
      ObjId objId_d = new ObjId("/d");
      ObjId objId_p = new ObjId("/p");
      ObjId objId_q = new ObjId("/q"); 

      byte[] value_2_crr = convertToArray(BayouTest.value_2);
      byte[] value_3_crr = convertToArray(BayouTest.value_3);

      ImmutableBytes value_2_imb = new ImmutableBytes(value_2_crr);
      ImmutableBytes value_3_imb = new ImmutableBytes(value_3_crr);

      ImmutableBytes imb;
     
      //
      // stage 1:
      //
      write(node2_interface,objId_d, value_2_crr);
      Env.dprintln(dbg, "Finished Stage 1");

      Thread.currentThread().sleep(5000);
      //
      // stage 2:
      //
      imb = read(node2_interface, objId_c);
      Env.dprintln(dbg, "Stage 2:");
      Env.dprintln(dbg, "...Exptected Value: " + value_2_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_2_imb)): "Read does not match expected value";
      

      Thread.currentThread().sleep(BayouTest.waitAntiEntropyEnd);

      //
      // stage 3:
      //
      
      write(node2_interface,objId_d, value_3_crr);
      write(node2_interface, objId_q, value_3_crr);      
 
      Env.dprintln(dbg, "Finished Stage 3");

      //
      // stage 4:
      //

       Env.dprintln(dbg, "Stage 4:");
      //Note: we need to read p before a so that we know that
      // the 2nd round of anti-entropy has occured
       imb = read(node2_interface, objId_p);
      Env.dprintln(dbg, "...Exptected Value: " + value_3_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_3_imb)): "Read does not match expected value";
      imb = read(node2_interface, objId_c);
      Env.dprintln(dbg, "...Exptected Value: " + value_3_imb);
      Env.dprintln(dbg, "...ReadValue: " + imb);
      assert(imb.equals(value_3_imb)): "Read does not match expected value";
       
      Env.dprintln(dbg, "Finished Stage 4");
      
      System.out.println("Test passed");
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

  public static void main(String[] args) {
    try {
      Env.verifyAssertEnabled();
      
      if(args.length < 1) {
	System.out.println("Usage: java BayouTest_Helper <overlog file>");
	System.exit(0);
      }
     
      String fileName = args[0];
      
      c =  new BarrierClient("localhost", 
			     BayouTest.BARRIER_PORT,
			     (int)BayouTest.NODE_2_ID);
      
      BayouTest_Helper testHelper = new BayouTest_Helper(fileName);
      c.sendBarrierRequest(0, BayouTest.NODE_2_ID); // Inform helper that stuff has be set up
      
      testHelper.startNodes();
      c.sendBarrierRequest(0, BayouTest.NODE_2_ID); // Inform helper that overlog node has been started

      testHelper.simpleTest();
      Thread.currentThread().sleep(1000);
      c.sendBarrierRequest(0, BayouTest.NODE_2_ID); // finished test 1
      testHelper.simpleTest2();
      c.sendBarrierRequest(0, BayouTest.NODE_2_ID); // Inform helper that test has completed shutting down

      testHelper.shutdownNodes();
      System.exit(0);
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }
  
}
 /** 
/* $Log: BayouTest_Helper.java,v $
/* Revision 1.3  2007/09/12 19:07:16  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.2  2007/01/12 22:30:08  nalini
/* Fixed informRemovedSubscription bug
/*
/* Revision 1.1  2007/01/12 19:09:32  nalini
/* Bayou case-study created
/*
 */
 **/ 
