 /** 
/*  Bayou_Exp2_Node1.java
 * 
 *  Bayou case study experiment 2
 *
 *  detects if node 1 is alive
 *  if so establishes anti-entropy and measures the time it takes
 *
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class Bayou_Exp2_Node2 {

  private BayouNode node2;
  private BayouFSLocalInterface node2_interface;
  
  private static String dbPathSuffix = "PopulatedDb";
  private static String node1Hostname = "localhost";
  private static String node2Hostname = "localhost";
  private String outFile;
  
  private String entropyStart;
  private String entropyEnd;
  private boolean node1Started = false;
  private boolean finishedMeasuring = false;

  private boolean dbg = true;


 /** 
 *  Constructor 
 **/ 

  public Bayou_Exp2_Node2(String dbPath,
			  String node1Hostname,
			  String node2Hostname,
			  String outFile){
    this.dbPathSuffix=dbPath;
    this.outFile = outFile;
    this.node1Hostname = node1Hostname;
    this.node2Hostname = node2Hostname;

    
    Bayou_Exp2_Node1.makePractiConfig(node1Hostname, node2Hostname, Bayou_Exp2_Node1.CONFIG_PATH);
    Bayou_Exp2_Node1.makeP2Config(Bayou_Exp2_Node1.CONFIG_P2_PATH);
    Bayou_Exp2_Node1.makeNodeIdMap(node1Hostname, node2Hostname,Bayou_Exp2_Node1.NODEID_MAP_PATH);

    node2 = new BayouNode(Bayou_Exp2_Node1.CONFIG_PATH, 
			  Bayou_Exp2_Node1.CONFIG_P2_PATH,
			  new NodeId(Bayou_Exp2_Node1.NODE_2_ID),
			  true, //clean DB
			  Bayou_Exp2_Node1.OVERLOG_PATH,
			  Bayou_Exp2_Node1.NODEID_MAP_PATH);

    node2_interface = node2.getFSInterface();


    TupleHandler th = new AliveTupleHandler(this);
    node2.registerHandler(th);
    th = new EntropyStartTupleHandler(this);
    node2.registerHandler(th);
    th = new EntropyEndTupleHandler(this);
    node2.registerHandler(th);
  }

 /** 
 *   Starts the Overlog  
 **/ 
   
  public synchronized void startNode() throws Exception{
    node2.start();
    wait(2000);
  }

 /** 
 *  shutdown 
 **/ 
   
  public synchronized void shutdown() throws Exception{
    node2.shutdown();
  }
  

 /** 
 *   print results 
 **/ 
 
 public synchronized void printResults() {
    try{
      PrintStream out = new PrintStream(new FileOutputStream(outFile, true));
      long entropyDuration = parseTime(entropyEnd) - parseTime(entropyStart);
      out.println(entropyDuration);
      out.close();
    }catch(Exception e) {
      e.printStackTrace();
    }
  }

 /** 
 *  Waits until node1 has started 
 **/ 
   
  public synchronized void waitForNode1Start(){
 
    while(!node1Started){
      String[] stringA = {"areYouAlive", 
                          NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_2_ID)).toString(),
                          NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_1_ID)).toString()};
      Tuple t = new Tuple(stringA);
      node2.insertTuple(t);
      try{
        wait(2000);
      }catch(InterruptedException e) {
        // do nothing and continue
      }
    }
  }

 /** 
 *  notified by tuplehandler when node1 has started 
 **/ 
 
  public synchronized void node1Started(){
    node1Started = true;
    notifyAll();
  }


 /** 
 *  entropyStart 
 **/ 
  public synchronized void entropyStart(String time){
    Env.dprintln(dbg, "EntropySTart = " + parseTime(time));
    entropyStart = time;
  }

 /** 
 *  entropyEnd 
 **/ 
   public synchronized void entropyEnd(String time){
    entropyEnd = time;
    finishedMeasuring = true;
    System.out.println("entropyEnd called");
    Env.dprintln(dbg, "EntropyEnd = " + parseTime(time));
    notifyAll();
  }


 /** 
 *  Parse time -- converts from P2 format to ms 
 *  Example Input String: [2007-Feb-20 20:03:57.445064000]  
 **/ 
  public synchronized long parseTime(String time) {
    long ret = 0;
    
    // skip over date
    int i = time.indexOf(' ');
    String tmpString = time.substring(i+1);
    
    // handle hour
    i = tmpString.indexOf(':');
    long value = (new Long(tmpString.substring(0,i))).longValue();
    ret+= value;
    tmpString = tmpString.substring(i+1);

    //handle minute
    i = tmpString.indexOf(':');
    value = (new Long(tmpString.substring(0,i))).longValue();
    ret = ret*60 + value;
    tmpString = tmpString.substring(i+1);

    //handle second
    i = tmpString.indexOf('.');
    value = (new Long(tmpString.substring(0,i))).longValue();
    ret = ret*60 + value;
    tmpString = tmpString.substring(i+1);

    //handle milliseconds
     value = (new Long(tmpString.substring(0,3))).longValue();
     ret = ret*1000 + value;

     return ret;
  }

    
    
 /** 
 *  startTest 
 **/ 
  public synchronized void startTest(int type) {

    String[] stringA;
    Tuple t;
    if(type==1){ // subscribe for all
    //insert a ss tuple
      stringA = new String[] {"insertedSS", 
                        NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_2_ID)).toString(),
                        "-*"};
    } else { // subscribe for 10% of the data
      stringA = new String[] {"insertedSS", 
			      NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_2_ID)).toString(),
			      "-100-1-*"};
    }

    t = new Tuple(stringA);
    node2.insertTuple(t);

    try{
      wait(50); // wait till the tuple registers
    }catch(Exception e) {
    }
    
    // wait for node1 to start
    System.out.println("Waiting for Node1 to start");
    waitForNode1Start();
    
    

   //insert a start Entropy tuple
    stringA = new String[]{"startEntropy", 
			   NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_2_ID)).toString(),
			   NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_1_ID)).toString()};
    t = new Tuple(stringA);
    node2.insertTuple(t);
    
    // do nothing until we receive "entropyEnd"
    while(!finishedMeasuring){
      try{
        wait();
      }catch(InterruptedException e) {
        // do nothing
      }
    }

    System.out.println("Received Entropy End.. printing results");
    //print results
    printResults();    

    //insert a finished measuring tuple
    stringA = new String[]{"finishedMeasuring", 
	       NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_2_ID)).toString(),
	       NodeIdMap.getOverlogId(new NodeId(Bayou_Exp2_Node1.NODE_1_ID)).toString()};
    t = new Tuple(stringA);
    node2.insertTuple(t);
    try{
      wait(100); // wait for the tuple to go across the line
    } catch(InterruptedException e) {
      // do nothing
    }
  }



 /** 
 *  main Method 
 **/ 
  public static void main(String[] args) {
    if(args.length < 5) {
      System.out.println("Usage: Bayou_Exp2_Node2 <type=1|2> <dbPathSuffix> <node1name> <node2name> <outFile>");
      return;
    }
    
    if(!args[0].equalsIgnoreCase("1") && !args[0].equalsIgnoreCase("2")){
      System.out.println("Type should be either 1 or 2");
      return;
    }

    int type;
    
    if(args[0].equalsIgnoreCase("1")){
      type=1;
    }else {
      type=2;
    }

    Bayou_Exp2_Node2 exp = new Bayou_Exp2_Node2(args[1],
						args[2],
						args[3],
						args[4]); 

    try{
      exp.startNode();
      exp.startTest(type);
      exp.shutdown();
    }catch(Exception e) {
      e.printStackTrace();
    }

    System.out.println("Test Ended");
    System.exit(0);
  }
  
 /** 
 *  AliveTupleHandler -- watches yesAlive tuple 
 **/ 

  class AliveTupleHandler extends TupleHandler {
    private Bayou_Exp2_Node2 test;

    public AliveTupleHandler(Bayou_Exp2_Node2 test) {
      super("yesAlive");
      this.test = test;
    }

    public void handleTuple(Tuple tp) {
      test.node1Started();
    }
  }  
 /** 
 *   Entropy Start Tuple Handler 
 **/ 
  class EntropyStartTupleHandler extends TupleHandler {
    private Bayou_Exp2_Node2 test;
    
    public EntropyStartTupleHandler(Bayou_Exp2_Node2 test) {
      super("entropyStart");
      this.test = test;
    }

    public void handleTuple(Tuple tp){
      test.entropyStart(tp.getItem(2));
    }
  }
 
 /** 
 *  Entropy End Tuple Handler 
 **/ 

  class EntropyEndTupleHandler extends TupleHandler {
    private Bayou_Exp2_Node2 test;
    
    public EntropyEndTupleHandler(Bayou_Exp2_Node2 test) {
      super("entropyEnd");
      this.test = test;
    }

    public void handleTuple(Tuple tp){
      test.entropyEnd(tp.getItem(2));
    }
  }
 
}
