 /** 
/*  Bayou_Exp2_PopulateDB.java
 * 
 *  Bayou case study experiment 2
 *
 *  Populates DB and log with bound writes as follows
 *
 * 
 * Option = 1 -- takes up too much heap space
 * Obj size   number	Total	%	path (each dir has 5% of the data)
 * 10b 	   150,000	1.5M	15%	/10/1, /10/2/, /10/3
 * 100b	   20,000	2M	20%	/100/1, /100/2, /100/3, /100/4
 * 1K	   3,000	3M	30%	/1000/1,/1000/2, /1000/3, /1000/4,/1000/5, /1000/6
 * 10K	   200		2M	20%	/10000/1, /1000/2, /10000/3, /10000/4		 
 * 100K	   15		1.5M	15%	/100000/1, /10000/2, /100000/3
 *
 *
 * Option = 2 
 * populate a DB as follows:
 * Obj size   number	Total	%	path (each dir has 10% of the data)
 * 10b 	   10,000	100k	10%	/10/1
 * 100b	   2,000	200K	20%	/100/1, /100/2
 * 1K	   400		400K	40%	/1000/1,/1000/2, /1000/3, /1000/4
 * 10K	   20		200K	20%	/10000/1, /1000/2
 * 100K	   1		100K	10%	/100000/1
 * 
 *
 * Option = 3 
 * populate a DB as follows:
 * Obj size   number	Total	%	path (each dir has 10% of the data)
 * 10b	      3000	 30K	30%	/10/1, /10/2, 10/3
 * 100b	      400	 40K	40%	/100/1,/100/2, /100/3, /100/4
 * 1K	      30	 30K	30%	/1000/1, /1000/2, /1000/3
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class Bayou_Exp2_PopulateDB {

  private BayouNode node1;
  private BayouFSLocalInterface node1_interface;
  
  public static long NODE_1_ID = 0;
  public static String CONFIG_PATH = "Bayou_Exp2_PopulateDB.config";
  public static String CONFIG_P2_PATH = "Bayou_Exp2_PopulateDB.p2config";
  public static String NODEID_MAP_PATH = "Bayou_Exp2_PopulateDB.nodemap";
  public static String OVERLOG_PATH = "Garbage.olg";

  private static String dbPathSuffix = "PopulatedDb";

 /** 
 *  Constructor 
 **/ 


  public Bayou_Exp2_PopulateDB (String dbPath) {
    this.dbPathSuffix = dbPath;
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(NODEID_MAP_PATH);
    
    node1 = new BayouNode(CONFIG_PATH, 
			  CONFIG_P2_PATH,
			  new NodeId(NODE_1_ID),
			  true,
			  OVERLOG_PATH,
			  NODEID_MAP_PATH);

    node1_interface = node1.getFSInterface();
  }

 /** 
 *  Make configuration 
 **/ 
  public static synchronized void makePractiConfig(String configPath) {
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            dbPathSuffix+".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    Config.writeToFile(configPath);
  }

 public static synchronized void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_1_ID), 5, 5, 30000000, 30000000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  public static synchronized  void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_1_ID), new OverlogId("localhost:5000"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
//  Starts the Overlog 
//---------------------------------------------------------------------------
   
  public synchronized void startNode() throws Exception{
    //start node without overlog
    node1.start(true);
    wait(2000);
  }

//---------------------------------------------------------------------------
// shutdown
//---------------------------------------------------------------------------
   
  public synchronized void shutdown() throws Exception{
    node1.shutdown();
  }
  

//---------------------------------------------------------------------------
//  Converts values to byte arrays
//---------------------------------------------------------------------------

  private synchronized byte[] convertToArray(int size, byte value){
    byte[] b = new byte[size];
    for(int i = 0; i < size; i++) {
      b[i] = value;
    }
    return b;
  }


//---------------------------------------------------------------------------
//  Writes <numObjects> of <size> to dir <path>
//---------------------------------------------------------------------------

  private synchronized void writeDir(String path, int numObjects, int size)
    throws IOException{
    byte[] b = convertToArray(size, (byte) 65);
    
    for(int i=0; i < numObjects; i++){
      ObjId objId = new ObjId(path + "/" + i);      
      node1_interface.write(objId, 0, size, b);
      try{
	wait(5);
      }catch(InterruptedException e) {
	// do nothing
      }
    }
  }



//---------------------------------------------------------------------------
// populate db -- 1
//---------------------------------------------------------------------------
  private synchronized void populateDB1() throws IOException {

    //populate 10byte directories
    writeDir("/10/1", 50000, 10);
    writeDir("/10/2", 50000, 10);
    writeDir("/10/3", 50000, 10);

    //populate 100byte directories
    writeDir("/100/1", 5000, 100);
    writeDir("/100/2", 5000, 100);
    writeDir("/100/3", 5000, 100);
    writeDir("/100/4", 5000, 100);

    //populate 1K directories
    writeDir("/1000/1", 500, 1000);
    writeDir("/1000/2", 500, 1000);
    writeDir("/1000/3", 500, 1000);
    writeDir("/1000/4", 500, 1000);
    writeDir("/1000/5", 500, 1000);
    writeDir("/1000/6", 500, 1000);

    //populate 10K directories
    writeDir("/10000/1", 50, 10000);
    writeDir("/10000/2", 50, 10000);
    writeDir("/10000/3", 50, 10000);
    writeDir("/10000/4", 50, 10000);

    //populate 100K directories
    writeDir("/100000/1", 5, 100000);
    writeDir("/100000/2", 5, 100000);
    writeDir("/100000/3", 5, 100000);
  }


//---------------------------------------------------------------------------
// populate db -- 2
//---------------------------------------------------------------------------
  private synchronized void populateDB2() throws IOException {

      System.out.println("Populating 1b dir");
    //populate 1byte directories
    writeDir("/1/1", 10000, 1);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }
  

    System.out.println("Populating 10b dir");
    //populate 10byte directories
    writeDir("/10/1", 1000, 10);
    writeDir("/10/2", 1000, 10);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }

    
    System.out.println("Populating 100b dir");
    //populate 100b directories
    writeDir("/100/1", 100, 100);
    writeDir("/100/2", 100, 100);

    writeDir("/100/3", 100, 100);
    writeDir("/100/4", 100, 100);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }    

    System.out.println("Populating 1K dir");
    //populate 1K directories
    writeDir("/1000/1", 10, 1000);
    writeDir("/1000/2", 10, 1000);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }

    System.out.println("Populating 10K dir");
    //populate 100K directories
    writeDir("/10000/1", 1, 10000);

  }

//---------------------------------------------------------------------------
// populate db -- 3
//---------------------------------------------------------------------------
  private synchronized void populateDB3() throws IOException {

   

    System.out.println("Populating 10b dir");
    //populate 10byte directories
    writeDir("/10/1", 1000, 10);
    writeDir("/10/2", 1000, 10);
    writeDir("/10/3", 1000, 10);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }

    
    System.out.println("Populating 100b dir");
    //populate 100b directories
    writeDir("/100/1", 100, 100);
    writeDir("/100/2", 100, 100);
    writeDir("/100/3", 100, 100);
    writeDir("/100/4", 100, 100);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }    

    System.out.println("Populating 1K dir");
    //populate 1K directories
    writeDir("/1000/1", 10, 1000);
    writeDir("/1000/2", 10, 1000);
    writeDir("/1000/3", 10, 1000);
    try{
      wait(50);
    }catch(InterruptedException e) {
    }

  }

//---------------------------------------------------------------------------
// main method
//---------------------------------------------------------------------------

  public static void main(String[] args) {

    if (args.length < 2) {
      System.out.println("Usage: Bayou_Exp2_PopulateDB <dbPathSuffix> <type=1|2|3>");
      return;
    }
    if(!args[1].equalsIgnoreCase("1") && !args[1].equalsIgnoreCase("2") && !args[1].equalsIgnoreCase("3")){
      System.out.println("Type should be either 1 or 2 or 3");
      return;
    }

    Bayou_Exp2_PopulateDB exp = new Bayou_Exp2_PopulateDB(args[0]);

    try{ 
      exp.startNode();
      if(args[1].equalsIgnoreCase("1")){
        exp.populateDB1();
      }else if (args[1].equalsIgnoreCase("2")) {
        exp.populateDB2();
      } else {
	exp.populateDB3();
      }
      exp.shutdown();
    }catch(Exception e) {
      e.printStackTrace();
    }
    System.out.println("DB Population Ended"); 
    System.exit(0);
  }
}

//---------------------------------------------------------------------------
/* $Log: Bayou_Exp2_PopulateDB.java,v $
/* Revision 1.5  2007/03/09 03:01:39  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.4  2007/02/28 20:51:20  nalini
/* *** empty log message ***
/*
/* Revision 1.3  2007/02/28 04:44:37  nalini
/* *** empty log message ***
/*
/* Revision 1.2  2007/02/28 03:45:33  nalini
/* minor updates
/*
/* Revision 1.1  2007/02/28 03:24:45  nalini
/* bayou exp2 added
/*
*/
 /** 
