 /** 
/*  Bayou_Exp3.java
 * 
 *  measures bandwidth for 500 writes
 *
 *  opens the pre-populated DB
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class Bayou_Exp3 {

  private BayouNode node;
  private BayouFSLocalInterface node_interface;
  private long myNodeId;

  public static long NODE_0_ID = 0;
  public static long NODE_1_ID = 1;
  public static String CONFIG_PATH = "Bayou_Exp3.config";
  public static String CONFIG_P2_PATH = "Bayou_Exp3.p2config";
  public static String NODEID_MAP_PATH = "Bayou_Exp3.nodemap";
  public static String OVERLOG_PATH = "Bayou_Exp3.olg";
  private static String dbPathPrefix = "Bayou_Exp3_";

  private String node1Hostname = "localhost";
  private String node2Hostname = "localhost";
  private String outFile="out.txt";
  private PrintStream out;

  protected static int sizeOfWrites = 3000; 
  protected static int numSets = 10;
  protected static int numWritesInSet = 50; // assume divisible by 10 
  protected static byte value = 65; //"A";
  protected static String fullSS;
  protected static String smallSS;
  protected static boolean full = true;


  private TupleWait nodeStartWait = new TupleWait("started");
  private TupleWait sendSetWait = new TupleWait("sendSet");
  private TupleWait expFinishedWait = new TupleWait("expFinished");
  private TupleWait yesAliveWait = new TupleWait("yesAlive");
  private TupleWait subscriptionWait = new TupleWait("subscriptionAdded");

  private boolean dbg = true;

 /** 
 *  Constructor 
 **/ 


  public Bayou_Exp3(long myId, String node1Hostname, String node2Hostname, String outFile) {
    if(myId == 0) {
      this.myNodeId = NODE_0_ID;
    }
    else {
      this.myNodeId = NODE_1_ID;
    }
    this.node1Hostname = node1Hostname;
    this.node2Hostname = node2Hostname;
    this.outFile = outFile;

    this.fullSS = "/*";
    
    String ss="";
    for(int i = 0; i < numSets; i++){
      ss+="/" + i + "/9/*";
      if(i != numSets-1) {
	ss+= ":";
      }
    }

    this.smallSS=ss;

    makePractiConfig(node1Hostname, node2Hostname, CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(node1Hostname, node2Hostname, NODEID_MAP_PATH);
    
    node = new BayouNode(CONFIG_PATH, 
                         CONFIG_P2_PATH,
                         new NodeId(myNodeId),
                         true,
                         OVERLOG_PATH,
                         NODEID_MAP_PATH,
                         false);

    node_interface = node.getFSInterface();

    node.registerHandler(nodeStartWait);
    node.registerHandler(sendSetWait);
    node.registerHandler(expFinishedWait);
    node.registerHandler(yesAliveWait);
    node.registerHandler(subscriptionWait);
  }
 

//---------------------------------------------------------------------------
// Reads an object from the specified interface
// If objNotFound -- sleeps for some time and tries again
//---------------------------------------------------------------------------

  public synchronized ImmutableBytes read(ObjId objId, int size) throws Exception {
    ImmutableBytes b=null;
    try{
      while (true) {
	try{
	  b = node_interface.read(objId, 0, size).getBody();
	  break;
	}catch(ObjNotFoundException e){
	  //wait until you recieve obj
	  //Env.dprintln(dbg, "ObjNotFoundException for " + objId);
	  Thread.currentThread().sleep(100);
	}
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    return b;
  }


//---------------------------------------------------------------------------
// makeWrites
//  - simply makes a bunch of writes
//---------------------------------------------------------------------------
  public void makeWrites(int set) {
    try{ 
      byte[] b = convertToArray(value);
      for(int i = 0; i < 10; i++){
	for(int j = 0; j < numWritesInSet/10; j++){
	  ObjId obj = new ObjId("/" + set + "/" + i + "/" + j);
	  node_interface.write(obj, 0, b.length, b);
	}
      }        
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }


//---------------------------------------------------------------------------
// readSett
//  - simply  reads the last obj in the set
//---------------------------------------------------------------------------
  public void readSet(int set) {
    try{ 
      ObjId obj = new ObjId("/" + set + "/9/" + (numWritesInSet/10-1));
      read(obj, sizeOfWrites);
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }
//---------------------------------------------------------------------------
//  Converts values to byte arrays
//---------------------------------------------------------------------------

  private byte[] convertToArray(byte value){
    byte[] b = new byte[sizeOfWrites];
    for(int i = 0; i < sizeOfWrites; i++) {
      b[i] = value;
    }
    return b;
  }


//---------------------------------------------------------------------------
// start test
//---------------------------------------------------------------------------
  public synchronized void startTest(){
    if(myNodeId == NODE_0_ID) {
      node_0_Test();
    }else {
      node_1_Test();
    }
  }

//---------------------------------------------------------------------------
// sender side test
//---------------------------------------------------------------------------
  public synchronized void node_0_Test(){
    Env.dprintln(dbg, "Node0: waiting for node start");
    nodeStartWait.waitForTuple();
    Stats.reset();
    for(int i = 0; i < numSets; i++){
      Env.dprintln(dbg, "Node0: waiting for sendSet for " + i);
      sendSetWait.waitForTuple();
      printStats(i); // we print Stats before each set -- we know
                     // the previous set has gone thro'
      Env.dprintln(dbg, "Node0: making writes");
     makeWrites(i);
    }

    Env.dprintln(dbg, "Node0: waiting for finished exp");
    expFinishedWait.waitForTuple();
    printStats(numSets); // we print Stats at the end.
    Env.dprintln(dbg, "Node0: finished exp");
  }



//---------------------------------------------------------------------------
// receiver side test
//---------------------------------------------------------------------------
  public synchronized void node_1_Test(){
    Env.dprintln(dbg, "Node1: waiting for node start");
    nodeStartWait.waitForTuple();
    Env.dprintln(dbg, "Node1: initializing SS");
    initializeSS();
    Env.dprintln(dbg, "Node1: checking if neighbor alive");
    checkIfAlive(NODE_0_ID);
    Env.dprintln(dbg, "Node1: waiting for inval subscription to be established");
    subscriptionWait.waitForTuple();
    for(int i = 0; i < numSets; i++) {
      Env.dprintln(dbg, "Node1: asking for set" + i);
      askForSet(NODE_0_ID);
      Env.dprintln(dbg, "Node1:reading set");
      readSet(i);
    }
    Env.dprintln(dbg, "Node1: notifying exp end");
    notifyExpFinished(NODE_0_ID);
    try{
      wait(2000);
    }catch(Exception e){
      //igonore
    }
  }

//---------------------------------------------------------------------------
// print stats
//---------------------------------------------------------------------------
  public synchronized void printStats(int set) {

      long bytes = Stats.createRecord().getTotal();
      out.println(set + "\t" + bytes);

      System.out.println("Set " + set + ": " + bytes + " bytes");
  }
      

//---------------------------------------------------------------------------
//  initialize SS for this experiment
//---------------------------------------------------------------------------
 
  public synchronized void initializeSS(){
    String[] stringA; 
    if(full) {
      stringA = new String[] {"insertSS", 
                              NodeIdMap.getOverlogId(new NodeId(myNodeId)).toString(),
                              fullSS};
    } else {
      stringA = new String[] {"insertSS", 
                              NodeIdMap.getOverlogId(new NodeId(myNodeId)).toString(),
                              smallSS};
    }
    Tuple t = new Tuple(stringA);
    node.insertTuple(t);
  }
 
//---------------------------------------------------------------------------
//  check if neighbor is alive
//---------------------------------------------------------------------------
 
  public synchronized void checkIfAlive(long neighbor) {
    boolean alive = false;
    while(!alive){
      String[] stringA = {"areYouAlive", 
                          NodeIdMap.getOverlogId(new NodeId(myNodeId)).toString(),
                          NodeIdMap.getOverlogId(new NodeId(neighbor)).toString()};
      Tuple t = new Tuple(stringA);
      node.insertTuple(t);
      alive = yesAliveWait.waitForTuple(2000);
    }
  }


//---------------------------------------------------------------------------
//  ask neighbor to send a set
//---------------------------------------------------------------------------
 
  public synchronized void askForSet(long neighbor) {

    String[] stringA = {"askForSet", 
                        NodeIdMap.getOverlogId(new NodeId(myNodeId)).toString(),
                        NodeIdMap.getOverlogId(new NodeId(neighbor)).toString()};
    Tuple t = new Tuple(stringA);
    node.insertTuple(t);
  }
 
//---------------------------------------------------------------------------
//  tell the neighbor that the exp finished
//---------------------------------------------------------------------------
 
  public synchronized void notifyExpFinished(long neighbor) {

    String[] stringA = {"notifyFinishedExp", 
                        NodeIdMap.getOverlogId(new NodeId(myNodeId)).toString(),
                        NodeIdMap.getOverlogId(new NodeId(neighbor)).toString()};
    Tuple t = new Tuple(stringA);
    node.insertTuple(t);
  }



//---------------------------------------------------------------------------
// Make configuration
//---------------------------------------------------------------------------
  public static void makePractiConfig(String node1host, String node2host,String configPath) {
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
                            node1host,
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            dbPathPrefix+".db",
                            "/*",
                            -1L,
                            node2host,
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
                            node2host,
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                            dbPathPrefix+ ".db",
			   "/*",
			   -1L,
			   node2host,
			   9893,
			   9894,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
    Config.writeToFile(configPath);
  }

 public static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_0_ID), 5, 5, 30000000, 30000000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_1_ID), 5, 5, 30000000, 30000000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

public static void makeNodeIdMap(String node1host, String node2host, String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_0_ID), new OverlogId(node1host+":5000"));
    NodeIdMap.add(new NodeId(NODE_1_ID), new OverlogId(node2host+":5001"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
//  Starts the Overlog 
//---------------------------------------------------------------------------
   
  public synchronized void startNode() throws Exception{
    node.start();
    //wait(2000);
    out = new PrintStream(new FileOutputStream(outFile)); 
  }

//---------------------------------------------------------------------------
// shutdown
//---------------------------------------------------------------------------
   
  public synchronized void shutdown() throws Exception{
    node.shutdown();
    out.close();
  }
  

//---------------------------------------------------------------------------
// main method
//---------------------------------------------------------------------------

  public static void main(String[] args) {

    if (args.length < 3) {
      System.out.println("Usage: Bayou_Exp3_Node1 <nodeid=[0|1]> <host1name> <host2name> <outFile>");
      return;
    }

    Bayou_Exp3 exp = new Bayou_Exp3((new Long(args[0])).longValue(), 
                                     args[1], 
                                     args[2],
                                     args[3]);

    try{ 
      exp.startNode();
      exp.startTest();
      exp.shutdown();
    }catch(Exception e) {
      e.printStackTrace();
    }
    System.out.println("Test Ended");
    System.exit(0);
  }

}

//---------------------------------------------------------------------------
/* $Log: Bayou_Exp3.java,v $
/* Revision 1.2  2007/09/12 19:07:16  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.1  2007/03/14 02:29:11  nalini
/* Bayou exp3 added
/*
*/
 /** 
