 /** 
/* ChainReplicationClient.java
 *  
 *  Implements an interface for a NO PRACTI & NO Overlog client to use
 *
 *  Implements read and write interface
 *  Assumes a fixed head and a fixed tail
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.IOException;
import java.io.EOFException;
import java.rmi.*;
import java.util.*;

public class ChainReplicationClient {

  private Hashtable headTable;
  private Hashtable tailTable;

  private boolean disableVol = true;

  
  private int MAX_TRIES = 10;
  private int RETRY_WAIT = 5000; // 5 seconds
  private boolean DEBUG = false;
 
  
 /** 
 *  Constructor 
 **/ 
  public 
  ChainReplicationClient(String configFile){
    this.headTable = new Hashtable();
    this.tailTable = new Hashtable();
    Config.readConfig(configFile);
  }


 /** 
 *  add head and tail 
 **/ 
  public 
  void putVolHeadTail(String volId, NodeId head, NodeId tail){
    headTable.put( volId, head);
    tailTable.put( volId, tail);
  }

 /** 
 *  Get Vol Id 
 **/ 

 public String getVolId(ObjId objId) {

   if (disableVol) return "0";
   
   String objStr = objId.toString();
  
   // if it is not thro NFS, this is simple
   int i = objStr.indexOf('/', 1); // we ignore the starting '/'
   String volId = objStr.substring(1, i);
   return volId;
 }

 /** 
 *  Read from an object 
 **/ 
  public synchronized BodyMsg
  read(ObjId objId,
       long offset,
       long length)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    BodyMsg result = null;

    Env.dprintln(DEBUG, "ChainRepClient: read for " + objId);
    
    int currentTry = 0;
    while(true){
      try{
        String volId  = getVolId(objId);
        Env.dprintln(DEBUG, "ChainRepClient: volId for " + objId + " = " + volId);
        NodeId tailId = getTail(volId);
        Env.dprintln(DEBUG, "ChainRepClient: tail for " + volId + " = " + tailId);
        String rmiURL = "rmi://" + Config.getDNS(tailId) + "/" + tailId.toString() + "-ChainReplicationFS"; 
        ChainReplicationFSRMIServer rmiServer = (ChainReplicationFSRMIServer)Naming.lookup(rmiURL);
        if(rmiServer != null) {
          result = rmiServer.read(objId, offset, length);
        }
        return(result);

      }catch(java.rmi.NotBoundException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepClient: Exception in read for " + objId + " retrying");
	if(DEBUG) { 
	  z.printStackTrace();
	}
        currentTry++; 
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepClient:reached max retries for read for " + objId);
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}

      }catch(java.rmi.RemoteException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepClient: Exception in read for " + objId + " retrying");
	if(DEBUG) { 
	  z.printStackTrace();
	}
        currentTry++;
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepClient:reached max retries for read for " + objId);
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}
      }
    }
  }   

   
 
  

 /** 
 *  Write method 
 **/ 
  public synchronized long
  write(ObjId objId, long offset, long length, byte buffer[]) 
    throws IOException {
    Env.dprintln(DEBUG, "ChainRepClient: write for " + objId);
    long numBytes = 0;
    int currentTry = 0;

    while(true) {
      try{
        String volId = getVolId(objId);
	Env.dprintln(DEBUG, "ChainRepClient: vol for " + objId + "= " + volId);
        NodeId headId = getHead(volId);
	Env.dprintln(DEBUG, "ChainRepClient: head for " + volId + " = " + headId);
        String rmiURL = "rmi://" + Config.getDNS(headId) + "/" + headId.toString() + "-ChainReplicationFS"; 
        ChainReplicationFSRMIServer rmiServer = (ChainReplicationFSRMIServer)Naming.lookup(rmiURL);
        if(rmiServer != null) {
          numBytes = rmiServer.write(objId, offset, length, buffer);
	  Env.dprintln(DEBUG, "ChainRepClient: finished writing from server");
        }
	  return numBytes;
      }catch(java.rmi.NotBoundException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepClient: Exception in write for " + objId + " retrying");
	if(DEBUG) { 
	  z.printStackTrace();
	}
        currentTry++; 
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepClient: reached max retires for write for  " + objId ); 
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}

      }catch(java.rmi.RemoteException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepClient: Exception in write for " + objId + " retrying");
	if(DEBUG) { 
	  z.printStackTrace();
	}
        currentTry++;
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepClient: reached max retries for write for  " + objId ); 
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}
      }
    }
  }   

 /** 
 *  getHead  
 *   -- from the hash table 
 **/ 

  protected NodeId getHead(String volId) {
    return (NodeId) headTable.get(volId);
  }

 /** 
 *  getTail from table 
 **/ 

  protected  NodeId getTail(String volId){
      return (NodeId) tailTable.get(volId);
  }

} 

 /** 
/* $Log: ChainReplicationClient.java,v $
/* Revision 1.3  2007/03/21 22:20:32  nalini
/* CR_Exp2 checked in
/*
/* Revision 1.2  2007/03/17 01:24:22  nalini
/* CR exp2 added
/*
/* Revision 1.1  2007/03/15 02:31:33  nalini
/* added ChainReplicationClient
/*
 */
 **/ 
