 /** 
/* ChainReplicationFS
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 
import java.io.*;

public class ChainReplicationFS extends PRACTIFS {

  private boolean DEBUG = false;

  public ChainReplicationFS(ChainReplicationFSLocalInterface newLocalInterface, boolean nfs) {

    //
    //  Note: makeRoot method in the super class uses the ChainReplicationFSLocalInterface
    //   read and write method, which will be forwarded to the head/tail.  But if
    //   the head and tail haven't initialized yet, it will throw an exception
    //   we re-write the makeroot method so that it is locally handled on 
    //   initialization.
    //
    super(newLocalInterface, false);    
    if(nfs) {
      makeRootLocally();
    }
  }



 /** 
 *  makeRootLocally 
 *   makes the root directory locally w/o forwarding them to head/tail 
 **/ 

  private void
  makeRootLocally(){
    Env.dprintln(DEBUG, "ChainRepFS: Going to make root locally");
    ObjId dirIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    PRACTIFSDirectory rootDir = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    FSWriteEntry[] fswArr = null;
    byte[] dirMetaBytes = null;
    long[] writeLengths = null;
    PRACTIFSFileAttributes dirAttr = null;
    ImmutableBytes dirIB = null;
    long numBlocks = 0;

    try{
      dirIdPrefix = super.fhCache.getObjIdPrefix(fhandle.makeAllZeroHandle());
      assert(dirIdPrefix.equals(new ObjId(PRACTIFS.ROOT_DIR_NAME)));
      dirMetaObjId = new ObjId(dirIdPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirIdPrefix.getPath() + DATA_SUFFIX);
      
      dirMetaToken = super.localInterface.acquireWriteLock(dirMetaObjId);
      dirDataToken = super.localInterface.acquireWriteLock(dirDataObjId);
      
      try{
        Env.dprintln(DEBUG, "readDir in makeRootDirectory, expect FileNotFoundException");
        rootDir = this.readLocalDir(dirDataToken, dirDataObjId);
        Env.dprintln(DEBUG, "root already exist");
        
      }catch(FileNotFoundException e){
        Env.dprintln(DEBUG, "FileNotFoundException expected");
        // Create the directory
        rootDir = new PRACTIFSDirectory();
        fswArr = new FSWriteEntry[2];

        // Prepare to write the root directory data
        dirIB = new ImmutableBytes(rootDir.dangerousGetBytes());
        fswArr[0] = new FSWriteEntry(dirDataObjId,
                                       0,
                                       rootDir.getSize(),                                       
                                       dirIB,
                                       false);

        // Write the new inner directory attributes
        numBlocks = (dirIB.getLength() + FS_BLOCK_SIZE - 1) / FS_BLOCK_SIZE;
        dirAttr = createFileAttributes(null,//newAttr 
                                       ftype.NFDIR,
                                       dirIB.getLength(), // size
                                       numBlocks, // blocks
                                       (long)dirDataObjId.hashCode(), // fileid
                                       null);//I'm the root
        dirMetaBytes = super.getAttributeBytes(dirAttr);
        fswArr[1] = new FSWriteEntry(dirMetaObjId,
                                       0,
                                       dirMetaBytes.length,
                                       new ImmutableBytes(dirMetaBytes),
                                       false);
        
        // Perform the writes
        writeLengths = ((ChainReplicationFSLocalInterface)super.localInterface).writeLocal(fswArr);
        /*
          System.out.println("PRACTIFS: Wrote: " +
          fswArr[0].getObjInvalTarget().getObjId() +
          ":" + writeLengths[0]);
          System.out.println("PRACTIFS: Wrote: " +
          fswArr[1].getObjInvalTarget().getObjId() +
          ":" + writeLengths[1]);
        */
      }
    }catch(BadFileHandleException e){
      // Did not find all-zero-file-handle in cache; this is an error!
      assert(false);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      assert(false);
    }catch(Exception e){
      // Program error
      e.printStackTrace();
      assert false : ("" + e);
    }finally{
      if((dirDataObjId != null) && (dirDataToken != null)){
        super.localInterface.releaseWriteLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        super.localInterface.releaseWriteLock(dirMetaToken, dirMetaObjId);
      }
    }
  }

 /** 
 *  Return the attributes for this object 
 **/ 
  protected PRACTIFSDirectory
  readLocalDir(PRACTIFSReadLockToken token, ObjId dirDataObjId)
    throws IOException, IsNotDirectoryException{
    PRACTIFSDirectory dir = null;
    ChainReplicationFileInputStream pfis = null;

    Env.dprintln(DEBUG, "readLocalDir " + dirDataObjId);
    assert(token.isAcquired());
    try{
      pfis = new ChainReplicationFileInputStream((ChainReplicationFSLocalInterface)super.localInterface,
                                                 dirDataObjId,
                                                 PRACTIFS.DEFAULT_READ_BLOCK_SIZE);
      //Env.dprintln(DEBUG, "readDir " + dirDataObjId + " 1 ");
      dir = new PRACTIFSDirectory(pfis);
      //Env.dprintln(DEBUG, "readDir " + dirDataObjId + " 2");
      
    }finally{
      if(pfis != null){
        // Close the stream
        pfis.close();
      }
    }
    return(dir);
  }


}

 /** 
/* $Log: ChainReplicationFS.java,v $
/* Revision 1.3  2007/03/11 21:19:54  nalini
/* chain replication consistency and volume revamped
/*
/* Revision 1.2  2007/01/21 19:23:19  nalini
/* ChainReplication works with NFS -- needed change NFSlib so that chainRep node makes root locally
/*
/* Revision 1.1  2007/01/18 23:47:13  nalini
/* chain replication case study added
/*
 */
 **/ 
