 /** 
/* ChainReplicationFSLocalInterface
 *  
 * all writes are bound 
 * all reads are blocked until valid and precise
 *  
 * Note: currently all methods in this are synchronized
 *       if there is a preformance bottleneck, we can incorporate
 *       mulitple threads to get head/tail from overlog.
 *       so that this interface does not block.
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.IOException;
import java.io.EOFException;
import java.rmi.*;
import java.util.*;

public class ChainReplicationFSLocalInterface extends PRACTIFSLocalInterface {
 
  public URAOverlogNode overlogNode;
  private Hashtable headTable;
  private Hashtable tailTable;
  private int OVERLOG_PUSH = 1000; // 1 second
  
  private ChainReplicationFSRMIServerImpl serverImpl;
  private int MAX_TRIES = 1;
  private int RETRY_WAIT = 5000; // 5 seconds
  private boolean DEBUG = false;

 /** 
 *  Constructor 
 **/ 
  public 
  ChainReplicationFSLocalInterface(URAOverlogNode overlogNode, 
                                   NodeId myNodeId, 
                                   ChainReplicationFSRMIServerImpl serverImpl) {
    super(overlogNode.getLocalInterface(), myNodeId, true);  // bound writes
    this.overlogNode = overlogNode;
    this.serverImpl = serverImpl;
    
    this.headTable = new Hashtable();
    this.tailTable = new Hashtable();
  }


 /** 
 *  Read from an object 
 **/ 
  public synchronized BodyMsg
  read(ObjId objId,
       long offset,
       long length)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    BodyMsg result = null;

    Env.dprintln(DEBUG, "ChainRepFSLI: read for " + objId);
    
    int currentTry = 0;
    while(true){
      try{
        String volId  = serverImpl.getVolId(objId);
        Env.dprintln(DEBUG, "ChainRepFSLI: volId for " + objId + " = " + volId);
        NodeId tailId = getTail(volId);
        Env.dprintln(DEBUG, "ChainRepFSLI: tail for " + volId + " = " + tailId);
        String rmiURL = "rmi://" + Config.getDNS(tailId) + "/" + tailId.toString() + "-ChainReplicationFS"; 
        ChainReplicationFSRMIServer rmiServer = (ChainReplicationFSRMIServer)Naming.lookup(rmiURL);
        if(rmiServer != null) {
          result = rmiServer.read(objId, offset, length);
        }
        return(result);

      }catch(java.rmi.NotBoundException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepFSLI: Exception in read for " + objId + " retrying");
        currentTry++; 
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepFSLI:reached max retries for read for " + objId);
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}

      }catch(java.rmi.RemoteException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepFSLI: Exception in read for " + objId + " retrying");
        currentTry++;
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepFSLI:reached max retries for read for " + objId);
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}
      }
    }
  }   

   
 

 /** 
 *  Write method called by PRACTIFS 
 **/ 
  public synchronized long[] write(FSWriteEntry[] fwe)
    throws IOException {
    long[] result = null;
    int currentTry = 0;
    
    Env.dprintln(DEBUG, "ChainRepFSLI: write for multiobj");
    // we assume that all the multiple entries is in a single volue
    
    while(true) {
      try{
        ObjInvalTarget oit = fwe[0].getObjInvalTarget();
        String volId = serverImpl.getVolId(oit.getObjId());
        Env.dprintln(DEBUG, "ChainRepFSLI: volId for " + oit.getObjId() + " = " + volId);
        NodeId headId = getHead(volId);
        Env.dprintln(DEBUG, "ChainRepFSLI: head for " + volId + " = " + headId);
        String rmiURL = "rmi://" + Config.getDNS(headId) + "/" + headId.toString() + "-ChainReplicationFS"; 
        ChainReplicationFSRMIServer rmiServer = (ChainReplicationFSRMIServer)Naming.lookup(rmiURL);
        if(rmiServer != null) {
          result = rmiServer.write(fwe);
        }
	return result;
      }catch(java.rmi.NotBoundException z){
        // the head have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepFSLI: Exception in write for retrying");
        currentTry++; 
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepFSLI:reached max retries for write");
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}

      }catch(java.rmi.RemoteException z){
        Env.dprintln(DEBUG, "ChainRepFSLI: Exception in write .. retrying");
        // the tail have died or is recovering, try again after some time
        currentTry++;
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepFSLI:reached max retries for write");
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}
      }
    }
  }   

 /** 
 *  Write method 
 **/ 
  public synchronized long
  write(ObjId objId, long offset, long length, byte buffer[]) 
    throws IOException {
    Env.dprintln(DEBUG, "ChainRepFSLI: write for " + objId);
    long numBytes = 0;
    int currentTry = 0;

    while(true) {
      try{
        String volId = serverImpl.getVolId(objId);
	Env.dprintln(DEBUG, "ChainRepFSLI: vol for " + objId + "= " + volId);
        NodeId headId = getHead(volId);
	Env.dprintln(DEBUG, "ChainRepFSLI: head for " + volId + " = " + headId);
        String rmiURL = "rmi://" + Config.getDNS(headId) + "/" + headId.toString() + "-ChainReplicationFS"; 
        ChainReplicationFSRMIServer rmiServer = (ChainReplicationFSRMIServer)Naming.lookup(rmiURL);
        if(rmiServer != null) {
          numBytes = rmiServer.write(objId, offset, length, buffer);
        }
	  return numBytes;
      }catch(java.rmi.NotBoundException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepFSLI: Exception in write for " + objId + " retrying");
        currentTry++; 
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepFSLI: reached max retires for write for  " + objId ); 
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}

      }catch(java.rmi.RemoteException z){
        // the tail have died or is recovering, try again after some time
        Env.dprintln(DEBUG, "ChainRepFSLI: Exception in write for " + objId + " retrying");
        currentTry++;
        if (currentTry >=  MAX_TRIES){
          Env.dprintln(DEBUG, "ChainRepFSLI: reached max retries for write for  " + objId ); 
          throw new IOException("Problem in read, reached max retries");
        }
        try{
          wait(RETRY_WAIT);
        }catch(InterruptedException e) {}
      }
    }
  }   
 


 /** 
 *  getHead  
 *   -- from the hash table 
 **/ 

  protected synchronized NodeId getHead(String volId) {
      while(headTable.get(volId) == null) {
	  try{
	      wait(OVERLOG_PUSH);
	  }catch(InterruptedException e){ }
      }
      return (NodeId) headTable.get(volId);
  }

 /** 
 *  getTail from Overlog  
 *   -- inserts a tuple and waits for response 
 **/ 


  protected synchronized NodeId getTail(String volId){
      // wait for overlog to push some tuples 
      while(tailTable.get(volId) == null) {
	  try{
	      wait(OVERLOG_PUSH);
	  }catch(InterruptedException e){ }
      }
     
      return (NodeId) tailTable.get(volId);
  }

 /** 
 *  recvHead  
 *   -- sets the head variable as received from Overlog and notifies blocked 
 *      threads 
 **/ 

  public synchronized void recvHead(String volId, NodeId headNodeId) {
    headTable.put(volId, headNodeId);
    notifyAll();
  }

 /** 
 *  recvTail 
 *   -- sets the tail variable as received from Overlog and notifies blocked 
 *      threads 
 **/ 

  public synchronized void recvTail(String volId, NodeId tailNodeId) {
    tailTable.put(volId, tailNodeId);
    notifyAll();
  }

 /** 
 *  ReadLocal from an object -- gets handled locally by serverImpl 
 **/ 
  public synchronized BodyMsg
  readLocal(ObjId objId,
            long offset,
            long length)
    throws ObjNotFoundException, IOException, EOFException, ReadOfHoleException{
    Env.dprintln(DEBUG, "ChainRepFSLI: readLocal for " + objId);
    return serverImpl.read(objId, offset, length);
  } 

 /** 
 *  WriteLocal  -- called when initializing root dir by nfs interface 
 *              -- goes around all P2 related stuff and goes directly 
 *              -- to the local interface 
 **/ 
  public synchronized long[] writeLocal(FSWriteEntry[] fwe)
    throws IOException {
    Env.dprintln(DEBUG, "ChainRepFSLI: writeLocal for multi obj");
    return serverImpl.writeLocal(fwe);
  }

 /** 
 *  WriteLocal  -- called when initializing root dir by nfs interface 
 *              -- goes around all P2 related stuff and goes directly 
 *              -- to the local interface 
 **/ 
  public synchronized long
  writeLocal(ObjId objId, long offset, long length, byte buffer[]) 
    throws IOException {
    Env.dprintln(DEBUG, "ChainRepFSLI: writeLocal for multi obj");
    return serverImpl.writeLocal(objId, offset, length, buffer);
  }
}
 /** 
/* $Log: ChainReplicationFSLocalInterface.java,v $
/* Revision 1.4  2007/03/11 21:19:54  nalini
/* chain replication consistency and volume revamped
/*
/* Revision 1.3  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.2  2007/01/21 19:23:19  nalini
/* ChainReplication works with NFS -- needed change NFSlib so that chainRep node makes root locally
/*
/* Revision 1.1  2007/01/18 23:47:13  nalini
/* chain replication case study added
/*
*/
 **/ 
