 /** 
/* ChainReplicationFSRMIServerImpl.java
 *
 *  RMIServer for ChainReplicationFS so that the reads can be
 *  redirected to the tail and the writes can be 
 *  redirected to the head.
 *
 *  All reads are blocked until Valid and Precise
 *  All writes are bound
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.rmi.*;
import java.rmi.server.*;
import java.io.*;

public class ChainReplicationFSRMIServerImpl extends UnicastRemoteObject
 implements ChainReplicationFSRMIServer {

  private LocalInterface li;
  private NodeId myNodeId;
  private SyncMailBox syncBox;
  private boolean nfs;
  private boolean disableVol;

  private boolean started = false;

  private boolean bound = false;
  private boolean embargoed = false;

  private boolean dbg = false;

 /** 
 *  Constructor 
 **/ 

  public ChainReplicationFSRMIServerImpl(LocalInterface li,
                                         NodeId myNodeId, 
                                         SyncMailBox syncBox,
                                         boolean nfs,
                                         boolean disableVol)
    throws java.rmi.RemoteException {
    this.li = li;
    this.myNodeId = myNodeId;
    this.syncBox = syncBox;
    this.nfs = nfs;
    this.disableVol = disableVol;
  }


 /** 
 *  Start() - registers the server with the rmi registry 
 **/ 
  
  public void start(){

    try {
      if (!started){
	Naming.rebind(myNodeId.toString()+"-ChainReplicationFS", this);
	started = true;
      }
    }catch(Exception e){
      e.printStackTrace();
      System.err.println("*****************\n"
                         + "ChainReplicationFSRMIServerImpl:start -- did you "
                         + "remember to run \'rmiregistry &\'?"
                         + "\nDid you remember to remake rmic "
                         + " after any changes?\n"
                         + "*****************");
      System.exit(-1);
    }

  }


 /** 
 *  getVolId 
 *    
 *   if diableVol = true, all volId = 1 
 *  
 *    assumes that the first part of the objId is the volId 
 *   i.e. if ObjId = /foo/bar 
 *         volId = foo 
 *  
 *   it gets more complicated for if it is called thro NFS 
 *  if I have the following view through nfs 
 *  dir1, dir1/file1,  and dir2, dir2/file2 
 *   there will be the following objIDs generated 
 *  - root.meta 
 *  - root.data 
 *  - root.data/dir1.data 
 *  - root.data/dir1.meta 
 *  - root.data/dir1.data/file1.data     
 *  - root.data/dir1.data/file1.meta 
 *  - root.data/dir2.data 
 *  - root.data/dir2.meta 
 *  - root.data/dir2.data/file2.data     
 *  - root.data/dir2.data/file2.meta 
 *  
 *  volId of root.meta and root.data  => "root" 
 *  for everything else, the name of the top-level directory is the volId 
//  e.g. root.data/foo.data/*  => "foo"
//  e.g. root.data/foo.meta  => "foo"
//-----------------------------------------------------------------------
  public String getVolId(ObjId objId) {

    if (disableVol) return "0";
    
    String objStr = objId.toString();
 
    if(!nfs) {
      // if it is not thro NFS, this is simple
      int i = objStr.indexOf('/', 1); // we ignore the starting '/'
      String volId = objStr.substring(1, i);
      return volId;
     }

    String rootdir = PRACTIFS.ROOT_DIR_NAME;
    String dataSuffix = PRACTIFS.DATA_SUFFIX;
    String metaSuffix = PRACTIFS.META_SUFFIX;

    if(objStr.equals(rootdir+dataSuffix) ||
       objStr.equals(rootdir+metaSuffix)) {
      //root dir
      return "root";
    }

    String rootData= rootdir + dataSuffix;
    assert(objStr.indexOf(rootData) == 0);
    
    String str =  objStr.substring(rootData.length());
    int i = str.indexOf(dataSuffix);
    String volId = objStr.substring(1, i);
    return volId;
  }
//-----------------------------------------------------------------------
// read
//-----------------------------------------------------------------------
  public BodyMsg read(ObjId objId, long offset, long length)
    throws java.rmi.RemoteException, ObjNotFoundException, 
	   EOFException, IOException, ReadOfHoleException {
    BodyMsg result = null;

    Env.dprintln(dbg, "ChainRepRMIServerImpl: read issued at node " + 
                 myNodeId  + " for obj" + objId);
   try{
      result = this.li.read(objId,
			    offset,
			    length,
			    true,//blockInvalid
			    false);//blockPrecise
   }catch(ReadOfInvalidRangeException roire){
     assert(false); // this should not happen 
   }
    return(result);
  }

    


//-----------------------------------------------------------------------
// write
//-----------------------------------------------------------------------
  public long write(ObjId objId, long offset, long length, byte buffer[])
    throws java.rmi.RemoteException, IOException {
    long numBytes = 0;
    double priority = 0.0;

    Env.dprintln(dbg, "ChainRepRMIServerImpl: write issued at node " + 
                 myNodeId  + " for obj" + objId);

    // Pass the call onto the localInterface
    numBytes = this.li.write(objId,
                             offset,
                             length,
                             priority,
                             buffer,
                             bound);

    BodyMsg bMsg=null;

    // find out what the accept stamp was
    try{
      bMsg = this.li.read(objId,
                          offset,
                          length,
                          false,//no need to block here
                          false);// no need to block here
    }catch(ReadOfInvalidRangeException roire){
      assert(false); // this should not happen
    } catch(ObjNotFoundException onfe){
      assert(false); // this should not happen
    } catch(ReadOfHoleException rohe) {
      assert(false); // this should not happen
    }
   
    assert(bMsg != null);
    AcceptStamp as = bMsg.getAcceptStamp();
    // wait till we receive an ack
    String volId = getVolId(objId);
    Env.dprintln(dbg, "ChainRepRMIServerImpl: waiting for ack for volId = " + volId + "as= "+ as);
    syncBox.waitForAck(volId, as);
    Env.dprintln(dbg, "ChainRepRMIServerImpl: recvd ack for volId = " + volId + "as= "+ as);
    return(numBytes);
  }



//-----------------------------------------------------------------------
// write
//-----------------------------------------------------------------------
  public long[] write(FSWriteEntry[] fwe)
    throws java.rmi.RemoteException, IOException {
     double priority = 0.0;
     long[] result = null;

    Env.dprintln(dbg, "ChainRepRMIServerImpl: write issued at node " + 
                 myNodeId  + " for multi obj");

    MultiWriteEntry[] mwe = createMultiWriteEntry(fwe, priority, bound);
    result = this.li.writeMulti(mwe, embargoed);

    ObjInvalTarget oit=null;
    BodyMsg bMsg = null;
    // find out what the accept stamp of the last entry was
    try{
      oit = mwe[mwe.length - 1].getObjInvalTarget();
      assert(oit != null);

      bMsg = this.li.read(oit.getObjId(),
                          oit.getOffset(),
                          oit.getLength(),
                          false,//no need to block here
                          false);// no need to block here
    }catch(ReadOfInvalidRangeException roire){
      assert(false); // this should not happen
    }catch(ObjNotFoundException onfe) {
      assert(false); // this should not happen
    }catch(ReadOfHoleException rohe) {
      assert(false); // this should not happen
    }
   
    assert(bMsg != null);

    AcceptStamp as = bMsg.getAcceptStamp();
    // wait till we receive an ack
    String volId = getVolId(oit.getObjId());
    Env.dprintln(dbg, "ChainRepRMIServerImpl: waiting for ack for volId = " + volId + "as= "+ as);
    syncBox.waitForAck(volId, as);  
    Env.dprintln(dbg, "ChainRepRMIServerImpl: recvd ack for volId = " + volId + "as= "+ as);  
    return result;    
  }


//-----------------------------------------------------------------------
// createMultiWriteEntry - converts FSWriteEntry to MultiWriteEntry
//  incorporating the priority and bound parameters
//-----------------------------------------------------------------------
  protected MultiWriteEntry[] 
  createMultiWriteEntry(FSWriteEntry[] fwe, double priority, boolean useBound) {
    MultiWriteEntry[] mwe = new MultiWriteEntry[fwe.length];
    boolean useBValue;

    for(int i=0; i<mwe.length; i++) {
      // try to avoid bounded deletes!
      useBValue = useBound;
      if(fwe[i].getDelete()) {
        useBValue = false;
      } 
      mwe[i] = new MultiWriteEntry(fwe[i].getObjInvalTarget(),
                                   priority,
                                   fwe[i].getImmutableBytes(),
                                   useBValue,
                                   fwe[i].getDelete());
    }

    return mwe;
  }

  //-----------------------------------------------------------------------
  // WriteLocal  -- called when initializing root dir by nfs interface
  //             -- goes around all P2 related stuff and goes directly
  //             -- to the local interface
  //-----------------------------------------------------------------------

  public long[] writeLocal(FSWriteEntry[] fwe)
    throws java.rmi.RemoteException, IOException {
     double priority = 0.0;
     long[] result = null;

    Env.dprintln(dbg, "ChainRepRMIServerImpl: write issued at node " + 
                 myNodeId  + " for multi obj");

    MultiWriteEntry[] mwe = createMultiWriteEntry(fwe, priority, bound);
    result = this.li.writeMulti(mwe, embargoed);
    
    return result;
  }

  //-----------------------------------------------------------------------
  // writeLocal   -- called when initializing root dir by nfs interface
  //             -- goes around all P2 related stuff and goes directly
  //             -- to the local interface
  //-----------------------------------------------------------------------
  public long writeLocal(ObjId objId, long offset, long length, byte buffer[])
    throws java.rmi.RemoteException, IOException {
    long numBytes = 0;
    double priority = 0.0;

    Env.dprintln(dbg, "ChainRepRMIServerImpl: write issued at node " + 
                 myNodeId  + " for obj" + objId);

    // Pass the call onto the localInterface
    numBytes = this.li.write(objId,
                             offset,
                             length,
                             priority,
                             buffer,
                             bound);
    return numBytes;
  }


}
//---------------------------------------------------------------------------
/* $Log: ChainReplicationFSRMIServerImpl.java,v $
/* Revision 1.6  2007/03/21 22:21:04  nalini
/* CR-Exp2 checked in
/*
/* Revision 1.5  2007/03/12 03:37:07  nalini
/* Chain Replication exp1 files added
/*
/* Revision 1.4  2007/03/11 21:19:54  nalini
/* chain replication consistency and volume revamped
/*
/* Revision 1.3  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.2  2007/01/21 19:23:19  nalini
/* ChainReplication works with NFS -- needed change NFSlib so that chainRep node makes root locally
/*
/* Revision 1.1  2007/01/18 23:47:13  nalini
/* chain replication case study added
/*
*/
 **/ 
