 /** 
/* ChainReplicationSimpleTest
 *
 * Tests if the reads are correctly forwarded to the head
 * and if the writes are forwarded to the tail.
 *
 * This test disables volumes.
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class ChainReplicationSimpleTest {

  protected static long NODE_0_ID = 0;
  protected static long NODE_1_ID = 1;
  protected static long NODE_2_ID = 2;
  protected static long NODE_3_ID = 3;
  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.ChainRep.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.ChainRep.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.ChainRep.nodemap";
  protected  String OVERLOG_PATH = "ChainReplication.olg";
  protected static boolean disableVol = true;


  protected static int sizeOfWrites = 20;
  protected static byte value_0 = 65; //"A";
  protected static byte value_1 = 66; //"B";
  protected static byte value_2 = 67; //"C";
  protected static byte value_3 = 68; //"D";
  protected static long timeToWait = 500; // time to retry a read after objNotFound


  static private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 3000;
  static private BarrierClient c;

  private long myNodeId;
  private ChainReplicationNode node;
  private ChainReplicationFSLocalInterface node_interface;
 
  private boolean dbg = true;


  
 /** 
 *  Constructor 
 **/ 
 
  public ChainReplicationSimpleTest(long nodeId, String overlogPath) {
    if(nodeId == 0) {
      myNodeId = NODE_0_ID;
    } else if (nodeId == 1){
      myNodeId = NODE_1_ID;
    } else if (nodeId == 2) {
      myNodeId = NODE_2_ID;
    } else if (nodeId == 3) {
      myNodeId = NODE_3_ID;
    }

    // set up environment
    if(myNodeId == NODE_0_ID) {
      OVERLOG_PATH = overlogPath;
      makePractiConfig(CONFIG_PATH);
      makeP2Config(CONFIG_P2_PATH);
      makeNodeIdMap(NODEID_MAP_PATH);

      barrierServer = new BarrierServer(BARRIER_PORT, 4, 9); // for simple test
       barrierServer.start();
      System.out.println("Started barrier Server");
    }

      c =  new BarrierClient("localhost", 
                             BARRIER_PORT,
                             (int) myNodeId);


      node = new ChainReplicationNode(CONFIG_PATH, 
                                      CONFIG_P2_PATH,
                                      new NodeId(myNodeId),
                                      true,
                                      OVERLOG_PATH,
                                      NODEID_MAP_PATH,
                                      false,
                                      disableVol);

      node_interface = node.getFSInterface();

  }

 /** 
 *  Start the nodes 
 **/ 

  public void startNode() throws Exception {
    node.start();
    Thread.currentThread().sleep(20000); // wait for overlog to install
    System.out.println("Started node");

    c.sendBarrierRequest(0, myNodeId); // Nodes set up now we can start

  }

 /** 
 *  Shutdown the nodes 
 **/ 

  public void shutdownNode() throws Exception {
    node.shutdown();
  }



 /** 
 *  Make configuration files 
 **/ 
  private static void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    
    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_0_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_1_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(NODE_2_ID),
			   "localhost",
			   9788,
			   9789,
			   9791,
			   9792,
                           9790,
                           "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			   NODE_2_ID+".db",
			   "/*",
			   -1L,
			   "localhost",
			   9793,
			   9794,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.addOneNodeConfig(new NodeId(NODE_3_ID),
                            "localhost",
                            9688,
                            9689,
                            9691,
                            9692,
                            9690,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_3_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9693,
                            9694,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
   Config.writeToFile(configPath);
  }

  private static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_0_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_1_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_2_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_3_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_0_ID), new OverlogId("localhost:5000"));
    NodeIdMap.add(new NodeId(NODE_1_ID), new OverlogId("localhost:5001"));
    NodeIdMap.add(new NodeId(NODE_2_ID), new OverlogId("localhost:5002"));
    NodeIdMap.add(new NodeId(NODE_3_ID), new OverlogId("localhost:5003"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
//  Converts values to byte arrays
//---------------------------------------------------------------------------

  private byte[] convertToArray(byte value){
    byte[] b = new byte[sizeOfWrites];
    for(int i = 0; i < sizeOfWrites; i++) {
      b[i] = value;
    }
    return b;
  }


//---------------------------------------------------------------------------
//  Writes a value to the specified interfac
//---------------------------------------------------------------------------

  private void write(ChainReplicationFSLocalInterface fsInterface, ObjId objId, byte[] b)
    throws IOException{
    fsInterface.write(objId, 0, sizeOfWrites, b);
  }

//---------------------------------------------------------------------------
// Reads an object from the specified interface
// If objNotFound -- sleeps for some time and tries again
//---------------------------------------------------------------------------

  private ImmutableBytes read(ChainReplicationFSLocalInterface fsInterface, ObjId objId) {
    ImmutableBytes b=null;
    try{
      while (true) {
	try{
          BodyMsg msg = fsInterface.read(objId, 0, sizeOfWrites);
	  if(msg != null) {
            b = msg.getBody();
          }
	  break;
	}catch(ObjNotFoundException e){
	  //wait until you recieve obj
	  Env.dprintln(dbg, "ObjNotFoundException for " + objId);
	  Thread.currentThread().sleep(timeToWait);
	}
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    return b;
  }


//---------------------------------------------------------------------------
// runTest
//  - simple test which just does a bunch of writes and then reads
//  - just to check if the writes and reads get redirected safely
//---------------------------------------------------------------------------
  public void runTest() {
    try{
      ObjId obj0 = new ObjId("/0");
      ObjId obj1 = new ObjId("/1");
      ObjId obj2 = new ObjId("/2");
      ObjId obj3 = new ObjId("/3");
    
      byte[] b0 = convertToArray(value_0);
      byte[] b1 = convertToArray(value_1);
      byte[] b2 = convertToArray(value_2);
      byte[] b3 = convertToArray(value_3);

      Thread.currentThread().sleep(10000);
      //
      //  write objects
      //

      if(myNodeId == NODE_0_ID) {
        write(node_interface, obj0, b0);
        Env.dprintln(dbg, "Finished writing objs");
      }        
      c.sendBarrierRequest(0, myNodeId); //  node 0 finished writing

      if(myNodeId == NODE_1_ID) {
        write(node_interface, obj1, b1);
        Env.dprintln(dbg, "Finished writing objs");
      }        
      c.sendBarrierRequest(0, myNodeId); //  node 1 finished writing

      if(myNodeId == NODE_2_ID) {
        write(node_interface, obj2, b2);
        Env.dprintln(dbg, "Finished writing objs");
      }          
      c.sendBarrierRequest(0, myNodeId); //  node 2 finished writing

      if(myNodeId == NODE_3_ID) {
        write(node_interface, obj3, b3);
        Env.dprintln(dbg, "Finished writing objs");
      }        
      c.sendBarrierRequest(0, myNodeId); //  node 3 finished writing
 


      //
      // Read objects
      //  
      ImmutableBytes imb = null;
      if(myNodeId == NODE_3_ID) {
        imb = read(node_interface, obj0);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj0 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b0)));
      }  
      c.sendBarrierRequest(0, myNodeId); //  node 3 finished reading
  
      if(myNodeId == NODE_2_ID) {
        imb = read(node_interface, obj1);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj1 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b1)));
      }      
      c.sendBarrierRequest(0, myNodeId); //  node 2 finished reading

      if(myNodeId == NODE_1_ID) {
        imb = read(node_interface, obj2);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj2 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b2)));
      }  
      c.sendBarrierRequest(0, myNodeId); //  node 1 finished reading

      if(myNodeId == NODE_0_ID) {
        imb = read(node_interface, obj3);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj0 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b3)));
      }  
     
      c.sendBarrierRequest(0, myNodeId); //  node 0 finished reading

      System.out.println("Finished test");

    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
//  Main  Method
//---------------------------------------------------------------------------


  public static void main(String[] args) {
    try {
      Env.verifyAssertEnabled();
      
      if(args.length < 2) {
	System.out.println("Usage: java ChainReplicationSimpleTest <nodeId> <overlog file>");
	System.exit(0);
      }
      
      long nodeId = (new Long(args[0])).longValue();
      String filename = args[1];

      ChainReplicationSimpleTest simpleTest = new ChainReplicationSimpleTest(nodeId, filename);
      simpleTest.startNode();
      System.out.println("Starting Test ..");
      simpleTest.runTest();
      System.exit(0);
      
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }
  
}
//---------------------------------------------------------------------------
/* $Log: ChainReplicationSimpleTest.java,v $
/* Revision 1.4  2007/03/11 21:19:55  nalini
/* chain replication consistency and volume revamped
/*
/* Revision 1.3  2007/03/09 03:01:39  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.2  2007/01/21 19:23:19  nalini
/* ChainReplication works with NFS -- needed change NFSlib so that chainRep node makes root locally
/*
/* Revision 1.1  2007/01/18 23:47:14  nalini
/* chain replication case study added
/*
 */
 /** 
