 /** 
/* ChainReplicationVolTest
 *
 * Tests if the reads are correctly forwarded to the head
 * and if the writes are forwarded to the tail for
 * different volumes
 * 
 * Note: node 0 is the master, it does not partake in the chain, but
 * it issues, reads and writes
 *
 * Suggestion for vol->node map
 *  VolId 1,  Node 1, 2, 3
 *  VolId 2,  Node 2
 *  VolId 3,  Node 2, 3
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class ChainReplicationVolTest {

  protected static long NODE_0_ID = 0;
  protected static long NODE_1_ID = 1;
  protected static long NODE_2_ID = 2;
  protected static long NODE_3_ID = 3;
  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.ChainRep.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.ChainRep.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.ChainRep.nodemap";
  protected  String OVERLOG_PATH = "ChainReplication.olg";
  protected static boolean disableVol = false;

  protected static String vol1Prefix = "/1/";
  protected static String vol2Prefix = "/2/";
  protected static String vol3Prefix = "/3/";


  protected static int sizeOfWrites = 20;
  protected static byte value_1 = 66; //"B";
  protected static byte value_2 = 67; //"C";
  protected static byte value_3 = 68; //"D";
  protected static long timeToWait = 500; // time to retry a read after objNotFound


  static private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 3000;
  static private BarrierClient c;

  private long myNodeId;
  private ChainReplicationNode node;
  private ChainReplicationFSLocalInterface node_interface;
 
  private boolean dbg = true;


  
 /** 
 *  Constructor 
 **/ 
 
  public ChainReplicationVolTest(long nodeId, String overlogPath) {
    if(nodeId == 0) {
      myNodeId = NODE_0_ID;
    } else if (nodeId == 1){
      myNodeId = NODE_1_ID;
    } else if (nodeId == 2) {
      myNodeId = NODE_2_ID;
    } else if (nodeId == 3) {
      myNodeId = NODE_3_ID;
    }

    // set up environment
    if(myNodeId == NODE_0_ID) {
      OVERLOG_PATH = overlogPath;
      makePractiConfig(CONFIG_PATH);
      makeP2Config(CONFIG_P2_PATH);
      makeNodeIdMap(NODEID_MAP_PATH);

      barrierServer = new BarrierServer(BARRIER_PORT, 4, 2); // for simple test
      barrierServer.start();
    }

      c =  new BarrierClient("localhost", 
                             BARRIER_PORT,
                             (int) myNodeId);


      node = new ChainReplicationNode(CONFIG_PATH, 
                                      CONFIG_P2_PATH,
                                      new NodeId(myNodeId),
                                      true,
                                      OVERLOG_PATH,
                                      NODEID_MAP_PATH,
                                      false,
                                      disableVol);

      node_interface = node.getFSInterface();

  }

 /** 
 *  Start the nodes 
 **/ 

  public void startNode() throws Exception {
    node.start();
    Thread.currentThread().sleep(15000); // wait for overlog to install
    System.out.println("Started node");

    c.sendBarrierRequest(0, myNodeId); // Nodes set up now we can start

  }

 /** 
 *  Shutdown the nodes 
 **/ 

  public void shutdownNode() throws Exception {
    node.shutdown();
  }



 /** 
 *  Make configuration files 
 **/ 
  private static void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    
    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_0_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_1_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(NODE_2_ID),
			   "localhost",
			   9788,
			   9789,
			   9791,
			   9792,
                           9790,
                           "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			   NODE_2_ID+".db",
			   "/*",
			   -1L,
			   "localhost",
			   9793,
			   9794,
                            -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.addOneNodeConfig(new NodeId(NODE_3_ID),
                            "localhost",
                            9688,
                            9689,
                            9691,
                            9692,
                            9690,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_3_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9693,
                            9694,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
   Config.writeToFile(configPath);
  }

  private static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_0_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_1_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_2_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_3_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_0_ID), new OverlogId("localhost:5000"));
    NodeIdMap.add(new NodeId(NODE_1_ID), new OverlogId("localhost:5001"));
    NodeIdMap.add(new NodeId(NODE_2_ID), new OverlogId("localhost:5002"));
    NodeIdMap.add(new NodeId(NODE_3_ID), new OverlogId("localhost:5003"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
//  Converts values to byte arrays
//---------------------------------------------------------------------------

  private byte[] convertToArray(byte value){
    byte[] b = new byte[sizeOfWrites];
    for(int i = 0; i < sizeOfWrites; i++) {
      b[i] = value;
    }
    return b;
  }


//---------------------------------------------------------------------------
//  Writes a value to the specified interfac
//---------------------------------------------------------------------------

  private void write(ChainReplicationFSLocalInterface fsInterface, ObjId objId, byte[] b)
    throws IOException{
    fsInterface.write(objId, 0, sizeOfWrites, b);
  }

//---------------------------------------------------------------------------
// Reads an object from the specified interface
// If objNotFound -- sleeps for some time and tries again
//---------------------------------------------------------------------------

  private ImmutableBytes read(ChainReplicationFSLocalInterface fsInterface, ObjId objId) {
    ImmutableBytes b=null;
    try{
      while (true) {
	try{
          BodyMsg msg = fsInterface.read(objId, 0, sizeOfWrites);
	  if(msg != null) {
            b = msg.getBody();
          }
	  break;
	}catch(ObjNotFoundException e){
	  //wait until you recieve obj
	  Env.dprintln(dbg, "ObjNotFoundException for " + objId);
	  Thread.currentThread().sleep(timeToWait);
	}
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    return b;
  }


//---------------------------------------------------------------------------
// runTest
//  - simple test which just does a bunch of writes and then reads
//  - just to check if the writes and reads get redirected safely
//---------------------------------------------------------------------------
  public void runTest() {
    try{
      ObjId obj1 = new ObjId(vol1Prefix+"obj");
      ObjId obj2 = new ObjId(vol2Prefix+"obj");
      ObjId obj3 = new ObjId(vol3Prefix+"obj");
    
      byte[] b1 = convertToArray(value_1);
      byte[] b2 = convertToArray(value_2);
      byte[] b3 = convertToArray(value_3);



      if(myNodeId == NODE_0_ID) {
        //
        //  write objects
        //
        write(node_interface, obj1, b1);
        Env.dprintln(dbg, "Finished writing obj for vol1");
	Thread.currentThread().sleep(2000);
        write(node_interface, obj2, b2);
        Env.dprintln(dbg, "Finished writing obj for vol2");
	Thread.currentThread().sleep(2000);
        write(node_interface, obj3, b3);
        Env.dprintln(dbg, "Finished writing obj for vol3");
	Thread.currentThread().sleep(5000);

        //
        // Read objects
        //  
        ImmutableBytes imb = null;
        imb = read(node_interface, obj1);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj1 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b1)));

	Thread.currentThread().sleep(2000);
        imb = read(node_interface, obj2);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj2 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b2)));

	Thread.currentThread().sleep(2000);
        imb = read(node_interface, obj3);
        Env.dprintln(dbg, "Read issued at node" + myNodeId + ": objId="+ obj3 +" value read="+ imb);
        assert(imb.equals(new ImmutableBytes(b3)));

      }        
      c.sendBarrierRequest(0, myNodeId); //  finished experiment
 
      System.out.println("Finished test");

    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
//  Main  Method
//---------------------------------------------------------------------------


  public static void main(String[] args) {
    try {
      Env.verifyAssertEnabled();
      
      if(args.length < 2) {
	System.out.println("Usage: java ChainReplicationVolTest <nodeId> <overlog file>");
	System.exit(0);
      }
      
      long nodeId = (new Long(args[0])).longValue();
      String filename = args[1];

      ChainReplicationVolTest simpleTest = new ChainReplicationVolTest(nodeId, filename);
      simpleTest.startNode();
      System.out.println("Starting Test ..");
      simpleTest.runTest();
      System.exit(0);
      
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }
  
}
//---------------------------------------------------------------------------
/* $Log: ChainReplicationVolTest.java,v $
/* Revision 1.1  2007/03/11 21:19:55  nalini
/* chain replication consistency and volume revamped
/*
 */
 /** 
