 /** 
/* SyncMailBox
 *
 * blocks a thread until an ack of a smaller timestamp is received
 * 
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.util.*; 

public class SyncMailBox {
  private LinkedList pendingAcks;
  private LinkedList receivedAcks;
  
  private boolean dbg = false;

 /** 
 *  Constructor 
 **/ 
  public SyncMailBox() {
    pendingAcks = new LinkedList();
    receivedAcks = new LinkedList();
  }

 /** 
 *  Adds to the pendingAcks list and waits 
 **/ 
  
  public synchronized void waitForAck(String volId, AcceptStamp as) {
    if(checkIfAlreadyReceivedAck(volId, as)) {
      return;
    }
    PendingAck requiredAck = new PendingAck(volId, as);
    pendingAcks.add(requiredAck);
    Env.dprintln(dbg, "SyncMailBox:: added " + requiredAck + " to pendingList");
    while(pendingAcks.contains(requiredAck)){
      try{
        wait();
      }catch(InterruptedException e) {
      }
    }
    assert(!pendingAcks.contains(requiredAck));
  }

 /** 
 *  Checks if we Already have received a ack for this 
 **/ 
  
  public synchronized boolean checkIfAlreadyReceivedAck(String volId, AcceptStamp as) {
    Iterator i =receivedAcks.iterator();
    while (i.hasNext()){
      PendingAck received = (PendingAck) i.next();
      if(received.getVolId().equals(volId) &&
         as.leq(received.getAcceptStamp())) {
        return true;
      }
    }
    return false;
  }

 /** 
 *  Removes all pending with lower acceptstamps and notifies all 
 **/ 

  public synchronized void recvAck(String volId, AcceptStamp as) {
    Env.dprintln(dbg, "SyncMailBox:: recvAck for volId = " + volId + " as = " + as);
    Iterator i = pendingAcks.iterator();
    while(i.hasNext()) {
      PendingAck pending = (PendingAck) i.next();
      if(pending.getVolId().equals(volId) &&
         pending.getAcceptStamp().leq(as)){
        Env.dprintln(dbg, "SyncMailBox:: removing " + pending + " from list");
        i.remove();
      }
    }
    addToReceivedAck(volId, as);
    notifyAll();
  }

 /** 
 *  Adds to received ack table... removes all ac with lower time stamps 
 **/ 

  public synchronized void addToReceivedAck(String volId, AcceptStamp as) {
    Iterator i = receivedAcks.iterator();
    while(i.hasNext()) {
      PendingAck received = (PendingAck) i.next();
      if(received.getVolId().equals(volId) &&
         received.getAcceptStamp().leq(as)){
        i.remove();
      }
    }
    receivedAcks.add(new PendingAck(volId, as));
  }


  public class PendingAck {
    private final String volId;
    private final AcceptStamp as;

    public PendingAck(String volId, AcceptStamp as) {
      this.volId = volId;
      this.as = as;
    }

    public String getVolId() {
      return volId;
    }
    
    public AcceptStamp getAcceptStamp() {
      return as;
    }

    public boolean equals(Object o) {
      if (this == o ) 
        return true;

      if (!(o instanceof PendingAck))
        return false;

      PendingAck other = (PendingAck) o;
      if(this.volId.equals(other.volId) && this.as.equals(other.as))
        return true;
      
      return false;
    }

    public String toString() {
      return "<" + volId + ", " + as + ">";
    }
  }
}

 /** 
/* $Log: SyncMailBox.java,v $
/* Revision 1.2  2007/03/15 20:26:00  nalini
/* keeping track of received acks to get rid of timing discrepancies
/*
/* Revision 1.1  2007/03/11 21:19:55  nalini
/* chain replication consistency and volume revamped
/*
 */
 **/ 
