 /** 
/* TestClient
 *  
 * Generates a random workload based
 * on percentage of updates
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.util.*;

public class TestClient implements Runnable {

  // we assume this is how the DB has been populated!
  private static int numVols = 1;
  private static int numObjPerVol = 1000;
  private static int sizeOfObj = 1000;
  private static byte value = 65; // A

  private static int numChainNodes;
  private int updatePercent;
  private int myId;

  private TreeSet requestList;
  private long numRequestCompleted = 0;
  private long startTime = 0;
  private long endTime = 0;

  private ChainReplicationClient cr;
  private Random ran1; // for requests type
  private Random ran2; // for volId
  private Random ran3; // for objId
  private byte [] buffer;

  private boolean  stopClient = false;
  
  private boolean dbg = true;

 /** 
 *   constructor 
 **/ 

  public TestClient(String configFile, int numChainNodes, int updatePercent, int myId){
    this.numChainNodes = numChainNodes;
    this.updatePercent = updatePercent;
    this.myId = myId;
    this.cr = new ChainReplicationClient(configFile);
    this.cr.putVolHeadTail("0", new NodeId(1), new NodeId(numChainNodes));
    this.requestList = new TreeSet();
    this.ran1 = new Random();
    this.ran2 = new Random();
    this.ran3 = new Random();
    this.buffer = convertToArray(sizeOfObj, value);

    generateRequestList();
  }

 /** 
 *   Converts values to byte arrays 
 **/ 

  private byte[] convertToArray(int sizeOfObj, byte value){
    byte[] b = new byte[sizeOfObj];
    for(int i = 0; i < sizeOfObj; i++) {
      b[i] = value;
    }
    return b;
  }

 /** 
 *  Generates 1000 requests and puts it in the requestlist 
 **/ 
  void generateRequestList() {
    for (int i = 0; i < updatePercent*10; i++){
      int r = ran1.nextInt();
      requestList.add(new Request(true, r));  // write
    }

    for(int i = updatePercent*10; i <1000; i++) {
      int r = ran1.nextInt();
      requestList.add(new Request(false, r)); //read
    }
  }

 /** 
 *  Generates objId 
 **/ 
  ObjId generateObjId() {
    int vol = 0;
    if(numVols > 1) {
      vol = ran2.nextInt(numVols); 
    }
    int obj = ran3.nextInt(numObjPerVol);

    return new ObjId("/" + vol + "/" + obj);
  }

 /** 
 *  stopClient 
 **/ 
  public void stopClient() {
    stopClient = true;
    Env.dprintln(dbg, "TestClient " + myId + ":: stopClient called");
  }

 /** 
 *  issue request 
 **/ 
  public void issueRequest(Request req, ObjId objId) throws Exception {
    Env.dprintln(dbg, "TestClient " + myId + ":: issuing Request " + req + " for obj=" + objId);
    if(req.getRequest()) {
      //write
      cr.write(objId, 0, buffer.length, buffer);
    } else {
      //read
      cr.read(objId, 0, sizeOfObj);
    }
    Env.dprintln(dbg, "TestClient " + myId + "::  finished request");
  }
  

 /** 
 *   Run method -- issues requests until it is told to stop 
 **/ 

  public void run() {
    Iterator i = requestList.iterator();

    Env.dprintln(dbg, "TestClient " + myId + ":: started");
    startTime = System.currentTimeMillis();
    while(!stopClient) {
      Request req = null;
      if(i.hasNext()) {
        req = (Request) i.next();
      } else {
        // reached the end of the list, restart from the head again
        i = requestList.iterator();
        req = (Request) i.next();
      }

      ObjId objId = generateObjId();

      try{
        issueRequest(req, objId);
        numRequestCompleted ++;
	endTime = System.currentTimeMillis();
      }catch(Exception e) {
        e.printStackTrace();
      }
    } 


    Env.dprintln(dbg, "TestClient " + myId + ":: ended client");
  }

 /** 
 *  getThroughput  (operations per second) 
 *  -- after we have finished running 
 **/ 
  public double getThroughput() {

    System.out.println("numRequest = " + numRequestCompleted);
    System.out.println( "TestClient " + myId + ":: ran for " + (endTime - startTime));
    double throughput = (double) numRequestCompleted * 1000/(endTime - startTime);
    System.out.println("TestClient " + myId + ":: throughput = " + throughput);

    return throughput;
  }       


 /** 
 *  getNumRequestCompleted 
 *   
 **/ 
  public long getNumRequestCompleted() {
    return numRequestCompleted;
  }       

 /** 
 *   Private class 
 **/ 

  public class Request implements Comparable {
    boolean req;
    int random;

    Request(boolean req, int random) {
      this.req = req;
      this.random = random;
    }

    public  boolean getRequest() {
      return req;
    }
   
    public int compareTo(Object o) {
      return ((Request)o).random - this.random;
    }

    public String toString() {
      return "" + req + ":" +  random;
    }
  }
      
}
 /** 
/* $Log: TestClient.java,v $
/* Revision 1.3  2007/03/21 22:20:32  nalini
/* CR_Exp2 checked in
/*
/* Revision 1.2  2007/03/17 01:24:22  nalini
/* CR exp2 added
/*
/* Revision 1.1  2007/03/15 02:30:54  nalini
/* added TestClient
/*
 */
 **/ 
