 /** 
/* ChainReplicationExp1
 *
 * Single Node writes and reads
 *   - makes a bunch of writes, then reads
 *   - calculates the avergage time it takes to 
 *     make write and a read.
 *
 *
 *  Note: Node 0 is the master, and Node 1 is part of the chain
 *  we use P2JavaWrapper to start the master
 *  Then we start this node as with NODEID= 1.
 *
 *  
 *  Reads and Writes are carried out by node 1
 *
 *  Volumes are disabled in this test
 * 
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;

public class ChainReplicationExp1 implements MasterWatcher{

  protected static long NODE_0_ID = 0;
  protected static long NODE_1_ID = 1;
  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.ChainRep.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.ChainRep.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.ChainRep.nodemap";
  protected  String OVERLOG_PATH = "ChainReplication.olg";
  protected static boolean disableVol = true;


  protected static int sizeOfWrites = 1000; 
  protected static int numWrites = 50; //50
  protected static byte value = 65; //"A";
  protected static long timeToWait = 500; // time to retry a read after objNotFound


  private long myNodeId;
  private ChainReplicationNode node;
  private ChainReplicationFSLocalInterface node_interface;
  private boolean masterAlive = false;
  
  private boolean dbg = false;


  
 /** 
 *  Constructor 
 **/ 
 
  public ChainReplicationExp1(String overlogPath) {

    myNodeId = NODE_1_ID;

    // set up environment
    OVERLOG_PATH = overlogPath;
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(NODEID_MAP_PATH);


    node = new ChainReplicationNode(CONFIG_PATH, 
				    CONFIG_P2_PATH,
				    new NodeId(myNodeId),
				    true,
				    OVERLOG_PATH,
				    NODEID_MAP_PATH,
				    false,
				    disableVol);

    TupleHandler th = new MasterAliveTupleHandler(this);
    node.registerHandler(th);

    node_interface = node.getFSInterface();

  }

 /** 
 *  Start the nodes 
 **/ 

  public void startNode() throws Exception {
    node.start();
    Thread.currentThread().sleep(1000); // wait for overlog to install
    System.out.println("Started node");
  }

 /** 
 *  Shutdown the nodes 
 **/ 

  public void shutdownNode() throws Exception {
    node.shutdown();
  }


 /** 
 *  master Alive 
 **/ 
  public synchronized void masterAlive(boolean status){
    masterAlive = status;
    notifyAll();
  }
 

 /** 
 *  wait for master to be alive 
 **/ 
  public synchronized void waitForMaster() {
    while(!masterAlive){
      try{
        wait();
      }catch(InterruptedException e) {
        // do nothing
      }
    }
  }
    

 /** 
 *  Make configuration files 
 **/ 
  private static void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    
    Config.addOneNodeConfig(new NodeId(NODE_0_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_0_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.addOneNodeConfig(new NodeId(NODE_1_ID),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                            "test" + File.separatorChar + "tmp.ChainRepTest-" + 
			    NODE_1_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);

    Config.writeToFile(configPath);
  }

  private static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_0_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_1_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_0_ID), new OverlogId("localhost:5000"));
    NodeIdMap.add(new NodeId(NODE_1_ID), new OverlogId("localhost:5001"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
//  Converts values to byte arrays
//---------------------------------------------------------------------------

  private byte[] convertToArray(byte value){
    byte[] b = new byte[sizeOfWrites];
    for(int i = 0; i < sizeOfWrites; i++) {
      b[i] = value;
    }
    return b;
  }


//---------------------------------------------------------------------------
//  Writes a value to the specified interfac
//---------------------------------------------------------------------------

  private void write(ChainReplicationFSLocalInterface fsInterface, ObjId objId, byte[] b)
    throws IOException{
    fsInterface.write(objId, 0, sizeOfWrites, b);
  }

//---------------------------------------------------------------------------
// Reads an object from the specified interface
// If objNotFound -- sleeps for some time and tries again
//---------------------------------------------------------------------------

  private ImmutableBytes read(ChainReplicationFSLocalInterface fsInterface, ObjId objId) {
    ImmutableBytes b=null;
    try{
      while (true) {
	try{
          BodyMsg msg = fsInterface.read(objId, 0, sizeOfWrites);
	  if(msg != null) {
            b = msg.getBody();
          }
	  break;
	}catch(ObjNotFoundException e){
	  //wait until you recieve obj
	  Env.dprintln(dbg, "ObjNotFoundException for " + objId);
	  Thread.currentThread().sleep(timeToWait);
	}
      }
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    return b;
  }


//---------------------------------------------------------------------------
// makeWrites
//  - simply makes a bunch of writes
//---------------------------------------------------------------------------
  public void makeWrites() {
    try{ 
      byte[] b = convertToArray(value);
      for(int i = 0; i < numWrites; i++){
	Env.dprintln(dbg, "Write: " + i);
	ObjId obj = new ObjId("/" + i);
        write(node_interface, obj, b);
      }        
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
// makeReads
//  - simply makes a bunch of Reads to previously written values
//---------------------------------------------------------------------------
  
  public void makeReads() {
    try{
      ImmutableBytes imb = null;
      for(int i = 0; i < numWrites; i ++){
	Env.dprintln(dbg, "Read: " + i);
	ObjId obj = new ObjId("/" + i);
	imb = read(node_interface, obj);
      }
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }
//---------------------------------------------------------------------------
//  print results
//---------------------------------------------------------------------------
 
 public static void printResults(long writeDuration, long readDuration, String outFile) {
    try{
      PrintStream out = new PrintStream(new FileOutputStream(outFile, true)); 
      out.println(writeDuration/numWrites + "\t" + readDuration/numWrites);
      out.close();
      
      System.out.println("Average write time = " + writeDuration/numWrites);
      System.out.println("Average read time = " + readDuration/numWrites);

    }catch(Exception e) {
      e.printStackTrace();
    }
  }

//---------------------------------------------------------------------------
//  Main  Method
//---------------------------------------------------------------------------


  public static void main(String[] args) {
    try {
      Env.verifyAssertEnabled();
      
      if(args.length < 2) {
	System.out.println("Usage: java ChainReplicationExp1 <overlog file> <resultFile>");
	System.exit(0);
      }
      

      String olgfilename = args[0];
      String resfilename = args[1];

      ChainReplicationExp1 exp = new ChainReplicationExp1(olgfilename);
      exp.startNode();
      
      System.out.println("Waiting for overlog to start");
      exp.waitForMaster();
      System.out.println("Starting Test ..");
      
      System.out.println("Making writes");
      long writeStart = System.currentTimeMillis();
      exp.makeWrites();
      long writeEnd = System.currentTimeMillis();

      System.out.println("Making reads");
      long readStart = System.currentTimeMillis();
      exp.makeReads();
      long readEnd = System.currentTimeMillis();


      printResults((writeEnd-writeStart), (readEnd-readStart), resfilename);
      System.out.println("Finished Test ... ");
      
    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
    System.exit(0);
  }
  
}
//---------------------------------------------------------------------------
/* $Log: ChainReplicationExp1.java,v $
/* Revision 1.2  2007/03/15 00:50:14  nalini
/* exp1 checked in
/*
/* Revision 1.1  2007/03/12 03:37:07  nalini
/* Chain Replication exp1 files added
/*
 */
 /** 
