 /** 
 *  Local interface created for the CodaFS 
 **/ 
import java.util.HashMap;
import java.util.Arrays;
import java.util.Map;
import java.util.Iterator;
import java.io.IOException;
import java.io.EOFException;

public class CodaLocalInterface extends PRACTIFSLocalInterface{
  private static final boolean cooperativeCaching = true;
  private static final long READTIMEOUT = 100000; //1 sec

  private static final boolean dbg = false;
 /** 
 *  Data members 
 **/ 
  private boolean connected;//server connection status
  private CodaP2MailBox mailBox;

 /** 
 *  Constructor 
 **/ 
  public
    CodaLocalInterface(LocalInterface li, NodeId myNodeId){
    super(li, myNodeId);
    connected = true;
    mailBox = new CodaP2MailBox();
  }

 /** 
 *  Constructor 
 **/ 
  public
    CodaLocalInterface(PRACTIFSLocalInterface li, NodeId myNodeId){
    super(li, myNodeId);
    connected = true;
    mailBox = new CodaP2MailBox();
    
  }

 /** 
 * called by P2 tuple handler to sync the connection status to server 
 **/ 
  public void setConnectedStatus(boolean isconnected){
   // Env.dprintln(dbg, "CodaLocalInterface::setConnectedStatus " + isconnected); 
    connected = isconnected;
    if(!isconnected){
      mailBox.quitAll();//release all threads and throws Exceptions
    }else{
      mailBox.waitForReset();//wait until all threads are released 
      //then reset the quitAll
    }
    //Env.dprintln(dbg, "CodaLocalInterface::setConnectedStatus " + isconnected + " done");
  }

 /** 
 * called by P2 tuple writeFinished handler to notify write done. 
 **/ 
  public void writeFinished(AcceptStamp as){
    //Env.dprintln(dbg, "CodaLocalInterface:: writeFinished "+as.toString());
    mailBox.got(as);
    //Env.dprintln(dbg, "CodaLocalInterface:: writeFinished "+as.toString() + " done");
  }

 /** 
 *  Read from an object  
 *  Coda sepcific parameters: 
 *      blockValid = true; blockPrecise = true 
 *  
 *  tbd: if disconnected from server while read is blocked because of 
 *       imprecise or valid, it will hang till the timeout. 
 *       Throws ReadTimeoutException in this case 
 *         
 *  side effect: 
 *    1. disconnected and imprecise or invalid 
 *       ---> throw ReadOfInvalidCacheObjException 
 *    2. block imprecise /invalid timeout 
 *       ---> throw ReadTimeoutException 
 *    both exception are subclass of IOException and treated as IOException 
 *    in NFS interface. 
 **/ 
  public BodyMsg
    read(ObjId objId,
	 long offset,
	 long length)
  throws ObjNotFoundException, IOException,
	 EOFException, ReadOfHoleException{
    Env.dprintln(dbg, "CodaLocalInterface:: read "+ objId + " off=" + offset + " len= " + length);
    if(!connected && !cooperativeCaching){//disconnected and imprecise
      if((!this.li.isPrecise(objId))||(!this.li.isValid(objId, offset, length))){
	throw new ReadOfInvalidCacheException(objId);
      }
    }
    
    BodyMsg result = null;    
    try{
      result = this.li.read(objId,
                            offset,
                            length,
                            true,//blockValid
                            true,
			    READTIMEOUT);//blockPrecise
    }catch(ReadOfInvalidRangeException rire){
      assert false;//because of blockValid is true

    }catch(ReadTimeoutException rte){
      if (!connected){
	throw new ReadOfInvalidCacheException(objId);
      }else{
	Env.dprintln(dbg, rte.toString() + " \n -- have you set the readtimeout"
		     + " too short?");
	throw rte;
      }
    }
    //Env.dprintln(dbg, "CodaLocalInterface:: read "+ objId + " off=" + offset + " len= " + length + " done");
    return(result);
  }

 /** 
 *  overload the original method in PRACTIFS 
 **/ 
  public long write(ObjId objId, long offset, long length, byte buffer[])
    throws IOException {
    Env.dprintln(dbg, "CodaLocalInterface:: write "
		 + objId + " off=" + offset + " len= " + length);
    PreciseInv resultSummary = null;
    resultSummary=li.preWrite(objId, offset, length, new ImmutableBytes(buffer), true, 1L);//bound = true, maxHop =1
    if(connected){
      try{
	mailBox.waitFor(resultSummary.getAcceptStamp());
      }catch(ServerDisconnectedException e){
	//ignore
      }
    }
    Env.dprintln(dbg, "CodaLocalInterface:: write done "+ objId 
		 + " off=" + offset + " len= " + length + " done");
    return resultSummary.getLength();
  }

  /*
    defer
  //-----------------------------------------------------------------------
  // Version of the write() method that allows writing to multiple
  // object atomically
  //-----------------------------------------------------------------------
  protected long[]
  writeMulti(MultiWriteEntry[] mwe, boolean embargoed)
    throws IOException{
    MultiObjPreciseInv result = null;
    Env.dprintln(dbgLock||dbg, "CodaLocalInterface::writeMulti");
    // Pass the call onto the real LocalInterface
    result = this.li.preWriteMulti(mwe, embargoed);
    
    if(connected){
      try{
	mailBox.waitFor(writeResult.getAcceptStamp());
      }catch(ServerDisconnectedException e){
	//ignore
      }
    }
    Env.dprintln(dbgLock||dbg, "return CodaLocalInterface::writeMulti");
    return result.getLength(); 
  }
  */
  
  
 /** 
 *  write 
 *  need to know the timestamp so that it knows when to wait 
 **/ 
  /*
  //-----------------------------------------------------------------------
  // Used for testing
  //-----------------------------------------------------------------------
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing CodaLocalInterface.java...");
    CodaLocalInterface.testSimple();
    System.out.println("...CodaLocalInterface.java");
    System.exit(0);
  }

  //-----------------------------------------------------------------------
  // Used for testing
  //-----------------------------------------------------------------------
  private static void
  testSimple(){
    CodaLocalInterface li = null;
    FakePRACTIFSLocalInterface fakeLI = null;
    PRACTIFSReadLockToken readToken = null;
    PRACTIFSWriteLockToken writeToken = null;

    fakeLI = new FakePRACTIFSLocalInterface(new NodeId(1));
    li = new CodaLocalInterface(fakeLI, new NodeId(1));
    try{
      // Try one objId
      readToken = li.acquireReadLock(new ObjId("a"));
      assert(readToken.getObjId().equals(new ObjId("a")));
      assert(readToken.isAcquired());
      li.releaseReadLock(readToken, new ObjId("a"));
      assert(!readToken.isAcquired());
      writeToken = li.acquireWriteLock(new ObjId("a"));
      assert(writeToken.getObjId().equals(new ObjId("a")));
      assert(writeToken.isAcquired());
      li.releaseWriteLock(writeToken, new ObjId("a"));
      assert(!writeToken.isAcquired());

      // Try two objIds
      readToken = li.acquireReadLock(new ObjId("a"));
      assert(readToken.getObjId().equals(new ObjId("a")));
      assert(readToken.isAcquired());
      writeToken = li.acquireWriteLock(new ObjId("b"));
      assert(writeToken.getObjId().equals(new ObjId("b")));
      assert(writeToken.isAcquired());
      li.releaseWriteLock(writeToken, new ObjId("b"));
      li.releaseReadLock(readToken, new ObjId("a"));
    }catch(InterruptedException e){
      System.err.println("" + e);
    }
  }
*/
}
 /** 
/* $Log: CodaLocalInterface.java,v $
/* Revision 1.6  2007/10/19 23:08:40  zjiandan
/* Coda demo for SOSP poster 2007.
/*
/* Revision 1.5  2007/10/03 20:26:00  zjiandan
/* Cooperative cache.
/*
/* Revision 1.4  2007/10/02 20:10:03  zjiandan
/*  Coda with recovery of server or client tested
/*
/* Revision 1.2  2007/03/13 08:15:24  zjiandan
/* Coda Tested with S-C1-C2 scenarios as described in CodaTest.olg.
/*
/* Revision 1.1  2007/03/12 03:07:26  zjiandan
/* Coda Localinterface.
/**/
 **/ 
