 /** 
/* PangaeaFS.java
 *
 * NFS interface wrapper for Pangaea to implement the java version of the
 * NFS loopback server
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;
import java.util.*;

public class PangaeaFS extends PRACTIFS{
  final static int GOLDNUM = 3;
  private final static boolean dbg = false;
  //private static final boolean PRINT_METHODS = false;
  private static final int TOTALREGION = GOLDNUM;
  private static final int REGIONSIZE = 1;
  private static boolean warnedP2InteractionInefficiency = false;
  private static boolean turnOnOP = true;
  
  private static boolean dbgPerformance = true;

  OverlogPolicy op;
  //PangaeaP2Synchronizer pps; //see comments for createDirEntry()
  HashSet objCache;

 /** 
 *  construction: 
 *     if this node is primary, create root directory 
 *     otherwise do not need to create root, the p2 rules will  
 *     automatically set up the subscription for the root directory 
 *   
 *     need to traverse the entire file system to recover the 
 *     connections for valid replicas 
 **/ 
  public PangaeaFS(PangaeaLocalInterface li, OverlogPolicy op, boolean primary){
    super(li, primary);
    this.objCache = new HashSet();
    Env.dprintln(dbg, "creating PRACTIFSLocalInterface done");
    this.op = op;
    //pps = new PangaeaP2Synchronizer(op);//see comments for createDirEntry()
    /*
    try{
      recover(new ObjId(super.ROOT_DIR_NAME), null);
    }catch(IOException e){
      e.printStackTrace();
      System.exit(-1);
    }
    Env.dprintln(dbg, "creating PangaeaFS done");
    */
  }


 /** 
 *  recover --- 
 *     traverse the entire file system starting from root 
 *  
 *  for each valid obj/dir 
 *     insert goldNodes tuples to p2 
 *     insert createNewReplica tuple to p2 
 *     cache.put(obj) 
 *  
 *  note: for root dir, the goldNodes are maintained in p2 and whenever a 
 *        node starts, the connections are initiated. 
 **/ 
  /*
  void recover(ObjId dirPrefix, 
	       HashSet goldNodes)
    throws IOException, EOFException{
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    PRACTIFSDirectory dir = null;
    PRACTIFSDirectory.DirIterator it = null;
    String dirEntryName = null;
    PRACTIFSDirEntry dirEntry = null;
    dirMetaObjId = new ObjId(dirPrefix.getPath() + super.META_SUFFIX);
    dirDataObjId = new ObjId(dirPrefix.getPath() + super.DATA_SUFFIX);
    try{
      if (super.localInterface.isValid(dirMetaObjId, 0, 1)
	  &&super.localInterface.isValid(dirDataObjId, 0, 1)){
	if(!dirPrefix.equals(new ObjId(super.ROOT_DIR_NAME))){
	  //root's connection is automatically setted
	  // by p2 rules
	  //
	  // this was a valid cache
	  // tell p2 to maintain the connections with m neighbors
	  // add it to the cache table
	  //
	  String objIdStr = dirPrefix.toString();
	  if(OverlogPolicy.P2VersionHasStringProblem){
	    assert !dirPrefix.toString().contains("-") :
	      "\"-\" is not allowed in ObjId because we are using the p2 version"
	      + "\nthat can't interpret \"/\" well and have to hack by replacing \"/\" with"
	      + "\n \"-\"."
	      + " If you are using a p2 version works with \"/\", "
	      + "turn off OverlogPolicy.P2VersionHasStringProblem";
	      
	      objIdStr = dirPrefix.toString().replaceAll("/", "-");
	  }
	  
	  assert goldNodes!= null:dirPrefix.toString();
	  // handle meta object
	  String metaObjIdStr = objIdStr + super.META_SUFFIX;
	  tellP2CreateNewReplica(metaObjIdStr, goldNodes);
	  // handle data object
	  String dataObjIdStr = objIdStr + super.DATA_SUFFIX;
	  tellP2CreateNewReplica(dataObjIdStr, goldNodes);
	  
	  objCache.add(dirPrefix);
	  Env.dprintln(dbg, "1 add to cache:" + dirPrefix.toString());
	}
	dir = this.readDir(null, dirDataObjId);
      }
    }catch(ObjNotFoundException e){
      Env.dprintln(dbg, "Warning: " + dirPrefix + " not found.");
      //cache doesn't exist, just ignore
      return;
    }catch(IsNotDirectoryException e){
      Env.dprintln(dbg, "warning: " + dirPrefix + " is not a directory.");
      //cache doesn't exist, just ignore
      return;
    }
    
    // if this object is a directory
    // iterate through the directory
    it = dir.getIterator();
    while(it.hasNext()){
      dirEntryName = it.getNext();
      try{
	dirEntry = dir.getEntry(dirEntryName);
      }catch(ObjNotInDirectoryException e){
	assert false;
      }
      recover(dirEntry.getObjIdPrefix(), (HashSet)dirEntry.getOtherItem());
    }
    
  }

  */
  
 /** 
 * infom that p2 has inserted a GoldNode tuple 
 **/ 
  public void informGoldNode(ObjId objId, String goldNodeOverlogId){
    Env.dprintln(PRINT_METHODS, "informGoldNode called");
    assert false;
    /*
   see comments for createDirEntry()
     try{
       pps.informGoldNode(objId, goldNodeOverlogId);
     } catch (UnexpectedTupleFromP2Exception e){
       Env.warn(e.toString());//ignore.
     }
    */  
  }
  
 
 /** 
 * infom that p2 has inserted a GoldNode tuple 
 **/ 
  public void replaceDeadGold(ObjId objId, NodeId deadGold, NodeId newGold){
    Env.dprintln(PRINT_METHODS, "replaceDeadGold called");
    assert false;
    //defer
  }

 /** 
 *  Implement the GETATTR NFS method 
 **/ 
  public attrstat
  getAttributes(fhandle fh){
    return super.getAttributes(fh);
  }

 /** 
 *  Implement the SETATTR NFS method 
 *  NOTE: This method does not currently allow increasing file sizes. 
 *  NOTE: We always set the time to the server time. Need to change this in 
 *  the future 
 *  Note: This method currently ignores the guard parameter. In the future 
 *  it should not do so. 
 *  Note: This method does not return attributes at all. Might want to 
 *  change this in the future. 
 **/ 
  public attrstat
  setAttributes(sattrargs args){
    return super.setAttributes(args);
  }

 /** 
 *  Implement the LOOKUP NFS method 
 *  
 *  NOTE: This method does not currently check whether the file being 
 *    opened is really a directory (by checking its attributes). It only 
 *    does a simple test of trying to open it as a directory and 
 *    complaining if it fails. 
 **/ 
  public diropres
  lookup(diropargs args){
    Env.dprintln(super.PRINT_METHODS, "PRACTIFS lookup " + args.getDir() + " " + args.getName());
    long start = System.currentTimeMillis();
    long markStart = start;
    long end;
    long totalTime=0;
    int stage = 0;
    ObjId dirPrefix = null;
    ObjId objIdPrefix = null;
    ObjId dirDataObjId = null;
    ObjId dirMetaObjId = null;
    ObjId objMetaObjId = null;
    PRACTIFSReadLockToken dirMetaToken = null;
    PRACTIFSReadLockToken dirDataToken = null;
    PRACTIFSReadLockToken objMetaToken = null;
    PRACTIFSDirectory dir = null;
    ObjId filePrefix = null;
    fhandle objHandle = null;
    diropres result = null;
    PRACTIFSFileAttributes objAttr = null;
    PRACTIFSFileAttributes dirAttr = null;

    try{
      dirPrefix = this.fhCache.getObjIdPrefix(args.getDir());
      dirMetaObjId = new ObjId(dirPrefix.getPath() + META_SUFFIX);
      dirDataObjId = new ObjId(dirPrefix.getPath() + DATA_SUFFIX);
      dirMetaToken = this.localInterface.acquireReadLock(dirMetaObjId);
      dirDataToken = this.localInterface.acquireReadLock(dirDataObjId);
     
      end = System.currentTimeMillis();
      totalTime = (end-start);
      start = end;
      stage++;
      if(totalTime > super.PRINT_TIME_FILTER ){
	Env.dprintln(dbgPerformance, "==== lookup stage " +stage 
		     + "  "+ args.getDir() + " " + args.getName()
		     + " took "+ (end-start) + "ms");
      }

      dirAttr = this.readAttributes(dirMetaToken, dirMetaObjId);

      end = System.currentTimeMillis();
      totalTime = (end-start);
      start = end;
      stage++;
      if(totalTime > super.PRINT_TIME_FILTER ){
	Env.dprintln(dbgPerformance, "==== lookup stage " +stage 
		     + " readDirAttributes "+ args.getDir() + " " + args.getName()
		     + " took "+totalTime + "ms");
      }

      if(dirAttr.getType() != ftype.NFDIR){
        throw new IsNotDirectoryException("" + dirPrefix + " not a directory");
      }
      dir = this.readDir(dirDataToken, dirDataObjId);
      
      end = System.currentTimeMillis();
      totalTime = (end-start);
      start = end;
      stage++;
      if(totalTime > super.PRINT_TIME_FILTER ){
	Env.dprintln(dbgPerformance, "==== lookup stage " +stage 
		     + " readDirData "+ args.getDir() + " " + args.getName()
		     + " took "+totalTime + "ms");
      }

      objIdPrefix = dir.getEntry(args.getName()).getObjIdPrefix();
      objMetaObjId = new ObjId(objIdPrefix.getPath() + META_SUFFIX);
      objMetaToken = this.localInterface.acquireReadLock(objMetaObjId);
      objHandle = this.fhCache.getFileHandle(objIdPrefix);
      //Pangaea Specific code - start
      
      end = System.currentTimeMillis();
      totalTime = (end-start);
      start = end;
      stage++;
      if(totalTime > super.PRINT_TIME_FILTER ){
	Env.dprintln(dbgPerformance, "==== lookup stage " +stage 
		     + " before readData "+ args.getDir() + " " + args.getName()
		     + " took "+totalTime + "ms");
      }

      if(!objCache.contains(objIdPrefix)){
        //
        // The file exists in the directory file,
        // but the corresponding body has not arrived yet.
        // This means we need to create a new replica for this file.
        //
        String objIdStr = objIdPrefix.toString();
        
	if(OverlogPolicy.P2VersionHasStringProblem){
	  assert !objIdPrefix.toString().contains("-") :
	    "\"-\" is not allowed in ObjId because we are using the p2 version"
	    + "\nthat can't interpret \"/\" well and have to hack by replacing \"/\" with"
	    + "\n \"-\"."
	    + " If you are using a p2 version works with \"/\", "
	      +"turn off OverlogPolicy.P2VersionHasStringProblem";
	    
	    objIdStr = objIdPrefix.toString().replaceAll("/", "-");
	}
       
	HashSet goldNodes = (HashSet) dir.getEntry(args.getName()).getOtherItem();
	
	// handle meta object
	String metaObjIdStr = objIdStr + super.META_SUFFIX;
        tellP2CreateNewReplica(this.op, metaObjIdStr, goldNodes);
	// handle data object
	String dataObjIdStr = objIdStr + super.DATA_SUFFIX;
	tellP2CreateNewReplica(this.op, dataObjIdStr, goldNodes);
	
	objCache.add(objIdPrefix);
	Env.dprintln(dbg, "2 add to cache:" + objIdPrefix.toString());
      
      }
      
      //Pangaea Specific code - end
      end = System.currentTimeMillis();
      totalTime = (end-start);
      start = end;
      stage++;
      if(totalTime > super.PRINT_TIME_FILTER ){
	Env.dprintln(dbgPerformance, "==== lookup stage " +stage 
		     + " insertCreateNewReplicaTupleToP2 "+ args.getDir() + " " + args.getName()
		     + " took "+totalTime + "ms");
      }

      
      //readstale and block if no bodyExistException
      objAttr = this.readAttributes(objMetaToken, objMetaObjId);

      end = System.currentTimeMillis();
      totalTime = (end-start);
      start = end;
      stage++;
      if(totalTime > super.PRINT_TIME_FILTER ){
	Env.dprintln(dbgPerformance, "==== lookup stage " +stage 
		     + " readFileMeta "+ args.getDir() + " " + args.getName()
		     + " took "+totalTime + "ms");
      }

      //objAttr = this.readAttributes(objMetaToken, objMetaObjId);
      result = new diropres(stat.NFS_OK, objHandle, objAttr);
      
    }catch(BadFileHandleException e){
      // Bad directory handle
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(IsNotDirectoryException e){
      // This object is not a directory
      result = new diropres(stat.NFSERR_NOTDIR, null, null);
    }catch(FileNotFoundException e){
      // Handle for directory exists, but directory doesn't
      result = new diropres(stat.NFSERR_STALE, null, null);
    }catch(ObjNotInDirectoryException e){
      // Directory has no problem; the file just doesn't exist
      result = new diropres(stat.NFSERR_NOENT, null, null);
    }catch(IOException e){
      // Don't know what caused this; generic IO exception
      result = new diropres(stat.NFSERR_IO, null, null);
    }catch(InterruptedException e){
      // This came from trying to acquire locks
      assert false : ("" + e);
    }finally{
      if((objMetaObjId != null) && (objMetaToken != null)){
        this.localInterface.releaseReadLock(objMetaToken, objMetaObjId);
      }
      if((dirDataObjId != null) && (dirDataToken != null)){
        this.localInterface.releaseReadLock(dirDataToken, dirDataObjId);
      }
      if((dirMetaObjId != null) && (dirMetaToken != null)){
        this.localInterface.releaseReadLock(dirMetaToken, dirMetaObjId);
      }
    }

    Env.dprintln(PRINT_METHODS, ".........PRACTIFS lookup " + args.getDir() + " " + args.getName()
      + " done.");

    end = System.currentTimeMillis();
    totalTime = end-markStart; //total for lookup
    if(totalTime > super.PRINT_TIME_FILTER ){
      Env.dprintln(PRINT_TIME, "........PRACTIFS lookup " + args.getDir() + " " + args.getName()
		   + " took "+totalTime + "ms");
    }
    return(result);
  }

 /** 
 *  Implement the READLINK NFS method 
 *  
 *  NOTE: This method does not currently correctly check whether the object 
 *    is in fact a symbolic link. It should check with the file attributes. 
 **/ 
  public readlinkres
  readlink(fhandle fh){
    return super.readlink(fh);
  }  

 /** 
 *  Implement the READ NFS method 
 *  
 *  NOTE: This method does not check whether the read is to a real file 
 *    or not. Need to change this in the future. 
 *  Note: This method makes a copy of the data bytes. This could be made  
 *    more efficient in the future (though it requires handling with care). 
 **/ 
  public readres
  read(readargs args){
    return super.read(args);
  }

  
 /** 
 *  Implement the WRITE NFS method 
 *  
 *  NOTE: This method does not change any attributes except the size; 
 *    need to change this 
 *  NOTE: This method does not check whether the write is to a real file 
 *    or not. Need to change this in the future. 
 *  NOTE: This method currently ignores the stability flag and makes all 
 *    writes DATA_SYNC stable 
 *  Note: This method makes a copy of the data bytes. This could be made  
 *    more efficient in the future (though it requires handling with care). 
 **/ 
  public attrstat
  write(fhandle file,
        long beginOffset,
        long offset,
        long totalCount,
        byte[] data){

   return super.write(file, beginOffset, offset, totalCount, data);
  }

 /** 
 *  Implement the CREATE NFS method 
 *  
 *  NOTE: This method does not check whether the write is to an existing 
 *    file or not. Need to change this in the future. 
 *  NOTE: This method does not work for the EXCLUSIVE createhow switch 
 *  NOTE: This method does not change the directory attributes timestamp 
 *  NOTE: The newly created file attributes give everyone permission to 
 *    do everything 
 **/ 
  public diropres
  create(createargs args){
    return super.create(args);
  }

  
 /** 
 *  Implement the REMOVE NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public int
  remove(diropargs args){
    return super.remove(args);
  }

 /** 
 *  Implement the RENAME NFS method 
 *  
 *  NOTE: This method does not change the directory attributes timestamp 
 **/ 
  public int
  rename(renameargs args){
    return super.rename(args);
    //issues: how to make existing subscriptionSet to adapt to receive
    //        the new object since in our current implementation
    //        the new object does not change its id.
    // for example: 
    //      We have two dirs: /a, /b. originally file foo is in /a,
    //      and node B is interested in /b/*. when rename /a/foo to /b/foo.
    //      then the subscriptionset /b/* will not bring file foo.
    //      Now when we read /b/foo, node B will add /a/foo into the subscriptionset.
    // 
    // two options:
    // 1. reimplement the rename so that the names match the ids, downside: preformance
    // 2. Assume rename is rare and leave the subscriptionset as it is. 
    // 
  }

  //-------------------------------------------------------------------------
  // Implement the MKDIR NFS method
  //
  // NOTE: This method does not check whether the write is to an existing
  //   file or not. Need to change this in the future.
  // NOTE: This method does not change the directory attributes timestamp
  // NOTE: The newly created directory attributes give everyone permission to
  //   do everything
  //-------------------------------------------------------------------------
  public diropres
  mkdir(createargs args){
    return super.mkdir(args);
  }

  //-------------------------------------------------------------------------
  // Implement the RMDIR NFS method
  //
  // NOTE: This method does not change the directory attributes timestamp
  //-------------------------------------------------------------------------
  public int
  rmdir(diropargs args){
    return super.rmdir(args);
  }

  //-------------------------------------------------------------------------
  // Implement the READDIR NFS method
  //
  // NOTE: This method isn't currently implemented
  // NOTE: We currently ignore the cookieverf token
  // NOTE: We currently ignore the usage of cookies (cookies must be 0)
  //-------------------------------------------------------------------------
  public readdirres
  readdir(readdirargs args){
    return super.readdir(args);
  }

  //----------------------------------------------------------------------
  // overload super's createDirEntry
  //
  //----------------------------------------------------------------------
  protected PRACTIFSDirEntry createDirEntry(ObjId objPrefix){

    if(!warnedP2InteractionInefficiency){
      Env.performanceWarning("Because for each Pangaea Object, we have two\n"
			     + " \t PRACTI native objects(meta and data), and there's no \n"
			     + " \t easy way to match the actual objects with Pangaea objects\n"
			     + " \t in overlog (especially considering the inform methods),\n"
			     + " \t therefore, we assume all objects in overlog\n"
			     + " \t is a PRACTI native object. This could have some performance penalty:\n"
			     + " \t When create a new Pangaea object, we need to ask P2 twice to return goldNodes\n"
			     + " \t for the meta object and for the data object. What's more, there's chance that\n"
			     + " \t the goldNodes for meta object might be different from data object, then what \n"
			     + " \t should we put in the directory entry for the goldNodes? \n"
			     + " \t currently we just have it retry until the goldNodes are the same. ");
      warnedP2InteractionInefficiency = true;
    }
    HashSet goldNodesForMeta = null;
    HashSet goldNodesForData = null;
    ObjId objMetaObjId = null;
    ObjId objDataObjId = null;
    assert objPrefix != null;
    
    /*
      //replace by method HashSet getGoldNodes(obj, goldnum).
      
      //ideally the policy for getting the initial goldNodes should located in p2 layer.
      //but currently the p2 has limited functionality to generate the goldnodes needed.
      //We fall back to write the policy in java and insert the goldNodes to the p2 layer
      //after inserting the createNewObj tuple

    objMetaObjId = new ObjId(objPrefix.getPath() + META_SUFFIX);
    objDataObjId = new ObjId(objPrefix.getPath() + DATA_SUFFIX);
    try{
      goldNodesForMeta = pps.getGoldNodesFromP2(objMetaObjId, GOLDNUM);
    }catch(SyncRequestAlreadyExistsException e){
      //someone else has already create this obj
      //shouldn't happen since we should have get the lock for dir
      assert false;
    }
    assert goldNodesForMeta.size() == GOLDNUM;

    try{
      goldNodesForData = pps.getGoldNodesFromP2(objDataObjId, GOLDNUM);
    }catch(SyncRequestAlreadyExistsException e){
      //someone else has already create this obj
      //shouldn't happen since we should have get the lock for dir
      assert false;
    }
    assert goldNodesForData.size() == GOLDNUM;
    for(Iterator iter = goldNodesForData.iterator(); iter.hasNext();){
	assert goldNodesForMeta.contains(iter.next());
    }
    */

    String objIdStr = objPrefix.toString();
    if(OverlogPolicy.P2VersionHasStringProblem){
      assert !objPrefix.toString().contains("-") :
	"\"-\" is not allowed in ObjId because we are using the p2 version"
	+ "\nthat can't interpret \"/\" well and have to hack by replacing \"/\" with"
	+ "\n \"-\"."
	+ " If you are using a p2 version works with \"/\", "
	+ "turn off OverlogPolicy.P2VersionHasStringProblem";
	
	objIdStr = objPrefix.toString().replaceAll("/", "-");
    }
    
    // handle meta object
    String metaObjIdStr = objIdStr + super.META_SUFFIX;
    // handle data object
    String dataObjIdStr = objIdStr + super.DATA_SUFFIX;
 

    HashSet goldNodes = null;
    goldNodes = getGoldNodes(this.op, objPrefix, GOLDNUM);
    assert goldNodes != null;
    assert goldNodes.size() == GOLDNUM;

    //
    // it is possible that the objPrefix already exist
    // but moved to different dir.
    // 
    if(!objCache.contains(objPrefix)){
      objCache.add(objPrefix);
      Env.dprintln(dbg, "3 add to cache:" + objPrefix.toString());
      tellP2CreateNewObj(this.op, metaObjIdStr, goldNodes);
      tellP2CreateNewObj(this.op, dataObjIdStr, goldNodes);
    }
    return new PRACTIFSDirEntry(objPrefix, goldNodes);
  }
  

 /** 
 *  Generate the gold node for the new object 
 *  by default: 
 *     <ME, randomly pick one from different regions> 
 *     suppose each region has r nodes. region number start from 0. 
 *     region i: { r*i, r*i + 1, ..., r*i + (r-1)} 
 **/ 
  public static HashSet getGoldNodes(OverlogPolicy theop, ObjId objPrefix, int total){
    assert total >= 1;
    
    int count = 0;
    int regionCount = 0;
    HashSet ret = new HashSet();
    Random r = new Random(1977);
    
    assert total <= TOTALREGION;
    //ret.add(theop.getMyOverlogId().toString());//myself is one of the gold nodes
    //count++;
    int myNodeId = (int)(NodeIdMap.getNodeId(theop.getMyOverlogId()).getIDint());
    
    int myRegion = myNodeId/REGIONSIZE;
    //System.out.println("goldNode " + myNodeId + " " + op.getMyOverlogId().toString() 
    //+ " region:" + myRegion); 
    NodeId goldnode;
    while(count < total){
      
      if(regionCount != myRegion){
	int pick = r.nextInt(REGIONSIZE);
	
	goldnode = new NodeId(regionCount*REGIONSIZE + pick);
	//System.out.println("pick: " + pick + " regionCount " + regionCount + "goldNode " + goldnode); 
	ret.add(NodeIdMap.getOverlogId(goldnode).toString());
	count++;
      }
      regionCount++;
    }
    assert ret.size() == total;
    return ret;
  }

 /** 
 *  Create a new file attribute for an object 
 *  by default there's no additional item in the entry except the  
 *  standard nfs attributes. 
 *  Note: dirPrefix is the parent of the file 
 *        it is reserved here to provide enough information for customizing 
 *        FileAttributes for different file systems. 
 *        if dirPrefix = null, then this fileAttributes belong to the root 
 **/ 
  protected PRACTIFSFileAttributes createFileAttributes(sattr newAttr, 
                                                        long fileType,
                                                        long size,
                                                        long numBlocks,
                                                        long myfileid,
                                                        ObjId dirPrefix){
    PRACTIFSFileAttributes ret = null;
    long now = System.currentTimeMillis();
    if (newAttr == null){
      ret = new PRACTIFSFileAttributes(fileType,
                                       0x041FF, // Giva all full permission
                                       1, // Only one link to this file so far
                                       0, // uid
                                       0, // gid
                                       size, // size
                                       FS_BLOCK_SIZE, // blocksize
                                       0, // rdev
                                       numBlocks, // blocks
                                       PRACTIFS.FSID, // fsid
                                       myfileid, // fileid
                                       new timeval(now), // atime
                                       new timeval(now), // mtime
                                       new timeval(now), // ctime
                                       dirPrefix);//otherAttributes if any
    }else{
      ret = new PRACTIFSFileAttributes(newAttr,
                                       fileType,
                                       0x041FF, // Giva all full permission
                                       1, // Only one link to this file so far
                                       0, // uid
                                       0, // gid
                                       size, // size
                                       FS_BLOCK_SIZE, // blocksize
                                       0, // rdev
                                       numBlocks, // blocks
                                       PRACTIFS.FSID, // fsid
                                       myfileid, // fileid
                                       new timeval(now), // atime
                                       new timeval(now), // mtime
                                       new timeval(now), // ctime
                                       dirPrefix);//otherAttributes if any
    }
      return ret;
  }

 /** 
 *  insert createNewReplica tuple to p2 
 *  note:  
 *     we insert goldNodes first then the createNewReplica tuple to avoid 
 *     the periodic check of the number of goldNodes. 
 *      
 **/ 
  public static void tellP2CreateNewReplica(OverlogPolicy theop, String objIdStr, HashSet goldNodes){
      Tuple tp = null;
      //insert the goldNodes
      
      assert goldNodes != null;
      //assert goldNodes.size() == GOLDNUM;
      for(Iterator iter = goldNodes.iterator(); iter.hasNext();){
	String goldId = (String) iter.next();
	String[] goldStr = {"goldNodes", 
			    theop.getMyOverlogId().toString(),
			    objIdStr,
			    goldId};
	tp = new Tuple(goldStr);
	Env.dprintln(PRINT_METHODS, "PangaeaFS.looup::insert tuple:" + tp); 
	if(turnOnOP){
	  theop.insertTuple(tp);
	}
      }
      
      String[] tpStr = {"createNewReplica", 
			theop.getMyOverlogId().toString(),
			objIdStr};
      tp = new Tuple(tpStr);
      Env.dprintln(PRINT_METHODS, "PangaeaFS.looup::insert tuple:" + tp); 
      if(turnOnOP){
	Env.dprintln(PRINT_METHODS, "PangaeaFS.looup::insert tuple:" + tp + " @ " + System.currentTimeMillis()); 
	theop.insertTuple(tp);
	Env.dprintln(PRINT_METHODS, "PangaeaFS.looup::insert tuple:" + tp + " return @ " + System.currentTimeMillis()); 
      }
    }

 /** 
 *  Used for testing 
 **/ 
  public static void tellP2CreateNewObj(OverlogPolicy theop, String objIdStr, HashSet goldNodes){
      
    //insert the goldNodes
    assert goldNodes != null;
    //assert goldNodes.size() == GOLDNUM;
    for(Iterator iter = goldNodes.iterator(); iter.hasNext();){
      String goldId = (String) iter.next();
      String[] goldStr = {"createNewObjGold", 
			  theop.getMyOverlogId().toString(),
			  objIdStr,
			  goldId};
      Tuple tp = new Tuple(goldStr);
      Env.dprintln(PRINT_METHODS, "PangaeaFS insert createNewObj tuple:" + tp); 
      if(turnOnOP){
	theop.insertTuple(tp);
      }
    }
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PangaeaFS.java...");
    PangaeaFS.testSimple();
    PangaeaFS.testLiveness();
    System.out.println("...Finished");
    System.exit(0);
  }
  
 /** 
 *  Test liveness 
 *  
 *    create image: 
 *       /root(P/V)/a(P/V)/a1(P/V)   
 *                        /a2(P/IV) 
 *                        /a3(IP/V) 
 *                        /a4(IP/IV) 
 *                 /b(P/IV)/b1(P/V) 
 *                         /b2(P/IV) 
 *                         /b3(IP/V) 
 *                         /b4(IP/IV) 
 *                 /c(IP/V)/c1(P/V) 
 *                         /c2(P/IV) 
 *                         /c3(IP/V) 
 *                         /c4(IP/IV) 
 *                 /d(IP/IV)/d1(P/V) 
 *                          /d2(P/IV) 
 *                          /d3(IP/V) 
 *                          /d4(IP/IV) 
 *  
 **/ 
  private static void
  testLiveness(){
    boolean withP2 = false;
    Env.dprintln(dbg, " PangaeaFS liveness with P2: " + withP2);
    PangaeaFS pfs = null;
    OverlogPolicy op;
    P2Runtime runtime;
    Core core;
    PangaeaLocalInterface li;

    boolean cleanDb;
    boolean noSyncLog;
    fhandle handle = null;
    fhandle handle2 = null;
    String configFile = "test" + File.separatorChar + "pfs.unittest.config";
    String p2ConfigFile = "test" + File.separatorChar + "pfs.unittest.p2config";
    NodeId myNodeId = new NodeId(2);
    String overlogFile = "test" + File.separatorChar + "pfs.unittest.olg";
    String nodeMapFile = "test" + File.separatorChar + "pfs.unittest.map";
    
    //members in URAOverlogNode
    if(!withP2){
      op = new TestOverlogPolicy(overlogFile, myNodeId, nodeMapFile);
    }else{
      op = new OverlogPolicy(overlogFile, myNodeId, nodeMapFile);
    }
    
    cleanDb = true;
    noSyncLog = false;
    runtime = new P2Runtime(configFile, p2ConfigFile, myNodeId, cleanDb, op, noSyncLog);
    core = runtime.getCore();
    li = new PangaeaLocalInterface(new LocalInterface(runtime.getController(), core), 
				   myNodeId);
    pfs = new PangaeaFS(li, op, true);
    
    assert false; //tbd
    
    
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    PangaeaLocalInterface li = null;
    PangaeaFS pfs = null;
    fhandle handle = null;
    fhandle handle2 = null;
    String configFile = "test" + File.separatorChar + "pfs.unittest.config";
    String p2ConfigFile = "test" + File.separatorChar + "pfs.unittest.p2config";
    NodeId myNodeId = new NodeId(1);
    String overlogFile = "test" + File.separatorChar + "pfs.unittest.olg";
    String nodeMapFile = "test" + File.separatorChar + "pfs.unittest.map";

    PangaeaURAOverlogNode uraNode = new PangaeaURAOverlogNode(configFile,
                                                              p2ConfigFile,
                                                              myNodeId,
                                                              true,
                                                              overlogFile,
                                                              nodeMapFile,
							      false);
    pfs = uraNode.getFS();

    //System.out.println("" + li);
    PangaeaFS.testCreate( pfs, fhandle.makeAllZeroHandle(), "a", stat.NFS_OK);
    handle = PangaeaFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle == null);
    handle = PangaeaFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "a");
    assert(handle != null);
    handle2 = PangaeaFS.testLookup( pfs, handle, "b");
    assert(handle2 == null);
    
    PangaeaFS.testGetAttributes( pfs, handle, ftype.NFREG);
    PangaeaFS.testGetAttributes( pfs, fhandle.makeAllZeroHandle(), ftype.NFDIR);
    PangaeaFS.testWrite( pfs, (byte)65, 0, 50, handle);
    PangaeaFS.testWrite( pfs, (byte)66, 50, 50, handle);
    PangaeaFS.testGetAttributes( pfs, handle,  ftype.NFREG);
    PangaeaFS.testRead( pfs, handle, 50, 50, (byte)66);
    PangaeaFS.testRead( pfs, handle, 0, 50, (byte)65);
    String[] expectedNames = new String[1];
    expectedNames[0] = "a";
    PangaeaFS.testReadDir( pfs, fhandle.makeAllZeroHandle(), expectedNames);
    PangaeaFS.testRename(
                        pfs,
                        fhandle.makeAllZeroHandle(),
                        "a",
                        fhandle.makeAllZeroHandle(),
                        "b");
    
    handle = PangaeaFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "a");
    assert(handle == null);
    
    expectedNames[0] = "b";
    PangaeaFS.testReadDir( pfs, fhandle.makeAllZeroHandle(), expectedNames);

    handle = PangaeaFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle != null);
    
    PangaeaFS.testMkdir( pfs, fhandle.makeAllZeroHandle(), "subDir");
    handle = PangaeaFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "subDir");
    assert(handle != null);
    
    expectedNames = new String[2];
    expectedNames[0] = "b";
    expectedNames[1] = "subDir";
    PangaeaFS.testReadDir( pfs, fhandle.makeAllZeroHandle(), expectedNames);

    expectedNames = new String[0];
    PangaeaFS.testReadDir( pfs, handle, expectedNames);//read subDir

    
    PangaeaFS.testRename( pfs, fhandle.makeAllZeroHandle(), "b", handle, "newB");
    handle2 = PangaeaFS.testLookup( pfs, handle, "newB");
    assert(handle2 != null);

    expectedNames = new String[1];
    expectedNames[0] = "newB";
    PangaeaFS.testReadDir( pfs, handle, expectedNames);//read subDir

    handle2 = PangaeaFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle2 == null);
    PangaeaFS.testSetAttributes( pfs, handle);
    try{
      uraNode.shutdown();
      Thread.sleep(2000);
    }catch(Exception e){
      e.printStackTrace();
    }
    
    
  }

 /** 
 *  Test the create method 
 **/ 
  private static void
    testCreate(PangaeaFS pfs, fhandle fh, String name, int status){
    createargs args = null;
    diropres result = null;
    diropargs dirOpArgs = null;
    sattr attr = null;

    dirOpArgs = new diropargs(fh, name);
    attr = new sattr(0, 0, 0, 0, new timeval(1, 2), new timeval(3, 4));

    args = new createargs(dirOpArgs, attr);
    result = pfs.create(args);

    assert(result.getStatus() == status);
  }

 /** 
 *  Test the lookup method 
 **/ 
  private static fhandle
  testLookup(PangaeaFS pfs,
             fhandle fh,
             String name){
    diropargs args = null;
    diropres result = null;

    args = new diropargs(fh, name);
    result = pfs.lookup(args);
    if(result.getStatus() == stat.NFS_OK){
      fh = result.getFile();
    }else{
      fh = null;
    }

    return(fh);
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  private static void
  testGetAttributes(
                    PangaeaFS pfs,
                    fhandle fh,
                    long expectedType){
    attrstat result = null;
    fattr attr = null;

    result = pfs.getAttributes(fh);
    assert(result.getStatus() == stat.NFS_OK);

    attr = result.getAttributes();
    
    assert attr.getType() == expectedType;
    /*
    assert attr.getMode() == expectedAttr.getMode();
    assert attr.getNLink() == expectedAttr.getNLink();
    assert attr.getUID() == expectedAttr.getUID();
    assert attr.getGID() == expectedAttr.getGID();
    assert attr.getSize() == expectedAttr.getSize();
    assert attr.getBlockSize() == expectedAttr.getBlockSize();
    assert attr.getBlocks() == expectedAttr.getBlocks();
    assert attr.getFSID() == expectedAttr.getFSID();
    assert attr.getFileId() == expectedAttr.getFileId();
    */
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  private static void
  testSetAttributes(
                    PangaeaFS pfs,
                    fhandle fh){
    attrstat result1 = null;
    attrstat result2 = null;
    fattr attr1 = null;
    fattr attr2 = null;
    sattrargs args = new sattrargs(fh,
                                  new sattr(1, 2, 3, 4, 
                                            new timeval(5, 6), 
                                            new timeval(7, 8)));
    result1 = pfs.setAttributes(args);
    assert(result1.getStatus() == stat.NFS_OK);
    
    attr1 = result1.getAttributes();
    
    result2 = pfs.getAttributes(fh);
    assert(result2.getStatus() == stat.NFS_OK);

    attr2 = result2.getAttributes();
    assert attr1.equals(attr2);
    //System.out.println("Size = " + attr.getSize());
  }

 /** 
 *  Test the write method 
 **/ 
  private static void
  testWrite(
            PangaeaFS pfs,
	    byte b,
	    int offset,
	    int len,
            fhandle fh){
    byte[] data = null;
    attrstat result = null;

    data = new byte[len];
    for(int i = 0; i < len; i++){
      data[i] = b;
    }
    result = pfs.write(fh, 0, offset, len, data);
    assert(result.getStatus() == stat.NFS_OK);

    // See what our fake local interface has
    //System.out.println("" + li);
  }

 /** 
 *  Test the read method 
 **/ 
  private static void
  testRead(
           PangaeaFS pfs,
           fhandle fh,
	   int offset, 
	   int len,
	   byte b){
    byte[] data = null;
    readargs args = null;
    readres result = null;

    // Read the first 50 bytes
    args = new readargs(fh, offset, len, len);
    result = pfs.read(args);
    assert(result.getStatus() == stat.NFS_OK);

    data = result.getData();
    //assert(data.length == 50);
    //System.out.println("" + data.length);
    assert data.length == len;
    for(int i = 0; i < len; i ++){
      assert(data[i] == b);
    }

  }

 /** 
 *  Test the readdir method 
 **/ 
  private static void
  testReadDir(
              PangaeaFS pfs,
              fhandle fh,
	      String[] names){
    readdirargs args = null;
    readdirres result = null;
    entry entryList = null;

    args = new readdirargs(fh, 0, 100);
    result = pfs.readdir(args);
    assert(result.getStatus() == stat.NFS_OK);
    assert(result.getEOF());

    /*
    entryList = result.getEntries();
    while(entryList != null){
      System.out.println("entryList: (" + entryList.getFileId() +
                         ", " + entryList.getName() +
                         ", " + entryList.getCookie() + ")");
      entryList = entryList.getNextEntry();
    }
    */

    entryList = result.getEntries();
    assert(entryList != null);
    assert(entryList.getName().equals("."));
    entryList = entryList.getNextEntry();
    assert(entryList != null);
    assert(entryList.getName().equals(".."));
    entryList = entryList.getNextEntry();
    
    for(int i = 0; i < names.length; i ++){
      assert(entryList != null);
      assert(entryList.getName().equals(names[i]));
      entryList = entryList.getNextEntry();
    }
    assert(entryList == null);
  }

 /** 
 *  Test the rename method 
 **/ 
  private static void
  testRename(
             PangaeaFS pfs,
             fhandle origFH,
             String origName,
             fhandle destFH,
             String destName){
    int result = 0;
    renameargs args = null;

    args = new renameargs(new diropargs(origFH, origName),
                          new diropargs(destFH, destName));
    result = pfs.rename(args);
    assert(result == stat.NFS_OK); 
    //System.out.println("" + li);
  }

 /** 
 *  Test the mkdir method 
 **/ 
  private static void
  testMkdir(
            PangaeaFS pfs,
            fhandle dirHandle,
            String newSubDirName){
    createargs args = null;
    diropres result = null;
    args = new createargs(new diropargs(dirHandle, newSubDirName),
                          new sattr(10, 20, 30, 40, new timeval(4, 8), new timeval(1, 2)));
    result = pfs.mkdir(args);
    assert(result.getStatus() == stat.NFS_OK);
  }

}

 /** 
 *  Stub for OverlogPolicy 
 *  to test code without P2 
 **/ 
class TestOverlogPolicy extends OverlogPolicy{
  
  
  private Vector insertedTuples;

  private boolean dbg = false;


 /** 
 *  Constructor 
 **/ 
  
  public TestOverlogPolicy(String olgFilename, 
			   NodeId myNodeId,
			   String nodeMapFilename){
    super(olgFilename, myNodeId, nodeMapFilename);
    insertedTuples = new Vector();
  }

 
 /** 
 *  Insert tuple - Can be used by LI to insert tuples into the 
 *  overlog file 
 **/ 
  public void insertTuple(Tuple tp) {
    insertedTuples.add(tp.toString());
    Env.dprintln(dbg, "insert overlog tuple: " + tp.toString());
  }
  
 /** 
 *  verify the inserted tuples  
 **/ 
  public void verifyTuples(Vector expected){
    assert insertedTuples.size() == expected.size();
    assert insertedTuples.containsAll(expected);
  }
}

 /** 
/* $Log: PangaeaFS.java,v $
/* Revision 1.19  2007/09/12 19:08:39  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.18  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.17  2007/04/02 21:11:39  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.16  2007/03/09 23:01:24  zjiandan
/* Make IncommingConnection not kick off an object when receiving an impreciseinv
/* if the receiver's local lpvv is not "less" then the inv.endVV.
/*
/* Revision 1.15  2007/03/07 20:39:07  zjiandan
/* add InvalIterator Filter.
/*
/* Revision 1.14  2007/03/06 18:25:52  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.13  2007/03/01 07:54:53  zjiandan
/* TierStore overlog and test helper, tested without PRACTI.
/*
/* Revision 1.12  2007/02/27 04:44:41  zjiandan
/* change readOfHole interface such that read of hole will throw an
/* ReadOfHoleException with the position of the next written byte.
/*
/* Revision 1.11  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.10  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.9  2007/01/10 21:07:39  nalini
/* cleaned up NFS interface so that all policy is in PRACTIFSLocalInterface
/*
/* Revision 1.8  2007/01/10 02:02:24  zjiandan
/* Fixed rmiregistry problems.
/*
/* Revision 1.7  2006/11/17 20:04:33  zjiandan
/* Tested new PRACTIFS.
/*
/* Revision 1.6  2006/11/16 21:22:57  zjiandan
/* *** empty log message ***
/*
/* Revision 1.5  2006/11/15 21:56:32  nayate
/* Fixed to pass in the new bound parameter to PRACTIFS.PRACTIFS()
/*
/* Revision 1.4  2006/11/14 02:02:15  zjiandan
/* Compiled Pangaea NFS version with some UnitTests.
/*
/* Revision 1.3  2006/11/02 21:28:39  zjiandan
/* fixed some minor bugs and turn off the debug printing.
/*
/* Revision 1.2  2006/11/02 21:11:19  zjiandan
/* Fixed some bugs in URAOverlogNode and TupleHandlers.
/*
/* Revision 1.1  2006/11/01 19:26:46  zjiandan
/* Integrate PangaeaFS and NFS interface.*/
 **/ 
