 /** 
/* PangaeaP2Synchronizer.java
 *
 *  synchronize between PRACTI and p2 for Pangaea
 *    - insert createNewObj to p2
 *    - wait for p2 to return the set of goldNodes for the new object
 *      so that we know what to put into the new object's directory entry
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Random;

public class PangaeaP2Synchronizer{
  private static final boolean DEBUG = false;
  private static final String DEBUGFLAG = "****PangaeaP2Synchronier:: ";
  HashMap gnSyncTable;
  OverlogPolicy op;
  
  public PangaeaP2Synchronizer(OverlogPolicy op){
    gnSyncTable = new HashMap();
    this.op = op;
  }

 /** 
 *  called from a InformGoldNodeHandler 
 *  when p2 inserts a "informGoldNode" tuple to PRACTI 
 **/ 
  public synchronized void informGoldNode(ObjId objId, String goldNodeOverlogId)
  throws UnexpectedTupleFromP2Exception{
    HashSet gn = (HashSet)(gnSyncTable.get(objId));
    Env.dprinterrln(DEBUG, DEBUGFLAG + "informGoldNode " + objId + " " + goldNodeOverlogId); 
    if (gn == null){
      Env.dprinterrln(DEBUG, DEBUGFLAG + "informGoldNode " + objId + " " 
                      + goldNodeOverlogId + " no threads wait for it!!!!!!!"); 
      throw new UnexpectedTupleFromP2Exception();  
    }
    gn.add(goldNodeOverlogId);
    notifyAll();
    Env.dprinterrln(DEBUG, DEBUGFLAG + "informGoldNode " + objId + " " 
                      + goldNodeOverlogId + " notifyAll() called"); 
  }

 /** 
 * called when creating a new object 
 * 1. insert a createNewObj tuple to ask p2 to tell me the initial goldNodes 
 * 2. block until p2 inform me of all the goldNodes  
 **/ 
  public synchronized HashSet getGoldNodesFromP2(ObjId objId, int goldNum)
  throws SyncRequestAlreadyExistsException{
    Env.dprinterrln(DEBUG, DEBUGFLAG + "getGoldNodesFromP2 " + objId + " " + goldNum); 
    HashSet gn = (HashSet) gnSyncTable.get(objId); 
    if(gn != null){
      Env.dprinterrln(DEBUG, DEBUGFLAG + "getGoldNodesFromP2 " + objId 
                      + " " + goldNum + " someone else is waiting for it already!!!!!!!!!"); 
      throw new SyncRequestAlreadyExistsException();
    }else{
        gn = new HashSet();
	gnSyncTable.put(objId, gn);
    }
    //note: 
    //  we should create the entry for this object
    //  before telling p2 to send PRACTI goldNode tuple so that
    //  the TupleHandler won't complain UnexpecedTupleFromP2Exception when it receives
    //  the informgoldNode tuple.
    String objIdStr = objId.toString();

    if(OverlogPolicy.P2VersionHasStringProblem){
      assert !objId.toString().contains("-") :
        "\"-\" is not allowed in ObjId because we are using the p2 version"
        + "\nthat can't interpret \"/\" well and have to hack by replacing \"/\" with"
        + "\n \"-\"."
        + " If you are using a p2 version works with \"/\", turn off OverlogPolicy.P2VersionHasStringProblem";

      objIdStr = objId.toString().replaceAll("/", "-");
    }

    Env.dprinterrln(DEBUG, DEBUGFLAG + " about to insert createNewObj tuple for " + objId + " and wait for reply \n");
    String[] tupleStr = {"createNewObj", op.getMyOverlogId().toString(), objIdStr};
    op.insertTuple(new Tuple(tupleStr));    
    while (gn.size() < goldNum){
      
      try{
        wait();
      }catch(InterruptedException e){
        //ignore;
      }
     
      gn = (HashSet)(gnSyncTable.get(objId));
      Env.dprinterrln(DEBUG, DEBUGFLAG + "getGoldNodesFromP2 " + objId + " " + goldNum 
                      + "wake up: " + gn.size() + " goldNodes received. "); 
    }
    gnSyncTable.remove(objId);
    return gn;
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing PangaeaP2Synchronizer.java...");
    PangaeaP2Synchronizer.testSimple();
    System.out.println("...PangaeaP2Synchronizer.java");
    System.exit(0);
  }
  
 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    FakeP2RequestQueue queue = new FakeP2RequestQueue();
    OverlogPolicy op = new FakeOverlogPolicy("case-studies/Pangaea/PangaeaTopology.olg", 
                                             new NodeId(1), "test/pfs.map", queue);
    PangaeaP2Synchronizer pps = new PangaeaP2Synchronizer(op);
    InformGoldNodeThread ignt = new InformGoldNodeThread(pps, queue);
    Thread it = new Thread(ignt);
    it.start();
    try{
      HashSet golds = pps.getGoldNodesFromP2(new ObjId("obj1"), PangaeaFS.GOLDNUM);
      for(Iterator iter = golds.iterator(); iter.hasNext();){
        String goldId = (String) iter.next();
        Env.dprintln(DEBUG, "goldNodes for obj1: " + goldId);
      }
    }catch(SyncRequestAlreadyExistsException e){
      assert false;
    }
    Thread g1 = (new Thread(new GetGoldNodeThread(pps, 1)));
    Thread g2 = (new Thread(new GetGoldNodeThread(pps, 2)));
    Thread g3 = (new Thread(new GetGoldNodeThread(pps, 3)));
    g1.start();
    g2.start();
    g3.start();
    try{
      g1.join();
      g2.join();
      g3.join();
    
      ignt.kill();
    }catch(InterruptedException e){
      assert false;
    }
  }
}

class FakeP2RequestQueue {
 /** 
 *  Data members 
 **/ 
  private LinkedList list;
  
 /** 
 *  Constructor 
 **/ 
  public
  FakeP2RequestQueue(){
    this.list = new LinkedList();
  }

 /** 
 *  Dequeue the next entry (wait until something available) 
 **/ 
  public synchronized Tuple
  dequeue() throws InterruptedException{
    Object obj = null;

    while(this.list.isEmpty()){
      this.wait();
    }
    obj = this.list.removeFirst();
    return((Tuple)obj);
  }

 /** 
 *  Enqueue the next entry 
 **/ 
  public synchronized void
  enqueue(Tuple obj){
    this.list.addLast(obj);
    this.notifyAll();
  }

}

class FakeOverlogPolicy extends OverlogPolicy{
 /** 
 *  Data members 
 **/ 
  private FakeP2RequestQueue queue;
  
 /** 
 *  Constructor 
 **/ 
  public
  FakeOverlogPolicy(String olgName,
                    NodeId myNodeId,
                    String mapFile,
                    FakeP2RequestQueue queue){
    super(olgName, myNodeId, mapFile);
    this.queue = queue;
  }

 /** 
 *  Insert tuple - Can be used by LI to insert tuples into the 
 *  overlog file 
 **/ 
  public void insertTuple(Tuple tp) {
    this.queue.enqueue(tp);
  }
}

class InformGoldNodeThread implements Runnable{

 /** 
 *  Data members 
 **/ 
  PangaeaP2Synchronizer pps = null; 
  FakeP2RequestQueue pq = null;
  boolean die = false;

 /** 
 *  Constructor 
 **/ 
  public
  InformGoldNodeThread(PangaeaP2Synchronizer pps, FakeP2RequestQueue pq){
    this.pps = pps;
    this.pq = pq;
    
  }
  
  public void kill(){
    die = true;
  }

 /** 
 *  Main thread method 
 **/ 
  public void
  run(){
    
    while(!die){
      
        int count = 0;
        try{
          Tuple tp = this.pq.dequeue();
          assert ((String)(tp.getTupleName())).equals("createNewObj");
          //System.out.println(tp);
          assert tp.getLength() == 3;
          String objId = tp.getItem(2);
          for(int i = 0; i < PangaeaFS.GOLDNUM;i++){
            try{
              Thread.sleep(3);
              pps.informGoldNode(new ObjId(objId), ""+count);
            }catch(UnexpectedTupleFromP2Exception e){
              assert false;
            }
            count++;
          }
        }catch(InterruptedException e){
          assert false;
        }
    }
    
  }
}

class GetGoldNodeThread implements Runnable{

 /** 
 *  Data members 
 **/ 
  PangaeaP2Synchronizer pps = null;
  int id;
  final static private boolean DEBUG = false;
 /** 
 *  Constructor 
 **/ 
  public
  GetGoldNodeThread(PangaeaP2Synchronizer pps, int id){
    this.pps = pps;
    this.id = id;
    
  }

 /** 
 *  Main thread method 
 **/ 
  public void
  run(){

    int count = 0;
    while(count < 100){
      try{
        Random r = new Random();
        Thread.sleep(r.nextInt(10));
        ObjId obj = new ObjId("obj." + id + "." + count);
        HashSet golds = pps.getGoldNodesFromP2(obj, PangaeaFS.GOLDNUM);
        for(Iterator iter = golds.iterator(); iter.hasNext();){
          String goldId = (String) iter.next();
          Env.dprintln(DEBUG, "goldNodes for " + obj + ": " +goldId);
        }
      }catch(SyncRequestAlreadyExistsException e){
        assert false;
      }catch(InterruptedException e){
        assert false;
      }
      count++;
    }
  }
}

 /** 
/* $Log: PangaeaP2Synchronizer.java,v $
/* Revision 1.7  2007/09/12 19:08:39  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.6  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.5  2007/01/10 06:34:01  zjiandan
/* fixed "mount: localhost:practidir: can't read superblock" problem
/* and the Java P2 wrapper string problem.
/*
/* Revision 1.4  2006/11/17 20:04:33  zjiandan
/* Tested new PRACTIFS.
/*
/* Revision 1.3  2006/11/16 21:22:57  zjiandan
/* *** empty log message ***
/*
/* Revision 1.2  2006/11/14 02:02:15  zjiandan
/* Compiled Pangaea NFS version with some UnitTests.
/*
/* Revision 1.1  2006/11/01 19:26:46  zjiandan
/* Integrate PangaeaFS and NFS interface.
/* */
 **/ 
