 /** 
/* TierStoreFS.java
 *
 * NFS interface wrapper for TierStore to implement the java version of the
 * NFS loopback server
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;
import java.util.*;

public class TierStoreFS extends PRACTIFS{
  
  private final static boolean dbg = false;
  private static final boolean PRINT_METHODS = false;
    
  private static boolean warnedP2InteractionInefficiency = false;

  OverlogPolicy op;
  //TierStoreP2Synchronizer pps; //see comments for createDirEntry()
  

 /** 
 *  construction: 
 *     if this node is primary, create root directory 
 *     otherwise do not need to create root, the p2 rules will  
 *     automatically set up the subscription for the root directory 
 *   
 *     need to traverse the entire file system to recover the 
 *     connections for valid replicas 
 **/ 
  public TierStoreFS(TierStoreLocalInterface li, OverlogPolicy op, boolean primary){
    super(li, primary);
    
    Env.dprintln(dbg, "creating TierSToreFSLocalInterface done");
    this.op = op;
    //pps = new TierStoreP2Synchronizer(op);//see comments for createDirEntry()
    /*
    try{
      recover(new ObjId(super.ROOT_DIR_NAME), null);
    }catch(IOException e){
      e.printStackTrace();
      System.exit(-1);
    }
    Env.dprintln(dbg, "creating TierStoreFS done");
    */
  }

 /** 
 *  Used for testing 
 **/ 
  public static void
  main(String[] argv){
    Env.verifyAssertEnabled();
    System.out.println("Testing TierStoreFS.java...");
    TierStoreFS.testSimple();
    TierStoreFS.testLiveness();
    System.out.println("...Finished");
    System.exit(0);
  }
  
 /** 
 *  Test liveness 
 *  
 *    create image: 
 *       /root(P/V)/a(P/V)/a1(P/V)   
 *                        /a2(P/IV) 
 *                        /a3(IP/V) 
 *                        /a4(IP/IV) 
 *                 /b(P/IV)/b1(P/V) 
 *                         /b2(P/IV) 
 *                         /b3(IP/V) 
 *                         /b4(IP/IV) 
 *                 /c(IP/V)/c1(P/V) 
 *                         /c2(P/IV) 
 *                         /c3(IP/V) 
 *                         /c4(IP/IV) 
 *                 /d(IP/IV)/d1(P/V) 
 *                          /d2(P/IV) 
 *                          /d3(IP/V) 
 *                          /d4(IP/IV) 
 *  
 **/ 
  private static void
  testLiveness(){
    boolean withP2 = false;
    Env.dprintln(dbg, " TierStoreFS liveness with P2: " + withP2);
    TierStoreFS pfs = null;
    OverlogPolicy op;
    P2Runtime runtime;
    Core core;
    TierStoreLocalInterface li;

    boolean cleanDb;
    boolean noSyncLog;
    fhandle handle = null;
    fhandle handle2 = null;
    String configFile = "test" + File.separatorChar + "pfs.unittest.config";
    String p2ConfigFile = "test" + File.separatorChar + "pfs.unittest.p2config";
    NodeId myNodeId = new NodeId(2);
    String overlogFile = "test" + File.separatorChar + "pfs.unittest.olg";
    String nodeMapFile = "test" + File.separatorChar + "pfs.unittest.map";
    
    //members in URAOverlogNode
    if(!withP2){
      op = new TestOverlogPolicy(overlogFile, myNodeId, nodeMapFile);
    }else{
      op = new OverlogPolicy(overlogFile, myNodeId, nodeMapFile);
    }
    
    cleanDb = true;
    noSyncLog = false;
    runtime = new P2Runtime(configFile, p2ConfigFile, myNodeId, cleanDb, op, noSyncLog);
    core = runtime.getCore();
    li = new TierStoreLocalInterface(new LocalInterface(runtime.getController(), core), 
				   myNodeId);
    pfs = new TierStoreFS(li, op, true);
    
    assert false; //tbd
    
    
  }

 /** 
 *  Used for testing 
 **/ 
  private static void
  testSimple(){
    TierStoreLocalInterface li = null;
    TierStoreFS pfs = null;
    fhandle handle = null;
    fhandle handle2 = null;
    String configFile = "test" + File.separatorChar + "pfs.unittest.config";
    String p2ConfigFile = "test" + File.separatorChar + "pfs.unittest.p2config";
    NodeId myNodeId = new NodeId(1);
    String overlogFile = "test" + File.separatorChar + "pfs.unittest.olg";
    String nodeMapFile = "test" + File.separatorChar + "pfs.unittest.map";

    TierStoreURAOverlogNode uraNode = new TierStoreURAOverlogNode(configFile,
                                                              p2ConfigFile,
                                                              myNodeId,
                                                              true,
                                                              overlogFile,
                                                              nodeMapFile,
							      false);
    pfs = uraNode.getFS();

    //System.out.println("" + li);
    TierStoreFS.testCreate( pfs, fhandle.makeAllZeroHandle(), "a", stat.NFS_OK);
    handle = TierStoreFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle == null);
    handle = TierStoreFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "a");
    assert(handle != null);
    handle2 = TierStoreFS.testLookup( pfs, handle, "b");
    assert(handle2 == null);
    
    TierStoreFS.testGetAttributes( pfs, handle, ftype.NFREG);
    TierStoreFS.testGetAttributes( pfs, fhandle.makeAllZeroHandle(), ftype.NFDIR);
    TierStoreFS.testWrite( pfs, (byte)65, 0, 50, handle);
    TierStoreFS.testWrite( pfs, (byte)66, 50, 50, handle);
    TierStoreFS.testGetAttributes( pfs, handle,  ftype.NFREG);
    TierStoreFS.testRead( pfs, handle, 50, 50, (byte)66);
    TierStoreFS.testRead( pfs, handle, 0, 50, (byte)65);
    String[] expectedNames = new String[1];
    expectedNames[0] = "a";
    TierStoreFS.testReadDir( pfs, fhandle.makeAllZeroHandle(), expectedNames);
    TierStoreFS.testRename(
                        pfs,
                        fhandle.makeAllZeroHandle(),
                        "a",
                        fhandle.makeAllZeroHandle(),
                        "b");
    
    handle = TierStoreFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "a");
    assert(handle == null);
    
    expectedNames[0] = "b";
    TierStoreFS.testReadDir( pfs, fhandle.makeAllZeroHandle(), expectedNames);

    handle = TierStoreFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle != null);
    
    TierStoreFS.testMkdir( pfs, fhandle.makeAllZeroHandle(), "subDir");
    handle = TierStoreFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "subDir");
    assert(handle != null);
    
    expectedNames = new String[2];
    expectedNames[0] = "b";
    expectedNames[1] = "subDir";
    TierStoreFS.testReadDir( pfs, fhandle.makeAllZeroHandle(), expectedNames);

    expectedNames = new String[0];
    TierStoreFS.testReadDir( pfs, handle, expectedNames);//read subDir

    
    TierStoreFS.testRename( pfs, fhandle.makeAllZeroHandle(), "b", handle, "newB");
    handle2 = TierStoreFS.testLookup( pfs, handle, "newB");
    assert(handle2 != null);

    expectedNames = new String[1];
    expectedNames[0] = "newB";
    TierStoreFS.testReadDir( pfs, handle, expectedNames);//read subDir

    handle2 = TierStoreFS.testLookup( pfs, fhandle.makeAllZeroHandle(), "b");
    assert(handle2 == null);
    TierStoreFS.testSetAttributes( pfs, handle);
    try{
      uraNode.shutdown();
      Thread.sleep(2000);
    }catch(Exception e){
      e.printStackTrace();
    }
    
    
  }

 /** 
 *  Test the create method 
 **/ 
  private static void
    testCreate(TierStoreFS pfs, fhandle fh, String name, int status){
    createargs args = null;
    diropres result = null;
    diropargs dirOpArgs = null;
    sattr attr = null;

    dirOpArgs = new diropargs(fh, name);
    attr = new sattr(0, 0, 0, 0, new timeval(1, 2), new timeval(3, 4));

    args = new createargs(dirOpArgs, attr);
    result = pfs.create(args);

    assert(result.getStatus() == status);
  }

 /** 
 *  Test the lookup method 
 **/ 
  private static fhandle
  testLookup(TierStoreFS pfs,
             fhandle fh,
             String name){
    diropargs args = null;
    diropres result = null;

    args = new diropargs(fh, name);
    result = pfs.lookup(args);
    if(result.getStatus() == stat.NFS_OK){
      fh = result.getFile();
    }else{
      fh = null;
    }

    return(fh);
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  private static void
  testGetAttributes(
                    TierStoreFS pfs,
                    fhandle fh,
                    long expectedType){
    attrstat result = null;
    fattr attr = null;

    result = pfs.getAttributes(fh);
    assert(result.getStatus() == stat.NFS_OK);

    attr = result.getAttributes();
    
    assert attr.getType() == expectedType;
    /*
    assert attr.getMode() == expectedAttr.getMode();
    assert attr.getNLink() == expectedAttr.getNLink();
    assert attr.getUID() == expectedAttr.getUID();
    assert attr.getGID() == expectedAttr.getGID();
    assert attr.getSize() == expectedAttr.getSize();
    assert attr.getBlockSize() == expectedAttr.getBlockSize();
    assert attr.getBlocks() == expectedAttr.getBlocks();
    assert attr.getFSID() == expectedAttr.getFSID();
    assert attr.getFileId() == expectedAttr.getFileId();
    */
  }

 /** 
 *  Test the getAttributes method 
 **/ 
  private static void
  testSetAttributes(
                    TierStoreFS pfs,
                    fhandle fh){
    attrstat result1 = null;
    attrstat result2 = null;
    fattr attr1 = null;
    fattr attr2 = null;
    sattrargs args = new sattrargs(fh,
                                  new sattr(1, 2, 3, 4, 
                                            new timeval(5, 6), 
                                            new timeval(7, 8)));
    result1 = pfs.setAttributes(args);
    assert(result1.getStatus() == stat.NFS_OK);
    
    attr1 = result1.getAttributes();
    
    result2 = pfs.getAttributes(fh);
    assert(result2.getStatus() == stat.NFS_OK);

    attr2 = result2.getAttributes();
    assert attr1.equals(attr2);
    //System.out.println("Size = " + attr.getSize());
  }

 /** 
 *  Test the write method 
 **/ 
  private static void
  testWrite(
            TierStoreFS pfs,
	    byte b,
	    int offset,
	    int len,
            fhandle fh){
    byte[] data = null;
    attrstat result = null;

    data = new byte[len];
    for(int i = 0; i < len; i++){
      data[i] = b;
    }
    result = pfs.write(fh, 0, offset, len, data);
    assert(result.getStatus() == stat.NFS_OK);

    // See what our fake local interface has
    //System.out.println("" + li);
  }

 /** 
 *  Test the read method 
 **/ 
  private static void
  testRead(
           TierStoreFS pfs,
           fhandle fh,
	   int offset, 
	   int len,
	   byte b){
    byte[] data = null;
    readargs args = null;
    readres result = null;

    // Read the first 50 bytes
    args = new readargs(fh, offset, len, len);
    result = pfs.read(args);
    assert(result.getStatus() == stat.NFS_OK);

    data = result.getData();
    //assert(data.length == 50);
    //System.out.println("" + data.length);
    assert data.length == len;
    for(int i = 0; i < len; i ++){
      assert(data[i] == b);
    }

  }

 /** 
 *  Test the readdir method 
 **/ 
  private static void
  testReadDir(
              TierStoreFS pfs,
              fhandle fh,
	      String[] names){
    readdirargs args = null;
    readdirres result = null;
    entry entryList = null;

    args = new readdirargs(fh, 0, 100);
    result = pfs.readdir(args);
    assert(result.getStatus() == stat.NFS_OK);
    assert(result.getEOF());

    /*
    entryList = result.getEntries();
    while(entryList != null){
      System.out.println("entryList: (" + entryList.getFileId() +
                         ", " + entryList.getName() +
                         ", " + entryList.getCookie() + ")");
      entryList = entryList.getNextEntry();
    }
    */

    entryList = result.getEntries();
    assert(entryList != null);
    assert(entryList.getName().equals("."));
    entryList = entryList.getNextEntry();
    assert(entryList != null);
    assert(entryList.getName().equals(".."));
    entryList = entryList.getNextEntry();
    
    for(int i = 0; i < names.length; i ++){
      assert(entryList != null);
      assert(entryList.getName().equals(names[i]));
      entryList = entryList.getNextEntry();
    }
    assert(entryList == null);
  }

 /** 
 *  Test the rename method 
 **/ 
  private static void
  testRename(
             TierStoreFS pfs,
             fhandle origFH,
             String origName,
             fhandle destFH,
             String destName){
    int result = 0;
    renameargs args = null;

    args = new renameargs(new diropargs(origFH, origName),
                          new diropargs(destFH, destName));
    result = pfs.rename(args);
    assert(result == stat.NFS_OK); 
    //System.out.println("" + li);
  }

 /** 
 *  Test the mkdir method 
 **/ 
  private static void
  testMkdir(
            TierStoreFS pfs,
            fhandle dirHandle,
            String newSubDirName){
    createargs args = null;
    diropres result = null;
    args = new createargs(new diropargs(dirHandle, newSubDirName),
                          new sattr(10, 20, 30, 40, new timeval(4, 8), new timeval(1, 2)));
    result = pfs.mkdir(args);
    assert(result.getStatus() == stat.NFS_OK);
  }

}

 /** 
 *  Stub for OverlogPolicy 
 *  to test code without P2 
 **/ 
class TestOverlogPolicy extends OverlogPolicy{
  
  
  private Vector insertedTuples;

  private boolean dbg = false;


 /** 
 *  Constructor 
 **/ 
  
  public TestOverlogPolicy(String olgFilename, 
			   NodeId myNodeId,
			   String nodeMapFilename){
    super(olgFilename, myNodeId, nodeMapFilename);
    insertedTuples = new Vector();
  }

 
 /** 
 *  Insert tuple - Can be used by LI to insert tuples into the 
 *  overlog file 
 **/ 
  public void insertTuple(Tuple tp) {
    insertedTuples.add(tp.toString());
    Env.dprintln(dbg, "insert overlog tuple: " + tp.toString());
  }
  
 /** 
 *  verify the inserted tuples  
 **/ 
  public void verifyTuples(Vector expected){
    assert insertedTuples.size() == expected.size();
    assert insertedTuples.containsAll(expected);
  }
}

 /** 
/* $Log: TierStoreFS.java,v $
/* Revision 1.1  2007/03/02 01:35:29  zjiandan
/* Yeah! Working Simplified TierStore.
/* */
 **/ 
