 /** 
 *  Global Coordinator 
 *  Read scripts from standard input 
 *  Currently have 4 kinds of commands: subscribe, subscribeBody, 
 *                                      issueDemandRead, and sleep 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Random;
import java.util.Vector;

public class CoordExptE extends CoordExptA
{
    public static long SUBSCRIPTION_DURATION_MS = 1000000;  // 1000 seconds
    public static long MAX_DELAY_MS = 1000; // 1 second

 /** 
 *  Constructor 
 **/ 
    public CoordExptE(String exptConfigFileName, String shellType_)
    {
	super(exptConfigFileName, shellType_);
    }

 /** 
 *  Return the file name generator based on the current experiment number 
 **/ 
    public RandFileBaseGen getRandFileBaseGen()
    {
	RandFileBaseGen rfbg = null;

	switch(this.exptConfig.getExptNum()){
	case 1:
	    // The (10/100) case
	    rfbg = new Case1RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	case 2:
	    // The (10/10) case
	    rfbg = new Case2RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	case 3:
	    // The (1/100) case
	    rfbg = new Case3RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	case 4:
	    // The (1/10) case
	    rfbg = new Case4RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	default:
	    assert (false):"Unimplemented";
	}
	return(rfbg);
    }

 /** 
 *  Return an interest set that contains exactly those files we are 
 *  interested in for cases 2 and 4; i.e. with 1 file per low-level 
 *  directory. 
 **/ 
    public DirectoryInterestSet makeFileInterestSet(int quotient)
    {
	int numFiles = 0;
	int fileBase = 0;
	String fileName = null;
	DirectoryInterestSet dis = null;
	Vector directoryVec = null;
	String[] allFiles = null;

	numFiles = this.exptConfig.getNumFiles();
	directoryVec = new Vector();
	for(int i = 0; i < ((numFiles / quotient) / 10); i++){
	    fileName = this.senderScriptGen.createFileName(i * 10, numFiles);
	    directoryVec.add(fileName);
	}
	allFiles = (String[])directoryVec.toArray(new String[0]);
	dis = new DirectoryInterestSet(allFiles);
	return(dis);
    }

 /** 
 *  Return an interest set that contains exactly those files we are 
 *  interested in 
 **/ 
    public DirectoryInterestSet makeInterestSet()
    {
	DirectoryInterestSet dis = null;

	switch(this.exptConfig.getExptNum()){
	case 1:
	    // The (10/100) case
	    dis = new DirectoryInterestSet("/0/*");
	    break;
	case 2:
	    // The (10/10) case
	    dis = this.makeFileInterestSet(10);
	    break;
	case 3:
	    // The (1/100) case
	    dis = new DirectoryInterestSet("/0/0/*");
	    break;
	case 4:
	    // The (1/10) case
	    dis = this.makeFileInterestSet(100);
	    break;
	default:
	    assert (false):"Unimplemented";
	}
	return(dis);
    }

    //-----------------------------------------------------------------------
    // Check if objId is a file that is in the interest set we care about
    //-----------------------------------------------------------------------
    protected boolean liesInInterestSet(String objName)
    {
	boolean result = false;

	switch(this.exptConfig.getExptNum()){
	case 1:
	    // The (10/100) case
	    result = objName.startsWith("/0/");
	    break;
	case 2:
	    // The (10/10) case
	    result = (objName.startsWith("/0/") &&
		      (objName.endsWith("/0") || objName.endsWith("/0/")));
	    break;
	case 3:
	    // The (1/100) case
	    result = objName.startsWith("/0/0/");
	    break;
	case 4:
	    // The (1/10) case
	    result = (objName.startsWith("/0/0/") &&
		      (objName.endsWith("/0") || objName.endsWith("/0/")));
	    break;
	default:
	    assert (false):"Unimplemented";
	}
	return(result);
    }

    //-----------------------------------------------------------------------
    // Make an array full of random writes
    //-----------------------------------------------------------------------
    protected WriteEntry[] makeRandomWrites()
    {
	long numFiles = 0;
	long fileBase = 0;
	long fileSize = 0;
	WriteEntry[] writeEntries = null;
	Random rand = null;
	String fileName = null;
	long offset = 0;
	long length = 0;

	numFiles = this.exptConfig.getNumFiles();
	fileSize = this.exptConfig.getFileSize();
	rand = new Random(1978);
	writeEntries = new WriteEntry[this.exptConfig.getNumWrites()];
	for(int i = 0; i < writeEntries.length; i++){
	    // Generate a random write entry
	    fileBase = (long)(rand.nextDouble() * numFiles);
	    offset = (long)(rand.nextDouble() * fileSize);
	    length = (long)(rand.nextDouble() * (fileSize - offset)) + 1;
	    assert(length > 0); // Otherwise our code chokes
	    assert((offset + length) <= fileSize);

	    fileName = this.senderScriptGen.createFileName(fileBase, numFiles);
	    writeEntries[i] = new WriteEntry(fileName, offset, length);
	}
	return(writeEntries);
    }

    /*
    //-----------------------------------------------------------------------
    // Ask nodeId to perform the random writes in writeEntries
    //-----------------------------------------------------------------------
    protected void sendRandomWritesScript(int nodeId,
					  WriteEntry[]
					  writeEntries)
    {
	OutputStream os = null;
	PrintStream ps = null;

	os = this.getProcessOutputStream(nodeId);
	ps = new PrintStream(os);
	for(int i = 0; i < writeEntries.length; i++){
	    ps.println("w " + writeEntries[i].getObjId() +
		       " " + writeEntries[i].getOffset() +
		       " " + writeEntries[i].getLength() +
		       " 0"); // 0 = unbound
	}
	ps.flush();
    }
    */

 /** 
 *  Ask nodeId to perform the random writes in writeEntries 
 **/ 
    protected void sendRandomWritesScript(int nodeId,
					  WriteEntry[]
					  writeEntries)
    {
	OutputStream os = null;
	PrintStream ps = null;

	os = this.getProcessOutputStream(nodeId);
	ps = new PrintStream(os);
	for(int i = 0; i < writeEntries.length; i++){
	    if(this.liesInInterestSet(writeEntries[i].getObjId())){
		// Send a bound inval
		ps.println("w " + writeEntries[i].getObjId() +
			   " " + writeEntries[i].getOffset() +
			   " " + writeEntries[i].getLength() +
			   " 1"// 1 = bound
                           + " " + 0); //no access rate
	    }else{
		ps.println("w " + writeEntries[i].getObjId() +
			   " " + writeEntries[i].getOffset() +
			   " " + writeEntries[i].getLength() +
			   " 0"// 0 = unbound
                           + " " + 0); //no access rate
	    }
	}
        ps.println("g " + nodeId);
        ps.flush();
    }

 /** 
 *  Ask nodeId to perform the reads to objects in writeEntries 
 **/ 
    protected void sendWaitReadsFinishScript(int nodeId,
					     WriteEntry[] writeEntries)
    {
	OutputStream os = null;
	PrintStream ps = null;
	CoordCommPacket ccp = null;

	os = this.getProcessOutputStream(nodeId);
	ps = new PrintStream(os);
	for(int i = 0; i < writeEntries.length; i++){
	    if(this.liesInInterestSet(writeEntries[i].getObjId())){
		ps.println("r " + writeEntries[i].getObjId() +
			   " " + writeEntries[i].getOffset() +
			   " " + writeEntries[i].getLength());
		ps.flush();
		// Wait for this read to finish
		ccp = this.recv(nodeId);
		assert(ccp.getSignalType() == CoordCommPacket.READ_DONE);
		assert(ccp.getData() instanceof AcceptStamp);
	    }
	}
	if(ccp != null){
	    System.out.println("Read AcceptStamp = " + ccp.getData());
	}
    }

 /** 
 *  Perform experiment actions 
 **/ 
    public void runExperimentE()
    {
	CoordCommPacket ccp = null;
	StatsRecord initStats = null;
	StatsRecord finalStats = null;
	StatsRecord diffStats = null;
	DirectoryInterestSet dis = null;
	WriteEntry[] writeEntries = null;
	AcceptStamp asArr[] = null;
	AcceptVV vv = null;

	/*
	 * Subscribe to B's interest set for invals and updates
	 */
	this.restartRMIRegistrySync();
	this.startNode(9);
	this.startNode(10);
	this.waitStartNode(9);
	this.waitStartNode(10);

	// Send invalidates in B's interest set to B
	asArr = new AcceptStamp[1];
	this.sendPopulateFullScript(9);
	ccp = this.recv(9);
	assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
	assert(ccp.getData() instanceof AcceptStamp);
	asArr[0] = (AcceptStamp)ccp.getData();
	dis = this.makeInterestSet();
	System.out.println("dis = " + dis);
	this.issueSubscribe(9,
			    10,
			    AcceptVV.makeVVAllNegatives(),
			    -1L,
			    dis,
			    SUBSCRIPTION_DURATION_MS,
			    MAX_DELAY_MS);
	this.sendSync(10, asArr[0]);
	ccp = this.recv(10);
	assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	assert(ccp.getData() == null);
	System.out.println("Received a SYNC_DONE");

	initStats = this.getAllStats();

	// Have A send bound invals for files that lie in B's interest
	// set and unbound invals for others
	vv = new AcceptVV(asArr);
	System.out.println("vv = " + vv);
	writeEntries = this.makeRandomWrites();
	// Have A perform bound writes to data B cares about and unbound
	// writes for the rest
	this.sendRandomWritesScript(9, writeEntries);
	// Ask A for an accept stamp and make B wait until that time
	ccp = this.recv(9);
	assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
	assert(ccp.getData() instanceof AcceptStamp);
	asArr[0] = (AcceptStamp)ccp.getData();
	this.sendSync(10, asArr[0]);

	try{
	    Thread.sleep(3000);
	}catch(InterruptedException e){
	    System.err.println("" + e);
	}

	finalStats = this.getAllStats();
	diffStats = finalStats.getDiff(initStats);
	System.out.println("---------------------------------------");
	System.out.println("" + diffStats);
	System.out.println("Exit value for node 9 = " + this.killNode(9));
	System.out.println("Exit value for node 10 = " + this.killNode(10));
    }

 /** 
 *  Program starting point 
 **/ 
    public static void main(String[] argv)
    {
	CoordExptE coord = null;

	if (argv.length < 2){
	    System.out.println("Usage: <exptConfigfile> <shellType>");
	    System.exit(-1);
	}
	coord = new CoordExptE(argv[0], argv[1]);
	coord.runExperimentE();
    }
}

 /** 
 *  $Log: CoordExptE.java,v $ 
 *  Revision 1.2  2005/01/18 22:49:43  zjiandan 
 *  Rewrited Class Serialization for network deliveringto reduce the overall bandwidth. 
 *  
 *  Revision 1.1  2004/07/07 21:26:49  nayate 
 *  Added new sub-experiments for experiment 1. 
 *  
 **/ 
