 /** 
 *  Global Coordinator 
 *  Read scripts from standard input 
 *  Currently have 4 kinds of commands: subscribe, subscribeBody, 
 *                                      issueDemandRead, and sleep 
 **/ 
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Random;

public class CoordExptSJ extends CoordExptA
{
    public static long SUBSCRIPTION_DURATION_MS = 1000000;  // 1000 seconds
    public static long MAX_DELAY_MS = 1000; // 1 second

 /** 
 *  Constructor 
 **/ 
    public CoordExptSJ(String exptConfigFileName, String shellType_)
    {
	super(exptConfigFileName, shellType_);
    }

 /** 
 *  Send a script to standard output for node nodeId to do random demand 
 *  fetches 
 **/ 
    public void sendRandomDemandFetches(int supplierNodeId,
					int receiverNodeId,
					RandFileBaseGen rfbg)
    {
	OutputStream os = null;
	PrintStream ps = null;
	long baseFileName = 0;
	String fileName = null;
	long numFiles = 0;
	long offset = 0;
	long length = 0;
	long fileSize = 0;
	Random rand = null;

	rand = new Random(4555);
	for(int i = 0; i < this.exptConfig.getNumReads(); i++){
	    baseFileName = rfbg.getNextBaseName();
	    numFiles = this.exptConfig.getNumFiles();
	    fileName = this.senderScriptGen.createFileName(baseFileName,
							   numFiles);
	    fileSize = this.exptConfig.getFileSize();
	    offset = (long)(rand.nextDouble() * fileSize);
	    length = (long)(rand.nextDouble() * (fileSize - offset)) + 1;
	    assert(length > 0); // Otherwise our code chokes
	    assert((offset + length) <= fileSize);
	    this.issueDemandRead(supplierNodeId,
				 receiverNodeId,
				 fileName,
				 offset,
				 length,
				 i);
	}
	// We must have had at least one read; wait until this is done
	assert(fileName != null);
	os = this.getProcessOutputStream(receiverNodeId);
	ps = new PrintStream(os);
	ps.println("r " + fileName + " " + offset + " " + length);
	ps.flush();
    }

 /** 
 *  Return the file name generator based on the current experiment number 
 **/ 
    public RandFileBaseGen getRandFileBaseGen()
    {
	RandFileBaseGen rfbg = null;

	switch(this.exptConfig.getExptNum()){
	case 1:
	    // The (10/100) case
	    rfbg = new Case1RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	case 2:
	    // The (10/10) case
	    rfbg = new Case2RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	case 3:
	    // The (1/100) case
	    rfbg = new Case3RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	case 4:
	    // The (1/10) case
	    rfbg = new Case4RandFileBaseGen(this.exptConfig.getNumFiles(), 10);
	    break;
	default:
	    assert (false):"Unimplemented";
	}
	return(rfbg);
    }

 /** 
 *  Return an interest set that contains the directories we are interested 
 *  in. Note that we may not care about all files in those directories. 
 **/ 
    public DirectoryInterestSet makeInterestSet()
    {
	DirectoryInterestSet dis = null;

	switch(this.exptConfig.getExptNum()){
	case 1:  // Fall through
	case 2:
	    // The (10/100) and the (10/10) cases
	    dis = new DirectoryInterestSet("/0/*");
	    break;
	case 3:
	case 4:
	    // The (1/100) and the (1/10) cases
	    dis = new DirectoryInterestSet("/0/0/*");
	    break;
	default:
	    assert (false):"Unimplemented";
	}
	return(dis);
    }

    //-----------------------------------------------------------------------
    // Check if objId lies in a directory that is in the interest set we
    // care about
    //-----------------------------------------------------------------------
    private boolean liesInInterestSet(String objName)
    {
	boolean result = false;

	switch(this.exptConfig.getExptNum()){
	case 1:  // Fall through
	case 2:
	    // The (10/100) and the (10/10) cases
	    result = objName.startsWith("/0/");
	    break;
	case 3:
	case 4:
	    // The (1/100) and the (1/10) cases
	    result = objName.startsWith("/0/0/");
	    break;
	default:
	    assert (false):"Unimplemented";
	}
	return(result);
    }

    //-----------------------------------------------------------------------
    // Make an array full of random writes
    //-----------------------------------------------------------------------
    private WriteEntry[] makeRandomWrites()
    {
	long numFiles = 0;
	long fileBase = 0;
	long fileSize = 0;
	WriteEntry[] writeEntries = null;
	Random rand = null;
	String fileName = null;
	long offset = 0;
	long length = 0;

	numFiles = this.exptConfig.getNumFiles();
	fileSize = this.exptConfig.getFileSize();
	rand = new Random(1978);
	writeEntries = new WriteEntry[this.exptConfig.getNumWrites()];
	for(int i = 0; i < writeEntries.length; i++){
	    // Generate a random write entry
	    fileBase = (long)(rand.nextDouble() * numFiles);
	    offset = (long)(rand.nextDouble() * fileSize);
	    length = (long)(rand.nextDouble() * (fileSize - offset)) + 1;
	    assert(length > 0); // Otherwise our code chokes
	    assert((offset + length) <= fileSize);

	    fileName = this.senderScriptGen.createFileName(fileBase, numFiles);
	    writeEntries[i] = new WriteEntry(fileName, offset, length);
	}
	return(writeEntries);
    }

    //-----------------------------------------------------------------------
    // Ask nodeId to perform the random writes in writeEntries
    //-----------------------------------------------------------------------
    private void sendRandomWritesScript(int nodeId, WriteEntry[] writeEntries)
    {
	OutputStream os = null;
	PrintStream ps = null;

	os = this.getProcessOutputStream(nodeId);
	ps = new PrintStream(os);
	for(int i = 0; i < writeEntries.length; i++){
	    ps.println("w " + writeEntries[i].getObjId() +
		       " " + writeEntries[i].getOffset() +
		       " " + writeEntries[i].getLength() +
		       " 0"
                       + " " + 0); //access rate
	}
	ps.flush();
    }

    //-----------------------------------------------------------------------
    // Ask nodeId to perform the reads to objects in writeEntries
    //-----------------------------------------------------------------------
    private void sendWaitReadsFinishScript(int nodeId,
					   WriteEntry[] writeEntries)
    {
	OutputStream os = null;
	PrintStream ps = null;
	CoordCommPacket ccp = null;

	os = this.getProcessOutputStream(nodeId);
	ps = new PrintStream(os);
	for(int i = 0; i < writeEntries.length; i++){
	    if(this.liesInInterestSet(writeEntries[i].getObjId())){
		ps.println("r " + writeEntries[i].getObjId() +
			   " " + writeEntries[i].getOffset() +
			   " " + writeEntries[i].getLength());
		ps.flush();
		// Wait for this read to finish
		ccp = this.recv(nodeId);
		assert(ccp.getSignalType() == CoordCommPacket.READ_DONE);
		assert(ccp.getData() instanceof AcceptStamp);
	    }
	}
	if(ccp != null){
	    System.out.println("Read AcceptStamp = " + ccp.getData());
	}
    }

    //-----------------------------------------------------------------------
    // Perform experiment actions
    //-----------------------------------------------------------------------
    public void runExperimentC()
    {
	CoordCommPacket ccp = null;
	StatsRecord initStats = null;
	StatsRecord finalStats = null;
	StatsRecord diffStats = null;
	DirectoryInterestSet dis = null;
	String dirs[] = null;

	/*
	 * (Each subscription is for data and invals)
	 * BETA subscribes to listen to "/a/ *:/d/ *" from ALPHA
	 * GAMMA subscribes to listen to "/c/ *:/d/ *" from ALPHA
	 * DELTA subscribes to listen to "/ *" from BETA
	 * DELTA subscribes to listen to "/ *" from GAMMA
	 * 
	 * ALPHA writes "/a/1", "/b/1", "/c/1"
	 *
	 * (ALPHA = 9, BETA = 10, GAMMA = 11, DELTA = 12)
	 */
	this.restartRMIRegistrySync();
	this.startNode(9);
	this.startNode(10);
	this.startNode(11);
	this.startNode(12);
	this.waitStartNode(9);
	this.waitStartNode(10);
	this.waitStartNode(11);
	this.waitStartNode(12);
	dirs = new String[2];

	// BETA subscribes to listen to "/a/ *:/d/ *" from ALPHA
	dirs[0] = "/a/*";
	dirs[1] = "/d/*";
	this.issueSubscribe(9,
			    10,
			    AcceptVV.makeVVAllNegatives(),
			    -1,
			    new DirectoryInterestSet(dirs),
			    SUBSCRIPTION_DURATION_MS,
			    MAX_DELAY_MS);
	this.issueSubscribeBody(9,
				10,
				AcceptVV.makeVVAllNegatives(),
				new DirectoryInterestSet(dirs),
				SUBSCRIPTION_DURATION_MS);

	// GAMMA subscribes to listen to "/c/ *:/d/ *" from ALPHA
	dirs[0] = "/c/*";
	dirs[1] = "/d/*";
	this.issueSubscribe(9,
			    11,
			    AcceptVV.makeVVAllNegatives(),
			    -1,
			    new DirectoryInterestSet(dirs),
			    SUBSCRIPTION_DURATION_MS,
			    MAX_DELAY_MS);
	this.issueSubscribeBody(9,
				11,
				AcceptVV.makeVVAllNegatives(),
				new DirectoryInterestSet(dirs),
				SUBSCRIPTION_DURATION_MS);

	// DELTA subscribes to listen to "/ *" from BETA
	this.issueSubscribe(10,
			    12,
			    AcceptVV.makeVVAllNegatives(),
			    -1,
			    new DirectoryInterestSet("/*"),
			    SUBSCRIPTION_DURATION_MS,
			    MAX_DELAY_MS);
	this.issueSubscribeBody(10,
				12,
				AcceptVV.makeVVAllNegatives(),
				new DirectoryInterestSet("/*"),
				SUBSCRIPTION_DURATION_MS);

	// ALPHA writes "/a/1", "/b/1", "/c/1"
	OutputStream os = this.getProcessOutputStream(9);
	PrintStream ps = new PrintStream(os);
	ps.println("w /a/1 0 100 0 0");
	ps.println("w /b/1 0 100 0 0");
	ps.println("w /c/1 0 100 0 0");
	ps.println("w /d/1 0 100 0 0");
	ps.println("g 9");
	ps.flush();

	// AMOL ADDED BEGIN
	// Wait for DELTA and GAMMA to receive all messages
	AcceptStamp as = null;
	ccp = this.recv(9);
	as = (AcceptStamp)ccp.getData();
	this.sendLogSync(11, (AcceptStamp)ccp.getData());
	ccp = this.recv(11);
	assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	os = this.getProcessOutputStream(11);
	ps = new PrintStream(os);
	ps.println("l 1000");
	ps.println("q");
	ps.flush();

	this.sendLogSync(12, as);
	ccp = this.recv(12);
	assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	os = this.getProcessOutputStream(12);
	ps = new PrintStream(os);
	ps.println("l 1000");
	ps.println("q");
	ps.flush();

	try{
	    Thread.sleep(10000);
	}catch(InterruptedException e){
	    System.err.println("" + e);
	}

	// DELTA subscribes to listen to "/ *" from GAMMA
	this.issueSubscribe(11,
			    12,
			    AcceptVV.makeVVAllNegatives(),
			    -1,
			    new DirectoryInterestSet("/*"),
			    SUBSCRIPTION_DURATION_MS,
			    MAX_DELAY_MS);
	this.issueSubscribeBody(11,
				12,
				AcceptVV.makeVVAllNegatives(),
				new DirectoryInterestSet("/*"),
				SUBSCRIPTION_DURATION_MS);
	//this.sendSync(12, as);
	//ccp = this.recv(12);
	//assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	os = this.getProcessOutputStream(12);
	ps = new PrintStream(os);
	ps.println("l 1000");
	ps.println("q");
	ps.flush();

	// AMOL ADDED END

	/*
	// Wait for DELTA to receive all invalidates
	ccp = this.recv(9);
	assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
	assert(ccp.getData() instanceof AcceptStamp);
	AcceptStamp as = (AcceptStamp)ccp.getData();
	this.sendSync(12, as);
	ccp = this.recv(12);
	assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	assert(ccp.getData() == null);

	// ALPHA writes "/d/1"
	ps.println("w /d/1 0 100 0 0");
	ps.println("g 9");
	ps.flush();

	// Wait for DELTA to receive this last invalidate
	ccp = this.recv(9);
	assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
	assert(ccp.getData() instanceof AcceptStamp);
	as = (AcceptStamp)ccp.getData();
	this.sendSync(12, as);
	ccp = this.recv(12);
	assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	assert(ccp.getData() == null);
	*/

	/*
	// Send demand reads and wait for them to arrive
	os = this.getProcessOutputStream(12);
	ps = new PrintStream(os);
	ps.println("r /a/1 0 100");
	ps.println("r /b/1 0 100");
	ps.println("r /c/1 0 100");
	ps.flush();
	finalStats = this.getAllStats();

	// Wait for demand reads
	ccp = this.recv(12);
	assert(ccp.getSignalType() == CoordCommPacket.READ_DONE);
	assert(ccp.getData() instanceof AcceptStamp);
	ccp = this.recv(12);
	assert(ccp.getSignalType() == CoordCommPacket.READ_DONE);
	assert(ccp.getData() instanceof AcceptStamp);
	ccp = this.recv(12);
	assert(ccp.getSignalType() == CoordCommPacket.READ_DONE);
	assert(ccp.getData() instanceof AcceptStamp);
	*/

	/*
	// Send invalidates in B's interest set to B
	asArr = new AcceptStamp[1];
	this.sendPopulateFullScript(9);
	ccp = this.recv(9);
	assert(ccp.getSignalType() == CoordCommPacket.ACCEPT_STAMP);
	assert(ccp.getData() instanceof AcceptStamp);
	asArr[0] = (AcceptStamp)ccp.getData();
	dis = this.makeInterestSet();
	System.out.println("dis = " + dis);
	this.issueSubscribe(9,
			    10,
			    AcceptVV.makeVVAllNegatives(),
			    -1L,
			    dis,
			    SUBSCRIPTION_DURATION_MS,
			    MAX_DELAY_MS);
	this.sendSync(10, asArr[0]);
	ccp = this.recv(10);
	assert(ccp.getSignalType() == CoordCommPacket.SYNC_DONE);
	assert(ccp.getData() == null);
	System.out.println("Received a SYNC_DONE");

	initStats = this.getAllStats();

	// Have B subscribe to A to receive updates
	vv = new AcceptVV(asArr);
	System.err.println("vv = " + vv);
	this.issueSubscribeBody(9,
				10,
				vv,
				dis,
				SUBSCRIPTION_DURATION_MS);
	writeEntries = this.makeRandomWrites();
	// Have A perform writes. Invals will flow to B on the already
	//   open channel, updates will flow to B on the newly-opened one.
	this.sendRandomWritesScript(9, writeEntries);
	// Have B read all writes done by A and sync back after each one.
	this.sendWaitReadsFinishScript(10, writeEntries);

	finalStats = this.getAllStats();
	diffStats = finalStats.getDiff(initStats);
	*/
	System.out.println("---------------------------------------");
	//System.out.println("" + diffStats);
	System.out.println("Exit value for node 9 = " + this.killNode(9));
	System.out.println("Exit value for node 10 = " + this.killNode(10));
	System.out.println("Exit value for node 11 = " + this.killNode(11));
	System.out.println("Exit value for node 12 = " + this.killNode(12));
    }

 /** 
 *  Program starting point 
 **/ 
    public static void main(String[] argv)
    {
	CoordExptSJ coord = null;

	if (argv.length < 2){
	    System.out.println("Usage: <exptConfigfile> <shellType>");
	    System.exit(-1);
	}
	coord = new CoordExptSJ(argv[0], argv[1]);
	coord.runExperimentC();
    }
}

 /** 
 *  $Log: CoordExptSJ.java,v $ 
 *  Revision 1.3  2005/01/18 22:49:43  zjiandan 
 *  Rewrited Class Serialization for network deliveringto reduce the overall bandwidth. 
 *  
 *  Revision 1.2  2004/05/26 02:01:01  nayate 
 *  Fixed the split-join test case 
 *  
 *  Revision 1.1  2004/05/25 06:32:42  nayate 
 *  Added a split/join coordinator 
 *  
 **/ 
