 /** 
 *  This class tests the garbage collection robustness during the 
 *  synchronization of two nodes  
 **/ 
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.Random;

public class TwoNodesLogExchange{
  public static void main(String[] argv)
    throws Exception{
    
    if(argv.length <1){
      System.out.println("Usage: SyncTimeWithStoreSize [file number]");
      System.exit(0);
    }

    //start barrier server
    int port = 9494;
    int count = 2;
    int fileNumber = new Integer(argv[0]).intValue();
    int fileSize = 5;
    long senderId = 0;
    long receiverId = 1;
    BarrierServer bs = new BarrierServer(port, count);
    bs.start();

    BarrierClient bcSender = new BarrierClient("localhost", 
                                               port, 
                                               0);
    BarrierClient bcReceiver = new BarrierClient("localhost",
                                                 port,
                                                 1);
    Config.readConfig("log.config");
    LogSenderThread st = new LogSenderThread(bcSender,
                                             fileNumber,
                                             fileSize,
                                             senderId,
                                             true,
                                             10);

    LogReceiverThread rt = new LogReceiverThread(bcReceiver,
                                                 fileNumber,
                                                 fileSize,
                                                 receiverId,
                                                 senderId,
                                                 true,
                                                 false,
                                                 10);
    st.start();
    rt.start();
    st.join();
    rt.join();
    
    st = null;
    rt = null;

    System.exit(0);
  }
}

 /** 
 **/ 
// Sender Thread 
 /** 
 **/ 
class LogSenderThread extends Thread{
  final static boolean dbg = true;
  final static boolean dbgVerbose = true;
  final static boolean dbgCorrectness = true;
  BarrierClient bc;
  int fileNumber;
  int fileSize;
  long myId;
  long totalWrites = Long.MAX_VALUE;
  //
  // URANode Members (all private)
  //
  private Core core;
  private RMIServerImpl rmiServer;         
  private SocketServer socketServer;
  private CPExptController controller;
  private LocalInterface localInterface;
  private RMIClient rmiClient;

  public LogSenderThread(BarrierClient bc,
                         int fileNumber,
                         int fileSize,
                         long senderId,
                         boolean cleanDb){
    super("senderThread");
    this.bc = bc;
    this.fileNumber = fileNumber;
    this.fileSize = fileSize;
    this.myId = senderId;
    
    try{

      this.rmiClient = new RMIClient();
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, 
                      new NodeId(myId));
      this.totalWrites = -1; //by default, total writes is infinity 
      controller = new CPExptController(core, totalWrites+fileNumber-1);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
      
    }catch(Exception e){
      System.err.println("SenderThread can't initiated:");
      e.printStackTrace();
      assert false;
    }
    
  }
  public LogSenderThread(BarrierClient bc,
                         int fileNumber,
                         int fileSize,
                         long senderId,
                         boolean cleanDb,
                         long totalWrites){
    super("senderThread");
    this.bc = bc;
    this.fileNumber = fileNumber;
    this.fileSize = fileSize;
    this.myId = senderId;
    
    try{

      this.rmiClient = new RMIClient();
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, 
                      new NodeId(myId));
      this.totalWrites = totalWrites; 
      controller = new CPExptController(core, totalWrites+fileNumber-1);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
      
    }catch(Exception e){
      System.err.println("SenderThread can't initiated:");
      e.printStackTrace();
      assert false;
    }
    
  }
  
 /** 
 *  endless write loop 
 **/ 
  private void repeatedWrite(long writeNum, boolean bound){
    boolean dbgMe = true;
   
    if(writeNum <=0){
      writeNum = Long.MAX_VALUE;//infinity
    }
    int offset = 0; 
    
    if(dbgVerbose){
      System.out.println("Start writing...");
    }
    long start = System.currentTimeMillis();
    long nextStart = System.currentTimeMillis();
    long write100Time = 0;
    int count = 0;
    while(count < writeNum){
      
      String fileName = createFileName(count%fileNumber, fileNumber);
      ImmutableBytes buffer = null;
      buffer = URANode.populateByteArray(fileSize);
      if((dbgCorrectness)&&(count==50)){
        fileName = "/bomb";
        byte[] buff = new byte[3];
        buff[0] = 'z';
        buff[1] = 'j';
        buff[2] = 'd';
        buffer = ImmutableBytes.dangerousCreateImmutableBytes(buff);
        try{
          localInterface.write(new ObjId(fileName), 
                               offset, 3, buffer, true);
          
        }catch(IOException e){
          assert false;
        }
      }else{
      
        try{
          localInterface.write(new ObjId(fileName), 
                               offset, fileSize, buffer, bound);
          
        }catch(IOException e){
          assert false;
        }
      }
      count ++;
      if(dbgMe&&(count%100 == 0)){
        nextStart = System.currentTimeMillis();
        write100Time = nextStart- start;
        
        System.out.println("total number of writes so far: " + count
                           + "    last 100 writes took time : " 
                           + write100Time);
        start = nextStart;
      } 
      
    }
   
  }
  
  public void run(){
    boolean dbgMe = true;
    bc.sendBarrierRequest(-1, -1); // wait for receiver to start
    repeatedWrite(this.fileNumber, true);//create the files
    if(dbgMe){
      System.err.println("after First repeatedWrites of " + fileNumber + " files");
      System.err.println("Sender's ISStatus:\n " + core.getISStatus());
    }
    bc.sendBarrierRequest(-1, -1);//wait for receiver to initiate states.
    repeatedWrite(totalWrites, false);
    if(dbgMe){
      System.err.println("after second repeatedWrites of " + totalWrites + " files");
      System.err.println("Sender's ISStatus:\n " + core.getISStatus());
    }
    bc.sendBarrierRequest(-1, -1);//tell receiver population is done
    bc.sendBarrierRequest(-1, -1); //wait for receiver to finish receiving
                                   // checkpoint data
  }
  
  private String createFileName(int fileNo, int total){
    assert fileNo < total;
    final boolean dbgme = false; 
    final int BRANCHING_FACTOR = 10;
    int numLen = (int)(Math.ceil(Math.log(total)/Math.log(BRANCHING_FACTOR)));
    int fileLen = 2 * numLen; // including /'s as seperator 
    byte [] bName = new byte[fileLen];
    for(int i=0; i < fileLen; i++){
      if(i%2 == 0){// even position
        bName[i] = '/';
      } else {// odd position
        long base = Math.round(Math.pow(BRANCHING_FACTOR, 
                                        numLen-(i+1)/2));
        bName[i] = (byte) ('0' + fileNo/base);
        fileNo = (int)(fileNo%base);
        if(!(bName[i]>='0' && bName[i]<='9')){
          System.out.println("bName["+i+"]:" + bName[i]
                             + " base:" + base + "fileNo:"+ fileNo + "i: " + i
                             + "bName: " + new String(bName)
                             + " fileLen: " + fileLen + " total: " + total);
        }
        assert(bName[i]>='0' && bName[i]<='9');
      }
    }
    if(dbgme){
      System.out.println("fileNo: " + fileNo + " total: " + total
                         + " fileName:" + new String(bName));
    }
    return new String(bName);
  }
  
}

 /** 
 **/ 
class LogReceiverThread extends Thread{
  final static boolean dbgCorrectness = true;
  
  final private static long SUB_DURATION = 360000000;//100 hours
  final private static long FORCE = 10000;
  
  BarrierClient bc;
  int fileNumber;
  int fileSize;
  long myId;
  long senderId;
  boolean useLog = true; //by default use log exchange
  long totalWrites = Long.MAX_VALUE; //by default infinity
  //
  // URANode Members (all private)
  //
  private Core core;
  private RMIServerImpl rmiServer;
  private SocketServer socketServer;
  private CPExptController controller;
  private LocalInterface localInterface;
  private RMIClient rmiClient;
  
  public LogReceiverThread(BarrierClient bc,
                           int fileNumber,
                           int fileSize,
                           long receiverId,
                           long senderId,
                           boolean cleanDb){
    super("receiverThread");
    this.bc = bc;
    this.fileNumber = fileNumber;
    this.fileSize = fileSize;
    this.myId = receiverId;
    this.senderId = senderId;
    
    try{
      
      this.rmiClient = new RMIClient();
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, 
                      new NodeId(myId));
      this.totalWrites = -1;
      controller = new CPExptController(core, totalWrites+fileNumber-1);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
    }catch(Exception e){
      System.err.println("ReceiverThread can't initiated:");
      e.printStackTrace();
      assert false;
    }
                           
  }
  public LogReceiverThread(BarrierClient bc,
                           int fileNumber,
                           int fileSize,
                           long receiverId,
                           long senderId,
                           boolean cleanDb,
                           boolean useLog){
    super("receiverThread");
    this.bc = bc;
    this.fileNumber = fileNumber;
    this.fileSize = fileSize;
    this.myId = receiverId;
    this.senderId = senderId;
    this.useLog = useLog;
    try{
      
      this.rmiClient = new RMIClient();
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, 
                      new NodeId(myId));
      this.totalWrites = -1;
      controller = new CPExptController(core, totalWrites+fileNumber-1);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
   
    }catch(Exception e){
      System.err.println("ReceiverThread can't initiated:");
      e.printStackTrace();
      assert false;
    }
  }

  public LogReceiverThread(BarrierClient bc,
                           int fileNumber,
                           int fileSize,
                           long receiverId,
                           long senderId,
                           boolean cleanDb,
                           boolean useLog,
                           long totalWrites){
    super("receiverThread");
    this.bc = bc;
    this.fileNumber = fileNumber;
    this.fileSize = fileSize;
    this.myId = receiverId;
    this.senderId = senderId;
    this.useLog = useLog;
    try{
      
      this.rmiClient = new RMIClient();
      boolean filterOn = true;
      core = new Core(rmiClient, filterOn, cleanDb, 
                      new NodeId(myId));
      this.totalWrites = totalWrites;
      controller = new CPExptController(core, totalWrites+fileNumber-1);
      rmiServer = new RMIServerImpl(core, controller);
      rmiServer.start();
      socketServer = new SocketServer(core, controller);
      localInterface = new LocalInterface(controller, core);
      assert(this.localInterface != null);
      core.recoverLocalState(rmiServer);
      
    }catch(Exception e){
      System.err.println("ReceiverThread can't initiated:");
      e.printStackTrace();
      assert false;
    }
  }

 /** 
 *  sync with receiver using log exchange 
 **/ 
  private void syncInitialFiles(){
    boolean dbgVerbose = true;

    if(dbgVerbose){
      System.out.println("Start sync initial states by log ...");
    }
    AcceptStamp[] as = new AcceptStamp[1];
    as[0] = new AcceptStamp(this.fileNumber-1,
                            new NodeId(senderId));
    AcceptVV endVV = new AcceptVV(as);
    try{
      rmiClient.subscribeInval(new NodeId(senderId),
                               new NodeId(myId),
                               Config.getDNS(new NodeId(myId)),
                               Config.getPortInval(new NodeId(myId)),
                               SubscriptionSet.makeSubscriptionSet("/*"),
                               AcceptVV.makeVVAllNegatives());
    }catch(Exception e){
      assert false;
    }
    while(true){
      try{
        localInterface.sync(new AcceptStamp(this.fileNumber-1,
                                            new NodeId(senderId)));
        Env.inform("Finish sync by log for "+ (this.fileNumber-1));
        Thread.sleep(100);//temparary workaround for waiting for core 
                          //to finish applying the inv to datastore
        break;
      }catch(InterruptedException ie){
        //ignore
      }
    }

    /*
    System.out.println("***** " + " fileNumber " + fileNumber
                       + " fileSize " + fileSize + " " 
                       + "numOfWrites " + totalWrites + " LOG " 
                       + (double)totalWrites/(double)fileNumber 
                       + " " + controller.logSyncTime());

    */
  }

 /** 
 *  sync with receiver using log exchange 
 **/ 
  private void syncUsingLog(){
    boolean dbgVerbose = true;

    if(dbgVerbose){
      System.out.println("Start sync by log ...");
    }
    
    AcceptStamp[] as = new AcceptStamp[1];
    as[0] = new AcceptStamp(this.fileNumber-1,
                            new NodeId(senderId));
    AcceptVV startVV = new AcceptVV(as);

    AcceptStamp[] as1 = new AcceptStamp[1];
    as1[0] = new AcceptStamp(this.fileNumber-1+this.totalWrites,
                            new NodeId(senderId));
    AcceptVV endVV = new AcceptVV(as1);
    try{

      rmiClient.subscribeInval(new NodeId(senderId),
                               new NodeId(myId),
                               Config.getDNS(new NodeId(myId)),
                               Config.getPortInval(new NodeId(myId)),
                               SubscriptionSet.makeSubscriptionSet("/*"),
                               startVV);
    }catch(Exception e){
      assert false;
    }
    while(true){
      try{
        localInterface.sync(new AcceptStamp(this.totalWrites+this.fileNumber-1,
                                            new NodeId(senderId)));
        Env.inform("Finish sync by log for "+ (this.totalWrites+this.fileNumber-1));
        Thread.sleep(100);//temparary workout for waiting for core to finish
                      //applying the inv to datastore
        break;
      }catch(InterruptedException ie){
        //ignore
      }
    }

    
    System.out.println("***** " + " fileNumber " + fileNumber
                       + " fileSize " + fileSize + " " 
                       + "numOfWrites " + totalWrites + " LOG " 
                       + (double)totalWrites/(double)fileNumber 
                       + " " + controller.logSyncTime());

  }

  //----------------------------------------------------------------
  // sync with receiver using cp
  //----------------------------------------------------------------
  private void syncUsingCP(){
    boolean dbgVerbose = true;

    if(dbgVerbose){
      System.out.println("Start sync by CP for ...");
    }
    
    try{
      String[] excludeChilds = new String[0];
      AcceptStamp[] as = new AcceptStamp[1];
      as[0] = new AcceptStamp(this.fileNumber-1,
                              new NodeId(senderId));
      AcceptVV startVV = new AcceptVV(as);

      rmiClient.subscribeInval(new NodeId(senderId),
                               new NodeId(myId),
                               Config.getDNS(new NodeId(myId)),
                               Config.getPortInval(new NodeId(myId)),
                               SubscriptionSet.makeSubscriptionSet("/*"),
                               startVV);

/*
      rmiClient.subscribeCheckpoint("/",
                                    excludeChilds,
                                    startVV,         
                                    new NodeId(senderId),
                                    new NodeId(myId),
                                    Config.getDNS(new NodeId(myId)),
                                    Config.getPortCP(new NodeId(myId)),
                                    false);
*/                        
    }catch(Exception e){
      assert false;
    }

    while(!controller.getCPUpdateStatus()){
      try{
        Thread.sleep(100);
      }catch(InterruptedException ie){
        //ignore
      }
    }
    
    System.out.println("***** " + " fileNumber " + fileNumber
                       + " fileSize " + fileSize + " " 
                       + "numOfWrites " + totalWrites + " CP " 
                       + (double)totalWrites/(double)fileNumber 
                       + " " + controller.cpSyncTime());
  }

  public void run(){
    
    bc.sendBarrierRequest(-1, -1);// wait for sender to start
    syncInitialFiles();
    bc.sendBarrierRequest(-1, -1);//tell sender that I'm done with initiation
    bc.sendBarrierRequest(-1, -1);//wait for sender to finish population
    
    if(this.useLog){
      syncUsingLog();
    }else{
      syncUsingCP();
    }
    if((dbgCorrectness)&&(fileNumber > 50)){
      try{
        System.out.println("start to read: ");
        BodyMsg bm = localInterface.read(new ObjId("/bomb"),
                                         0,
                                         3,
                                         false,
                                         false);
        System.out.println("receiver read /bomb: " + bm);
      }catch(Exception e){
        System.err.println("read fail:" + e);
        System.err.println("Receiver.ISStatus: " + core.getISStatus());
        assert false;
      }
    }
    bc.sendBarrierRequest(-1, -1);//tell sender that I'm done with sync
  }
  
}
